var reddit = (function() {
  "use strict";
  function defineContentScript(definition2) {
    return definition2;
  }
  const browser$1 = globalThis.browser?.runtime?.id ? globalThis.browser : globalThis.chrome;
  const browser = browser$1;
  const definition = defineContentScript({
    matches: ["https://www.reddit.com/*"],
    runAt: "document_start",
    async main() {
      const fetchInterceptorUrl = browser.runtime.getURL("fetchInterceptor.js");
      const missionDataFetcherUrl = browser.runtime.getURL("missionDataFetcher.js");
      const scriptFetch = document.createElement("script");
      scriptFetch.src = fetchInterceptorUrl;
      scriptFetch.type = "text/javascript";
      (document.head || document.documentElement).appendChild(scriptFetch);
      const scriptMission = document.createElement("script");
      scriptMission.src = missionDataFetcherUrl;
      scriptMission.type = "text/javascript";
      (document.head || document.documentElement).appendChild(scriptMission);
      await Promise.resolve().then(() => reddit);
    }
  });
  function print$1(method, ...args) {
    return;
  }
  const logger$1 = {
    debug: (...args) => print$1(console.debug, ...args),
    log: (...args) => print$1(console.log, ...args),
    warn: (...args) => print$1(console.warn, ...args),
    error: (...args) => print$1(console.error, ...args)
  };
  class WxtLocationChangeEvent extends Event {
    constructor(newUrl, oldUrl) {
      super(WxtLocationChangeEvent.EVENT_NAME, {});
      this.newUrl = newUrl;
      this.oldUrl = oldUrl;
    }
    static EVENT_NAME = getUniqueEventName("wxt:locationchange");
  }
  function getUniqueEventName(eventName) {
    return `${browser?.runtime?.id}:${"reddit"}:${eventName}`;
  }
  function createLocationWatcher(ctx) {
    let interval;
    let oldUrl;
    return {
      /**
       * Ensure the location watcher is actively looking for URL changes. If it's already watching,
       * this is a noop.
       */
      run() {
        if (interval != null) return;
        oldUrl = new URL(location.href);
        interval = ctx.setInterval(() => {
          let newUrl = new URL(location.href);
          if (newUrl.href !== oldUrl.href) {
            window.dispatchEvent(new WxtLocationChangeEvent(newUrl, oldUrl));
            oldUrl = newUrl;
          }
        }, 1e3);
      }
    };
  }
  class ContentScriptContext {
    constructor(contentScriptName, options) {
      this.contentScriptName = contentScriptName;
      this.options = options;
      this.abortController = new AbortController();
      if (this.isTopFrame) {
        this.listenForNewerScripts({ ignoreFirstEvent: true });
        this.stopOldScripts();
      } else {
        this.listenForNewerScripts();
      }
    }
    static SCRIPT_STARTED_MESSAGE_TYPE = getUniqueEventName(
      "wxt:content-script-started"
    );
    isTopFrame = window.self === window.top;
    abortController;
    locationWatcher = createLocationWatcher(this);
    receivedMessageIds = /* @__PURE__ */ new Set();
    get signal() {
      return this.abortController.signal;
    }
    abort(reason) {
      return this.abortController.abort(reason);
    }
    get isInvalid() {
      if (browser.runtime.id == null) {
        this.notifyInvalidated();
      }
      return this.signal.aborted;
    }
    get isValid() {
      return !this.isInvalid;
    }
    /**
     * Add a listener that is called when the content script's context is invalidated.
     *
     * @returns A function to remove the listener.
     *
     * @example
     * browser.runtime.onMessage.addListener(cb);
     * const removeInvalidatedListener = ctx.onInvalidated(() => {
     *   browser.runtime.onMessage.removeListener(cb);
     * })
     * // ...
     * removeInvalidatedListener();
     */
    onInvalidated(cb) {
      this.signal.addEventListener("abort", cb);
      return () => this.signal.removeEventListener("abort", cb);
    }
    /**
     * Return a promise that never resolves. Useful if you have an async function that shouldn't run
     * after the context is expired.
     *
     * @example
     * const getValueFromStorage = async () => {
     *   if (ctx.isInvalid) return ctx.block();
     *
     *   // ...
     * }
     */
    block() {
      return new Promise(() => {
      });
    }
    /**
     * Wrapper around `window.setInterval` that automatically clears the interval when invalidated.
     *
     * Intervals can be cleared by calling the normal `clearInterval` function.
     */
    setInterval(handler, timeout) {
      const id = setInterval(() => {
        if (this.isValid) handler();
      }, timeout);
      this.onInvalidated(() => clearInterval(id));
      return id;
    }
    /**
     * Wrapper around `window.setTimeout` that automatically clears the interval when invalidated.
     *
     * Timeouts can be cleared by calling the normal `setTimeout` function.
     */
    setTimeout(handler, timeout) {
      const id = setTimeout(() => {
        if (this.isValid) handler();
      }, timeout);
      this.onInvalidated(() => clearTimeout(id));
      return id;
    }
    /**
     * Wrapper around `window.requestAnimationFrame` that automatically cancels the request when
     * invalidated.
     *
     * Callbacks can be canceled by calling the normal `cancelAnimationFrame` function.
     */
    requestAnimationFrame(callback) {
      const id = requestAnimationFrame((...args) => {
        if (this.isValid) callback(...args);
      });
      this.onInvalidated(() => cancelAnimationFrame(id));
      return id;
    }
    /**
     * Wrapper around `window.requestIdleCallback` that automatically cancels the request when
     * invalidated.
     *
     * Callbacks can be canceled by calling the normal `cancelIdleCallback` function.
     */
    requestIdleCallback(callback, options) {
      const id = requestIdleCallback((...args) => {
        if (!this.signal.aborted) callback(...args);
      }, options);
      this.onInvalidated(() => cancelIdleCallback(id));
      return id;
    }
    addEventListener(target, type, handler, options) {
      if (type === "wxt:locationchange") {
        if (this.isValid) this.locationWatcher.run();
      }
      target.addEventListener?.(
        type.startsWith("wxt:") ? getUniqueEventName(type) : type,
        handler,
        {
          ...options,
          signal: this.signal
        }
      );
    }
    /**
     * @internal
     * Abort the abort controller and execute all `onInvalidated` listeners.
     */
    notifyInvalidated() {
      this.abort("Content script context invalidated");
      logger$1.debug(
        `Content script "${this.contentScriptName}" context invalidated`
      );
    }
    stopOldScripts() {
      window.postMessage(
        {
          type: ContentScriptContext.SCRIPT_STARTED_MESSAGE_TYPE,
          contentScriptName: this.contentScriptName,
          messageId: Math.random().toString(36).slice(2)
        },
        "*"
      );
    }
    verifyScriptStartedEvent(event) {
      const isScriptStartedEvent = event.data?.type === ContentScriptContext.SCRIPT_STARTED_MESSAGE_TYPE;
      const isSameContentScript = event.data?.contentScriptName === this.contentScriptName;
      const isNotDuplicate = !this.receivedMessageIds.has(event.data?.messageId);
      return isScriptStartedEvent && isSameContentScript && isNotDuplicate;
    }
    listenForNewerScripts(options) {
      let isFirst = true;
      const cb = (event) => {
        if (this.verifyScriptStartedEvent(event)) {
          this.receivedMessageIds.add(event.data.messageId);
          const wasFirst = isFirst;
          isFirst = false;
          if (wasFirst && options?.ignoreFirstEvent) return;
          this.notifyInvalidated();
        }
      };
      addEventListener("message", cb);
      this.onInvalidated(() => removeEventListener("message", cb));
    }
  }
  function initPlugins() {
  }
  function print(method, ...args) {
    return;
  }
  const logger = {
    debug: (...args) => print(console.debug, ...args),
    log: (...args) => print(console.log, ...args),
    warn: (...args) => print(console.warn, ...args),
    error: (...args) => print(console.error, ...args)
  };
  const result = (async () => {
    try {
      initPlugins();
      const { main, ...options } = definition;
      const ctx = new ContentScriptContext("reddit", options);
      return await main(ctx);
    } catch (err) {
      logger.error(
        `The content script "${"reddit"}" crashed on startup!`,
        err
      );
      throw err;
    }
  })();
  const DEFAULT_MAX_STORED_LOGS = 5e3;
  const BATCH_WRITE_INTERVAL = 1e4;
  const BATCH_SIZE_THRESHOLD = 50;
  const _Logger = class _Logger {
    constructor(context, config, parentContext) {
      this.parentContext = parentContext;
      this.config = {
        context,
        remoteLogging: config?.remoteLogging ?? true,
        remoteUrl: config?.remoteUrl ?? "http://localhost:7856/log",
        consoleLogging: config?.consoleLogging ?? true,
        storeLogs: config?.storeLogs ?? true,
        maxStoredLogs: config?.maxStoredLogs ?? DEFAULT_MAX_STORED_LOGS
      };
      if (typeof chrome !== "undefined" && chrome.storage) {
        chrome.storage.local.get(["automationConfig"], (result2) => {
          if (result2.automationConfig?.remoteLogging !== void 0) {
            this.config.remoteLogging = result2.automationConfig.remoteLogging;
          }
          if (result2.automationConfig?.storeLogs !== void 0) {
            this.config.storeLogs = result2.automationConfig.storeLogs;
          }
          if (result2.automationConfig?.maxStoredLogs !== void 0) {
            this.config.maxStoredLogs = result2.automationConfig.maxStoredLogs;
          }
        });
        chrome.storage.onChanged.addListener((changes, areaName) => {
          if (areaName === "local" && changes.automationConfig?.newValue) {
            const newConfig = changes.automationConfig.newValue;
            if (newConfig.remoteLogging !== void 0) {
              this.config.remoteLogging = newConfig.remoteLogging;
            }
            if (newConfig.storeLogs !== void 0) {
              this.config.storeLogs = newConfig.storeLogs;
            }
            if (newConfig.maxStoredLogs !== void 0) {
              this.config.maxStoredLogs = newConfig.maxStoredLogs;
            }
          }
        });
      }
    }
    /**
     * Send log to remote server
     */
    async sendToRemote(entry) {
      if (!this.config.remoteLogging) return;
      try {
        await fetch(this.config.remoteUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify(entry)
        }).catch(() => {
        });
      } catch (error) {
      }
    }
    /**
     * Flush buffered logs to chrome.storage
     * This is called periodically or when buffer reaches threshold
     */
    static async flushLogsToStorage() {
      if (typeof chrome === "undefined" || !chrome.storage) return;
      if (_Logger.logBuffer.length === 0) return;
      if (_Logger.flushTimer) {
        clearTimeout(_Logger.flushTimer);
        _Logger.flushTimer = null;
      }
      _Logger.isFlushScheduled = false;
      const logsToFlush = [..._Logger.logBuffer];
      _Logger.logBuffer = [];
      try {
        const result2 = await chrome.storage.local.get(["debugLogs", "automationConfig"]);
        const existingLogs = result2.debugLogs || [];
        const maxLogs = result2.automationConfig?.maxStoredLogs ?? DEFAULT_MAX_STORED_LOGS;
        const allLogs = [...existingLogs, ...logsToFlush];
        if (allLogs.length > maxLogs) {
          allLogs.splice(0, allLogs.length - maxLogs);
        }
        await chrome.storage.local.set({ debugLogs: allLogs });
      } catch (error) {
        _Logger.logBuffer.unshift(...logsToFlush);
        console.error("[LF] Failed to flush logs:", error);
      }
      if (_Logger.logBuffer.length > 0) {
        _Logger.scheduleFlush();
      }
    }
    /**
     * Schedule a flush to happen after the interval
     */
    static scheduleFlush() {
      if (_Logger.isFlushScheduled) return;
      _Logger.isFlushScheduled = true;
      _Logger.flushTimer = setTimeout(() => {
        _Logger.flushLogsToStorage();
      }, BATCH_WRITE_INTERVAL);
    }
    /**
     * Store log entry in buffer (will be flushed periodically)
     */
    storeLog(entry) {
      if (!this.config.storeLogs) return;
      if (typeof chrome === "undefined" || !chrome.storage) return;
      try {
        _Logger.logBuffer.push(entry);
        if (_Logger.logBuffer.length >= BATCH_SIZE_THRESHOLD) {
          _Logger.flushLogsToStorage();
        } else {
          _Logger.scheduleFlush();
        }
      } catch (error) {
        console.error("[LF] Failed to buffer log:", error);
      }
    }
    /**
     * Format message with prefix
     */
    formatMessage(message) {
      const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
      return `[LF][${fullContext}] ${message}`;
    }
    /**
     * Serialize data for logging
     */
    serializeData(data) {
      if (data === void 0) return void 0;
      try {
        return JSON.parse(JSON.stringify(data));
      } catch (error) {
        return String(data);
      }
    }
    /**
     * Core logging function
     */
    logInternal(level, ...args) {
      const message = args.map((arg) => {
        if (typeof arg === "string") return arg;
        if (typeof arg === "object") {
          try {
            return JSON.stringify(arg);
          } catch (error) {
            return "[Circular Reference]";
          }
        }
        return String(arg);
      }).join(" ");
      const entry = {
        timestamp: (/* @__PURE__ */ new Date()).toISOString(),
        context: this.config.context,
        level,
        message,
        data: args.length > 1 ? this.serializeData(args.slice(1)) : void 0
      };
      if (this.config.consoleLogging) {
        const consoleMethod = console[level] || console.log;
        const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
        consoleMethod(`[LF][${fullContext}]`, ...args);
      }
      this.storeLog(entry);
      this.sendToRemote(entry);
    }
    /**
     * Public logging methods - support unlimited parameters like console.log()
     */
    log(...args) {
      this.logInternal("log", ...args);
    }
    info(...args) {
      this.logInternal("info", ...args);
    }
    warn(...args) {
      this.logInternal("warn", ...args);
    }
    error(...args) {
      this.logInternal("error", ...args);
    }
    debug(...args) {
      this.logInternal("debug", ...args);
    }
    /**
     * Update logger configuration
     */
    setConfig(config) {
      this.config = { ...this.config, ...config };
    }
    /**
     * Enable/disable remote logging
     */
    setRemoteLogging(enabled) {
      this.config.remoteLogging = enabled;
    }
    /**
     * Enable/disable console logging
     */
    setConsoleLogging(enabled) {
      this.config.consoleLogging = enabled;
    }
    /**
     * Create a nested logger with additional context
     */
    createNestedLogger(nestedContext) {
      const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
      return new _Logger(
        nestedContext,
        {
          remoteLogging: this.config.remoteLogging,
          remoteUrl: this.config.remoteUrl,
          consoleLogging: this.config.consoleLogging
        },
        fullContext
      );
    }
    /**
     * Flush all buffered logs to storage immediately
     * Call this before extension unload to prevent log loss
     */
    static async flushLogs() {
      return _Logger.flushLogsToStorage();
    }
  };
  _Logger.logBuffer = [];
  _Logger.flushTimer = null;
  _Logger.isFlushScheduled = false;
  let Logger = _Logger;
  function createLogger(context, config, parentContext) {
    return new Logger(context, config, parentContext);
  }
  createLogger("POPUP");
  createLogger("SW");
  const redditLogger = createLogger("REDDIT");
  createLogger("DEVVIT");
  createLogger("DEVVIT-GIAE");
  function normalizeSelector(sel) {
    function saveUnmatched() {
      if (unmatched) {
        if (tokens.length > 0 && /^[~+>]$/.test(tokens[tokens.length - 1])) {
          tokens.push(" ");
        }
        tokens.push(unmatched);
      }
    }
    var tokens = [], match, unmatched, regex, state = [0], next_match_idx = 0, prev_match_idx, not_escaped_pattern = /(?:[^\\]|(?:^|[^\\])(?:\\\\)+)$/, whitespace_pattern = /^\s+$/, state_patterns = [
      /\s+|\/\*|["'>~+[(]/g,
      // general
      /\s+|\/\*|["'[\]()]/g,
      // [..] set
      /\s+|\/\*|["'[\]()]/g,
      // (..) set
      null,
      // string literal (placeholder)
      /\*\//g
      // comment
    ];
    sel = sel.trim();
    while (true) {
      unmatched = "";
      regex = state_patterns[state[state.length - 1]];
      regex.lastIndex = next_match_idx;
      match = regex.exec(sel);
      if (match) {
        prev_match_idx = next_match_idx;
        next_match_idx = regex.lastIndex;
        if (prev_match_idx < next_match_idx - match[0].length) {
          unmatched = sel.substring(
            prev_match_idx,
            next_match_idx - match[0].length
          );
        }
        if (state[state.length - 1] < 3) {
          saveUnmatched();
          if (match[0] === "[") {
            state.push(1);
          } else if (match[0] === "(") {
            state.push(2);
          } else if (/^["']$/.test(match[0])) {
            state.push(3);
            state_patterns[3] = new RegExp(match[0], "g");
          } else if (match[0] === "/*") {
            state.push(4);
          } else if (/^[\])]$/.test(match[0]) && state.length > 0) {
            state.pop();
          } else if (/^(?:\s+|[~+>])$/.test(match[0])) {
            if (tokens.length > 0 && !whitespace_pattern.test(tokens[tokens.length - 1]) && state[state.length - 1] === 0) {
              tokens.push(" ");
            }
            if (state[state.length - 1] === 1 && tokens.length === 5 && tokens[2].charAt(tokens[2].length - 1) === "=") {
              tokens[4] = " " + tokens[4];
            }
            if (whitespace_pattern.test(match[0])) {
              continue;
            }
          }
          tokens.push(match[0]);
        } else {
          tokens[tokens.length - 1] += unmatched;
          if (not_escaped_pattern.test(tokens[tokens.length - 1])) {
            if (state[state.length - 1] === 4) {
              if (tokens.length < 2 || whitespace_pattern.test(tokens[tokens.length - 2])) {
                tokens.pop();
              } else {
                tokens[tokens.length - 1] = " ";
              }
              match[0] = "";
            }
            state.pop();
          }
          tokens[tokens.length - 1] += match[0];
        }
      } else {
        unmatched = sel.substr(next_match_idx);
        saveUnmatched();
        break;
      }
    }
    return tokens.join("").trim();
  }
  function querySelectorDeep(selector, root2 = document, allElements = null) {
    return _querySelectorDeep(selector, false, root2, allElements);
  }
  function _querySelectorDeep(selector, findMany, root2, allElements = null) {
    selector = normalizeSelector(selector);
    let lightElement = root2.querySelector(selector);
    if (document.head.createShadowRoot || document.head.attachShadow) {
      if (lightElement) {
        return lightElement;
      }
      const selectionsToMake = splitByCharacterUnlessQuoted(selector, ",");
      return selectionsToMake.reduce((acc, minimalSelector) => {
        if (acc) {
          return acc;
        }
        const splitSelector = splitByCharacterUnlessQuoted(minimalSelector.replace(/^\s+/g, "").replace(/\s*([>+~]+)\s*/g, "$1"), " ").filter((entry) => !!entry).map((entry) => splitByCharacterUnlessQuoted(entry, ">"));
        const possibleElementsIndex = splitSelector.length - 1;
        const lastSplitPart = splitSelector[possibleElementsIndex][splitSelector[possibleElementsIndex].length - 1];
        const possibleElements = collectAllElementsDeep(lastSplitPart, root2, allElements);
        const findElements = findMatchingElement(splitSelector, possibleElementsIndex, root2);
        {
          acc = possibleElements.find(findElements);
          return acc || null;
        }
      }, null);
    } else {
      {
        return lightElement;
      }
    }
  }
  function findMatchingElement(splitSelector, possibleElementsIndex, root2) {
    return (element) => {
      let position = possibleElementsIndex;
      let parent = element;
      let foundElement = false;
      while (parent && !isDocumentNode(parent)) {
        let foundMatch = true;
        if (splitSelector[position].length === 1) {
          foundMatch = parent.matches(splitSelector[position]);
        } else {
          const reversedParts = [].concat(splitSelector[position]).reverse();
          let newParent = parent;
          for (const part of reversedParts) {
            if (!newParent || !newParent.matches(part)) {
              foundMatch = false;
              break;
            }
            newParent = findParentOrHost(newParent, root2);
          }
        }
        if (foundMatch && position === 0) {
          foundElement = true;
          break;
        }
        if (foundMatch) {
          position--;
        }
        parent = findParentOrHost(parent, root2);
      }
      return foundElement;
    };
  }
  function splitByCharacterUnlessQuoted(selector, character) {
    return selector.match(/\\?.|^$/g).reduce((p, c) => {
      if (c === '"' && !p.sQuote) {
        p.quote ^= 1;
        p.a[p.a.length - 1] += c;
      } else if (c === "'" && !p.quote) {
        p.sQuote ^= 1;
        p.a[p.a.length - 1] += c;
      } else if (!p.quote && !p.sQuote && c === character) {
        p.a.push("");
      } else {
        p.a[p.a.length - 1] += c;
      }
      return p;
    }, { a: [""] }).a;
  }
  function isDocumentNode(node) {
    return node.nodeType === Node.DOCUMENT_FRAGMENT_NODE || node.nodeType === Node.DOCUMENT_NODE;
  }
  function findParentOrHost(element, root2) {
    const parentNode = element.parentNode;
    return parentNode && parentNode.host && parentNode.nodeType === 11 ? parentNode.host : parentNode === root2 ? null : parentNode;
  }
  function collectAllElementsDeep(selector = null, root2, cachedElements = null) {
    let allElements = [];
    if (cachedElements) {
      allElements = cachedElements;
    } else {
      const findAllElements = function(nodes) {
        for (let i = 0; i < nodes.length; i++) {
          const el = nodes[i];
          allElements.push(el);
          if (el.shadowRoot) {
            findAllElements(el.shadowRoot.querySelectorAll("*"));
          }
        }
      };
      if (root2.shadowRoot) {
        findAllElements(root2.shadowRoot.querySelectorAll("*"));
      }
      findAllElements(root2.querySelectorAll("*"));
    }
    return selector ? allElements.filter((el) => el.matches(selector)) : allElements;
  }
  const findGameIframe = () => {
    const gameIframe = querySelectorDeep('iframe[src*="devvit.net"]');
    return gameIframe;
  };
  const isGameDialogOpen = () => {
    const gameIframe = findGameIframe();
    if (!gameIframe) return false;
    const iframeLoaded = !!(gameIframe.src && gameIframe.src.includes("devvit.net"));
    redditLogger.log("Game dialog status check", {
      hasIframe: !!gameIframe,
      iframeLoaded,
      isOpen: iframeLoaded
    });
    return iframeLoaded;
  };
  function parseLevelFromPost(post) {
    try {
      const title = post.getAttribute("post-title") || "";
      const permalink = post.getAttribute("permalink") || "";
      const postId = post.getAttribute("id") || "";
      const author = post.getAttribute("author") || "";
      if (!title) {
        redditLogger.warn("Post has no title", { post });
        return null;
      }
      const metaLevels = ["The Inn"];
      if (metaLevels.some((meta) => title === meta)) {
        redditLogger.log(`Skipping meta level: "${title}"`);
        return null;
      }
      const flairLink = post.querySelector('a[href*="flair_name"]');
      const levelRange = flairLink ? flairLink.textContent?.trim() || null : null;
      const levelMatch = title.match(/(?:level|mission)\s*(\d+)/i);
      const levelNumber = levelMatch ? parseInt(levelMatch[1]) : null;
      let levelRangeMin = null;
      let levelRangeMax = null;
      if (levelRange) {
        const rangeMatch = levelRange.match(/Level\s+(\d+)-(\d+)/i);
        if (rangeMatch) {
          levelRangeMin = parseInt(rangeMatch[1]);
          levelRangeMax = parseInt(rangeMatch[2]);
        }
      }
      const starDifficulty = 0;
      const isTitleCleared = title.toLowerCase().includes("cleared") || title.toLowerCase().includes("completed") || title.includes("✓") || title.includes("✔") || title.includes("[done]") || title.toLowerCase().includes("solved");
      const href = permalink ? `https://www.reddit.com${permalink}` : null;
      return {
        title,
        href,
        postId,
        author,
        levelNumber,
        levelRange,
        levelRangeMin,
        levelRangeMax,
        stars: starDifficulty,
        cleared: isTitleCleared,
        element: post
      };
    } catch (error) {
      redditLogger.error("Error parsing level", { error: String(error) });
      return null;
    }
  }
  async function saveScannedMission(level) {
    if (!level.postId || !level.href) {
      return;
    }
    if (!level.stars || level.stars === 0) {
      return;
    }
    if (!level.levelRangeMin || !level.levelRangeMax) {
      return;
    }
    if (level.levelRangeMax < level.levelRangeMin) {
      return;
    }
    try {
      if (level.cleared) {
        const { isMissionCleared: isMissionCleared2 } = await Promise.resolve().then(() => userProgress);
        const isCleared = await isMissionCleared2(level.postId);
        if (!isCleared) {
          redditLogger.log("Mission cleared status detected - updating user progress", {
            postId: level.postId,
            title: level.title.substring(0, 50)
          });
          const { markMissionCleared: markMissionCleared2 } = await Promise.resolve().then(() => missions);
          await markMissionCleared2(level.postId);
        }
      }
      redditLogger.log(`Found mission: ${level.postId}`, {
        title: level.title.substring(0, 50),
        starDifficulty: level.stars,
        levelRange: level.levelRange,
        author: level.author,
        cleared: level.cleared
      });
    } catch (error) {
      redditLogger.error("Failed to save mission", {
        error: error instanceof Error ? error.message : String(error),
        errorStack: error instanceof Error ? error.stack : void 0,
        level: {
          title: level.title,
          postId: level.postId,
          starDifficulty: level.stars,
          href: level.href,
          author: level.author
        }
      });
    }
  }
  function getAllLevels() {
    const posts = document.querySelectorAll("shreddit-post");
    const levels = [];
    let savedCount = 0;
    let skippedCount = 0;
    let parseFailedCount = 0;
    let missingDataCount = 0;
    redditLogger.log("Found posts on page, parsing...", {
      postsCount: posts.length
    });
    posts.forEach((post, index) => {
      const level = parseLevelFromPost(post);
      if (level) {
        if (!level.postId || !level.href) {
          missingDataCount++;
          redditLogger.warn(`Post ${index + 1} missing data`, {
            title: level.title.substring(0, 50),
            hasPostId: !!level.postId,
            hasHref: !!level.href
          });
          return;
        }
        levels.push(level);
        saveScannedMission(level).then(() => {
          savedCount++;
        }).catch((err) => {
          redditLogger.error("Failed to save mission", {
            error: err instanceof Error ? err.message : String(err),
            postId: level.postId
          });
          skippedCount++;
        });
      } else {
        parseFailedCount++;
      }
    });
    redditLogger.log("Parsed valid missions from posts", {
      validMissions: levels.length,
      totalPosts: posts.length
    });
    redditLogger.log("Parse stats", {
      parseFailedCount,
      missingDataCount
    });
    setTimeout(() => {
      redditLogger.log("Save summary", {
        savedCount,
        skippedCount
      });
    }, 2e3);
    return levels;
  }
  function filterLevels(levels, filters) {
    return levels.filter((level) => {
      if (!filters.stars.includes(level.stars)) {
        return false;
      }
      if (level.levelNumber !== null) {
        if (level.levelNumber < filters.minLevel || level.levelNumber > filters.maxLevel) {
          return false;
        }
      }
      if (level.levelNumber === null && level.levelRangeMax !== null) {
        if (level.levelRangeMax < filters.minLevel || level.levelRangeMax > filters.maxLevel) {
          return false;
        }
      }
      if (level.cleared) {
        return false;
      }
      return true;
    });
  }
  const STORAGE_KEYS = {
    MISSIONS: "missions",
    // Mission data (static, from database)
    USER_PROGRESS: "userProgress"
  };
  const CACHE_DURATION = 5 * 60 * 1e3;
  const STORAGE_KEY = "redditUserCache";
  async function getCurrentRedditUser() {
    const cachedUser = await getCachedUserIncludingExpired();
    const isCacheFresh = cachedUser ? Date.now() - cachedUser.timestamp < CACHE_DURATION : false;
    if (isCacheFresh && cachedUser) {
      return cachedUser.username;
    }
    try {
      const response = await fetch("https://www.reddit.com/api/me.json", {
        credentials: "include"
      });
      if (!response.ok) {
        if (cachedUser) {
          return cachedUser.username;
        }
        return "default";
      }
      const data = await response.json();
      const username = data?.data?.name;
      if (username && typeof username === "string") {
        await cacheUser(username);
        return username;
      } else {
        if (cachedUser) {
          return cachedUser.username;
        }
        return "default";
      }
    } catch (error) {
      if (cachedUser) {
        return cachedUser.username;
      }
      return "default";
    }
  }
  async function getCachedUserIncludingExpired() {
    return new Promise((resolve) => {
      chrome.storage.local.get([STORAGE_KEY], (result2) => {
        if (chrome.runtime.lastError) {
          resolve(null);
          return;
        }
        const cache = result2[STORAGE_KEY];
        resolve(cache || null);
      });
    });
  }
  async function cacheUser(username) {
    return new Promise((resolve, reject) => {
      const cache = {
        username,
        timestamp: Date.now()
      };
      chrome.storage.local.set({ [STORAGE_KEY]: cache }, () => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve();
        }
      });
    });
  }
  const userDetection = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
    __proto__: null,
    getCurrentRedditUser
  }, Symbol.toStringTag, { value: "Module" }));
  function createEmptyProgressData() {
    return {
      cleared: [],
      disabled: [],
      clearedAt: {},
      loot: {}
    };
  }
  async function getMultiUserProgress() {
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.USER_PROGRESS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve(result2[STORAGE_KEYS.USER_PROGRESS] || {});
        }
      });
    });
  }
  async function setMultiUserProgress(data) {
    return new Promise((resolve, reject) => {
      chrome.storage.local.set({ [STORAGE_KEYS.USER_PROGRESS]: data }, () => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve();
        }
      });
    });
  }
  async function isMissionCleared(postId) {
    const progress = await getAllUserProgress();
    return progress.cleared.includes(postId);
  }
  async function getAllUserProgress() {
    const username = await getCurrentRedditUser();
    const multiUserData = await getMultiUserProgress();
    return multiUserData[username] || createEmptyProgressData();
  }
  async function markMissionCleared$1(postId) {
    const username = await getCurrentRedditUser();
    const multiUserData = await getMultiUserProgress();
    const userProgress2 = multiUserData[username] || createEmptyProgressData();
    if (!userProgress2.cleared.includes(postId)) {
      userProgress2.cleared.push(postId);
    }
    userProgress2.clearedAt[postId] = {
      timestamp: Date.now(),
      duration: void 0
    };
    multiUserData[username] = userProgress2;
    await setMultiUserProgress(multiUserData);
  }
  async function syncClearedMissionsFromServer(clearedMissions) {
    const username = await getCurrentRedditUser();
    const multiUserData = await getMultiUserProgress();
    const userProgress2 = multiUserData[username] || createEmptyProgressData();
    for (const [postId, serverData] of Object.entries(clearedMissions)) {
      if (!userProgress2.cleared.includes(postId)) {
        userProgress2.cleared.push(postId);
      }
      userProgress2.clearedAt[postId] = {
        timestamp: serverData.cleared_timestamp,
        duration: serverData.duration
      };
    }
    multiUserData[username] = userProgress2;
    await setMultiUserProgress(multiUserData);
  }
  const userProgress = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
    __proto__: null,
    getAllUserProgress,
    isMissionCleared,
    markMissionCleared: markMissionCleared$1,
    syncClearedMissionsFromServer
  }, Symbol.toStringTag, { value: "Module" }));
  function isLegacyFormat(record) {
    return record && typeof record === "object" && "metadata" in record && record.metadata !== void 0;
  }
  function migrateLegacyRecord(legacy) {
    const mission = legacy.metadata?.mission;
    const record = {
      // Core identification
      postId: legacy.postId,
      timestamp: legacy.timestamp,
      permalink: legacy.permalink,
      // Mission metadata
      missionTitle: legacy.metadata?.missionTitle || legacy.missionTitle || `Mission ${legacy.postId.slice(3)}`,
      missionAuthorName: legacy.metadata?.missionAuthorName || "Unknown",
      // Mission data (from nested mission object or top-level fields)
      environment: mission?.environment || legacy.environment || "haunted_forest",
      encounters: mission?.encounters || [],
      minLevel: mission?.minLevel || legacy.minLevel || 1,
      maxLevel: mission?.maxLevel || legacy.maxLevel || 340,
      difficulty: mission?.difficulty || legacy.difficulty || 0,
      foodImage: mission?.foodImage || "",
      foodName: mission?.foodName || legacy.foodName || "",
      authorWeaponId: mission?.authorWeaponId || "",
      chef: mission?.chef || "",
      cart: mission?.cart || "",
      rarity: mission?.rarity || "common",
      type: mission?.type
    };
    return record;
  }
  function normalizeMissionRecord(record) {
    if (isLegacyFormat(record)) {
      return migrateLegacyRecord(record);
    }
    return record;
  }
  async function saveMission(mission) {
    return new Promise((resolve, reject) => {
      if (!chrome.runtime?.id) {
        reject(new Error("Extension context invalidated"));
        return;
      }
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        const missions2 = result2[STORAGE_KEYS.MISSIONS] || {};
        missions2[mission.postId] = mission;
        chrome.storage.local.set({ [STORAGE_KEYS.MISSIONS]: missions2 }, () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            chrome.runtime.sendMessage({
              type: "MISSIONS_UPDATED"
            }).catch(() => {
            });
            resolve();
          }
        });
      });
    });
  }
  async function getAllMissions() {
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          const rawMissions = result2[STORAGE_KEYS.MISSIONS] || {};
          const migratedMissions = {};
          for (const postId in rawMissions) {
            migratedMissions[postId] = normalizeMissionRecord(rawMissions[postId]);
          }
          resolve(migratedMissions);
        }
      });
    });
  }
  async function getMission(postId) {
    const missions2 = await getAllMissions();
    return missions2[postId] || null;
  }
  async function markMissionCleared(postId) {
    return markMissionCleared$1(postId);
  }
  const missions = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
    __proto__: null,
    getAllMissions,
    getMission,
    markMissionCleared,
    saveMission
  }, Symbol.toStringTag, { value: "Module" }));
  async function getFilteredUnclearedMissions(filters) {
    const [missions2, progress] = await Promise.all([getAllMissions(), getAllUserProgress()]);
    const allStarsSelected = !filters?.stars || filters.stars.length === 5 && [1, 2, 3, 4, 5].every((d) => filters.stars.includes(d));
    let unclearedMissions = Object.values(missions2).filter(
      (m) => !progress.cleared.includes(m.postId) && !progress.disabled.includes(m.postId) && m.minLevel !== void 0 && m.maxLevel !== void 0 && // If all stars selected or no star filter, include missions with null difficulty
      (allStarsSelected || (m.difficulty ?? 0) > 0)
    );
    if (filters) {
      unclearedMissions = unclearedMissions.filter((m) => {
        if (filters.stars && filters.stars.length > 0) {
          if (allStarsSelected) ;
          else if (!filters.stars.includes(m.difficulty || 0)) {
            return false;
          }
        }
        if (filters.minLevel !== void 0) {
          if (m.minLevel < filters.minLevel) {
            return false;
          }
        }
        if (filters.maxLevel !== void 0) {
          if (m.maxLevel > filters.maxLevel) {
            return false;
          }
        }
        return true;
      });
    }
    unclearedMissions.sort((a, b) => (b.timestamp || 0) - (a.timestamp || 0));
    return unclearedMissions;
  }
  async function getNextUnclearedMission(filters) {
    const unclearedMissions = await getFilteredUnclearedMissions(filters);
    console.log("[getNextUnclearedMission] Found uncleared missions", {
      count: unclearedMissions.length,
      excludePostIds: filters?.excludePostIds,
      firstFew: unclearedMissions.slice(0, 3).map((m) => ({
        postId: m.postId,
        title: m.missionTitle?.substring(0, 30)
      }))
    });
    const filteredMissions = filters?.excludePostIds ? unclearedMissions.filter((m) => !filters.excludePostIds.includes(m.postId)) : unclearedMissions;
    console.log("[getNextUnclearedMission] After exclusion", {
      count: filteredMissions.length,
      nextMission: filteredMissions[0] ? {
        postId: filteredMissions[0].postId,
        title: filteredMissions[0].missionTitle?.substring(0, 30)
      } : null
    });
    return filteredMissions[0] || null;
  }
  function safeSendMessage(message, callback) {
    try {
      chrome.runtime.sendMessage(message, (response) => {
        if (chrome.runtime.lastError) {
          handleExtensionContextError(chrome.runtime.lastError);
          return;
        }
        if (callback) ;
      });
    } catch (error) {
      handleExtensionContextError(error);
    }
  }
  function handleExtensionContextError(error) {
    const errorMsg = String(error.message || error);
    if (errorMsg.includes("Extension context invalidated")) {
      showExtensionReloadNotification();
      redditLogger.log("[ExtensionContext] Extension was updated/reloaded, page needs refresh", {
        error: errorMsg,
        url: window.location.href
      });
    } else {
      redditLogger.error("[ExtensionContext] Runtime error", {
        error: errorMsg
      });
    }
  }
  function showExtensionReloadNotification() {
    const existingNotification = document.getElementById("lazyfrog-reload-notification");
    if (existingNotification) {
      existingNotification.remove();
    }
    const notification = document.createElement("div");
    notification.id = "lazyfrog-reload-notification";
    notification.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 10000;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 16px 24px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    max-width: 400px;
    animation: slideIn 0.3s ease-out;
  `;
    notification.innerHTML = `
    <div style="display: flex; align-items: start; gap: 12px;">
      <div style="flex: 1;">
        <div style="font-weight: 600; font-size: 16px; margin-bottom: 8px;">🔄 LazyFrog Extension Updated</div>
        <div style="font-size: 14px; opacity: 0.95; margin-bottom: 12px;">The extension was updated or reloaded. Please refresh this page.</div>
        <button id="lazyfrog-reload-btn" style="
          background: white;
          color: #667eea;
          border: none;
          padding: 8px 16px;
          border-radius: 4px;
          font-weight: 600;
          cursor: pointer;
          font-size: 14px;
          transition: transform 0.1s;
        ">Reload Page Now</button>
      </div>
      <button id="lazyfrog-close-notification" style="
        background: none;
        border: none;
        color: white;
        font-size: 24px;
        cursor: pointer;
        padding: 0;
        width: 24px;
        height: 24px;
        display: flex;
        align-items: center;
        justify-content: center;
        line-height: 1;
        opacity: 0.8;
        transition: opacity 0.2s;
      ">×</button>
    </div>
  `;
    const style = document.createElement("style");
    style.textContent = `
    @keyframes slideIn {
      from {
        transform: translateX(400px);
        opacity: 0;
      }
      to {
        transform: translateX(0);
        opacity: 1;
      }
    }
    @keyframes slideOut {
      to {
        transform: translateX(400px);
        opacity: 0;
      }
    }
    #lazyfrog-reload-btn:hover {
      transform: scale(1.05);
      opacity: 1;
    }
  `;
    document.head.appendChild(style);
    document.body.appendChild(notification);
    const reloadBtn = document.getElementById("lazyfrog-reload-btn");
    const closeBtn = document.getElementById("lazyfrog-close-notification");
    if (reloadBtn) {
      reloadBtn.addEventListener("click", () => {
        window.location.reload();
      });
    }
    if (closeBtn) {
      closeBtn.addEventListener("click", () => {
        notification.remove();
      });
    }
    setTimeout(() => {
      if (notification.parentElement) {
        notification.style.animation = "slideOut 0.3s ease-in";
        setTimeout(() => notification.remove(), 300);
      }
    }, 3e4);
  }
  function initializeDebugFunctions(currentBotState2, currentBotContext2, renderControlPanel2) {
    window.autoSupperDebug = {
      getAllLevels,
      parseLevelFromPost,
      filterLevels,
      getState: () => ({ state: currentBotState2, context: currentBotContext2 }),
      sendEvent: (event) => safeSendMessage(event),
      renderControlPanel: renderControlPanel2,
      testSelectors: () => {
        redditLogger.log("Testing selectors");
        const selectors = [
          '[data-testid="post-container"]',
          "shreddit-post",
          '[data-click-id="background"]',
          "article",
          '[id^="t3_"]'
        ];
        selectors.forEach((sel) => {
          const found = document.querySelectorAll(sel);
          redditLogger.log(`Selector "${sel}"`, { count: found.length });
        });
      }
    };
    redditLogger.log("Debug functions available: window.autoSupperDebug");
  }
  var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
  function getDefaultExportFromCjs(x) {
    return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, "default") ? x["default"] : x;
  }
  var indexMinimal = {};
  var minimal$1 = {};
  var aspromise;
  var hasRequiredAspromise;
  function requireAspromise() {
    if (hasRequiredAspromise) return aspromise;
    hasRequiredAspromise = 1;
    aspromise = asPromise;
    function asPromise(fn, ctx) {
      var params = new Array(arguments.length - 1), offset = 0, index = 2, pending = true;
      while (index < arguments.length)
        params[offset++] = arguments[index++];
      return new Promise(function executor(resolve, reject) {
        params[offset] = function callback(err) {
          if (pending) {
            pending = false;
            if (err)
              reject(err);
            else {
              var params2 = new Array(arguments.length - 1), offset2 = 0;
              while (offset2 < params2.length)
                params2[offset2++] = arguments[offset2];
              resolve.apply(null, params2);
            }
          }
        };
        try {
          fn.apply(ctx || null, params);
        } catch (err) {
          if (pending) {
            pending = false;
            reject(err);
          }
        }
      });
    }
    return aspromise;
  }
  var base64 = {};
  var hasRequiredBase64;
  function requireBase64() {
    if (hasRequiredBase64) return base64;
    hasRequiredBase64 = 1;
    (function(exports) {
      var base642 = exports;
      base642.length = function length(string) {
        var p = string.length;
        if (!p)
          return 0;
        var n = 0;
        while (--p % 4 > 1 && string.charAt(p) === "=")
          ++n;
        return Math.ceil(string.length * 3) / 4 - n;
      };
      var b64 = new Array(64);
      var s64 = new Array(123);
      for (var i = 0; i < 64; )
        s64[b64[i] = i < 26 ? i + 65 : i < 52 ? i + 71 : i < 62 ? i - 4 : i - 59 | 43] = i++;
      base642.encode = function encode(buffer, start, end) {
        var parts = null, chunk = [];
        var i2 = 0, j = 0, t;
        while (start < end) {
          var b = buffer[start++];
          switch (j) {
            case 0:
              chunk[i2++] = b64[b >> 2];
              t = (b & 3) << 4;
              j = 1;
              break;
            case 1:
              chunk[i2++] = b64[t | b >> 4];
              t = (b & 15) << 2;
              j = 2;
              break;
            case 2:
              chunk[i2++] = b64[t | b >> 6];
              chunk[i2++] = b64[b & 63];
              j = 0;
              break;
          }
          if (i2 > 8191) {
            (parts || (parts = [])).push(String.fromCharCode.apply(String, chunk));
            i2 = 0;
          }
        }
        if (j) {
          chunk[i2++] = b64[t];
          chunk[i2++] = 61;
          if (j === 1)
            chunk[i2++] = 61;
        }
        if (parts) {
          if (i2)
            parts.push(String.fromCharCode.apply(String, chunk.slice(0, i2)));
          return parts.join("");
        }
        return String.fromCharCode.apply(String, chunk.slice(0, i2));
      };
      var invalidEncoding = "invalid encoding";
      base642.decode = function decode(string, buffer, offset) {
        var start = offset;
        var j = 0, t;
        for (var i2 = 0; i2 < string.length; ) {
          var c = string.charCodeAt(i2++);
          if (c === 61 && j > 1)
            break;
          if ((c = s64[c]) === void 0)
            throw Error(invalidEncoding);
          switch (j) {
            case 0:
              t = c;
              j = 1;
              break;
            case 1:
              buffer[offset++] = t << 2 | (c & 48) >> 4;
              t = c;
              j = 2;
              break;
            case 2:
              buffer[offset++] = (t & 15) << 4 | (c & 60) >> 2;
              t = c;
              j = 3;
              break;
            case 3:
              buffer[offset++] = (t & 3) << 6 | c;
              j = 0;
              break;
          }
        }
        if (j === 1)
          throw Error(invalidEncoding);
        return offset - start;
      };
      base642.test = function test(string) {
        return /^(?:[A-Za-z0-9+/]{4})*(?:[A-Za-z0-9+/]{2}==|[A-Za-z0-9+/]{3}=)?$/.test(string);
      };
    })(base64);
    return base64;
  }
  var eventemitter;
  var hasRequiredEventemitter;
  function requireEventemitter() {
    if (hasRequiredEventemitter) return eventemitter;
    hasRequiredEventemitter = 1;
    eventemitter = EventEmitter;
    function EventEmitter() {
      this._listeners = {};
    }
    EventEmitter.prototype.on = function on(evt, fn, ctx) {
      (this._listeners[evt] || (this._listeners[evt] = [])).push({
        fn,
        ctx: ctx || this
      });
      return this;
    };
    EventEmitter.prototype.off = function off(evt, fn) {
      if (evt === void 0)
        this._listeners = {};
      else {
        if (fn === void 0)
          this._listeners[evt] = [];
        else {
          var listeners = this._listeners[evt];
          for (var i = 0; i < listeners.length; )
            if (listeners[i].fn === fn)
              listeners.splice(i, 1);
            else
              ++i;
        }
      }
      return this;
    };
    EventEmitter.prototype.emit = function emit(evt) {
      var listeners = this._listeners[evt];
      if (listeners) {
        var args = [], i = 1;
        for (; i < arguments.length; )
          args.push(arguments[i++]);
        for (i = 0; i < listeners.length; )
          listeners[i].fn.apply(listeners[i++].ctx, args);
      }
      return this;
    };
    return eventemitter;
  }
  var float;
  var hasRequiredFloat;
  function requireFloat() {
    if (hasRequiredFloat) return float;
    hasRequiredFloat = 1;
    float = factory(factory);
    function factory(exports) {
      if (typeof Float32Array !== "undefined") (function() {
        var f32 = new Float32Array([-0]), f8b = new Uint8Array(f32.buffer), le = f8b[3] === 128;
        function writeFloat_f32_cpy(val, buf, pos) {
          f32[0] = val;
          buf[pos] = f8b[0];
          buf[pos + 1] = f8b[1];
          buf[pos + 2] = f8b[2];
          buf[pos + 3] = f8b[3];
        }
        function writeFloat_f32_rev(val, buf, pos) {
          f32[0] = val;
          buf[pos] = f8b[3];
          buf[pos + 1] = f8b[2];
          buf[pos + 2] = f8b[1];
          buf[pos + 3] = f8b[0];
        }
        exports.writeFloatLE = le ? writeFloat_f32_cpy : writeFloat_f32_rev;
        exports.writeFloatBE = le ? writeFloat_f32_rev : writeFloat_f32_cpy;
        function readFloat_f32_cpy(buf, pos) {
          f8b[0] = buf[pos];
          f8b[1] = buf[pos + 1];
          f8b[2] = buf[pos + 2];
          f8b[3] = buf[pos + 3];
          return f32[0];
        }
        function readFloat_f32_rev(buf, pos) {
          f8b[3] = buf[pos];
          f8b[2] = buf[pos + 1];
          f8b[1] = buf[pos + 2];
          f8b[0] = buf[pos + 3];
          return f32[0];
        }
        exports.readFloatLE = le ? readFloat_f32_cpy : readFloat_f32_rev;
        exports.readFloatBE = le ? readFloat_f32_rev : readFloat_f32_cpy;
      })();
      else (function() {
        function writeFloat_ieee754(writeUint, val, buf, pos) {
          var sign = val < 0 ? 1 : 0;
          if (sign)
            val = -val;
          if (val === 0)
            writeUint(1 / val > 0 ? (
              /* positive */
              0
            ) : (
              /* negative 0 */
              2147483648
            ), buf, pos);
          else if (isNaN(val))
            writeUint(2143289344, buf, pos);
          else if (val > 34028234663852886e22)
            writeUint((sign << 31 | 2139095040) >>> 0, buf, pos);
          else if (val < 11754943508222875e-54)
            writeUint((sign << 31 | Math.round(val / 1401298464324817e-60)) >>> 0, buf, pos);
          else {
            var exponent = Math.floor(Math.log(val) / Math.LN2), mantissa = Math.round(val * Math.pow(2, -exponent) * 8388608) & 8388607;
            writeUint((sign << 31 | exponent + 127 << 23 | mantissa) >>> 0, buf, pos);
          }
        }
        exports.writeFloatLE = writeFloat_ieee754.bind(null, writeUintLE);
        exports.writeFloatBE = writeFloat_ieee754.bind(null, writeUintBE);
        function readFloat_ieee754(readUint, buf, pos) {
          var uint = readUint(buf, pos), sign = (uint >> 31) * 2 + 1, exponent = uint >>> 23 & 255, mantissa = uint & 8388607;
          return exponent === 255 ? mantissa ? NaN : sign * Infinity : exponent === 0 ? sign * 1401298464324817e-60 * mantissa : sign * Math.pow(2, exponent - 150) * (mantissa + 8388608);
        }
        exports.readFloatLE = readFloat_ieee754.bind(null, readUintLE);
        exports.readFloatBE = readFloat_ieee754.bind(null, readUintBE);
      })();
      if (typeof Float64Array !== "undefined") (function() {
        var f64 = new Float64Array([-0]), f8b = new Uint8Array(f64.buffer), le = f8b[7] === 128;
        function writeDouble_f64_cpy(val, buf, pos) {
          f64[0] = val;
          buf[pos] = f8b[0];
          buf[pos + 1] = f8b[1];
          buf[pos + 2] = f8b[2];
          buf[pos + 3] = f8b[3];
          buf[pos + 4] = f8b[4];
          buf[pos + 5] = f8b[5];
          buf[pos + 6] = f8b[6];
          buf[pos + 7] = f8b[7];
        }
        function writeDouble_f64_rev(val, buf, pos) {
          f64[0] = val;
          buf[pos] = f8b[7];
          buf[pos + 1] = f8b[6];
          buf[pos + 2] = f8b[5];
          buf[pos + 3] = f8b[4];
          buf[pos + 4] = f8b[3];
          buf[pos + 5] = f8b[2];
          buf[pos + 6] = f8b[1];
          buf[pos + 7] = f8b[0];
        }
        exports.writeDoubleLE = le ? writeDouble_f64_cpy : writeDouble_f64_rev;
        exports.writeDoubleBE = le ? writeDouble_f64_rev : writeDouble_f64_cpy;
        function readDouble_f64_cpy(buf, pos) {
          f8b[0] = buf[pos];
          f8b[1] = buf[pos + 1];
          f8b[2] = buf[pos + 2];
          f8b[3] = buf[pos + 3];
          f8b[4] = buf[pos + 4];
          f8b[5] = buf[pos + 5];
          f8b[6] = buf[pos + 6];
          f8b[7] = buf[pos + 7];
          return f64[0];
        }
        function readDouble_f64_rev(buf, pos) {
          f8b[7] = buf[pos];
          f8b[6] = buf[pos + 1];
          f8b[5] = buf[pos + 2];
          f8b[4] = buf[pos + 3];
          f8b[3] = buf[pos + 4];
          f8b[2] = buf[pos + 5];
          f8b[1] = buf[pos + 6];
          f8b[0] = buf[pos + 7];
          return f64[0];
        }
        exports.readDoubleLE = le ? readDouble_f64_cpy : readDouble_f64_rev;
        exports.readDoubleBE = le ? readDouble_f64_rev : readDouble_f64_cpy;
      })();
      else (function() {
        function writeDouble_ieee754(writeUint, off0, off1, val, buf, pos) {
          var sign = val < 0 ? 1 : 0;
          if (sign)
            val = -val;
          if (val === 0) {
            writeUint(0, buf, pos + off0);
            writeUint(1 / val > 0 ? (
              /* positive */
              0
            ) : (
              /* negative 0 */
              2147483648
            ), buf, pos + off1);
          } else if (isNaN(val)) {
            writeUint(0, buf, pos + off0);
            writeUint(2146959360, buf, pos + off1);
          } else if (val > 17976931348623157e292) {
            writeUint(0, buf, pos + off0);
            writeUint((sign << 31 | 2146435072) >>> 0, buf, pos + off1);
          } else {
            var mantissa;
            if (val < 22250738585072014e-324) {
              mantissa = val / 5e-324;
              writeUint(mantissa >>> 0, buf, pos + off0);
              writeUint((sign << 31 | mantissa / 4294967296) >>> 0, buf, pos + off1);
            } else {
              var exponent = Math.floor(Math.log(val) / Math.LN2);
              if (exponent === 1024)
                exponent = 1023;
              mantissa = val * Math.pow(2, -exponent);
              writeUint(mantissa * 4503599627370496 >>> 0, buf, pos + off0);
              writeUint((sign << 31 | exponent + 1023 << 20 | mantissa * 1048576 & 1048575) >>> 0, buf, pos + off1);
            }
          }
        }
        exports.writeDoubleLE = writeDouble_ieee754.bind(null, writeUintLE, 0, 4);
        exports.writeDoubleBE = writeDouble_ieee754.bind(null, writeUintBE, 4, 0);
        function readDouble_ieee754(readUint, off0, off1, buf, pos) {
          var lo = readUint(buf, pos + off0), hi = readUint(buf, pos + off1);
          var sign = (hi >> 31) * 2 + 1, exponent = hi >>> 20 & 2047, mantissa = 4294967296 * (hi & 1048575) + lo;
          return exponent === 2047 ? mantissa ? NaN : sign * Infinity : exponent === 0 ? sign * 5e-324 * mantissa : sign * Math.pow(2, exponent - 1075) * (mantissa + 4503599627370496);
        }
        exports.readDoubleLE = readDouble_ieee754.bind(null, readUintLE, 0, 4);
        exports.readDoubleBE = readDouble_ieee754.bind(null, readUintBE, 4, 0);
      })();
      return exports;
    }
    function writeUintLE(val, buf, pos) {
      buf[pos] = val & 255;
      buf[pos + 1] = val >>> 8 & 255;
      buf[pos + 2] = val >>> 16 & 255;
      buf[pos + 3] = val >>> 24;
    }
    function writeUintBE(val, buf, pos) {
      buf[pos] = val >>> 24;
      buf[pos + 1] = val >>> 16 & 255;
      buf[pos + 2] = val >>> 8 & 255;
      buf[pos + 3] = val & 255;
    }
    function readUintLE(buf, pos) {
      return (buf[pos] | buf[pos + 1] << 8 | buf[pos + 2] << 16 | buf[pos + 3] << 24) >>> 0;
    }
    function readUintBE(buf, pos) {
      return (buf[pos] << 24 | buf[pos + 1] << 16 | buf[pos + 2] << 8 | buf[pos + 3]) >>> 0;
    }
    return float;
  }
  var inquire_1;
  var hasRequiredInquire;
  function requireInquire() {
    if (hasRequiredInquire) return inquire_1;
    hasRequiredInquire = 1;
    inquire_1 = inquire;
    function inquire(moduleName) {
      try {
        var mod = eval("quire".replace(/^/, "re"))(moduleName);
        if (mod && (mod.length || Object.keys(mod).length))
          return mod;
      } catch (e) {
      }
      return null;
    }
    return inquire_1;
  }
  var utf8 = {};
  var hasRequiredUtf8;
  function requireUtf8() {
    if (hasRequiredUtf8) return utf8;
    hasRequiredUtf8 = 1;
    (function(exports) {
      var utf82 = exports;
      utf82.length = function utf8_length(string) {
        var len = 0, c = 0;
        for (var i = 0; i < string.length; ++i) {
          c = string.charCodeAt(i);
          if (c < 128)
            len += 1;
          else if (c < 2048)
            len += 2;
          else if ((c & 64512) === 55296 && (string.charCodeAt(i + 1) & 64512) === 56320) {
            ++i;
            len += 4;
          } else
            len += 3;
        }
        return len;
      };
      utf82.read = function utf8_read(buffer, start, end) {
        var len = end - start;
        if (len < 1)
          return "";
        var parts = null, chunk = [], i = 0, t;
        while (start < end) {
          t = buffer[start++];
          if (t < 128)
            chunk[i++] = t;
          else if (t > 191 && t < 224)
            chunk[i++] = (t & 31) << 6 | buffer[start++] & 63;
          else if (t > 239 && t < 365) {
            t = ((t & 7) << 18 | (buffer[start++] & 63) << 12 | (buffer[start++] & 63) << 6 | buffer[start++] & 63) - 65536;
            chunk[i++] = 55296 + (t >> 10);
            chunk[i++] = 56320 + (t & 1023);
          } else
            chunk[i++] = (t & 15) << 12 | (buffer[start++] & 63) << 6 | buffer[start++] & 63;
          if (i > 8191) {
            (parts || (parts = [])).push(String.fromCharCode.apply(String, chunk));
            i = 0;
          }
        }
        if (parts) {
          if (i)
            parts.push(String.fromCharCode.apply(String, chunk.slice(0, i)));
          return parts.join("");
        }
        return String.fromCharCode.apply(String, chunk.slice(0, i));
      };
      utf82.write = function utf8_write(string, buffer, offset) {
        var start = offset, c1, c2;
        for (var i = 0; i < string.length; ++i) {
          c1 = string.charCodeAt(i);
          if (c1 < 128) {
            buffer[offset++] = c1;
          } else if (c1 < 2048) {
            buffer[offset++] = c1 >> 6 | 192;
            buffer[offset++] = c1 & 63 | 128;
          } else if ((c1 & 64512) === 55296 && ((c2 = string.charCodeAt(i + 1)) & 64512) === 56320) {
            c1 = 65536 + ((c1 & 1023) << 10) + (c2 & 1023);
            ++i;
            buffer[offset++] = c1 >> 18 | 240;
            buffer[offset++] = c1 >> 12 & 63 | 128;
            buffer[offset++] = c1 >> 6 & 63 | 128;
            buffer[offset++] = c1 & 63 | 128;
          } else {
            buffer[offset++] = c1 >> 12 | 224;
            buffer[offset++] = c1 >> 6 & 63 | 128;
            buffer[offset++] = c1 & 63 | 128;
          }
        }
        return offset - start;
      };
    })(utf8);
    return utf8;
  }
  var pool_1;
  var hasRequiredPool;
  function requirePool() {
    if (hasRequiredPool) return pool_1;
    hasRequiredPool = 1;
    pool_1 = pool;
    function pool(alloc, slice, size) {
      var SIZE = size || 8192;
      var MAX = SIZE >>> 1;
      var slab = null;
      var offset = SIZE;
      return function pool_alloc(size2) {
        if (size2 < 1 || size2 > MAX)
          return alloc(size2);
        if (offset + size2 > SIZE) {
          slab = alloc(SIZE);
          offset = 0;
        }
        var buf = slice.call(slab, offset, offset += size2);
        if (offset & 7)
          offset = (offset | 7) + 1;
        return buf;
      };
    }
    return pool_1;
  }
  var longbits;
  var hasRequiredLongbits;
  function requireLongbits() {
    if (hasRequiredLongbits) return longbits;
    hasRequiredLongbits = 1;
    longbits = LongBits;
    var util = requireMinimal$1();
    function LongBits(lo, hi) {
      this.lo = lo >>> 0;
      this.hi = hi >>> 0;
    }
    var zero = LongBits.zero = new LongBits(0, 0);
    zero.toNumber = function() {
      return 0;
    };
    zero.zzEncode = zero.zzDecode = function() {
      return this;
    };
    zero.length = function() {
      return 1;
    };
    var zeroHash = LongBits.zeroHash = "\0\0\0\0\0\0\0\0";
    LongBits.fromNumber = function fromNumber2(value) {
      if (value === 0)
        return zero;
      var sign = value < 0;
      if (sign)
        value = -value;
      var lo = value >>> 0, hi = (value - lo) / 4294967296 >>> 0;
      if (sign) {
        hi = ~hi >>> 0;
        lo = ~lo >>> 0;
        if (++lo > 4294967295) {
          lo = 0;
          if (++hi > 4294967295)
            hi = 0;
        }
      }
      return new LongBits(lo, hi);
    };
    LongBits.from = function from(value) {
      if (typeof value === "number")
        return LongBits.fromNumber(value);
      if (util.isString(value)) {
        if (util.Long)
          value = util.Long.fromString(value);
        else
          return LongBits.fromNumber(parseInt(value, 10));
      }
      return value.low || value.high ? new LongBits(value.low >>> 0, value.high >>> 0) : zero;
    };
    LongBits.prototype.toNumber = function toNumber(unsigned) {
      if (!unsigned && this.hi >>> 31) {
        var lo = ~this.lo + 1 >>> 0, hi = ~this.hi >>> 0;
        if (!lo)
          hi = hi + 1 >>> 0;
        return -(lo + hi * 4294967296);
      }
      return this.lo + this.hi * 4294967296;
    };
    LongBits.prototype.toLong = function toLong(unsigned) {
      return util.Long ? new util.Long(this.lo | 0, this.hi | 0, Boolean(unsigned)) : { low: this.lo | 0, high: this.hi | 0, unsigned: Boolean(unsigned) };
    };
    var charCodeAt = String.prototype.charCodeAt;
    LongBits.fromHash = function fromHash(hash) {
      if (hash === zeroHash)
        return zero;
      return new LongBits(
        (charCodeAt.call(hash, 0) | charCodeAt.call(hash, 1) << 8 | charCodeAt.call(hash, 2) << 16 | charCodeAt.call(hash, 3) << 24) >>> 0,
        (charCodeAt.call(hash, 4) | charCodeAt.call(hash, 5) << 8 | charCodeAt.call(hash, 6) << 16 | charCodeAt.call(hash, 7) << 24) >>> 0
      );
    };
    LongBits.prototype.toHash = function toHash() {
      return String.fromCharCode(
        this.lo & 255,
        this.lo >>> 8 & 255,
        this.lo >>> 16 & 255,
        this.lo >>> 24,
        this.hi & 255,
        this.hi >>> 8 & 255,
        this.hi >>> 16 & 255,
        this.hi >>> 24
      );
    };
    LongBits.prototype.zzEncode = function zzEncode() {
      var mask = this.hi >> 31;
      this.hi = ((this.hi << 1 | this.lo >>> 31) ^ mask) >>> 0;
      this.lo = (this.lo << 1 ^ mask) >>> 0;
      return this;
    };
    LongBits.prototype.zzDecode = function zzDecode() {
      var mask = -(this.lo & 1);
      this.lo = ((this.lo >>> 1 | this.hi << 31) ^ mask) >>> 0;
      this.hi = (this.hi >>> 1 ^ mask) >>> 0;
      return this;
    };
    LongBits.prototype.length = function length() {
      var part0 = this.lo, part1 = (this.lo >>> 28 | this.hi << 4) >>> 0, part2 = this.hi >>> 24;
      return part2 === 0 ? part1 === 0 ? part0 < 16384 ? part0 < 128 ? 1 : 2 : part0 < 2097152 ? 3 : 4 : part1 < 16384 ? part1 < 128 ? 5 : 6 : part1 < 2097152 ? 7 : 8 : part2 < 128 ? 9 : 10;
    };
    return longbits;
  }
  var hasRequiredMinimal$1;
  function requireMinimal$1() {
    if (hasRequiredMinimal$1) return minimal$1;
    hasRequiredMinimal$1 = 1;
    (function(exports) {
      var util = exports;
      util.asPromise = requireAspromise();
      util.base64 = requireBase64();
      util.EventEmitter = requireEventemitter();
      util.float = requireFloat();
      util.inquire = requireInquire();
      util.utf8 = requireUtf8();
      util.pool = requirePool();
      util.LongBits = requireLongbits();
      util.isNode = Boolean(typeof commonjsGlobal !== "undefined" && commonjsGlobal && commonjsGlobal.process && commonjsGlobal.process.versions && commonjsGlobal.process.versions.node);
      util.global = util.isNode && commonjsGlobal || typeof window !== "undefined" && window || typeof self !== "undefined" && self || minimal$1;
      util.emptyArray = Object.freeze ? Object.freeze([]) : (
        /* istanbul ignore next */
        []
      );
      util.emptyObject = Object.freeze ? Object.freeze({}) : (
        /* istanbul ignore next */
        {}
      );
      util.isInteger = Number.isInteger || /* istanbul ignore next */
      function isInteger(value) {
        return typeof value === "number" && isFinite(value) && Math.floor(value) === value;
      };
      util.isString = function isString(value) {
        return typeof value === "string" || value instanceof String;
      };
      util.isObject = function isObject2(value) {
        return value && typeof value === "object";
      };
      util.isset = /**
       * Checks if a property on a message is considered to be present.
       * @param {Object} obj Plain object or message instance
       * @param {string} prop Property name
       * @returns {boolean} `true` if considered to be present, otherwise `false`
       */
      util.isSet = function isSet2(obj, prop) {
        var value = obj[prop];
        if (value != null && obj.hasOwnProperty(prop))
          return typeof value !== "object" || (Array.isArray(value) ? value.length : Object.keys(value).length) > 0;
        return false;
      };
      util.Buffer = (function() {
        try {
          var Buffer = util.inquire("buffer").Buffer;
          return Buffer.prototype.utf8Write ? Buffer : (
            /* istanbul ignore next */
            null
          );
        } catch (e) {
          return null;
        }
      })();
      util._Buffer_from = null;
      util._Buffer_allocUnsafe = null;
      util.newBuffer = function newBuffer(sizeOrArray) {
        return typeof sizeOrArray === "number" ? util.Buffer ? util._Buffer_allocUnsafe(sizeOrArray) : new util.Array(sizeOrArray) : util.Buffer ? util._Buffer_from(sizeOrArray) : typeof Uint8Array === "undefined" ? sizeOrArray : new Uint8Array(sizeOrArray);
      };
      util.Array = typeof Uint8Array !== "undefined" ? Uint8Array : Array;
      util.Long = /* istanbul ignore next */
      util.global.dcodeIO && /* istanbul ignore next */
      util.global.dcodeIO.Long || /* istanbul ignore next */
      util.global.Long || util.inquire("long");
      util.key2Re = /^true|false|0|1$/;
      util.key32Re = /^-?(?:0|[1-9][0-9]*)$/;
      util.key64Re = /^(?:[\\x00-\\xff]{8}|-?(?:0|[1-9][0-9]*))$/;
      util.longToHash = function longToHash(value) {
        return value ? util.LongBits.from(value).toHash() : util.LongBits.zeroHash;
      };
      util.longFromHash = function longFromHash(hash, unsigned) {
        var bits = util.LongBits.fromHash(hash);
        if (util.Long)
          return util.Long.fromBits(bits.lo, bits.hi, unsigned);
        return bits.toNumber(Boolean(unsigned));
      };
      function merge(dst, src, ifNotSet) {
        for (var keys = Object.keys(src), i = 0; i < keys.length; ++i)
          if (dst[keys[i]] === void 0 || !ifNotSet)
            dst[keys[i]] = src[keys[i]];
        return dst;
      }
      util.merge = merge;
      util.lcFirst = function lcFirst(str) {
        return str.charAt(0).toLowerCase() + str.substring(1);
      };
      function newError(name) {
        function CustomError(message, properties) {
          if (!(this instanceof CustomError))
            return new CustomError(message, properties);
          Object.defineProperty(this, "message", { get: function() {
            return message;
          } });
          if (Error.captureStackTrace)
            Error.captureStackTrace(this, CustomError);
          else
            Object.defineProperty(this, "stack", { value: new Error().stack || "" });
          if (properties)
            merge(this, properties);
        }
        CustomError.prototype = Object.create(Error.prototype, {
          constructor: {
            value: CustomError,
            writable: true,
            enumerable: false,
            configurable: true
          },
          name: {
            get: function get() {
              return name;
            },
            set: void 0,
            enumerable: false,
            // configurable: false would accurately preserve the behavior of
            // the original, but I'm guessing that was not intentional.
            // For an actual error subclass, this property would
            // be configurable.
            configurable: true
          },
          toString: {
            value: function value() {
              return this.name + ": " + this.message;
            },
            writable: true,
            enumerable: false,
            configurable: true
          }
        });
        return CustomError;
      }
      util.newError = newError;
      util.ProtocolError = newError("ProtocolError");
      util.oneOfGetter = function getOneOf(fieldNames) {
        var fieldMap = {};
        for (var i = 0; i < fieldNames.length; ++i)
          fieldMap[fieldNames[i]] = 1;
        return function() {
          for (var keys = Object.keys(this), i2 = keys.length - 1; i2 > -1; --i2)
            if (fieldMap[keys[i2]] === 1 && this[keys[i2]] !== void 0 && this[keys[i2]] !== null)
              return keys[i2];
        };
      };
      util.oneOfSetter = function setOneOf(fieldNames) {
        return function(name) {
          for (var i = 0; i < fieldNames.length; ++i)
            if (fieldNames[i] !== name)
              delete this[fieldNames[i]];
        };
      };
      util.toJSONOptions = {
        longs: String,
        enums: String,
        bytes: String,
        json: true
      };
      util._configure = function() {
        var Buffer = util.Buffer;
        if (!Buffer) {
          util._Buffer_from = util._Buffer_allocUnsafe = null;
          return;
        }
        util._Buffer_from = Buffer.from !== Uint8Array.from && Buffer.from || /* istanbul ignore next */
        function Buffer_from(value, encoding) {
          return new Buffer(value, encoding);
        };
        util._Buffer_allocUnsafe = Buffer.allocUnsafe || /* istanbul ignore next */
        function Buffer_allocUnsafe(size) {
          return new Buffer(size);
        };
      };
    })(minimal$1);
    return minimal$1;
  }
  var writer;
  var hasRequiredWriter;
  function requireWriter() {
    if (hasRequiredWriter) return writer;
    hasRequiredWriter = 1;
    writer = Writer;
    var util = requireMinimal$1();
    var BufferWriter;
    var LongBits = util.LongBits, base642 = util.base64, utf82 = util.utf8;
    function Op(fn, len, val) {
      this.fn = fn;
      this.len = len;
      this.next = void 0;
      this.val = val;
    }
    function noop() {
    }
    function State(writer2) {
      this.head = writer2.head;
      this.tail = writer2.tail;
      this.len = writer2.len;
      this.next = writer2.states;
    }
    function Writer() {
      this.len = 0;
      this.head = new Op(noop, 0, 0);
      this.tail = this.head;
      this.states = null;
    }
    var create = function create2() {
      return util.Buffer ? function create_buffer_setup() {
        return (Writer.create = function create_buffer() {
          return new BufferWriter();
        })();
      } : function create_array() {
        return new Writer();
      };
    };
    Writer.create = create();
    Writer.alloc = function alloc(size) {
      return new util.Array(size);
    };
    if (util.Array !== Array)
      Writer.alloc = util.pool(Writer.alloc, util.Array.prototype.subarray);
    Writer.prototype._push = function push(fn, len, val) {
      this.tail = this.tail.next = new Op(fn, len, val);
      this.len += len;
      return this;
    };
    function writeByte(val, buf, pos) {
      buf[pos] = val & 255;
    }
    function writeVarint32(val, buf, pos) {
      while (val > 127) {
        buf[pos++] = val & 127 | 128;
        val >>>= 7;
      }
      buf[pos] = val;
    }
    function VarintOp(len, val) {
      this.len = len;
      this.next = void 0;
      this.val = val;
    }
    VarintOp.prototype = Object.create(Op.prototype);
    VarintOp.prototype.fn = writeVarint32;
    Writer.prototype.uint32 = function write_uint32(value) {
      this.len += (this.tail = this.tail.next = new VarintOp(
        (value = value >>> 0) < 128 ? 1 : value < 16384 ? 2 : value < 2097152 ? 3 : value < 268435456 ? 4 : 5,
        value
      )).len;
      return this;
    };
    Writer.prototype.int32 = function write_int32(value) {
      return value < 0 ? this._push(writeVarint64, 10, LongBits.fromNumber(value)) : this.uint32(value);
    };
    Writer.prototype.sint32 = function write_sint32(value) {
      return this.uint32((value << 1 ^ value >> 31) >>> 0);
    };
    function writeVarint64(val, buf, pos) {
      while (val.hi) {
        buf[pos++] = val.lo & 127 | 128;
        val.lo = (val.lo >>> 7 | val.hi << 25) >>> 0;
        val.hi >>>= 7;
      }
      while (val.lo > 127) {
        buf[pos++] = val.lo & 127 | 128;
        val.lo = val.lo >>> 7;
      }
      buf[pos++] = val.lo;
    }
    Writer.prototype.uint64 = function write_uint64(value) {
      var bits = LongBits.from(value);
      return this._push(writeVarint64, bits.length(), bits);
    };
    Writer.prototype.int64 = Writer.prototype.uint64;
    Writer.prototype.sint64 = function write_sint64(value) {
      var bits = LongBits.from(value).zzEncode();
      return this._push(writeVarint64, bits.length(), bits);
    };
    Writer.prototype.bool = function write_bool(value) {
      return this._push(writeByte, 1, value ? 1 : 0);
    };
    function writeFixed32(val, buf, pos) {
      buf[pos] = val & 255;
      buf[pos + 1] = val >>> 8 & 255;
      buf[pos + 2] = val >>> 16 & 255;
      buf[pos + 3] = val >>> 24;
    }
    Writer.prototype.fixed32 = function write_fixed32(value) {
      return this._push(writeFixed32, 4, value >>> 0);
    };
    Writer.prototype.sfixed32 = Writer.prototype.fixed32;
    Writer.prototype.fixed64 = function write_fixed64(value) {
      var bits = LongBits.from(value);
      return this._push(writeFixed32, 4, bits.lo)._push(writeFixed32, 4, bits.hi);
    };
    Writer.prototype.sfixed64 = Writer.prototype.fixed64;
    Writer.prototype.float = function write_float(value) {
      return this._push(util.float.writeFloatLE, 4, value);
    };
    Writer.prototype.double = function write_double(value) {
      return this._push(util.float.writeDoubleLE, 8, value);
    };
    var writeBytes = util.Array.prototype.set ? function writeBytes_set(val, buf, pos) {
      buf.set(val, pos);
    } : function writeBytes_for(val, buf, pos) {
      for (var i = 0; i < val.length; ++i)
        buf[pos + i] = val[i];
    };
    Writer.prototype.bytes = function write_bytes(value) {
      var len = value.length >>> 0;
      if (!len)
        return this._push(writeByte, 1, 0);
      if (util.isString(value)) {
        var buf = Writer.alloc(len = base642.length(value));
        base642.decode(value, buf, 0);
        value = buf;
      }
      return this.uint32(len)._push(writeBytes, len, value);
    };
    Writer.prototype.string = function write_string(value) {
      var len = utf82.length(value);
      return len ? this.uint32(len)._push(utf82.write, len, value) : this._push(writeByte, 1, 0);
    };
    Writer.prototype.fork = function fork() {
      this.states = new State(this);
      this.head = this.tail = new Op(noop, 0, 0);
      this.len = 0;
      return this;
    };
    Writer.prototype.reset = function reset() {
      if (this.states) {
        this.head = this.states.head;
        this.tail = this.states.tail;
        this.len = this.states.len;
        this.states = this.states.next;
      } else {
        this.head = this.tail = new Op(noop, 0, 0);
        this.len = 0;
      }
      return this;
    };
    Writer.prototype.ldelim = function ldelim() {
      var head = this.head, tail = this.tail, len = this.len;
      this.reset().uint32(len);
      if (len) {
        this.tail.next = head.next;
        this.tail = tail;
        this.len += len;
      }
      return this;
    };
    Writer.prototype.finish = function finish() {
      var head = this.head.next, buf = this.constructor.alloc(this.len), pos = 0;
      while (head) {
        head.fn(head.val, buf, pos);
        pos += head.len;
        head = head.next;
      }
      return buf;
    };
    Writer._configure = function(BufferWriter_) {
      BufferWriter = BufferWriter_;
      Writer.create = create();
      BufferWriter._configure();
    };
    return writer;
  }
  var writer_buffer;
  var hasRequiredWriter_buffer;
  function requireWriter_buffer() {
    if (hasRequiredWriter_buffer) return writer_buffer;
    hasRequiredWriter_buffer = 1;
    writer_buffer = BufferWriter;
    var Writer = requireWriter();
    (BufferWriter.prototype = Object.create(Writer.prototype)).constructor = BufferWriter;
    var util = requireMinimal$1();
    function BufferWriter() {
      Writer.call(this);
    }
    BufferWriter._configure = function() {
      BufferWriter.alloc = util._Buffer_allocUnsafe;
      BufferWriter.writeBytesBuffer = util.Buffer && util.Buffer.prototype instanceof Uint8Array && util.Buffer.prototype.set.name === "set" ? function writeBytesBuffer_set(val, buf, pos) {
        buf.set(val, pos);
      } : function writeBytesBuffer_copy(val, buf, pos) {
        if (val.copy)
          val.copy(buf, pos, 0, val.length);
        else for (var i = 0; i < val.length; )
          buf[pos++] = val[i++];
      };
    };
    BufferWriter.prototype.bytes = function write_bytes_buffer(value) {
      if (util.isString(value))
        value = util._Buffer_from(value, "base64");
      var len = value.length >>> 0;
      this.uint32(len);
      if (len)
        this._push(BufferWriter.writeBytesBuffer, len, value);
      return this;
    };
    function writeStringBuffer(val, buf, pos) {
      if (val.length < 40)
        util.utf8.write(val, buf, pos);
      else if (buf.utf8Write)
        buf.utf8Write(val, pos);
      else
        buf.write(val, pos);
    }
    BufferWriter.prototype.string = function write_string_buffer(value) {
      var len = util.Buffer.byteLength(value);
      this.uint32(len);
      if (len)
        this._push(writeStringBuffer, len, value);
      return this;
    };
    BufferWriter._configure();
    return writer_buffer;
  }
  var reader;
  var hasRequiredReader;
  function requireReader() {
    if (hasRequiredReader) return reader;
    hasRequiredReader = 1;
    reader = Reader;
    var util = requireMinimal$1();
    var BufferReader;
    var LongBits = util.LongBits, utf82 = util.utf8;
    function indexOutOfRange(reader2, writeLength) {
      return RangeError("index out of range: " + reader2.pos + " + " + (writeLength || 1) + " > " + reader2.len);
    }
    function Reader(buffer) {
      this.buf = buffer;
      this.pos = 0;
      this.len = buffer.length;
    }
    var create_array = typeof Uint8Array !== "undefined" ? function create_typed_array(buffer) {
      if (buffer instanceof Uint8Array || Array.isArray(buffer))
        return new Reader(buffer);
      throw Error("illegal buffer");
    } : function create_array2(buffer) {
      if (Array.isArray(buffer))
        return new Reader(buffer);
      throw Error("illegal buffer");
    };
    var create = function create2() {
      return util.Buffer ? function create_buffer_setup(buffer) {
        return (Reader.create = function create_buffer(buffer2) {
          return util.Buffer.isBuffer(buffer2) ? new BufferReader(buffer2) : create_array(buffer2);
        })(buffer);
      } : create_array;
    };
    Reader.create = create();
    Reader.prototype._slice = util.Array.prototype.subarray || /* istanbul ignore next */
    util.Array.prototype.slice;
    Reader.prototype.uint32 = /* @__PURE__ */ (function read_uint32_setup() {
      var value = 4294967295;
      return function read_uint32() {
        value = (this.buf[this.pos] & 127) >>> 0;
        if (this.buf[this.pos++] < 128) return value;
        value = (value | (this.buf[this.pos] & 127) << 7) >>> 0;
        if (this.buf[this.pos++] < 128) return value;
        value = (value | (this.buf[this.pos] & 127) << 14) >>> 0;
        if (this.buf[this.pos++] < 128) return value;
        value = (value | (this.buf[this.pos] & 127) << 21) >>> 0;
        if (this.buf[this.pos++] < 128) return value;
        value = (value | (this.buf[this.pos] & 15) << 28) >>> 0;
        if (this.buf[this.pos++] < 128) return value;
        if ((this.pos += 5) > this.len) {
          this.pos = this.len;
          throw indexOutOfRange(this, 10);
        }
        return value;
      };
    })();
    Reader.prototype.int32 = function read_int32() {
      return this.uint32() | 0;
    };
    Reader.prototype.sint32 = function read_sint32() {
      var value = this.uint32();
      return value >>> 1 ^ -(value & 1) | 0;
    };
    function readLongVarint() {
      var bits = new LongBits(0, 0);
      var i = 0;
      if (this.len - this.pos > 4) {
        for (; i < 4; ++i) {
          bits.lo = (bits.lo | (this.buf[this.pos] & 127) << i * 7) >>> 0;
          if (this.buf[this.pos++] < 128)
            return bits;
        }
        bits.lo = (bits.lo | (this.buf[this.pos] & 127) << 28) >>> 0;
        bits.hi = (bits.hi | (this.buf[this.pos] & 127) >> 4) >>> 0;
        if (this.buf[this.pos++] < 128)
          return bits;
        i = 0;
      } else {
        for (; i < 3; ++i) {
          if (this.pos >= this.len)
            throw indexOutOfRange(this);
          bits.lo = (bits.lo | (this.buf[this.pos] & 127) << i * 7) >>> 0;
          if (this.buf[this.pos++] < 128)
            return bits;
        }
        bits.lo = (bits.lo | (this.buf[this.pos++] & 127) << i * 7) >>> 0;
        return bits;
      }
      if (this.len - this.pos > 4) {
        for (; i < 5; ++i) {
          bits.hi = (bits.hi | (this.buf[this.pos] & 127) << i * 7 + 3) >>> 0;
          if (this.buf[this.pos++] < 128)
            return bits;
        }
      } else {
        for (; i < 5; ++i) {
          if (this.pos >= this.len)
            throw indexOutOfRange(this);
          bits.hi = (bits.hi | (this.buf[this.pos] & 127) << i * 7 + 3) >>> 0;
          if (this.buf[this.pos++] < 128)
            return bits;
        }
      }
      throw Error("invalid varint encoding");
    }
    Reader.prototype.bool = function read_bool() {
      return this.uint32() !== 0;
    };
    function readFixed32_end(buf, end) {
      return (buf[end - 4] | buf[end - 3] << 8 | buf[end - 2] << 16 | buf[end - 1] << 24) >>> 0;
    }
    Reader.prototype.fixed32 = function read_fixed32() {
      if (this.pos + 4 > this.len)
        throw indexOutOfRange(this, 4);
      return readFixed32_end(this.buf, this.pos += 4);
    };
    Reader.prototype.sfixed32 = function read_sfixed32() {
      if (this.pos + 4 > this.len)
        throw indexOutOfRange(this, 4);
      return readFixed32_end(this.buf, this.pos += 4) | 0;
    };
    function readFixed64() {
      if (this.pos + 8 > this.len)
        throw indexOutOfRange(this, 8);
      return new LongBits(readFixed32_end(this.buf, this.pos += 4), readFixed32_end(this.buf, this.pos += 4));
    }
    Reader.prototype.float = function read_float() {
      if (this.pos + 4 > this.len)
        throw indexOutOfRange(this, 4);
      var value = util.float.readFloatLE(this.buf, this.pos);
      this.pos += 4;
      return value;
    };
    Reader.prototype.double = function read_double() {
      if (this.pos + 8 > this.len)
        throw indexOutOfRange(this, 4);
      var value = util.float.readDoubleLE(this.buf, this.pos);
      this.pos += 8;
      return value;
    };
    Reader.prototype.bytes = function read_bytes() {
      var length = this.uint32(), start = this.pos, end = this.pos + length;
      if (end > this.len)
        throw indexOutOfRange(this, length);
      this.pos += length;
      if (Array.isArray(this.buf))
        return this.buf.slice(start, end);
      if (start === end) {
        var nativeBuffer = util.Buffer;
        return nativeBuffer ? nativeBuffer.alloc(0) : new this.buf.constructor(0);
      }
      return this._slice.call(this.buf, start, end);
    };
    Reader.prototype.string = function read_string() {
      var bytes = this.bytes();
      return utf82.read(bytes, 0, bytes.length);
    };
    Reader.prototype.skip = function skip(length) {
      if (typeof length === "number") {
        if (this.pos + length > this.len)
          throw indexOutOfRange(this, length);
        this.pos += length;
      } else {
        do {
          if (this.pos >= this.len)
            throw indexOutOfRange(this);
        } while (this.buf[this.pos++] & 128);
      }
      return this;
    };
    Reader.prototype.skipType = function(wireType) {
      switch (wireType) {
        case 0:
          this.skip();
          break;
        case 1:
          this.skip(8);
          break;
        case 2:
          this.skip(this.uint32());
          break;
        case 3:
          while ((wireType = this.uint32() & 7) !== 4) {
            this.skipType(wireType);
          }
          break;
        case 5:
          this.skip(4);
          break;
        /* istanbul ignore next */
        default:
          throw Error("invalid wire type " + wireType + " at offset " + this.pos);
      }
      return this;
    };
    Reader._configure = function(BufferReader_) {
      BufferReader = BufferReader_;
      Reader.create = create();
      BufferReader._configure();
      var fn = util.Long ? "toLong" : (
        /* istanbul ignore next */
        "toNumber"
      );
      util.merge(Reader.prototype, {
        int64: function read_int64() {
          return readLongVarint.call(this)[fn](false);
        },
        uint64: function read_uint64() {
          return readLongVarint.call(this)[fn](true);
        },
        sint64: function read_sint64() {
          return readLongVarint.call(this).zzDecode()[fn](false);
        },
        fixed64: function read_fixed64() {
          return readFixed64.call(this)[fn](true);
        },
        sfixed64: function read_sfixed64() {
          return readFixed64.call(this)[fn](false);
        }
      });
    };
    return reader;
  }
  var reader_buffer;
  var hasRequiredReader_buffer;
  function requireReader_buffer() {
    if (hasRequiredReader_buffer) return reader_buffer;
    hasRequiredReader_buffer = 1;
    reader_buffer = BufferReader;
    var Reader = requireReader();
    (BufferReader.prototype = Object.create(Reader.prototype)).constructor = BufferReader;
    var util = requireMinimal$1();
    function BufferReader(buffer) {
      Reader.call(this, buffer);
    }
    BufferReader._configure = function() {
      if (util.Buffer)
        BufferReader.prototype._slice = util.Buffer.prototype.slice;
    };
    BufferReader.prototype.string = function read_string_buffer() {
      var len = this.uint32();
      return this.buf.utf8Slice ? this.buf.utf8Slice(this.pos, this.pos = Math.min(this.pos + len, this.len)) : this.buf.toString("utf-8", this.pos, this.pos = Math.min(this.pos + len, this.len));
    };
    BufferReader._configure();
    return reader_buffer;
  }
  var rpc = {};
  var service;
  var hasRequiredService;
  function requireService() {
    if (hasRequiredService) return service;
    hasRequiredService = 1;
    service = Service;
    var util = requireMinimal$1();
    (Service.prototype = Object.create(util.EventEmitter.prototype)).constructor = Service;
    function Service(rpcImpl, requestDelimited, responseDelimited) {
      if (typeof rpcImpl !== "function")
        throw TypeError("rpcImpl must be a function");
      util.EventEmitter.call(this);
      this.rpcImpl = rpcImpl;
      this.requestDelimited = Boolean(requestDelimited);
      this.responseDelimited = Boolean(responseDelimited);
    }
    Service.prototype.rpcCall = function rpcCall(method, requestCtor, responseCtor, request, callback) {
      if (!request)
        throw TypeError("request must be specified");
      var self2 = this;
      if (!callback)
        return util.asPromise(rpcCall, self2, method, requestCtor, responseCtor, request);
      if (!self2.rpcImpl) {
        setTimeout(function() {
          callback(Error("already ended"));
        }, 0);
        return void 0;
      }
      try {
        return self2.rpcImpl(
          method,
          requestCtor[self2.requestDelimited ? "encodeDelimited" : "encode"](request).finish(),
          function rpcCallback(err, response) {
            if (err) {
              self2.emit("error", err, method);
              return callback(err);
            }
            if (response === null) {
              self2.end(
                /* endedByRPC */
                true
              );
              return void 0;
            }
            if (!(response instanceof responseCtor)) {
              try {
                response = responseCtor[self2.responseDelimited ? "decodeDelimited" : "decode"](response);
              } catch (err2) {
                self2.emit("error", err2, method);
                return callback(err2);
              }
            }
            self2.emit("data", response, method);
            return callback(null, response);
          }
        );
      } catch (err) {
        self2.emit("error", err, method);
        setTimeout(function() {
          callback(err);
        }, 0);
        return void 0;
      }
    };
    Service.prototype.end = function end(endedByRPC) {
      if (this.rpcImpl) {
        if (!endedByRPC)
          this.rpcImpl(null, null, null);
        this.rpcImpl = null;
        this.emit("end").off();
      }
      return this;
    };
    return service;
  }
  var hasRequiredRpc;
  function requireRpc() {
    if (hasRequiredRpc) return rpc;
    hasRequiredRpc = 1;
    (function(exports) {
      var rpc2 = exports;
      rpc2.Service = requireService();
    })(rpc);
    return rpc;
  }
  var roots;
  var hasRequiredRoots;
  function requireRoots() {
    if (hasRequiredRoots) return roots;
    hasRequiredRoots = 1;
    roots = {};
    return roots;
  }
  var hasRequiredIndexMinimal;
  function requireIndexMinimal() {
    if (hasRequiredIndexMinimal) return indexMinimal;
    hasRequiredIndexMinimal = 1;
    (function(exports) {
      var protobuf = exports;
      protobuf.build = "minimal";
      protobuf.Writer = requireWriter();
      protobuf.BufferWriter = requireWriter_buffer();
      protobuf.Reader = requireReader();
      protobuf.BufferReader = requireReader_buffer();
      protobuf.util = requireMinimal$1();
      protobuf.rpc = requireRpc();
      protobuf.roots = requireRoots();
      protobuf.configure = configure;
      function configure() {
        protobuf.util._configure();
        protobuf.Writer._configure(protobuf.BufferWriter);
        protobuf.Reader._configure(protobuf.BufferReader);
      }
      configure();
    })(indexMinimal);
    return indexMinimal;
  }
  var minimal;
  var hasRequiredMinimal;
  function requireMinimal() {
    if (hasRequiredMinimal) return minimal;
    hasRequiredMinimal = 1;
    minimal = requireIndexMinimal();
    return minimal;
  }
  var minimalExports = requireMinimal();
  const _m0 = /* @__PURE__ */ getDefaultExportFromCjs(minimalExports);
  const messageTypeRegistry = /* @__PURE__ */ new Map();
  var NullValue;
  (function(NullValue2) {
    NullValue2[NullValue2["NULL_VALUE"] = 0] = "NULL_VALUE";
    NullValue2[NullValue2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(NullValue || (NullValue = {}));
  function nullValueFromJSON(object) {
    switch (object) {
      case 0:
      case "NULL_VALUE":
        return NullValue.NULL_VALUE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return NullValue.UNRECOGNIZED;
    }
  }
  function nullValueToJSON(object) {
    switch (object) {
      case NullValue.NULL_VALUE:
        return 0;
      case NullValue.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseStruct() {
    return { fields: {} };
  }
  const Struct = {
    $type: "google.protobuf.Struct",
    encode(message, writer2 = _m0.Writer.create()) {
      Object.entries(message.fields).forEach(([key, value]) => {
        if (value !== void 0) {
          Struct_FieldsEntry.encode({ key, value }, writer2.uint32(10).fork()).ldelim();
        }
      });
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseStruct();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            const entry1 = Struct_FieldsEntry.decode(reader2, reader2.uint32());
            if (entry1.value !== void 0) {
              message.fields[entry1.key] = entry1.value;
            }
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        fields: isObject$9(object.fields) ? Object.entries(object.fields).reduce((acc, [key, value]) => {
          acc[key] = value;
          return acc;
        }, {}) : {}
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.fields) {
        const entries = Object.entries(message.fields);
        if (entries.length > 0) {
          obj.fields = {};
          entries.forEach(([k, v]) => {
            obj.fields[k] = v;
          });
        }
      }
      return obj;
    },
    create(base) {
      return Struct.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseStruct();
      message.fields = Object.entries(object.fields ?? {}).reduce((acc, [key, value]) => {
        if (value !== void 0) {
          acc[key] = value;
        }
        return acc;
      }, {});
      return message;
    },
    wrap(object) {
      const struct = createBaseStruct();
      if (object !== void 0) {
        for (const key of Object.keys(object)) {
          struct.fields[key] = object[key];
        }
      }
      return struct;
    },
    unwrap(message) {
      const object = {};
      if (message.fields) {
        for (const key of Object.keys(message.fields)) {
          object[key] = message.fields[key];
        }
      }
      return object;
    }
  };
  messageTypeRegistry.set(Struct.$type, Struct);
  function createBaseStruct_FieldsEntry() {
    return { key: "", value: void 0 };
  }
  const Struct_FieldsEntry = {
    $type: "google.protobuf.Struct.FieldsEntry",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.key !== "") {
        writer2.uint32(10).string(message.key);
      }
      if (message.value !== void 0) {
        Value.encode(Value.wrap(message.value), writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseStruct_FieldsEntry();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.key = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.value = Value.unwrap(Value.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        key: isSet$v(object.key) ? globalThis.String(object.key) : "",
        value: isSet$v(object?.value) ? object.value : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.key !== "") {
        obj.key = message.key;
      }
      if (message.value !== void 0) {
        obj.value = message.value;
      }
      return obj;
    },
    create(base) {
      return Struct_FieldsEntry.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseStruct_FieldsEntry();
      message.key = object.key ?? "";
      message.value = object.value ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(Struct_FieldsEntry.$type, Struct_FieldsEntry);
  function createBaseValue() {
    return {
      nullValue: void 0,
      numberValue: void 0,
      stringValue: void 0,
      boolValue: void 0,
      structValue: void 0,
      listValue: void 0
    };
  }
  const Value = {
    $type: "google.protobuf.Value",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.nullValue !== void 0) {
        writer2.uint32(8).int32(message.nullValue);
      }
      if (message.numberValue !== void 0) {
        writer2.uint32(17).double(message.numberValue);
      }
      if (message.stringValue !== void 0) {
        writer2.uint32(26).string(message.stringValue);
      }
      if (message.boolValue !== void 0) {
        writer2.uint32(32).bool(message.boolValue);
      }
      if (message.structValue !== void 0) {
        Struct.encode(Struct.wrap(message.structValue), writer2.uint32(42).fork()).ldelim();
      }
      if (message.listValue !== void 0) {
        ListValue.encode(ListValue.wrap(message.listValue), writer2.uint32(50).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.nullValue = reader2.int32();
            continue;
          case 2:
            if (tag !== 17) {
              break;
            }
            message.numberValue = reader2.double();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.stringValue = reader2.string();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.boolValue = reader2.bool();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.structValue = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.listValue = ListValue.unwrap(ListValue.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        nullValue: isSet$v(object.nullValue) ? nullValueFromJSON(object.nullValue) : void 0,
        numberValue: isSet$v(object.numberValue) ? globalThis.Number(object.numberValue) : void 0,
        stringValue: isSet$v(object.stringValue) ? globalThis.String(object.stringValue) : void 0,
        boolValue: isSet$v(object.boolValue) ? globalThis.Boolean(object.boolValue) : void 0,
        structValue: isObject$9(object.structValue) ? object.structValue : void 0,
        listValue: globalThis.Array.isArray(object.listValue) ? [...object.listValue] : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.nullValue !== void 0) {
        obj.nullValue = nullValueToJSON(message.nullValue);
      }
      if (message.numberValue !== void 0) {
        obj.numberValue = message.numberValue;
      }
      if (message.stringValue !== void 0) {
        obj.stringValue = message.stringValue;
      }
      if (message.boolValue !== void 0) {
        obj.boolValue = message.boolValue;
      }
      if (message.structValue !== void 0) {
        obj.structValue = message.structValue;
      }
      if (message.listValue !== void 0) {
        obj.listValue = message.listValue;
      }
      return obj;
    },
    create(base) {
      return Value.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseValue();
      message.nullValue = object.nullValue ?? void 0;
      message.numberValue = object.numberValue ?? void 0;
      message.stringValue = object.stringValue ?? void 0;
      message.boolValue = object.boolValue ?? void 0;
      message.structValue = object.structValue ?? void 0;
      message.listValue = object.listValue ?? void 0;
      return message;
    },
    wrap(value) {
      const result2 = createBaseValue();
      if (value === null) {
        result2.nullValue = NullValue.NULL_VALUE;
      } else if (typeof value === "boolean") {
        result2.boolValue = value;
      } else if (typeof value === "number") {
        result2.numberValue = value;
      } else if (typeof value === "string") {
        result2.stringValue = value;
      } else if (globalThis.Array.isArray(value)) {
        result2.listValue = value;
      } else if (typeof value === "object") {
        result2.structValue = value;
      } else if (typeof value !== "undefined") {
        throw new globalThis.Error("Unsupported any value type: " + typeof value);
      }
      return result2;
    },
    unwrap(message) {
      if (message.stringValue !== void 0) {
        return message.stringValue;
      } else if (message?.numberValue !== void 0) {
        return message.numberValue;
      } else if (message?.boolValue !== void 0) {
        return message.boolValue;
      } else if (message?.structValue !== void 0) {
        return message.structValue;
      } else if (message?.listValue !== void 0) {
        return message.listValue;
      } else if (message?.nullValue !== void 0) {
        return null;
      }
      return void 0;
    }
  };
  messageTypeRegistry.set(Value.$type, Value);
  function createBaseListValue() {
    return { values: [] };
  }
  const ListValue = {
    $type: "google.protobuf.ListValue",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.values) {
        Value.encode(Value.wrap(v), writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseListValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.values.push(Value.unwrap(Value.decode(reader2, reader2.uint32())));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { values: globalThis.Array.isArray(object?.values) ? [...object.values] : [] };
    },
    toJSON(message) {
      const obj = {};
      if (message.values?.length) {
        obj.values = message.values;
      }
      return obj;
    },
    create(base) {
      return ListValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseListValue();
      message.values = object.values?.map((e) => e) || [];
      return message;
    },
    wrap(array) {
      const result2 = createBaseListValue();
      result2.values = array ?? [];
      return result2;
    },
    unwrap(message) {
      if (message?.hasOwnProperty("values") && globalThis.Array.isArray(message.values)) {
        return message.values;
      } else {
        return message;
      }
    }
  };
  messageTypeRegistry.set(ListValue.$type, ListValue);
  function isObject$9(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$v(value) {
    return value !== null && value !== void 0;
  }
  function createBaseUIDimensions() {
    return { height: 0, width: 0, scale: 0, fontScale: void 0 };
  }
  const UIDimensions = {
    $type: "devvit.ui.common.v1alpha.UIDimensions",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.height !== 0) {
        writer2.uint32(8).int32(message.height);
      }
      if (message.width !== 0) {
        writer2.uint32(16).int32(message.width);
      }
      if (message.scale !== 0) {
        writer2.uint32(29).float(message.scale);
      }
      if (message.fontScale !== void 0) {
        writer2.uint32(37).float(message.fontScale);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUIDimensions();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.height = reader2.int32();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.width = reader2.int32();
            continue;
          case 3:
            if (tag !== 29) {
              break;
            }
            message.scale = reader2.float();
            continue;
          case 4:
            if (tag !== 37) {
              break;
            }
            message.fontScale = reader2.float();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        height: isSet$u(object.height) ? globalThis.Number(object.height) : 0,
        width: isSet$u(object.width) ? globalThis.Number(object.width) : 0,
        scale: isSet$u(object.scale) ? globalThis.Number(object.scale) : 0,
        fontScale: isSet$u(object.fontScale) ? globalThis.Number(object.fontScale) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.height !== 0) {
        obj.height = Math.round(message.height);
      }
      if (message.width !== 0) {
        obj.width = Math.round(message.width);
      }
      if (message.scale !== 0) {
        obj.scale = message.scale;
      }
      if (message.fontScale !== void 0) {
        obj.fontScale = message.fontScale;
      }
      return obj;
    },
    create(base) {
      return UIDimensions.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUIDimensions();
      message.height = object.height ?? 0;
      message.width = object.width ?? 0;
      message.scale = object.scale ?? 0;
      message.fontScale = object.fontScale ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(UIDimensions.$type, UIDimensions);
  function isSet$u(value) {
    return value !== null && value !== void 0;
  }
  function createBaseNavigateToUrlEffect() {
    return { url: "", target: void 0 };
  }
  const NavigateToUrlEffect = {
    $type: "devvit.ui.effect_types.v1alpha.NavigateToUrlEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.url !== "") {
        writer2.uint32(10).string(message.url);
      }
      if (message.target !== void 0) {
        writer2.uint32(18).string(message.target);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseNavigateToUrlEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.url = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.target = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        url: isSet$t(object.url) ? globalThis.String(object.url) : "",
        target: isSet$t(object.target) ? globalThis.String(object.target) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.url !== "") {
        obj.url = message.url;
      }
      if (message.target !== void 0) {
        obj.target = message.target;
      }
      return obj;
    },
    create(base) {
      return NavigateToUrlEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseNavigateToUrlEffect();
      message.url = object.url ?? "";
      message.target = object.target ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(NavigateToUrlEffect.$type, NavigateToUrlEffect);
  function isSet$t(value) {
    return value !== null && value !== void 0;
  }
  var FormFieldType;
  (function(FormFieldType2) {
    FormFieldType2[FormFieldType2["STRING"] = 0] = "STRING";
    FormFieldType2[FormFieldType2["PARAGRAPH"] = 1] = "PARAGRAPH";
    FormFieldType2[FormFieldType2["NUMBER"] = 2] = "NUMBER";
    FormFieldType2[FormFieldType2["BOOLEAN"] = 3] = "BOOLEAN";
    FormFieldType2[FormFieldType2["LIST"] = 4] = "LIST";
    FormFieldType2[FormFieldType2["SELECTION"] = 5] = "SELECTION";
    FormFieldType2[FormFieldType2["GROUP"] = 6] = "GROUP";
    FormFieldType2[FormFieldType2["IMAGE"] = 7] = "IMAGE";
    FormFieldType2[FormFieldType2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(FormFieldType || (FormFieldType = {}));
  function formFieldTypeFromJSON(object) {
    switch (object) {
      case 0:
      case "STRING":
        return FormFieldType.STRING;
      case 1:
      case "PARAGRAPH":
        return FormFieldType.PARAGRAPH;
      case 2:
      case "NUMBER":
        return FormFieldType.NUMBER;
      case 3:
      case "BOOLEAN":
        return FormFieldType.BOOLEAN;
      case 4:
      case "LIST":
        return FormFieldType.LIST;
      case 5:
      case "SELECTION":
        return FormFieldType.SELECTION;
      case 6:
      case "GROUP":
        return FormFieldType.GROUP;
      case 7:
      case "IMAGE":
        return FormFieldType.IMAGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return FormFieldType.UNRECOGNIZED;
    }
  }
  function formFieldTypeToJSON(object) {
    switch (object) {
      case FormFieldType.STRING:
        return 0;
      case FormFieldType.PARAGRAPH:
        return 1;
      case FormFieldType.NUMBER:
        return 2;
      case FormFieldType.BOOLEAN:
        return 3;
      case FormFieldType.LIST:
        return 4;
      case FormFieldType.SELECTION:
        return 5;
      case FormFieldType.GROUP:
        return 6;
      case FormFieldType.IMAGE:
        return 7;
      case FormFieldType.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseFormFieldValue() {
    return {
      fieldType: 0,
      isSecret: void 0,
      stringValue: void 0,
      numberValue: void 0,
      boolValue: void 0,
      listValue: void 0,
      selectionValue: void 0,
      groupValue: void 0
    };
  }
  const FormFieldValue = {
    $type: "devvit.ui.form_builder.v1alpha.FormFieldValue",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.fieldType !== 0) {
        writer2.uint32(8).int32(message.fieldType);
      }
      if (message.isSecret !== void 0) {
        writer2.uint32(800).bool(message.isSecret);
      }
      if (message.stringValue !== void 0) {
        writer2.uint32(18).string(message.stringValue);
      }
      if (message.numberValue !== void 0) {
        writer2.uint32(25).double(message.numberValue);
      }
      if (message.boolValue !== void 0) {
        writer2.uint32(32).bool(message.boolValue);
      }
      if (message.listValue !== void 0) {
        FormFieldValue_ListValue.encode(message.listValue, writer2.uint32(42).fork()).ldelim();
      }
      if (message.selectionValue !== void 0) {
        FormFieldValue_SelectionValue.encode(message.selectionValue, writer2.uint32(50).fork()).ldelim();
      }
      if (message.groupValue !== void 0) {
        FormFieldValue_GroupValue.encode(message.groupValue, writer2.uint32(58).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormFieldValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.fieldType = reader2.int32();
            continue;
          case 100:
            if (tag !== 800) {
              break;
            }
            message.isSecret = reader2.bool();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.stringValue = reader2.string();
            continue;
          case 3:
            if (tag !== 25) {
              break;
            }
            message.numberValue = reader2.double();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.boolValue = reader2.bool();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.listValue = FormFieldValue_ListValue.decode(reader2, reader2.uint32());
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.selectionValue = FormFieldValue_SelectionValue.decode(reader2, reader2.uint32());
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.groupValue = FormFieldValue_GroupValue.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        fieldType: isSet$s(object.fieldType) ? formFieldTypeFromJSON(object.fieldType) : 0,
        isSecret: isSet$s(object.isSecret) ? globalThis.Boolean(object.isSecret) : void 0,
        stringValue: isSet$s(object.stringValue) ? globalThis.String(object.stringValue) : void 0,
        numberValue: isSet$s(object.numberValue) ? globalThis.Number(object.numberValue) : void 0,
        boolValue: isSet$s(object.boolValue) ? globalThis.Boolean(object.boolValue) : void 0,
        listValue: isSet$s(object.listValue) ? FormFieldValue_ListValue.fromJSON(object.listValue) : void 0,
        selectionValue: isSet$s(object.selectionValue) ? FormFieldValue_SelectionValue.fromJSON(object.selectionValue) : void 0,
        groupValue: isSet$s(object.groupValue) ? FormFieldValue_GroupValue.fromJSON(object.groupValue) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.fieldType !== 0) {
        obj.fieldType = formFieldTypeToJSON(message.fieldType);
      }
      if (message.isSecret !== void 0) {
        obj.isSecret = message.isSecret;
      }
      if (message.stringValue !== void 0) {
        obj.stringValue = message.stringValue;
      }
      if (message.numberValue !== void 0) {
        obj.numberValue = message.numberValue;
      }
      if (message.boolValue !== void 0) {
        obj.boolValue = message.boolValue;
      }
      if (message.listValue !== void 0) {
        obj.listValue = FormFieldValue_ListValue.toJSON(message.listValue);
      }
      if (message.selectionValue !== void 0) {
        obj.selectionValue = FormFieldValue_SelectionValue.toJSON(message.selectionValue);
      }
      if (message.groupValue !== void 0) {
        obj.groupValue = FormFieldValue_GroupValue.toJSON(message.groupValue);
      }
      return obj;
    },
    create(base) {
      return FormFieldValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFormFieldValue();
      message.fieldType = object.fieldType ?? 0;
      message.isSecret = object.isSecret ?? void 0;
      message.stringValue = object.stringValue ?? void 0;
      message.numberValue = object.numberValue ?? void 0;
      message.boolValue = object.boolValue ?? void 0;
      message.listValue = object.listValue !== void 0 && object.listValue !== null ? FormFieldValue_ListValue.fromPartial(object.listValue) : void 0;
      message.selectionValue = object.selectionValue !== void 0 && object.selectionValue !== null ? FormFieldValue_SelectionValue.fromPartial(object.selectionValue) : void 0;
      message.groupValue = object.groupValue !== void 0 && object.groupValue !== null ? FormFieldValue_GroupValue.fromPartial(object.groupValue) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FormFieldValue.$type, FormFieldValue);
  function createBaseFormFieldValue_ListValue() {
    return { itemType: 0, items: [] };
  }
  const FormFieldValue_ListValue = {
    $type: "devvit.ui.form_builder.v1alpha.FormFieldValue.ListValue",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.itemType !== 0) {
        writer2.uint32(8).int32(message.itemType);
      }
      for (const v of message.items) {
        FormFieldValue.encode(v, writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormFieldValue_ListValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.itemType = reader2.int32();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.items.push(FormFieldValue.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        itemType: isSet$s(object.itemType) ? formFieldTypeFromJSON(object.itemType) : 0,
        items: globalThis.Array.isArray(object?.items) ? object.items.map((e) => FormFieldValue.fromJSON(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.itemType !== 0) {
        obj.itemType = formFieldTypeToJSON(message.itemType);
      }
      if (message.items?.length) {
        obj.items = message.items.map((e) => FormFieldValue.toJSON(e));
      }
      return obj;
    },
    create(base) {
      return FormFieldValue_ListValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFormFieldValue_ListValue();
      message.itemType = object.itemType ?? 0;
      message.items = object.items?.map((e) => FormFieldValue.fromPartial(e)) || [];
      return message;
    }
  };
  messageTypeRegistry.set(FormFieldValue_ListValue.$type, FormFieldValue_ListValue);
  function createBaseFormFieldValue_SelectionValue() {
    return { values: [] };
  }
  const FormFieldValue_SelectionValue = {
    $type: "devvit.ui.form_builder.v1alpha.FormFieldValue.SelectionValue",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.values) {
        writer2.uint32(10).string(v);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormFieldValue_SelectionValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.values.push(reader2.string());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        values: globalThis.Array.isArray(object?.values) ? object.values.map((e) => globalThis.String(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.values?.length) {
        obj.values = message.values;
      }
      return obj;
    },
    create(base) {
      return FormFieldValue_SelectionValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFormFieldValue_SelectionValue();
      message.values = object.values?.map((e) => e) || [];
      return message;
    }
  };
  messageTypeRegistry.set(FormFieldValue_SelectionValue.$type, FormFieldValue_SelectionValue);
  function createBaseFormFieldValue_GroupValue() {
    return {};
  }
  const FormFieldValue_GroupValue = {
    $type: "devvit.ui.form_builder.v1alpha.FormFieldValue.GroupValue",
    encode(_, writer2 = _m0.Writer.create()) {
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormFieldValue_GroupValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(_) {
      return {};
    },
    toJSON(_) {
      const obj = {};
      return obj;
    },
    create(base) {
      return FormFieldValue_GroupValue.fromPartial(base ?? {});
    },
    fromPartial(_) {
      const message = createBaseFormFieldValue_GroupValue();
      return message;
    }
  };
  messageTypeRegistry.set(FormFieldValue_GroupValue.$type, FormFieldValue_GroupValue);
  function isSet$s(value) {
    return value !== null && value !== void 0;
  }
  function createBaseFormField() {
    return {
      fieldId: "",
      fieldType: 0,
      label: "",
      helpText: void 0,
      defaultValue: void 0,
      required: void 0,
      disabled: void 0,
      fieldConfig: void 0,
      isSecret: void 0
    };
  }
  const FormField = {
    $type: "devvit.ui.form_builder.v1alpha.FormField",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.fieldId !== "") {
        writer2.uint32(10).string(message.fieldId);
      }
      if (message.fieldType !== 0) {
        writer2.uint32(16).int32(message.fieldType);
      }
      if (message.label !== "") {
        writer2.uint32(26).string(message.label);
      }
      if (message.helpText !== void 0) {
        writer2.uint32(34).string(message.helpText);
      }
      if (message.defaultValue !== void 0) {
        FormFieldValue.encode(message.defaultValue, writer2.uint32(42).fork()).ldelim();
      }
      if (message.required !== void 0) {
        writer2.uint32(48).bool(message.required);
      }
      if (message.disabled !== void 0) {
        writer2.uint32(56).bool(message.disabled);
      }
      if (message.fieldConfig !== void 0) {
        FieldConfig.encode(message.fieldConfig, writer2.uint32(66).fork()).ldelim();
      }
      if (message.isSecret !== void 0) {
        writer2.uint32(72).bool(message.isSecret);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormField();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.fieldId = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.fieldType = reader2.int32();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.label = reader2.string();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.helpText = reader2.string();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.defaultValue = FormFieldValue.decode(reader2, reader2.uint32());
            continue;
          case 6:
            if (tag !== 48) {
              break;
            }
            message.required = reader2.bool();
            continue;
          case 7:
            if (tag !== 56) {
              break;
            }
            message.disabled = reader2.bool();
            continue;
          case 8:
            if (tag !== 66) {
              break;
            }
            message.fieldConfig = FieldConfig.decode(reader2, reader2.uint32());
            continue;
          case 9:
            if (tag !== 72) {
              break;
            }
            message.isSecret = reader2.bool();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        fieldId: isSet$r(object.fieldId) ? globalThis.String(object.fieldId) : "",
        fieldType: isSet$r(object.fieldType) ? formFieldTypeFromJSON(object.fieldType) : 0,
        label: isSet$r(object.label) ? globalThis.String(object.label) : "",
        helpText: isSet$r(object.helpText) ? globalThis.String(object.helpText) : void 0,
        defaultValue: isSet$r(object.defaultValue) ? FormFieldValue.fromJSON(object.defaultValue) : void 0,
        required: isSet$r(object.required) ? globalThis.Boolean(object.required) : void 0,
        disabled: isSet$r(object.disabled) ? globalThis.Boolean(object.disabled) : void 0,
        fieldConfig: isSet$r(object.fieldConfig) ? FieldConfig.fromJSON(object.fieldConfig) : void 0,
        isSecret: isSet$r(object.isSecret) ? globalThis.Boolean(object.isSecret) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.fieldId !== "") {
        obj.fieldId = message.fieldId;
      }
      if (message.fieldType !== 0) {
        obj.fieldType = formFieldTypeToJSON(message.fieldType);
      }
      if (message.label !== "") {
        obj.label = message.label;
      }
      if (message.helpText !== void 0) {
        obj.helpText = message.helpText;
      }
      if (message.defaultValue !== void 0) {
        obj.defaultValue = FormFieldValue.toJSON(message.defaultValue);
      }
      if (message.required !== void 0) {
        obj.required = message.required;
      }
      if (message.disabled !== void 0) {
        obj.disabled = message.disabled;
      }
      if (message.fieldConfig !== void 0) {
        obj.fieldConfig = FieldConfig.toJSON(message.fieldConfig);
      }
      if (message.isSecret !== void 0) {
        obj.isSecret = message.isSecret;
      }
      return obj;
    },
    create(base) {
      return FormField.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFormField();
      message.fieldId = object.fieldId ?? "";
      message.fieldType = object.fieldType ?? 0;
      message.label = object.label ?? "";
      message.helpText = object.helpText ?? void 0;
      message.defaultValue = object.defaultValue !== void 0 && object.defaultValue !== null ? FormFieldValue.fromPartial(object.defaultValue) : void 0;
      message.required = object.required ?? void 0;
      message.disabled = object.disabled ?? void 0;
      message.fieldConfig = object.fieldConfig !== void 0 && object.fieldConfig !== null ? FieldConfig.fromPartial(object.fieldConfig) : void 0;
      message.isSecret = object.isSecret ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FormField.$type, FormField);
  function createBaseFieldConfig() {
    return {
      stringConfig: void 0,
      paragraphConfig: void 0,
      numberConfig: void 0,
      booleanConfig: void 0,
      listConfig: void 0,
      selectionConfig: void 0,
      groupConfig: void 0
    };
  }
  const FieldConfig = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.stringConfig !== void 0) {
        FieldConfig_String.encode(message.stringConfig, writer2.uint32(10).fork()).ldelim();
      }
      if (message.paragraphConfig !== void 0) {
        FieldConfig_Paragraph.encode(message.paragraphConfig, writer2.uint32(18).fork()).ldelim();
      }
      if (message.numberConfig !== void 0) {
        FieldConfig_Number.encode(message.numberConfig, writer2.uint32(26).fork()).ldelim();
      }
      if (message.booleanConfig !== void 0) {
        FieldConfig_Boolean.encode(message.booleanConfig, writer2.uint32(34).fork()).ldelim();
      }
      if (message.listConfig !== void 0) {
        FieldConfig_List.encode(message.listConfig, writer2.uint32(42).fork()).ldelim();
      }
      if (message.selectionConfig !== void 0) {
        FieldConfig_Selection.encode(message.selectionConfig, writer2.uint32(50).fork()).ldelim();
      }
      if (message.groupConfig !== void 0) {
        FieldConfig_Group.encode(message.groupConfig, writer2.uint32(58).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.stringConfig = FieldConfig_String.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.paragraphConfig = FieldConfig_Paragraph.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.numberConfig = FieldConfig_Number.decode(reader2, reader2.uint32());
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.booleanConfig = FieldConfig_Boolean.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.listConfig = FieldConfig_List.decode(reader2, reader2.uint32());
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.selectionConfig = FieldConfig_Selection.decode(reader2, reader2.uint32());
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.groupConfig = FieldConfig_Group.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        stringConfig: isSet$r(object.stringConfig) ? FieldConfig_String.fromJSON(object.stringConfig) : void 0,
        paragraphConfig: isSet$r(object.paragraphConfig) ? FieldConfig_Paragraph.fromJSON(object.paragraphConfig) : void 0,
        numberConfig: isSet$r(object.numberConfig) ? FieldConfig_Number.fromJSON(object.numberConfig) : void 0,
        booleanConfig: isSet$r(object.booleanConfig) ? FieldConfig_Boolean.fromJSON(object.booleanConfig) : void 0,
        listConfig: isSet$r(object.listConfig) ? FieldConfig_List.fromJSON(object.listConfig) : void 0,
        selectionConfig: isSet$r(object.selectionConfig) ? FieldConfig_Selection.fromJSON(object.selectionConfig) : void 0,
        groupConfig: isSet$r(object.groupConfig) ? FieldConfig_Group.fromJSON(object.groupConfig) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.stringConfig !== void 0) {
        obj.stringConfig = FieldConfig_String.toJSON(message.stringConfig);
      }
      if (message.paragraphConfig !== void 0) {
        obj.paragraphConfig = FieldConfig_Paragraph.toJSON(message.paragraphConfig);
      }
      if (message.numberConfig !== void 0) {
        obj.numberConfig = FieldConfig_Number.toJSON(message.numberConfig);
      }
      if (message.booleanConfig !== void 0) {
        obj.booleanConfig = FieldConfig_Boolean.toJSON(message.booleanConfig);
      }
      if (message.listConfig !== void 0) {
        obj.listConfig = FieldConfig_List.toJSON(message.listConfig);
      }
      if (message.selectionConfig !== void 0) {
        obj.selectionConfig = FieldConfig_Selection.toJSON(message.selectionConfig);
      }
      if (message.groupConfig !== void 0) {
        obj.groupConfig = FieldConfig_Group.toJSON(message.groupConfig);
      }
      return obj;
    },
    create(base) {
      return FieldConfig.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig();
      message.stringConfig = object.stringConfig !== void 0 && object.stringConfig !== null ? FieldConfig_String.fromPartial(object.stringConfig) : void 0;
      message.paragraphConfig = object.paragraphConfig !== void 0 && object.paragraphConfig !== null ? FieldConfig_Paragraph.fromPartial(object.paragraphConfig) : void 0;
      message.numberConfig = object.numberConfig !== void 0 && object.numberConfig !== null ? FieldConfig_Number.fromPartial(object.numberConfig) : void 0;
      message.booleanConfig = object.booleanConfig !== void 0 && object.booleanConfig !== null ? FieldConfig_Boolean.fromPartial(object.booleanConfig) : void 0;
      message.listConfig = object.listConfig !== void 0 && object.listConfig !== null ? FieldConfig_List.fromPartial(object.listConfig) : void 0;
      message.selectionConfig = object.selectionConfig !== void 0 && object.selectionConfig !== null ? FieldConfig_Selection.fromPartial(object.selectionConfig) : void 0;
      message.groupConfig = object.groupConfig !== void 0 && object.groupConfig !== null ? FieldConfig_Group.fromPartial(object.groupConfig) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig.$type, FieldConfig);
  function createBaseFieldConfig_String() {
    return { minLength: void 0, maxLength: void 0, placeholder: void 0 };
  }
  const FieldConfig_String = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.String",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.minLength !== void 0) {
        writer2.uint32(8).int32(message.minLength);
      }
      if (message.maxLength !== void 0) {
        writer2.uint32(16).int32(message.maxLength);
      }
      if (message.placeholder !== void 0) {
        writer2.uint32(26).string(message.placeholder);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_String();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.minLength = reader2.int32();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.maxLength = reader2.int32();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.placeholder = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        minLength: isSet$r(object.minLength) ? globalThis.Number(object.minLength) : void 0,
        maxLength: isSet$r(object.maxLength) ? globalThis.Number(object.maxLength) : void 0,
        placeholder: isSet$r(object.placeholder) ? globalThis.String(object.placeholder) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.minLength !== void 0) {
        obj.minLength = Math.round(message.minLength);
      }
      if (message.maxLength !== void 0) {
        obj.maxLength = Math.round(message.maxLength);
      }
      if (message.placeholder !== void 0) {
        obj.placeholder = message.placeholder;
      }
      return obj;
    },
    create(base) {
      return FieldConfig_String.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig_String();
      message.minLength = object.minLength ?? void 0;
      message.maxLength = object.maxLength ?? void 0;
      message.placeholder = object.placeholder ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_String.$type, FieldConfig_String);
  function createBaseFieldConfig_Paragraph() {
    return { maxCharacters: void 0, lineHeight: void 0, placeholder: void 0 };
  }
  const FieldConfig_Paragraph = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.Paragraph",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.maxCharacters !== void 0) {
        writer2.uint32(8).int32(message.maxCharacters);
      }
      if (message.lineHeight !== void 0) {
        writer2.uint32(16).int32(message.lineHeight);
      }
      if (message.placeholder !== void 0) {
        writer2.uint32(26).string(message.placeholder);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_Paragraph();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.maxCharacters = reader2.int32();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.lineHeight = reader2.int32();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.placeholder = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        maxCharacters: isSet$r(object.maxCharacters) ? globalThis.Number(object.maxCharacters) : void 0,
        lineHeight: isSet$r(object.lineHeight) ? globalThis.Number(object.lineHeight) : void 0,
        placeholder: isSet$r(object.placeholder) ? globalThis.String(object.placeholder) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.maxCharacters !== void 0) {
        obj.maxCharacters = Math.round(message.maxCharacters);
      }
      if (message.lineHeight !== void 0) {
        obj.lineHeight = Math.round(message.lineHeight);
      }
      if (message.placeholder !== void 0) {
        obj.placeholder = message.placeholder;
      }
      return obj;
    },
    create(base) {
      return FieldConfig_Paragraph.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig_Paragraph();
      message.maxCharacters = object.maxCharacters ?? void 0;
      message.lineHeight = object.lineHeight ?? void 0;
      message.placeholder = object.placeholder ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_Paragraph.$type, FieldConfig_Paragraph);
  function createBaseFieldConfig_Number() {
    return { step: void 0, min: void 0, max: void 0 };
  }
  const FieldConfig_Number = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.Number",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.step !== void 0) {
        writer2.uint32(9).double(message.step);
      }
      if (message.min !== void 0) {
        writer2.uint32(17).double(message.min);
      }
      if (message.max !== void 0) {
        writer2.uint32(25).double(message.max);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_Number();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 9) {
              break;
            }
            message.step = reader2.double();
            continue;
          case 2:
            if (tag !== 17) {
              break;
            }
            message.min = reader2.double();
            continue;
          case 3:
            if (tag !== 25) {
              break;
            }
            message.max = reader2.double();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        step: isSet$r(object.step) ? globalThis.Number(object.step) : void 0,
        min: isSet$r(object.min) ? globalThis.Number(object.min) : void 0,
        max: isSet$r(object.max) ? globalThis.Number(object.max) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.step !== void 0) {
        obj.step = message.step;
      }
      if (message.min !== void 0) {
        obj.min = message.min;
      }
      if (message.max !== void 0) {
        obj.max = message.max;
      }
      return obj;
    },
    create(base) {
      return FieldConfig_Number.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig_Number();
      message.step = object.step ?? void 0;
      message.min = object.min ?? void 0;
      message.max = object.max ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_Number.$type, FieldConfig_Number);
  function createBaseFieldConfig_Boolean() {
    return {};
  }
  const FieldConfig_Boolean = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.Boolean",
    encode(_, writer2 = _m0.Writer.create()) {
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_Boolean();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(_) {
      return {};
    },
    toJSON(_) {
      const obj = {};
      return obj;
    },
    create(base) {
      return FieldConfig_Boolean.fromPartial(base ?? {});
    },
    fromPartial(_) {
      const message = createBaseFieldConfig_Boolean();
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_Boolean.$type, FieldConfig_Boolean);
  function createBaseFieldConfig_List() {
    return { itemType: 0, itemConfig: void 0, minEntries: void 0, maxEntries: void 0, entryLabel: void 0 };
  }
  const FieldConfig_List = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.List",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.itemType !== 0) {
        writer2.uint32(8).int32(message.itemType);
      }
      if (message.itemConfig !== void 0) {
        FieldConfig.encode(message.itemConfig, writer2.uint32(18).fork()).ldelim();
      }
      if (message.minEntries !== void 0) {
        writer2.uint32(24).int32(message.minEntries);
      }
      if (message.maxEntries !== void 0) {
        writer2.uint32(32).int32(message.maxEntries);
      }
      if (message.entryLabel !== void 0) {
        writer2.uint32(42).string(message.entryLabel);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_List();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.itemType = reader2.int32();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.itemConfig = FieldConfig.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.minEntries = reader2.int32();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.maxEntries = reader2.int32();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.entryLabel = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        itemType: isSet$r(object.itemType) ? formFieldTypeFromJSON(object.itemType) : 0,
        itemConfig: isSet$r(object.itemConfig) ? FieldConfig.fromJSON(object.itemConfig) : void 0,
        minEntries: isSet$r(object.minEntries) ? globalThis.Number(object.minEntries) : void 0,
        maxEntries: isSet$r(object.maxEntries) ? globalThis.Number(object.maxEntries) : void 0,
        entryLabel: isSet$r(object.entryLabel) ? globalThis.String(object.entryLabel) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.itemType !== 0) {
        obj.itemType = formFieldTypeToJSON(message.itemType);
      }
      if (message.itemConfig !== void 0) {
        obj.itemConfig = FieldConfig.toJSON(message.itemConfig);
      }
      if (message.minEntries !== void 0) {
        obj.minEntries = Math.round(message.minEntries);
      }
      if (message.maxEntries !== void 0) {
        obj.maxEntries = Math.round(message.maxEntries);
      }
      if (message.entryLabel !== void 0) {
        obj.entryLabel = message.entryLabel;
      }
      return obj;
    },
    create(base) {
      return FieldConfig_List.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig_List();
      message.itemType = object.itemType ?? 0;
      message.itemConfig = object.itemConfig !== void 0 && object.itemConfig !== null ? FieldConfig.fromPartial(object.itemConfig) : void 0;
      message.minEntries = object.minEntries ?? void 0;
      message.maxEntries = object.maxEntries ?? void 0;
      message.entryLabel = object.entryLabel ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_List.$type, FieldConfig_List);
  function createBaseFieldConfig_Selection() {
    return {
      choices: [],
      multiSelect: void 0,
      minSelections: void 0,
      maxSelections: void 0,
      renderAsList: void 0
    };
  }
  const FieldConfig_Selection = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.Selection",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.choices) {
        FieldConfig_Selection_Item.encode(v, writer2.uint32(10).fork()).ldelim();
      }
      if (message.multiSelect !== void 0) {
        writer2.uint32(16).bool(message.multiSelect);
      }
      if (message.minSelections !== void 0) {
        writer2.uint32(24).int32(message.minSelections);
      }
      if (message.maxSelections !== void 0) {
        writer2.uint32(32).int32(message.maxSelections);
      }
      if (message.renderAsList !== void 0) {
        writer2.uint32(40).bool(message.renderAsList);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_Selection();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.choices.push(FieldConfig_Selection_Item.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.multiSelect = reader2.bool();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.minSelections = reader2.int32();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.maxSelections = reader2.int32();
            continue;
          case 5:
            if (tag !== 40) {
              break;
            }
            message.renderAsList = reader2.bool();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        choices: globalThis.Array.isArray(object?.choices) ? object.choices.map((e) => FieldConfig_Selection_Item.fromJSON(e)) : [],
        multiSelect: isSet$r(object.multiSelect) ? globalThis.Boolean(object.multiSelect) : void 0,
        minSelections: isSet$r(object.minSelections) ? globalThis.Number(object.minSelections) : void 0,
        maxSelections: isSet$r(object.maxSelections) ? globalThis.Number(object.maxSelections) : void 0,
        renderAsList: isSet$r(object.renderAsList) ? globalThis.Boolean(object.renderAsList) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.choices?.length) {
        obj.choices = message.choices.map((e) => FieldConfig_Selection_Item.toJSON(e));
      }
      if (message.multiSelect !== void 0) {
        obj.multiSelect = message.multiSelect;
      }
      if (message.minSelections !== void 0) {
        obj.minSelections = Math.round(message.minSelections);
      }
      if (message.maxSelections !== void 0) {
        obj.maxSelections = Math.round(message.maxSelections);
      }
      if (message.renderAsList !== void 0) {
        obj.renderAsList = message.renderAsList;
      }
      return obj;
    },
    create(base) {
      return FieldConfig_Selection.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig_Selection();
      message.choices = object.choices?.map((e) => FieldConfig_Selection_Item.fromPartial(e)) || [];
      message.multiSelect = object.multiSelect ?? void 0;
      message.minSelections = object.minSelections ?? void 0;
      message.maxSelections = object.maxSelections ?? void 0;
      message.renderAsList = object.renderAsList ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_Selection.$type, FieldConfig_Selection);
  function createBaseFieldConfig_Selection_Item() {
    return { label: "", value: "" };
  }
  const FieldConfig_Selection_Item = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.Selection.Item",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.label !== "") {
        writer2.uint32(10).string(message.label);
      }
      if (message.value !== "") {
        writer2.uint32(18).string(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_Selection_Item();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.label = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.value = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        label: isSet$r(object.label) ? globalThis.String(object.label) : "",
        value: isSet$r(object.value) ? globalThis.String(object.value) : ""
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.label !== "") {
        obj.label = message.label;
      }
      if (message.value !== "") {
        obj.value = message.value;
      }
      return obj;
    },
    create(base) {
      return FieldConfig_Selection_Item.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig_Selection_Item();
      message.label = object.label ?? "";
      message.value = object.value ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_Selection_Item.$type, FieldConfig_Selection_Item);
  function createBaseFieldConfig_Group() {
    return { fields: [] };
  }
  const FieldConfig_Group = {
    $type: "devvit.ui.form_builder.v1alpha.FieldConfig.Group",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.fields) {
        FormField.encode(v, writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFieldConfig_Group();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.fields.push(FormField.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        fields: globalThis.Array.isArray(object?.fields) ? object.fields.map((e) => FormField.fromJSON(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.fields?.length) {
        obj.fields = message.fields.map((e) => FormField.toJSON(e));
      }
      return obj;
    },
    create(base) {
      return FieldConfig_Group.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFieldConfig_Group();
      message.fields = object.fields?.map((e) => FormField.fromPartial(e)) || [];
      return message;
    }
  };
  messageTypeRegistry.set(FieldConfig_Group.$type, FieldConfig_Group);
  function isSet$r(value) {
    return value !== null && value !== void 0;
  }
  function createBaseForm() {
    return {
      fields: [],
      title: void 0,
      shortDescription: void 0,
      acceptLabel: void 0,
      cancelLabel: void 0,
      id: void 0
    };
  }
  const Form = {
    $type: "devvit.ui.form_builder.v1alpha.Form",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.fields) {
        FormField.encode(v, writer2.uint32(10).fork()).ldelim();
      }
      if (message.title !== void 0) {
        writer2.uint32(18).string(message.title);
      }
      if (message.shortDescription !== void 0) {
        writer2.uint32(26).string(message.shortDescription);
      }
      if (message.acceptLabel !== void 0) {
        writer2.uint32(34).string(message.acceptLabel);
      }
      if (message.cancelLabel !== void 0) {
        writer2.uint32(42).string(message.cancelLabel);
      }
      if (message.id !== void 0) {
        writer2.uint32(50).string(message.id);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseForm();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.fields.push(FormField.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.title = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.shortDescription = reader2.string();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.acceptLabel = reader2.string();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.cancelLabel = reader2.string();
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.id = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        fields: globalThis.Array.isArray(object?.fields) ? object.fields.map((e) => FormField.fromJSON(e)) : [],
        title: isSet$q(object.title) ? globalThis.String(object.title) : void 0,
        shortDescription: isSet$q(object.shortDescription) ? globalThis.String(object.shortDescription) : void 0,
        acceptLabel: isSet$q(object.acceptLabel) ? globalThis.String(object.acceptLabel) : void 0,
        cancelLabel: isSet$q(object.cancelLabel) ? globalThis.String(object.cancelLabel) : void 0,
        id: isSet$q(object.id) ? globalThis.String(object.id) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.fields?.length) {
        obj.fields = message.fields.map((e) => FormField.toJSON(e));
      }
      if (message.title !== void 0) {
        obj.title = message.title;
      }
      if (message.shortDescription !== void 0) {
        obj.shortDescription = message.shortDescription;
      }
      if (message.acceptLabel !== void 0) {
        obj.acceptLabel = message.acceptLabel;
      }
      if (message.cancelLabel !== void 0) {
        obj.cancelLabel = message.cancelLabel;
      }
      if (message.id !== void 0) {
        obj.id = message.id;
      }
      return obj;
    },
    create(base) {
      return Form.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseForm();
      message.fields = object.fields?.map((e) => FormField.fromPartial(e)) || [];
      message.title = object.title ?? void 0;
      message.shortDescription = object.shortDescription ?? void 0;
      message.acceptLabel = object.acceptLabel ?? void 0;
      message.cancelLabel = object.cancelLabel ?? void 0;
      message.id = object.id ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(Form.$type, Form);
  function isSet$q(value) {
    return value !== null && value !== void 0;
  }
  function createBaseShowFormEffect() {
    return { form: void 0 };
  }
  const ShowFormEffect = {
    $type: "devvit.ui.effect_types.v1alpha.ShowFormEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.form !== void 0) {
        Form.encode(message.form, writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseShowFormEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.form = Form.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { form: isSet$p(object.form) ? Form.fromJSON(object.form) : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.form !== void 0) {
        obj.form = Form.toJSON(message.form);
      }
      return obj;
    },
    create(base) {
      return ShowFormEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseShowFormEffect();
      message.form = object.form !== void 0 && object.form !== null ? Form.fromPartial(object.form) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(ShowFormEffect.$type, ShowFormEffect);
  function createBaseFormSubmittedEvent() {
    return { results: {}, formId: void 0 };
  }
  const FormSubmittedEvent = {
    $type: "devvit.ui.effect_types.v1alpha.FormSubmittedEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      Object.entries(message.results).forEach(([key, value]) => {
        FormSubmittedEvent_ResultsEntry.encode({ key, value }, writer2.uint32(10).fork()).ldelim();
      });
      if (message.formId !== void 0) {
        writer2.uint32(18).string(message.formId);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormSubmittedEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            const entry1 = FormSubmittedEvent_ResultsEntry.decode(reader2, reader2.uint32());
            if (entry1.value !== void 0) {
              message.results[entry1.key] = entry1.value;
            }
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.formId = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        results: isObject$8(object.results) ? Object.entries(object.results).reduce((acc, [key, value]) => {
          acc[key] = FormFieldValue.fromJSON(value);
          return acc;
        }, {}) : {},
        formId: isSet$p(object.formId) ? globalThis.String(object.formId) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.results) {
        const entries = Object.entries(message.results);
        if (entries.length > 0) {
          obj.results = {};
          entries.forEach(([k, v]) => {
            obj.results[k] = FormFieldValue.toJSON(v);
          });
        }
      }
      if (message.formId !== void 0) {
        obj.formId = message.formId;
      }
      return obj;
    },
    create(base) {
      return FormSubmittedEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFormSubmittedEvent();
      message.results = Object.entries(object.results ?? {}).reduce((acc, [key, value]) => {
        if (value !== void 0) {
          acc[key] = FormFieldValue.fromPartial(value);
        }
        return acc;
      }, {});
      message.formId = object.formId ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FormSubmittedEvent.$type, FormSubmittedEvent);
  function createBaseFormSubmittedEvent_ResultsEntry() {
    return { key: "", value: void 0 };
  }
  const FormSubmittedEvent_ResultsEntry = {
    $type: "devvit.ui.effect_types.v1alpha.FormSubmittedEvent.ResultsEntry",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.key !== "") {
        writer2.uint32(10).string(message.key);
      }
      if (message.value !== void 0) {
        FormFieldValue.encode(message.value, writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormSubmittedEvent_ResultsEntry();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.key = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.value = FormFieldValue.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        key: isSet$p(object.key) ? globalThis.String(object.key) : "",
        value: isSet$p(object.value) ? FormFieldValue.fromJSON(object.value) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.key !== "") {
        obj.key = message.key;
      }
      if (message.value !== void 0) {
        obj.value = FormFieldValue.toJSON(message.value);
      }
      return obj;
    },
    create(base) {
      return FormSubmittedEvent_ResultsEntry.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFormSubmittedEvent_ResultsEntry();
      message.key = object.key ?? "";
      message.value = object.value !== void 0 && object.value !== null ? FormFieldValue.fromPartial(object.value) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FormSubmittedEvent_ResultsEntry.$type, FormSubmittedEvent_ResultsEntry);
  function createBaseFormCanceledEvent() {
    return { formId: void 0 };
  }
  const FormCanceledEvent = {
    $type: "devvit.ui.effect_types.v1alpha.FormCanceledEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.formId !== void 0) {
        writer2.uint32(10).string(message.formId);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFormCanceledEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.formId = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { formId: isSet$p(object.formId) ? globalThis.String(object.formId) : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.formId !== void 0) {
        obj.formId = message.formId;
      }
      return obj;
    },
    create(base) {
      return FormCanceledEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFormCanceledEvent();
      message.formId = object.formId ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(FormCanceledEvent.$type, FormCanceledEvent);
  function isObject$8(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$p(value) {
    return value !== null && value !== void 0;
  }
  var ToastAppearance;
  (function(ToastAppearance2) {
    ToastAppearance2[ToastAppearance2["NEUTRAL"] = 0] = "NEUTRAL";
    ToastAppearance2[ToastAppearance2["SUCCESS"] = 1] = "SUCCESS";
    ToastAppearance2[ToastAppearance2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(ToastAppearance || (ToastAppearance = {}));
  function toastAppearanceFromJSON(object) {
    switch (object) {
      case 0:
      case "NEUTRAL":
        return ToastAppearance.NEUTRAL;
      case 1:
      case "SUCCESS":
        return ToastAppearance.SUCCESS;
      case -1:
      case "UNRECOGNIZED":
      default:
        return ToastAppearance.UNRECOGNIZED;
    }
  }
  function toastAppearanceToJSON(object) {
    switch (object) {
      case ToastAppearance.NEUTRAL:
        return 0;
      case ToastAppearance.SUCCESS:
        return 1;
      case ToastAppearance.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseToastLeadingElement() {
    return { icon: void 0, emoji: void 0, avatar: void 0 };
  }
  const ToastLeadingElement = {
    $type: "devvit.ui.toast.ToastLeadingElement",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.icon !== void 0) {
        writer2.uint32(10).string(message.icon);
      }
      if (message.emoji !== void 0) {
        writer2.uint32(18).string(message.emoji);
      }
      if (message.avatar !== void 0) {
        writer2.uint32(26).string(message.avatar);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseToastLeadingElement();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.icon = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.emoji = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.avatar = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        icon: isSet$o(object.icon) ? globalThis.String(object.icon) : void 0,
        emoji: isSet$o(object.emoji) ? globalThis.String(object.emoji) : void 0,
        avatar: isSet$o(object.avatar) ? globalThis.String(object.avatar) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.icon !== void 0) {
        obj.icon = message.icon;
      }
      if (message.emoji !== void 0) {
        obj.emoji = message.emoji;
      }
      if (message.avatar !== void 0) {
        obj.avatar = message.avatar;
      }
      return obj;
    },
    create(base) {
      return ToastLeadingElement.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseToastLeadingElement();
      message.icon = object.icon ?? void 0;
      message.emoji = object.emoji ?? void 0;
      message.avatar = object.avatar ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(ToastLeadingElement.$type, ToastLeadingElement);
  function createBaseToastTrailingElement() {
    return { label: void 0, icon: void 0 };
  }
  const ToastTrailingElement = {
    $type: "devvit.ui.toast.ToastTrailingElement",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.label !== void 0) {
        writer2.uint32(10).string(message.label);
      }
      if (message.icon !== void 0) {
        writer2.uint32(18).string(message.icon);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseToastTrailingElement();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.label = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.icon = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        label: isSet$o(object.label) ? globalThis.String(object.label) : void 0,
        icon: isSet$o(object.icon) ? globalThis.String(object.icon) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.label !== void 0) {
        obj.label = message.label;
      }
      if (message.icon !== void 0) {
        obj.icon = message.icon;
      }
      return obj;
    },
    create(base) {
      return ToastTrailingElement.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseToastTrailingElement();
      message.label = object.label ?? void 0;
      message.icon = object.icon ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(ToastTrailingElement.$type, ToastTrailingElement);
  function createBaseToast() {
    return { text: "", appearance: void 0, leadingElement: void 0, trailingElement: void 0 };
  }
  const Toast = {
    $type: "devvit.ui.toast.Toast",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.text !== "") {
        writer2.uint32(10).string(message.text);
      }
      if (message.appearance !== void 0) {
        writer2.uint32(16).int32(message.appearance);
      }
      if (message.leadingElement !== void 0) {
        ToastLeadingElement.encode(message.leadingElement, writer2.uint32(26).fork()).ldelim();
      }
      if (message.trailingElement !== void 0) {
        ToastTrailingElement.encode(message.trailingElement, writer2.uint32(34).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseToast();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.text = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.appearance = reader2.int32();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.leadingElement = ToastLeadingElement.decode(reader2, reader2.uint32());
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.trailingElement = ToastTrailingElement.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        text: isSet$o(object.text) ? globalThis.String(object.text) : "",
        appearance: isSet$o(object.appearance) ? toastAppearanceFromJSON(object.appearance) : void 0,
        leadingElement: isSet$o(object.leadingElement) ? ToastLeadingElement.fromJSON(object.leadingElement) : void 0,
        trailingElement: isSet$o(object.trailingElement) ? ToastTrailingElement.fromJSON(object.trailingElement) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.text !== "") {
        obj.text = message.text;
      }
      if (message.appearance !== void 0) {
        obj.appearance = toastAppearanceToJSON(message.appearance);
      }
      if (message.leadingElement !== void 0) {
        obj.leadingElement = ToastLeadingElement.toJSON(message.leadingElement);
      }
      if (message.trailingElement !== void 0) {
        obj.trailingElement = ToastTrailingElement.toJSON(message.trailingElement);
      }
      return obj;
    },
    create(base) {
      return Toast.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseToast();
      message.text = object.text ?? "";
      message.appearance = object.appearance ?? void 0;
      message.leadingElement = object.leadingElement !== void 0 && object.leadingElement !== null ? ToastLeadingElement.fromPartial(object.leadingElement) : void 0;
      message.trailingElement = object.trailingElement !== void 0 && object.trailingElement !== null ? ToastTrailingElement.fromPartial(object.trailingElement) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(Toast.$type, Toast);
  function isSet$o(value) {
    return value !== null && value !== void 0;
  }
  function createBaseShowToastEffect() {
    return { toast: void 0 };
  }
  const ShowToastEffect = {
    $type: "devvit.ui.effect_types.v1alpha.ShowToastEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.toast !== void 0) {
        Toast.encode(message.toast, writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseShowToastEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.toast = Toast.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { toast: isSet$n(object.toast) ? Toast.fromJSON(object.toast) : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.toast !== void 0) {
        obj.toast = Toast.toJSON(message.toast);
      }
      return obj;
    },
    create(base) {
      return ShowToastEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseShowToastEffect();
      message.toast = object.toast !== void 0 && object.toast !== null ? Toast.fromPartial(object.toast) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(ShowToastEffect.$type, ShowToastEffect);
  function createBaseToastActionEvent() {
    return {};
  }
  const ToastActionEvent = {
    $type: "devvit.ui.effect_types.v1alpha.ToastActionEvent",
    encode(_, writer2 = _m0.Writer.create()) {
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseToastActionEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(_) {
      return {};
    },
    toJSON(_) {
      const obj = {};
      return obj;
    },
    create(base) {
      return ToastActionEvent.fromPartial(base ?? {});
    },
    fromPartial(_) {
      const message = createBaseToastActionEvent();
      return message;
    }
  };
  messageTypeRegistry.set(ToastActionEvent.$type, ToastActionEvent);
  function isSet$n(value) {
    return value !== null && value !== void 0;
  }
  function createBaseWebViewFullScreenVisibilityEffect() {
    return { show: false, url: "", id: "" };
  }
  const WebViewFullScreenVisibilityEffect = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewFullScreenVisibilityEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.show !== false) {
        writer2.uint32(8).bool(message.show);
      }
      if (message.url !== "") {
        writer2.uint32(18).string(message.url);
      }
      if (message.id !== "") {
        writer2.uint32(26).string(message.id);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewFullScreenVisibilityEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.show = reader2.bool();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.url = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.id = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        show: isSet$m(object.show) ? globalThis.Boolean(object.show) : false,
        url: isSet$m(object.url) ? globalThis.String(object.url) : "",
        id: isSet$m(object.id) ? globalThis.String(object.id) : ""
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.show !== false) {
        obj.show = message.show;
      }
      if (message.url !== "") {
        obj.url = message.url;
      }
      if (message.id !== "") {
        obj.id = message.id;
      }
      return obj;
    },
    create(base) {
      return WebViewFullScreenVisibilityEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewFullScreenVisibilityEffect();
      message.show = object.show ?? false;
      message.url = object.url ?? "";
      message.id = object.id ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(WebViewFullScreenVisibilityEffect.$type, WebViewFullScreenVisibilityEffect);
  function isSet$m(value) {
    return value !== null && value !== void 0;
  }
  function createBaseEmpty() {
    return {};
  }
  const Empty = {
    $type: "google.protobuf.Empty",
    encode(_, writer2 = _m0.Writer.create()) {
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseEmpty();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(_) {
      return {};
    },
    toJSON(_) {
      const obj = {};
      return obj;
    },
    create(base) {
      return Empty.fromPartial(base ?? {});
    },
    fromPartial(_) {
      const message = createBaseEmpty();
      return message;
    }
  };
  messageTypeRegistry.set(Empty.$type, Empty);
  function createBaseRealtimeRequest() {
    return { channels: [] };
  }
  const RealtimeRequest = {
    $type: "devvit.events.v1alpha.RealtimeRequest",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.channels) {
        writer2.uint32(10).string(v);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRealtimeRequest();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.channels.push(reader2.string());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        channels: globalThis.Array.isArray(object?.channels) ? object.channels.map((e) => globalThis.String(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.channels?.length) {
        obj.channels = message.channels;
      }
      return obj;
    },
    create(base) {
      return RealtimeRequest.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRealtimeRequest();
      message.channels = object.channels?.map((e) => e) || [];
      return message;
    }
  };
  messageTypeRegistry.set(RealtimeRequest.$type, RealtimeRequest);
  function createBaseRealtimeEvent() {
    return { channel: "", data: void 0 };
  }
  const RealtimeEvent = {
    $type: "devvit.events.v1alpha.RealtimeEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.channel !== "") {
        writer2.uint32(10).string(message.channel);
      }
      if (message.data !== void 0) {
        Struct.encode(Struct.wrap(message.data), writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRealtimeEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.channel = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.data = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        channel: isSet$l(object.channel) ? globalThis.String(object.channel) : "",
        data: isObject$7(object.data) ? object.data : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.channel !== "") {
        obj.channel = message.channel;
      }
      if (message.data !== void 0) {
        obj.data = message.data;
      }
      return obj;
    },
    create(base) {
      return RealtimeEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRealtimeEvent();
      message.channel = object.channel ?? "";
      message.data = object.data ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(RealtimeEvent.$type, RealtimeEvent);
  function isObject$7(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$l(value) {
    return value !== null && value !== void 0;
  }
  var RealtimeSubscriptionStatus$1;
  (function(RealtimeSubscriptionStatus2) {
    RealtimeSubscriptionStatus2[RealtimeSubscriptionStatus2["REALTIME_SUBSCRIBED"] = 0] = "REALTIME_SUBSCRIBED";
    RealtimeSubscriptionStatus2[RealtimeSubscriptionStatus2["REALTIME_UNSUBSCRIBED"] = 1] = "REALTIME_UNSUBSCRIBED";
    RealtimeSubscriptionStatus2[RealtimeSubscriptionStatus2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(RealtimeSubscriptionStatus$1 || (RealtimeSubscriptionStatus$1 = {}));
  function realtimeSubscriptionStatusFromJSON$1(object) {
    switch (object) {
      case 0:
      case "REALTIME_SUBSCRIBED":
        return RealtimeSubscriptionStatus$1.REALTIME_SUBSCRIBED;
      case 1:
      case "REALTIME_UNSUBSCRIBED":
        return RealtimeSubscriptionStatus$1.REALTIME_UNSUBSCRIBED;
      case -1:
      case "UNRECOGNIZED":
      default:
        return RealtimeSubscriptionStatus$1.UNRECOGNIZED;
    }
  }
  function realtimeSubscriptionStatusToJSON$1(object) {
    switch (object) {
      case RealtimeSubscriptionStatus$1.REALTIME_SUBSCRIBED:
        return 0;
      case RealtimeSubscriptionStatus$1.REALTIME_UNSUBSCRIBED:
        return 1;
      case RealtimeSubscriptionStatus$1.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseRealtimeSubscriptionsEffect$1() {
    return { subscriptionIds: [] };
  }
  const RealtimeSubscriptionsEffect$1 = {
    $type: "devvit.ui.effect_types.v1alpha.RealtimeSubscriptionsEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.subscriptionIds) {
        writer2.uint32(10).string(v);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRealtimeSubscriptionsEffect$1();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.subscriptionIds.push(reader2.string());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        subscriptionIds: globalThis.Array.isArray(object?.subscriptionIds) ? object.subscriptionIds.map((e) => globalThis.String(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.subscriptionIds?.length) {
        obj.subscriptionIds = message.subscriptionIds;
      }
      return obj;
    },
    create(base) {
      return RealtimeSubscriptionsEffect$1.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRealtimeSubscriptionsEffect$1();
      message.subscriptionIds = object.subscriptionIds?.map((e) => e) || [];
      return message;
    }
  };
  messageTypeRegistry.set(RealtimeSubscriptionsEffect$1.$type, RealtimeSubscriptionsEffect$1);
  function createBaseRealtimeSubscriptionEvent$1() {
    return { event: void 0, status: void 0 };
  }
  const RealtimeSubscriptionEvent$1 = {
    $type: "devvit.ui.effect_types.v1alpha.RealtimeSubscriptionEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.event !== void 0) {
        RealtimeEvent.encode(message.event, writer2.uint32(10).fork()).ldelim();
      }
      if (message.status !== void 0) {
        writer2.uint32(16).int32(message.status);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRealtimeSubscriptionEvent$1();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.event = RealtimeEvent.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.status = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        event: isSet$k(object.event) ? RealtimeEvent.fromJSON(object.event) : void 0,
        status: isSet$k(object.status) ? realtimeSubscriptionStatusFromJSON$1(object.status) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.event !== void 0) {
        obj.event = RealtimeEvent.toJSON(message.event);
      }
      if (message.status !== void 0) {
        obj.status = realtimeSubscriptionStatusToJSON$1(message.status);
      }
      return obj;
    },
    create(base) {
      return RealtimeSubscriptionEvent$1.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRealtimeSubscriptionEvent$1();
      message.event = object.event !== void 0 && object.event !== null ? RealtimeEvent.fromPartial(object.event) : void 0;
      message.status = object.status ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(RealtimeSubscriptionEvent$1.$type, RealtimeSubscriptionEvent$1);
  function isSet$k(value) {
    return value !== null && value !== void 0;
  }
  var WebViewImmersiveMode;
  (function(WebViewImmersiveMode2) {
    WebViewImmersiveMode2[WebViewImmersiveMode2["UNSPECIFIED"] = 0] = "UNSPECIFIED";
    WebViewImmersiveMode2[WebViewImmersiveMode2["INLINE_MODE"] = 1] = "INLINE_MODE";
    WebViewImmersiveMode2[WebViewImmersiveMode2["IMMERSIVE_MODE"] = 2] = "IMMERSIVE_MODE";
    WebViewImmersiveMode2[WebViewImmersiveMode2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(WebViewImmersiveMode || (WebViewImmersiveMode = {}));
  function webViewImmersiveModeFromJSON(object) {
    switch (object) {
      case 0:
      case "UNSPECIFIED":
        return WebViewImmersiveMode.UNSPECIFIED;
      case 1:
      case "INLINE_MODE":
        return WebViewImmersiveMode.INLINE_MODE;
      case 2:
      case "IMMERSIVE_MODE":
        return WebViewImmersiveMode.IMMERSIVE_MODE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return WebViewImmersiveMode.UNRECOGNIZED;
    }
  }
  function webViewImmersiveModeToJSON(object) {
    switch (object) {
      case WebViewImmersiveMode.UNSPECIFIED:
        return 0;
      case WebViewImmersiveMode.INLINE_MODE:
        return 1;
      case WebViewImmersiveMode.IMMERSIVE_MODE:
        return 2;
      case WebViewImmersiveMode.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseWebViewImmersiveModeEffect() {
    return { immersiveMode: 0, entryUrl: void 0 };
  }
  const WebViewImmersiveModeEffect = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewImmersiveModeEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.immersiveMode !== 0) {
        writer2.uint32(8).int32(message.immersiveMode);
      }
      if (message.entryUrl !== void 0) {
        writer2.uint32(18).string(message.entryUrl);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewImmersiveModeEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.immersiveMode = reader2.int32();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.entryUrl = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        immersiveMode: isSet$j(object.immersiveMode) ? webViewImmersiveModeFromJSON(object.immersiveMode) : 0,
        entryUrl: isSet$j(object.entryUrl) ? globalThis.String(object.entryUrl) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.immersiveMode !== 0) {
        obj.immersiveMode = webViewImmersiveModeToJSON(message.immersiveMode);
      }
      if (message.entryUrl !== void 0) {
        obj.entryUrl = message.entryUrl;
      }
      return obj;
    },
    create(base) {
      return WebViewImmersiveModeEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewImmersiveModeEffect();
      message.immersiveMode = object.immersiveMode ?? 0;
      message.entryUrl = object.entryUrl ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewImmersiveModeEffect.$type, WebViewImmersiveModeEffect);
  function isSet$j(value) {
    return value !== null && value !== void 0;
  }
  function createBaseWebViewShareEffect() {
    return { userData: void 0, title: void 0, text: void 0, appIconUri: void 0 };
  }
  const WebViewShareEffect = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewShareEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.userData !== void 0) {
        writer2.uint32(10).string(message.userData);
      }
      if (message.title !== void 0) {
        writer2.uint32(18).string(message.title);
      }
      if (message.text !== void 0) {
        writer2.uint32(26).string(message.text);
      }
      if (message.appIconUri !== void 0) {
        writer2.uint32(34).string(message.appIconUri);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewShareEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.userData = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.title = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.text = reader2.string();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.appIconUri = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        userData: isSet$i(object.userData) ? globalThis.String(object.userData) : void 0,
        title: isSet$i(object.title) ? globalThis.String(object.title) : void 0,
        text: isSet$i(object.text) ? globalThis.String(object.text) : void 0,
        appIconUri: isSet$i(object.appIconUri) ? globalThis.String(object.appIconUri) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.userData !== void 0) {
        obj.userData = message.userData;
      }
      if (message.title !== void 0) {
        obj.title = message.title;
      }
      if (message.text !== void 0) {
        obj.text = message.text;
      }
      if (message.appIconUri !== void 0) {
        obj.appIconUri = message.appIconUri;
      }
      return obj;
    },
    create(base) {
      return WebViewShareEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewShareEffect();
      message.userData = object.userData ?? void 0;
      message.title = object.title ?? void 0;
      message.text = object.text ?? void 0;
      message.appIconUri = object.appIconUri ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewShareEffect.$type, WebViewShareEffect);
  function isSet$i(value) {
    return value !== null && value !== void 0;
  }
  var WebViewInternalMessageScope;
  (function(WebViewInternalMessageScope2) {
    WebViewInternalMessageScope2[WebViewInternalMessageScope2["CLIENT"] = 0] = "CLIENT";
    WebViewInternalMessageScope2[WebViewInternalMessageScope2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(WebViewInternalMessageScope || (WebViewInternalMessageScope = {}));
  function webViewInternalMessageScopeFromJSON(object) {
    switch (object) {
      case 0:
      case "CLIENT":
        return WebViewInternalMessageScope.CLIENT;
      case -1:
      case "UNRECOGNIZED":
      default:
        return WebViewInternalMessageScope.UNRECOGNIZED;
    }
  }
  function webViewInternalMessageScopeToJSON(object) {
    switch (object) {
      case WebViewInternalMessageScope.CLIENT:
        return 0;
      case WebViewInternalMessageScope.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseWebViewInternalMessage() {
    return {
      type: "",
      scope: 0,
      analytics: void 0,
      effect: void 0,
      realtimeEffect: void 0,
      immersiveMode: void 0,
      share: void 0,
      showToast: void 0,
      navigateToUrl: void 0,
      showForm: void 0,
      id: void 0
    };
  }
  const WebViewInternalMessage = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewInternalMessage",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.type !== "") {
        writer2.uint32(10).string(message.type);
      }
      if (message.scope !== 0) {
        writer2.uint32(16).int32(message.scope);
      }
      if (message.analytics !== void 0) {
        Struct.encode(Struct.wrap(message.analytics), writer2.uint32(26).fork()).ldelim();
      }
      if (message.effect !== void 0) {
        Struct.encode(Struct.wrap(message.effect), writer2.uint32(34).fork()).ldelim();
      }
      if (message.realtimeEffect !== void 0) {
        RealtimeSubscriptionsEffect$1.encode(message.realtimeEffect, writer2.uint32(50).fork()).ldelim();
      }
      if (message.immersiveMode !== void 0) {
        WebViewImmersiveModeEffect.encode(message.immersiveMode, writer2.uint32(58).fork()).ldelim();
      }
      if (message.share !== void 0) {
        WebViewShareEffect.encode(message.share, writer2.uint32(66).fork()).ldelim();
      }
      if (message.showToast !== void 0) {
        ShowToastEffect.encode(message.showToast, writer2.uint32(74).fork()).ldelim();
      }
      if (message.navigateToUrl !== void 0) {
        NavigateToUrlEffect.encode(message.navigateToUrl, writer2.uint32(82).fork()).ldelim();
      }
      if (message.showForm !== void 0) {
        ShowFormEffect.encode(message.showForm, writer2.uint32(90).fork()).ldelim();
      }
      if (message.id !== void 0) {
        writer2.uint32(42).string(message.id);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewInternalMessage();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.type = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.scope = reader2.int32();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.analytics = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.effect = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.realtimeEffect = RealtimeSubscriptionsEffect$1.decode(reader2, reader2.uint32());
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.immersiveMode = WebViewImmersiveModeEffect.decode(reader2, reader2.uint32());
            continue;
          case 8:
            if (tag !== 66) {
              break;
            }
            message.share = WebViewShareEffect.decode(reader2, reader2.uint32());
            continue;
          case 9:
            if (tag !== 74) {
              break;
            }
            message.showToast = ShowToastEffect.decode(reader2, reader2.uint32());
            continue;
          case 10:
            if (tag !== 82) {
              break;
            }
            message.navigateToUrl = NavigateToUrlEffect.decode(reader2, reader2.uint32());
            continue;
          case 11:
            if (tag !== 90) {
              break;
            }
            message.showForm = ShowFormEffect.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.id = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        type: isSet$h(object.type) ? globalThis.String(object.type) : "",
        scope: isSet$h(object.scope) ? webViewInternalMessageScopeFromJSON(object.scope) : 0,
        analytics: isObject$6(object.analytics) ? object.analytics : void 0,
        effect: isObject$6(object.effect) ? object.effect : void 0,
        realtimeEffect: isSet$h(object.realtimeEffect) ? RealtimeSubscriptionsEffect$1.fromJSON(object.realtimeEffect) : void 0,
        immersiveMode: isSet$h(object.immersiveMode) ? WebViewImmersiveModeEffect.fromJSON(object.immersiveMode) : void 0,
        share: isSet$h(object.share) ? WebViewShareEffect.fromJSON(object.share) : void 0,
        showToast: isSet$h(object.showToast) ? ShowToastEffect.fromJSON(object.showToast) : void 0,
        navigateToUrl: isSet$h(object.navigateToUrl) ? NavigateToUrlEffect.fromJSON(object.navigateToUrl) : void 0,
        showForm: isSet$h(object.showForm) ? ShowFormEffect.fromJSON(object.showForm) : void 0,
        id: isSet$h(object.id) ? globalThis.String(object.id) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.type !== "") {
        obj.type = message.type;
      }
      if (message.scope !== 0) {
        obj.scope = webViewInternalMessageScopeToJSON(message.scope);
      }
      if (message.analytics !== void 0) {
        obj.analytics = message.analytics;
      }
      if (message.effect !== void 0) {
        obj.effect = message.effect;
      }
      if (message.realtimeEffect !== void 0) {
        obj.realtimeEffect = RealtimeSubscriptionsEffect$1.toJSON(message.realtimeEffect);
      }
      if (message.immersiveMode !== void 0) {
        obj.immersiveMode = WebViewImmersiveModeEffect.toJSON(message.immersiveMode);
      }
      if (message.share !== void 0) {
        obj.share = WebViewShareEffect.toJSON(message.share);
      }
      if (message.showToast !== void 0) {
        obj.showToast = ShowToastEffect.toJSON(message.showToast);
      }
      if (message.navigateToUrl !== void 0) {
        obj.navigateToUrl = NavigateToUrlEffect.toJSON(message.navigateToUrl);
      }
      if (message.showForm !== void 0) {
        obj.showForm = ShowFormEffect.toJSON(message.showForm);
      }
      if (message.id !== void 0) {
        obj.id = message.id;
      }
      return obj;
    },
    create(base) {
      return WebViewInternalMessage.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewInternalMessage();
      message.type = object.type ?? "";
      message.scope = object.scope ?? 0;
      message.analytics = object.analytics ?? void 0;
      message.effect = object.effect ?? void 0;
      message.realtimeEffect = object.realtimeEffect !== void 0 && object.realtimeEffect !== null ? RealtimeSubscriptionsEffect$1.fromPartial(object.realtimeEffect) : void 0;
      message.immersiveMode = object.immersiveMode !== void 0 && object.immersiveMode !== null ? WebViewImmersiveModeEffect.fromPartial(object.immersiveMode) : void 0;
      message.share = object.share !== void 0 && object.share !== null ? WebViewShareEffect.fromPartial(object.share) : void 0;
      message.showToast = object.showToast !== void 0 && object.showToast !== null ? ShowToastEffect.fromPartial(object.showToast) : void 0;
      message.navigateToUrl = object.navigateToUrl !== void 0 && object.navigateToUrl !== null ? NavigateToUrlEffect.fromPartial(object.navigateToUrl) : void 0;
      message.showForm = object.showForm !== void 0 && object.showForm !== null ? ShowFormEffect.fromPartial(object.showForm) : void 0;
      message.id = object.id ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewInternalMessage.$type, WebViewInternalMessage);
  function createBaseWebViewPostMessage() {
    return { type: "", data: void 0 };
  }
  const WebViewPostMessage = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewPostMessage",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.type !== "") {
        writer2.uint32(10).string(message.type);
      }
      if (message.data !== void 0) {
        Struct.encode(Struct.wrap(message.data), writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewPostMessage();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.type = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.data = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        type: isSet$h(object.type) ? globalThis.String(object.type) : "",
        data: isObject$6(object.data) ? object.data : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.type !== "") {
        obj.type = message.type;
      }
      if (message.data !== void 0) {
        obj.data = message.data;
      }
      return obj;
    },
    create(base) {
      return WebViewPostMessage.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewPostMessage();
      message.type = object.type ?? "";
      message.data = object.data ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewPostMessage.$type, WebViewPostMessage);
  function createBaseWebViewAppMessage() {
    return { message: void 0, jsonString: void 0 };
  }
  const WebViewAppMessage = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewAppMessage",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.message !== void 0) {
        Value.encode(Value.wrap(message.message), writer2.uint32(10).fork()).ldelim();
      }
      if (message.jsonString !== void 0) {
        writer2.uint32(18).string(message.jsonString);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewAppMessage();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.message = Value.unwrap(Value.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.jsonString = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        message: isSet$h(object?.message) ? object.message : void 0,
        jsonString: isSet$h(object.jsonString) ? globalThis.String(object.jsonString) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.message !== void 0) {
        obj.message = message.message;
      }
      if (message.jsonString !== void 0) {
        obj.jsonString = message.jsonString;
      }
      return obj;
    },
    create(base) {
      return WebViewAppMessage.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewAppMessage();
      message.message = object.message ?? void 0;
      message.jsonString = object.jsonString ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewAppMessage.$type, WebViewAppMessage);
  function createBaseWebViewStateMessage() {
    return { state: void 0 };
  }
  const WebViewStateMessage = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewStateMessage",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.state !== void 0) {
        Struct.encode(Struct.wrap(message.state), writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewStateMessage();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.state = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { state: isObject$6(object.state) ? object.state : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.state !== void 0) {
        obj.state = message.state;
      }
      return obj;
    },
    create(base) {
      return WebViewStateMessage.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewStateMessage();
      message.state = object.state ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewStateMessage.$type, WebViewStateMessage);
  function createBaseWebViewPostMessageEffect() {
    return { webViewId: "", app: void 0, state: void 0 };
  }
  const WebViewPostMessageEffect = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewPostMessageEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.webViewId !== "") {
        writer2.uint32(10).string(message.webViewId);
      }
      if (message.app !== void 0) {
        WebViewAppMessage.encode(message.app, writer2.uint32(18).fork()).ldelim();
      }
      if (message.state !== void 0) {
        WebViewStateMessage.encode(message.state, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewPostMessageEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.webViewId = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.app = WebViewAppMessage.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.state = WebViewStateMessage.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        webViewId: isSet$h(object.webViewId) ? globalThis.String(object.webViewId) : "",
        app: isSet$h(object.app) ? WebViewAppMessage.fromJSON(object.app) : void 0,
        state: isSet$h(object.state) ? WebViewStateMessage.fromJSON(object.state) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.webViewId !== "") {
        obj.webViewId = message.webViewId;
      }
      if (message.app !== void 0) {
        obj.app = WebViewAppMessage.toJSON(message.app);
      }
      if (message.state !== void 0) {
        obj.state = WebViewStateMessage.toJSON(message.state);
      }
      return obj;
    },
    create(base) {
      return WebViewPostMessageEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewPostMessageEffect();
      message.webViewId = object.webViewId ?? "";
      message.app = object.app !== void 0 && object.app !== null ? WebViewAppMessage.fromPartial(object.app) : void 0;
      message.state = object.state !== void 0 && object.state !== null ? WebViewStateMessage.fromPartial(object.state) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewPostMessageEffect.$type, WebViewPostMessageEffect);
  function isObject$6(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$h(value) {
    return value !== null && value !== void 0;
  }
  function createBaseWebViewEffect() {
    return { postMessage: void 0, fullscreen: void 0 };
  }
  const WebViewEffect = {
    $type: "devvit.ui.effects.web_view.v1alpha.WebViewEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.postMessage !== void 0) {
        WebViewPostMessageEffect.encode(message.postMessage, writer2.uint32(10).fork()).ldelim();
      }
      if (message.fullscreen !== void 0) {
        WebViewFullScreenVisibilityEffect.encode(message.fullscreen, writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.postMessage = WebViewPostMessageEffect.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.fullscreen = WebViewFullScreenVisibilityEffect.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        postMessage: isSet$g(object.postMessage) ? WebViewPostMessageEffect.fromJSON(object.postMessage) : void 0,
        fullscreen: isSet$g(object.fullscreen) ? WebViewFullScreenVisibilityEffect.fromJSON(object.fullscreen) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.postMessage !== void 0) {
        obj.postMessage = WebViewPostMessageEffect.toJSON(message.postMessage);
      }
      if (message.fullscreen !== void 0) {
        obj.fullscreen = WebViewFullScreenVisibilityEffect.toJSON(message.fullscreen);
      }
      return obj;
    },
    create(base) {
      return WebViewEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewEffect();
      message.postMessage = object.postMessage !== void 0 && object.postMessage !== null ? WebViewPostMessageEffect.fromPartial(object.postMessage) : void 0;
      message.fullscreen = object.fullscreen !== void 0 && object.fullscreen !== null ? WebViewFullScreenVisibilityEffect.fromPartial(object.fullscreen) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewEffect.$type, WebViewEffect);
  function isSet$g(value) {
    return value !== null && value !== void 0;
  }
  /**
   * @license
   * Copyright 2009 The Closure Library Authors
   * Copyright 2020 Daniel Wirtz / The long.js Authors.
   *
   * Licensed under the Apache License, Version 2.0 (the "License");
   * you may not use this file except in compliance with the License.
   * You may obtain a copy of the License at
   *
   *     http://www.apache.org/licenses/LICENSE-2.0
   *
   * Unless required by applicable law or agreed to in writing, software
   * distributed under the License is distributed on an "AS IS" BASIS,
   * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   * See the License for the specific language governing permissions and
   * limitations under the License.
   *
   * SPDX-License-Identifier: Apache-2.0
   */
  var wasm = null;
  try {
    wasm = new WebAssembly.Instance(
      new WebAssembly.Module(
        new Uint8Array([
          // \0asm
          0,
          97,
          115,
          109,
          // version 1
          1,
          0,
          0,
          0,
          // section "type"
          1,
          13,
          2,
          // 0, () => i32
          96,
          0,
          1,
          127,
          // 1, (i32, i32, i32, i32) => i32
          96,
          4,
          127,
          127,
          127,
          127,
          1,
          127,
          // section "function"
          3,
          7,
          6,
          // 0, type 0
          0,
          // 1, type 1
          1,
          // 2, type 1
          1,
          // 3, type 1
          1,
          // 4, type 1
          1,
          // 5, type 1
          1,
          // section "global"
          6,
          6,
          1,
          // 0, "high", mutable i32
          127,
          1,
          65,
          0,
          11,
          // section "export"
          7,
          50,
          6,
          // 0, "mul"
          3,
          109,
          117,
          108,
          0,
          1,
          // 1, "div_s"
          5,
          100,
          105,
          118,
          95,
          115,
          0,
          2,
          // 2, "div_u"
          5,
          100,
          105,
          118,
          95,
          117,
          0,
          3,
          // 3, "rem_s"
          5,
          114,
          101,
          109,
          95,
          115,
          0,
          4,
          // 4, "rem_u"
          5,
          114,
          101,
          109,
          95,
          117,
          0,
          5,
          // 5, "get_high"
          8,
          103,
          101,
          116,
          95,
          104,
          105,
          103,
          104,
          0,
          0,
          // section "code"
          10,
          191,
          1,
          6,
          // 0, "get_high"
          4,
          0,
          35,
          0,
          11,
          // 1, "mul"
          36,
          1,
          1,
          126,
          32,
          0,
          173,
          32,
          1,
          173,
          66,
          32,
          134,
          132,
          32,
          2,
          173,
          32,
          3,
          173,
          66,
          32,
          134,
          132,
          126,
          34,
          4,
          66,
          32,
          135,
          167,
          36,
          0,
          32,
          4,
          167,
          11,
          // 2, "div_s"
          36,
          1,
          1,
          126,
          32,
          0,
          173,
          32,
          1,
          173,
          66,
          32,
          134,
          132,
          32,
          2,
          173,
          32,
          3,
          173,
          66,
          32,
          134,
          132,
          127,
          34,
          4,
          66,
          32,
          135,
          167,
          36,
          0,
          32,
          4,
          167,
          11,
          // 3, "div_u"
          36,
          1,
          1,
          126,
          32,
          0,
          173,
          32,
          1,
          173,
          66,
          32,
          134,
          132,
          32,
          2,
          173,
          32,
          3,
          173,
          66,
          32,
          134,
          132,
          128,
          34,
          4,
          66,
          32,
          135,
          167,
          36,
          0,
          32,
          4,
          167,
          11,
          // 4, "rem_s"
          36,
          1,
          1,
          126,
          32,
          0,
          173,
          32,
          1,
          173,
          66,
          32,
          134,
          132,
          32,
          2,
          173,
          32,
          3,
          173,
          66,
          32,
          134,
          132,
          129,
          34,
          4,
          66,
          32,
          135,
          167,
          36,
          0,
          32,
          4,
          167,
          11,
          // 5, "rem_u"
          36,
          1,
          1,
          126,
          32,
          0,
          173,
          32,
          1,
          173,
          66,
          32,
          134,
          132,
          32,
          2,
          173,
          32,
          3,
          173,
          66,
          32,
          134,
          132,
          130,
          34,
          4,
          66,
          32,
          135,
          167,
          36,
          0,
          32,
          4,
          167,
          11
        ])
      ),
      {}
    ).exports;
  } catch {
  }
  function Long(low, high, unsigned) {
    this.low = low | 0;
    this.high = high | 0;
    this.unsigned = !!unsigned;
  }
  Long.prototype.__isLong__;
  Object.defineProperty(Long.prototype, "__isLong__", { value: true });
  function isLong(obj) {
    return (obj && obj["__isLong__"]) === true;
  }
  function ctz32(value) {
    var c = Math.clz32(value & -value);
    return value ? 31 - c : c;
  }
  Long.isLong = isLong;
  var INT_CACHE = {};
  var UINT_CACHE = {};
  function fromInt(value, unsigned) {
    var obj, cachedObj, cache;
    if (unsigned) {
      value >>>= 0;
      if (cache = 0 <= value && value < 256) {
        cachedObj = UINT_CACHE[value];
        if (cachedObj) return cachedObj;
      }
      obj = fromBits(value, 0, true);
      if (cache) UINT_CACHE[value] = obj;
      return obj;
    } else {
      value |= 0;
      if (cache = -128 <= value && value < 128) {
        cachedObj = INT_CACHE[value];
        if (cachedObj) return cachedObj;
      }
      obj = fromBits(value, value < 0 ? -1 : 0, false);
      if (cache) INT_CACHE[value] = obj;
      return obj;
    }
  }
  Long.fromInt = fromInt;
  function fromNumber(value, unsigned) {
    if (isNaN(value)) return unsigned ? UZERO : ZERO;
    if (unsigned) {
      if (value < 0) return UZERO;
      if (value >= TWO_PWR_64_DBL) return MAX_UNSIGNED_VALUE;
    } else {
      if (value <= -TWO_PWR_63_DBL) return MIN_VALUE;
      if (value + 1 >= TWO_PWR_63_DBL) return MAX_VALUE;
    }
    if (value < 0) return fromNumber(-value, unsigned).neg();
    return fromBits(
      value % TWO_PWR_32_DBL | 0,
      value / TWO_PWR_32_DBL | 0,
      unsigned
    );
  }
  Long.fromNumber = fromNumber;
  function fromBits(lowBits, highBits, unsigned) {
    return new Long(lowBits, highBits, unsigned);
  }
  Long.fromBits = fromBits;
  var pow_dbl = Math.pow;
  function fromString(str, unsigned, radix) {
    if (str.length === 0) throw Error("empty string");
    if (typeof unsigned === "number") {
      radix = unsigned;
      unsigned = false;
    } else {
      unsigned = !!unsigned;
    }
    if (str === "NaN" || str === "Infinity" || str === "+Infinity" || str === "-Infinity")
      return unsigned ? UZERO : ZERO;
    radix = radix || 10;
    if (radix < 2 || 36 < radix) throw RangeError("radix");
    var p;
    if ((p = str.indexOf("-")) > 0) throw Error("interior hyphen");
    else if (p === 0) {
      return fromString(str.substring(1), unsigned, radix).neg();
    }
    var radixToPower = fromNumber(pow_dbl(radix, 8));
    var result2 = ZERO;
    for (var i = 0; i < str.length; i += 8) {
      var size = Math.min(8, str.length - i), value = parseInt(str.substring(i, i + size), radix);
      if (size < 8) {
        var power = fromNumber(pow_dbl(radix, size));
        result2 = result2.mul(power).add(fromNumber(value));
      } else {
        result2 = result2.mul(radixToPower);
        result2 = result2.add(fromNumber(value));
      }
    }
    result2.unsigned = unsigned;
    return result2;
  }
  Long.fromString = fromString;
  function fromValue(val, unsigned) {
    if (typeof val === "number") return fromNumber(val, unsigned);
    if (typeof val === "string") return fromString(val, unsigned);
    return fromBits(
      val.low,
      val.high,
      typeof unsigned === "boolean" ? unsigned : val.unsigned
    );
  }
  Long.fromValue = fromValue;
  var TWO_PWR_16_DBL = 1 << 16;
  var TWO_PWR_24_DBL = 1 << 24;
  var TWO_PWR_32_DBL = TWO_PWR_16_DBL * TWO_PWR_16_DBL;
  var TWO_PWR_64_DBL = TWO_PWR_32_DBL * TWO_PWR_32_DBL;
  var TWO_PWR_63_DBL = TWO_PWR_64_DBL / 2;
  var TWO_PWR_24 = fromInt(TWO_PWR_24_DBL);
  var ZERO = fromInt(0);
  Long.ZERO = ZERO;
  var UZERO = fromInt(0, true);
  Long.UZERO = UZERO;
  var ONE = fromInt(1);
  Long.ONE = ONE;
  var UONE = fromInt(1, true);
  Long.UONE = UONE;
  var NEG_ONE = fromInt(-1);
  Long.NEG_ONE = NEG_ONE;
  var MAX_VALUE = fromBits(4294967295 | 0, 2147483647 | 0, false);
  Long.MAX_VALUE = MAX_VALUE;
  var MAX_UNSIGNED_VALUE = fromBits(4294967295 | 0, 4294967295 | 0, true);
  Long.MAX_UNSIGNED_VALUE = MAX_UNSIGNED_VALUE;
  var MIN_VALUE = fromBits(0, 2147483648 | 0, false);
  Long.MIN_VALUE = MIN_VALUE;
  var LongPrototype = Long.prototype;
  LongPrototype.toInt = function toInt() {
    return this.unsigned ? this.low >>> 0 : this.low;
  };
  LongPrototype.toNumber = function toNumber() {
    if (this.unsigned)
      return (this.high >>> 0) * TWO_PWR_32_DBL + (this.low >>> 0);
    return this.high * TWO_PWR_32_DBL + (this.low >>> 0);
  };
  LongPrototype.toString = function toString(radix) {
    radix = radix || 10;
    if (radix < 2 || 36 < radix) throw RangeError("radix");
    if (this.isZero()) return "0";
    if (this.isNegative()) {
      if (this.eq(MIN_VALUE)) {
        var radixLong = fromNumber(radix), div = this.div(radixLong), rem1 = div.mul(radixLong).sub(this);
        return div.toString(radix) + rem1.toInt().toString(radix);
      } else return "-" + this.neg().toString(radix);
    }
    var radixToPower = fromNumber(pow_dbl(radix, 6), this.unsigned), rem = this;
    var result2 = "";
    while (true) {
      var remDiv = rem.div(radixToPower), intval = rem.sub(remDiv.mul(radixToPower)).toInt() >>> 0, digits = intval.toString(radix);
      rem = remDiv;
      if (rem.isZero()) return digits + result2;
      else {
        while (digits.length < 6) digits = "0" + digits;
        result2 = "" + digits + result2;
      }
    }
  };
  LongPrototype.getHighBits = function getHighBits() {
    return this.high;
  };
  LongPrototype.getHighBitsUnsigned = function getHighBitsUnsigned() {
    return this.high >>> 0;
  };
  LongPrototype.getLowBits = function getLowBits() {
    return this.low;
  };
  LongPrototype.getLowBitsUnsigned = function getLowBitsUnsigned() {
    return this.low >>> 0;
  };
  LongPrototype.getNumBitsAbs = function getNumBitsAbs() {
    if (this.isNegative())
      return this.eq(MIN_VALUE) ? 64 : this.neg().getNumBitsAbs();
    var val = this.high != 0 ? this.high : this.low;
    for (var bit = 31; bit > 0; bit--) if ((val & 1 << bit) != 0) break;
    return this.high != 0 ? bit + 33 : bit + 1;
  };
  LongPrototype.isSafeInteger = function isSafeInteger() {
    var top11Bits = this.high >> 21;
    if (!top11Bits) return true;
    if (this.unsigned) return false;
    return top11Bits === -1 && !(this.low === 0 && this.high === -2097152);
  };
  LongPrototype.isZero = function isZero() {
    return this.high === 0 && this.low === 0;
  };
  LongPrototype.eqz = LongPrototype.isZero;
  LongPrototype.isNegative = function isNegative() {
    return !this.unsigned && this.high < 0;
  };
  LongPrototype.isPositive = function isPositive() {
    return this.unsigned || this.high >= 0;
  };
  LongPrototype.isOdd = function isOdd() {
    return (this.low & 1) === 1;
  };
  LongPrototype.isEven = function isEven() {
    return (this.low & 1) === 0;
  };
  LongPrototype.equals = function equals(other) {
    if (!isLong(other)) other = fromValue(other);
    if (this.unsigned !== other.unsigned && this.high >>> 31 === 1 && other.high >>> 31 === 1)
      return false;
    return this.high === other.high && this.low === other.low;
  };
  LongPrototype.eq = LongPrototype.equals;
  LongPrototype.notEquals = function notEquals(other) {
    return !this.eq(
      /* validates */
      other
    );
  };
  LongPrototype.neq = LongPrototype.notEquals;
  LongPrototype.ne = LongPrototype.notEquals;
  LongPrototype.lessThan = function lessThan(other) {
    return this.comp(
      /* validates */
      other
    ) < 0;
  };
  LongPrototype.lt = LongPrototype.lessThan;
  LongPrototype.lessThanOrEqual = function lessThanOrEqual(other) {
    return this.comp(
      /* validates */
      other
    ) <= 0;
  };
  LongPrototype.lte = LongPrototype.lessThanOrEqual;
  LongPrototype.le = LongPrototype.lessThanOrEqual;
  LongPrototype.greaterThan = function greaterThan(other) {
    return this.comp(
      /* validates */
      other
    ) > 0;
  };
  LongPrototype.gt = LongPrototype.greaterThan;
  LongPrototype.greaterThanOrEqual = function greaterThanOrEqual(other) {
    return this.comp(
      /* validates */
      other
    ) >= 0;
  };
  LongPrototype.gte = LongPrototype.greaterThanOrEqual;
  LongPrototype.ge = LongPrototype.greaterThanOrEqual;
  LongPrototype.compare = function compare(other) {
    if (!isLong(other)) other = fromValue(other);
    if (this.eq(other)) return 0;
    var thisNeg = this.isNegative(), otherNeg = other.isNegative();
    if (thisNeg && !otherNeg) return -1;
    if (!thisNeg && otherNeg) return 1;
    if (!this.unsigned) return this.sub(other).isNegative() ? -1 : 1;
    return other.high >>> 0 > this.high >>> 0 || other.high === this.high && other.low >>> 0 > this.low >>> 0 ? -1 : 1;
  };
  LongPrototype.comp = LongPrototype.compare;
  LongPrototype.negate = function negate() {
    if (!this.unsigned && this.eq(MIN_VALUE)) return MIN_VALUE;
    return this.not().add(ONE);
  };
  LongPrototype.neg = LongPrototype.negate;
  LongPrototype.add = function add(addend) {
    if (!isLong(addend)) addend = fromValue(addend);
    var a48 = this.high >>> 16;
    var a32 = this.high & 65535;
    var a16 = this.low >>> 16;
    var a00 = this.low & 65535;
    var b48 = addend.high >>> 16;
    var b32 = addend.high & 65535;
    var b16 = addend.low >>> 16;
    var b00 = addend.low & 65535;
    var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
    c00 += a00 + b00;
    c16 += c00 >>> 16;
    c00 &= 65535;
    c16 += a16 + b16;
    c32 += c16 >>> 16;
    c16 &= 65535;
    c32 += a32 + b32;
    c48 += c32 >>> 16;
    c32 &= 65535;
    c48 += a48 + b48;
    c48 &= 65535;
    return fromBits(c16 << 16 | c00, c48 << 16 | c32, this.unsigned);
  };
  LongPrototype.subtract = function subtract(subtrahend) {
    if (!isLong(subtrahend)) subtrahend = fromValue(subtrahend);
    return this.add(subtrahend.neg());
  };
  LongPrototype.sub = LongPrototype.subtract;
  LongPrototype.multiply = function multiply(multiplier) {
    if (this.isZero()) return this;
    if (!isLong(multiplier)) multiplier = fromValue(multiplier);
    if (wasm) {
      var low = wasm["mul"](this.low, this.high, multiplier.low, multiplier.high);
      return fromBits(low, wasm["get_high"](), this.unsigned);
    }
    if (multiplier.isZero()) return this.unsigned ? UZERO : ZERO;
    if (this.eq(MIN_VALUE)) return multiplier.isOdd() ? MIN_VALUE : ZERO;
    if (multiplier.eq(MIN_VALUE)) return this.isOdd() ? MIN_VALUE : ZERO;
    if (this.isNegative()) {
      if (multiplier.isNegative()) return this.neg().mul(multiplier.neg());
      else return this.neg().mul(multiplier).neg();
    } else if (multiplier.isNegative()) return this.mul(multiplier.neg()).neg();
    if (this.lt(TWO_PWR_24) && multiplier.lt(TWO_PWR_24))
      return fromNumber(this.toNumber() * multiplier.toNumber(), this.unsigned);
    var a48 = this.high >>> 16;
    var a32 = this.high & 65535;
    var a16 = this.low >>> 16;
    var a00 = this.low & 65535;
    var b48 = multiplier.high >>> 16;
    var b32 = multiplier.high & 65535;
    var b16 = multiplier.low >>> 16;
    var b00 = multiplier.low & 65535;
    var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
    c00 += a00 * b00;
    c16 += c00 >>> 16;
    c00 &= 65535;
    c16 += a16 * b00;
    c32 += c16 >>> 16;
    c16 &= 65535;
    c16 += a00 * b16;
    c32 += c16 >>> 16;
    c16 &= 65535;
    c32 += a32 * b00;
    c48 += c32 >>> 16;
    c32 &= 65535;
    c32 += a16 * b16;
    c48 += c32 >>> 16;
    c32 &= 65535;
    c32 += a00 * b32;
    c48 += c32 >>> 16;
    c32 &= 65535;
    c48 += a48 * b00 + a32 * b16 + a16 * b32 + a00 * b48;
    c48 &= 65535;
    return fromBits(c16 << 16 | c00, c48 << 16 | c32, this.unsigned);
  };
  LongPrototype.mul = LongPrototype.multiply;
  LongPrototype.divide = function divide(divisor) {
    if (!isLong(divisor)) divisor = fromValue(divisor);
    if (divisor.isZero()) throw Error("division by zero");
    if (wasm) {
      if (!this.unsigned && this.high === -2147483648 && divisor.low === -1 && divisor.high === -1) {
        return this;
      }
      var low = (this.unsigned ? wasm["div_u"] : wasm["div_s"])(
        this.low,
        this.high,
        divisor.low,
        divisor.high
      );
      return fromBits(low, wasm["get_high"](), this.unsigned);
    }
    if (this.isZero()) return this.unsigned ? UZERO : ZERO;
    var approx, rem, res;
    if (!this.unsigned) {
      if (this.eq(MIN_VALUE)) {
        if (divisor.eq(ONE) || divisor.eq(NEG_ONE))
          return MIN_VALUE;
        else if (divisor.eq(MIN_VALUE)) return ONE;
        else {
          var halfThis = this.shr(1);
          approx = halfThis.div(divisor).shl(1);
          if (approx.eq(ZERO)) {
            return divisor.isNegative() ? ONE : NEG_ONE;
          } else {
            rem = this.sub(divisor.mul(approx));
            res = approx.add(rem.div(divisor));
            return res;
          }
        }
      } else if (divisor.eq(MIN_VALUE)) return this.unsigned ? UZERO : ZERO;
      if (this.isNegative()) {
        if (divisor.isNegative()) return this.neg().div(divisor.neg());
        return this.neg().div(divisor).neg();
      } else if (divisor.isNegative()) return this.div(divisor.neg()).neg();
      res = ZERO;
    } else {
      if (!divisor.unsigned) divisor = divisor.toUnsigned();
      if (divisor.gt(this)) return UZERO;
      if (divisor.gt(this.shru(1)))
        return UONE;
      res = UZERO;
    }
    rem = this;
    while (rem.gte(divisor)) {
      approx = Math.max(1, Math.floor(rem.toNumber() / divisor.toNumber()));
      var log2 = Math.ceil(Math.log(approx) / Math.LN2), delta = log2 <= 48 ? 1 : pow_dbl(2, log2 - 48), approxRes = fromNumber(approx), approxRem = approxRes.mul(divisor);
      while (approxRem.isNegative() || approxRem.gt(rem)) {
        approx -= delta;
        approxRes = fromNumber(approx, this.unsigned);
        approxRem = approxRes.mul(divisor);
      }
      if (approxRes.isZero()) approxRes = ONE;
      res = res.add(approxRes);
      rem = rem.sub(approxRem);
    }
    return res;
  };
  LongPrototype.div = LongPrototype.divide;
  LongPrototype.modulo = function modulo(divisor) {
    if (!isLong(divisor)) divisor = fromValue(divisor);
    if (wasm) {
      var low = (this.unsigned ? wasm["rem_u"] : wasm["rem_s"])(
        this.low,
        this.high,
        divisor.low,
        divisor.high
      );
      return fromBits(low, wasm["get_high"](), this.unsigned);
    }
    return this.sub(this.div(divisor).mul(divisor));
  };
  LongPrototype.mod = LongPrototype.modulo;
  LongPrototype.rem = LongPrototype.modulo;
  LongPrototype.not = function not() {
    return fromBits(~this.low, ~this.high, this.unsigned);
  };
  LongPrototype.countLeadingZeros = function countLeadingZeros() {
    return this.high ? Math.clz32(this.high) : Math.clz32(this.low) + 32;
  };
  LongPrototype.clz = LongPrototype.countLeadingZeros;
  LongPrototype.countTrailingZeros = function countTrailingZeros() {
    return this.low ? ctz32(this.low) : ctz32(this.high) + 32;
  };
  LongPrototype.ctz = LongPrototype.countTrailingZeros;
  LongPrototype.and = function and(other) {
    if (!isLong(other)) other = fromValue(other);
    return fromBits(this.low & other.low, this.high & other.high, this.unsigned);
  };
  LongPrototype.or = function or(other) {
    if (!isLong(other)) other = fromValue(other);
    return fromBits(this.low | other.low, this.high | other.high, this.unsigned);
  };
  LongPrototype.xor = function xor(other) {
    if (!isLong(other)) other = fromValue(other);
    return fromBits(this.low ^ other.low, this.high ^ other.high, this.unsigned);
  };
  LongPrototype.shiftLeft = function shiftLeft(numBits) {
    if (isLong(numBits)) numBits = numBits.toInt();
    if ((numBits &= 63) === 0) return this;
    else if (numBits < 32)
      return fromBits(
        this.low << numBits,
        this.high << numBits | this.low >>> 32 - numBits,
        this.unsigned
      );
    else return fromBits(0, this.low << numBits - 32, this.unsigned);
  };
  LongPrototype.shl = LongPrototype.shiftLeft;
  LongPrototype.shiftRight = function shiftRight(numBits) {
    if (isLong(numBits)) numBits = numBits.toInt();
    if ((numBits &= 63) === 0) return this;
    else if (numBits < 32)
      return fromBits(
        this.low >>> numBits | this.high << 32 - numBits,
        this.high >> numBits,
        this.unsigned
      );
    else
      return fromBits(
        this.high >> numBits - 32,
        this.high >= 0 ? 0 : -1,
        this.unsigned
      );
  };
  LongPrototype.shr = LongPrototype.shiftRight;
  LongPrototype.shiftRightUnsigned = function shiftRightUnsigned(numBits) {
    if (isLong(numBits)) numBits = numBits.toInt();
    if ((numBits &= 63) === 0) return this;
    if (numBits < 32)
      return fromBits(
        this.low >>> numBits | this.high << 32 - numBits,
        this.high >>> numBits,
        this.unsigned
      );
    if (numBits === 32) return fromBits(this.high, 0, this.unsigned);
    return fromBits(this.high >>> numBits - 32, 0, this.unsigned);
  };
  LongPrototype.shru = LongPrototype.shiftRightUnsigned;
  LongPrototype.shr_u = LongPrototype.shiftRightUnsigned;
  LongPrototype.rotateLeft = function rotateLeft(numBits) {
    var b;
    if (isLong(numBits)) numBits = numBits.toInt();
    if ((numBits &= 63) === 0) return this;
    if (numBits === 32) return fromBits(this.high, this.low, this.unsigned);
    if (numBits < 32) {
      b = 32 - numBits;
      return fromBits(
        this.low << numBits | this.high >>> b,
        this.high << numBits | this.low >>> b,
        this.unsigned
      );
    }
    numBits -= 32;
    b = 32 - numBits;
    return fromBits(
      this.high << numBits | this.low >>> b,
      this.low << numBits | this.high >>> b,
      this.unsigned
    );
  };
  LongPrototype.rotl = LongPrototype.rotateLeft;
  LongPrototype.rotateRight = function rotateRight(numBits) {
    var b;
    if (isLong(numBits)) numBits = numBits.toInt();
    if ((numBits &= 63) === 0) return this;
    if (numBits === 32) return fromBits(this.high, this.low, this.unsigned);
    if (numBits < 32) {
      b = 32 - numBits;
      return fromBits(
        this.high << b | this.low >>> numBits,
        this.low << b | this.high >>> numBits,
        this.unsigned
      );
    }
    numBits -= 32;
    b = 32 - numBits;
    return fromBits(
      this.low << b | this.high >>> numBits,
      this.high << b | this.low >>> numBits,
      this.unsigned
    );
  };
  LongPrototype.rotr = LongPrototype.rotateRight;
  LongPrototype.toSigned = function toSigned() {
    if (!this.unsigned) return this;
    return fromBits(this.low, this.high, false);
  };
  LongPrototype.toUnsigned = function toUnsigned() {
    if (this.unsigned) return this;
    return fromBits(this.low, this.high, true);
  };
  LongPrototype.toBytes = function toBytes(le) {
    return le ? this.toBytesLE() : this.toBytesBE();
  };
  LongPrototype.toBytesLE = function toBytesLE() {
    var hi = this.high, lo = this.low;
    return [
      lo & 255,
      lo >>> 8 & 255,
      lo >>> 16 & 255,
      lo >>> 24,
      hi & 255,
      hi >>> 8 & 255,
      hi >>> 16 & 255,
      hi >>> 24
    ];
  };
  LongPrototype.toBytesBE = function toBytesBE() {
    var hi = this.high, lo = this.low;
    return [
      hi >>> 24,
      hi >>> 16 & 255,
      hi >>> 8 & 255,
      hi & 255,
      lo >>> 24,
      lo >>> 16 & 255,
      lo >>> 8 & 255,
      lo & 255
    ];
  };
  Long.fromBytes = function fromBytes(bytes, unsigned, le) {
    return le ? Long.fromBytesLE(bytes, unsigned) : Long.fromBytesBE(bytes, unsigned);
  };
  Long.fromBytesLE = function fromBytesLE(bytes, unsigned) {
    return new Long(
      bytes[0] | bytes[1] << 8 | bytes[2] << 16 | bytes[3] << 24,
      bytes[4] | bytes[5] << 8 | bytes[6] << 16 | bytes[7] << 24,
      unsigned
    );
  };
  Long.fromBytesBE = function fromBytesBE(bytes, unsigned) {
    return new Long(
      bytes[4] << 24 | bytes[5] << 16 | bytes[6] << 8 | bytes[7],
      bytes[0] << 24 | bytes[1] << 16 | bytes[2] << 8 | bytes[3],
      unsigned
    );
  };
  if (typeof BigInt === "function") {
    Long.fromBigInt = function fromBigInt(value, unsigned) {
      var lowBits = Number(BigInt.asIntN(32, value));
      var highBits = Number(BigInt.asIntN(32, value >> BigInt(32)));
      return fromBits(lowBits, highBits, unsigned);
    };
    Long.fromValue = function fromValueWithBigInt(value, unsigned) {
      if (typeof value === "bigint") return Long.fromBigInt(value, unsigned);
      return fromValue(value, unsigned);
    };
    LongPrototype.toBigInt = function toBigInt() {
      var lowBigInt = BigInt(this.low >>> 0);
      var highBigInt = BigInt(this.unsigned ? this.high >>> 0 : this.high);
      return highBigInt << BigInt(32) | lowBigInt;
    };
  }
  function createBaseTimestamp() {
    return { seconds: 0, nanos: 0 };
  }
  const Timestamp = {
    $type: "google.protobuf.Timestamp",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.seconds !== 0) {
        writer2.uint32(8).int64(message.seconds);
      }
      if (message.nanos !== 0) {
        writer2.uint32(16).int32(message.nanos);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseTimestamp();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.seconds = longToNumber$2(reader2.int64());
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.nanos = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        seconds: isSet$f(object.seconds) ? globalThis.Number(object.seconds) : 0,
        nanos: isSet$f(object.nanos) ? globalThis.Number(object.nanos) : 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.seconds !== 0) {
        obj.seconds = Math.round(message.seconds);
      }
      if (message.nanos !== 0) {
        obj.nanos = Math.round(message.nanos);
      }
      return obj;
    },
    create(base) {
      return Timestamp.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseTimestamp();
      message.seconds = object.seconds ?? 0;
      message.nanos = object.nanos ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(Timestamp.$type, Timestamp);
  function longToNumber$2(long) {
    if (long.gt(globalThis.Number.MAX_SAFE_INTEGER)) {
      throw new globalThis.Error("Value is larger than Number.MAX_SAFE_INTEGER");
    }
    return long.toNumber();
  }
  if (_m0.util.Long !== Long) {
    _m0.util.Long = Long;
    _m0.configure();
  }
  function isSet$f(value) {
    return value !== null && value !== void 0;
  }
  var ConsentStatus;
  (function(ConsentStatus2) {
    ConsentStatus2[ConsentStatus2["CONSENT_STATUS_UNKNOWN"] = 0] = "CONSENT_STATUS_UNKNOWN";
    ConsentStatus2[ConsentStatus2["REVOKED"] = 1] = "REVOKED";
    ConsentStatus2[ConsentStatus2["GRANTED"] = 2] = "GRANTED";
    ConsentStatus2[ConsentStatus2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(ConsentStatus || (ConsentStatus = {}));
  function consentStatusFromJSON(object) {
    switch (object) {
      case 0:
      case "CONSENT_STATUS_UNKNOWN":
        return ConsentStatus.CONSENT_STATUS_UNKNOWN;
      case 1:
      case "REVOKED":
        return ConsentStatus.REVOKED;
      case 2:
      case "GRANTED":
        return ConsentStatus.GRANTED;
      case -1:
      case "UNRECOGNIZED":
      default:
        return ConsentStatus.UNRECOGNIZED;
    }
  }
  function consentStatusToJSON(object) {
    switch (object) {
      case ConsentStatus.CONSENT_STATUS_UNKNOWN:
        return 0;
      case ConsentStatus.REVOKED:
        return 1;
      case ConsentStatus.GRANTED:
        return 2;
      case ConsentStatus.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var Scope;
  (function(Scope2) {
    Scope2[Scope2["SCOPE_UNKNOWN"] = 0] = "SCOPE_UNKNOWN";
    Scope2[Scope2["SUBMIT_POST"] = 1] = "SUBMIT_POST";
    Scope2[Scope2["SUBMIT_COMMENT"] = 2] = "SUBMIT_COMMENT";
    Scope2[Scope2["SUBSCRIBE_TO_SUBREDDIT"] = 3] = "SUBSCRIBE_TO_SUBREDDIT";
    Scope2[Scope2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(Scope || (Scope = {}));
  function scopeFromJSON(object) {
    switch (object) {
      case 0:
      case "SCOPE_UNKNOWN":
        return Scope.SCOPE_UNKNOWN;
      case 1:
      case "SUBMIT_POST":
        return Scope.SUBMIT_POST;
      case 2:
      case "SUBMIT_COMMENT":
        return Scope.SUBMIT_COMMENT;
      case 3:
      case "SUBSCRIBE_TO_SUBREDDIT":
        return Scope.SUBSCRIBE_TO_SUBREDDIT;
      case -1:
      case "UNRECOGNIZED":
      default:
        return Scope.UNRECOGNIZED;
    }
  }
  function scopeToJSON(object) {
    switch (object) {
      case Scope.SCOPE_UNKNOWN:
        return 0;
      case Scope.SUBMIT_POST:
        return 1;
      case Scope.SUBMIT_COMMENT:
        return 2;
      case Scope.SUBSCRIBE_TO_SUBREDDIT:
        return 3;
      case Scope.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseAppPermission() {
    return { appSlug: "", subredditId: "", scopes: [], consentStatus: 0, updatedAt: void 0 };
  }
  const AppPermission = {
    $type: "reddit.devvit.app_permission.v1.AppPermission",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.appSlug !== "") {
        writer2.uint32(10).string(message.appSlug);
      }
      if (message.subredditId !== "") {
        writer2.uint32(18).string(message.subredditId);
      }
      writer2.uint32(26).fork();
      for (const v of message.scopes) {
        writer2.int32(v);
      }
      writer2.ldelim();
      if (message.consentStatus !== 0) {
        writer2.uint32(32).int32(message.consentStatus);
      }
      if (message.updatedAt !== void 0) {
        Timestamp.encode(toTimestamp(message.updatedAt), writer2.uint32(42).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseAppPermission();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.appSlug = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.subredditId = reader2.string();
            continue;
          case 3:
            if (tag === 24) {
              message.scopes.push(reader2.int32());
              continue;
            }
            if (tag === 26) {
              const end2 = reader2.uint32() + reader2.pos;
              while (reader2.pos < end2) {
                message.scopes.push(reader2.int32());
              }
              continue;
            }
            break;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.consentStatus = reader2.int32();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.updatedAt = fromTimestamp(Timestamp.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        appSlug: isSet$e(object.appSlug) ? globalThis.String(object.appSlug) : "",
        subredditId: isSet$e(object.subredditId) ? globalThis.String(object.subredditId) : "",
        scopes: globalThis.Array.isArray(object?.scopes) ? object.scopes.map((e) => scopeFromJSON(e)) : [],
        consentStatus: isSet$e(object.consentStatus) ? consentStatusFromJSON(object.consentStatus) : 0,
        updatedAt: isSet$e(object.updatedAt) ? fromJsonTimestamp(object.updatedAt) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.appSlug !== "") {
        obj.appSlug = message.appSlug;
      }
      if (message.subredditId !== "") {
        obj.subredditId = message.subredditId;
      }
      if (message.scopes?.length) {
        obj.scopes = message.scopes.map((e) => scopeToJSON(e));
      }
      if (message.consentStatus !== 0) {
        obj.consentStatus = consentStatusToJSON(message.consentStatus);
      }
      if (message.updatedAt !== void 0) {
        obj.updatedAt = message.updatedAt.toISOString();
      }
      return obj;
    },
    create(base) {
      return AppPermission.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseAppPermission();
      message.appSlug = object.appSlug ?? "";
      message.subredditId = object.subredditId ?? "";
      message.scopes = object.scopes?.map((e) => e) || [];
      message.consentStatus = object.consentStatus ?? 0;
      message.updatedAt = object.updatedAt ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(AppPermission.$type, AppPermission);
  function createBaseDevvitApp() {
    return { appSlug: "", appName: "" };
  }
  const DevvitApp = {
    $type: "reddit.devvit.app_permission.v1.DevvitApp",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.appSlug !== "") {
        writer2.uint32(10).string(message.appSlug);
      }
      if (message.appName !== "") {
        writer2.uint32(18).string(message.appName);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseDevvitApp();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.appSlug = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.appName = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        appSlug: isSet$e(object.appSlug) ? globalThis.String(object.appSlug) : "",
        appName: isSet$e(object.appName) ? globalThis.String(object.appName) : ""
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.appSlug !== "") {
        obj.appSlug = message.appSlug;
      }
      if (message.appName !== "") {
        obj.appName = message.appName;
      }
      return obj;
    },
    create(base) {
      return DevvitApp.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseDevvitApp();
      message.appSlug = object.appSlug ?? "";
      message.appName = object.appName ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(DevvitApp.$type, DevvitApp);
  function createBaseGetAppPermissionsByUserIdRequest() {
    return { userId: "" };
  }
  const GetAppPermissionsByUserIdRequest = {
    $type: "reddit.devvit.app_permission.v1.GetAppPermissionsByUserIdRequest",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.userId !== "") {
        writer2.uint32(10).string(message.userId);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseGetAppPermissionsByUserIdRequest();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.userId = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { userId: isSet$e(object.userId) ? globalThis.String(object.userId) : "" };
    },
    toJSON(message) {
      const obj = {};
      if (message.userId !== "") {
        obj.userId = message.userId;
      }
      return obj;
    },
    create(base) {
      return GetAppPermissionsByUserIdRequest.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseGetAppPermissionsByUserIdRequest();
      message.userId = object.userId ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(GetAppPermissionsByUserIdRequest.$type, GetAppPermissionsByUserIdRequest);
  function createBaseGetAppPermissionsByUserIdResponse() {
    return { appPermissions: [], devvitApps: [] };
  }
  const GetAppPermissionsByUserIdResponse = {
    $type: "reddit.devvit.app_permission.v1.GetAppPermissionsByUserIdResponse",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.appPermissions) {
        AppPermission.encode(v, writer2.uint32(10).fork()).ldelim();
      }
      for (const v of message.devvitApps) {
        DevvitApp.encode(v, writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseGetAppPermissionsByUserIdResponse();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.appPermissions.push(AppPermission.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.devvitApps.push(DevvitApp.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        appPermissions: globalThis.Array.isArray(object?.appPermissions) ? object.appPermissions.map((e) => AppPermission.fromJSON(e)) : [],
        devvitApps: globalThis.Array.isArray(object?.devvitApps) ? object.devvitApps.map((e) => DevvitApp.fromJSON(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.appPermissions?.length) {
        obj.appPermissions = message.appPermissions.map((e) => AppPermission.toJSON(e));
      }
      if (message.devvitApps?.length) {
        obj.devvitApps = message.devvitApps.map((e) => DevvitApp.toJSON(e));
      }
      return obj;
    },
    create(base) {
      return GetAppPermissionsByUserIdResponse.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseGetAppPermissionsByUserIdResponse();
      message.appPermissions = object.appPermissions?.map((e) => AppPermission.fromPartial(e)) || [];
      message.devvitApps = object.devvitApps?.map((e) => DevvitApp.fromPartial(e)) || [];
      return message;
    }
  };
  messageTypeRegistry.set(GetAppPermissionsByUserIdResponse.$type, GetAppPermissionsByUserIdResponse);
  function createBaseGrantAppPermissionRequest() {
    return { userId: "", appSlug: "", subredditId: "", scopes: [] };
  }
  const GrantAppPermissionRequest = {
    $type: "reddit.devvit.app_permission.v1.GrantAppPermissionRequest",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.userId !== "") {
        writer2.uint32(10).string(message.userId);
      }
      if (message.appSlug !== "") {
        writer2.uint32(18).string(message.appSlug);
      }
      if (message.subredditId !== "") {
        writer2.uint32(26).string(message.subredditId);
      }
      writer2.uint32(34).fork();
      for (const v of message.scopes) {
        writer2.int32(v);
      }
      writer2.ldelim();
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseGrantAppPermissionRequest();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.userId = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.appSlug = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.subredditId = reader2.string();
            continue;
          case 4:
            if (tag === 32) {
              message.scopes.push(reader2.int32());
              continue;
            }
            if (tag === 34) {
              const end2 = reader2.uint32() + reader2.pos;
              while (reader2.pos < end2) {
                message.scopes.push(reader2.int32());
              }
              continue;
            }
            break;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        userId: isSet$e(object.userId) ? globalThis.String(object.userId) : "",
        appSlug: isSet$e(object.appSlug) ? globalThis.String(object.appSlug) : "",
        subredditId: isSet$e(object.subredditId) ? globalThis.String(object.subredditId) : "",
        scopes: globalThis.Array.isArray(object?.scopes) ? object.scopes.map((e) => scopeFromJSON(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.userId !== "") {
        obj.userId = message.userId;
      }
      if (message.appSlug !== "") {
        obj.appSlug = message.appSlug;
      }
      if (message.subredditId !== "") {
        obj.subredditId = message.subredditId;
      }
      if (message.scopes?.length) {
        obj.scopes = message.scopes.map((e) => scopeToJSON(e));
      }
      return obj;
    },
    create(base) {
      return GrantAppPermissionRequest.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseGrantAppPermissionRequest();
      message.userId = object.userId ?? "";
      message.appSlug = object.appSlug ?? "";
      message.subredditId = object.subredditId ?? "";
      message.scopes = object.scopes?.map((e) => e) || [];
      return message;
    }
  };
  messageTypeRegistry.set(GrantAppPermissionRequest.$type, GrantAppPermissionRequest);
  function createBaseGrantAppPermissionResponse() {
    return { error: void 0 };
  }
  const GrantAppPermissionResponse = {
    $type: "reddit.devvit.app_permission.v1.GrantAppPermissionResponse",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.error !== void 0) {
        ErrorMessage.encode(message.error, writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseGrantAppPermissionResponse();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.error = ErrorMessage.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { error: isSet$e(object.error) ? ErrorMessage.fromJSON(object.error) : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.error !== void 0) {
        obj.error = ErrorMessage.toJSON(message.error);
      }
      return obj;
    },
    create(base) {
      return GrantAppPermissionResponse.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseGrantAppPermissionResponse();
      message.error = object.error !== void 0 && object.error !== null ? ErrorMessage.fromPartial(object.error) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(GrantAppPermissionResponse.$type, GrantAppPermissionResponse);
  function createBaseRevokeAppPermissionRequest() {
    return { appSlug: "", userId: "", subredditId: "" };
  }
  const RevokeAppPermissionRequest = {
    $type: "reddit.devvit.app_permission.v1.RevokeAppPermissionRequest",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.appSlug !== "") {
        writer2.uint32(10).string(message.appSlug);
      }
      if (message.userId !== "") {
        writer2.uint32(18).string(message.userId);
      }
      if (message.subredditId !== "") {
        writer2.uint32(26).string(message.subredditId);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRevokeAppPermissionRequest();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.appSlug = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.userId = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.subredditId = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        appSlug: isSet$e(object.appSlug) ? globalThis.String(object.appSlug) : "",
        userId: isSet$e(object.userId) ? globalThis.String(object.userId) : "",
        subredditId: isSet$e(object.subredditId) ? globalThis.String(object.subredditId) : ""
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.appSlug !== "") {
        obj.appSlug = message.appSlug;
      }
      if (message.userId !== "") {
        obj.userId = message.userId;
      }
      if (message.subredditId !== "") {
        obj.subredditId = message.subredditId;
      }
      return obj;
    },
    create(base) {
      return RevokeAppPermissionRequest.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRevokeAppPermissionRequest();
      message.appSlug = object.appSlug ?? "";
      message.userId = object.userId ?? "";
      message.subredditId = object.subredditId ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(RevokeAppPermissionRequest.$type, RevokeAppPermissionRequest);
  function createBaseRevokeAppPermissionResponse() {
    return { error: void 0 };
  }
  const RevokeAppPermissionResponse = {
    $type: "reddit.devvit.app_permission.v1.RevokeAppPermissionResponse",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.error !== void 0) {
        ErrorMessage.encode(message.error, writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRevokeAppPermissionResponse();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.error = ErrorMessage.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { error: isSet$e(object.error) ? ErrorMessage.fromJSON(object.error) : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.error !== void 0) {
        obj.error = ErrorMessage.toJSON(message.error);
      }
      return obj;
    },
    create(base) {
      return RevokeAppPermissionResponse.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRevokeAppPermissionResponse();
      message.error = object.error !== void 0 && object.error !== null ? ErrorMessage.fromPartial(object.error) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(RevokeAppPermissionResponse.$type, RevokeAppPermissionResponse);
  function createBaseErrorMessage() {
    return { message: "" };
  }
  const ErrorMessage = {
    $type: "reddit.devvit.app_permission.v1.ErrorMessage",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.message !== "") {
        writer2.uint32(10).string(message.message);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseErrorMessage();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.message = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { message: isSet$e(object.message) ? globalThis.String(object.message) : "" };
    },
    toJSON(message) {
      const obj = {};
      if (message.message !== "") {
        obj.message = message.message;
      }
      return obj;
    },
    create(base) {
      return ErrorMessage.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseErrorMessage();
      message.message = object.message ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(ErrorMessage.$type, ErrorMessage);
  function toTimestamp(date) {
    const seconds = Math.trunc(date.getTime() / 1e3);
    const nanos = date.getTime() % 1e3 * 1e6;
    return { seconds, nanos };
  }
  function fromTimestamp(t) {
    let millis = (t.seconds || 0) * 1e3;
    millis += (t.nanos || 0) / 1e6;
    return new globalThis.Date(millis);
  }
  function fromJsonTimestamp(o) {
    if (o instanceof globalThis.Date) {
      return o;
    } else if (typeof o === "string") {
      return new globalThis.Date(o);
    } else {
      return fromTimestamp(Timestamp.fromJSON(o));
    }
  }
  function isSet$e(value) {
    return value !== null && value !== void 0;
  }
  function createBaseCanRunAsUserEffect() {
    return { postId: void 0, appSlug: void 0, subredditId: void 0 };
  }
  const CanRunAsUserEffect = {
    $type: "devvit.ui.effects.v1alpha.CanRunAsUserEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.postId !== void 0) {
        writer2.uint32(10).string(message.postId);
      }
      if (message.appSlug !== void 0) {
        writer2.uint32(18).string(message.appSlug);
      }
      if (message.subredditId !== void 0) {
        writer2.uint32(26).string(message.subredditId);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseCanRunAsUserEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.postId = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.appSlug = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.subredditId = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        postId: isSet$d(object.postId) ? globalThis.String(object.postId) : void 0,
        appSlug: isSet$d(object.appSlug) ? globalThis.String(object.appSlug) : void 0,
        subredditId: isSet$d(object.subredditId) ? globalThis.String(object.subredditId) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.postId !== void 0) {
        obj.postId = message.postId;
      }
      if (message.appSlug !== void 0) {
        obj.appSlug = message.appSlug;
      }
      if (message.subredditId !== void 0) {
        obj.subredditId = message.subredditId;
      }
      return obj;
    },
    create(base) {
      return CanRunAsUserEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseCanRunAsUserEffect();
      message.postId = object.postId ?? void 0;
      message.appSlug = object.appSlug ?? void 0;
      message.subredditId = object.subredditId ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(CanRunAsUserEffect.$type, CanRunAsUserEffect);
  function createBaseConsentStatusEvent() {
    return { consentStatus: 0 };
  }
  const ConsentStatusEvent = {
    $type: "devvit.ui.effects.v1alpha.ConsentStatusEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.consentStatus !== 0) {
        writer2.uint32(8).int32(message.consentStatus);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseConsentStatusEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.consentStatus = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { consentStatus: isSet$d(object.consentStatus) ? consentStatusFromJSON(object.consentStatus) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.consentStatus !== 0) {
        obj.consentStatus = consentStatusToJSON(message.consentStatus);
      }
      return obj;
    },
    create(base) {
      return ConsentStatusEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseConsentStatusEvent();
      message.consentStatus = object.consentStatus ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(ConsentStatusEvent.$type, ConsentStatusEvent);
  function isSet$d(value) {
    return value !== null && value !== void 0;
  }
  var OrderResultStatus;
  (function(OrderResultStatus2) {
    OrderResultStatus2[OrderResultStatus2["STATUS_CANCELLED"] = 0] = "STATUS_CANCELLED";
    OrderResultStatus2[OrderResultStatus2["STATUS_SUCCESS"] = 1] = "STATUS_SUCCESS";
    OrderResultStatus2[OrderResultStatus2["STATUS_ERROR"] = 2] = "STATUS_ERROR";
    OrderResultStatus2[OrderResultStatus2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(OrderResultStatus || (OrderResultStatus = {}));
  function orderResultStatusFromJSON(object) {
    switch (object) {
      case 0:
      case "STATUS_CANCELLED":
        return OrderResultStatus.STATUS_CANCELLED;
      case 1:
      case "STATUS_SUCCESS":
        return OrderResultStatus.STATUS_SUCCESS;
      case 2:
      case "STATUS_ERROR":
        return OrderResultStatus.STATUS_ERROR;
      case -1:
      case "UNRECOGNIZED":
      default:
        return OrderResultStatus.UNRECOGNIZED;
    }
  }
  function orderResultStatusToJSON(object) {
    switch (object) {
      case OrderResultStatus.STATUS_CANCELLED:
        return 0;
      case OrderResultStatus.STATUS_SUCCESS:
        return 1;
      case OrderResultStatus.STATUS_ERROR:
        return 2;
      case OrderResultStatus.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseCreateOrderEffect() {
    return { id: "", skus: [], metadata: {} };
  }
  const CreateOrderEffect = {
    $type: "devvit.ui.effects.v1alpha.CreateOrderEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.id !== "") {
        writer2.uint32(10).string(message.id);
      }
      for (const v of message.skus) {
        writer2.uint32(18).string(v);
      }
      Object.entries(message.metadata).forEach(([key, value]) => {
        CreateOrderEffect_MetadataEntry.encode({ key, value }, writer2.uint32(26).fork()).ldelim();
      });
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseCreateOrderEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.id = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.skus.push(reader2.string());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            const entry3 = CreateOrderEffect_MetadataEntry.decode(reader2, reader2.uint32());
            if (entry3.value !== void 0) {
              message.metadata[entry3.key] = entry3.value;
            }
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        id: isSet$c(object.id) ? globalThis.String(object.id) : "",
        skus: globalThis.Array.isArray(object?.skus) ? object.skus.map((e) => globalThis.String(e)) : [],
        metadata: isObject$5(object.metadata) ? Object.entries(object.metadata).reduce((acc, [key, value]) => {
          acc[key] = String(value);
          return acc;
        }, {}) : {}
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.id !== "") {
        obj.id = message.id;
      }
      if (message.skus?.length) {
        obj.skus = message.skus;
      }
      if (message.metadata) {
        const entries = Object.entries(message.metadata);
        if (entries.length > 0) {
          obj.metadata = {};
          entries.forEach(([k, v]) => {
            obj.metadata[k] = v;
          });
        }
      }
      return obj;
    },
    create(base) {
      return CreateOrderEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseCreateOrderEffect();
      message.id = object.id ?? "";
      message.skus = object.skus?.map((e) => e) || [];
      message.metadata = Object.entries(object.metadata ?? {}).reduce((acc, [key, value]) => {
        if (value !== void 0) {
          acc[key] = globalThis.String(value);
        }
        return acc;
      }, {});
      return message;
    }
  };
  messageTypeRegistry.set(CreateOrderEffect.$type, CreateOrderEffect);
  function createBaseCreateOrderEffect_MetadataEntry() {
    return { key: "", value: "" };
  }
  const CreateOrderEffect_MetadataEntry = {
    $type: "devvit.ui.effects.v1alpha.CreateOrderEffect.MetadataEntry",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.key !== "") {
        writer2.uint32(10).string(message.key);
      }
      if (message.value !== "") {
        writer2.uint32(18).string(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseCreateOrderEffect_MetadataEntry();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.key = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.value = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        key: isSet$c(object.key) ? globalThis.String(object.key) : "",
        value: isSet$c(object.value) ? globalThis.String(object.value) : ""
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.key !== "") {
        obj.key = message.key;
      }
      if (message.value !== "") {
        obj.value = message.value;
      }
      return obj;
    },
    create(base) {
      return CreateOrderEffect_MetadataEntry.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseCreateOrderEffect_MetadataEntry();
      message.key = object.key ?? "";
      message.value = object.value ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(CreateOrderEffect_MetadataEntry.$type, CreateOrderEffect_MetadataEntry);
  function createBaseOrderResultEvent() {
    return { errorMessage: void 0, orderId: void 0, order: void 0, status: 0, errorCode: void 0 };
  }
  const OrderResultEvent = {
    $type: "devvit.ui.effects.v1alpha.OrderResultEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.errorMessage !== void 0) {
        writer2.uint32(18).string(message.errorMessage);
      }
      if (message.orderId !== void 0) {
        writer2.uint32(26).string(message.orderId);
      }
      if (message.order !== void 0) {
        CreateOrderEffect.encode(message.order, writer2.uint32(34).fork()).ldelim();
      }
      if (message.status !== 0) {
        writer2.uint32(40).int32(message.status);
      }
      if (message.errorCode !== void 0) {
        writer2.uint32(48).int32(message.errorCode);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseOrderResultEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 2:
            if (tag !== 18) {
              break;
            }
            message.errorMessage = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.orderId = reader2.string();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.order = CreateOrderEffect.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 40) {
              break;
            }
            message.status = reader2.int32();
            continue;
          case 6:
            if (tag !== 48) {
              break;
            }
            message.errorCode = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        errorMessage: isSet$c(object.errorMessage) ? globalThis.String(object.errorMessage) : void 0,
        orderId: isSet$c(object.orderId) ? globalThis.String(object.orderId) : void 0,
        order: isSet$c(object.order) ? CreateOrderEffect.fromJSON(object.order) : void 0,
        status: isSet$c(object.status) ? orderResultStatusFromJSON(object.status) : 0,
        errorCode: isSet$c(object.errorCode) ? globalThis.Number(object.errorCode) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.errorMessage !== void 0) {
        obj.errorMessage = message.errorMessage;
      }
      if (message.orderId !== void 0) {
        obj.orderId = message.orderId;
      }
      if (message.order !== void 0) {
        obj.order = CreateOrderEffect.toJSON(message.order);
      }
      if (message.status !== 0) {
        obj.status = orderResultStatusToJSON(message.status);
      }
      if (message.errorCode !== void 0) {
        obj.errorCode = Math.round(message.errorCode);
      }
      return obj;
    },
    create(base) {
      return OrderResultEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseOrderResultEvent();
      message.errorMessage = object.errorMessage ?? void 0;
      message.orderId = object.orderId ?? void 0;
      message.order = object.order !== void 0 && object.order !== null ? CreateOrderEffect.fromPartial(object.order) : void 0;
      message.status = object.status ?? 0;
      message.errorCode = object.errorCode ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(OrderResultEvent.$type, OrderResultEvent);
  function isObject$5(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$c(value) {
    return value !== null && value !== void 0;
  }
  function createBaseDuration() {
    return { seconds: 0, nanos: 0 };
  }
  const Duration = {
    $type: "google.protobuf.Duration",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.seconds !== 0) {
        writer2.uint32(8).int64(message.seconds);
      }
      if (message.nanos !== 0) {
        writer2.uint32(16).int32(message.nanos);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseDuration();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.seconds = longToNumber$1(reader2.int64());
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.nanos = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        seconds: isSet$b(object.seconds) ? globalThis.Number(object.seconds) : 0,
        nanos: isSet$b(object.nanos) ? globalThis.Number(object.nanos) : 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.seconds !== 0) {
        obj.seconds = Math.round(message.seconds);
      }
      if (message.nanos !== 0) {
        obj.nanos = Math.round(message.nanos);
      }
      return obj;
    },
    create(base) {
      return Duration.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseDuration();
      message.seconds = object.seconds ?? 0;
      message.nanos = object.nanos ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(Duration.$type, Duration);
  function longToNumber$1(long) {
    if (long.gt(globalThis.Number.MAX_SAFE_INTEGER)) {
      throw new globalThis.Error("Value is larger than Number.MAX_SAFE_INTEGER");
    }
    return long.toNumber();
  }
  if (_m0.util.Long !== Long) {
    _m0.util.Long = Long;
    _m0.configure();
  }
  function isSet$b(value) {
    return value !== null && value !== void 0;
  }
  function createBaseDoubleValue() {
    return { value: 0 };
  }
  const DoubleValue = {
    $type: "google.protobuf.DoubleValue",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== 0) {
        writer2.uint32(9).double(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseDoubleValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 9) {
              break;
            }
            message.value = reader2.double();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.Number(object.value) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== 0) {
        obj.value = message.value;
      }
      return obj;
    },
    create(base) {
      return DoubleValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseDoubleValue();
      message.value = object.value ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(DoubleValue.$type, DoubleValue);
  function createBaseFloatValue() {
    return { value: 0 };
  }
  const FloatValue = {
    $type: "google.protobuf.FloatValue",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== 0) {
        writer2.uint32(13).float(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseFloatValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 13) {
              break;
            }
            message.value = reader2.float();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.Number(object.value) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== 0) {
        obj.value = message.value;
      }
      return obj;
    },
    create(base) {
      return FloatValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseFloatValue();
      message.value = object.value ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(FloatValue.$type, FloatValue);
  function createBaseInt64Value() {
    return { value: 0 };
  }
  const Int64Value = {
    $type: "google.protobuf.Int64Value",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== 0) {
        writer2.uint32(8).int64(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseInt64Value();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.value = longToNumber(reader2.int64());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.Number(object.value) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== 0) {
        obj.value = Math.round(message.value);
      }
      return obj;
    },
    create(base) {
      return Int64Value.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseInt64Value();
      message.value = object.value ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(Int64Value.$type, Int64Value);
  function createBaseUInt64Value() {
    return { value: 0 };
  }
  const UInt64Value = {
    $type: "google.protobuf.UInt64Value",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== 0) {
        writer2.uint32(8).uint64(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUInt64Value();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.value = longToNumber(reader2.uint64());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.Number(object.value) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== 0) {
        obj.value = Math.round(message.value);
      }
      return obj;
    },
    create(base) {
      return UInt64Value.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUInt64Value();
      message.value = object.value ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(UInt64Value.$type, UInt64Value);
  function createBaseInt32Value() {
    return { value: 0 };
  }
  const Int32Value = {
    $type: "google.protobuf.Int32Value",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== 0) {
        writer2.uint32(8).int32(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseInt32Value();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.value = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.Number(object.value) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== 0) {
        obj.value = Math.round(message.value);
      }
      return obj;
    },
    create(base) {
      return Int32Value.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseInt32Value();
      message.value = object.value ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(Int32Value.$type, Int32Value);
  function createBaseUInt32Value() {
    return { value: 0 };
  }
  const UInt32Value = {
    $type: "google.protobuf.UInt32Value",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== 0) {
        writer2.uint32(8).uint32(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUInt32Value();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.value = reader2.uint32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.Number(object.value) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== 0) {
        obj.value = Math.round(message.value);
      }
      return obj;
    },
    create(base) {
      return UInt32Value.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUInt32Value();
      message.value = object.value ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(UInt32Value.$type, UInt32Value);
  function createBaseBoolValue() {
    return { value: false };
  }
  const BoolValue = {
    $type: "google.protobuf.BoolValue",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== false) {
        writer2.uint32(8).bool(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBoolValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.value = reader2.bool();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.Boolean(object.value) : false };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== false) {
        obj.value = message.value;
      }
      return obj;
    },
    create(base) {
      return BoolValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBoolValue();
      message.value = object.value ?? false;
      return message;
    }
  };
  messageTypeRegistry.set(BoolValue.$type, BoolValue);
  function createBaseStringValue() {
    return { value: "" };
  }
  const StringValue = {
    $type: "google.protobuf.StringValue",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== "") {
        writer2.uint32(10).string(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseStringValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.value = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? globalThis.String(object.value) : "" };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== "") {
        obj.value = message.value;
      }
      return obj;
    },
    create(base) {
      return StringValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseStringValue();
      message.value = object.value ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(StringValue.$type, StringValue);
  function createBaseBytesValue() {
    return { value: new Uint8Array(0) };
  }
  const BytesValue = {
    $type: "google.protobuf.BytesValue",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value.length !== 0) {
        writer2.uint32(10).bytes(message.value);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBytesValue();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.value = reader2.bytes();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { value: isSet$a(object.value) ? bytesFromBase64(object.value) : new Uint8Array(0) };
    },
    toJSON(message) {
      const obj = {};
      if (message.value.length !== 0) {
        obj.value = base64FromBytes(message.value);
      }
      return obj;
    },
    create(base) {
      return BytesValue.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBytesValue();
      message.value = object.value ?? new Uint8Array(0);
      return message;
    }
  };
  messageTypeRegistry.set(BytesValue.$type, BytesValue);
  function bytesFromBase64(b64) {
    if (globalThis.Buffer) {
      return Uint8Array.from(globalThis.Buffer.from(b64, "base64"));
    } else {
      const bin = globalThis.atob(b64);
      const arr = new Uint8Array(bin.length);
      for (let i = 0; i < bin.length; ++i) {
        arr[i] = bin.charCodeAt(i);
      }
      return arr;
    }
  }
  function base64FromBytes(arr) {
    if (globalThis.Buffer) {
      return globalThis.Buffer.from(arr).toString("base64");
    } else {
      const bin = [];
      arr.forEach((byte) => {
        bin.push(globalThis.String.fromCharCode(byte));
      });
      return globalThis.btoa(bin.join(""));
    }
  }
  function longToNumber(long) {
    if (long.gt(globalThis.Number.MAX_SAFE_INTEGER)) {
      throw new globalThis.Error("Value is larger than Number.MAX_SAFE_INTEGER");
    }
    return long.toNumber();
  }
  if (_m0.util.Long !== Long) {
    _m0.util.Long = Long;
    _m0.configure();
  }
  function isSet$a(value) {
    return value !== null && value !== void 0;
  }
  function createBaseSetIntervalsEffect() {
    return { intervals: {} };
  }
  const SetIntervalsEffect = {
    $type: "devvit.ui.effects.v1alpha.SetIntervalsEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      Object.entries(message.intervals).forEach(([key, value]) => {
        SetIntervalsEffect_IntervalsEntry.encode({ key, value }, writer2.uint32(10).fork()).ldelim();
      });
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseSetIntervalsEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            const entry1 = SetIntervalsEffect_IntervalsEntry.decode(reader2, reader2.uint32());
            if (entry1.value !== void 0) {
              message.intervals[entry1.key] = entry1.value;
            }
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        intervals: isObject$4(object.intervals) ? Object.entries(object.intervals).reduce((acc, [key, value]) => {
          acc[key] = IntervalDetails.fromJSON(value);
          return acc;
        }, {}) : {}
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.intervals) {
        const entries = Object.entries(message.intervals);
        if (entries.length > 0) {
          obj.intervals = {};
          entries.forEach(([k, v]) => {
            obj.intervals[k] = IntervalDetails.toJSON(v);
          });
        }
      }
      return obj;
    },
    create(base) {
      return SetIntervalsEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseSetIntervalsEffect();
      message.intervals = Object.entries(object.intervals ?? {}).reduce((acc, [key, value]) => {
        if (value !== void 0) {
          acc[key] = IntervalDetails.fromPartial(value);
        }
        return acc;
      }, {});
      return message;
    }
  };
  messageTypeRegistry.set(SetIntervalsEffect.$type, SetIntervalsEffect);
  function createBaseSetIntervalsEffect_IntervalsEntry() {
    return { key: "", value: void 0 };
  }
  const SetIntervalsEffect_IntervalsEntry = {
    $type: "devvit.ui.effects.v1alpha.SetIntervalsEffect.IntervalsEntry",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.key !== "") {
        writer2.uint32(10).string(message.key);
      }
      if (message.value !== void 0) {
        IntervalDetails.encode(message.value, writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseSetIntervalsEffect_IntervalsEntry();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.key = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.value = IntervalDetails.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        key: isSet$9(object.key) ? globalThis.String(object.key) : "",
        value: isSet$9(object.value) ? IntervalDetails.fromJSON(object.value) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.key !== "") {
        obj.key = message.key;
      }
      if (message.value !== void 0) {
        obj.value = IntervalDetails.toJSON(message.value);
      }
      return obj;
    },
    create(base) {
      return SetIntervalsEffect_IntervalsEntry.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseSetIntervalsEffect_IntervalsEntry();
      message.key = object.key ?? "";
      message.value = object.value !== void 0 && object.value !== null ? IntervalDetails.fromPartial(object.value) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(SetIntervalsEffect_IntervalsEntry.$type, SetIntervalsEffect_IntervalsEntry);
  function createBaseIntervalDetails() {
    return { duration: void 0, async: void 0 };
  }
  const IntervalDetails = {
    $type: "devvit.ui.effects.v1alpha.IntervalDetails",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.duration !== void 0) {
        Duration.encode(message.duration, writer2.uint32(10).fork()).ldelim();
      }
      if (message.async !== void 0) {
        BoolValue.encode({ value: message.async }, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseIntervalDetails();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.duration = Duration.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.async = BoolValue.decode(reader2, reader2.uint32()).value;
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        duration: isSet$9(object.duration) ? Duration.fromJSON(object.duration) : void 0,
        async: isSet$9(object.async) ? Boolean(object.async) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.duration !== void 0) {
        obj.duration = Duration.toJSON(message.duration);
      }
      if (message.async !== void 0) {
        obj.async = message.async;
      }
      return obj;
    },
    create(base) {
      return IntervalDetails.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseIntervalDetails();
      message.duration = object.duration !== void 0 && object.duration !== null ? Duration.fromPartial(object.duration) : void 0;
      message.async = object.async ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(IntervalDetails.$type, IntervalDetails);
  function isObject$4(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$9(value) {
    return value !== null && value !== void 0;
  }
  var RealtimeSubscriptionStatus;
  (function(RealtimeSubscriptionStatus2) {
    RealtimeSubscriptionStatus2[RealtimeSubscriptionStatus2["REALTIME_SUBSCRIBED"] = 0] = "REALTIME_SUBSCRIBED";
    RealtimeSubscriptionStatus2[RealtimeSubscriptionStatus2["REALTIME_UNSUBSCRIBED"] = 1] = "REALTIME_UNSUBSCRIBED";
    RealtimeSubscriptionStatus2[RealtimeSubscriptionStatus2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(RealtimeSubscriptionStatus || (RealtimeSubscriptionStatus = {}));
  function realtimeSubscriptionStatusFromJSON(object) {
    switch (object) {
      case 0:
      case "REALTIME_SUBSCRIBED":
        return RealtimeSubscriptionStatus.REALTIME_SUBSCRIBED;
      case 1:
      case "REALTIME_UNSUBSCRIBED":
        return RealtimeSubscriptionStatus.REALTIME_UNSUBSCRIBED;
      case -1:
      case "UNRECOGNIZED":
      default:
        return RealtimeSubscriptionStatus.UNRECOGNIZED;
    }
  }
  function realtimeSubscriptionStatusToJSON(object) {
    switch (object) {
      case RealtimeSubscriptionStatus.REALTIME_SUBSCRIBED:
        return 0;
      case RealtimeSubscriptionStatus.REALTIME_UNSUBSCRIBED:
        return 1;
      case RealtimeSubscriptionStatus.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseRealtimeSubscriptionsEffect() {
    return { subscriptionIds: [] };
  }
  const RealtimeSubscriptionsEffect = {
    $type: "devvit.ui.effects.v1alpha.RealtimeSubscriptionsEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.subscriptionIds) {
        writer2.uint32(10).string(v);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRealtimeSubscriptionsEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.subscriptionIds.push(reader2.string());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        subscriptionIds: globalThis.Array.isArray(object?.subscriptionIds) ? object.subscriptionIds.map((e) => globalThis.String(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.subscriptionIds?.length) {
        obj.subscriptionIds = message.subscriptionIds;
      }
      return obj;
    },
    create(base) {
      return RealtimeSubscriptionsEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRealtimeSubscriptionsEffect();
      message.subscriptionIds = object.subscriptionIds?.map((e) => e) || [];
      return message;
    }
  };
  messageTypeRegistry.set(RealtimeSubscriptionsEffect.$type, RealtimeSubscriptionsEffect);
  function createBaseRealtimeSubscriptionEvent() {
    return { event: void 0, status: void 0 };
  }
  const RealtimeSubscriptionEvent = {
    $type: "devvit.ui.effects.v1alpha.RealtimeSubscriptionEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.event !== void 0) {
        RealtimeEvent.encode(message.event, writer2.uint32(10).fork()).ldelim();
      }
      if (message.status !== void 0) {
        writer2.uint32(16).int32(message.status);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRealtimeSubscriptionEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.event = RealtimeEvent.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.status = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        event: isSet$8(object.event) ? RealtimeEvent.fromJSON(object.event) : void 0,
        status: isSet$8(object.status) ? realtimeSubscriptionStatusFromJSON(object.status) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.event !== void 0) {
        obj.event = RealtimeEvent.toJSON(message.event);
      }
      if (message.status !== void 0) {
        obj.status = realtimeSubscriptionStatusToJSON(message.status);
      }
      return obj;
    },
    create(base) {
      return RealtimeSubscriptionEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRealtimeSubscriptionEvent();
      message.event = object.event !== void 0 && object.event !== null ? RealtimeEvent.fromPartial(object.event) : void 0;
      message.status = object.status ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(RealtimeSubscriptionEvent.$type, RealtimeSubscriptionEvent);
  function isSet$8(value) {
    return value !== null && value !== void 0;
  }
  function createBaseReloadPartEffect() {
    return { subreddit: void 0, post: void 0, comment: void 0 };
  }
  const ReloadPartEffect = {
    $type: "devvit.ui.effects.v1alpha.ReloadPartEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.subreddit !== void 0) {
        ReloadPartEffect_Subreddit.encode(message.subreddit, writer2.uint32(10).fork()).ldelim();
      }
      if (message.post !== void 0) {
        ReloadPartEffect_Post.encode(message.post, writer2.uint32(18).fork()).ldelim();
      }
      if (message.comment !== void 0) {
        ReloadPartEffect_Comment.encode(message.comment, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseReloadPartEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.subreddit = ReloadPartEffect_Subreddit.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.post = ReloadPartEffect_Post.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.comment = ReloadPartEffect_Comment.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        subreddit: isSet$7(object.subreddit) ? ReloadPartEffect_Subreddit.fromJSON(object.subreddit) : void 0,
        post: isSet$7(object.post) ? ReloadPartEffect_Post.fromJSON(object.post) : void 0,
        comment: isSet$7(object.comment) ? ReloadPartEffect_Comment.fromJSON(object.comment) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.subreddit !== void 0) {
        obj.subreddit = ReloadPartEffect_Subreddit.toJSON(message.subreddit);
      }
      if (message.post !== void 0) {
        obj.post = ReloadPartEffect_Post.toJSON(message.post);
      }
      if (message.comment !== void 0) {
        obj.comment = ReloadPartEffect_Comment.toJSON(message.comment);
      }
      return obj;
    },
    create(base) {
      return ReloadPartEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseReloadPartEffect();
      message.subreddit = object.subreddit !== void 0 && object.subreddit !== null ? ReloadPartEffect_Subreddit.fromPartial(object.subreddit) : void 0;
      message.post = object.post !== void 0 && object.post !== null ? ReloadPartEffect_Post.fromPartial(object.post) : void 0;
      message.comment = object.comment !== void 0 && object.comment !== null ? ReloadPartEffect_Comment.fromPartial(object.comment) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(ReloadPartEffect.$type, ReloadPartEffect);
  function createBaseReloadPartEffect_Subreddit() {
    return { subredditId: "" };
  }
  const ReloadPartEffect_Subreddit = {
    $type: "devvit.ui.effects.v1alpha.ReloadPartEffect.Subreddit",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.subredditId !== "") {
        writer2.uint32(10).string(message.subredditId);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseReloadPartEffect_Subreddit();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.subredditId = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { subredditId: isSet$7(object.subredditId) ? globalThis.String(object.subredditId) : "" };
    },
    toJSON(message) {
      const obj = {};
      if (message.subredditId !== "") {
        obj.subredditId = message.subredditId;
      }
      return obj;
    },
    create(base) {
      return ReloadPartEffect_Subreddit.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseReloadPartEffect_Subreddit();
      message.subredditId = object.subredditId ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(ReloadPartEffect_Subreddit.$type, ReloadPartEffect_Subreddit);
  function createBaseReloadPartEffect_Post() {
    return { postId: "", body: void 0, comments: void 0 };
  }
  const ReloadPartEffect_Post = {
    $type: "devvit.ui.effects.v1alpha.ReloadPartEffect.Post",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.postId !== "") {
        writer2.uint32(10).string(message.postId);
      }
      if (message.body !== void 0) {
        writer2.uint32(16).bool(message.body);
      }
      if (message.comments !== void 0) {
        writer2.uint32(24).bool(message.comments);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseReloadPartEffect_Post();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.postId = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.body = reader2.bool();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.comments = reader2.bool();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        postId: isSet$7(object.postId) ? globalThis.String(object.postId) : "",
        body: isSet$7(object.body) ? globalThis.Boolean(object.body) : void 0,
        comments: isSet$7(object.comments) ? globalThis.Boolean(object.comments) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.postId !== "") {
        obj.postId = message.postId;
      }
      if (message.body !== void 0) {
        obj.body = message.body;
      }
      if (message.comments !== void 0) {
        obj.comments = message.comments;
      }
      return obj;
    },
    create(base) {
      return ReloadPartEffect_Post.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseReloadPartEffect_Post();
      message.postId = object.postId ?? "";
      message.body = object.body ?? void 0;
      message.comments = object.comments ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(ReloadPartEffect_Post.$type, ReloadPartEffect_Post);
  function createBaseReloadPartEffect_Comment() {
    return { postId: "", commentId: "", replies: void 0 };
  }
  const ReloadPartEffect_Comment = {
    $type: "devvit.ui.effects.v1alpha.ReloadPartEffect.Comment",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.postId !== "") {
        writer2.uint32(10).string(message.postId);
      }
      if (message.commentId !== "") {
        writer2.uint32(18).string(message.commentId);
      }
      if (message.replies !== void 0) {
        writer2.uint32(24).bool(message.replies);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseReloadPartEffect_Comment();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.postId = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.commentId = reader2.string();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.replies = reader2.bool();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        postId: isSet$7(object.postId) ? globalThis.String(object.postId) : "",
        commentId: isSet$7(object.commentId) ? globalThis.String(object.commentId) : "",
        replies: isSet$7(object.replies) ? globalThis.Boolean(object.replies) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.postId !== "") {
        obj.postId = message.postId;
      }
      if (message.commentId !== "") {
        obj.commentId = message.commentId;
      }
      if (message.replies !== void 0) {
        obj.replies = message.replies;
      }
      return obj;
    },
    create(base) {
      return ReloadPartEffect_Comment.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseReloadPartEffect_Comment();
      message.postId = object.postId ?? "";
      message.commentId = object.commentId ?? "";
      message.replies = object.replies ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(ReloadPartEffect_Comment.$type, ReloadPartEffect_Comment);
  function isSet$7(value) {
    return value !== null && value !== void 0;
  }
  function createBaseRerenderEffect() {
    return { delaySeconds: void 0 };
  }
  const RerenderEffect = {
    $type: "devvit.ui.effects.v1alpha.RerenderEffect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.delaySeconds !== void 0) {
        writer2.uint32(13).float(message.delaySeconds);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseRerenderEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 13) {
              break;
            }
            message.delaySeconds = reader2.float();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { delaySeconds: isSet$6(object.delaySeconds) ? globalThis.Number(object.delaySeconds) : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.delaySeconds !== void 0) {
        obj.delaySeconds = message.delaySeconds;
      }
      return obj;
    },
    create(base) {
      return RerenderEffect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseRerenderEffect();
      message.delaySeconds = object.delaySeconds ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(RerenderEffect.$type, RerenderEffect);
  function isSet$6(value) {
    return value !== null && value !== void 0;
  }
  var EffectType;
  (function(EffectType2) {
    EffectType2[EffectType2["EFFECT_REALTIME_SUB"] = 0] = "EFFECT_REALTIME_SUB";
    EffectType2[EffectType2["EFFECT_RERENDER_UI"] = 1] = "EFFECT_RERENDER_UI";
    EffectType2[EffectType2["EFFECT_RELOAD_PART"] = 2] = "EFFECT_RELOAD_PART";
    EffectType2[EffectType2["EFFECT_SHOW_FORM"] = 3] = "EFFECT_SHOW_FORM";
    EffectType2[EffectType2["EFFECT_SHOW_TOAST"] = 4] = "EFFECT_SHOW_TOAST";
    EffectType2[EffectType2["EFFECT_NAVIGATE_TO_URL"] = 5] = "EFFECT_NAVIGATE_TO_URL";
    EffectType2[EffectType2["EFFECT_SET_INTERVALS"] = 7] = "EFFECT_SET_INTERVALS";
    EffectType2[EffectType2["EFFECT_CREATE_ORDER"] = 8] = "EFFECT_CREATE_ORDER";
    EffectType2[EffectType2["EFFECT_WEB_VIEW"] = 9] = "EFFECT_WEB_VIEW";
    EffectType2[EffectType2["EFFECT_CAN_RUN_AS_USER"] = 11] = "EFFECT_CAN_RUN_AS_USER";
    EffectType2[EffectType2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(EffectType || (EffectType = {}));
  function effectTypeFromJSON(object) {
    switch (object) {
      case 0:
      case "EFFECT_REALTIME_SUB":
        return EffectType.EFFECT_REALTIME_SUB;
      case 1:
      case "EFFECT_RERENDER_UI":
        return EffectType.EFFECT_RERENDER_UI;
      case 2:
      case "EFFECT_RELOAD_PART":
        return EffectType.EFFECT_RELOAD_PART;
      case 3:
      case "EFFECT_SHOW_FORM":
        return EffectType.EFFECT_SHOW_FORM;
      case 4:
      case "EFFECT_SHOW_TOAST":
        return EffectType.EFFECT_SHOW_TOAST;
      case 5:
      case "EFFECT_NAVIGATE_TO_URL":
        return EffectType.EFFECT_NAVIGATE_TO_URL;
      case 7:
      case "EFFECT_SET_INTERVALS":
        return EffectType.EFFECT_SET_INTERVALS;
      case 8:
      case "EFFECT_CREATE_ORDER":
        return EffectType.EFFECT_CREATE_ORDER;
      case 9:
      case "EFFECT_WEB_VIEW":
        return EffectType.EFFECT_WEB_VIEW;
      case 11:
      case "EFFECT_CAN_RUN_AS_USER":
        return EffectType.EFFECT_CAN_RUN_AS_USER;
      case -1:
      case "UNRECOGNIZED":
      default:
        return EffectType.UNRECOGNIZED;
    }
  }
  function effectTypeToJSON(object) {
    switch (object) {
      case EffectType.EFFECT_REALTIME_SUB:
        return 0;
      case EffectType.EFFECT_RERENDER_UI:
        return 1;
      case EffectType.EFFECT_RELOAD_PART:
        return 2;
      case EffectType.EFFECT_SHOW_FORM:
        return 3;
      case EffectType.EFFECT_SHOW_TOAST:
        return 4;
      case EffectType.EFFECT_NAVIGATE_TO_URL:
        return 5;
      case EffectType.EFFECT_SET_INTERVALS:
        return 7;
      case EffectType.EFFECT_CREATE_ORDER:
        return 8;
      case EffectType.EFFECT_WEB_VIEW:
        return 9;
      case EffectType.EFFECT_CAN_RUN_AS_USER:
        return 11;
      case EffectType.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseEffect() {
    return {
      realtimeSubscriptions: void 0,
      rerenderUi: void 0,
      reloadPart: void 0,
      showForm: void 0,
      showToast: void 0,
      navigateToUrl: void 0,
      interval: void 0,
      createOrder: void 0,
      webView: void 0,
      canRunAsUser: void 0,
      type: 0
    };
  }
  const Effect = {
    $type: "devvit.ui.effects.v1alpha.Effect",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.realtimeSubscriptions !== void 0) {
        RealtimeSubscriptionsEffect.encode(message.realtimeSubscriptions, writer2.uint32(10).fork()).ldelim();
      }
      if (message.rerenderUi !== void 0) {
        RerenderEffect.encode(message.rerenderUi, writer2.uint32(18).fork()).ldelim();
      }
      if (message.reloadPart !== void 0) {
        ReloadPartEffect.encode(message.reloadPart, writer2.uint32(26).fork()).ldelim();
      }
      if (message.showForm !== void 0) {
        ShowFormEffect.encode(message.showForm, writer2.uint32(34).fork()).ldelim();
      }
      if (message.showToast !== void 0) {
        ShowToastEffect.encode(message.showToast, writer2.uint32(42).fork()).ldelim();
      }
      if (message.navigateToUrl !== void 0) {
        NavigateToUrlEffect.encode(message.navigateToUrl, writer2.uint32(50).fork()).ldelim();
      }
      if (message.interval !== void 0) {
        SetIntervalsEffect.encode(message.interval, writer2.uint32(74).fork()).ldelim();
      }
      if (message.createOrder !== void 0) {
        CreateOrderEffect.encode(message.createOrder, writer2.uint32(82).fork()).ldelim();
      }
      if (message.webView !== void 0) {
        WebViewEffect.encode(message.webView, writer2.uint32(90).fork()).ldelim();
      }
      if (message.canRunAsUser !== void 0) {
        CanRunAsUserEffect.encode(message.canRunAsUser, writer2.uint32(106).fork()).ldelim();
      }
      if (message.type !== 0) {
        writer2.uint32(56).int32(message.type);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseEffect();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.realtimeSubscriptions = RealtimeSubscriptionsEffect.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.rerenderUi = RerenderEffect.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.reloadPart = ReloadPartEffect.decode(reader2, reader2.uint32());
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.showForm = ShowFormEffect.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.showToast = ShowToastEffect.decode(reader2, reader2.uint32());
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.navigateToUrl = NavigateToUrlEffect.decode(reader2, reader2.uint32());
            continue;
          case 9:
            if (tag !== 74) {
              break;
            }
            message.interval = SetIntervalsEffect.decode(reader2, reader2.uint32());
            continue;
          case 10:
            if (tag !== 82) {
              break;
            }
            message.createOrder = CreateOrderEffect.decode(reader2, reader2.uint32());
            continue;
          case 11:
            if (tag !== 90) {
              break;
            }
            message.webView = WebViewEffect.decode(reader2, reader2.uint32());
            continue;
          case 13:
            if (tag !== 106) {
              break;
            }
            message.canRunAsUser = CanRunAsUserEffect.decode(reader2, reader2.uint32());
            continue;
          case 7:
            if (tag !== 56) {
              break;
            }
            message.type = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        realtimeSubscriptions: isSet$5(object.realtimeSubscriptions) ? RealtimeSubscriptionsEffect.fromJSON(object.realtimeSubscriptions) : void 0,
        rerenderUi: isSet$5(object.rerenderUi) ? RerenderEffect.fromJSON(object.rerenderUi) : void 0,
        reloadPart: isSet$5(object.reloadPart) ? ReloadPartEffect.fromJSON(object.reloadPart) : void 0,
        showForm: isSet$5(object.showForm) ? ShowFormEffect.fromJSON(object.showForm) : void 0,
        showToast: isSet$5(object.showToast) ? ShowToastEffect.fromJSON(object.showToast) : void 0,
        navigateToUrl: isSet$5(object.navigateToUrl) ? NavigateToUrlEffect.fromJSON(object.navigateToUrl) : void 0,
        interval: isSet$5(object.interval) ? SetIntervalsEffect.fromJSON(object.interval) : void 0,
        createOrder: isSet$5(object.createOrder) ? CreateOrderEffect.fromJSON(object.createOrder) : void 0,
        webView: isSet$5(object.webView) ? WebViewEffect.fromJSON(object.webView) : void 0,
        canRunAsUser: isSet$5(object.canRunAsUser) ? CanRunAsUserEffect.fromJSON(object.canRunAsUser) : void 0,
        type: isSet$5(object.type) ? effectTypeFromJSON(object.type) : 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.realtimeSubscriptions !== void 0) {
        obj.realtimeSubscriptions = RealtimeSubscriptionsEffect.toJSON(message.realtimeSubscriptions);
      }
      if (message.rerenderUi !== void 0) {
        obj.rerenderUi = RerenderEffect.toJSON(message.rerenderUi);
      }
      if (message.reloadPart !== void 0) {
        obj.reloadPart = ReloadPartEffect.toJSON(message.reloadPart);
      }
      if (message.showForm !== void 0) {
        obj.showForm = ShowFormEffect.toJSON(message.showForm);
      }
      if (message.showToast !== void 0) {
        obj.showToast = ShowToastEffect.toJSON(message.showToast);
      }
      if (message.navigateToUrl !== void 0) {
        obj.navigateToUrl = NavigateToUrlEffect.toJSON(message.navigateToUrl);
      }
      if (message.interval !== void 0) {
        obj.interval = SetIntervalsEffect.toJSON(message.interval);
      }
      if (message.createOrder !== void 0) {
        obj.createOrder = CreateOrderEffect.toJSON(message.createOrder);
      }
      if (message.webView !== void 0) {
        obj.webView = WebViewEffect.toJSON(message.webView);
      }
      if (message.canRunAsUser !== void 0) {
        obj.canRunAsUser = CanRunAsUserEffect.toJSON(message.canRunAsUser);
      }
      if (message.type !== 0) {
        obj.type = effectTypeToJSON(message.type);
      }
      return obj;
    },
    create(base) {
      return Effect.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseEffect();
      message.realtimeSubscriptions = object.realtimeSubscriptions !== void 0 && object.realtimeSubscriptions !== null ? RealtimeSubscriptionsEffect.fromPartial(object.realtimeSubscriptions) : void 0;
      message.rerenderUi = object.rerenderUi !== void 0 && object.rerenderUi !== null ? RerenderEffect.fromPartial(object.rerenderUi) : void 0;
      message.reloadPart = object.reloadPart !== void 0 && object.reloadPart !== null ? ReloadPartEffect.fromPartial(object.reloadPart) : void 0;
      message.showForm = object.showForm !== void 0 && object.showForm !== null ? ShowFormEffect.fromPartial(object.showForm) : void 0;
      message.showToast = object.showToast !== void 0 && object.showToast !== null ? ShowToastEffect.fromPartial(object.showToast) : void 0;
      message.navigateToUrl = object.navigateToUrl !== void 0 && object.navigateToUrl !== null ? NavigateToUrlEffect.fromPartial(object.navigateToUrl) : void 0;
      message.interval = object.interval !== void 0 && object.interval !== null ? SetIntervalsEffect.fromPartial(object.interval) : void 0;
      message.createOrder = object.createOrder !== void 0 && object.createOrder !== null ? CreateOrderEffect.fromPartial(object.createOrder) : void 0;
      message.webView = object.webView !== void 0 && object.webView !== null ? WebViewEffect.fromPartial(object.webView) : void 0;
      message.canRunAsUser = object.canRunAsUser !== void 0 && object.canRunAsUser !== null ? CanRunAsUserEffect.fromPartial(object.canRunAsUser) : void 0;
      message.type = object.type ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(Effect.$type, Effect);
  function isSet$5(value) {
    return value !== null && value !== void 0;
  }
  var WebViewVisibility;
  (function(WebViewVisibility2) {
    WebViewVisibility2[WebViewVisibility2["WEBVIEW_VISIBLE"] = 0] = "WEBVIEW_VISIBLE";
    WebViewVisibility2[WebViewVisibility2["WEBVIEW_HIDDEN"] = 1] = "WEBVIEW_HIDDEN";
    WebViewVisibility2[WebViewVisibility2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(WebViewVisibility || (WebViewVisibility = {}));
  function webViewVisibilityFromJSON(object) {
    switch (object) {
      case 0:
      case "WEBVIEW_VISIBLE":
        return WebViewVisibility.WEBVIEW_VISIBLE;
      case 1:
      case "WEBVIEW_HIDDEN":
        return WebViewVisibility.WEBVIEW_HIDDEN;
      case -1:
      case "UNRECOGNIZED":
      default:
        return WebViewVisibility.UNRECOGNIZED;
    }
  }
  function webViewVisibilityToJSON(object) {
    switch (object) {
      case WebViewVisibility.WEBVIEW_VISIBLE:
        return 0;
      case WebViewVisibility.WEBVIEW_HIDDEN:
        return 1;
      case WebViewVisibility.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseWebViewPostMessageEvent() {
    return { message: void 0, jsonString: "" };
  }
  const WebViewPostMessageEvent = {
    $type: "devvit.ui.events.v1alpha.WebViewPostMessageEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.message !== void 0) {
        Value.encode(Value.wrap(message.message), writer2.uint32(10).fork()).ldelim();
      }
      if (message.jsonString !== "") {
        writer2.uint32(18).string(message.jsonString);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewPostMessageEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.message = Value.unwrap(Value.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.jsonString = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        message: isSet$4(object?.message) ? object.message : void 0,
        jsonString: isSet$4(object.jsonString) ? globalThis.String(object.jsonString) : ""
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.message !== void 0) {
        obj.message = message.message;
      }
      if (message.jsonString !== "") {
        obj.jsonString = message.jsonString;
      }
      return obj;
    },
    create(base) {
      return WebViewPostMessageEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewPostMessageEvent();
      message.message = object.message ?? void 0;
      message.jsonString = object.jsonString ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(WebViewPostMessageEvent.$type, WebViewPostMessageEvent);
  function createBaseWebViewFullScreenEvent() {
    return { visibility: 0 };
  }
  const WebViewFullScreenEvent = {
    $type: "devvit.ui.events.v1alpha.WebViewFullScreenEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.visibility !== 0) {
        writer2.uint32(8).int32(message.visibility);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewFullScreenEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.visibility = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { visibility: isSet$4(object.visibility) ? webViewVisibilityFromJSON(object.visibility) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.visibility !== 0) {
        obj.visibility = webViewVisibilityToJSON(message.visibility);
      }
      return obj;
    },
    create(base) {
      return WebViewFullScreenEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewFullScreenEvent();
      message.visibility = object.visibility ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewFullScreenEvent.$type, WebViewFullScreenEvent);
  function createBaseWebViewEvent() {
    return { postMessage: void 0, fullScreen: void 0 };
  }
  const WebViewEvent = {
    $type: "devvit.ui.events.v1alpha.WebViewEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.postMessage !== void 0) {
        WebViewPostMessageEvent.encode(message.postMessage, writer2.uint32(10).fork()).ldelim();
      }
      if (message.fullScreen !== void 0) {
        WebViewFullScreenEvent.encode(message.fullScreen, writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.postMessage = WebViewPostMessageEvent.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.fullScreen = WebViewFullScreenEvent.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        postMessage: isSet$4(object.postMessage) ? WebViewPostMessageEvent.fromJSON(object.postMessage) : void 0,
        fullScreen: isSet$4(object.fullScreen) ? WebViewFullScreenEvent.fromJSON(object.fullScreen) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.postMessage !== void 0) {
        obj.postMessage = WebViewPostMessageEvent.toJSON(message.postMessage);
      }
      if (message.fullScreen !== void 0) {
        obj.fullScreen = WebViewFullScreenEvent.toJSON(message.fullScreen);
      }
      return obj;
    },
    create(base) {
      return WebViewEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewEvent();
      message.postMessage = object.postMessage !== void 0 && object.postMessage !== null ? WebViewPostMessageEvent.fromPartial(object.postMessage) : void 0;
      message.fullScreen = object.fullScreen !== void 0 && object.fullScreen !== null ? WebViewFullScreenEvent.fromPartial(object.fullScreen) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewEvent.$type, WebViewEvent);
  function createBaseWebViewImmersiveModeChangedEvent() {
    return { immersiveMode: 0 };
  }
  const WebViewImmersiveModeChangedEvent = {
    $type: "devvit.ui.events.v1alpha.WebViewImmersiveModeChangedEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.immersiveMode !== 0) {
        writer2.uint32(8).int32(message.immersiveMode);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewImmersiveModeChangedEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.immersiveMode = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { immersiveMode: isSet$4(object.immersiveMode) ? webViewImmersiveModeFromJSON(object.immersiveMode) : 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.immersiveMode !== 0) {
        obj.immersiveMode = webViewImmersiveModeToJSON(message.immersiveMode);
      }
      return obj;
    },
    create(base) {
      return WebViewImmersiveModeChangedEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewImmersiveModeChangedEvent();
      message.immersiveMode = object.immersiveMode ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewImmersiveModeChangedEvent.$type, WebViewImmersiveModeChangedEvent);
  function createBaseWebViewInternalEventMessage() {
    return {
      id: "",
      formCanceled: void 0,
      formSubmitted: void 0,
      realtimeEvent: void 0,
      immersiveModeEvent: void 0,
      consentStatus: void 0
    };
  }
  const WebViewInternalEventMessage = {
    $type: "devvit.ui.events.v1alpha.WebViewInternalEventMessage",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.id !== "") {
        writer2.uint32(10).string(message.id);
      }
      if (message.formCanceled !== void 0) {
        FormCanceledEvent.encode(message.formCanceled, writer2.uint32(18).fork()).ldelim();
      }
      if (message.formSubmitted !== void 0) {
        FormSubmittedEvent.encode(message.formSubmitted, writer2.uint32(26).fork()).ldelim();
      }
      if (message.realtimeEvent !== void 0) {
        RealtimeSubscriptionEvent.encode(message.realtimeEvent, writer2.uint32(34).fork()).ldelim();
      }
      if (message.immersiveModeEvent !== void 0) {
        WebViewImmersiveModeChangedEvent.encode(message.immersiveModeEvent, writer2.uint32(42).fork()).ldelim();
      }
      if (message.consentStatus !== void 0) {
        ConsentStatusEvent.encode(message.consentStatus, writer2.uint32(58).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewInternalEventMessage();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.id = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.formCanceled = FormCanceledEvent.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.formSubmitted = FormSubmittedEvent.decode(reader2, reader2.uint32());
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.realtimeEvent = RealtimeSubscriptionEvent.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.immersiveModeEvent = WebViewImmersiveModeChangedEvent.decode(reader2, reader2.uint32());
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.consentStatus = ConsentStatusEvent.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        id: isSet$4(object.id) ? globalThis.String(object.id) : "",
        formCanceled: isSet$4(object.formCanceled) ? FormCanceledEvent.fromJSON(object.formCanceled) : void 0,
        formSubmitted: isSet$4(object.formSubmitted) ? FormSubmittedEvent.fromJSON(object.formSubmitted) : void 0,
        realtimeEvent: isSet$4(object.realtimeEvent) ? RealtimeSubscriptionEvent.fromJSON(object.realtimeEvent) : void 0,
        immersiveModeEvent: isSet$4(object.immersiveModeEvent) ? WebViewImmersiveModeChangedEvent.fromJSON(object.immersiveModeEvent) : void 0,
        consentStatus: isSet$4(object.consentStatus) ? ConsentStatusEvent.fromJSON(object.consentStatus) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.id !== "") {
        obj.id = message.id;
      }
      if (message.formCanceled !== void 0) {
        obj.formCanceled = FormCanceledEvent.toJSON(message.formCanceled);
      }
      if (message.formSubmitted !== void 0) {
        obj.formSubmitted = FormSubmittedEvent.toJSON(message.formSubmitted);
      }
      if (message.realtimeEvent !== void 0) {
        obj.realtimeEvent = RealtimeSubscriptionEvent.toJSON(message.realtimeEvent);
      }
      if (message.immersiveModeEvent !== void 0) {
        obj.immersiveModeEvent = WebViewImmersiveModeChangedEvent.toJSON(message.immersiveModeEvent);
      }
      if (message.consentStatus !== void 0) {
        obj.consentStatus = ConsentStatusEvent.toJSON(message.consentStatus);
      }
      return obj;
    },
    create(base) {
      return WebViewInternalEventMessage.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewInternalEventMessage();
      message.id = object.id ?? "";
      message.formCanceled = object.formCanceled !== void 0 && object.formCanceled !== null ? FormCanceledEvent.fromPartial(object.formCanceled) : void 0;
      message.formSubmitted = object.formSubmitted !== void 0 && object.formSubmitted !== null ? FormSubmittedEvent.fromPartial(object.formSubmitted) : void 0;
      message.realtimeEvent = object.realtimeEvent !== void 0 && object.realtimeEvent !== null ? RealtimeSubscriptionEvent.fromPartial(object.realtimeEvent) : void 0;
      message.immersiveModeEvent = object.immersiveModeEvent !== void 0 && object.immersiveModeEvent !== null ? WebViewImmersiveModeChangedEvent.fromPartial(object.immersiveModeEvent) : void 0;
      message.consentStatus = object.consentStatus !== void 0 && object.consentStatus !== null ? ConsentStatusEvent.fromPartial(object.consentStatus) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewInternalEventMessage.$type, WebViewInternalEventMessage);
  function createBaseWebViewMessageEvent() {
    return { data: void 0 };
  }
  const WebViewMessageEvent = {
    $type: "devvit.ui.events.v1alpha.WebViewMessageEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.data !== void 0) {
        WebViewMessageEvent_MessageData.encode(message.data, writer2.uint32(10).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewMessageEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.data = WebViewMessageEvent_MessageData.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return { data: isSet$4(object.data) ? WebViewMessageEvent_MessageData.fromJSON(object.data) : void 0 };
    },
    toJSON(message) {
      const obj = {};
      if (message.data !== void 0) {
        obj.data = WebViewMessageEvent_MessageData.toJSON(message.data);
      }
      return obj;
    },
    create(base) {
      return WebViewMessageEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewMessageEvent();
      message.data = object.data !== void 0 && object.data !== null ? WebViewMessageEvent_MessageData.fromPartial(object.data) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewMessageEvent.$type, WebViewMessageEvent);
  function createBaseWebViewMessageEvent_MessageData() {
    return { type: "", data: void 0 };
  }
  const WebViewMessageEvent_MessageData = {
    $type: "devvit.ui.events.v1alpha.WebViewMessageEvent.MessageData",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.type !== "") {
        writer2.uint32(10).string(message.type);
      }
      if (message.data !== void 0) {
        WebViewInternalEventMessage.encode(message.data, writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseWebViewMessageEvent_MessageData();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.type = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.data = WebViewInternalEventMessage.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        type: isSet$4(object.type) ? globalThis.String(object.type) : "",
        data: isSet$4(object.data) ? WebViewInternalEventMessage.fromJSON(object.data) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.type !== "") {
        obj.type = message.type;
      }
      if (message.data !== void 0) {
        obj.data = WebViewInternalEventMessage.toJSON(message.data);
      }
      return obj;
    },
    create(base) {
      return WebViewMessageEvent_MessageData.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseWebViewMessageEvent_MessageData();
      message.type = object.type ?? "";
      message.data = object.data !== void 0 && object.data !== null ? WebViewInternalEventMessage.fromPartial(object.data) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(WebViewMessageEvent_MessageData.$type, WebViewMessageEvent_MessageData);
  function isSet$4(value) {
    return value !== null && value !== void 0;
  }
  var UIEventScope;
  (function(UIEventScope2) {
    UIEventScope2[UIEventScope2["ALL"] = 0] = "ALL";
    UIEventScope2[UIEventScope2["LOCAL"] = 1] = "LOCAL";
    UIEventScope2[UIEventScope2["REMOTE"] = 2] = "REMOTE";
    UIEventScope2[UIEventScope2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(UIEventScope || (UIEventScope = {}));
  function uIEventScopeFromJSON(object) {
    switch (object) {
      case 0:
      case "ALL":
        return UIEventScope.ALL;
      case 1:
      case "LOCAL":
        return UIEventScope.LOCAL;
      case 2:
      case "REMOTE":
        return UIEventScope.REMOTE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return UIEventScope.UNRECOGNIZED;
    }
  }
  function uIEventScopeToJSON(object) {
    switch (object) {
      case UIEventScope.ALL:
        return 0;
      case UIEventScope.LOCAL:
        return 1;
      case UIEventScope.REMOTE:
        return 2;
      case UIEventScope.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseUIEvent() {
    return {
      realtimeEvent: void 0,
      formSubmitted: void 0,
      orderResult: void 0,
      toastAction: void 0,
      userAction: void 0,
      asyncRequest: void 0,
      asyncResponse: void 0,
      timer: void 0,
      blocking: void 0,
      resize: void 0,
      webView: void 0,
      formCanceled: void 0,
      consentStatus: void 0,
      async: void 0,
      hook: void 0,
      retry: void 0,
      scope: void 0
    };
  }
  const UIEvent = {
    $type: "devvit.ui.events.v1alpha.UIEvent",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.realtimeEvent !== void 0) {
        RealtimeSubscriptionEvent.encode(message.realtimeEvent, writer2.uint32(18).fork()).ldelim();
      }
      if (message.formSubmitted !== void 0) {
        FormSubmittedEvent.encode(message.formSubmitted, writer2.uint32(26).fork()).ldelim();
      }
      if (message.orderResult !== void 0) {
        OrderResultEvent.encode(message.orderResult, writer2.uint32(154).fork()).ldelim();
      }
      if (message.toastAction !== void 0) {
        ToastActionEvent.encode(message.toastAction, writer2.uint32(34).fork()).ldelim();
      }
      if (message.userAction !== void 0) {
        UserAction.encode(message.userAction, writer2.uint32(50).fork()).ldelim();
      }
      if (message.asyncRequest !== void 0) {
        AsyncRequest.encode(message.asyncRequest, writer2.uint32(58).fork()).ldelim();
      }
      if (message.asyncResponse !== void 0) {
        AsyncResponse.encode(message.asyncResponse, writer2.uint32(66).fork()).ldelim();
      }
      if (message.timer !== void 0) {
        TimerEvent.encode(message.timer, writer2.uint32(98).fork()).ldelim();
      }
      if (message.blocking !== void 0) {
        BlockingRenderEvent.encode(message.blocking, writer2.uint32(106).fork()).ldelim();
      }
      if (message.resize !== void 0) {
        ResizeEvent.encode(message.resize, writer2.uint32(114).fork()).ldelim();
      }
      if (message.webView !== void 0) {
        WebViewEvent.encode(message.webView, writer2.uint32(162).fork()).ldelim();
      }
      if (message.formCanceled !== void 0) {
        FormCanceledEvent.encode(message.formCanceled, writer2.uint32(170).fork()).ldelim();
      }
      if (message.consentStatus !== void 0) {
        ConsentStatusEvent.encode(message.consentStatus, writer2.uint32(186).fork()).ldelim();
      }
      if (message.async !== void 0) {
        BoolValue.encode({ value: message.async }, writer2.uint32(122).fork()).ldelim();
      }
      if (message.hook !== void 0) {
        StringValue.encode({ value: message.hook }, writer2.uint32(82).fork()).ldelim();
      }
      if (message.retry !== void 0) {
        BoolValue.encode({ value: message.retry }, writer2.uint32(138).fork()).ldelim();
      }
      if (message.scope !== void 0) {
        writer2.uint32(144).int32(message.scope);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUIEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 2:
            if (tag !== 18) {
              break;
            }
            message.realtimeEvent = RealtimeSubscriptionEvent.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.formSubmitted = FormSubmittedEvent.decode(reader2, reader2.uint32());
            continue;
          case 19:
            if (tag !== 154) {
              break;
            }
            message.orderResult = OrderResultEvent.decode(reader2, reader2.uint32());
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.toastAction = ToastActionEvent.decode(reader2, reader2.uint32());
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.userAction = UserAction.decode(reader2, reader2.uint32());
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.asyncRequest = AsyncRequest.decode(reader2, reader2.uint32());
            continue;
          case 8:
            if (tag !== 66) {
              break;
            }
            message.asyncResponse = AsyncResponse.decode(reader2, reader2.uint32());
            continue;
          case 12:
            if (tag !== 98) {
              break;
            }
            message.timer = TimerEvent.decode(reader2, reader2.uint32());
            continue;
          case 13:
            if (tag !== 106) {
              break;
            }
            message.blocking = BlockingRenderEvent.decode(reader2, reader2.uint32());
            continue;
          case 14:
            if (tag !== 114) {
              break;
            }
            message.resize = ResizeEvent.decode(reader2, reader2.uint32());
            continue;
          case 20:
            if (tag !== 162) {
              break;
            }
            message.webView = WebViewEvent.decode(reader2, reader2.uint32());
            continue;
          case 21:
            if (tag !== 170) {
              break;
            }
            message.formCanceled = FormCanceledEvent.decode(reader2, reader2.uint32());
            continue;
          case 23:
            if (tag !== 186) {
              break;
            }
            message.consentStatus = ConsentStatusEvent.decode(reader2, reader2.uint32());
            continue;
          case 15:
            if (tag !== 122) {
              break;
            }
            message.async = BoolValue.decode(reader2, reader2.uint32()).value;
            continue;
          case 10:
            if (tag !== 82) {
              break;
            }
            message.hook = StringValue.decode(reader2, reader2.uint32()).value;
            continue;
          case 17:
            if (tag !== 138) {
              break;
            }
            message.retry = BoolValue.decode(reader2, reader2.uint32()).value;
            continue;
          case 18:
            if (tag !== 144) {
              break;
            }
            message.scope = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        realtimeEvent: isSet$3(object.realtimeEvent) ? RealtimeSubscriptionEvent.fromJSON(object.realtimeEvent) : void 0,
        formSubmitted: isSet$3(object.formSubmitted) ? FormSubmittedEvent.fromJSON(object.formSubmitted) : void 0,
        orderResult: isSet$3(object.orderResult) ? OrderResultEvent.fromJSON(object.orderResult) : void 0,
        toastAction: isSet$3(object.toastAction) ? ToastActionEvent.fromJSON(object.toastAction) : void 0,
        userAction: isSet$3(object.userAction) ? UserAction.fromJSON(object.userAction) : void 0,
        asyncRequest: isSet$3(object.asyncRequest) ? AsyncRequest.fromJSON(object.asyncRequest) : void 0,
        asyncResponse: isSet$3(object.asyncResponse) ? AsyncResponse.fromJSON(object.asyncResponse) : void 0,
        timer: isSet$3(object.timer) ? TimerEvent.fromJSON(object.timer) : void 0,
        blocking: isSet$3(object.blocking) ? BlockingRenderEvent.fromJSON(object.blocking) : void 0,
        resize: isSet$3(object.resize) ? ResizeEvent.fromJSON(object.resize) : void 0,
        webView: isSet$3(object.webView) ? WebViewEvent.fromJSON(object.webView) : void 0,
        formCanceled: isSet$3(object.formCanceled) ? FormCanceledEvent.fromJSON(object.formCanceled) : void 0,
        consentStatus: isSet$3(object.consentStatus) ? ConsentStatusEvent.fromJSON(object.consentStatus) : void 0,
        async: isSet$3(object.async) ? Boolean(object.async) : void 0,
        hook: isSet$3(object.hook) ? String(object.hook) : void 0,
        retry: isSet$3(object.retry) ? Boolean(object.retry) : void 0,
        scope: isSet$3(object.scope) ? uIEventScopeFromJSON(object.scope) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.realtimeEvent !== void 0) {
        obj.realtimeEvent = RealtimeSubscriptionEvent.toJSON(message.realtimeEvent);
      }
      if (message.formSubmitted !== void 0) {
        obj.formSubmitted = FormSubmittedEvent.toJSON(message.formSubmitted);
      }
      if (message.orderResult !== void 0) {
        obj.orderResult = OrderResultEvent.toJSON(message.orderResult);
      }
      if (message.toastAction !== void 0) {
        obj.toastAction = ToastActionEvent.toJSON(message.toastAction);
      }
      if (message.userAction !== void 0) {
        obj.userAction = UserAction.toJSON(message.userAction);
      }
      if (message.asyncRequest !== void 0) {
        obj.asyncRequest = AsyncRequest.toJSON(message.asyncRequest);
      }
      if (message.asyncResponse !== void 0) {
        obj.asyncResponse = AsyncResponse.toJSON(message.asyncResponse);
      }
      if (message.timer !== void 0) {
        obj.timer = TimerEvent.toJSON(message.timer);
      }
      if (message.blocking !== void 0) {
        obj.blocking = BlockingRenderEvent.toJSON(message.blocking);
      }
      if (message.resize !== void 0) {
        obj.resize = ResizeEvent.toJSON(message.resize);
      }
      if (message.webView !== void 0) {
        obj.webView = WebViewEvent.toJSON(message.webView);
      }
      if (message.formCanceled !== void 0) {
        obj.formCanceled = FormCanceledEvent.toJSON(message.formCanceled);
      }
      if (message.consentStatus !== void 0) {
        obj.consentStatus = ConsentStatusEvent.toJSON(message.consentStatus);
      }
      if (message.async !== void 0) {
        obj.async = message.async;
      }
      if (message.hook !== void 0) {
        obj.hook = message.hook;
      }
      if (message.retry !== void 0) {
        obj.retry = message.retry;
      }
      if (message.scope !== void 0) {
        obj.scope = uIEventScopeToJSON(message.scope);
      }
      return obj;
    },
    create(base) {
      return UIEvent.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUIEvent();
      message.realtimeEvent = object.realtimeEvent !== void 0 && object.realtimeEvent !== null ? RealtimeSubscriptionEvent.fromPartial(object.realtimeEvent) : void 0;
      message.formSubmitted = object.formSubmitted !== void 0 && object.formSubmitted !== null ? FormSubmittedEvent.fromPartial(object.formSubmitted) : void 0;
      message.orderResult = object.orderResult !== void 0 && object.orderResult !== null ? OrderResultEvent.fromPartial(object.orderResult) : void 0;
      message.toastAction = object.toastAction !== void 0 && object.toastAction !== null ? ToastActionEvent.fromPartial(object.toastAction) : void 0;
      message.userAction = object.userAction !== void 0 && object.userAction !== null ? UserAction.fromPartial(object.userAction) : void 0;
      message.asyncRequest = object.asyncRequest !== void 0 && object.asyncRequest !== null ? AsyncRequest.fromPartial(object.asyncRequest) : void 0;
      message.asyncResponse = object.asyncResponse !== void 0 && object.asyncResponse !== null ? AsyncResponse.fromPartial(object.asyncResponse) : void 0;
      message.timer = object.timer !== void 0 && object.timer !== null ? TimerEvent.fromPartial(object.timer) : void 0;
      message.blocking = object.blocking !== void 0 && object.blocking !== null ? BlockingRenderEvent.fromPartial(object.blocking) : void 0;
      message.resize = object.resize !== void 0 && object.resize !== null ? ResizeEvent.fromPartial(object.resize) : void 0;
      message.webView = object.webView !== void 0 && object.webView !== null ? WebViewEvent.fromPartial(object.webView) : void 0;
      message.formCanceled = object.formCanceled !== void 0 && object.formCanceled !== null ? FormCanceledEvent.fromPartial(object.formCanceled) : void 0;
      message.consentStatus = object.consentStatus !== void 0 && object.consentStatus !== null ? ConsentStatusEvent.fromPartial(object.consentStatus) : void 0;
      message.async = object.async ?? void 0;
      message.hook = object.hook ?? void 0;
      message.retry = object.retry ?? void 0;
      message.scope = object.scope ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(UIEvent.$type, UIEvent);
  function createBaseBlockingRenderEvent() {
    return {};
  }
  const BlockingRenderEvent = {
    $type: "devvit.ui.events.v1alpha.BlockingRenderEvent",
    encode(_, writer2 = _m0.Writer.create()) {
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockingRenderEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(_) {
      return {};
    },
    toJSON(_) {
      const obj = {};
      return obj;
    },
    create(base) {
      return BlockingRenderEvent.fromPartial(base ?? {});
    },
    fromPartial(_) {
      const message = createBaseBlockingRenderEvent();
      return message;
    }
  };
  messageTypeRegistry.set(BlockingRenderEvent.$type, BlockingRenderEvent);
  function createBaseResizeEvent() {
    return {};
  }
  const ResizeEvent = {
    $type: "devvit.ui.events.v1alpha.ResizeEvent",
    encode(_, writer2 = _m0.Writer.create()) {
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseResizeEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(_) {
      return {};
    },
    toJSON(_) {
      const obj = {};
      return obj;
    },
    create(base) {
      return ResizeEvent.fromPartial(base ?? {});
    },
    fromPartial(_) {
      const message = createBaseResizeEvent();
      return message;
    }
  };
  messageTypeRegistry.set(ResizeEvent.$type, ResizeEvent);
  function createBaseTimerEvent() {
    return {};
  }
  const TimerEvent = {
    $type: "devvit.ui.events.v1alpha.TimerEvent",
    encode(_, writer2 = _m0.Writer.create()) {
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseTimerEvent();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(_) {
      return {};
    },
    toJSON(_) {
      const obj = {};
      return obj;
    },
    create(base) {
      return TimerEvent.fromPartial(base ?? {});
    },
    fromPartial(_) {
      const message = createBaseTimerEvent();
      return message;
    }
  };
  messageTypeRegistry.set(TimerEvent.$type, TimerEvent);
  function createBaseAsyncError() {
    return { message: "", details: "" };
  }
  const AsyncError = {
    $type: "devvit.ui.events.v1alpha.AsyncError",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.message !== "") {
        writer2.uint32(10).string(message.message);
      }
      if (message.details !== "") {
        writer2.uint32(18).string(message.details);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseAsyncError();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.message = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.details = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        message: isSet$3(object.message) ? globalThis.String(object.message) : "",
        details: isSet$3(object.details) ? globalThis.String(object.details) : ""
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.message !== "") {
        obj.message = message.message;
      }
      if (message.details !== "") {
        obj.details = message.details;
      }
      return obj;
    },
    create(base) {
      return AsyncError.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseAsyncError();
      message.message = object.message ?? "";
      message.details = object.details ?? "";
      return message;
    }
  };
  messageTypeRegistry.set(AsyncError.$type, AsyncError);
  function createBaseAsyncRequest() {
    return { requestId: "", data: void 0 };
  }
  const AsyncRequest = {
    $type: "devvit.ui.events.v1alpha.AsyncRequest",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.requestId !== "") {
        writer2.uint32(10).string(message.requestId);
      }
      if (message.data !== void 0) {
        Struct.encode(Struct.wrap(message.data), writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseAsyncRequest();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.requestId = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.data = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        requestId: isSet$3(object.requestId) ? globalThis.String(object.requestId) : "",
        data: isObject$3(object.data) ? object.data : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.requestId !== "") {
        obj.requestId = message.requestId;
      }
      if (message.data !== void 0) {
        obj.data = message.data;
      }
      return obj;
    },
    create(base) {
      return AsyncRequest.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseAsyncRequest();
      message.requestId = object.requestId ?? "";
      message.data = object.data ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(AsyncRequest.$type, AsyncRequest);
  function createBaseAsyncResponse() {
    return { requestId: "", data: void 0, error: void 0 };
  }
  const AsyncResponse = {
    $type: "devvit.ui.events.v1alpha.AsyncResponse",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.requestId !== "") {
        writer2.uint32(10).string(message.requestId);
      }
      if (message.data !== void 0) {
        Struct.encode(Struct.wrap(message.data), writer2.uint32(18).fork()).ldelim();
      }
      if (message.error !== void 0) {
        AsyncError.encode(message.error, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseAsyncResponse();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.requestId = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.data = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.error = AsyncError.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        requestId: isSet$3(object.requestId) ? globalThis.String(object.requestId) : "",
        data: isObject$3(object.data) ? object.data : void 0,
        error: isSet$3(object.error) ? AsyncError.fromJSON(object.error) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.requestId !== "") {
        obj.requestId = message.requestId;
      }
      if (message.data !== void 0) {
        obj.data = message.data;
      }
      if (message.error !== void 0) {
        obj.error = AsyncError.toJSON(message.error);
      }
      return obj;
    },
    create(base) {
      return AsyncResponse.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseAsyncResponse();
      message.requestId = object.requestId ?? "";
      message.data = object.data ?? void 0;
      message.error = object.error !== void 0 && object.error !== null ? AsyncError.fromPartial(object.error) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(AsyncResponse.$type, AsyncResponse);
  function createBaseUserAction() {
    return { actionId: "", data: void 0 };
  }
  const UserAction = {
    $type: "devvit.ui.events.v1alpha.UserAction",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.actionId !== "") {
        writer2.uint32(10).string(message.actionId);
      }
      if (message.data !== void 0) {
        Struct.encode(Struct.wrap(message.data), writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUserAction();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.actionId = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.data = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        actionId: isSet$3(object.actionId) ? globalThis.String(object.actionId) : "",
        data: isObject$3(object.data) ? object.data : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.actionId !== "") {
        obj.actionId = message.actionId;
      }
      if (message.data !== void 0) {
        obj.data = message.data;
      }
      return obj;
    },
    create(base) {
      return UserAction.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUserAction();
      message.actionId = object.actionId ?? "";
      message.data = object.data ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(UserAction.$type, UserAction);
  function isObject$3(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$3(value) {
    return value !== null && value !== void 0;
  }
  var BlockType;
  (function(BlockType2) {
    BlockType2[BlockType2["BLOCK_ROOT"] = 0] = "BLOCK_ROOT";
    BlockType2[BlockType2["BLOCK_STACK"] = 1] = "BLOCK_STACK";
    BlockType2[BlockType2["BLOCK_TEXT"] = 2] = "BLOCK_TEXT";
    BlockType2[BlockType2["BLOCK_BUTTON"] = 3] = "BLOCK_BUTTON";
    BlockType2[BlockType2["BLOCK_IMAGE"] = 4] = "BLOCK_IMAGE";
    BlockType2[BlockType2["BLOCK_SPACER"] = 5] = "BLOCK_SPACER";
    BlockType2[BlockType2["BLOCK_ICON"] = 6] = "BLOCK_ICON";
    BlockType2[BlockType2["BLOCK_AVATAR"] = 7] = "BLOCK_AVATAR";
    BlockType2[BlockType2["BLOCK_FULLSNOO"] = 8] = "BLOCK_FULLSNOO";
    BlockType2[BlockType2["BLOCK_ANIMATION"] = 9] = "BLOCK_ANIMATION";
    BlockType2[BlockType2["BLOCK_WEBVIEW"] = 100] = "BLOCK_WEBVIEW";
    BlockType2[BlockType2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockType || (BlockType = {}));
  function blockTypeFromJSON(object) {
    switch (object) {
      case 0:
      case "BLOCK_ROOT":
        return BlockType.BLOCK_ROOT;
      case 1:
      case "BLOCK_STACK":
        return BlockType.BLOCK_STACK;
      case 2:
      case "BLOCK_TEXT":
        return BlockType.BLOCK_TEXT;
      case 3:
      case "BLOCK_BUTTON":
        return BlockType.BLOCK_BUTTON;
      case 4:
      case "BLOCK_IMAGE":
        return BlockType.BLOCK_IMAGE;
      case 5:
      case "BLOCK_SPACER":
        return BlockType.BLOCK_SPACER;
      case 6:
      case "BLOCK_ICON":
        return BlockType.BLOCK_ICON;
      case 7:
      case "BLOCK_AVATAR":
        return BlockType.BLOCK_AVATAR;
      case 8:
      case "BLOCK_FULLSNOO":
        return BlockType.BLOCK_FULLSNOO;
      case 9:
      case "BLOCK_ANIMATION":
        return BlockType.BLOCK_ANIMATION;
      case 100:
      case "BLOCK_WEBVIEW":
        return BlockType.BLOCK_WEBVIEW;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockType.UNRECOGNIZED;
    }
  }
  function blockTypeToJSON(object) {
    switch (object) {
      case BlockType.BLOCK_ROOT:
        return 0;
      case BlockType.BLOCK_STACK:
        return 1;
      case BlockType.BLOCK_TEXT:
        return 2;
      case BlockType.BLOCK_BUTTON:
        return 3;
      case BlockType.BLOCK_IMAGE:
        return 4;
      case BlockType.BLOCK_SPACER:
        return 5;
      case BlockType.BLOCK_ICON:
        return 6;
      case BlockType.BLOCK_AVATAR:
        return 7;
      case BlockType.BLOCK_FULLSNOO:
        return 8;
      case BlockType.BLOCK_ANIMATION:
        return 9;
      case BlockType.BLOCK_WEBVIEW:
        return 100;
      case BlockType.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockActionType;
  (function(BlockActionType2) {
    BlockActionType2[BlockActionType2["ACTION_CLICK"] = 0] = "ACTION_CLICK";
    BlockActionType2[BlockActionType2["ACTION_WEBVIEW"] = 100] = "ACTION_WEBVIEW";
    BlockActionType2[BlockActionType2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockActionType || (BlockActionType = {}));
  function blockActionTypeFromJSON(object) {
    switch (object) {
      case 0:
      case "ACTION_CLICK":
        return BlockActionType.ACTION_CLICK;
      case 100:
      case "ACTION_WEBVIEW":
        return BlockActionType.ACTION_WEBVIEW;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockActionType.UNRECOGNIZED;
    }
  }
  function blockActionTypeToJSON(object) {
    switch (object) {
      case BlockActionType.ACTION_CLICK:
        return 0;
      case BlockActionType.ACTION_WEBVIEW:
        return 100;
      case BlockActionType.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockRenderEventType;
  (function(BlockRenderEventType2) {
    BlockRenderEventType2[BlockRenderEventType2["RENDER_INITIAL"] = 0] = "RENDER_INITIAL";
    BlockRenderEventType2[BlockRenderEventType2["RENDER_USER_ACTION"] = 1] = "RENDER_USER_ACTION";
    BlockRenderEventType2[BlockRenderEventType2["RENDER_EFFECT_EVENT"] = 2] = "RENDER_EFFECT_EVENT";
    BlockRenderEventType2[BlockRenderEventType2["RENDER_CACHED"] = 3] = "RENDER_CACHED";
    BlockRenderEventType2[BlockRenderEventType2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockRenderEventType || (BlockRenderEventType = {}));
  var BlockStackDirection;
  (function(BlockStackDirection2) {
    BlockStackDirection2[BlockStackDirection2["STACK_HORIZONTAL"] = 0] = "STACK_HORIZONTAL";
    BlockStackDirection2[BlockStackDirection2["STACK_VERTICAL"] = 1] = "STACK_VERTICAL";
    BlockStackDirection2[BlockStackDirection2["STACK_DEPTH"] = 2] = "STACK_DEPTH";
    BlockStackDirection2[BlockStackDirection2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockStackDirection || (BlockStackDirection = {}));
  function blockStackDirectionFromJSON(object) {
    switch (object) {
      case 0:
      case "STACK_HORIZONTAL":
        return BlockStackDirection.STACK_HORIZONTAL;
      case 1:
      case "STACK_VERTICAL":
        return BlockStackDirection.STACK_VERTICAL;
      case 2:
      case "STACK_DEPTH":
        return BlockStackDirection.STACK_DEPTH;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockStackDirection.UNRECOGNIZED;
    }
  }
  function blockStackDirectionToJSON(object) {
    switch (object) {
      case BlockStackDirection.STACK_HORIZONTAL:
        return 0;
      case BlockStackDirection.STACK_VERTICAL:
        return 1;
      case BlockStackDirection.STACK_DEPTH:
        return 2;
      case BlockStackDirection.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockBorderWidth;
  (function(BlockBorderWidth2) {
    BlockBorderWidth2[BlockBorderWidth2["BORDER_WIDTH_NONE"] = 0] = "BORDER_WIDTH_NONE";
    BlockBorderWidth2[BlockBorderWidth2["BORDER_WIDTH_THIN"] = 1] = "BORDER_WIDTH_THIN";
    BlockBorderWidth2[BlockBorderWidth2["BORDER_WIDTH_THICK"] = 2] = "BORDER_WIDTH_THICK";
    BlockBorderWidth2[BlockBorderWidth2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockBorderWidth || (BlockBorderWidth = {}));
  function blockBorderWidthFromJSON(object) {
    switch (object) {
      case 0:
      case "BORDER_WIDTH_NONE":
        return BlockBorderWidth.BORDER_WIDTH_NONE;
      case 1:
      case "BORDER_WIDTH_THIN":
        return BlockBorderWidth.BORDER_WIDTH_THIN;
      case 2:
      case "BORDER_WIDTH_THICK":
        return BlockBorderWidth.BORDER_WIDTH_THICK;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockBorderWidth.UNRECOGNIZED;
    }
  }
  function blockBorderWidthToJSON(object) {
    switch (object) {
      case BlockBorderWidth.BORDER_WIDTH_NONE:
        return 0;
      case BlockBorderWidth.BORDER_WIDTH_THIN:
        return 1;
      case BlockBorderWidth.BORDER_WIDTH_THICK:
        return 2;
      case BlockBorderWidth.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockVerticalAlignment;
  (function(BlockVerticalAlignment2) {
    BlockVerticalAlignment2[BlockVerticalAlignment2["ALIGN_TOP"] = 0] = "ALIGN_TOP";
    BlockVerticalAlignment2[BlockVerticalAlignment2["ALIGN_MIDDLE"] = 1] = "ALIGN_MIDDLE";
    BlockVerticalAlignment2[BlockVerticalAlignment2["ALIGN_BOTTOM"] = 2] = "ALIGN_BOTTOM";
    BlockVerticalAlignment2[BlockVerticalAlignment2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockVerticalAlignment || (BlockVerticalAlignment = {}));
  function blockVerticalAlignmentFromJSON(object) {
    switch (object) {
      case 0:
      case "ALIGN_TOP":
        return BlockVerticalAlignment.ALIGN_TOP;
      case 1:
      case "ALIGN_MIDDLE":
        return BlockVerticalAlignment.ALIGN_MIDDLE;
      case 2:
      case "ALIGN_BOTTOM":
        return BlockVerticalAlignment.ALIGN_BOTTOM;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockVerticalAlignment.UNRECOGNIZED;
    }
  }
  function blockVerticalAlignmentToJSON(object) {
    switch (object) {
      case BlockVerticalAlignment.ALIGN_TOP:
        return 0;
      case BlockVerticalAlignment.ALIGN_MIDDLE:
        return 1;
      case BlockVerticalAlignment.ALIGN_BOTTOM:
        return 2;
      case BlockVerticalAlignment.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockHorizontalAlignment;
  (function(BlockHorizontalAlignment2) {
    BlockHorizontalAlignment2[BlockHorizontalAlignment2["ALIGN_START"] = 0] = "ALIGN_START";
    BlockHorizontalAlignment2[BlockHorizontalAlignment2["ALIGN_CENTER"] = 1] = "ALIGN_CENTER";
    BlockHorizontalAlignment2[BlockHorizontalAlignment2["ALIGN_END"] = 2] = "ALIGN_END";
    BlockHorizontalAlignment2[BlockHorizontalAlignment2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockHorizontalAlignment || (BlockHorizontalAlignment = {}));
  function blockHorizontalAlignmentFromJSON(object) {
    switch (object) {
      case 0:
      case "ALIGN_START":
        return BlockHorizontalAlignment.ALIGN_START;
      case 1:
      case "ALIGN_CENTER":
        return BlockHorizontalAlignment.ALIGN_CENTER;
      case 2:
      case "ALIGN_END":
        return BlockHorizontalAlignment.ALIGN_END;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockHorizontalAlignment.UNRECOGNIZED;
    }
  }
  function blockHorizontalAlignmentToJSON(object) {
    switch (object) {
      case BlockHorizontalAlignment.ALIGN_START:
        return 0;
      case BlockHorizontalAlignment.ALIGN_CENTER:
        return 1;
      case BlockHorizontalAlignment.ALIGN_END:
        return 2;
      case BlockHorizontalAlignment.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockPadding;
  (function(BlockPadding2) {
    BlockPadding2[BlockPadding2["PADDING_NONE"] = 0] = "PADDING_NONE";
    BlockPadding2[BlockPadding2["PADDING_XSMALL"] = 1] = "PADDING_XSMALL";
    BlockPadding2[BlockPadding2["PADDING_SMALL"] = 2] = "PADDING_SMALL";
    BlockPadding2[BlockPadding2["PADDING_MEDIUM"] = 3] = "PADDING_MEDIUM";
    BlockPadding2[BlockPadding2["PADDING_LARGE"] = 4] = "PADDING_LARGE";
    BlockPadding2[BlockPadding2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockPadding || (BlockPadding = {}));
  function blockPaddingFromJSON(object) {
    switch (object) {
      case 0:
      case "PADDING_NONE":
        return BlockPadding.PADDING_NONE;
      case 1:
      case "PADDING_XSMALL":
        return BlockPadding.PADDING_XSMALL;
      case 2:
      case "PADDING_SMALL":
        return BlockPadding.PADDING_SMALL;
      case 3:
      case "PADDING_MEDIUM":
        return BlockPadding.PADDING_MEDIUM;
      case 4:
      case "PADDING_LARGE":
        return BlockPadding.PADDING_LARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockPadding.UNRECOGNIZED;
    }
  }
  function blockPaddingToJSON(object) {
    switch (object) {
      case BlockPadding.PADDING_NONE:
        return 0;
      case BlockPadding.PADDING_XSMALL:
        return 1;
      case BlockPadding.PADDING_SMALL:
        return 2;
      case BlockPadding.PADDING_MEDIUM:
        return 3;
      case BlockPadding.PADDING_LARGE:
        return 4;
      case BlockPadding.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockGap;
  (function(BlockGap2) {
    BlockGap2[BlockGap2["GAP_NONE"] = 0] = "GAP_NONE";
    BlockGap2[BlockGap2["GAP_SMALL"] = 1] = "GAP_SMALL";
    BlockGap2[BlockGap2["GAP_MEDIUM"] = 2] = "GAP_MEDIUM";
    BlockGap2[BlockGap2["GAP_LARGE"] = 3] = "GAP_LARGE";
    BlockGap2[BlockGap2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockGap || (BlockGap = {}));
  function blockGapFromJSON(object) {
    switch (object) {
      case 0:
      case "GAP_NONE":
        return BlockGap.GAP_NONE;
      case 1:
      case "GAP_SMALL":
        return BlockGap.GAP_SMALL;
      case 2:
      case "GAP_MEDIUM":
        return BlockGap.GAP_MEDIUM;
      case 3:
      case "GAP_LARGE":
        return BlockGap.GAP_LARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockGap.UNRECOGNIZED;
    }
  }
  function blockGapToJSON(object) {
    switch (object) {
      case BlockGap.GAP_NONE:
        return 0;
      case BlockGap.GAP_SMALL:
        return 1;
      case BlockGap.GAP_MEDIUM:
        return 2;
      case BlockGap.GAP_LARGE:
        return 3;
      case BlockGap.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockRadius;
  (function(BlockRadius2) {
    BlockRadius2[BlockRadius2["RADIUS_NONE"] = 0] = "RADIUS_NONE";
    BlockRadius2[BlockRadius2["RADIUS_SMALL"] = 1] = "RADIUS_SMALL";
    BlockRadius2[BlockRadius2["RADIUS_MEDIUM"] = 2] = "RADIUS_MEDIUM";
    BlockRadius2[BlockRadius2["RADIUS_LARGE"] = 3] = "RADIUS_LARGE";
    BlockRadius2[BlockRadius2["RADIUS_FULL"] = 100] = "RADIUS_FULL";
    BlockRadius2[BlockRadius2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockRadius || (BlockRadius = {}));
  function blockRadiusFromJSON(object) {
    switch (object) {
      case 0:
      case "RADIUS_NONE":
        return BlockRadius.RADIUS_NONE;
      case 1:
      case "RADIUS_SMALL":
        return BlockRadius.RADIUS_SMALL;
      case 2:
      case "RADIUS_MEDIUM":
        return BlockRadius.RADIUS_MEDIUM;
      case 3:
      case "RADIUS_LARGE":
        return BlockRadius.RADIUS_LARGE;
      case 100:
      case "RADIUS_FULL":
        return BlockRadius.RADIUS_FULL;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockRadius.UNRECOGNIZED;
    }
  }
  function blockRadiusToJSON(object) {
    switch (object) {
      case BlockRadius.RADIUS_NONE:
        return 0;
      case BlockRadius.RADIUS_SMALL:
        return 1;
      case BlockRadius.RADIUS_MEDIUM:
        return 2;
      case BlockRadius.RADIUS_LARGE:
        return 3;
      case BlockRadius.RADIUS_FULL:
        return 100;
      case BlockRadius.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockTextSize;
  (function(BlockTextSize2) {
    BlockTextSize2[BlockTextSize2["TEXT_SIZE_MEDIUM"] = 0] = "TEXT_SIZE_MEDIUM";
    BlockTextSize2[BlockTextSize2["TEXT_SIZE_XSMALL"] = 1] = "TEXT_SIZE_XSMALL";
    BlockTextSize2[BlockTextSize2["TEXT_SIZE_SMALL"] = 2] = "TEXT_SIZE_SMALL";
    BlockTextSize2[BlockTextSize2["TEXT_SIZE_LARGE"] = 3] = "TEXT_SIZE_LARGE";
    BlockTextSize2[BlockTextSize2["TEXT_SIZE_XLARGE"] = 4] = "TEXT_SIZE_XLARGE";
    BlockTextSize2[BlockTextSize2["TEXT_SIZE_XXLARGE"] = 5] = "TEXT_SIZE_XXLARGE";
    BlockTextSize2[BlockTextSize2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockTextSize || (BlockTextSize = {}));
  function blockTextSizeFromJSON(object) {
    switch (object) {
      case 0:
      case "TEXT_SIZE_MEDIUM":
        return BlockTextSize.TEXT_SIZE_MEDIUM;
      case 1:
      case "TEXT_SIZE_XSMALL":
        return BlockTextSize.TEXT_SIZE_XSMALL;
      case 2:
      case "TEXT_SIZE_SMALL":
        return BlockTextSize.TEXT_SIZE_SMALL;
      case 3:
      case "TEXT_SIZE_LARGE":
        return BlockTextSize.TEXT_SIZE_LARGE;
      case 4:
      case "TEXT_SIZE_XLARGE":
        return BlockTextSize.TEXT_SIZE_XLARGE;
      case 5:
      case "TEXT_SIZE_XXLARGE":
        return BlockTextSize.TEXT_SIZE_XXLARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockTextSize.UNRECOGNIZED;
    }
  }
  function blockTextSizeToJSON(object) {
    switch (object) {
      case BlockTextSize.TEXT_SIZE_MEDIUM:
        return 0;
      case BlockTextSize.TEXT_SIZE_XSMALL:
        return 1;
      case BlockTextSize.TEXT_SIZE_SMALL:
        return 2;
      case BlockTextSize.TEXT_SIZE_LARGE:
        return 3;
      case BlockTextSize.TEXT_SIZE_XLARGE:
        return 4;
      case BlockTextSize.TEXT_SIZE_XXLARGE:
        return 5;
      case BlockTextSize.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockTextWeight;
  (function(BlockTextWeight2) {
    BlockTextWeight2[BlockTextWeight2["TEXT_WEIGHT_REGULAR"] = 0] = "TEXT_WEIGHT_REGULAR";
    BlockTextWeight2[BlockTextWeight2["TEXT_WEIGHT_BOLD"] = 1] = "TEXT_WEIGHT_BOLD";
    BlockTextWeight2[BlockTextWeight2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockTextWeight || (BlockTextWeight = {}));
  function blockTextWeightFromJSON(object) {
    switch (object) {
      case 0:
      case "TEXT_WEIGHT_REGULAR":
        return BlockTextWeight.TEXT_WEIGHT_REGULAR;
      case 1:
      case "TEXT_WEIGHT_BOLD":
        return BlockTextWeight.TEXT_WEIGHT_BOLD;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockTextWeight.UNRECOGNIZED;
    }
  }
  function blockTextWeightToJSON(object) {
    switch (object) {
      case BlockTextWeight.TEXT_WEIGHT_REGULAR:
        return 0;
      case BlockTextWeight.TEXT_WEIGHT_BOLD:
        return 1;
      case BlockTextWeight.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockTextOutline;
  (function(BlockTextOutline2) {
    BlockTextOutline2[BlockTextOutline2["TEXT_OUTLINE_NONE"] = 0] = "TEXT_OUTLINE_NONE";
    BlockTextOutline2[BlockTextOutline2["TEXT_OUTLINE_THIN"] = 1] = "TEXT_OUTLINE_THIN";
    BlockTextOutline2[BlockTextOutline2["TEXT_OUTLINE_THICK"] = 2] = "TEXT_OUTLINE_THICK";
    BlockTextOutline2[BlockTextOutline2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockTextOutline || (BlockTextOutline = {}));
  function blockTextOutlineFromJSON(object) {
    switch (object) {
      case 0:
      case "TEXT_OUTLINE_NONE":
        return BlockTextOutline.TEXT_OUTLINE_NONE;
      case 1:
      case "TEXT_OUTLINE_THIN":
        return BlockTextOutline.TEXT_OUTLINE_THIN;
      case 2:
      case "TEXT_OUTLINE_THICK":
        return BlockTextOutline.TEXT_OUTLINE_THICK;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockTextOutline.UNRECOGNIZED;
    }
  }
  function blockTextOutlineToJSON(object) {
    switch (object) {
      case BlockTextOutline.TEXT_OUTLINE_NONE:
        return 0;
      case BlockTextOutline.TEXT_OUTLINE_THIN:
        return 1;
      case BlockTextOutline.TEXT_OUTLINE_THICK:
        return 2;
      case BlockTextOutline.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockTextStyle;
  (function(BlockTextStyle2) {
    BlockTextStyle2[BlockTextStyle2["TEXT_STYLE_BODY"] = 0] = "TEXT_STYLE_BODY";
    BlockTextStyle2[BlockTextStyle2["TEXT_STYLE_METADATA"] = 1] = "TEXT_STYLE_METADATA";
    BlockTextStyle2[BlockTextStyle2["TEXT_STYLE_HEADING"] = 2] = "TEXT_STYLE_HEADING";
    BlockTextStyle2[BlockTextStyle2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockTextStyle || (BlockTextStyle = {}));
  function blockTextStyleFromJSON(object) {
    switch (object) {
      case 0:
      case "TEXT_STYLE_BODY":
        return BlockTextStyle.TEXT_STYLE_BODY;
      case 1:
      case "TEXT_STYLE_METADATA":
        return BlockTextStyle.TEXT_STYLE_METADATA;
      case 2:
      case "TEXT_STYLE_HEADING":
        return BlockTextStyle.TEXT_STYLE_HEADING;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockTextStyle.UNRECOGNIZED;
    }
  }
  function blockTextStyleToJSON(object) {
    switch (object) {
      case BlockTextStyle.TEXT_STYLE_BODY:
        return 0;
      case BlockTextStyle.TEXT_STYLE_METADATA:
        return 1;
      case BlockTextStyle.TEXT_STYLE_HEADING:
        return 2;
      case BlockTextStyle.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockButtonSize;
  (function(BlockButtonSize2) {
    BlockButtonSize2[BlockButtonSize2["BUTTON_SIZE_MEDIUM"] = 0] = "BUTTON_SIZE_MEDIUM";
    BlockButtonSize2[BlockButtonSize2["BUTTON_SIZE_SMALL"] = 1] = "BUTTON_SIZE_SMALL";
    BlockButtonSize2[BlockButtonSize2["BUTTON_SIZE_LARGE"] = 2] = "BUTTON_SIZE_LARGE";
    BlockButtonSize2[BlockButtonSize2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockButtonSize || (BlockButtonSize = {}));
  function blockButtonSizeFromJSON(object) {
    switch (object) {
      case 0:
      case "BUTTON_SIZE_MEDIUM":
        return BlockButtonSize.BUTTON_SIZE_MEDIUM;
      case 1:
      case "BUTTON_SIZE_SMALL":
        return BlockButtonSize.BUTTON_SIZE_SMALL;
      case 2:
      case "BUTTON_SIZE_LARGE":
        return BlockButtonSize.BUTTON_SIZE_LARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockButtonSize.UNRECOGNIZED;
    }
  }
  function blockButtonSizeToJSON(object) {
    switch (object) {
      case BlockButtonSize.BUTTON_SIZE_MEDIUM:
        return 0;
      case BlockButtonSize.BUTTON_SIZE_SMALL:
        return 1;
      case BlockButtonSize.BUTTON_SIZE_LARGE:
        return 2;
      case BlockButtonSize.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockButtonAppearance;
  (function(BlockButtonAppearance2) {
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_SECONDARY"] = 0] = "BUTTON_APPEARANCE_SECONDARY";
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_PRIMARY"] = 1] = "BUTTON_APPEARANCE_PRIMARY";
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_PLAIN"] = 2] = "BUTTON_APPEARANCE_PLAIN";
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_BORDERED"] = 3] = "BUTTON_APPEARANCE_BORDERED";
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_MEDIA"] = 4] = "BUTTON_APPEARANCE_MEDIA";
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_DESTRUCTIVE"] = 5] = "BUTTON_APPEARANCE_DESTRUCTIVE";
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_CAUTION"] = 6] = "BUTTON_APPEARANCE_CAUTION";
    BlockButtonAppearance2[BlockButtonAppearance2["BUTTON_APPEARANCE_SUCCESS"] = 7] = "BUTTON_APPEARANCE_SUCCESS";
    BlockButtonAppearance2[BlockButtonAppearance2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockButtonAppearance || (BlockButtonAppearance = {}));
  function blockButtonAppearanceFromJSON(object) {
    switch (object) {
      case 0:
      case "BUTTON_APPEARANCE_SECONDARY":
        return BlockButtonAppearance.BUTTON_APPEARANCE_SECONDARY;
      case 1:
      case "BUTTON_APPEARANCE_PRIMARY":
        return BlockButtonAppearance.BUTTON_APPEARANCE_PRIMARY;
      case 2:
      case "BUTTON_APPEARANCE_PLAIN":
        return BlockButtonAppearance.BUTTON_APPEARANCE_PLAIN;
      case 3:
      case "BUTTON_APPEARANCE_BORDERED":
        return BlockButtonAppearance.BUTTON_APPEARANCE_BORDERED;
      case 4:
      case "BUTTON_APPEARANCE_MEDIA":
        return BlockButtonAppearance.BUTTON_APPEARANCE_MEDIA;
      case 5:
      case "BUTTON_APPEARANCE_DESTRUCTIVE":
        return BlockButtonAppearance.BUTTON_APPEARANCE_DESTRUCTIVE;
      case 6:
      case "BUTTON_APPEARANCE_CAUTION":
        return BlockButtonAppearance.BUTTON_APPEARANCE_CAUTION;
      case 7:
      case "BUTTON_APPEARANCE_SUCCESS":
        return BlockButtonAppearance.BUTTON_APPEARANCE_SUCCESS;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockButtonAppearance.UNRECOGNIZED;
    }
  }
  function blockButtonAppearanceToJSON(object) {
    switch (object) {
      case BlockButtonAppearance.BUTTON_APPEARANCE_SECONDARY:
        return 0;
      case BlockButtonAppearance.BUTTON_APPEARANCE_PRIMARY:
        return 1;
      case BlockButtonAppearance.BUTTON_APPEARANCE_PLAIN:
        return 2;
      case BlockButtonAppearance.BUTTON_APPEARANCE_BORDERED:
        return 3;
      case BlockButtonAppearance.BUTTON_APPEARANCE_MEDIA:
        return 4;
      case BlockButtonAppearance.BUTTON_APPEARANCE_DESTRUCTIVE:
        return 5;
      case BlockButtonAppearance.BUTTON_APPEARANCE_CAUTION:
        return 6;
      case BlockButtonAppearance.BUTTON_APPEARANCE_SUCCESS:
        return 7;
      case BlockButtonAppearance.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockImageResizeMode;
  (function(BlockImageResizeMode2) {
    BlockImageResizeMode2[BlockImageResizeMode2["IMAGE_RESIZE_FIT"] = 0] = "IMAGE_RESIZE_FIT";
    BlockImageResizeMode2[BlockImageResizeMode2["IMAGE_RESIZE_FILL"] = 1] = "IMAGE_RESIZE_FILL";
    BlockImageResizeMode2[BlockImageResizeMode2["IMAGE_RESIZE_COVER"] = 2] = "IMAGE_RESIZE_COVER";
    BlockImageResizeMode2[BlockImageResizeMode2["IMAGE_RESIZE_NONE"] = 3] = "IMAGE_RESIZE_NONE";
    BlockImageResizeMode2[BlockImageResizeMode2["IMAGE_RESIZE_SCALE_DOWN"] = 4] = "IMAGE_RESIZE_SCALE_DOWN";
    BlockImageResizeMode2[BlockImageResizeMode2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockImageResizeMode || (BlockImageResizeMode = {}));
  function blockImageResizeModeFromJSON(object) {
    switch (object) {
      case 0:
      case "IMAGE_RESIZE_FIT":
        return BlockImageResizeMode.IMAGE_RESIZE_FIT;
      case 1:
      case "IMAGE_RESIZE_FILL":
        return BlockImageResizeMode.IMAGE_RESIZE_FILL;
      case 2:
      case "IMAGE_RESIZE_COVER":
        return BlockImageResizeMode.IMAGE_RESIZE_COVER;
      case 3:
      case "IMAGE_RESIZE_NONE":
        return BlockImageResizeMode.IMAGE_RESIZE_NONE;
      case 4:
      case "IMAGE_RESIZE_SCALE_DOWN":
        return BlockImageResizeMode.IMAGE_RESIZE_SCALE_DOWN;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockImageResizeMode.UNRECOGNIZED;
    }
  }
  function blockImageResizeModeToJSON(object) {
    switch (object) {
      case BlockImageResizeMode.IMAGE_RESIZE_FIT:
        return 0;
      case BlockImageResizeMode.IMAGE_RESIZE_FILL:
        return 1;
      case BlockImageResizeMode.IMAGE_RESIZE_COVER:
        return 2;
      case BlockImageResizeMode.IMAGE_RESIZE_NONE:
        return 3;
      case BlockImageResizeMode.IMAGE_RESIZE_SCALE_DOWN:
        return 4;
      case BlockImageResizeMode.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockSpacerSize;
  (function(BlockSpacerSize2) {
    BlockSpacerSize2[BlockSpacerSize2["SPACER_SMALL"] = 0] = "SPACER_SMALL";
    BlockSpacerSize2[BlockSpacerSize2["SPACER_XSMALL"] = 1] = "SPACER_XSMALL";
    BlockSpacerSize2[BlockSpacerSize2["SPACER_MEDIUM"] = 2] = "SPACER_MEDIUM";
    BlockSpacerSize2[BlockSpacerSize2["SPACER_LARGE"] = 3] = "SPACER_LARGE";
    BlockSpacerSize2[BlockSpacerSize2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockSpacerSize || (BlockSpacerSize = {}));
  function blockSpacerSizeFromJSON(object) {
    switch (object) {
      case 0:
      case "SPACER_SMALL":
        return BlockSpacerSize.SPACER_SMALL;
      case 1:
      case "SPACER_XSMALL":
        return BlockSpacerSize.SPACER_XSMALL;
      case 2:
      case "SPACER_MEDIUM":
        return BlockSpacerSize.SPACER_MEDIUM;
      case 3:
      case "SPACER_LARGE":
        return BlockSpacerSize.SPACER_LARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockSpacerSize.UNRECOGNIZED;
    }
  }
  function blockSpacerSizeToJSON(object) {
    switch (object) {
      case BlockSpacerSize.SPACER_SMALL:
        return 0;
      case BlockSpacerSize.SPACER_XSMALL:
        return 1;
      case BlockSpacerSize.SPACER_MEDIUM:
        return 2;
      case BlockSpacerSize.SPACER_LARGE:
        return 3;
      case BlockSpacerSize.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockSpacerShape;
  (function(BlockSpacerShape2) {
    BlockSpacerShape2[BlockSpacerShape2["SPACER_INVISIBLE"] = 0] = "SPACER_INVISIBLE";
    BlockSpacerShape2[BlockSpacerShape2["SPACER_THIN"] = 1] = "SPACER_THIN";
    BlockSpacerShape2[BlockSpacerShape2["SPACER_SQUARE"] = 2] = "SPACER_SQUARE";
    BlockSpacerShape2[BlockSpacerShape2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockSpacerShape || (BlockSpacerShape = {}));
  function blockSpacerShapeFromJSON(object) {
    switch (object) {
      case 0:
      case "SPACER_INVISIBLE":
        return BlockSpacerShape.SPACER_INVISIBLE;
      case 1:
      case "SPACER_THIN":
        return BlockSpacerShape.SPACER_THIN;
      case 2:
      case "SPACER_SQUARE":
        return BlockSpacerShape.SPACER_SQUARE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockSpacerShape.UNRECOGNIZED;
    }
  }
  function blockSpacerShapeToJSON(object) {
    switch (object) {
      case BlockSpacerShape.SPACER_INVISIBLE:
        return 0;
      case BlockSpacerShape.SPACER_THIN:
        return 1;
      case BlockSpacerShape.SPACER_SQUARE:
        return 2;
      case BlockSpacerShape.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockIconSize;
  (function(BlockIconSize2) {
    BlockIconSize2[BlockIconSize2["ICON_SIZE_MEDIUM"] = 0] = "ICON_SIZE_MEDIUM";
    BlockIconSize2[BlockIconSize2["ICON_SIZE_XSMALL"] = 1] = "ICON_SIZE_XSMALL";
    BlockIconSize2[BlockIconSize2["ICON_SIZE_SMALL"] = 2] = "ICON_SIZE_SMALL";
    BlockIconSize2[BlockIconSize2["ICON_SIZE_LARGE"] = 3] = "ICON_SIZE_LARGE";
    BlockIconSize2[BlockIconSize2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockIconSize || (BlockIconSize = {}));
  function blockIconSizeFromJSON(object) {
    switch (object) {
      case 0:
      case "ICON_SIZE_MEDIUM":
        return BlockIconSize.ICON_SIZE_MEDIUM;
      case 1:
      case "ICON_SIZE_XSMALL":
        return BlockIconSize.ICON_SIZE_XSMALL;
      case 2:
      case "ICON_SIZE_SMALL":
        return BlockIconSize.ICON_SIZE_SMALL;
      case 3:
      case "ICON_SIZE_LARGE":
        return BlockIconSize.ICON_SIZE_LARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockIconSize.UNRECOGNIZED;
    }
  }
  function blockIconSizeToJSON(object) {
    switch (object) {
      case BlockIconSize.ICON_SIZE_MEDIUM:
        return 0;
      case BlockIconSize.ICON_SIZE_XSMALL:
        return 1;
      case BlockIconSize.ICON_SIZE_SMALL:
        return 2;
      case BlockIconSize.ICON_SIZE_LARGE:
        return 3;
      case BlockIconSize.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockAvatarFacing;
  (function(BlockAvatarFacing2) {
    BlockAvatarFacing2[BlockAvatarFacing2["AVATAR_FACING_LEFT"] = 0] = "AVATAR_FACING_LEFT";
    BlockAvatarFacing2[BlockAvatarFacing2["AVATAR_FACING_RIGHT"] = 1] = "AVATAR_FACING_RIGHT";
    BlockAvatarFacing2[BlockAvatarFacing2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockAvatarFacing || (BlockAvatarFacing = {}));
  function blockAvatarFacingFromJSON(object) {
    switch (object) {
      case 0:
      case "AVATAR_FACING_LEFT":
        return BlockAvatarFacing.AVATAR_FACING_LEFT;
      case 1:
      case "AVATAR_FACING_RIGHT":
        return BlockAvatarFacing.AVATAR_FACING_RIGHT;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockAvatarFacing.UNRECOGNIZED;
    }
  }
  function blockAvatarFacingToJSON(object) {
    switch (object) {
      case BlockAvatarFacing.AVATAR_FACING_LEFT:
        return 0;
      case BlockAvatarFacing.AVATAR_FACING_RIGHT:
        return 1;
      case BlockAvatarFacing.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockAvatarSize;
  (function(BlockAvatarSize2) {
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_SMALL"] = 0] = "AVATAR_SIZE_SMALL";
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_XXSMALL"] = 1] = "AVATAR_SIZE_XXSMALL";
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_XSMALL"] = 2] = "AVATAR_SIZE_XSMALL";
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_MEDIUM"] = 3] = "AVATAR_SIZE_MEDIUM";
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_LARGE"] = 4] = "AVATAR_SIZE_LARGE";
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_XLARGE"] = 5] = "AVATAR_SIZE_XLARGE";
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_XXLARGE"] = 6] = "AVATAR_SIZE_XXLARGE";
    BlockAvatarSize2[BlockAvatarSize2["AVATAR_SIZE_XXXLARGE"] = 7] = "AVATAR_SIZE_XXXLARGE";
    BlockAvatarSize2[BlockAvatarSize2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockAvatarSize || (BlockAvatarSize = {}));
  function blockAvatarSizeFromJSON(object) {
    switch (object) {
      case 0:
      case "AVATAR_SIZE_SMALL":
        return BlockAvatarSize.AVATAR_SIZE_SMALL;
      case 1:
      case "AVATAR_SIZE_XXSMALL":
        return BlockAvatarSize.AVATAR_SIZE_XXSMALL;
      case 2:
      case "AVATAR_SIZE_XSMALL":
        return BlockAvatarSize.AVATAR_SIZE_XSMALL;
      case 3:
      case "AVATAR_SIZE_MEDIUM":
        return BlockAvatarSize.AVATAR_SIZE_MEDIUM;
      case 4:
      case "AVATAR_SIZE_LARGE":
        return BlockAvatarSize.AVATAR_SIZE_LARGE;
      case 5:
      case "AVATAR_SIZE_XLARGE":
        return BlockAvatarSize.AVATAR_SIZE_XLARGE;
      case 6:
      case "AVATAR_SIZE_XXLARGE":
        return BlockAvatarSize.AVATAR_SIZE_XXLARGE;
      case 7:
      case "AVATAR_SIZE_XXXLARGE":
        return BlockAvatarSize.AVATAR_SIZE_XXXLARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockAvatarSize.UNRECOGNIZED;
    }
  }
  function blockAvatarSizeToJSON(object) {
    switch (object) {
      case BlockAvatarSize.AVATAR_SIZE_SMALL:
        return 0;
      case BlockAvatarSize.AVATAR_SIZE_XXSMALL:
        return 1;
      case BlockAvatarSize.AVATAR_SIZE_XSMALL:
        return 2;
      case BlockAvatarSize.AVATAR_SIZE_MEDIUM:
        return 3;
      case BlockAvatarSize.AVATAR_SIZE_LARGE:
        return 4;
      case BlockAvatarSize.AVATAR_SIZE_XLARGE:
        return 5;
      case BlockAvatarSize.AVATAR_SIZE_XXLARGE:
        return 6;
      case BlockAvatarSize.AVATAR_SIZE_XXXLARGE:
        return 7;
      case BlockAvatarSize.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockAvatarBackground;
  (function(BlockAvatarBackground2) {
    BlockAvatarBackground2[BlockAvatarBackground2["AVATAR_BG_LIGHT"] = 0] = "AVATAR_BG_LIGHT";
    BlockAvatarBackground2[BlockAvatarBackground2["AVATAR_BG_DARK"] = 1] = "AVATAR_BG_DARK";
    BlockAvatarBackground2[BlockAvatarBackground2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockAvatarBackground || (BlockAvatarBackground = {}));
  function blockAvatarBackgroundFromJSON(object) {
    switch (object) {
      case 0:
      case "AVATAR_BG_LIGHT":
        return BlockAvatarBackground.AVATAR_BG_LIGHT;
      case 1:
      case "AVATAR_BG_DARK":
        return BlockAvatarBackground.AVATAR_BG_DARK;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockAvatarBackground.UNRECOGNIZED;
    }
  }
  function blockAvatarBackgroundToJSON(object) {
    switch (object) {
      case BlockAvatarBackground.AVATAR_BG_LIGHT:
        return 0;
      case BlockAvatarBackground.AVATAR_BG_DARK:
        return 1;
      case BlockAvatarBackground.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockFullSnooSize;
  (function(BlockFullSnooSize2) {
    BlockFullSnooSize2[BlockFullSnooSize2["FULLSNOO_SMALL"] = 0] = "FULLSNOO_SMALL";
    BlockFullSnooSize2[BlockFullSnooSize2["FULLSNOO_XSMALL"] = 1] = "FULLSNOO_XSMALL";
    BlockFullSnooSize2[BlockFullSnooSize2["FULLSNOO_MEDIUM"] = 2] = "FULLSNOO_MEDIUM";
    BlockFullSnooSize2[BlockFullSnooSize2["FULLSNOO_LARGE"] = 3] = "FULLSNOO_LARGE";
    BlockFullSnooSize2[BlockFullSnooSize2["FULLSNOO_XLARGE"] = 4] = "FULLSNOO_XLARGE";
    BlockFullSnooSize2[BlockFullSnooSize2["FULLSNOO_XXLARGE"] = 5] = "FULLSNOO_XXLARGE";
    BlockFullSnooSize2[BlockFullSnooSize2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockFullSnooSize || (BlockFullSnooSize = {}));
  function blockFullSnooSizeFromJSON(object) {
    switch (object) {
      case 0:
      case "FULLSNOO_SMALL":
        return BlockFullSnooSize.FULLSNOO_SMALL;
      case 1:
      case "FULLSNOO_XSMALL":
        return BlockFullSnooSize.FULLSNOO_XSMALL;
      case 2:
      case "FULLSNOO_MEDIUM":
        return BlockFullSnooSize.FULLSNOO_MEDIUM;
      case 3:
      case "FULLSNOO_LARGE":
        return BlockFullSnooSize.FULLSNOO_LARGE;
      case 4:
      case "FULLSNOO_XLARGE":
        return BlockFullSnooSize.FULLSNOO_XLARGE;
      case 5:
      case "FULLSNOO_XXLARGE":
        return BlockFullSnooSize.FULLSNOO_XXLARGE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockFullSnooSize.UNRECOGNIZED;
    }
  }
  function blockFullSnooSizeToJSON(object) {
    switch (object) {
      case BlockFullSnooSize.FULLSNOO_SMALL:
        return 0;
      case BlockFullSnooSize.FULLSNOO_XSMALL:
        return 1;
      case BlockFullSnooSize.FULLSNOO_MEDIUM:
        return 2;
      case BlockFullSnooSize.FULLSNOO_LARGE:
        return 3;
      case BlockFullSnooSize.FULLSNOO_XLARGE:
        return 4;
      case BlockFullSnooSize.FULLSNOO_XXLARGE:
        return 5;
      case BlockFullSnooSize.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockAnimationType;
  (function(BlockAnimationType2) {
    BlockAnimationType2[BlockAnimationType2["ANIM_LOTTIE"] = 0] = "ANIM_LOTTIE";
    BlockAnimationType2[BlockAnimationType2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockAnimationType || (BlockAnimationType = {}));
  function blockAnimationTypeFromJSON(object) {
    switch (object) {
      case 0:
      case "ANIM_LOTTIE":
        return BlockAnimationType.ANIM_LOTTIE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockAnimationType.UNRECOGNIZED;
    }
  }
  function blockAnimationTypeToJSON(object) {
    switch (object) {
      case BlockAnimationType.ANIM_LOTTIE:
        return 0;
      case BlockAnimationType.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockAnimationLoopMode;
  (function(BlockAnimationLoopMode2) {
    BlockAnimationLoopMode2[BlockAnimationLoopMode2["ANIM_LOOP_REPEAT"] = 0] = "ANIM_LOOP_REPEAT";
    BlockAnimationLoopMode2[BlockAnimationLoopMode2["ANIM_LOOP_BOUNCE"] = 1] = "ANIM_LOOP_BOUNCE";
    BlockAnimationLoopMode2[BlockAnimationLoopMode2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockAnimationLoopMode || (BlockAnimationLoopMode = {}));
  function blockAnimationLoopModeFromJSON(object) {
    switch (object) {
      case 0:
      case "ANIM_LOOP_REPEAT":
        return BlockAnimationLoopMode.ANIM_LOOP_REPEAT;
      case 1:
      case "ANIM_LOOP_BOUNCE":
        return BlockAnimationLoopMode.ANIM_LOOP_BOUNCE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockAnimationLoopMode.UNRECOGNIZED;
    }
  }
  function blockAnimationLoopModeToJSON(object) {
    switch (object) {
      case BlockAnimationLoopMode.ANIM_LOOP_REPEAT:
        return 0;
      case BlockAnimationLoopMode.ANIM_LOOP_BOUNCE:
        return 1;
      case BlockAnimationLoopMode.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockAnimationDirection;
  (function(BlockAnimationDirection2) {
    BlockAnimationDirection2[BlockAnimationDirection2["ANIM_DIR_FORWARD"] = 0] = "ANIM_DIR_FORWARD";
    BlockAnimationDirection2[BlockAnimationDirection2["ANIM_DIR_BACKWARD"] = 1] = "ANIM_DIR_BACKWARD";
    BlockAnimationDirection2[BlockAnimationDirection2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockAnimationDirection || (BlockAnimationDirection = {}));
  function blockAnimationDirectionFromJSON(object) {
    switch (object) {
      case 0:
      case "ANIM_DIR_FORWARD":
        return BlockAnimationDirection.ANIM_DIR_FORWARD;
      case 1:
      case "ANIM_DIR_BACKWARD":
        return BlockAnimationDirection.ANIM_DIR_BACKWARD;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockAnimationDirection.UNRECOGNIZED;
    }
  }
  function blockAnimationDirectionToJSON(object) {
    switch (object) {
      case BlockAnimationDirection.ANIM_DIR_FORWARD:
        return 0;
      case BlockAnimationDirection.ANIM_DIR_BACKWARD:
        return 1;
      case BlockAnimationDirection.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockSizeUnit;
  (function(BlockSizeUnit2) {
    BlockSizeUnit2[BlockSizeUnit2["SIZE_UNIT_PERCENT"] = 0] = "SIZE_UNIT_PERCENT";
    BlockSizeUnit2[BlockSizeUnit2["SIZE_UNIT_PIXELS"] = 1] = "SIZE_UNIT_PIXELS";
    BlockSizeUnit2[BlockSizeUnit2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockSizeUnit || (BlockSizeUnit = {}));
  function blockSizeUnitFromJSON(object) {
    switch (object) {
      case 0:
      case "SIZE_UNIT_PERCENT":
        return BlockSizeUnit.SIZE_UNIT_PERCENT;
      case 1:
      case "SIZE_UNIT_PIXELS":
        return BlockSizeUnit.SIZE_UNIT_PIXELS;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockSizeUnit.UNRECOGNIZED;
    }
  }
  function blockSizeUnitToJSON(object) {
    switch (object) {
      case BlockSizeUnit.SIZE_UNIT_PERCENT:
        return 0;
      case BlockSizeUnit.SIZE_UNIT_PIXELS:
        return 1;
      case BlockSizeUnit.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  var BlockTextOverflow;
  (function(BlockTextOverflow2) {
    BlockTextOverflow2[BlockTextOverflow2["TEXT_OVERFLOW_CLIP"] = 0] = "TEXT_OVERFLOW_CLIP";
    BlockTextOverflow2[BlockTextOverflow2["TEXT_OVERFLOW_ELLIPSE"] = 1] = "TEXT_OVERFLOW_ELLIPSE";
    BlockTextOverflow2[BlockTextOverflow2["UNRECOGNIZED"] = -1] = "UNRECOGNIZED";
  })(BlockTextOverflow || (BlockTextOverflow = {}));
  function blockTextOverflowFromJSON(object) {
    switch (object) {
      case 0:
      case "TEXT_OVERFLOW_CLIP":
        return BlockTextOverflow.TEXT_OVERFLOW_CLIP;
      case 1:
      case "TEXT_OVERFLOW_ELLIPSE":
        return BlockTextOverflow.TEXT_OVERFLOW_ELLIPSE;
      case -1:
      case "UNRECOGNIZED":
      default:
        return BlockTextOverflow.UNRECOGNIZED;
    }
  }
  function blockTextOverflowToJSON(object) {
    switch (object) {
      case BlockTextOverflow.TEXT_OVERFLOW_CLIP:
        return 0;
      case BlockTextOverflow.TEXT_OVERFLOW_ELLIPSE:
        return 1;
      case BlockTextOverflow.UNRECOGNIZED:
      default:
        return -1;
    }
  }
  function createBaseBlockAlignment() {
    return { vertical: void 0, horizontal: void 0 };
  }
  const BlockAlignment = {
    $type: "devvit.ui.block_kit.v1beta.BlockAlignment",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.vertical !== void 0) {
        writer2.uint32(8).int32(message.vertical);
      }
      if (message.horizontal !== void 0) {
        writer2.uint32(16).int32(message.horizontal);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockAlignment();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.vertical = reader2.int32();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.horizontal = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        vertical: isSet$2(object.vertical) ? blockVerticalAlignmentFromJSON(object.vertical) : void 0,
        horizontal: isSet$2(object.horizontal) ? blockHorizontalAlignmentFromJSON(object.horizontal) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.vertical !== void 0) {
        obj.vertical = blockVerticalAlignmentToJSON(message.vertical);
      }
      if (message.horizontal !== void 0) {
        obj.horizontal = blockHorizontalAlignmentToJSON(message.horizontal);
      }
      return obj;
    },
    create(base) {
      return BlockAlignment.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockAlignment();
      message.vertical = object.vertical ?? void 0;
      message.horizontal = object.horizontal ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockAlignment.$type, BlockAlignment);
  function createBaseBlockBorder() {
    return { color: void 0, width: void 0, colors: void 0 };
  }
  const BlockBorder = {
    $type: "devvit.ui.block_kit.v1beta.BlockBorder",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.color !== void 0) {
        writer2.uint32(10).string(message.color);
      }
      if (message.width !== void 0) {
        writer2.uint32(16).int32(message.width);
      }
      if (message.colors !== void 0) {
        BlockColor.encode(message.colors, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockBorder();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.color = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.width = reader2.int32();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.colors = BlockColor.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        color: isSet$2(object.color) ? globalThis.String(object.color) : void 0,
        width: isSet$2(object.width) ? blockBorderWidthFromJSON(object.width) : void 0,
        colors: isSet$2(object.colors) ? BlockColor.fromJSON(object.colors) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.color !== void 0) {
        obj.color = message.color;
      }
      if (message.width !== void 0) {
        obj.width = blockBorderWidthToJSON(message.width);
      }
      if (message.colors !== void 0) {
        obj.colors = BlockColor.toJSON(message.colors);
      }
      return obj;
    },
    create(base) {
      return BlockBorder.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockBorder();
      message.color = object.color ?? void 0;
      message.width = object.width ?? void 0;
      message.colors = object.colors !== void 0 && object.colors !== null ? BlockColor.fromPartial(object.colors) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockBorder.$type, BlockBorder);
  function createBaseBlockSizes() {
    return { grow: void 0, width: void 0, height: void 0 };
  }
  const BlockSizes = {
    $type: "devvit.ui.block_kit.v1beta.BlockSizes",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.grow !== void 0) {
        writer2.uint32(8).bool(message.grow);
      }
      if (message.width !== void 0) {
        BlockSizes_Dimension.encode(message.width, writer2.uint32(18).fork()).ldelim();
      }
      if (message.height !== void 0) {
        BlockSizes_Dimension.encode(message.height, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockSizes();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.grow = reader2.bool();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.width = BlockSizes_Dimension.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.height = BlockSizes_Dimension.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        grow: isSet$2(object.grow) ? globalThis.Boolean(object.grow) : void 0,
        width: isSet$2(object.width) ? BlockSizes_Dimension.fromJSON(object.width) : void 0,
        height: isSet$2(object.height) ? BlockSizes_Dimension.fromJSON(object.height) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.grow !== void 0) {
        obj.grow = message.grow;
      }
      if (message.width !== void 0) {
        obj.width = BlockSizes_Dimension.toJSON(message.width);
      }
      if (message.height !== void 0) {
        obj.height = BlockSizes_Dimension.toJSON(message.height);
      }
      return obj;
    },
    create(base) {
      return BlockSizes.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockSizes();
      message.grow = object.grow ?? void 0;
      message.width = object.width !== void 0 && object.width !== null ? BlockSizes_Dimension.fromPartial(object.width) : void 0;
      message.height = object.height !== void 0 && object.height !== null ? BlockSizes_Dimension.fromPartial(object.height) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockSizes.$type, BlockSizes);
  function createBaseBlockSizes_Dimension() {
    return { value: void 0, min: void 0, max: void 0 };
  }
  const BlockSizes_Dimension = {
    $type: "devvit.ui.block_kit.v1beta.BlockSizes.Dimension",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== void 0) {
        BlockSizes_Dimension_Value.encode(message.value, writer2.uint32(10).fork()).ldelim();
      }
      if (message.min !== void 0) {
        BlockSizes_Dimension_Value.encode(message.min, writer2.uint32(18).fork()).ldelim();
      }
      if (message.max !== void 0) {
        BlockSizes_Dimension_Value.encode(message.max, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockSizes_Dimension();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.value = BlockSizes_Dimension_Value.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.min = BlockSizes_Dimension_Value.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.max = BlockSizes_Dimension_Value.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        value: isSet$2(object.value) ? BlockSizes_Dimension_Value.fromJSON(object.value) : void 0,
        min: isSet$2(object.min) ? BlockSizes_Dimension_Value.fromJSON(object.min) : void 0,
        max: isSet$2(object.max) ? BlockSizes_Dimension_Value.fromJSON(object.max) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== void 0) {
        obj.value = BlockSizes_Dimension_Value.toJSON(message.value);
      }
      if (message.min !== void 0) {
        obj.min = BlockSizes_Dimension_Value.toJSON(message.min);
      }
      if (message.max !== void 0) {
        obj.max = BlockSizes_Dimension_Value.toJSON(message.max);
      }
      return obj;
    },
    create(base) {
      return BlockSizes_Dimension.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockSizes_Dimension();
      message.value = object.value !== void 0 && object.value !== null ? BlockSizes_Dimension_Value.fromPartial(object.value) : void 0;
      message.min = object.min !== void 0 && object.min !== null ? BlockSizes_Dimension_Value.fromPartial(object.min) : void 0;
      message.max = object.max !== void 0 && object.max !== null ? BlockSizes_Dimension_Value.fromPartial(object.max) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockSizes_Dimension.$type, BlockSizes_Dimension);
  function createBaseBlockSizes_Dimension_Value() {
    return { value: 0, unit: 0 };
  }
  const BlockSizes_Dimension_Value = {
    $type: "devvit.ui.block_kit.v1beta.BlockSizes.Dimension.Value",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.value !== 0) {
        writer2.uint32(13).float(message.value);
      }
      if (message.unit !== 0) {
        writer2.uint32(16).int32(message.unit);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockSizes_Dimension_Value();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 13) {
              break;
            }
            message.value = reader2.float();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.unit = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        value: isSet$2(object.value) ? globalThis.Number(object.value) : 0,
        unit: isSet$2(object.unit) ? blockSizeUnitFromJSON(object.unit) : 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.value !== 0) {
        obj.value = message.value;
      }
      if (message.unit !== 0) {
        obj.unit = blockSizeUnitToJSON(message.unit);
      }
      return obj;
    },
    create(base) {
      return BlockSizes_Dimension_Value.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockSizes_Dimension_Value();
      message.value = object.value ?? 0;
      message.unit = object.unit ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockSizes_Dimension_Value.$type, BlockSizes_Dimension_Value);
  function createBaseBlockSize() {
    return { grow: void 0, width: void 0, widthUnit: void 0, height: void 0, heightUnit: void 0 };
  }
  const BlockSize = {
    $type: "devvit.ui.block_kit.v1beta.BlockSize",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.grow !== void 0) {
        writer2.uint32(8).bool(message.grow);
      }
      if (message.width !== void 0) {
        writer2.uint32(21).float(message.width);
      }
      if (message.widthUnit !== void 0) {
        writer2.uint32(32).int32(message.widthUnit);
      }
      if (message.height !== void 0) {
        writer2.uint32(29).float(message.height);
      }
      if (message.heightUnit !== void 0) {
        writer2.uint32(40).int32(message.heightUnit);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockSize();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.grow = reader2.bool();
            continue;
          case 2:
            if (tag !== 21) {
              break;
            }
            message.width = reader2.float();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.widthUnit = reader2.int32();
            continue;
          case 3:
            if (tag !== 29) {
              break;
            }
            message.height = reader2.float();
            continue;
          case 5:
            if (tag !== 40) {
              break;
            }
            message.heightUnit = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        grow: isSet$2(object.grow) ? globalThis.Boolean(object.grow) : void 0,
        width: isSet$2(object.width) ? globalThis.Number(object.width) : void 0,
        widthUnit: isSet$2(object.widthUnit) ? blockSizeUnitFromJSON(object.widthUnit) : void 0,
        height: isSet$2(object.height) ? globalThis.Number(object.height) : void 0,
        heightUnit: isSet$2(object.heightUnit) ? blockSizeUnitFromJSON(object.heightUnit) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.grow !== void 0) {
        obj.grow = message.grow;
      }
      if (message.width !== void 0) {
        obj.width = message.width;
      }
      if (message.widthUnit !== void 0) {
        obj.widthUnit = blockSizeUnitToJSON(message.widthUnit);
      }
      if (message.height !== void 0) {
        obj.height = message.height;
      }
      if (message.heightUnit !== void 0) {
        obj.heightUnit = blockSizeUnitToJSON(message.heightUnit);
      }
      return obj;
    },
    create(base) {
      return BlockSize.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockSize();
      message.grow = object.grow ?? void 0;
      message.width = object.width ?? void 0;
      message.widthUnit = object.widthUnit ?? void 0;
      message.height = object.height ?? void 0;
      message.heightUnit = object.heightUnit ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockSize.$type, BlockSize);
  function createBaseBlockAction() {
    return { type: 0, id: "", data: void 0 };
  }
  const BlockAction = {
    $type: "devvit.ui.block_kit.v1beta.BlockAction",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.type !== 0) {
        writer2.uint32(8).int32(message.type);
      }
      if (message.id !== "") {
        writer2.uint32(18).string(message.id);
      }
      if (message.data !== void 0) {
        Struct.encode(Struct.wrap(message.data), writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockAction();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.type = reader2.int32();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.id = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.data = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        type: isSet$2(object.type) ? blockActionTypeFromJSON(object.type) : 0,
        id: isSet$2(object.id) ? globalThis.String(object.id) : "",
        data: isObject$2(object.data) ? object.data : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.type !== 0) {
        obj.type = blockActionTypeToJSON(message.type);
      }
      if (message.id !== "") {
        obj.id = message.id;
      }
      if (message.data !== void 0) {
        obj.data = message.data;
      }
      return obj;
    },
    create(base) {
      return BlockAction.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockAction();
      message.type = object.type ?? 0;
      message.id = object.id ?? "";
      message.data = object.data ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockAction.$type, BlockAction);
  function createBaseBlockColor() {
    return { light: void 0, dark: void 0 };
  }
  const BlockColor = {
    $type: "devvit.ui.block_kit.v1beta.BlockColor",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.light !== void 0) {
        writer2.uint32(10).string(message.light);
      }
      if (message.dark !== void 0) {
        writer2.uint32(18).string(message.dark);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockColor();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.light = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.dark = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        light: isSet$2(object.light) ? globalThis.String(object.light) : void 0,
        dark: isSet$2(object.dark) ? globalThis.String(object.dark) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.light !== void 0) {
        obj.light = message.light;
      }
      if (message.dark !== void 0) {
        obj.dark = message.dark;
      }
      return obj;
    },
    create(base) {
      return BlockColor.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockColor();
      message.light = object.light ?? void 0;
      message.dark = object.dark ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockColor.$type, BlockColor);
  function isObject$2(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$2(value) {
    return value !== null && value !== void 0;
  }
  function createBaseBlock() {
    return { type: 0, size: void 0, sizes: void 0, config: void 0, actions: [], id: void 0, key: void 0 };
  }
  const Block = {
    $type: "devvit.ui.block_kit.v1beta.Block",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.type !== 0) {
        writer2.uint32(8).int32(message.type);
      }
      if (message.size !== void 0) {
        BlockSize.encode(message.size, writer2.uint32(18).fork()).ldelim();
      }
      if (message.sizes !== void 0) {
        BlockSizes.encode(message.sizes, writer2.uint32(42).fork()).ldelim();
      }
      if (message.config !== void 0) {
        BlockConfig.encode(message.config, writer2.uint32(26).fork()).ldelim();
      }
      for (const v of message.actions) {
        BlockAction.encode(v, writer2.uint32(34).fork()).ldelim();
      }
      if (message.id !== void 0) {
        writer2.uint32(50).string(message.id);
      }
      if (message.key !== void 0) {
        writer2.uint32(58).string(message.key);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlock();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.type = reader2.int32();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.size = BlockSize.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.sizes = BlockSizes.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.config = BlockConfig.decode(reader2, reader2.uint32());
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.actions.push(BlockAction.decode(reader2, reader2.uint32()));
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.id = reader2.string();
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.key = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        type: isSet$1(object.type) ? blockTypeFromJSON(object.type) : 0,
        size: isSet$1(object.size) ? BlockSize.fromJSON(object.size) : void 0,
        sizes: isSet$1(object.sizes) ? BlockSizes.fromJSON(object.sizes) : void 0,
        config: isSet$1(object.config) ? BlockConfig.fromJSON(object.config) : void 0,
        actions: globalThis.Array.isArray(object?.actions) ? object.actions.map((e) => BlockAction.fromJSON(e)) : [],
        id: isSet$1(object.id) ? globalThis.String(object.id) : void 0,
        key: isSet$1(object.key) ? globalThis.String(object.key) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.type !== 0) {
        obj.type = blockTypeToJSON(message.type);
      }
      if (message.size !== void 0) {
        obj.size = BlockSize.toJSON(message.size);
      }
      if (message.sizes !== void 0) {
        obj.sizes = BlockSizes.toJSON(message.sizes);
      }
      if (message.config !== void 0) {
        obj.config = BlockConfig.toJSON(message.config);
      }
      if (message.actions?.length) {
        obj.actions = message.actions.map((e) => BlockAction.toJSON(e));
      }
      if (message.id !== void 0) {
        obj.id = message.id;
      }
      if (message.key !== void 0) {
        obj.key = message.key;
      }
      return obj;
    },
    create(base) {
      return Block.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlock();
      message.type = object.type ?? 0;
      message.size = object.size !== void 0 && object.size !== null ? BlockSize.fromPartial(object.size) : void 0;
      message.sizes = object.sizes !== void 0 && object.sizes !== null ? BlockSizes.fromPartial(object.sizes) : void 0;
      message.config = object.config !== void 0 && object.config !== null ? BlockConfig.fromPartial(object.config) : void 0;
      message.actions = object.actions?.map((e) => BlockAction.fromPartial(e)) || [];
      message.id = object.id ?? void 0;
      message.key = object.key ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(Block.$type, Block);
  function createBaseBlockConfig() {
    return {
      rootConfig: void 0,
      stackConfig: void 0,
      textConfig: void 0,
      buttonConfig: void 0,
      imageConfig: void 0,
      spacerConfig: void 0,
      iconConfig: void 0,
      avatarConfig: void 0,
      fullsnooConfig: void 0,
      animationConfig: void 0,
      webviewConfig: void 0
    };
  }
  const BlockConfig = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.rootConfig !== void 0) {
        BlockConfig_Root.encode(message.rootConfig, writer2.uint32(10).fork()).ldelim();
      }
      if (message.stackConfig !== void 0) {
        BlockConfig_Stack.encode(message.stackConfig, writer2.uint32(18).fork()).ldelim();
      }
      if (message.textConfig !== void 0) {
        BlockConfig_Text.encode(message.textConfig, writer2.uint32(26).fork()).ldelim();
      }
      if (message.buttonConfig !== void 0) {
        BlockConfig_Button.encode(message.buttonConfig, writer2.uint32(34).fork()).ldelim();
      }
      if (message.imageConfig !== void 0) {
        BlockConfig_Image.encode(message.imageConfig, writer2.uint32(42).fork()).ldelim();
      }
      if (message.spacerConfig !== void 0) {
        BlockConfig_Spacer.encode(message.spacerConfig, writer2.uint32(50).fork()).ldelim();
      }
      if (message.iconConfig !== void 0) {
        BlockConfig_Icon.encode(message.iconConfig, writer2.uint32(58).fork()).ldelim();
      }
      if (message.avatarConfig !== void 0) {
        BlockConfig_Avatar.encode(message.avatarConfig, writer2.uint32(66).fork()).ldelim();
      }
      if (message.fullsnooConfig !== void 0) {
        BlockConfig_FullSnoo.encode(message.fullsnooConfig, writer2.uint32(74).fork()).ldelim();
      }
      if (message.animationConfig !== void 0) {
        BlockConfig_Animation.encode(message.animationConfig, writer2.uint32(82).fork()).ldelim();
      }
      if (message.webviewConfig !== void 0) {
        BlockConfig_WebView.encode(message.webviewConfig, writer2.uint32(802).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.rootConfig = BlockConfig_Root.decode(reader2, reader2.uint32());
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.stackConfig = BlockConfig_Stack.decode(reader2, reader2.uint32());
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.textConfig = BlockConfig_Text.decode(reader2, reader2.uint32());
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.buttonConfig = BlockConfig_Button.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.imageConfig = BlockConfig_Image.decode(reader2, reader2.uint32());
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.spacerConfig = BlockConfig_Spacer.decode(reader2, reader2.uint32());
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.iconConfig = BlockConfig_Icon.decode(reader2, reader2.uint32());
            continue;
          case 8:
            if (tag !== 66) {
              break;
            }
            message.avatarConfig = BlockConfig_Avatar.decode(reader2, reader2.uint32());
            continue;
          case 9:
            if (tag !== 74) {
              break;
            }
            message.fullsnooConfig = BlockConfig_FullSnoo.decode(reader2, reader2.uint32());
            continue;
          case 10:
            if (tag !== 82) {
              break;
            }
            message.animationConfig = BlockConfig_Animation.decode(reader2, reader2.uint32());
            continue;
          case 100:
            if (tag !== 802) {
              break;
            }
            message.webviewConfig = BlockConfig_WebView.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        rootConfig: isSet$1(object.rootConfig) ? BlockConfig_Root.fromJSON(object.rootConfig) : void 0,
        stackConfig: isSet$1(object.stackConfig) ? BlockConfig_Stack.fromJSON(object.stackConfig) : void 0,
        textConfig: isSet$1(object.textConfig) ? BlockConfig_Text.fromJSON(object.textConfig) : void 0,
        buttonConfig: isSet$1(object.buttonConfig) ? BlockConfig_Button.fromJSON(object.buttonConfig) : void 0,
        imageConfig: isSet$1(object.imageConfig) ? BlockConfig_Image.fromJSON(object.imageConfig) : void 0,
        spacerConfig: isSet$1(object.spacerConfig) ? BlockConfig_Spacer.fromJSON(object.spacerConfig) : void 0,
        iconConfig: isSet$1(object.iconConfig) ? BlockConfig_Icon.fromJSON(object.iconConfig) : void 0,
        avatarConfig: isSet$1(object.avatarConfig) ? BlockConfig_Avatar.fromJSON(object.avatarConfig) : void 0,
        fullsnooConfig: isSet$1(object.fullsnooConfig) ? BlockConfig_FullSnoo.fromJSON(object.fullsnooConfig) : void 0,
        animationConfig: isSet$1(object.animationConfig) ? BlockConfig_Animation.fromJSON(object.animationConfig) : void 0,
        webviewConfig: isSet$1(object.webviewConfig) ? BlockConfig_WebView.fromJSON(object.webviewConfig) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.rootConfig !== void 0) {
        obj.rootConfig = BlockConfig_Root.toJSON(message.rootConfig);
      }
      if (message.stackConfig !== void 0) {
        obj.stackConfig = BlockConfig_Stack.toJSON(message.stackConfig);
      }
      if (message.textConfig !== void 0) {
        obj.textConfig = BlockConfig_Text.toJSON(message.textConfig);
      }
      if (message.buttonConfig !== void 0) {
        obj.buttonConfig = BlockConfig_Button.toJSON(message.buttonConfig);
      }
      if (message.imageConfig !== void 0) {
        obj.imageConfig = BlockConfig_Image.toJSON(message.imageConfig);
      }
      if (message.spacerConfig !== void 0) {
        obj.spacerConfig = BlockConfig_Spacer.toJSON(message.spacerConfig);
      }
      if (message.iconConfig !== void 0) {
        obj.iconConfig = BlockConfig_Icon.toJSON(message.iconConfig);
      }
      if (message.avatarConfig !== void 0) {
        obj.avatarConfig = BlockConfig_Avatar.toJSON(message.avatarConfig);
      }
      if (message.fullsnooConfig !== void 0) {
        obj.fullsnooConfig = BlockConfig_FullSnoo.toJSON(message.fullsnooConfig);
      }
      if (message.animationConfig !== void 0) {
        obj.animationConfig = BlockConfig_Animation.toJSON(message.animationConfig);
      }
      if (message.webviewConfig !== void 0) {
        obj.webviewConfig = BlockConfig_WebView.toJSON(message.webviewConfig);
      }
      return obj;
    },
    create(base) {
      return BlockConfig.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig();
      message.rootConfig = object.rootConfig !== void 0 && object.rootConfig !== null ? BlockConfig_Root.fromPartial(object.rootConfig) : void 0;
      message.stackConfig = object.stackConfig !== void 0 && object.stackConfig !== null ? BlockConfig_Stack.fromPartial(object.stackConfig) : void 0;
      message.textConfig = object.textConfig !== void 0 && object.textConfig !== null ? BlockConfig_Text.fromPartial(object.textConfig) : void 0;
      message.buttonConfig = object.buttonConfig !== void 0 && object.buttonConfig !== null ? BlockConfig_Button.fromPartial(object.buttonConfig) : void 0;
      message.imageConfig = object.imageConfig !== void 0 && object.imageConfig !== null ? BlockConfig_Image.fromPartial(object.imageConfig) : void 0;
      message.spacerConfig = object.spacerConfig !== void 0 && object.spacerConfig !== null ? BlockConfig_Spacer.fromPartial(object.spacerConfig) : void 0;
      message.iconConfig = object.iconConfig !== void 0 && object.iconConfig !== null ? BlockConfig_Icon.fromPartial(object.iconConfig) : void 0;
      message.avatarConfig = object.avatarConfig !== void 0 && object.avatarConfig !== null ? BlockConfig_Avatar.fromPartial(object.avatarConfig) : void 0;
      message.fullsnooConfig = object.fullsnooConfig !== void 0 && object.fullsnooConfig !== null ? BlockConfig_FullSnoo.fromPartial(object.fullsnooConfig) : void 0;
      message.animationConfig = object.animationConfig !== void 0 && object.animationConfig !== null ? BlockConfig_Animation.fromPartial(object.animationConfig) : void 0;
      message.webviewConfig = object.webviewConfig !== void 0 && object.webviewConfig !== null ? BlockConfig_WebView.fromPartial(object.webviewConfig) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig.$type, BlockConfig);
  function createBaseBlockConfig_Root() {
    return { children: [], height: 0 };
  }
  const BlockConfig_Root = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Root",
    encode(message, writer2 = _m0.Writer.create()) {
      for (const v of message.children) {
        Block.encode(v, writer2.uint32(10).fork()).ldelim();
      }
      if (message.height !== 0) {
        writer2.uint32(16).int32(message.height);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Root();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.children.push(Block.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.height = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        children: globalThis.Array.isArray(object?.children) ? object.children.map((e) => Block.fromJSON(e)) : [],
        height: isSet$1(object.height) ? globalThis.Number(object.height) : 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.children?.length) {
        obj.children = message.children.map((e) => Block.toJSON(e));
      }
      if (message.height !== 0) {
        obj.height = Math.round(message.height);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Root.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Root();
      message.children = object.children?.map((e) => Block.fromPartial(e)) || [];
      message.height = object.height ?? 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Root.$type, BlockConfig_Root);
  function createBaseBlockConfig_Stack() {
    return {
      direction: 0,
      children: [],
      reverse: void 0,
      alignment: void 0,
      padding: void 0,
      gap: void 0,
      border: void 0,
      cornerRadius: void 0,
      backgroundColor: void 0,
      backgroundColors: void 0
    };
  }
  const BlockConfig_Stack = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Stack",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.direction !== 0) {
        writer2.uint32(8).int32(message.direction);
      }
      for (const v of message.children) {
        Block.encode(v, writer2.uint32(18).fork()).ldelim();
      }
      if (message.reverse !== void 0) {
        writer2.uint32(24).bool(message.reverse);
      }
      if (message.alignment !== void 0) {
        BlockAlignment.encode(message.alignment, writer2.uint32(34).fork()).ldelim();
      }
      if (message.padding !== void 0) {
        writer2.uint32(40).int32(message.padding);
      }
      if (message.gap !== void 0) {
        writer2.uint32(48).int32(message.gap);
      }
      if (message.border !== void 0) {
        BlockBorder.encode(message.border, writer2.uint32(58).fork()).ldelim();
      }
      if (message.cornerRadius !== void 0) {
        writer2.uint32(64).int32(message.cornerRadius);
      }
      if (message.backgroundColor !== void 0) {
        writer2.uint32(74).string(message.backgroundColor);
      }
      if (message.backgroundColors !== void 0) {
        BlockColor.encode(message.backgroundColors, writer2.uint32(82).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Stack();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.direction = reader2.int32();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.children.push(Block.decode(reader2, reader2.uint32()));
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.reverse = reader2.bool();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.alignment = BlockAlignment.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 40) {
              break;
            }
            message.padding = reader2.int32();
            continue;
          case 6:
            if (tag !== 48) {
              break;
            }
            message.gap = reader2.int32();
            continue;
          case 7:
            if (tag !== 58) {
              break;
            }
            message.border = BlockBorder.decode(reader2, reader2.uint32());
            continue;
          case 8:
            if (tag !== 64) {
              break;
            }
            message.cornerRadius = reader2.int32();
            continue;
          case 9:
            if (tag !== 74) {
              break;
            }
            message.backgroundColor = reader2.string();
            continue;
          case 10:
            if (tag !== 82) {
              break;
            }
            message.backgroundColors = BlockColor.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        direction: isSet$1(object.direction) ? blockStackDirectionFromJSON(object.direction) : 0,
        children: globalThis.Array.isArray(object?.children) ? object.children.map((e) => Block.fromJSON(e)) : [],
        reverse: isSet$1(object.reverse) ? globalThis.Boolean(object.reverse) : void 0,
        alignment: isSet$1(object.alignment) ? BlockAlignment.fromJSON(object.alignment) : void 0,
        padding: isSet$1(object.padding) ? blockPaddingFromJSON(object.padding) : void 0,
        gap: isSet$1(object.gap) ? blockGapFromJSON(object.gap) : void 0,
        border: isSet$1(object.border) ? BlockBorder.fromJSON(object.border) : void 0,
        cornerRadius: isSet$1(object.cornerRadius) ? blockRadiusFromJSON(object.cornerRadius) : void 0,
        backgroundColor: isSet$1(object.backgroundColor) ? globalThis.String(object.backgroundColor) : void 0,
        backgroundColors: isSet$1(object.backgroundColors) ? BlockColor.fromJSON(object.backgroundColors) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.direction !== 0) {
        obj.direction = blockStackDirectionToJSON(message.direction);
      }
      if (message.children?.length) {
        obj.children = message.children.map((e) => Block.toJSON(e));
      }
      if (message.reverse !== void 0) {
        obj.reverse = message.reverse;
      }
      if (message.alignment !== void 0) {
        obj.alignment = BlockAlignment.toJSON(message.alignment);
      }
      if (message.padding !== void 0) {
        obj.padding = blockPaddingToJSON(message.padding);
      }
      if (message.gap !== void 0) {
        obj.gap = blockGapToJSON(message.gap);
      }
      if (message.border !== void 0) {
        obj.border = BlockBorder.toJSON(message.border);
      }
      if (message.cornerRadius !== void 0) {
        obj.cornerRadius = blockRadiusToJSON(message.cornerRadius);
      }
      if (message.backgroundColor !== void 0) {
        obj.backgroundColor = message.backgroundColor;
      }
      if (message.backgroundColors !== void 0) {
        obj.backgroundColors = BlockColor.toJSON(message.backgroundColors);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Stack.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Stack();
      message.direction = object.direction ?? 0;
      message.children = object.children?.map((e) => Block.fromPartial(e)) || [];
      message.reverse = object.reverse ?? void 0;
      message.alignment = object.alignment !== void 0 && object.alignment !== null ? BlockAlignment.fromPartial(object.alignment) : void 0;
      message.padding = object.padding ?? void 0;
      message.gap = object.gap ?? void 0;
      message.border = object.border !== void 0 && object.border !== null ? BlockBorder.fromPartial(object.border) : void 0;
      message.cornerRadius = object.cornerRadius ?? void 0;
      message.backgroundColor = object.backgroundColor ?? void 0;
      message.backgroundColors = object.backgroundColors !== void 0 && object.backgroundColors !== null ? BlockColor.fromPartial(object.backgroundColors) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Stack.$type, BlockConfig_Stack);
  function createBaseBlockConfig_Text() {
    return {
      text: "",
      size: void 0,
      weight: void 0,
      color: void 0,
      alignment: void 0,
      outline: void 0,
      style: void 0,
      selectable: void 0,
      colors: void 0,
      wrap: void 0,
      overflow: void 0
    };
  }
  const BlockConfig_Text = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Text",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.text !== "") {
        writer2.uint32(10).string(message.text);
      }
      if (message.size !== void 0) {
        writer2.uint32(16).int32(message.size);
      }
      if (message.weight !== void 0) {
        writer2.uint32(24).int32(message.weight);
      }
      if (message.color !== void 0) {
        writer2.uint32(34).string(message.color);
      }
      if (message.alignment !== void 0) {
        BlockAlignment.encode(message.alignment, writer2.uint32(42).fork()).ldelim();
      }
      if (message.outline !== void 0) {
        writer2.uint32(48).int32(message.outline);
      }
      if (message.style !== void 0) {
        writer2.uint32(56).int32(message.style);
      }
      if (message.selectable !== void 0) {
        writer2.uint32(64).bool(message.selectable);
      }
      if (message.colors !== void 0) {
        BlockColor.encode(message.colors, writer2.uint32(74).fork()).ldelim();
      }
      if (message.wrap !== void 0) {
        writer2.uint32(80).bool(message.wrap);
      }
      if (message.overflow !== void 0) {
        writer2.uint32(88).int32(message.overflow);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Text();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.text = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.size = reader2.int32();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.weight = reader2.int32();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.color = reader2.string();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.alignment = BlockAlignment.decode(reader2, reader2.uint32());
            continue;
          case 6:
            if (tag !== 48) {
              break;
            }
            message.outline = reader2.int32();
            continue;
          case 7:
            if (tag !== 56) {
              break;
            }
            message.style = reader2.int32();
            continue;
          case 8:
            if (tag !== 64) {
              break;
            }
            message.selectable = reader2.bool();
            continue;
          case 9:
            if (tag !== 74) {
              break;
            }
            message.colors = BlockColor.decode(reader2, reader2.uint32());
            continue;
          case 10:
            if (tag !== 80) {
              break;
            }
            message.wrap = reader2.bool();
            continue;
          case 11:
            if (tag !== 88) {
              break;
            }
            message.overflow = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        text: isSet$1(object.text) ? globalThis.String(object.text) : "",
        size: isSet$1(object.size) ? blockTextSizeFromJSON(object.size) : void 0,
        weight: isSet$1(object.weight) ? blockTextWeightFromJSON(object.weight) : void 0,
        color: isSet$1(object.color) ? globalThis.String(object.color) : void 0,
        alignment: isSet$1(object.alignment) ? BlockAlignment.fromJSON(object.alignment) : void 0,
        outline: isSet$1(object.outline) ? blockTextOutlineFromJSON(object.outline) : void 0,
        style: isSet$1(object.style) ? blockTextStyleFromJSON(object.style) : void 0,
        selectable: isSet$1(object.selectable) ? globalThis.Boolean(object.selectable) : void 0,
        colors: isSet$1(object.colors) ? BlockColor.fromJSON(object.colors) : void 0,
        wrap: isSet$1(object.wrap) ? globalThis.Boolean(object.wrap) : void 0,
        overflow: isSet$1(object.overflow) ? blockTextOverflowFromJSON(object.overflow) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.text !== "") {
        obj.text = message.text;
      }
      if (message.size !== void 0) {
        obj.size = blockTextSizeToJSON(message.size);
      }
      if (message.weight !== void 0) {
        obj.weight = blockTextWeightToJSON(message.weight);
      }
      if (message.color !== void 0) {
        obj.color = message.color;
      }
      if (message.alignment !== void 0) {
        obj.alignment = BlockAlignment.toJSON(message.alignment);
      }
      if (message.outline !== void 0) {
        obj.outline = blockTextOutlineToJSON(message.outline);
      }
      if (message.style !== void 0) {
        obj.style = blockTextStyleToJSON(message.style);
      }
      if (message.selectable !== void 0) {
        obj.selectable = message.selectable;
      }
      if (message.colors !== void 0) {
        obj.colors = BlockColor.toJSON(message.colors);
      }
      if (message.wrap !== void 0) {
        obj.wrap = message.wrap;
      }
      if (message.overflow !== void 0) {
        obj.overflow = blockTextOverflowToJSON(message.overflow);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Text.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Text();
      message.text = object.text ?? "";
      message.size = object.size ?? void 0;
      message.weight = object.weight ?? void 0;
      message.color = object.color ?? void 0;
      message.alignment = object.alignment !== void 0 && object.alignment !== null ? BlockAlignment.fromPartial(object.alignment) : void 0;
      message.outline = object.outline ?? void 0;
      message.style = object.style ?? void 0;
      message.selectable = object.selectable ?? void 0;
      message.colors = object.colors !== void 0 && object.colors !== null ? BlockColor.fromPartial(object.colors) : void 0;
      message.wrap = object.wrap ?? void 0;
      message.overflow = object.overflow ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Text.$type, BlockConfig_Text);
  function createBaseBlockConfig_Button() {
    return {
      text: void 0,
      icon: void 0,
      buttonSize: void 0,
      buttonAppearance: void 0,
      textColor: void 0,
      backgroundColor: void 0,
      disabled: void 0,
      textColors: void 0,
      backgroundColors: void 0
    };
  }
  const BlockConfig_Button = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Button",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.text !== void 0) {
        writer2.uint32(10).string(message.text);
      }
      if (message.icon !== void 0) {
        writer2.uint32(18).string(message.icon);
      }
      if (message.buttonSize !== void 0) {
        writer2.uint32(24).int32(message.buttonSize);
      }
      if (message.buttonAppearance !== void 0) {
        writer2.uint32(32).int32(message.buttonAppearance);
      }
      if (message.textColor !== void 0) {
        writer2.uint32(42).string(message.textColor);
      }
      if (message.backgroundColor !== void 0) {
        writer2.uint32(50).string(message.backgroundColor);
      }
      if (message.disabled !== void 0) {
        writer2.uint32(56).bool(message.disabled);
      }
      if (message.textColors !== void 0) {
        BlockColor.encode(message.textColors, writer2.uint32(66).fork()).ldelim();
      }
      if (message.backgroundColors !== void 0) {
        BlockColor.encode(message.backgroundColors, writer2.uint32(74).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Button();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.text = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.icon = reader2.string();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.buttonSize = reader2.int32();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.buttonAppearance = reader2.int32();
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.textColor = reader2.string();
            continue;
          case 6:
            if (tag !== 50) {
              break;
            }
            message.backgroundColor = reader2.string();
            continue;
          case 7:
            if (tag !== 56) {
              break;
            }
            message.disabled = reader2.bool();
            continue;
          case 8:
            if (tag !== 66) {
              break;
            }
            message.textColors = BlockColor.decode(reader2, reader2.uint32());
            continue;
          case 9:
            if (tag !== 74) {
              break;
            }
            message.backgroundColors = BlockColor.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        text: isSet$1(object.text) ? globalThis.String(object.text) : void 0,
        icon: isSet$1(object.icon) ? globalThis.String(object.icon) : void 0,
        buttonSize: isSet$1(object.buttonSize) ? blockButtonSizeFromJSON(object.buttonSize) : void 0,
        buttonAppearance: isSet$1(object.buttonAppearance) ? blockButtonAppearanceFromJSON(object.buttonAppearance) : void 0,
        textColor: isSet$1(object.textColor) ? globalThis.String(object.textColor) : void 0,
        backgroundColor: isSet$1(object.backgroundColor) ? globalThis.String(object.backgroundColor) : void 0,
        disabled: isSet$1(object.disabled) ? globalThis.Boolean(object.disabled) : void 0,
        textColors: isSet$1(object.textColors) ? BlockColor.fromJSON(object.textColors) : void 0,
        backgroundColors: isSet$1(object.backgroundColors) ? BlockColor.fromJSON(object.backgroundColors) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.text !== void 0) {
        obj.text = message.text;
      }
      if (message.icon !== void 0) {
        obj.icon = message.icon;
      }
      if (message.buttonSize !== void 0) {
        obj.buttonSize = blockButtonSizeToJSON(message.buttonSize);
      }
      if (message.buttonAppearance !== void 0) {
        obj.buttonAppearance = blockButtonAppearanceToJSON(message.buttonAppearance);
      }
      if (message.textColor !== void 0) {
        obj.textColor = message.textColor;
      }
      if (message.backgroundColor !== void 0) {
        obj.backgroundColor = message.backgroundColor;
      }
      if (message.disabled !== void 0) {
        obj.disabled = message.disabled;
      }
      if (message.textColors !== void 0) {
        obj.textColors = BlockColor.toJSON(message.textColors);
      }
      if (message.backgroundColors !== void 0) {
        obj.backgroundColors = BlockColor.toJSON(message.backgroundColors);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Button.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Button();
      message.text = object.text ?? void 0;
      message.icon = object.icon ?? void 0;
      message.buttonSize = object.buttonSize ?? void 0;
      message.buttonAppearance = object.buttonAppearance ?? void 0;
      message.textColor = object.textColor ?? void 0;
      message.backgroundColor = object.backgroundColor ?? void 0;
      message.disabled = object.disabled ?? void 0;
      message.textColors = object.textColors !== void 0 && object.textColors !== null ? BlockColor.fromPartial(object.textColors) : void 0;
      message.backgroundColors = object.backgroundColors !== void 0 && object.backgroundColors !== null ? BlockColor.fromPartial(object.backgroundColors) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Button.$type, BlockConfig_Button);
  function createBaseBlockConfig_Image() {
    return { url: "", width: 0, height: 0, description: void 0, resizeMode: void 0 };
  }
  const BlockConfig_Image = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Image",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.url !== "") {
        writer2.uint32(10).string(message.url);
      }
      if (message.width !== 0) {
        writer2.uint32(16).int32(message.width);
      }
      if (message.height !== 0) {
        writer2.uint32(24).int32(message.height);
      }
      if (message.description !== void 0) {
        writer2.uint32(34).string(message.description);
      }
      if (message.resizeMode !== void 0) {
        writer2.uint32(40).int32(message.resizeMode);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Image();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.url = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.width = reader2.int32();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.height = reader2.int32();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.description = reader2.string();
            continue;
          case 5:
            if (tag !== 40) {
              break;
            }
            message.resizeMode = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        url: isSet$1(object.url) ? globalThis.String(object.url) : "",
        width: isSet$1(object.width) ? globalThis.Number(object.width) : 0,
        height: isSet$1(object.height) ? globalThis.Number(object.height) : 0,
        description: isSet$1(object.description) ? globalThis.String(object.description) : void 0,
        resizeMode: isSet$1(object.resizeMode) ? blockImageResizeModeFromJSON(object.resizeMode) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.url !== "") {
        obj.url = message.url;
      }
      if (message.width !== 0) {
        obj.width = Math.round(message.width);
      }
      if (message.height !== 0) {
        obj.height = Math.round(message.height);
      }
      if (message.description !== void 0) {
        obj.description = message.description;
      }
      if (message.resizeMode !== void 0) {
        obj.resizeMode = blockImageResizeModeToJSON(message.resizeMode);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Image.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Image();
      message.url = object.url ?? "";
      message.width = object.width ?? 0;
      message.height = object.height ?? 0;
      message.description = object.description ?? void 0;
      message.resizeMode = object.resizeMode ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Image.$type, BlockConfig_Image);
  function createBaseBlockConfig_Spacer() {
    return { size: void 0, shape: void 0 };
  }
  const BlockConfig_Spacer = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Spacer",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.size !== void 0) {
        writer2.uint32(8).int32(message.size);
      }
      if (message.shape !== void 0) {
        writer2.uint32(16).int32(message.shape);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Spacer();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 8) {
              break;
            }
            message.size = reader2.int32();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.shape = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        size: isSet$1(object.size) ? blockSpacerSizeFromJSON(object.size) : void 0,
        shape: isSet$1(object.shape) ? blockSpacerShapeFromJSON(object.shape) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.size !== void 0) {
        obj.size = blockSpacerSizeToJSON(message.size);
      }
      if (message.shape !== void 0) {
        obj.shape = blockSpacerShapeToJSON(message.shape);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Spacer.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Spacer();
      message.size = object.size ?? void 0;
      message.shape = object.shape ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Spacer.$type, BlockConfig_Spacer);
  function createBaseBlockConfig_Icon() {
    return { icon: "", color: void 0, size: void 0, colors: void 0 };
  }
  const BlockConfig_Icon = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Icon",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.icon !== "") {
        writer2.uint32(10).string(message.icon);
      }
      if (message.color !== void 0) {
        writer2.uint32(18).string(message.color);
      }
      if (message.size !== void 0) {
        writer2.uint32(24).int32(message.size);
      }
      if (message.colors !== void 0) {
        BlockColor.encode(message.colors, writer2.uint32(34).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Icon();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.icon = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.color = reader2.string();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.size = reader2.int32();
            continue;
          case 4:
            if (tag !== 34) {
              break;
            }
            message.colors = BlockColor.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        icon: isSet$1(object.icon) ? globalThis.String(object.icon) : "",
        color: isSet$1(object.color) ? globalThis.String(object.color) : void 0,
        size: isSet$1(object.size) ? blockIconSizeFromJSON(object.size) : void 0,
        colors: isSet$1(object.colors) ? BlockColor.fromJSON(object.colors) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.icon !== "") {
        obj.icon = message.icon;
      }
      if (message.color !== void 0) {
        obj.color = message.color;
      }
      if (message.size !== void 0) {
        obj.size = blockIconSizeToJSON(message.size);
      }
      if (message.colors !== void 0) {
        obj.colors = BlockColor.toJSON(message.colors);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Icon.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Icon();
      message.icon = object.icon ?? "";
      message.color = object.color ?? void 0;
      message.size = object.size ?? void 0;
      message.colors = object.colors !== void 0 && object.colors !== null ? BlockColor.fromPartial(object.colors) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Icon.$type, BlockConfig_Icon);
  function createBaseBlockConfig_Avatar() {
    return { thingId: "", facing: void 0, size: void 0, background: void 0 };
  }
  const BlockConfig_Avatar = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Avatar",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.thingId !== "") {
        writer2.uint32(10).string(message.thingId);
      }
      if (message.facing !== void 0) {
        writer2.uint32(16).int32(message.facing);
      }
      if (message.size !== void 0) {
        writer2.uint32(24).int32(message.size);
      }
      if (message.background !== void 0) {
        writer2.uint32(32).int32(message.background);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Avatar();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.thingId = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.facing = reader2.int32();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.size = reader2.int32();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.background = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        thingId: isSet$1(object.thingId) ? globalThis.String(object.thingId) : "",
        facing: isSet$1(object.facing) ? blockAvatarFacingFromJSON(object.facing) : void 0,
        size: isSet$1(object.size) ? blockAvatarSizeFromJSON(object.size) : void 0,
        background: isSet$1(object.background) ? blockAvatarBackgroundFromJSON(object.background) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.thingId !== "") {
        obj.thingId = message.thingId;
      }
      if (message.facing !== void 0) {
        obj.facing = blockAvatarFacingToJSON(message.facing);
      }
      if (message.size !== void 0) {
        obj.size = blockAvatarSizeToJSON(message.size);
      }
      if (message.background !== void 0) {
        obj.background = blockAvatarBackgroundToJSON(message.background);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Avatar.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Avatar();
      message.thingId = object.thingId ?? "";
      message.facing = object.facing ?? void 0;
      message.size = object.size ?? void 0;
      message.background = object.background ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Avatar.$type, BlockConfig_Avatar);
  function createBaseBlockConfig_FullSnoo() {
    return { userId: "", facing: void 0, size: void 0 };
  }
  const BlockConfig_FullSnoo = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.FullSnoo",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.userId !== "") {
        writer2.uint32(10).string(message.userId);
      }
      if (message.facing !== void 0) {
        writer2.uint32(16).int32(message.facing);
      }
      if (message.size !== void 0) {
        writer2.uint32(24).int32(message.size);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_FullSnoo();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.userId = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.facing = reader2.int32();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.size = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        userId: isSet$1(object.userId) ? globalThis.String(object.userId) : "",
        facing: isSet$1(object.facing) ? blockAvatarFacingFromJSON(object.facing) : void 0,
        size: isSet$1(object.size) ? blockFullSnooSizeFromJSON(object.size) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.userId !== "") {
        obj.userId = message.userId;
      }
      if (message.facing !== void 0) {
        obj.facing = blockAvatarFacingToJSON(message.facing);
      }
      if (message.size !== void 0) {
        obj.size = blockFullSnooSizeToJSON(message.size);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_FullSnoo.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_FullSnoo();
      message.userId = object.userId ?? "";
      message.facing = object.facing ?? void 0;
      message.size = object.size ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_FullSnoo.$type, BlockConfig_FullSnoo);
  function createBaseBlockConfig_Animation() {
    return {
      url: "",
      width: 0,
      height: 0,
      type: 0,
      loop: void 0,
      loopMode: void 0,
      autoplay: void 0,
      speed: void 0,
      direction: void 0
    };
  }
  const BlockConfig_Animation = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.Animation",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.url !== "") {
        writer2.uint32(10).string(message.url);
      }
      if (message.width !== 0) {
        writer2.uint32(16).int32(message.width);
      }
      if (message.height !== 0) {
        writer2.uint32(24).int32(message.height);
      }
      if (message.type !== 0) {
        writer2.uint32(32).int32(message.type);
      }
      if (message.loop !== void 0) {
        writer2.uint32(40).bool(message.loop);
      }
      if (message.loopMode !== void 0) {
        writer2.uint32(48).int32(message.loopMode);
      }
      if (message.autoplay !== void 0) {
        writer2.uint32(56).bool(message.autoplay);
      }
      if (message.speed !== void 0) {
        writer2.uint32(69).float(message.speed);
      }
      if (message.direction !== void 0) {
        writer2.uint32(72).int32(message.direction);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_Animation();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.url = reader2.string();
            continue;
          case 2:
            if (tag !== 16) {
              break;
            }
            message.width = reader2.int32();
            continue;
          case 3:
            if (tag !== 24) {
              break;
            }
            message.height = reader2.int32();
            continue;
          case 4:
            if (tag !== 32) {
              break;
            }
            message.type = reader2.int32();
            continue;
          case 5:
            if (tag !== 40) {
              break;
            }
            message.loop = reader2.bool();
            continue;
          case 6:
            if (tag !== 48) {
              break;
            }
            message.loopMode = reader2.int32();
            continue;
          case 7:
            if (tag !== 56) {
              break;
            }
            message.autoplay = reader2.bool();
            continue;
          case 8:
            if (tag !== 69) {
              break;
            }
            message.speed = reader2.float();
            continue;
          case 9:
            if (tag !== 72) {
              break;
            }
            message.direction = reader2.int32();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        url: isSet$1(object.url) ? globalThis.String(object.url) : "",
        width: isSet$1(object.width) ? globalThis.Number(object.width) : 0,
        height: isSet$1(object.height) ? globalThis.Number(object.height) : 0,
        type: isSet$1(object.type) ? blockAnimationTypeFromJSON(object.type) : 0,
        loop: isSet$1(object.loop) ? globalThis.Boolean(object.loop) : void 0,
        loopMode: isSet$1(object.loopMode) ? blockAnimationLoopModeFromJSON(object.loopMode) : void 0,
        autoplay: isSet$1(object.autoplay) ? globalThis.Boolean(object.autoplay) : void 0,
        speed: isSet$1(object.speed) ? globalThis.Number(object.speed) : void 0,
        direction: isSet$1(object.direction) ? blockAnimationDirectionFromJSON(object.direction) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.url !== "") {
        obj.url = message.url;
      }
      if (message.width !== 0) {
        obj.width = Math.round(message.width);
      }
      if (message.height !== 0) {
        obj.height = Math.round(message.height);
      }
      if (message.type !== 0) {
        obj.type = blockAnimationTypeToJSON(message.type);
      }
      if (message.loop !== void 0) {
        obj.loop = message.loop;
      }
      if (message.loopMode !== void 0) {
        obj.loopMode = blockAnimationLoopModeToJSON(message.loopMode);
      }
      if (message.autoplay !== void 0) {
        obj.autoplay = message.autoplay;
      }
      if (message.speed !== void 0) {
        obj.speed = message.speed;
      }
      if (message.direction !== void 0) {
        obj.direction = blockAnimationDirectionToJSON(message.direction);
      }
      return obj;
    },
    create(base) {
      return BlockConfig_Animation.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_Animation();
      message.url = object.url ?? "";
      message.width = object.width ?? 0;
      message.height = object.height ?? 0;
      message.type = object.type ?? 0;
      message.loop = object.loop ?? void 0;
      message.loopMode = object.loopMode ?? void 0;
      message.autoplay = object.autoplay ?? void 0;
      message.speed = object.speed ?? void 0;
      message.direction = object.direction ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_Animation.$type, BlockConfig_Animation);
  function createBaseBlockConfig_WebView() {
    return { url: "", state: void 0 };
  }
  const BlockConfig_WebView = {
    $type: "devvit.ui.block_kit.v1beta.BlockConfig.WebView",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.url !== "") {
        writer2.uint32(10).string(message.url);
      }
      if (message.state !== void 0) {
        Struct.encode(Struct.wrap(message.state), writer2.uint32(18).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseBlockConfig_WebView();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.url = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.state = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        url: isSet$1(object.url) ? globalThis.String(object.url) : "",
        state: isObject$1(object.state) ? object.state : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.url !== "") {
        obj.url = message.url;
      }
      if (message.state !== void 0) {
        obj.state = message.state;
      }
      return obj;
    },
    create(base) {
      return BlockConfig_WebView.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseBlockConfig_WebView();
      message.url = object.url ?? "";
      message.state = object.state ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(BlockConfig_WebView.$type, BlockConfig_WebView);
  function isObject$1(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet$1(value) {
    return value !== null && value !== void 0;
  }
  function createBaseUIRequest() {
    return { env: void 0, props: void 0, state: void 0, events: [] };
  }
  const UIRequest = {
    $type: "devvit.ui.block_kit.v1beta.UIRequest",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.env !== void 0) {
        UIEnvironment.encode(message.env, writer2.uint32(42).fork()).ldelim();
      }
      if (message.props !== void 0) {
        Struct.encode(Struct.wrap(message.props), writer2.uint32(10).fork()).ldelim();
      }
      if (message.state !== void 0) {
        Struct.encode(Struct.wrap(message.state), writer2.uint32(18).fork()).ldelim();
      }
      for (const v of message.events) {
        UIEvent.encode(v, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUIRequest();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 5:
            if (tag !== 42) {
              break;
            }
            message.env = UIEnvironment.decode(reader2, reader2.uint32());
            continue;
          case 1:
            if (tag !== 10) {
              break;
            }
            message.props = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.state = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.events.push(UIEvent.decode(reader2, reader2.uint32()));
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        env: isSet(object.env) ? UIEnvironment.fromJSON(object.env) : void 0,
        props: isObject(object.props) ? object.props : void 0,
        state: isObject(object.state) ? object.state : void 0,
        events: globalThis.Array.isArray(object?.events) ? object.events.map((e) => UIEvent.fromJSON(e)) : []
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.env !== void 0) {
        obj.env = UIEnvironment.toJSON(message.env);
      }
      if (message.props !== void 0) {
        obj.props = message.props;
      }
      if (message.state !== void 0) {
        obj.state = message.state;
      }
      if (message.events?.length) {
        obj.events = message.events.map((e) => UIEvent.toJSON(e));
      }
      return obj;
    },
    create(base) {
      return UIRequest.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUIRequest();
      message.env = object.env !== void 0 && object.env !== null ? UIEnvironment.fromPartial(object.env) : void 0;
      message.props = object.props ?? void 0;
      message.state = object.state ?? void 0;
      message.events = object.events?.map((e) => UIEvent.fromPartial(e)) || [];
      return message;
    }
  };
  messageTypeRegistry.set(UIRequest.$type, UIRequest);
  function createBaseUIResponse() {
    return { state: void 0, effects: [], events: [], blocks: void 0 };
  }
  const UIResponse = {
    $type: "devvit.ui.block_kit.v1beta.UIResponse",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.state !== void 0) {
        Struct.encode(Struct.wrap(message.state), writer2.uint32(10).fork()).ldelim();
      }
      for (const v of message.effects) {
        Effect.encode(v, writer2.uint32(18).fork()).ldelim();
      }
      for (const v of message.events) {
        UIEvent.encode(v, writer2.uint32(42).fork()).ldelim();
      }
      if (message.blocks !== void 0) {
        Block.encode(message.blocks, writer2.uint32(26).fork()).ldelim();
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUIResponse();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.state = Struct.unwrap(Struct.decode(reader2, reader2.uint32()));
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.effects.push(Effect.decode(reader2, reader2.uint32()));
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.events.push(UIEvent.decode(reader2, reader2.uint32()));
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.blocks = Block.decode(reader2, reader2.uint32());
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        state: isObject(object.state) ? object.state : void 0,
        effects: globalThis.Array.isArray(object?.effects) ? object.effects.map((e) => Effect.fromJSON(e)) : [],
        events: globalThis.Array.isArray(object?.events) ? object.events.map((e) => UIEvent.fromJSON(e)) : [],
        blocks: isSet(object.blocks) ? Block.fromJSON(object.blocks) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.state !== void 0) {
        obj.state = message.state;
      }
      if (message.effects?.length) {
        obj.effects = message.effects.map((e) => Effect.toJSON(e));
      }
      if (message.events?.length) {
        obj.events = message.events.map((e) => UIEvent.toJSON(e));
      }
      if (message.blocks !== void 0) {
        obj.blocks = Block.toJSON(message.blocks);
      }
      return obj;
    },
    create(base) {
      return UIResponse.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUIResponse();
      message.state = object.state ?? void 0;
      message.effects = object.effects?.map((e) => Effect.fromPartial(e)) || [];
      message.events = object.events?.map((e) => UIEvent.fromPartial(e)) || [];
      message.blocks = object.blocks !== void 0 && object.blocks !== null ? Block.fromPartial(object.blocks) : void 0;
      return message;
    }
  };
  messageTypeRegistry.set(UIResponse.$type, UIResponse);
  function createBaseUIEnvironment() {
    return { locale: void 0, colorScheme: void 0, dimensions: void 0, timezone: void 0 };
  }
  const UIEnvironment = {
    $type: "devvit.ui.block_kit.v1beta.UIEnvironment",
    encode(message, writer2 = _m0.Writer.create()) {
      if (message.locale !== void 0) {
        writer2.uint32(10).string(message.locale);
      }
      if (message.colorScheme !== void 0) {
        writer2.uint32(18).string(message.colorScheme);
      }
      if (message.dimensions !== void 0) {
        UIDimensions.encode(message.dimensions, writer2.uint32(26).fork()).ldelim();
      }
      if (message.timezone !== void 0) {
        writer2.uint32(42).string(message.timezone);
      }
      return writer2;
    },
    decode(input, length) {
      const reader2 = input instanceof _m0.Reader ? input : _m0.Reader.create(input);
      let end = length === void 0 ? reader2.len : reader2.pos + length;
      const message = createBaseUIEnvironment();
      while (reader2.pos < end) {
        const tag = reader2.uint32();
        switch (tag >>> 3) {
          case 1:
            if (tag !== 10) {
              break;
            }
            message.locale = reader2.string();
            continue;
          case 2:
            if (tag !== 18) {
              break;
            }
            message.colorScheme = reader2.string();
            continue;
          case 3:
            if (tag !== 26) {
              break;
            }
            message.dimensions = UIDimensions.decode(reader2, reader2.uint32());
            continue;
          case 5:
            if (tag !== 42) {
              break;
            }
            message.timezone = reader2.string();
            continue;
        }
        if ((tag & 7) === 4 || tag === 0) {
          break;
        }
        reader2.skipType(tag & 7);
      }
      return message;
    },
    fromJSON(object) {
      return {
        locale: isSet(object.locale) ? globalThis.String(object.locale) : void 0,
        colorScheme: isSet(object.colorScheme) ? globalThis.String(object.colorScheme) : void 0,
        dimensions: isSet(object.dimensions) ? UIDimensions.fromJSON(object.dimensions) : void 0,
        timezone: isSet(object.timezone) ? globalThis.String(object.timezone) : void 0
      };
    },
    toJSON(message) {
      const obj = {};
      if (message.locale !== void 0) {
        obj.locale = message.locale;
      }
      if (message.colorScheme !== void 0) {
        obj.colorScheme = message.colorScheme;
      }
      if (message.dimensions !== void 0) {
        obj.dimensions = UIDimensions.toJSON(message.dimensions);
      }
      if (message.timezone !== void 0) {
        obj.timezone = message.timezone;
      }
      return obj;
    },
    create(base) {
      return UIEnvironment.fromPartial(base ?? {});
    },
    fromPartial(object) {
      const message = createBaseUIEnvironment();
      message.locale = object.locale ?? void 0;
      message.colorScheme = object.colorScheme ?? void 0;
      message.dimensions = object.dimensions !== void 0 && object.dimensions !== null ? UIDimensions.fromPartial(object.dimensions) : void 0;
      message.timezone = object.timezone ?? void 0;
      return message;
    }
  };
  messageTypeRegistry.set(UIEnvironment.$type, UIEnvironment);
  function isObject(value) {
    return typeof value === "object" && value !== null;
  }
  function isSet(value) {
    return value !== null && value !== void 0;
  }
  const ui = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
    __proto__: null,
    UIEnvironment,
    UIRequest,
    UIResponse
  }, Symbol.toStringTag, { value: "Module" }));
  function parseMissionData(arrayBuffer, postId) {
    try {
      let buffer = arrayBuffer;
      if (buffer.byteLength > 5) {
        buffer = buffer.slice(5);
      }
      const uint8Array = new Uint8Array(buffer);
      const response = UIResponse.decode(uint8Array);
      const json = UIResponse.toJSON(response);
      const data = { postId };
      if (json.state) {
        redditLogger.debug(`Decoded UIResponse for ${postId}:`, json);
        for (const [key, value] of Object.entries(json.state)) {
          if (typeof value === "object" && value !== null) {
            const stateValue = value.value;
            if (stateValue && typeof stateValue === "object") {
              if (stateValue.mission) {
                const mission = stateValue.mission;
                if (mission.difficulty !== void 0) {
                  data.difficulty = Math.round(mission.difficulty);
                }
                if (mission.minLevel !== void 0) {
                  data.minLevel = Math.round(mission.minLevel);
                }
                if (mission.maxLevel !== void 0) {
                  data.maxLevel = Math.round(mission.maxLevel);
                }
                if (mission.environment) {
                  data.environment = mission.environment;
                }
                if (mission.foodName) {
                  data.foodName = mission.foodName;
                }
                if (mission.foodImage) {
                  data.foodImage = mission.foodImage;
                }
                if (mission.authorWeaponId) {
                  data.authorWeaponId = mission.authorWeaponId;
                }
                if (mission.chef) {
                  data.chef = mission.chef;
                }
                if (mission.cart) {
                  data.cart = mission.cart;
                }
                if (mission.rarity) {
                  data.rarity = mission.rarity;
                }
                if (mission.type) {
                  data.type = mission.type;
                }
                if (mission.encounters && Array.isArray(mission.encounters)) {
                  data.encounters = mission.encounters;
                }
              }
              if (stateValue.isInnPost !== void 0) {
                data.isInnPost = stateValue.isInnPost;
              }
              if (stateValue.authorName) {
                data.authorName = stateValue.authorName;
              }
              if (stateValue.title) {
                data.title = stateValue.title;
              }
              if (stateValue.encounters && Array.isArray(stateValue.encounters)) {
                data.encounters = stateValue.encounters;
                redditLogger.log(`Found ${stateValue.encounters.length} encounters in ${key}`, {
                  encounters: stateValue.encounters
                });
              }
              if (stateValue.clearedMissions && typeof stateValue.clearedMissions === "object") {
                data.clearedMissions = stateValue.clearedMissions;
                redditLogger.log(`Found ${Object.keys(stateValue.clearedMissions).length} cleared missions in ${key}`, {
                  sample: Object.keys(stateValue.clearedMissions).slice(0, 5)
                });
              }
            }
          }
        }
      } else {
        redditLogger.warn(`No state object found in UIResponse for ${postId}`);
      }
      const jsonString = JSON.stringify(json);
      if (jsonString.includes("fxlui9egtgbf1.png")) {
        redditLogger.log(`Detected Inn image (fxlui9egtgbf1.png) in UIResponse for ${postId}`);
        data.__cleared = true;
      }
      redditLogger.log(`Parsed mission (${postId})`, data);
      return data;
    } catch (decoderError) {
      redditLogger.warn("UIResponse decoder failed, falling back to legacy parser", {
        error: decoderError instanceof Error ? decoderError.message : String(decoderError)
      });
    }
    try {
      const uint8Array = new Uint8Array(arrayBuffer);
      const text = new TextDecoder().decode(uint8Array);
      const data = { postId };
      const diffIndex = text.indexOf("difficulty");
      if (diffIndex >= 0) {
        const offsets = [10, 11, 12, 13, 14, 15, 16, 17, 18];
        for (const offset of offsets) {
          if (diffIndex + offset + 8 <= arrayBuffer.byteLength) {
            try {
              const dataView = new DataView(arrayBuffer, diffIndex + offset, 8);
              const difficultyValue = dataView.getFloat64(0, true);
              if (difficultyValue >= 1 && difficultyValue <= 5) {
                data.difficulty = Math.round(difficultyValue);
                break;
              }
            } catch (e) {
            }
          }
        }
      }
      const minIndex = text.indexOf("minLevel");
      if (minIndex >= 0) {
        const offsets = [8, 10, 12, 14, 16, 18, 20];
        for (const offset of offsets) {
          if (minIndex + offset + 8 <= arrayBuffer.byteLength) {
            try {
              const dataView = new DataView(arrayBuffer, minIndex + offset, 8);
              const minLevelValue = dataView.getFloat64(0, true);
              if (minLevelValue >= 1 && minLevelValue <= 340) {
                data.minLevel = Math.round(minLevelValue);
                break;
              }
            } catch (e) {
            }
          }
        }
      }
      const maxIndex = text.indexOf("maxLevel");
      if (maxIndex >= 0) {
        const offsets = [8, 10, 12, 14, 16, 18, 20];
        for (const offset of offsets) {
          if (maxIndex + offset + 8 <= arrayBuffer.byteLength) {
            try {
              const dataView = new DataView(arrayBuffer, maxIndex + offset, 8);
              const maxLevelValue = dataView.getFloat64(0, true);
              if (maxLevelValue >= 1 && maxLevelValue <= 340) {
                data.maxLevel = Math.round(maxLevelValue);
                break;
              }
            } catch (e) {
            }
          }
        }
      }
      const envMatch = text.match(/environment[\x00-\x1f]*([a-z_]+)/);
      if (envMatch) {
        data.environment = envMatch[1];
      }
      const foodNameMatch = text.match(/foodName[\x00-\x1f]*([A-Za-z0-9 ]+)/);
      if (foodNameMatch) {
        data.foodName = foodNameMatch[1].trim();
      }
      const authorMatch = text.match(/authorName[\x00-\x1f]*([a-zA-Z0-9_-]+)/);
      if (authorMatch) {
        data.authorName = authorMatch[1];
      }
      const titleIndex = text.indexOf("title");
      if (titleIndex >= 0) {
        const afterTitle = text.substring(titleIndex + 5);
        const titleStart = 4;
        const titleEnd = afterTitle.indexOf("\n", titleStart);
        if (titleEnd > titleStart) {
          data.title = afterTitle.substring(titleStart, titleEnd).trim();
        }
      }
      data.isInnPost = text.includes("isInnPost");
      if (text.includes("fxlui9egtgbf1.png")) {
        data.__cleared = true;
      }
      redditLogger.log(`Parsed mission (${data.postId}) data from API`, {
        ...data
      });
      return data;
    } catch (error) {
      redditLogger.error("Failed to parse mission data", {
        error: error instanceof Error ? error.message : String(error),
        postId
      });
      return null;
    }
  }
  function normalizePostId(id) {
    if (!id || typeof id !== "string") {
      return null;
    }
    if (id.startsWith("t3_")) {
      const postIdPart = id.slice(3);
      if (postIdPart && /^[a-z0-9]+$/i.test(postIdPart)) {
        return id;
      }
      return null;
    }
    if (/^[a-z0-9]+$/i.test(id)) {
      return `t3_${id}`;
    }
    return null;
  }
  function normalizeRedditPermalink(input) {
    const base = "https://www.reddit.com";
    const stripT3 = (id) => id?.startsWith("t3_") ? id.slice(3) : id;
    let postId = "";
    try {
      if (input?.startsWith("http")) {
        const url = new URL(input);
        const match = url.pathname.match(/\/comments\/([^/]+)/);
        if (match && match[1]) {
          postId = stripT3(match[1]);
        }
      } else if (input) {
        postId = stripT3(input);
      }
    } catch {
      postId = stripT3(input);
    }
    if (!postId) {
      return `${base}/r/SwordAndSupperGame/`;
    }
    return `${base}/r/SwordAndSupperGame/comments/${postId}/`;
  }
  function convertMissionDataToRecord(data) {
    if (!data.difficulty || !data.minLevel || !data.maxLevel || !data.environment) {
      return null;
    }
    if (!data.encounters || data.encounters.length === 0) {
      return null;
    }
    const record = {
      // Core identification
      postId: data.postId,
      timestamp: Date.now(),
      permalink: normalizeRedditPermalink(data.postId),
      // Mission metadata
      missionTitle: data.title || data.foodName || `Mission ${data.postId}`,
      missionAuthorName: data.authorName || "Unknown",
      // Mission data (from game state)
      environment: data.environment,
      encounters: data.encounters,
      minLevel: data.minLevel,
      maxLevel: data.maxLevel,
      difficulty: data.difficulty,
      foodImage: data.foodImage || "",
      foodName: data.foodName || "",
      authorWeaponId: data.authorWeaponId || "",
      chef: data.chef || "",
      cart: data.cart || "",
      rarity: data.rarity || "common",
      type: data.type
    };
    return record;
  }
  function isCompleteMissionData(data) {
    return !!(data.difficulty && data.minLevel && data.maxLevel && data.environment && data.foodName && data.encounters && data.encounters.length > 0);
  }
  async function saveMissionFromAPI(data) {
    if (isCompleteMissionData(data)) {
      const record = convertMissionDataToRecord(data);
      if (record) {
        try {
          const { getMission: getMission2 } = await Promise.resolve().then(() => missions);
          const existingMission = await getMission2(data.postId);
          if (existingMission?.timestamp) {
            record.timestamp = existingMission.timestamp;
          }
          await saveMission(record);
          redditLogger.log(`Saved complete mission from RenderPostContent: ${data.postId}`, {
            name: data.foodName,
            difficulty: data.difficulty,
            levels: `${data.minLevel}-${data.maxLevel}`,
            encounters: data.encounters?.length || 0
          });
          return;
        } catch (error) {
          redditLogger.error("Failed to save complete mission from RenderPostContent", {
            error: error instanceof Error ? error.message : String(error),
            postId: data.postId
          });
        }
      }
    }
    const missing = [];
    if (!data.difficulty) missing.push("difficulty");
    if (!data.minLevel) missing.push("minLevel");
    if (!data.maxLevel) missing.push("maxLevel");
    if (!data.environment) missing.push("environment");
    if (!data.foodName) missing.push("foodName");
    if (!data.encounters || data.encounters.length === 0) missing.push("encounters");
    redditLogger.warn("Incomplete mission data from RenderPostContent - cannot save", {
      postId: data.postId,
      missingFields: missing,
      hasData: {
        difficulty: !!data.difficulty,
        levels: !!(data.minLevel && data.maxLevel),
        environment: !!data.environment,
        foodName: !!data.foodName,
        encounters: data.encounters?.length || 0
      }
    });
  }
  function injectPageScript() {
    const scriptUrl = chrome.runtime.getURL("fetchInterceptor.js");
    redditLogger.log("Attempting to inject fetch interceptor", { scriptUrl });
    const script = document.createElement("script");
    script.src = scriptUrl;
    script.onload = () => {
      redditLogger.log("Fetch interceptor script loaded successfully");
      script.remove();
    };
    script.onerror = (error) => {
      redditLogger.error("Failed to load fetch interceptor script", { error: String(error) });
    };
    const target = document.head || document.documentElement;
    if (target) {
      target.appendChild(script);
      redditLogger.log("Script tag appended to DOM", { targetTag: target.tagName });
    } else {
      redditLogger.error("No document.head or documentElement available for injection");
    }
  }
  function installMissionDataHandler() {
    injectPageScript();
    window.addEventListener("autosupper:raw-mission-data", async (event) => {
      const customEvent = event;
      const { postId, arrayBuffer } = customEvent.detail;
      if (postId && arrayBuffer) {
        const data = parseMissionData(arrayBuffer, postId);
        if (data) {
          if (data.difficulty) {
            await saveMissionFromAPI(data);
          }
          if (data.__cleared === true || data.isInnPost === true) {
            redditLogger.log(
              "[missionDataHandler] Mission completion detected from PostRenderContent",
              {
                postId: data.postId,
                hasInnImage: data.__cleared,
                hasInnPostFlag: data.isInnPost
              }
            );
            await handleMissionCompletionFromData(data.postId);
          }
          if (data.clearedMissions && Object.keys(data.clearedMissions).length > 0) {
            try {
              await syncClearedMissionsFromServer(data.clearedMissions);
              redditLogger.log(
                "[missionDataHandler] Synced cleared missions from server",
                {
                  postId: data.postId,
                  count: Object.keys(data.clearedMissions).length
                }
              );
            } catch (error) {
              redditLogger.error("[missionDataHandler] Failed to sync cleared missions", {
                error: error instanceof Error ? error.message : String(error),
                postId: data.postId
              });
            }
          }
        }
      }
    });
    redditLogger.log("Mission data handler installed - listening for raw mission data events");
  }
  async function handleMissionCompletionFromData(postId) {
    redditLogger.log("[missionDataHandler] Handling mission completion", {
      postId
    });
    safeSendMessage({
      type: "MISSION_COMPLETED",
      postId,
      source: "postrendercontent-data"
    });
  }
  let pingInterval = null;
  function startPinging() {
    if (pingInterval) {
      clearInterval(pingInterval);
    }
    pingInterval = setInterval(() => {
      chrome.runtime.sendMessage({ type: "PING" }, (response) => {
        if (chrome.runtime.lastError) {
          redditLogger.error("[Ping] Failed to ping service worker", {
            error: chrome.runtime.lastError.message
          });
          stopPinging();
        } else if (response?.success) {
          redditLogger.log("[Ping] Service worker alive", {
            state: response.state,
            timestamp: response.timestamp
          });
        }
      });
    }, 2e4);
    redditLogger.log("[Ping] Started pinging service worker every 20s");
  }
  function stopPinging() {
    if (pingInterval) {
      clearInterval(pingInterval);
      pingInterval = null;
      redditLogger.log("[Ping] Stopped pinging service worker");
    }
  }
  function isRedditUrl(url) {
    try {
      const urlObj = new URL(url, window.location.origin);
      return urlObj.hostname === window.location.hostname;
    } catch {
      return false;
    }
  }
  function navigateToUrl(url, forceReload = false) {
    if (forceReload || !isRedditUrl(url)) {
      window.location.href = url;
      return;
    }
    try {
      window.history.pushState({}, "", url);
      window.dispatchEvent(
        new CustomEvent("spa-navigation", {
          detail: { url, timestamp: Date.now() }
        })
      );
    } catch (error) {
      redditLogger.error("[Navigation] SPA navigation failed", { error: String(error) });
      window.location.href = url;
    }
  }
  function onUrlChange(callback) {
    let lastUrl = window.location.href;
    const popstateHandler = () => {
      const newUrl = window.location.href;
      if (newUrl !== lastUrl) {
        lastUrl = newUrl;
        callback(newUrl);
      }
    };
    const spaNavHandler = ((e) => {
      const newUrl = e.detail.url;
      if (newUrl !== lastUrl) {
        lastUrl = newUrl;
        callback(newUrl);
      }
    });
    const observer2 = new MutationObserver(() => {
      const newUrl = window.location.href;
      if (newUrl !== lastUrl) {
        lastUrl = newUrl;
        callback(newUrl);
      }
    });
    window.addEventListener("popstate", popstateHandler);
    window.addEventListener("spa-navigation", spaNavHandler);
    observer2.observe(document.body, {
      childList: true,
      subtree: true
    });
    return () => {
      window.removeEventListener("popstate", popstateHandler);
      window.removeEventListener("spa-navigation", spaNavHandler);
      observer2.disconnect();
    };
  }
  const DOM_UPDATE_DELAY = 500;
  const GAME_LOADER_CHECK_INTERVAL = 500;
  const GAME_LOADER_MAX_WAIT = 1e4;
  var jsxRuntime = { exports: {} };
  var reactJsxRuntime_production = {};
  /**
   * @license React
   * react-jsx-runtime.production.js
   *
   * Copyright (c) Meta Platforms, Inc. and affiliates.
   *
   * This source code is licensed under the MIT license found in the
   * LICENSE file in the root directory of this source tree.
   */
  var hasRequiredReactJsxRuntime_production;
  function requireReactJsxRuntime_production() {
    if (hasRequiredReactJsxRuntime_production) return reactJsxRuntime_production;
    hasRequiredReactJsxRuntime_production = 1;
    var REACT_ELEMENT_TYPE = Symbol.for("react.transitional.element"), REACT_FRAGMENT_TYPE = Symbol.for("react.fragment");
    function jsxProd(type, config, maybeKey) {
      var key = null;
      void 0 !== maybeKey && (key = "" + maybeKey);
      void 0 !== config.key && (key = "" + config.key);
      if ("key" in config) {
        maybeKey = {};
        for (var propName in config)
          "key" !== propName && (maybeKey[propName] = config[propName]);
      } else maybeKey = config;
      config = maybeKey.ref;
      return {
        $$typeof: REACT_ELEMENT_TYPE,
        type,
        key,
        ref: void 0 !== config ? config : null,
        props: maybeKey
      };
    }
    reactJsxRuntime_production.Fragment = REACT_FRAGMENT_TYPE;
    reactJsxRuntime_production.jsx = jsxProd;
    reactJsxRuntime_production.jsxs = jsxProd;
    return reactJsxRuntime_production;
  }
  var hasRequiredJsxRuntime;
  function requireJsxRuntime() {
    if (hasRequiredJsxRuntime) return jsxRuntime.exports;
    hasRequiredJsxRuntime = 1;
    {
      jsxRuntime.exports = requireReactJsxRuntime_production();
    }
    return jsxRuntime.exports;
  }
  var jsxRuntimeExports = requireJsxRuntime();
  var client = { exports: {} };
  var reactDomClient_production = {};
  var scheduler = { exports: {} };
  var scheduler_production = {};
  /**
   * @license React
   * scheduler.production.js
   *
   * Copyright (c) Meta Platforms, Inc. and affiliates.
   *
   * This source code is licensed under the MIT license found in the
   * LICENSE file in the root directory of this source tree.
   */
  var hasRequiredScheduler_production;
  function requireScheduler_production() {
    if (hasRequiredScheduler_production) return scheduler_production;
    hasRequiredScheduler_production = 1;
    (function(exports) {
      function push(heap, node) {
        var index = heap.length;
        heap.push(node);
        a: for (; 0 < index; ) {
          var parentIndex = index - 1 >>> 1, parent = heap[parentIndex];
          if (0 < compare(parent, node))
            heap[parentIndex] = node, heap[index] = parent, index = parentIndex;
          else break a;
        }
      }
      function peek(heap) {
        return 0 === heap.length ? null : heap[0];
      }
      function pop(heap) {
        if (0 === heap.length) return null;
        var first = heap[0], last = heap.pop();
        if (last !== first) {
          heap[0] = last;
          a: for (var index = 0, length = heap.length, halfLength = length >>> 1; index < halfLength; ) {
            var leftIndex = 2 * (index + 1) - 1, left = heap[leftIndex], rightIndex = leftIndex + 1, right = heap[rightIndex];
            if (0 > compare(left, last))
              rightIndex < length && 0 > compare(right, left) ? (heap[index] = right, heap[rightIndex] = last, index = rightIndex) : (heap[index] = left, heap[leftIndex] = last, index = leftIndex);
            else if (rightIndex < length && 0 > compare(right, last))
              heap[index] = right, heap[rightIndex] = last, index = rightIndex;
            else break a;
          }
        }
        return first;
      }
      function compare(a, b) {
        var diff = a.sortIndex - b.sortIndex;
        return 0 !== diff ? diff : a.id - b.id;
      }
      exports.unstable_now = void 0;
      if ("object" === typeof performance && "function" === typeof performance.now) {
        var localPerformance = performance;
        exports.unstable_now = function() {
          return localPerformance.now();
        };
      } else {
        var localDate = Date, initialTime = localDate.now();
        exports.unstable_now = function() {
          return localDate.now() - initialTime;
        };
      }
      var taskQueue = [], timerQueue = [], taskIdCounter = 1, currentTask = null, currentPriorityLevel = 3, isPerformingWork = false, isHostCallbackScheduled = false, isHostTimeoutScheduled = false, needsPaint = false, localSetTimeout = "function" === typeof setTimeout ? setTimeout : null, localClearTimeout = "function" === typeof clearTimeout ? clearTimeout : null, localSetImmediate = "undefined" !== typeof setImmediate ? setImmediate : null;
      function advanceTimers(currentTime) {
        for (var timer = peek(timerQueue); null !== timer; ) {
          if (null === timer.callback) pop(timerQueue);
          else if (timer.startTime <= currentTime)
            pop(timerQueue), timer.sortIndex = timer.expirationTime, push(taskQueue, timer);
          else break;
          timer = peek(timerQueue);
        }
      }
      function handleTimeout(currentTime) {
        isHostTimeoutScheduled = false;
        advanceTimers(currentTime);
        if (!isHostCallbackScheduled)
          if (null !== peek(taskQueue))
            isHostCallbackScheduled = true, isMessageLoopRunning || (isMessageLoopRunning = true, schedulePerformWorkUntilDeadline());
          else {
            var firstTimer = peek(timerQueue);
            null !== firstTimer && requestHostTimeout(handleTimeout, firstTimer.startTime - currentTime);
          }
      }
      var isMessageLoopRunning = false, taskTimeoutID = -1, frameInterval = 5, startTime = -1;
      function shouldYieldToHost() {
        return needsPaint ? true : exports.unstable_now() - startTime < frameInterval ? false : true;
      }
      function performWorkUntilDeadline() {
        needsPaint = false;
        if (isMessageLoopRunning) {
          var currentTime = exports.unstable_now();
          startTime = currentTime;
          var hasMoreWork = true;
          try {
            a: {
              isHostCallbackScheduled = false;
              isHostTimeoutScheduled && (isHostTimeoutScheduled = false, localClearTimeout(taskTimeoutID), taskTimeoutID = -1);
              isPerformingWork = true;
              var previousPriorityLevel = currentPriorityLevel;
              try {
                b: {
                  advanceTimers(currentTime);
                  for (currentTask = peek(taskQueue); null !== currentTask && !(currentTask.expirationTime > currentTime && shouldYieldToHost()); ) {
                    var callback = currentTask.callback;
                    if ("function" === typeof callback) {
                      currentTask.callback = null;
                      currentPriorityLevel = currentTask.priorityLevel;
                      var continuationCallback = callback(
                        currentTask.expirationTime <= currentTime
                      );
                      currentTime = exports.unstable_now();
                      if ("function" === typeof continuationCallback) {
                        currentTask.callback = continuationCallback;
                        advanceTimers(currentTime);
                        hasMoreWork = true;
                        break b;
                      }
                      currentTask === peek(taskQueue) && pop(taskQueue);
                      advanceTimers(currentTime);
                    } else pop(taskQueue);
                    currentTask = peek(taskQueue);
                  }
                  if (null !== currentTask) hasMoreWork = true;
                  else {
                    var firstTimer = peek(timerQueue);
                    null !== firstTimer && requestHostTimeout(
                      handleTimeout,
                      firstTimer.startTime - currentTime
                    );
                    hasMoreWork = false;
                  }
                }
                break a;
              } finally {
                currentTask = null, currentPriorityLevel = previousPriorityLevel, isPerformingWork = false;
              }
              hasMoreWork = void 0;
            }
          } finally {
            hasMoreWork ? schedulePerformWorkUntilDeadline() : isMessageLoopRunning = false;
          }
        }
      }
      var schedulePerformWorkUntilDeadline;
      if ("function" === typeof localSetImmediate)
        schedulePerformWorkUntilDeadline = function() {
          localSetImmediate(performWorkUntilDeadline);
        };
      else if ("undefined" !== typeof MessageChannel) {
        var channel = new MessageChannel(), port = channel.port2;
        channel.port1.onmessage = performWorkUntilDeadline;
        schedulePerformWorkUntilDeadline = function() {
          port.postMessage(null);
        };
      } else
        schedulePerformWorkUntilDeadline = function() {
          localSetTimeout(performWorkUntilDeadline, 0);
        };
      function requestHostTimeout(callback, ms) {
        taskTimeoutID = localSetTimeout(function() {
          callback(exports.unstable_now());
        }, ms);
      }
      exports.unstable_IdlePriority = 5;
      exports.unstable_ImmediatePriority = 1;
      exports.unstable_LowPriority = 4;
      exports.unstable_NormalPriority = 3;
      exports.unstable_Profiling = null;
      exports.unstable_UserBlockingPriority = 2;
      exports.unstable_cancelCallback = function(task) {
        task.callback = null;
      };
      exports.unstable_forceFrameRate = function(fps) {
        0 > fps || 125 < fps ? console.error(
          "forceFrameRate takes a positive int between 0 and 125, forcing frame rates higher than 125 fps is not supported"
        ) : frameInterval = 0 < fps ? Math.floor(1e3 / fps) : 5;
      };
      exports.unstable_getCurrentPriorityLevel = function() {
        return currentPriorityLevel;
      };
      exports.unstable_next = function(eventHandler) {
        switch (currentPriorityLevel) {
          case 1:
          case 2:
          case 3:
            var priorityLevel = 3;
            break;
          default:
            priorityLevel = currentPriorityLevel;
        }
        var previousPriorityLevel = currentPriorityLevel;
        currentPriorityLevel = priorityLevel;
        try {
          return eventHandler();
        } finally {
          currentPriorityLevel = previousPriorityLevel;
        }
      };
      exports.unstable_requestPaint = function() {
        needsPaint = true;
      };
      exports.unstable_runWithPriority = function(priorityLevel, eventHandler) {
        switch (priorityLevel) {
          case 1:
          case 2:
          case 3:
          case 4:
          case 5:
            break;
          default:
            priorityLevel = 3;
        }
        var previousPriorityLevel = currentPriorityLevel;
        currentPriorityLevel = priorityLevel;
        try {
          return eventHandler();
        } finally {
          currentPriorityLevel = previousPriorityLevel;
        }
      };
      exports.unstable_scheduleCallback = function(priorityLevel, callback, options) {
        var currentTime = exports.unstable_now();
        "object" === typeof options && null !== options ? (options = options.delay, options = "number" === typeof options && 0 < options ? currentTime + options : currentTime) : options = currentTime;
        switch (priorityLevel) {
          case 1:
            var timeout = -1;
            break;
          case 2:
            timeout = 250;
            break;
          case 5:
            timeout = 1073741823;
            break;
          case 4:
            timeout = 1e4;
            break;
          default:
            timeout = 5e3;
        }
        timeout = options + timeout;
        priorityLevel = {
          id: taskIdCounter++,
          callback,
          priorityLevel,
          startTime: options,
          expirationTime: timeout,
          sortIndex: -1
        };
        options > currentTime ? (priorityLevel.sortIndex = options, push(timerQueue, priorityLevel), null === peek(taskQueue) && priorityLevel === peek(timerQueue) && (isHostTimeoutScheduled ? (localClearTimeout(taskTimeoutID), taskTimeoutID = -1) : isHostTimeoutScheduled = true, requestHostTimeout(handleTimeout, options - currentTime))) : (priorityLevel.sortIndex = timeout, push(taskQueue, priorityLevel), isHostCallbackScheduled || isPerformingWork || (isHostCallbackScheduled = true, isMessageLoopRunning || (isMessageLoopRunning = true, schedulePerformWorkUntilDeadline())));
        return priorityLevel;
      };
      exports.unstable_shouldYield = shouldYieldToHost;
      exports.unstable_wrapCallback = function(callback) {
        var parentPriorityLevel = currentPriorityLevel;
        return function() {
          var previousPriorityLevel = currentPriorityLevel;
          currentPriorityLevel = parentPriorityLevel;
          try {
            return callback.apply(this, arguments);
          } finally {
            currentPriorityLevel = previousPriorityLevel;
          }
        };
      };
    })(scheduler_production);
    return scheduler_production;
  }
  var hasRequiredScheduler;
  function requireScheduler() {
    if (hasRequiredScheduler) return scheduler.exports;
    hasRequiredScheduler = 1;
    {
      scheduler.exports = requireScheduler_production();
    }
    return scheduler.exports;
  }
  var react = { exports: {} };
  var react_production = {};
  /**
   * @license React
   * react.production.js
   *
   * Copyright (c) Meta Platforms, Inc. and affiliates.
   *
   * This source code is licensed under the MIT license found in the
   * LICENSE file in the root directory of this source tree.
   */
  var hasRequiredReact_production;
  function requireReact_production() {
    if (hasRequiredReact_production) return react_production;
    hasRequiredReact_production = 1;
    var REACT_ELEMENT_TYPE = Symbol.for("react.transitional.element"), REACT_PORTAL_TYPE = Symbol.for("react.portal"), REACT_FRAGMENT_TYPE = Symbol.for("react.fragment"), REACT_STRICT_MODE_TYPE = Symbol.for("react.strict_mode"), REACT_PROFILER_TYPE = Symbol.for("react.profiler"), REACT_CONSUMER_TYPE = Symbol.for("react.consumer"), REACT_CONTEXT_TYPE = Symbol.for("react.context"), REACT_FORWARD_REF_TYPE = Symbol.for("react.forward_ref"), REACT_SUSPENSE_TYPE = Symbol.for("react.suspense"), REACT_MEMO_TYPE = Symbol.for("react.memo"), REACT_LAZY_TYPE = Symbol.for("react.lazy"), REACT_ACTIVITY_TYPE = Symbol.for("react.activity"), MAYBE_ITERATOR_SYMBOL = Symbol.iterator;
    function getIteratorFn(maybeIterable) {
      if (null === maybeIterable || "object" !== typeof maybeIterable) return null;
      maybeIterable = MAYBE_ITERATOR_SYMBOL && maybeIterable[MAYBE_ITERATOR_SYMBOL] || maybeIterable["@@iterator"];
      return "function" === typeof maybeIterable ? maybeIterable : null;
    }
    var ReactNoopUpdateQueue = {
      isMounted: function() {
        return false;
      },
      enqueueForceUpdate: function() {
      },
      enqueueReplaceState: function() {
      },
      enqueueSetState: function() {
      }
    }, assign = Object.assign, emptyObject = {};
    function Component(props, context, updater) {
      this.props = props;
      this.context = context;
      this.refs = emptyObject;
      this.updater = updater || ReactNoopUpdateQueue;
    }
    Component.prototype.isReactComponent = {};
    Component.prototype.setState = function(partialState, callback) {
      if ("object" !== typeof partialState && "function" !== typeof partialState && null != partialState)
        throw Error(
          "takes an object of state variables to update or a function which returns an object of state variables."
        );
      this.updater.enqueueSetState(this, partialState, callback, "setState");
    };
    Component.prototype.forceUpdate = function(callback) {
      this.updater.enqueueForceUpdate(this, callback, "forceUpdate");
    };
    function ComponentDummy() {
    }
    ComponentDummy.prototype = Component.prototype;
    function PureComponent(props, context, updater) {
      this.props = props;
      this.context = context;
      this.refs = emptyObject;
      this.updater = updater || ReactNoopUpdateQueue;
    }
    var pureComponentPrototype = PureComponent.prototype = new ComponentDummy();
    pureComponentPrototype.constructor = PureComponent;
    assign(pureComponentPrototype, Component.prototype);
    pureComponentPrototype.isPureReactComponent = true;
    var isArrayImpl = Array.isArray;
    function noop() {
    }
    var ReactSharedInternals = { H: null, A: null, T: null, S: null }, hasOwnProperty = Object.prototype.hasOwnProperty;
    function ReactElement(type, key, props) {
      var refProp = props.ref;
      return {
        $$typeof: REACT_ELEMENT_TYPE,
        type,
        key,
        ref: void 0 !== refProp ? refProp : null,
        props
      };
    }
    function cloneAndReplaceKey(oldElement, newKey) {
      return ReactElement(oldElement.type, newKey, oldElement.props);
    }
    function isValidElement(object) {
      return "object" === typeof object && null !== object && object.$$typeof === REACT_ELEMENT_TYPE;
    }
    function escape(key) {
      var escaperLookup = { "=": "=0", ":": "=2" };
      return "$" + key.replace(/[=:]/g, function(match) {
        return escaperLookup[match];
      });
    }
    var userProvidedKeyEscapeRegex = /\/+/g;
    function getElementKey(element, index) {
      return "object" === typeof element && null !== element && null != element.key ? escape("" + element.key) : index.toString(36);
    }
    function resolveThenable(thenable) {
      switch (thenable.status) {
        case "fulfilled":
          return thenable.value;
        case "rejected":
          throw thenable.reason;
        default:
          switch ("string" === typeof thenable.status ? thenable.then(noop, noop) : (thenable.status = "pending", thenable.then(
            function(fulfilledValue) {
              "pending" === thenable.status && (thenable.status = "fulfilled", thenable.value = fulfilledValue);
            },
            function(error) {
              "pending" === thenable.status && (thenable.status = "rejected", thenable.reason = error);
            }
          )), thenable.status) {
            case "fulfilled":
              return thenable.value;
            case "rejected":
              throw thenable.reason;
          }
      }
      throw thenable;
    }
    function mapIntoArray(children, array, escapedPrefix, nameSoFar, callback) {
      var type = typeof children;
      if ("undefined" === type || "boolean" === type) children = null;
      var invokeCallback = false;
      if (null === children) invokeCallback = true;
      else
        switch (type) {
          case "bigint":
          case "string":
          case "number":
            invokeCallback = true;
            break;
          case "object":
            switch (children.$$typeof) {
              case REACT_ELEMENT_TYPE:
              case REACT_PORTAL_TYPE:
                invokeCallback = true;
                break;
              case REACT_LAZY_TYPE:
                return invokeCallback = children._init, mapIntoArray(
                  invokeCallback(children._payload),
                  array,
                  escapedPrefix,
                  nameSoFar,
                  callback
                );
            }
        }
      if (invokeCallback)
        return callback = callback(children), invokeCallback = "" === nameSoFar ? "." + getElementKey(children, 0) : nameSoFar, isArrayImpl(callback) ? (escapedPrefix = "", null != invokeCallback && (escapedPrefix = invokeCallback.replace(userProvidedKeyEscapeRegex, "$&/") + "/"), mapIntoArray(callback, array, escapedPrefix, "", function(c) {
          return c;
        })) : null != callback && (isValidElement(callback) && (callback = cloneAndReplaceKey(
          callback,
          escapedPrefix + (null == callback.key || children && children.key === callback.key ? "" : ("" + callback.key).replace(
            userProvidedKeyEscapeRegex,
            "$&/"
          ) + "/") + invokeCallback
        )), array.push(callback)), 1;
      invokeCallback = 0;
      var nextNamePrefix = "" === nameSoFar ? "." : nameSoFar + ":";
      if (isArrayImpl(children))
        for (var i = 0; i < children.length; i++)
          nameSoFar = children[i], type = nextNamePrefix + getElementKey(nameSoFar, i), invokeCallback += mapIntoArray(
            nameSoFar,
            array,
            escapedPrefix,
            type,
            callback
          );
      else if (i = getIteratorFn(children), "function" === typeof i)
        for (children = i.call(children), i = 0; !(nameSoFar = children.next()).done; )
          nameSoFar = nameSoFar.value, type = nextNamePrefix + getElementKey(nameSoFar, i++), invokeCallback += mapIntoArray(
            nameSoFar,
            array,
            escapedPrefix,
            type,
            callback
          );
      else if ("object" === type) {
        if ("function" === typeof children.then)
          return mapIntoArray(
            resolveThenable(children),
            array,
            escapedPrefix,
            nameSoFar,
            callback
          );
        array = String(children);
        throw Error(
          "Objects are not valid as a React child (found: " + ("[object Object]" === array ? "object with keys {" + Object.keys(children).join(", ") + "}" : array) + "). If you meant to render a collection of children, use an array instead."
        );
      }
      return invokeCallback;
    }
    function mapChildren(children, func, context) {
      if (null == children) return children;
      var result2 = [], count = 0;
      mapIntoArray(children, result2, "", "", function(child) {
        return func.call(context, child, count++);
      });
      return result2;
    }
    function lazyInitializer(payload) {
      if (-1 === payload._status) {
        var ctor = payload._result;
        ctor = ctor();
        ctor.then(
          function(moduleObject) {
            if (0 === payload._status || -1 === payload._status)
              payload._status = 1, payload._result = moduleObject;
          },
          function(error) {
            if (0 === payload._status || -1 === payload._status)
              payload._status = 2, payload._result = error;
          }
        );
        -1 === payload._status && (payload._status = 0, payload._result = ctor);
      }
      if (1 === payload._status) return payload._result.default;
      throw payload._result;
    }
    var reportGlobalError = "function" === typeof reportError ? reportError : function(error) {
      if ("object" === typeof window && "function" === typeof window.ErrorEvent) {
        var event = new window.ErrorEvent("error", {
          bubbles: true,
          cancelable: true,
          message: "object" === typeof error && null !== error && "string" === typeof error.message ? String(error.message) : String(error),
          error
        });
        if (!window.dispatchEvent(event)) return;
      } else if ("object" === typeof process && "function" === typeof process.emit) {
        process.emit("uncaughtException", error);
        return;
      }
      console.error(error);
    }, Children = {
      map: mapChildren,
      forEach: function(children, forEachFunc, forEachContext) {
        mapChildren(
          children,
          function() {
            forEachFunc.apply(this, arguments);
          },
          forEachContext
        );
      },
      count: function(children) {
        var n = 0;
        mapChildren(children, function() {
          n++;
        });
        return n;
      },
      toArray: function(children) {
        return mapChildren(children, function(child) {
          return child;
        }) || [];
      },
      only: function(children) {
        if (!isValidElement(children))
          throw Error(
            "React.Children.only expected to receive a single React element child."
          );
        return children;
      }
    };
    react_production.Activity = REACT_ACTIVITY_TYPE;
    react_production.Children = Children;
    react_production.Component = Component;
    react_production.Fragment = REACT_FRAGMENT_TYPE;
    react_production.Profiler = REACT_PROFILER_TYPE;
    react_production.PureComponent = PureComponent;
    react_production.StrictMode = REACT_STRICT_MODE_TYPE;
    react_production.Suspense = REACT_SUSPENSE_TYPE;
    react_production.__CLIENT_INTERNALS_DO_NOT_USE_OR_WARN_USERS_THEY_CANNOT_UPGRADE = ReactSharedInternals;
    react_production.__COMPILER_RUNTIME = {
      __proto__: null,
      c: function(size) {
        return ReactSharedInternals.H.useMemoCache(size);
      }
    };
    react_production.cache = function(fn) {
      return function() {
        return fn.apply(null, arguments);
      };
    };
    react_production.cacheSignal = function() {
      return null;
    };
    react_production.cloneElement = function(element, config, children) {
      if (null === element || void 0 === element)
        throw Error(
          "The argument must be a React element, but you passed " + element + "."
        );
      var props = assign({}, element.props), key = element.key;
      if (null != config)
        for (propName in void 0 !== config.key && (key = "" + config.key), config)
          !hasOwnProperty.call(config, propName) || "key" === propName || "__self" === propName || "__source" === propName || "ref" === propName && void 0 === config.ref || (props[propName] = config[propName]);
      var propName = arguments.length - 2;
      if (1 === propName) props.children = children;
      else if (1 < propName) {
        for (var childArray = Array(propName), i = 0; i < propName; i++)
          childArray[i] = arguments[i + 2];
        props.children = childArray;
      }
      return ReactElement(element.type, key, props);
    };
    react_production.createContext = function(defaultValue) {
      defaultValue = {
        $$typeof: REACT_CONTEXT_TYPE,
        _currentValue: defaultValue,
        _currentValue2: defaultValue,
        _threadCount: 0,
        Provider: null,
        Consumer: null
      };
      defaultValue.Provider = defaultValue;
      defaultValue.Consumer = {
        $$typeof: REACT_CONSUMER_TYPE,
        _context: defaultValue
      };
      return defaultValue;
    };
    react_production.createElement = function(type, config, children) {
      var propName, props = {}, key = null;
      if (null != config)
        for (propName in void 0 !== config.key && (key = "" + config.key), config)
          hasOwnProperty.call(config, propName) && "key" !== propName && "__self" !== propName && "__source" !== propName && (props[propName] = config[propName]);
      var childrenLength = arguments.length - 2;
      if (1 === childrenLength) props.children = children;
      else if (1 < childrenLength) {
        for (var childArray = Array(childrenLength), i = 0; i < childrenLength; i++)
          childArray[i] = arguments[i + 2];
        props.children = childArray;
      }
      if (type && type.defaultProps)
        for (propName in childrenLength = type.defaultProps, childrenLength)
          void 0 === props[propName] && (props[propName] = childrenLength[propName]);
      return ReactElement(type, key, props);
    };
    react_production.createRef = function() {
      return { current: null };
    };
    react_production.forwardRef = function(render) {
      return { $$typeof: REACT_FORWARD_REF_TYPE, render };
    };
    react_production.isValidElement = isValidElement;
    react_production.lazy = function(ctor) {
      return {
        $$typeof: REACT_LAZY_TYPE,
        _payload: { _status: -1, _result: ctor },
        _init: lazyInitializer
      };
    };
    react_production.memo = function(type, compare) {
      return {
        $$typeof: REACT_MEMO_TYPE,
        type,
        compare: void 0 === compare ? null : compare
      };
    };
    react_production.startTransition = function(scope) {
      var prevTransition = ReactSharedInternals.T, currentTransition = {};
      ReactSharedInternals.T = currentTransition;
      try {
        var returnValue = scope(), onStartTransitionFinish = ReactSharedInternals.S;
        null !== onStartTransitionFinish && onStartTransitionFinish(currentTransition, returnValue);
        "object" === typeof returnValue && null !== returnValue && "function" === typeof returnValue.then && returnValue.then(noop, reportGlobalError);
      } catch (error) {
        reportGlobalError(error);
      } finally {
        null !== prevTransition && null !== currentTransition.types && (prevTransition.types = currentTransition.types), ReactSharedInternals.T = prevTransition;
      }
    };
    react_production.unstable_useCacheRefresh = function() {
      return ReactSharedInternals.H.useCacheRefresh();
    };
    react_production.use = function(usable) {
      return ReactSharedInternals.H.use(usable);
    };
    react_production.useActionState = function(action, initialState, permalink) {
      return ReactSharedInternals.H.useActionState(action, initialState, permalink);
    };
    react_production.useCallback = function(callback, deps) {
      return ReactSharedInternals.H.useCallback(callback, deps);
    };
    react_production.useContext = function(Context) {
      return ReactSharedInternals.H.useContext(Context);
    };
    react_production.useDebugValue = function() {
    };
    react_production.useDeferredValue = function(value, initialValue) {
      return ReactSharedInternals.H.useDeferredValue(value, initialValue);
    };
    react_production.useEffect = function(create, deps) {
      return ReactSharedInternals.H.useEffect(create, deps);
    };
    react_production.useEffectEvent = function(callback) {
      return ReactSharedInternals.H.useEffectEvent(callback);
    };
    react_production.useId = function() {
      return ReactSharedInternals.H.useId();
    };
    react_production.useImperativeHandle = function(ref, create, deps) {
      return ReactSharedInternals.H.useImperativeHandle(ref, create, deps);
    };
    react_production.useInsertionEffect = function(create, deps) {
      return ReactSharedInternals.H.useInsertionEffect(create, deps);
    };
    react_production.useLayoutEffect = function(create, deps) {
      return ReactSharedInternals.H.useLayoutEffect(create, deps);
    };
    react_production.useMemo = function(create, deps) {
      return ReactSharedInternals.H.useMemo(create, deps);
    };
    react_production.useOptimistic = function(passthrough, reducer) {
      return ReactSharedInternals.H.useOptimistic(passthrough, reducer);
    };
    react_production.useReducer = function(reducer, initialArg, init) {
      return ReactSharedInternals.H.useReducer(reducer, initialArg, init);
    };
    react_production.useRef = function(initialValue) {
      return ReactSharedInternals.H.useRef(initialValue);
    };
    react_production.useState = function(initialState) {
      return ReactSharedInternals.H.useState(initialState);
    };
    react_production.useSyncExternalStore = function(subscribe, getSnapshot, getServerSnapshot) {
      return ReactSharedInternals.H.useSyncExternalStore(
        subscribe,
        getSnapshot,
        getServerSnapshot
      );
    };
    react_production.useTransition = function() {
      return ReactSharedInternals.H.useTransition();
    };
    react_production.version = "19.2.0";
    return react_production;
  }
  var hasRequiredReact;
  function requireReact() {
    if (hasRequiredReact) return react.exports;
    hasRequiredReact = 1;
    {
      react.exports = requireReact_production();
    }
    return react.exports;
  }
  var reactDom = { exports: {} };
  var reactDom_production = {};
  /**
   * @license React
   * react-dom.production.js
   *
   * Copyright (c) Meta Platforms, Inc. and affiliates.
   *
   * This source code is licensed under the MIT license found in the
   * LICENSE file in the root directory of this source tree.
   */
  var hasRequiredReactDom_production;
  function requireReactDom_production() {
    if (hasRequiredReactDom_production) return reactDom_production;
    hasRequiredReactDom_production = 1;
    var React = requireReact();
    function formatProdErrorMessage(code) {
      var url = "https://react.dev/errors/" + code;
      if (1 < arguments.length) {
        url += "?args[]=" + encodeURIComponent(arguments[1]);
        for (var i = 2; i < arguments.length; i++)
          url += "&args[]=" + encodeURIComponent(arguments[i]);
      }
      return "Minified React error #" + code + "; visit " + url + " for the full message or use the non-minified dev environment for full errors and additional helpful warnings.";
    }
    function noop() {
    }
    var Internals = {
      d: {
        f: noop,
        r: function() {
          throw Error(formatProdErrorMessage(522));
        },
        D: noop,
        C: noop,
        L: noop,
        m: noop,
        X: noop,
        S: noop,
        M: noop
      },
      p: 0,
      findDOMNode: null
    }, REACT_PORTAL_TYPE = Symbol.for("react.portal");
    function createPortal$1(children, containerInfo, implementation) {
      var key = 3 < arguments.length && void 0 !== arguments[3] ? arguments[3] : null;
      return {
        $$typeof: REACT_PORTAL_TYPE,
        key: null == key ? null : "" + key,
        children,
        containerInfo,
        implementation
      };
    }
    var ReactSharedInternals = React.__CLIENT_INTERNALS_DO_NOT_USE_OR_WARN_USERS_THEY_CANNOT_UPGRADE;
    function getCrossOriginStringAs(as, input) {
      if ("font" === as) return "";
      if ("string" === typeof input)
        return "use-credentials" === input ? input : "";
    }
    reactDom_production.__DOM_INTERNALS_DO_NOT_USE_OR_WARN_USERS_THEY_CANNOT_UPGRADE = Internals;
    reactDom_production.createPortal = function(children, container) {
      var key = 2 < arguments.length && void 0 !== arguments[2] ? arguments[2] : null;
      if (!container || 1 !== container.nodeType && 9 !== container.nodeType && 11 !== container.nodeType)
        throw Error(formatProdErrorMessage(299));
      return createPortal$1(children, container, null, key);
    };
    reactDom_production.flushSync = function(fn) {
      var previousTransition = ReactSharedInternals.T, previousUpdatePriority = Internals.p;
      try {
        if (ReactSharedInternals.T = null, Internals.p = 2, fn) return fn();
      } finally {
        ReactSharedInternals.T = previousTransition, Internals.p = previousUpdatePriority, Internals.d.f();
      }
    };
    reactDom_production.preconnect = function(href, options) {
      "string" === typeof href && (options ? (options = options.crossOrigin, options = "string" === typeof options ? "use-credentials" === options ? options : "" : void 0) : options = null, Internals.d.C(href, options));
    };
    reactDom_production.prefetchDNS = function(href) {
      "string" === typeof href && Internals.d.D(href);
    };
    reactDom_production.preinit = function(href, options) {
      if ("string" === typeof href && options && "string" === typeof options.as) {
        var as = options.as, crossOrigin = getCrossOriginStringAs(as, options.crossOrigin), integrity = "string" === typeof options.integrity ? options.integrity : void 0, fetchPriority = "string" === typeof options.fetchPriority ? options.fetchPriority : void 0;
        "style" === as ? Internals.d.S(
          href,
          "string" === typeof options.precedence ? options.precedence : void 0,
          {
            crossOrigin,
            integrity,
            fetchPriority
          }
        ) : "script" === as && Internals.d.X(href, {
          crossOrigin,
          integrity,
          fetchPriority,
          nonce: "string" === typeof options.nonce ? options.nonce : void 0
        });
      }
    };
    reactDom_production.preinitModule = function(href, options) {
      if ("string" === typeof href)
        if ("object" === typeof options && null !== options) {
          if (null == options.as || "script" === options.as) {
            var crossOrigin = getCrossOriginStringAs(
              options.as,
              options.crossOrigin
            );
            Internals.d.M(href, {
              crossOrigin,
              integrity: "string" === typeof options.integrity ? options.integrity : void 0,
              nonce: "string" === typeof options.nonce ? options.nonce : void 0
            });
          }
        } else null == options && Internals.d.M(href);
    };
    reactDom_production.preload = function(href, options) {
      if ("string" === typeof href && "object" === typeof options && null !== options && "string" === typeof options.as) {
        var as = options.as, crossOrigin = getCrossOriginStringAs(as, options.crossOrigin);
        Internals.d.L(href, as, {
          crossOrigin,
          integrity: "string" === typeof options.integrity ? options.integrity : void 0,
          nonce: "string" === typeof options.nonce ? options.nonce : void 0,
          type: "string" === typeof options.type ? options.type : void 0,
          fetchPriority: "string" === typeof options.fetchPriority ? options.fetchPriority : void 0,
          referrerPolicy: "string" === typeof options.referrerPolicy ? options.referrerPolicy : void 0,
          imageSrcSet: "string" === typeof options.imageSrcSet ? options.imageSrcSet : void 0,
          imageSizes: "string" === typeof options.imageSizes ? options.imageSizes : void 0,
          media: "string" === typeof options.media ? options.media : void 0
        });
      }
    };
    reactDom_production.preloadModule = function(href, options) {
      if ("string" === typeof href)
        if (options) {
          var crossOrigin = getCrossOriginStringAs(options.as, options.crossOrigin);
          Internals.d.m(href, {
            as: "string" === typeof options.as && "script" !== options.as ? options.as : void 0,
            crossOrigin,
            integrity: "string" === typeof options.integrity ? options.integrity : void 0
          });
        } else Internals.d.m(href);
    };
    reactDom_production.requestFormReset = function(form) {
      Internals.d.r(form);
    };
    reactDom_production.unstable_batchedUpdates = function(fn, a) {
      return fn(a);
    };
    reactDom_production.useFormState = function(action, initialState, permalink) {
      return ReactSharedInternals.H.useFormState(action, initialState, permalink);
    };
    reactDom_production.useFormStatus = function() {
      return ReactSharedInternals.H.useHostTransitionStatus();
    };
    reactDom_production.version = "19.2.0";
    return reactDom_production;
  }
  var hasRequiredReactDom;
  function requireReactDom() {
    if (hasRequiredReactDom) return reactDom.exports;
    hasRequiredReactDom = 1;
    function checkDCE() {
      if (typeof __REACT_DEVTOOLS_GLOBAL_HOOK__ === "undefined" || typeof __REACT_DEVTOOLS_GLOBAL_HOOK__.checkDCE !== "function") {
        return;
      }
      try {
        __REACT_DEVTOOLS_GLOBAL_HOOK__.checkDCE(checkDCE);
      } catch (err) {
        console.error(err);
      }
    }
    {
      checkDCE();
      reactDom.exports = requireReactDom_production();
    }
    return reactDom.exports;
  }
  /**
   * @license React
   * react-dom-client.production.js
   *
   * Copyright (c) Meta Platforms, Inc. and affiliates.
   *
   * This source code is licensed under the MIT license found in the
   * LICENSE file in the root directory of this source tree.
   */
  var hasRequiredReactDomClient_production;
  function requireReactDomClient_production() {
    if (hasRequiredReactDomClient_production) return reactDomClient_production;
    hasRequiredReactDomClient_production = 1;
    var Scheduler = requireScheduler(), React = requireReact(), ReactDOM = requireReactDom();
    function formatProdErrorMessage(code) {
      var url = "https://react.dev/errors/" + code;
      if (1 < arguments.length) {
        url += "?args[]=" + encodeURIComponent(arguments[1]);
        for (var i = 2; i < arguments.length; i++)
          url += "&args[]=" + encodeURIComponent(arguments[i]);
      }
      return "Minified React error #" + code + "; visit " + url + " for the full message or use the non-minified dev environment for full errors and additional helpful warnings.";
    }
    function isValidContainer(node) {
      return !(!node || 1 !== node.nodeType && 9 !== node.nodeType && 11 !== node.nodeType);
    }
    function getNearestMountedFiber(fiber) {
      var node = fiber, nearestMounted = fiber;
      if (fiber.alternate) for (; node.return; ) node = node.return;
      else {
        fiber = node;
        do
          node = fiber, 0 !== (node.flags & 4098) && (nearestMounted = node.return), fiber = node.return;
        while (fiber);
      }
      return 3 === node.tag ? nearestMounted : null;
    }
    function getSuspenseInstanceFromFiber(fiber) {
      if (13 === fiber.tag) {
        var suspenseState = fiber.memoizedState;
        null === suspenseState && (fiber = fiber.alternate, null !== fiber && (suspenseState = fiber.memoizedState));
        if (null !== suspenseState) return suspenseState.dehydrated;
      }
      return null;
    }
    function getActivityInstanceFromFiber(fiber) {
      if (31 === fiber.tag) {
        var activityState = fiber.memoizedState;
        null === activityState && (fiber = fiber.alternate, null !== fiber && (activityState = fiber.memoizedState));
        if (null !== activityState) return activityState.dehydrated;
      }
      return null;
    }
    function assertIsMounted(fiber) {
      if (getNearestMountedFiber(fiber) !== fiber)
        throw Error(formatProdErrorMessage(188));
    }
    function findCurrentFiberUsingSlowPath(fiber) {
      var alternate = fiber.alternate;
      if (!alternate) {
        alternate = getNearestMountedFiber(fiber);
        if (null === alternate) throw Error(formatProdErrorMessage(188));
        return alternate !== fiber ? null : fiber;
      }
      for (var a = fiber, b = alternate; ; ) {
        var parentA = a.return;
        if (null === parentA) break;
        var parentB = parentA.alternate;
        if (null === parentB) {
          b = parentA.return;
          if (null !== b) {
            a = b;
            continue;
          }
          break;
        }
        if (parentA.child === parentB.child) {
          for (parentB = parentA.child; parentB; ) {
            if (parentB === a) return assertIsMounted(parentA), fiber;
            if (parentB === b) return assertIsMounted(parentA), alternate;
            parentB = parentB.sibling;
          }
          throw Error(formatProdErrorMessage(188));
        }
        if (a.return !== b.return) a = parentA, b = parentB;
        else {
          for (var didFindChild = false, child$0 = parentA.child; child$0; ) {
            if (child$0 === a) {
              didFindChild = true;
              a = parentA;
              b = parentB;
              break;
            }
            if (child$0 === b) {
              didFindChild = true;
              b = parentA;
              a = parentB;
              break;
            }
            child$0 = child$0.sibling;
          }
          if (!didFindChild) {
            for (child$0 = parentB.child; child$0; ) {
              if (child$0 === a) {
                didFindChild = true;
                a = parentB;
                b = parentA;
                break;
              }
              if (child$0 === b) {
                didFindChild = true;
                b = parentB;
                a = parentA;
                break;
              }
              child$0 = child$0.sibling;
            }
            if (!didFindChild) throw Error(formatProdErrorMessage(189));
          }
        }
        if (a.alternate !== b) throw Error(formatProdErrorMessage(190));
      }
      if (3 !== a.tag) throw Error(formatProdErrorMessage(188));
      return a.stateNode.current === a ? fiber : alternate;
    }
    function findCurrentHostFiberImpl(node) {
      var tag = node.tag;
      if (5 === tag || 26 === tag || 27 === tag || 6 === tag) return node;
      for (node = node.child; null !== node; ) {
        tag = findCurrentHostFiberImpl(node);
        if (null !== tag) return tag;
        node = node.sibling;
      }
      return null;
    }
    var assign = Object.assign, REACT_LEGACY_ELEMENT_TYPE = Symbol.for("react.element"), REACT_ELEMENT_TYPE = Symbol.for("react.transitional.element"), REACT_PORTAL_TYPE = Symbol.for("react.portal"), REACT_FRAGMENT_TYPE = Symbol.for("react.fragment"), REACT_STRICT_MODE_TYPE = Symbol.for("react.strict_mode"), REACT_PROFILER_TYPE = Symbol.for("react.profiler"), REACT_CONSUMER_TYPE = Symbol.for("react.consumer"), REACT_CONTEXT_TYPE = Symbol.for("react.context"), REACT_FORWARD_REF_TYPE = Symbol.for("react.forward_ref"), REACT_SUSPENSE_TYPE = Symbol.for("react.suspense"), REACT_SUSPENSE_LIST_TYPE = Symbol.for("react.suspense_list"), REACT_MEMO_TYPE = Symbol.for("react.memo"), REACT_LAZY_TYPE = Symbol.for("react.lazy");
    var REACT_ACTIVITY_TYPE = Symbol.for("react.activity");
    var REACT_MEMO_CACHE_SENTINEL = Symbol.for("react.memo_cache_sentinel");
    var MAYBE_ITERATOR_SYMBOL = Symbol.iterator;
    function getIteratorFn(maybeIterable) {
      if (null === maybeIterable || "object" !== typeof maybeIterable) return null;
      maybeIterable = MAYBE_ITERATOR_SYMBOL && maybeIterable[MAYBE_ITERATOR_SYMBOL] || maybeIterable["@@iterator"];
      return "function" === typeof maybeIterable ? maybeIterable : null;
    }
    var REACT_CLIENT_REFERENCE = Symbol.for("react.client.reference");
    function getComponentNameFromType(type) {
      if (null == type) return null;
      if ("function" === typeof type)
        return type.$$typeof === REACT_CLIENT_REFERENCE ? null : type.displayName || type.name || null;
      if ("string" === typeof type) return type;
      switch (type) {
        case REACT_FRAGMENT_TYPE:
          return "Fragment";
        case REACT_PROFILER_TYPE:
          return "Profiler";
        case REACT_STRICT_MODE_TYPE:
          return "StrictMode";
        case REACT_SUSPENSE_TYPE:
          return "Suspense";
        case REACT_SUSPENSE_LIST_TYPE:
          return "SuspenseList";
        case REACT_ACTIVITY_TYPE:
          return "Activity";
      }
      if ("object" === typeof type)
        switch (type.$$typeof) {
          case REACT_PORTAL_TYPE:
            return "Portal";
          case REACT_CONTEXT_TYPE:
            return type.displayName || "Context";
          case REACT_CONSUMER_TYPE:
            return (type._context.displayName || "Context") + ".Consumer";
          case REACT_FORWARD_REF_TYPE:
            var innerType = type.render;
            type = type.displayName;
            type || (type = innerType.displayName || innerType.name || "", type = "" !== type ? "ForwardRef(" + type + ")" : "ForwardRef");
            return type;
          case REACT_MEMO_TYPE:
            return innerType = type.displayName || null, null !== innerType ? innerType : getComponentNameFromType(type.type) || "Memo";
          case REACT_LAZY_TYPE:
            innerType = type._payload;
            type = type._init;
            try {
              return getComponentNameFromType(type(innerType));
            } catch (x) {
            }
        }
      return null;
    }
    var isArrayImpl = Array.isArray, ReactSharedInternals = React.__CLIENT_INTERNALS_DO_NOT_USE_OR_WARN_USERS_THEY_CANNOT_UPGRADE, ReactDOMSharedInternals = ReactDOM.__DOM_INTERNALS_DO_NOT_USE_OR_WARN_USERS_THEY_CANNOT_UPGRADE, sharedNotPendingObject = {
      pending: false,
      data: null,
      method: null,
      action: null
    }, valueStack = [], index = -1;
    function createCursor(defaultValue) {
      return { current: defaultValue };
    }
    function pop(cursor) {
      0 > index || (cursor.current = valueStack[index], valueStack[index] = null, index--);
    }
    function push(cursor, value) {
      index++;
      valueStack[index] = cursor.current;
      cursor.current = value;
    }
    var contextStackCursor = createCursor(null), contextFiberStackCursor = createCursor(null), rootInstanceStackCursor = createCursor(null), hostTransitionProviderCursor = createCursor(null);
    function pushHostContainer(fiber, nextRootInstance) {
      push(rootInstanceStackCursor, nextRootInstance);
      push(contextFiberStackCursor, fiber);
      push(contextStackCursor, null);
      switch (nextRootInstance.nodeType) {
        case 9:
        case 11:
          fiber = (fiber = nextRootInstance.documentElement) ? (fiber = fiber.namespaceURI) ? getOwnHostContext(fiber) : 0 : 0;
          break;
        default:
          if (fiber = nextRootInstance.tagName, nextRootInstance = nextRootInstance.namespaceURI)
            nextRootInstance = getOwnHostContext(nextRootInstance), fiber = getChildHostContextProd(nextRootInstance, fiber);
          else
            switch (fiber) {
              case "svg":
                fiber = 1;
                break;
              case "math":
                fiber = 2;
                break;
              default:
                fiber = 0;
            }
      }
      pop(contextStackCursor);
      push(contextStackCursor, fiber);
    }
    function popHostContainer() {
      pop(contextStackCursor);
      pop(contextFiberStackCursor);
      pop(rootInstanceStackCursor);
    }
    function pushHostContext(fiber) {
      null !== fiber.memoizedState && push(hostTransitionProviderCursor, fiber);
      var context = contextStackCursor.current;
      var JSCompiler_inline_result = getChildHostContextProd(context, fiber.type);
      context !== JSCompiler_inline_result && (push(contextFiberStackCursor, fiber), push(contextStackCursor, JSCompiler_inline_result));
    }
    function popHostContext(fiber) {
      contextFiberStackCursor.current === fiber && (pop(contextStackCursor), pop(contextFiberStackCursor));
      hostTransitionProviderCursor.current === fiber && (pop(hostTransitionProviderCursor), HostTransitionContext._currentValue = sharedNotPendingObject);
    }
    var prefix, suffix;
    function describeBuiltInComponentFrame(name) {
      if (void 0 === prefix)
        try {
          throw Error();
        } catch (x) {
          var match = x.stack.trim().match(/\n( *(at )?)/);
          prefix = match && match[1] || "";
          suffix = -1 < x.stack.indexOf("\n    at") ? " (<anonymous>)" : -1 < x.stack.indexOf("@") ? "@unknown:0:0" : "";
        }
      return "\n" + prefix + name + suffix;
    }
    var reentry = false;
    function describeNativeComponentFrame(fn, construct) {
      if (!fn || reentry) return "";
      reentry = true;
      var previousPrepareStackTrace = Error.prepareStackTrace;
      Error.prepareStackTrace = void 0;
      try {
        var RunInRootFrame = {
          DetermineComponentFrameRoot: function() {
            try {
              if (construct) {
                var Fake = function() {
                  throw Error();
                };
                Object.defineProperty(Fake.prototype, "props", {
                  set: function() {
                    throw Error();
                  }
                });
                if ("object" === typeof Reflect && Reflect.construct) {
                  try {
                    Reflect.construct(Fake, []);
                  } catch (x) {
                    var control = x;
                  }
                  Reflect.construct(fn, [], Fake);
                } else {
                  try {
                    Fake.call();
                  } catch (x$1) {
                    control = x$1;
                  }
                  fn.call(Fake.prototype);
                }
              } else {
                try {
                  throw Error();
                } catch (x$2) {
                  control = x$2;
                }
                (Fake = fn()) && "function" === typeof Fake.catch && Fake.catch(function() {
                });
              }
            } catch (sample) {
              if (sample && control && "string" === typeof sample.stack)
                return [sample.stack, control.stack];
            }
            return [null, null];
          }
        };
        RunInRootFrame.DetermineComponentFrameRoot.displayName = "DetermineComponentFrameRoot";
        var namePropDescriptor = Object.getOwnPropertyDescriptor(
          RunInRootFrame.DetermineComponentFrameRoot,
          "name"
        );
        namePropDescriptor && namePropDescriptor.configurable && Object.defineProperty(
          RunInRootFrame.DetermineComponentFrameRoot,
          "name",
          { value: "DetermineComponentFrameRoot" }
        );
        var _RunInRootFrame$Deter = RunInRootFrame.DetermineComponentFrameRoot(), sampleStack = _RunInRootFrame$Deter[0], controlStack = _RunInRootFrame$Deter[1];
        if (sampleStack && controlStack) {
          var sampleLines = sampleStack.split("\n"), controlLines = controlStack.split("\n");
          for (namePropDescriptor = RunInRootFrame = 0; RunInRootFrame < sampleLines.length && !sampleLines[RunInRootFrame].includes("DetermineComponentFrameRoot"); )
            RunInRootFrame++;
          for (; namePropDescriptor < controlLines.length && !controlLines[namePropDescriptor].includes(
            "DetermineComponentFrameRoot"
          ); )
            namePropDescriptor++;
          if (RunInRootFrame === sampleLines.length || namePropDescriptor === controlLines.length)
            for (RunInRootFrame = sampleLines.length - 1, namePropDescriptor = controlLines.length - 1; 1 <= RunInRootFrame && 0 <= namePropDescriptor && sampleLines[RunInRootFrame] !== controlLines[namePropDescriptor]; )
              namePropDescriptor--;
          for (; 1 <= RunInRootFrame && 0 <= namePropDescriptor; RunInRootFrame--, namePropDescriptor--)
            if (sampleLines[RunInRootFrame] !== controlLines[namePropDescriptor]) {
              if (1 !== RunInRootFrame || 1 !== namePropDescriptor) {
                do
                  if (RunInRootFrame--, namePropDescriptor--, 0 > namePropDescriptor || sampleLines[RunInRootFrame] !== controlLines[namePropDescriptor]) {
                    var frame = "\n" + sampleLines[RunInRootFrame].replace(" at new ", " at ");
                    fn.displayName && frame.includes("<anonymous>") && (frame = frame.replace("<anonymous>", fn.displayName));
                    return frame;
                  }
                while (1 <= RunInRootFrame && 0 <= namePropDescriptor);
              }
              break;
            }
        }
      } finally {
        reentry = false, Error.prepareStackTrace = previousPrepareStackTrace;
      }
      return (previousPrepareStackTrace = fn ? fn.displayName || fn.name : "") ? describeBuiltInComponentFrame(previousPrepareStackTrace) : "";
    }
    function describeFiber(fiber, childFiber) {
      switch (fiber.tag) {
        case 26:
        case 27:
        case 5:
          return describeBuiltInComponentFrame(fiber.type);
        case 16:
          return describeBuiltInComponentFrame("Lazy");
        case 13:
          return fiber.child !== childFiber && null !== childFiber ? describeBuiltInComponentFrame("Suspense Fallback") : describeBuiltInComponentFrame("Suspense");
        case 19:
          return describeBuiltInComponentFrame("SuspenseList");
        case 0:
        case 15:
          return describeNativeComponentFrame(fiber.type, false);
        case 11:
          return describeNativeComponentFrame(fiber.type.render, false);
        case 1:
          return describeNativeComponentFrame(fiber.type, true);
        case 31:
          return describeBuiltInComponentFrame("Activity");
        default:
          return "";
      }
    }
    function getStackByFiberInDevAndProd(workInProgress2) {
      try {
        var info = "", previous = null;
        do
          info += describeFiber(workInProgress2, previous), previous = workInProgress2, workInProgress2 = workInProgress2.return;
        while (workInProgress2);
        return info;
      } catch (x) {
        return "\nError generating stack: " + x.message + "\n" + x.stack;
      }
    }
    var hasOwnProperty = Object.prototype.hasOwnProperty, scheduleCallback$3 = Scheduler.unstable_scheduleCallback, cancelCallback$1 = Scheduler.unstable_cancelCallback, shouldYield = Scheduler.unstable_shouldYield, requestPaint = Scheduler.unstable_requestPaint, now = Scheduler.unstable_now, getCurrentPriorityLevel = Scheduler.unstable_getCurrentPriorityLevel, ImmediatePriority = Scheduler.unstable_ImmediatePriority, UserBlockingPriority = Scheduler.unstable_UserBlockingPriority, NormalPriority$1 = Scheduler.unstable_NormalPriority, LowPriority = Scheduler.unstable_LowPriority, IdlePriority = Scheduler.unstable_IdlePriority, log$1 = Scheduler.log, unstable_setDisableYieldValue = Scheduler.unstable_setDisableYieldValue, rendererID = null, injectedHook = null;
    function setIsStrictModeForDevtools(newIsStrictMode) {
      "function" === typeof log$1 && unstable_setDisableYieldValue(newIsStrictMode);
      if (injectedHook && "function" === typeof injectedHook.setStrictMode)
        try {
          injectedHook.setStrictMode(rendererID, newIsStrictMode);
        } catch (err) {
        }
    }
    var clz32 = Math.clz32 ? Math.clz32 : clz32Fallback, log = Math.log, LN2 = Math.LN2;
    function clz32Fallback(x) {
      x >>>= 0;
      return 0 === x ? 32 : 31 - (log(x) / LN2 | 0) | 0;
    }
    var nextTransitionUpdateLane = 256, nextTransitionDeferredLane = 262144, nextRetryLane = 4194304;
    function getHighestPriorityLanes(lanes) {
      var pendingSyncLanes = lanes & 42;
      if (0 !== pendingSyncLanes) return pendingSyncLanes;
      switch (lanes & -lanes) {
        case 1:
          return 1;
        case 2:
          return 2;
        case 4:
          return 4;
        case 8:
          return 8;
        case 16:
          return 16;
        case 32:
          return 32;
        case 64:
          return 64;
        case 128:
          return 128;
        case 256:
        case 512:
        case 1024:
        case 2048:
        case 4096:
        case 8192:
        case 16384:
        case 32768:
        case 65536:
        case 131072:
          return lanes & 261888;
        case 262144:
        case 524288:
        case 1048576:
        case 2097152:
          return lanes & 3932160;
        case 4194304:
        case 8388608:
        case 16777216:
        case 33554432:
          return lanes & 62914560;
        case 67108864:
          return 67108864;
        case 134217728:
          return 134217728;
        case 268435456:
          return 268435456;
        case 536870912:
          return 536870912;
        case 1073741824:
          return 0;
        default:
          return lanes;
      }
    }
    function getNextLanes(root3, wipLanes, rootHasPendingCommit) {
      var pendingLanes = root3.pendingLanes;
      if (0 === pendingLanes) return 0;
      var nextLanes = 0, suspendedLanes = root3.suspendedLanes, pingedLanes = root3.pingedLanes;
      root3 = root3.warmLanes;
      var nonIdlePendingLanes = pendingLanes & 134217727;
      0 !== nonIdlePendingLanes ? (pendingLanes = nonIdlePendingLanes & ~suspendedLanes, 0 !== pendingLanes ? nextLanes = getHighestPriorityLanes(pendingLanes) : (pingedLanes &= nonIdlePendingLanes, 0 !== pingedLanes ? nextLanes = getHighestPriorityLanes(pingedLanes) : rootHasPendingCommit || (rootHasPendingCommit = nonIdlePendingLanes & ~root3, 0 !== rootHasPendingCommit && (nextLanes = getHighestPriorityLanes(rootHasPendingCommit))))) : (nonIdlePendingLanes = pendingLanes & ~suspendedLanes, 0 !== nonIdlePendingLanes ? nextLanes = getHighestPriorityLanes(nonIdlePendingLanes) : 0 !== pingedLanes ? nextLanes = getHighestPriorityLanes(pingedLanes) : rootHasPendingCommit || (rootHasPendingCommit = pendingLanes & ~root3, 0 !== rootHasPendingCommit && (nextLanes = getHighestPriorityLanes(rootHasPendingCommit))));
      return 0 === nextLanes ? 0 : 0 !== wipLanes && wipLanes !== nextLanes && 0 === (wipLanes & suspendedLanes) && (suspendedLanes = nextLanes & -nextLanes, rootHasPendingCommit = wipLanes & -wipLanes, suspendedLanes >= rootHasPendingCommit || 32 === suspendedLanes && 0 !== (rootHasPendingCommit & 4194048)) ? wipLanes : nextLanes;
    }
    function checkIfRootIsPrerendering(root3, renderLanes2) {
      return 0 === (root3.pendingLanes & ~(root3.suspendedLanes & ~root3.pingedLanes) & renderLanes2);
    }
    function computeExpirationTime(lane, currentTime) {
      switch (lane) {
        case 1:
        case 2:
        case 4:
        case 8:
        case 64:
          return currentTime + 250;
        case 16:
        case 32:
        case 128:
        case 256:
        case 512:
        case 1024:
        case 2048:
        case 4096:
        case 8192:
        case 16384:
        case 32768:
        case 65536:
        case 131072:
        case 262144:
        case 524288:
        case 1048576:
        case 2097152:
          return currentTime + 5e3;
        case 4194304:
        case 8388608:
        case 16777216:
        case 33554432:
          return -1;
        case 67108864:
        case 134217728:
        case 268435456:
        case 536870912:
        case 1073741824:
          return -1;
        default:
          return -1;
      }
    }
    function claimNextRetryLane() {
      var lane = nextRetryLane;
      nextRetryLane <<= 1;
      0 === (nextRetryLane & 62914560) && (nextRetryLane = 4194304);
      return lane;
    }
    function createLaneMap(initial) {
      for (var laneMap = [], i = 0; 31 > i; i++) laneMap.push(initial);
      return laneMap;
    }
    function markRootUpdated$1(root3, updateLane) {
      root3.pendingLanes |= updateLane;
      268435456 !== updateLane && (root3.suspendedLanes = 0, root3.pingedLanes = 0, root3.warmLanes = 0);
    }
    function markRootFinished(root3, finishedLanes, remainingLanes, spawnedLane, updatedLanes, suspendedRetryLanes) {
      var previouslyPendingLanes = root3.pendingLanes;
      root3.pendingLanes = remainingLanes;
      root3.suspendedLanes = 0;
      root3.pingedLanes = 0;
      root3.warmLanes = 0;
      root3.expiredLanes &= remainingLanes;
      root3.entangledLanes &= remainingLanes;
      root3.errorRecoveryDisabledLanes &= remainingLanes;
      root3.shellSuspendCounter = 0;
      var entanglements = root3.entanglements, expirationTimes = root3.expirationTimes, hiddenUpdates = root3.hiddenUpdates;
      for (remainingLanes = previouslyPendingLanes & ~remainingLanes; 0 < remainingLanes; ) {
        var index$7 = 31 - clz32(remainingLanes), lane = 1 << index$7;
        entanglements[index$7] = 0;
        expirationTimes[index$7] = -1;
        var hiddenUpdatesForLane = hiddenUpdates[index$7];
        if (null !== hiddenUpdatesForLane)
          for (hiddenUpdates[index$7] = null, index$7 = 0; index$7 < hiddenUpdatesForLane.length; index$7++) {
            var update = hiddenUpdatesForLane[index$7];
            null !== update && (update.lane &= -536870913);
          }
        remainingLanes &= ~lane;
      }
      0 !== spawnedLane && markSpawnedDeferredLane(root3, spawnedLane, 0);
      0 !== suspendedRetryLanes && 0 === updatedLanes && 0 !== root3.tag && (root3.suspendedLanes |= suspendedRetryLanes & ~(previouslyPendingLanes & ~finishedLanes));
    }
    function markSpawnedDeferredLane(root3, spawnedLane, entangledLanes) {
      root3.pendingLanes |= spawnedLane;
      root3.suspendedLanes &= ~spawnedLane;
      var spawnedLaneIndex = 31 - clz32(spawnedLane);
      root3.entangledLanes |= spawnedLane;
      root3.entanglements[spawnedLaneIndex] = root3.entanglements[spawnedLaneIndex] | 1073741824 | entangledLanes & 261930;
    }
    function markRootEntangled(root3, entangledLanes) {
      var rootEntangledLanes = root3.entangledLanes |= entangledLanes;
      for (root3 = root3.entanglements; rootEntangledLanes; ) {
        var index$8 = 31 - clz32(rootEntangledLanes), lane = 1 << index$8;
        lane & entangledLanes | root3[index$8] & entangledLanes && (root3[index$8] |= entangledLanes);
        rootEntangledLanes &= ~lane;
      }
    }
    function getBumpedLaneForHydration(root3, renderLanes2) {
      var renderLane = renderLanes2 & -renderLanes2;
      renderLane = 0 !== (renderLane & 42) ? 1 : getBumpedLaneForHydrationByLane(renderLane);
      return 0 !== (renderLane & (root3.suspendedLanes | renderLanes2)) ? 0 : renderLane;
    }
    function getBumpedLaneForHydrationByLane(lane) {
      switch (lane) {
        case 2:
          lane = 1;
          break;
        case 8:
          lane = 4;
          break;
        case 32:
          lane = 16;
          break;
        case 256:
        case 512:
        case 1024:
        case 2048:
        case 4096:
        case 8192:
        case 16384:
        case 32768:
        case 65536:
        case 131072:
        case 262144:
        case 524288:
        case 1048576:
        case 2097152:
        case 4194304:
        case 8388608:
        case 16777216:
        case 33554432:
          lane = 128;
          break;
        case 268435456:
          lane = 134217728;
          break;
        default:
          lane = 0;
      }
      return lane;
    }
    function lanesToEventPriority(lanes) {
      lanes &= -lanes;
      return 2 < lanes ? 8 < lanes ? 0 !== (lanes & 134217727) ? 32 : 268435456 : 8 : 2;
    }
    function resolveUpdatePriority() {
      var updatePriority = ReactDOMSharedInternals.p;
      if (0 !== updatePriority) return updatePriority;
      updatePriority = window.event;
      return void 0 === updatePriority ? 32 : getEventPriority(updatePriority.type);
    }
    function runWithPriority(priority, fn) {
      var previousPriority = ReactDOMSharedInternals.p;
      try {
        return ReactDOMSharedInternals.p = priority, fn();
      } finally {
        ReactDOMSharedInternals.p = previousPriority;
      }
    }
    var randomKey = Math.random().toString(36).slice(2), internalInstanceKey = "__reactFiber$" + randomKey, internalPropsKey = "__reactProps$" + randomKey, internalContainerInstanceKey = "__reactContainer$" + randomKey, internalEventHandlersKey = "__reactEvents$" + randomKey, internalEventHandlerListenersKey = "__reactListeners$" + randomKey, internalEventHandlesSetKey = "__reactHandles$" + randomKey, internalRootNodeResourcesKey = "__reactResources$" + randomKey, internalHoistableMarker = "__reactMarker$" + randomKey;
    function detachDeletedInstance(node) {
      delete node[internalInstanceKey];
      delete node[internalPropsKey];
      delete node[internalEventHandlersKey];
      delete node[internalEventHandlerListenersKey];
      delete node[internalEventHandlesSetKey];
    }
    function getClosestInstanceFromNode(targetNode) {
      var targetInst = targetNode[internalInstanceKey];
      if (targetInst) return targetInst;
      for (var parentNode = targetNode.parentNode; parentNode; ) {
        if (targetInst = parentNode[internalContainerInstanceKey] || parentNode[internalInstanceKey]) {
          parentNode = targetInst.alternate;
          if (null !== targetInst.child || null !== parentNode && null !== parentNode.child)
            for (targetNode = getParentHydrationBoundary(targetNode); null !== targetNode; ) {
              if (parentNode = targetNode[internalInstanceKey]) return parentNode;
              targetNode = getParentHydrationBoundary(targetNode);
            }
          return targetInst;
        }
        targetNode = parentNode;
        parentNode = targetNode.parentNode;
      }
      return null;
    }
    function getInstanceFromNode(node) {
      if (node = node[internalInstanceKey] || node[internalContainerInstanceKey]) {
        var tag = node.tag;
        if (5 === tag || 6 === tag || 13 === tag || 31 === tag || 26 === tag || 27 === tag || 3 === tag)
          return node;
      }
      return null;
    }
    function getNodeFromInstance(inst) {
      var tag = inst.tag;
      if (5 === tag || 26 === tag || 27 === tag || 6 === tag) return inst.stateNode;
      throw Error(formatProdErrorMessage(33));
    }
    function getResourcesFromRoot(root3) {
      var resources = root3[internalRootNodeResourcesKey];
      resources || (resources = root3[internalRootNodeResourcesKey] = { hoistableStyles: /* @__PURE__ */ new Map(), hoistableScripts: /* @__PURE__ */ new Map() });
      return resources;
    }
    function markNodeAsHoistable(node) {
      node[internalHoistableMarker] = true;
    }
    var allNativeEvents = /* @__PURE__ */ new Set(), registrationNameDependencies = {};
    function registerTwoPhaseEvent(registrationName, dependencies) {
      registerDirectEvent(registrationName, dependencies);
      registerDirectEvent(registrationName + "Capture", dependencies);
    }
    function registerDirectEvent(registrationName, dependencies) {
      registrationNameDependencies[registrationName] = dependencies;
      for (registrationName = 0; registrationName < dependencies.length; registrationName++)
        allNativeEvents.add(dependencies[registrationName]);
    }
    var VALID_ATTRIBUTE_NAME_REGEX = RegExp(
      "^[:A-Z_a-z\\u00C0-\\u00D6\\u00D8-\\u00F6\\u00F8-\\u02FF\\u0370-\\u037D\\u037F-\\u1FFF\\u200C-\\u200D\\u2070-\\u218F\\u2C00-\\u2FEF\\u3001-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFFD][:A-Z_a-z\\u00C0-\\u00D6\\u00D8-\\u00F6\\u00F8-\\u02FF\\u0370-\\u037D\\u037F-\\u1FFF\\u200C-\\u200D\\u2070-\\u218F\\u2C00-\\u2FEF\\u3001-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFFD\\-.0-9\\u00B7\\u0300-\\u036F\\u203F-\\u2040]*$"
    ), illegalAttributeNameCache = {}, validatedAttributeNameCache = {};
    function isAttributeNameSafe(attributeName) {
      if (hasOwnProperty.call(validatedAttributeNameCache, attributeName))
        return true;
      if (hasOwnProperty.call(illegalAttributeNameCache, attributeName)) return false;
      if (VALID_ATTRIBUTE_NAME_REGEX.test(attributeName))
        return validatedAttributeNameCache[attributeName] = true;
      illegalAttributeNameCache[attributeName] = true;
      return false;
    }
    function setValueForAttribute(node, name, value) {
      if (isAttributeNameSafe(name))
        if (null === value) node.removeAttribute(name);
        else {
          switch (typeof value) {
            case "undefined":
            case "function":
            case "symbol":
              node.removeAttribute(name);
              return;
            case "boolean":
              var prefix$10 = name.toLowerCase().slice(0, 5);
              if ("data-" !== prefix$10 && "aria-" !== prefix$10) {
                node.removeAttribute(name);
                return;
              }
          }
          node.setAttribute(name, "" + value);
        }
    }
    function setValueForKnownAttribute(node, name, value) {
      if (null === value) node.removeAttribute(name);
      else {
        switch (typeof value) {
          case "undefined":
          case "function":
          case "symbol":
          case "boolean":
            node.removeAttribute(name);
            return;
        }
        node.setAttribute(name, "" + value);
      }
    }
    function setValueForNamespacedAttribute(node, namespace, name, value) {
      if (null === value) node.removeAttribute(name);
      else {
        switch (typeof value) {
          case "undefined":
          case "function":
          case "symbol":
          case "boolean":
            node.removeAttribute(name);
            return;
        }
        node.setAttributeNS(namespace, name, "" + value);
      }
    }
    function getToStringValue(value) {
      switch (typeof value) {
        case "bigint":
        case "boolean":
        case "number":
        case "string":
        case "undefined":
          return value;
        case "object":
          return value;
        default:
          return "";
      }
    }
    function isCheckable(elem) {
      var type = elem.type;
      return (elem = elem.nodeName) && "input" === elem.toLowerCase() && ("checkbox" === type || "radio" === type);
    }
    function trackValueOnNode(node, valueField, currentValue) {
      var descriptor = Object.getOwnPropertyDescriptor(
        node.constructor.prototype,
        valueField
      );
      if (!node.hasOwnProperty(valueField) && "undefined" !== typeof descriptor && "function" === typeof descriptor.get && "function" === typeof descriptor.set) {
        var get = descriptor.get, set = descriptor.set;
        Object.defineProperty(node, valueField, {
          configurable: true,
          get: function() {
            return get.call(this);
          },
          set: function(value) {
            currentValue = "" + value;
            set.call(this, value);
          }
        });
        Object.defineProperty(node, valueField, {
          enumerable: descriptor.enumerable
        });
        return {
          getValue: function() {
            return currentValue;
          },
          setValue: function(value) {
            currentValue = "" + value;
          },
          stopTracking: function() {
            node._valueTracker = null;
            delete node[valueField];
          }
        };
      }
    }
    function track(node) {
      if (!node._valueTracker) {
        var valueField = isCheckable(node) ? "checked" : "value";
        node._valueTracker = trackValueOnNode(
          node,
          valueField,
          "" + node[valueField]
        );
      }
    }
    function updateValueIfChanged(node) {
      if (!node) return false;
      var tracker = node._valueTracker;
      if (!tracker) return true;
      var lastValue = tracker.getValue();
      var value = "";
      node && (value = isCheckable(node) ? node.checked ? "true" : "false" : node.value);
      node = value;
      return node !== lastValue ? (tracker.setValue(node), true) : false;
    }
    function getActiveElement(doc) {
      doc = doc || ("undefined" !== typeof document ? document : void 0);
      if ("undefined" === typeof doc) return null;
      try {
        return doc.activeElement || doc.body;
      } catch (e) {
        return doc.body;
      }
    }
    var escapeSelectorAttributeValueInsideDoubleQuotesRegex = /[\n"\\]/g;
    function escapeSelectorAttributeValueInsideDoubleQuotes(value) {
      return value.replace(
        escapeSelectorAttributeValueInsideDoubleQuotesRegex,
        function(ch) {
          return "\\" + ch.charCodeAt(0).toString(16) + " ";
        }
      );
    }
    function updateInput(element, value, defaultValue, lastDefaultValue, checked, defaultChecked, type, name) {
      element.name = "";
      null != type && "function" !== typeof type && "symbol" !== typeof type && "boolean" !== typeof type ? element.type = type : element.removeAttribute("type");
      if (null != value)
        if ("number" === type) {
          if (0 === value && "" === element.value || element.value != value)
            element.value = "" + getToStringValue(value);
        } else
          element.value !== "" + getToStringValue(value) && (element.value = "" + getToStringValue(value));
      else
        "submit" !== type && "reset" !== type || element.removeAttribute("value");
      null != value ? setDefaultValue(element, type, getToStringValue(value)) : null != defaultValue ? setDefaultValue(element, type, getToStringValue(defaultValue)) : null != lastDefaultValue && element.removeAttribute("value");
      null == checked && null != defaultChecked && (element.defaultChecked = !!defaultChecked);
      null != checked && (element.checked = checked && "function" !== typeof checked && "symbol" !== typeof checked);
      null != name && "function" !== typeof name && "symbol" !== typeof name && "boolean" !== typeof name ? element.name = "" + getToStringValue(name) : element.removeAttribute("name");
    }
    function initInput(element, value, defaultValue, checked, defaultChecked, type, name, isHydrating2) {
      null != type && "function" !== typeof type && "symbol" !== typeof type && "boolean" !== typeof type && (element.type = type);
      if (null != value || null != defaultValue) {
        if (!("submit" !== type && "reset" !== type || void 0 !== value && null !== value)) {
          track(element);
          return;
        }
        defaultValue = null != defaultValue ? "" + getToStringValue(defaultValue) : "";
        value = null != value ? "" + getToStringValue(value) : defaultValue;
        isHydrating2 || value === element.value || (element.value = value);
        element.defaultValue = value;
      }
      checked = null != checked ? checked : defaultChecked;
      checked = "function" !== typeof checked && "symbol" !== typeof checked && !!checked;
      element.checked = isHydrating2 ? element.checked : !!checked;
      element.defaultChecked = !!checked;
      null != name && "function" !== typeof name && "symbol" !== typeof name && "boolean" !== typeof name && (element.name = name);
      track(element);
    }
    function setDefaultValue(node, type, value) {
      "number" === type && getActiveElement(node.ownerDocument) === node || node.defaultValue === "" + value || (node.defaultValue = "" + value);
    }
    function updateOptions(node, multiple, propValue, setDefaultSelected) {
      node = node.options;
      if (multiple) {
        multiple = {};
        for (var i = 0; i < propValue.length; i++)
          multiple["$" + propValue[i]] = true;
        for (propValue = 0; propValue < node.length; propValue++)
          i = multiple.hasOwnProperty("$" + node[propValue].value), node[propValue].selected !== i && (node[propValue].selected = i), i && setDefaultSelected && (node[propValue].defaultSelected = true);
      } else {
        propValue = "" + getToStringValue(propValue);
        multiple = null;
        for (i = 0; i < node.length; i++) {
          if (node[i].value === propValue) {
            node[i].selected = true;
            setDefaultSelected && (node[i].defaultSelected = true);
            return;
          }
          null !== multiple || node[i].disabled || (multiple = node[i]);
        }
        null !== multiple && (multiple.selected = true);
      }
    }
    function updateTextarea(element, value, defaultValue) {
      if (null != value && (value = "" + getToStringValue(value), value !== element.value && (element.value = value), null == defaultValue)) {
        element.defaultValue !== value && (element.defaultValue = value);
        return;
      }
      element.defaultValue = null != defaultValue ? "" + getToStringValue(defaultValue) : "";
    }
    function initTextarea(element, value, defaultValue, children) {
      if (null == value) {
        if (null != children) {
          if (null != defaultValue) throw Error(formatProdErrorMessage(92));
          if (isArrayImpl(children)) {
            if (1 < children.length) throw Error(formatProdErrorMessage(93));
            children = children[0];
          }
          defaultValue = children;
        }
        null == defaultValue && (defaultValue = "");
        value = defaultValue;
      }
      defaultValue = getToStringValue(value);
      element.defaultValue = defaultValue;
      children = element.textContent;
      children === defaultValue && "" !== children && null !== children && (element.value = children);
      track(element);
    }
    function setTextContent(node, text) {
      if (text) {
        var firstChild = node.firstChild;
        if (firstChild && firstChild === node.lastChild && 3 === firstChild.nodeType) {
          firstChild.nodeValue = text;
          return;
        }
      }
      node.textContent = text;
    }
    var unitlessNumbers = new Set(
      "animationIterationCount aspectRatio borderImageOutset borderImageSlice borderImageWidth boxFlex boxFlexGroup boxOrdinalGroup columnCount columns flex flexGrow flexPositive flexShrink flexNegative flexOrder gridArea gridRow gridRowEnd gridRowSpan gridRowStart gridColumn gridColumnEnd gridColumnSpan gridColumnStart fontWeight lineClamp lineHeight opacity order orphans scale tabSize widows zIndex zoom fillOpacity floodOpacity stopOpacity strokeDasharray strokeDashoffset strokeMiterlimit strokeOpacity strokeWidth MozAnimationIterationCount MozBoxFlex MozBoxFlexGroup MozLineClamp msAnimationIterationCount msFlex msZoom msFlexGrow msFlexNegative msFlexOrder msFlexPositive msFlexShrink msGridColumn msGridColumnSpan msGridRow msGridRowSpan WebkitAnimationIterationCount WebkitBoxFlex WebKitBoxFlexGroup WebkitBoxOrdinalGroup WebkitColumnCount WebkitColumns WebkitFlex WebkitFlexGrow WebkitFlexPositive WebkitFlexShrink WebkitLineClamp".split(
        " "
      )
    );
    function setValueForStyle(style2, styleName, value) {
      var isCustomProperty = 0 === styleName.indexOf("--");
      null == value || "boolean" === typeof value || "" === value ? isCustomProperty ? style2.setProperty(styleName, "") : "float" === styleName ? style2.cssFloat = "" : style2[styleName] = "" : isCustomProperty ? style2.setProperty(styleName, value) : "number" !== typeof value || 0 === value || unitlessNumbers.has(styleName) ? "float" === styleName ? style2.cssFloat = value : style2[styleName] = ("" + value).trim() : style2[styleName] = value + "px";
    }
    function setValueForStyles(node, styles, prevStyles) {
      if (null != styles && "object" !== typeof styles)
        throw Error(formatProdErrorMessage(62));
      node = node.style;
      if (null != prevStyles) {
        for (var styleName in prevStyles)
          !prevStyles.hasOwnProperty(styleName) || null != styles && styles.hasOwnProperty(styleName) || (0 === styleName.indexOf("--") ? node.setProperty(styleName, "") : "float" === styleName ? node.cssFloat = "" : node[styleName] = "");
        for (var styleName$16 in styles)
          styleName = styles[styleName$16], styles.hasOwnProperty(styleName$16) && prevStyles[styleName$16] !== styleName && setValueForStyle(node, styleName$16, styleName);
      } else
        for (var styleName$17 in styles)
          styles.hasOwnProperty(styleName$17) && setValueForStyle(node, styleName$17, styles[styleName$17]);
    }
    function isCustomElement(tagName) {
      if (-1 === tagName.indexOf("-")) return false;
      switch (tagName) {
        case "annotation-xml":
        case "color-profile":
        case "font-face":
        case "font-face-src":
        case "font-face-uri":
        case "font-face-format":
        case "font-face-name":
        case "missing-glyph":
          return false;
        default:
          return true;
      }
    }
    var aliases = /* @__PURE__ */ new Map([
      ["acceptCharset", "accept-charset"],
      ["htmlFor", "for"],
      ["httpEquiv", "http-equiv"],
      ["crossOrigin", "crossorigin"],
      ["accentHeight", "accent-height"],
      ["alignmentBaseline", "alignment-baseline"],
      ["arabicForm", "arabic-form"],
      ["baselineShift", "baseline-shift"],
      ["capHeight", "cap-height"],
      ["clipPath", "clip-path"],
      ["clipRule", "clip-rule"],
      ["colorInterpolation", "color-interpolation"],
      ["colorInterpolationFilters", "color-interpolation-filters"],
      ["colorProfile", "color-profile"],
      ["colorRendering", "color-rendering"],
      ["dominantBaseline", "dominant-baseline"],
      ["enableBackground", "enable-background"],
      ["fillOpacity", "fill-opacity"],
      ["fillRule", "fill-rule"],
      ["floodColor", "flood-color"],
      ["floodOpacity", "flood-opacity"],
      ["fontFamily", "font-family"],
      ["fontSize", "font-size"],
      ["fontSizeAdjust", "font-size-adjust"],
      ["fontStretch", "font-stretch"],
      ["fontStyle", "font-style"],
      ["fontVariant", "font-variant"],
      ["fontWeight", "font-weight"],
      ["glyphName", "glyph-name"],
      ["glyphOrientationHorizontal", "glyph-orientation-horizontal"],
      ["glyphOrientationVertical", "glyph-orientation-vertical"],
      ["horizAdvX", "horiz-adv-x"],
      ["horizOriginX", "horiz-origin-x"],
      ["imageRendering", "image-rendering"],
      ["letterSpacing", "letter-spacing"],
      ["lightingColor", "lighting-color"],
      ["markerEnd", "marker-end"],
      ["markerMid", "marker-mid"],
      ["markerStart", "marker-start"],
      ["overlinePosition", "overline-position"],
      ["overlineThickness", "overline-thickness"],
      ["paintOrder", "paint-order"],
      ["panose-1", "panose-1"],
      ["pointerEvents", "pointer-events"],
      ["renderingIntent", "rendering-intent"],
      ["shapeRendering", "shape-rendering"],
      ["stopColor", "stop-color"],
      ["stopOpacity", "stop-opacity"],
      ["strikethroughPosition", "strikethrough-position"],
      ["strikethroughThickness", "strikethrough-thickness"],
      ["strokeDasharray", "stroke-dasharray"],
      ["strokeDashoffset", "stroke-dashoffset"],
      ["strokeLinecap", "stroke-linecap"],
      ["strokeLinejoin", "stroke-linejoin"],
      ["strokeMiterlimit", "stroke-miterlimit"],
      ["strokeOpacity", "stroke-opacity"],
      ["strokeWidth", "stroke-width"],
      ["textAnchor", "text-anchor"],
      ["textDecoration", "text-decoration"],
      ["textRendering", "text-rendering"],
      ["transformOrigin", "transform-origin"],
      ["underlinePosition", "underline-position"],
      ["underlineThickness", "underline-thickness"],
      ["unicodeBidi", "unicode-bidi"],
      ["unicodeRange", "unicode-range"],
      ["unitsPerEm", "units-per-em"],
      ["vAlphabetic", "v-alphabetic"],
      ["vHanging", "v-hanging"],
      ["vIdeographic", "v-ideographic"],
      ["vMathematical", "v-mathematical"],
      ["vectorEffect", "vector-effect"],
      ["vertAdvY", "vert-adv-y"],
      ["vertOriginX", "vert-origin-x"],
      ["vertOriginY", "vert-origin-y"],
      ["wordSpacing", "word-spacing"],
      ["writingMode", "writing-mode"],
      ["xmlnsXlink", "xmlns:xlink"],
      ["xHeight", "x-height"]
    ]), isJavaScriptProtocol = /^[\u0000-\u001F ]*j[\r\n\t]*a[\r\n\t]*v[\r\n\t]*a[\r\n\t]*s[\r\n\t]*c[\r\n\t]*r[\r\n\t]*i[\r\n\t]*p[\r\n\t]*t[\r\n\t]*:/i;
    function sanitizeURL(url) {
      return isJavaScriptProtocol.test("" + url) ? "javascript:throw new Error('React has blocked a javascript: URL as a security precaution.')" : url;
    }
    function noop$1() {
    }
    var currentReplayingEvent = null;
    function getEventTarget(nativeEvent) {
      nativeEvent = nativeEvent.target || nativeEvent.srcElement || window;
      nativeEvent.correspondingUseElement && (nativeEvent = nativeEvent.correspondingUseElement);
      return 3 === nativeEvent.nodeType ? nativeEvent.parentNode : nativeEvent;
    }
    var restoreTarget = null, restoreQueue = null;
    function restoreStateOfTarget(target) {
      var internalInstance = getInstanceFromNode(target);
      if (internalInstance && (target = internalInstance.stateNode)) {
        var props = target[internalPropsKey] || null;
        a: switch (target = internalInstance.stateNode, internalInstance.type) {
          case "input":
            updateInput(
              target,
              props.value,
              props.defaultValue,
              props.defaultValue,
              props.checked,
              props.defaultChecked,
              props.type,
              props.name
            );
            internalInstance = props.name;
            if ("radio" === props.type && null != internalInstance) {
              for (props = target; props.parentNode; ) props = props.parentNode;
              props = props.querySelectorAll(
                'input[name="' + escapeSelectorAttributeValueInsideDoubleQuotes(
                  "" + internalInstance
                ) + '"][type="radio"]'
              );
              for (internalInstance = 0; internalInstance < props.length; internalInstance++) {
                var otherNode = props[internalInstance];
                if (otherNode !== target && otherNode.form === target.form) {
                  var otherProps = otherNode[internalPropsKey] || null;
                  if (!otherProps) throw Error(formatProdErrorMessage(90));
                  updateInput(
                    otherNode,
                    otherProps.value,
                    otherProps.defaultValue,
                    otherProps.defaultValue,
                    otherProps.checked,
                    otherProps.defaultChecked,
                    otherProps.type,
                    otherProps.name
                  );
                }
              }
              for (internalInstance = 0; internalInstance < props.length; internalInstance++)
                otherNode = props[internalInstance], otherNode.form === target.form && updateValueIfChanged(otherNode);
            }
            break a;
          case "textarea":
            updateTextarea(target, props.value, props.defaultValue);
            break a;
          case "select":
            internalInstance = props.value, null != internalInstance && updateOptions(target, !!props.multiple, internalInstance, false);
        }
      }
    }
    var isInsideEventHandler = false;
    function batchedUpdates$1(fn, a, b) {
      if (isInsideEventHandler) return fn(a, b);
      isInsideEventHandler = true;
      try {
        var JSCompiler_inline_result = fn(a);
        return JSCompiler_inline_result;
      } finally {
        if (isInsideEventHandler = false, null !== restoreTarget || null !== restoreQueue) {
          if (flushSyncWork$1(), restoreTarget && (a = restoreTarget, fn = restoreQueue, restoreQueue = restoreTarget = null, restoreStateOfTarget(a), fn))
            for (a = 0; a < fn.length; a++) restoreStateOfTarget(fn[a]);
        }
      }
    }
    function getListener(inst, registrationName) {
      var stateNode = inst.stateNode;
      if (null === stateNode) return null;
      var props = stateNode[internalPropsKey] || null;
      if (null === props) return null;
      stateNode = props[registrationName];
      a: switch (registrationName) {
        case "onClick":
        case "onClickCapture":
        case "onDoubleClick":
        case "onDoubleClickCapture":
        case "onMouseDown":
        case "onMouseDownCapture":
        case "onMouseMove":
        case "onMouseMoveCapture":
        case "onMouseUp":
        case "onMouseUpCapture":
        case "onMouseEnter":
          (props = !props.disabled) || (inst = inst.type, props = !("button" === inst || "input" === inst || "select" === inst || "textarea" === inst));
          inst = !props;
          break a;
        default:
          inst = false;
      }
      if (inst) return null;
      if (stateNode && "function" !== typeof stateNode)
        throw Error(
          formatProdErrorMessage(231, registrationName, typeof stateNode)
        );
      return stateNode;
    }
    var canUseDOM = !("undefined" === typeof window || "undefined" === typeof window.document || "undefined" === typeof window.document.createElement), passiveBrowserEventsSupported = false;
    if (canUseDOM)
      try {
        var options = {};
        Object.defineProperty(options, "passive", {
          get: function() {
            passiveBrowserEventsSupported = true;
          }
        });
        window.addEventListener("test", options, options);
        window.removeEventListener("test", options, options);
      } catch (e) {
        passiveBrowserEventsSupported = false;
      }
    var root2 = null, startText = null, fallbackText = null;
    function getData() {
      if (fallbackText) return fallbackText;
      var start, startValue = startText, startLength = startValue.length, end, endValue = "value" in root2 ? root2.value : root2.textContent, endLength = endValue.length;
      for (start = 0; start < startLength && startValue[start] === endValue[start]; start++) ;
      var minEnd = startLength - start;
      for (end = 1; end <= minEnd && startValue[startLength - end] === endValue[endLength - end]; end++) ;
      return fallbackText = endValue.slice(start, 1 < end ? 1 - end : void 0);
    }
    function getEventCharCode(nativeEvent) {
      var keyCode = nativeEvent.keyCode;
      "charCode" in nativeEvent ? (nativeEvent = nativeEvent.charCode, 0 === nativeEvent && 13 === keyCode && (nativeEvent = 13)) : nativeEvent = keyCode;
      10 === nativeEvent && (nativeEvent = 13);
      return 32 <= nativeEvent || 13 === nativeEvent ? nativeEvent : 0;
    }
    function functionThatReturnsTrue() {
      return true;
    }
    function functionThatReturnsFalse() {
      return false;
    }
    function createSyntheticEvent(Interface) {
      function SyntheticBaseEvent(reactName, reactEventType, targetInst, nativeEvent, nativeEventTarget) {
        this._reactName = reactName;
        this._targetInst = targetInst;
        this.type = reactEventType;
        this.nativeEvent = nativeEvent;
        this.target = nativeEventTarget;
        this.currentTarget = null;
        for (var propName in Interface)
          Interface.hasOwnProperty(propName) && (reactName = Interface[propName], this[propName] = reactName ? reactName(nativeEvent) : nativeEvent[propName]);
        this.isDefaultPrevented = (null != nativeEvent.defaultPrevented ? nativeEvent.defaultPrevented : false === nativeEvent.returnValue) ? functionThatReturnsTrue : functionThatReturnsFalse;
        this.isPropagationStopped = functionThatReturnsFalse;
        return this;
      }
      assign(SyntheticBaseEvent.prototype, {
        preventDefault: function() {
          this.defaultPrevented = true;
          var event = this.nativeEvent;
          event && (event.preventDefault ? event.preventDefault() : "unknown" !== typeof event.returnValue && (event.returnValue = false), this.isDefaultPrevented = functionThatReturnsTrue);
        },
        stopPropagation: function() {
          var event = this.nativeEvent;
          event && (event.stopPropagation ? event.stopPropagation() : "unknown" !== typeof event.cancelBubble && (event.cancelBubble = true), this.isPropagationStopped = functionThatReturnsTrue);
        },
        persist: function() {
        },
        isPersistent: functionThatReturnsTrue
      });
      return SyntheticBaseEvent;
    }
    var EventInterface = {
      eventPhase: 0,
      bubbles: 0,
      cancelable: 0,
      timeStamp: function(event) {
        return event.timeStamp || Date.now();
      },
      defaultPrevented: 0,
      isTrusted: 0
    }, SyntheticEvent = createSyntheticEvent(EventInterface), UIEventInterface = assign({}, EventInterface, { view: 0, detail: 0 }), SyntheticUIEvent = createSyntheticEvent(UIEventInterface), lastMovementX, lastMovementY, lastMouseEvent, MouseEventInterface = assign({}, UIEventInterface, {
      screenX: 0,
      screenY: 0,
      clientX: 0,
      clientY: 0,
      pageX: 0,
      pageY: 0,
      ctrlKey: 0,
      shiftKey: 0,
      altKey: 0,
      metaKey: 0,
      getModifierState: getEventModifierState,
      button: 0,
      buttons: 0,
      relatedTarget: function(event) {
        return void 0 === event.relatedTarget ? event.fromElement === event.srcElement ? event.toElement : event.fromElement : event.relatedTarget;
      },
      movementX: function(event) {
        if ("movementX" in event) return event.movementX;
        event !== lastMouseEvent && (lastMouseEvent && "mousemove" === event.type ? (lastMovementX = event.screenX - lastMouseEvent.screenX, lastMovementY = event.screenY - lastMouseEvent.screenY) : lastMovementY = lastMovementX = 0, lastMouseEvent = event);
        return lastMovementX;
      },
      movementY: function(event) {
        return "movementY" in event ? event.movementY : lastMovementY;
      }
    }), SyntheticMouseEvent = createSyntheticEvent(MouseEventInterface), DragEventInterface = assign({}, MouseEventInterface, { dataTransfer: 0 }), SyntheticDragEvent = createSyntheticEvent(DragEventInterface), FocusEventInterface = assign({}, UIEventInterface, { relatedTarget: 0 }), SyntheticFocusEvent = createSyntheticEvent(FocusEventInterface), AnimationEventInterface = assign({}, EventInterface, {
      animationName: 0,
      elapsedTime: 0,
      pseudoElement: 0
    }), SyntheticAnimationEvent = createSyntheticEvent(AnimationEventInterface), ClipboardEventInterface = assign({}, EventInterface, {
      clipboardData: function(event) {
        return "clipboardData" in event ? event.clipboardData : window.clipboardData;
      }
    }), SyntheticClipboardEvent = createSyntheticEvent(ClipboardEventInterface), CompositionEventInterface = assign({}, EventInterface, { data: 0 }), SyntheticCompositionEvent = createSyntheticEvent(CompositionEventInterface), normalizeKey = {
      Esc: "Escape",
      Spacebar: " ",
      Left: "ArrowLeft",
      Up: "ArrowUp",
      Right: "ArrowRight",
      Down: "ArrowDown",
      Del: "Delete",
      Win: "OS",
      Menu: "ContextMenu",
      Apps: "ContextMenu",
      Scroll: "ScrollLock",
      MozPrintableKey: "Unidentified"
    }, translateToKey = {
      8: "Backspace",
      9: "Tab",
      12: "Clear",
      13: "Enter",
      16: "Shift",
      17: "Control",
      18: "Alt",
      19: "Pause",
      20: "CapsLock",
      27: "Escape",
      32: " ",
      33: "PageUp",
      34: "PageDown",
      35: "End",
      36: "Home",
      37: "ArrowLeft",
      38: "ArrowUp",
      39: "ArrowRight",
      40: "ArrowDown",
      45: "Insert",
      46: "Delete",
      112: "F1",
      113: "F2",
      114: "F3",
      115: "F4",
      116: "F5",
      117: "F6",
      118: "F7",
      119: "F8",
      120: "F9",
      121: "F10",
      122: "F11",
      123: "F12",
      144: "NumLock",
      145: "ScrollLock",
      224: "Meta"
    }, modifierKeyToProp = {
      Alt: "altKey",
      Control: "ctrlKey",
      Meta: "metaKey",
      Shift: "shiftKey"
    };
    function modifierStateGetter(keyArg) {
      var nativeEvent = this.nativeEvent;
      return nativeEvent.getModifierState ? nativeEvent.getModifierState(keyArg) : (keyArg = modifierKeyToProp[keyArg]) ? !!nativeEvent[keyArg] : false;
    }
    function getEventModifierState() {
      return modifierStateGetter;
    }
    var KeyboardEventInterface = assign({}, UIEventInterface, {
      key: function(nativeEvent) {
        if (nativeEvent.key) {
          var key = normalizeKey[nativeEvent.key] || nativeEvent.key;
          if ("Unidentified" !== key) return key;
        }
        return "keypress" === nativeEvent.type ? (nativeEvent = getEventCharCode(nativeEvent), 13 === nativeEvent ? "Enter" : String.fromCharCode(nativeEvent)) : "keydown" === nativeEvent.type || "keyup" === nativeEvent.type ? translateToKey[nativeEvent.keyCode] || "Unidentified" : "";
      },
      code: 0,
      location: 0,
      ctrlKey: 0,
      shiftKey: 0,
      altKey: 0,
      metaKey: 0,
      repeat: 0,
      locale: 0,
      getModifierState: getEventModifierState,
      charCode: function(event) {
        return "keypress" === event.type ? getEventCharCode(event) : 0;
      },
      keyCode: function(event) {
        return "keydown" === event.type || "keyup" === event.type ? event.keyCode : 0;
      },
      which: function(event) {
        return "keypress" === event.type ? getEventCharCode(event) : "keydown" === event.type || "keyup" === event.type ? event.keyCode : 0;
      }
    }), SyntheticKeyboardEvent = createSyntheticEvent(KeyboardEventInterface), PointerEventInterface = assign({}, MouseEventInterface, {
      pointerId: 0,
      width: 0,
      height: 0,
      pressure: 0,
      tangentialPressure: 0,
      tiltX: 0,
      tiltY: 0,
      twist: 0,
      pointerType: 0,
      isPrimary: 0
    }), SyntheticPointerEvent = createSyntheticEvent(PointerEventInterface), TouchEventInterface = assign({}, UIEventInterface, {
      touches: 0,
      targetTouches: 0,
      changedTouches: 0,
      altKey: 0,
      metaKey: 0,
      ctrlKey: 0,
      shiftKey: 0,
      getModifierState: getEventModifierState
    }), SyntheticTouchEvent = createSyntheticEvent(TouchEventInterface), TransitionEventInterface = assign({}, EventInterface, {
      propertyName: 0,
      elapsedTime: 0,
      pseudoElement: 0
    }), SyntheticTransitionEvent = createSyntheticEvent(TransitionEventInterface), WheelEventInterface = assign({}, MouseEventInterface, {
      deltaX: function(event) {
        return "deltaX" in event ? event.deltaX : "wheelDeltaX" in event ? -event.wheelDeltaX : 0;
      },
      deltaY: function(event) {
        return "deltaY" in event ? event.deltaY : "wheelDeltaY" in event ? -event.wheelDeltaY : "wheelDelta" in event ? -event.wheelDelta : 0;
      },
      deltaZ: 0,
      deltaMode: 0
    }), SyntheticWheelEvent = createSyntheticEvent(WheelEventInterface), ToggleEventInterface = assign({}, EventInterface, {
      newState: 0,
      oldState: 0
    }), SyntheticToggleEvent = createSyntheticEvent(ToggleEventInterface), END_KEYCODES = [9, 13, 27, 32], canUseCompositionEvent = canUseDOM && "CompositionEvent" in window, documentMode = null;
    canUseDOM && "documentMode" in document && (documentMode = document.documentMode);
    var canUseTextInputEvent = canUseDOM && "TextEvent" in window && !documentMode, useFallbackCompositionData = canUseDOM && (!canUseCompositionEvent || documentMode && 8 < documentMode && 11 >= documentMode), SPACEBAR_CHAR = String.fromCharCode(32), hasSpaceKeypress = false;
    function isFallbackCompositionEnd(domEventName, nativeEvent) {
      switch (domEventName) {
        case "keyup":
          return -1 !== END_KEYCODES.indexOf(nativeEvent.keyCode);
        case "keydown":
          return 229 !== nativeEvent.keyCode;
        case "keypress":
        case "mousedown":
        case "focusout":
          return true;
        default:
          return false;
      }
    }
    function getDataFromCustomEvent(nativeEvent) {
      nativeEvent = nativeEvent.detail;
      return "object" === typeof nativeEvent && "data" in nativeEvent ? nativeEvent.data : null;
    }
    var isComposing = false;
    function getNativeBeforeInputChars(domEventName, nativeEvent) {
      switch (domEventName) {
        case "compositionend":
          return getDataFromCustomEvent(nativeEvent);
        case "keypress":
          if (32 !== nativeEvent.which) return null;
          hasSpaceKeypress = true;
          return SPACEBAR_CHAR;
        case "textInput":
          return domEventName = nativeEvent.data, domEventName === SPACEBAR_CHAR && hasSpaceKeypress ? null : domEventName;
        default:
          return null;
      }
    }
    function getFallbackBeforeInputChars(domEventName, nativeEvent) {
      if (isComposing)
        return "compositionend" === domEventName || !canUseCompositionEvent && isFallbackCompositionEnd(domEventName, nativeEvent) ? (domEventName = getData(), fallbackText = startText = root2 = null, isComposing = false, domEventName) : null;
      switch (domEventName) {
        case "paste":
          return null;
        case "keypress":
          if (!(nativeEvent.ctrlKey || nativeEvent.altKey || nativeEvent.metaKey) || nativeEvent.ctrlKey && nativeEvent.altKey) {
            if (nativeEvent.char && 1 < nativeEvent.char.length)
              return nativeEvent.char;
            if (nativeEvent.which) return String.fromCharCode(nativeEvent.which);
          }
          return null;
        case "compositionend":
          return useFallbackCompositionData && "ko" !== nativeEvent.locale ? null : nativeEvent.data;
        default:
          return null;
      }
    }
    var supportedInputTypes = {
      color: true,
      date: true,
      datetime: true,
      "datetime-local": true,
      email: true,
      month: true,
      number: true,
      password: true,
      range: true,
      search: true,
      tel: true,
      text: true,
      time: true,
      url: true,
      week: true
    };
    function isTextInputElement(elem) {
      var nodeName = elem && elem.nodeName && elem.nodeName.toLowerCase();
      return "input" === nodeName ? !!supportedInputTypes[elem.type] : "textarea" === nodeName ? true : false;
    }
    function createAndAccumulateChangeEvent(dispatchQueue, inst, nativeEvent, target) {
      restoreTarget ? restoreQueue ? restoreQueue.push(target) : restoreQueue = [target] : restoreTarget = target;
      inst = accumulateTwoPhaseListeners(inst, "onChange");
      0 < inst.length && (nativeEvent = new SyntheticEvent(
        "onChange",
        "change",
        null,
        nativeEvent,
        target
      ), dispatchQueue.push({ event: nativeEvent, listeners: inst }));
    }
    var activeElement$1 = null, activeElementInst$1 = null;
    function runEventInBatch(dispatchQueue) {
      processDispatchQueue(dispatchQueue, 0);
    }
    function getInstIfValueChanged(targetInst) {
      var targetNode = getNodeFromInstance(targetInst);
      if (updateValueIfChanged(targetNode)) return targetInst;
    }
    function getTargetInstForChangeEvent(domEventName, targetInst) {
      if ("change" === domEventName) return targetInst;
    }
    var isInputEventSupported = false;
    if (canUseDOM) {
      var JSCompiler_inline_result$jscomp$286;
      if (canUseDOM) {
        var isSupported$jscomp$inline_427 = "oninput" in document;
        if (!isSupported$jscomp$inline_427) {
          var element$jscomp$inline_428 = document.createElement("div");
          element$jscomp$inline_428.setAttribute("oninput", "return;");
          isSupported$jscomp$inline_427 = "function" === typeof element$jscomp$inline_428.oninput;
        }
        JSCompiler_inline_result$jscomp$286 = isSupported$jscomp$inline_427;
      } else JSCompiler_inline_result$jscomp$286 = false;
      isInputEventSupported = JSCompiler_inline_result$jscomp$286 && (!document.documentMode || 9 < document.documentMode);
    }
    function stopWatchingForValueChange() {
      activeElement$1 && (activeElement$1.detachEvent("onpropertychange", handlePropertyChange), activeElementInst$1 = activeElement$1 = null);
    }
    function handlePropertyChange(nativeEvent) {
      if ("value" === nativeEvent.propertyName && getInstIfValueChanged(activeElementInst$1)) {
        var dispatchQueue = [];
        createAndAccumulateChangeEvent(
          dispatchQueue,
          activeElementInst$1,
          nativeEvent,
          getEventTarget(nativeEvent)
        );
        batchedUpdates$1(runEventInBatch, dispatchQueue);
      }
    }
    function handleEventsForInputEventPolyfill(domEventName, target, targetInst) {
      "focusin" === domEventName ? (stopWatchingForValueChange(), activeElement$1 = target, activeElementInst$1 = targetInst, activeElement$1.attachEvent("onpropertychange", handlePropertyChange)) : "focusout" === domEventName && stopWatchingForValueChange();
    }
    function getTargetInstForInputEventPolyfill(domEventName) {
      if ("selectionchange" === domEventName || "keyup" === domEventName || "keydown" === domEventName)
        return getInstIfValueChanged(activeElementInst$1);
    }
    function getTargetInstForClickEvent(domEventName, targetInst) {
      if ("click" === domEventName) return getInstIfValueChanged(targetInst);
    }
    function getTargetInstForInputOrChangeEvent(domEventName, targetInst) {
      if ("input" === domEventName || "change" === domEventName)
        return getInstIfValueChanged(targetInst);
    }
    function is(x, y) {
      return x === y && (0 !== x || 1 / x === 1 / y) || x !== x && y !== y;
    }
    var objectIs = "function" === typeof Object.is ? Object.is : is;
    function shallowEqual(objA, objB) {
      if (objectIs(objA, objB)) return true;
      if ("object" !== typeof objA || null === objA || "object" !== typeof objB || null === objB)
        return false;
      var keysA = Object.keys(objA), keysB = Object.keys(objB);
      if (keysA.length !== keysB.length) return false;
      for (keysB = 0; keysB < keysA.length; keysB++) {
        var currentKey = keysA[keysB];
        if (!hasOwnProperty.call(objB, currentKey) || !objectIs(objA[currentKey], objB[currentKey]))
          return false;
      }
      return true;
    }
    function getLeafNode(node) {
      for (; node && node.firstChild; ) node = node.firstChild;
      return node;
    }
    function getNodeForCharacterOffset(root3, offset) {
      var node = getLeafNode(root3);
      root3 = 0;
      for (var nodeEnd; node; ) {
        if (3 === node.nodeType) {
          nodeEnd = root3 + node.textContent.length;
          if (root3 <= offset && nodeEnd >= offset)
            return { node, offset: offset - root3 };
          root3 = nodeEnd;
        }
        a: {
          for (; node; ) {
            if (node.nextSibling) {
              node = node.nextSibling;
              break a;
            }
            node = node.parentNode;
          }
          node = void 0;
        }
        node = getLeafNode(node);
      }
    }
    function containsNode(outerNode, innerNode) {
      return outerNode && innerNode ? outerNode === innerNode ? true : outerNode && 3 === outerNode.nodeType ? false : innerNode && 3 === innerNode.nodeType ? containsNode(outerNode, innerNode.parentNode) : "contains" in outerNode ? outerNode.contains(innerNode) : outerNode.compareDocumentPosition ? !!(outerNode.compareDocumentPosition(innerNode) & 16) : false : false;
    }
    function getActiveElementDeep(containerInfo) {
      containerInfo = null != containerInfo && null != containerInfo.ownerDocument && null != containerInfo.ownerDocument.defaultView ? containerInfo.ownerDocument.defaultView : window;
      for (var element = getActiveElement(containerInfo.document); element instanceof containerInfo.HTMLIFrameElement; ) {
        try {
          var JSCompiler_inline_result = "string" === typeof element.contentWindow.location.href;
        } catch (err) {
          JSCompiler_inline_result = false;
        }
        if (JSCompiler_inline_result) containerInfo = element.contentWindow;
        else break;
        element = getActiveElement(containerInfo.document);
      }
      return element;
    }
    function hasSelectionCapabilities(elem) {
      var nodeName = elem && elem.nodeName && elem.nodeName.toLowerCase();
      return nodeName && ("input" === nodeName && ("text" === elem.type || "search" === elem.type || "tel" === elem.type || "url" === elem.type || "password" === elem.type) || "textarea" === nodeName || "true" === elem.contentEditable);
    }
    var skipSelectionChangeEvent = canUseDOM && "documentMode" in document && 11 >= document.documentMode, activeElement = null, activeElementInst = null, lastSelection = null, mouseDown = false;
    function constructSelectEvent(dispatchQueue, nativeEvent, nativeEventTarget) {
      var doc = nativeEventTarget.window === nativeEventTarget ? nativeEventTarget.document : 9 === nativeEventTarget.nodeType ? nativeEventTarget : nativeEventTarget.ownerDocument;
      mouseDown || null == activeElement || activeElement !== getActiveElement(doc) || (doc = activeElement, "selectionStart" in doc && hasSelectionCapabilities(doc) ? doc = { start: doc.selectionStart, end: doc.selectionEnd } : (doc = (doc.ownerDocument && doc.ownerDocument.defaultView || window).getSelection(), doc = {
        anchorNode: doc.anchorNode,
        anchorOffset: doc.anchorOffset,
        focusNode: doc.focusNode,
        focusOffset: doc.focusOffset
      }), lastSelection && shallowEqual(lastSelection, doc) || (lastSelection = doc, doc = accumulateTwoPhaseListeners(activeElementInst, "onSelect"), 0 < doc.length && (nativeEvent = new SyntheticEvent(
        "onSelect",
        "select",
        null,
        nativeEvent,
        nativeEventTarget
      ), dispatchQueue.push({ event: nativeEvent, listeners: doc }), nativeEvent.target = activeElement)));
    }
    function makePrefixMap(styleProp, eventName) {
      var prefixes = {};
      prefixes[styleProp.toLowerCase()] = eventName.toLowerCase();
      prefixes["Webkit" + styleProp] = "webkit" + eventName;
      prefixes["Moz" + styleProp] = "moz" + eventName;
      return prefixes;
    }
    var vendorPrefixes = {
      animationend: makePrefixMap("Animation", "AnimationEnd"),
      animationiteration: makePrefixMap("Animation", "AnimationIteration"),
      animationstart: makePrefixMap("Animation", "AnimationStart"),
      transitionrun: makePrefixMap("Transition", "TransitionRun"),
      transitionstart: makePrefixMap("Transition", "TransitionStart"),
      transitioncancel: makePrefixMap("Transition", "TransitionCancel"),
      transitionend: makePrefixMap("Transition", "TransitionEnd")
    }, prefixedEventNames = {}, style = {};
    canUseDOM && (style = document.createElement("div").style, "AnimationEvent" in window || (delete vendorPrefixes.animationend.animation, delete vendorPrefixes.animationiteration.animation, delete vendorPrefixes.animationstart.animation), "TransitionEvent" in window || delete vendorPrefixes.transitionend.transition);
    function getVendorPrefixedEventName(eventName) {
      if (prefixedEventNames[eventName]) return prefixedEventNames[eventName];
      if (!vendorPrefixes[eventName]) return eventName;
      var prefixMap = vendorPrefixes[eventName], styleProp;
      for (styleProp in prefixMap)
        if (prefixMap.hasOwnProperty(styleProp) && styleProp in style)
          return prefixedEventNames[eventName] = prefixMap[styleProp];
      return eventName;
    }
    var ANIMATION_END = getVendorPrefixedEventName("animationend"), ANIMATION_ITERATION = getVendorPrefixedEventName("animationiteration"), ANIMATION_START = getVendorPrefixedEventName("animationstart"), TRANSITION_RUN = getVendorPrefixedEventName("transitionrun"), TRANSITION_START = getVendorPrefixedEventName("transitionstart"), TRANSITION_CANCEL = getVendorPrefixedEventName("transitioncancel"), TRANSITION_END = getVendorPrefixedEventName("transitionend"), topLevelEventsToReactNames = /* @__PURE__ */ new Map(), simpleEventPluginEvents = "abort auxClick beforeToggle cancel canPlay canPlayThrough click close contextMenu copy cut drag dragEnd dragEnter dragExit dragLeave dragOver dragStart drop durationChange emptied encrypted ended error gotPointerCapture input invalid keyDown keyPress keyUp load loadedData loadedMetadata loadStart lostPointerCapture mouseDown mouseMove mouseOut mouseOver mouseUp paste pause play playing pointerCancel pointerDown pointerMove pointerOut pointerOver pointerUp progress rateChange reset resize seeked seeking stalled submit suspend timeUpdate touchCancel touchEnd touchStart volumeChange scroll toggle touchMove waiting wheel".split(
      " "
    );
    simpleEventPluginEvents.push("scrollEnd");
    function registerSimpleEvent(domEventName, reactName) {
      topLevelEventsToReactNames.set(domEventName, reactName);
      registerTwoPhaseEvent(reactName, [domEventName]);
    }
    var reportGlobalError = "function" === typeof reportError ? reportError : function(error) {
      if ("object" === typeof window && "function" === typeof window.ErrorEvent) {
        var event = new window.ErrorEvent("error", {
          bubbles: true,
          cancelable: true,
          message: "object" === typeof error && null !== error && "string" === typeof error.message ? String(error.message) : String(error),
          error
        });
        if (!window.dispatchEvent(event)) return;
      } else if ("object" === typeof process && "function" === typeof process.emit) {
        process.emit("uncaughtException", error);
        return;
      }
      console.error(error);
    }, concurrentQueues = [], concurrentQueuesIndex = 0, concurrentlyUpdatedLanes = 0;
    function finishQueueingConcurrentUpdates() {
      for (var endIndex = concurrentQueuesIndex, i = concurrentlyUpdatedLanes = concurrentQueuesIndex = 0; i < endIndex; ) {
        var fiber = concurrentQueues[i];
        concurrentQueues[i++] = null;
        var queue = concurrentQueues[i];
        concurrentQueues[i++] = null;
        var update = concurrentQueues[i];
        concurrentQueues[i++] = null;
        var lane = concurrentQueues[i];
        concurrentQueues[i++] = null;
        if (null !== queue && null !== update) {
          var pending = queue.pending;
          null === pending ? update.next = update : (update.next = pending.next, pending.next = update);
          queue.pending = update;
        }
        0 !== lane && markUpdateLaneFromFiberToRoot(fiber, update, lane);
      }
    }
    function enqueueUpdate$1(fiber, queue, update, lane) {
      concurrentQueues[concurrentQueuesIndex++] = fiber;
      concurrentQueues[concurrentQueuesIndex++] = queue;
      concurrentQueues[concurrentQueuesIndex++] = update;
      concurrentQueues[concurrentQueuesIndex++] = lane;
      concurrentlyUpdatedLanes |= lane;
      fiber.lanes |= lane;
      fiber = fiber.alternate;
      null !== fiber && (fiber.lanes |= lane);
    }
    function enqueueConcurrentHookUpdate(fiber, queue, update, lane) {
      enqueueUpdate$1(fiber, queue, update, lane);
      return getRootForUpdatedFiber(fiber);
    }
    function enqueueConcurrentRenderForLane(fiber, lane) {
      enqueueUpdate$1(fiber, null, null, lane);
      return getRootForUpdatedFiber(fiber);
    }
    function markUpdateLaneFromFiberToRoot(sourceFiber, update, lane) {
      sourceFiber.lanes |= lane;
      var alternate = sourceFiber.alternate;
      null !== alternate && (alternate.lanes |= lane);
      for (var isHidden = false, parent = sourceFiber.return; null !== parent; )
        parent.childLanes |= lane, alternate = parent.alternate, null !== alternate && (alternate.childLanes |= lane), 22 === parent.tag && (sourceFiber = parent.stateNode, null === sourceFiber || sourceFiber._visibility & 1 || (isHidden = true)), sourceFiber = parent, parent = parent.return;
      return 3 === sourceFiber.tag ? (parent = sourceFiber.stateNode, isHidden && null !== update && (isHidden = 31 - clz32(lane), sourceFiber = parent.hiddenUpdates, alternate = sourceFiber[isHidden], null === alternate ? sourceFiber[isHidden] = [update] : alternate.push(update), update.lane = lane | 536870912), parent) : null;
    }
    function getRootForUpdatedFiber(sourceFiber) {
      if (50 < nestedUpdateCount)
        throw nestedUpdateCount = 0, rootWithNestedUpdates = null, Error(formatProdErrorMessage(185));
      for (var parent = sourceFiber.return; null !== parent; )
        sourceFiber = parent, parent = sourceFiber.return;
      return 3 === sourceFiber.tag ? sourceFiber.stateNode : null;
    }
    var emptyContextObject = {};
    function FiberNode(tag, pendingProps, key, mode) {
      this.tag = tag;
      this.key = key;
      this.sibling = this.child = this.return = this.stateNode = this.type = this.elementType = null;
      this.index = 0;
      this.refCleanup = this.ref = null;
      this.pendingProps = pendingProps;
      this.dependencies = this.memoizedState = this.updateQueue = this.memoizedProps = null;
      this.mode = mode;
      this.subtreeFlags = this.flags = 0;
      this.deletions = null;
      this.childLanes = this.lanes = 0;
      this.alternate = null;
    }
    function createFiberImplClass(tag, pendingProps, key, mode) {
      return new FiberNode(tag, pendingProps, key, mode);
    }
    function shouldConstruct(Component) {
      Component = Component.prototype;
      return !(!Component || !Component.isReactComponent);
    }
    function createWorkInProgress(current, pendingProps) {
      var workInProgress2 = current.alternate;
      null === workInProgress2 ? (workInProgress2 = createFiberImplClass(
        current.tag,
        pendingProps,
        current.key,
        current.mode
      ), workInProgress2.elementType = current.elementType, workInProgress2.type = current.type, workInProgress2.stateNode = current.stateNode, workInProgress2.alternate = current, current.alternate = workInProgress2) : (workInProgress2.pendingProps = pendingProps, workInProgress2.type = current.type, workInProgress2.flags = 0, workInProgress2.subtreeFlags = 0, workInProgress2.deletions = null);
      workInProgress2.flags = current.flags & 65011712;
      workInProgress2.childLanes = current.childLanes;
      workInProgress2.lanes = current.lanes;
      workInProgress2.child = current.child;
      workInProgress2.memoizedProps = current.memoizedProps;
      workInProgress2.memoizedState = current.memoizedState;
      workInProgress2.updateQueue = current.updateQueue;
      pendingProps = current.dependencies;
      workInProgress2.dependencies = null === pendingProps ? null : { lanes: pendingProps.lanes, firstContext: pendingProps.firstContext };
      workInProgress2.sibling = current.sibling;
      workInProgress2.index = current.index;
      workInProgress2.ref = current.ref;
      workInProgress2.refCleanup = current.refCleanup;
      return workInProgress2;
    }
    function resetWorkInProgress(workInProgress2, renderLanes2) {
      workInProgress2.flags &= 65011714;
      var current = workInProgress2.alternate;
      null === current ? (workInProgress2.childLanes = 0, workInProgress2.lanes = renderLanes2, workInProgress2.child = null, workInProgress2.subtreeFlags = 0, workInProgress2.memoizedProps = null, workInProgress2.memoizedState = null, workInProgress2.updateQueue = null, workInProgress2.dependencies = null, workInProgress2.stateNode = null) : (workInProgress2.childLanes = current.childLanes, workInProgress2.lanes = current.lanes, workInProgress2.child = current.child, workInProgress2.subtreeFlags = 0, workInProgress2.deletions = null, workInProgress2.memoizedProps = current.memoizedProps, workInProgress2.memoizedState = current.memoizedState, workInProgress2.updateQueue = current.updateQueue, workInProgress2.type = current.type, renderLanes2 = current.dependencies, workInProgress2.dependencies = null === renderLanes2 ? null : {
        lanes: renderLanes2.lanes,
        firstContext: renderLanes2.firstContext
      });
      return workInProgress2;
    }
    function createFiberFromTypeAndProps(type, key, pendingProps, owner, mode, lanes) {
      var fiberTag = 0;
      owner = type;
      if ("function" === typeof type) shouldConstruct(type) && (fiberTag = 1);
      else if ("string" === typeof type)
        fiberTag = isHostHoistableType(
          type,
          pendingProps,
          contextStackCursor.current
        ) ? 26 : "html" === type || "head" === type || "body" === type ? 27 : 5;
      else
        a: switch (type) {
          case REACT_ACTIVITY_TYPE:
            return type = createFiberImplClass(31, pendingProps, key, mode), type.elementType = REACT_ACTIVITY_TYPE, type.lanes = lanes, type;
          case REACT_FRAGMENT_TYPE:
            return createFiberFromFragment(pendingProps.children, mode, lanes, key);
          case REACT_STRICT_MODE_TYPE:
            fiberTag = 8;
            mode |= 24;
            break;
          case REACT_PROFILER_TYPE:
            return type = createFiberImplClass(12, pendingProps, key, mode | 2), type.elementType = REACT_PROFILER_TYPE, type.lanes = lanes, type;
          case REACT_SUSPENSE_TYPE:
            return type = createFiberImplClass(13, pendingProps, key, mode), type.elementType = REACT_SUSPENSE_TYPE, type.lanes = lanes, type;
          case REACT_SUSPENSE_LIST_TYPE:
            return type = createFiberImplClass(19, pendingProps, key, mode), type.elementType = REACT_SUSPENSE_LIST_TYPE, type.lanes = lanes, type;
          default:
            if ("object" === typeof type && null !== type)
              switch (type.$$typeof) {
                case REACT_CONTEXT_TYPE:
                  fiberTag = 10;
                  break a;
                case REACT_CONSUMER_TYPE:
                  fiberTag = 9;
                  break a;
                case REACT_FORWARD_REF_TYPE:
                  fiberTag = 11;
                  break a;
                case REACT_MEMO_TYPE:
                  fiberTag = 14;
                  break a;
                case REACT_LAZY_TYPE:
                  fiberTag = 16;
                  owner = null;
                  break a;
              }
            fiberTag = 29;
            pendingProps = Error(
              formatProdErrorMessage(130, null === type ? "null" : typeof type, "")
            );
            owner = null;
        }
      key = createFiberImplClass(fiberTag, pendingProps, key, mode);
      key.elementType = type;
      key.type = owner;
      key.lanes = lanes;
      return key;
    }
    function createFiberFromFragment(elements, mode, lanes, key) {
      elements = createFiberImplClass(7, elements, key, mode);
      elements.lanes = lanes;
      return elements;
    }
    function createFiberFromText(content, mode, lanes) {
      content = createFiberImplClass(6, content, null, mode);
      content.lanes = lanes;
      return content;
    }
    function createFiberFromDehydratedFragment(dehydratedNode) {
      var fiber = createFiberImplClass(18, null, null, 0);
      fiber.stateNode = dehydratedNode;
      return fiber;
    }
    function createFiberFromPortal(portal, mode, lanes) {
      mode = createFiberImplClass(
        4,
        null !== portal.children ? portal.children : [],
        portal.key,
        mode
      );
      mode.lanes = lanes;
      mode.stateNode = {
        containerInfo: portal.containerInfo,
        pendingChildren: null,
        implementation: portal.implementation
      };
      return mode;
    }
    var CapturedStacks = /* @__PURE__ */ new WeakMap();
    function createCapturedValueAtFiber(value, source) {
      if ("object" === typeof value && null !== value) {
        var existing = CapturedStacks.get(value);
        if (void 0 !== existing) return existing;
        source = {
          value,
          source,
          stack: getStackByFiberInDevAndProd(source)
        };
        CapturedStacks.set(value, source);
        return source;
      }
      return {
        value,
        source,
        stack: getStackByFiberInDevAndProd(source)
      };
    }
    var forkStack = [], forkStackIndex = 0, treeForkProvider = null, treeForkCount = 0, idStack = [], idStackIndex = 0, treeContextProvider = null, treeContextId = 1, treeContextOverflow = "";
    function pushTreeFork(workInProgress2, totalChildren) {
      forkStack[forkStackIndex++] = treeForkCount;
      forkStack[forkStackIndex++] = treeForkProvider;
      treeForkProvider = workInProgress2;
      treeForkCount = totalChildren;
    }
    function pushTreeId(workInProgress2, totalChildren, index2) {
      idStack[idStackIndex++] = treeContextId;
      idStack[idStackIndex++] = treeContextOverflow;
      idStack[idStackIndex++] = treeContextProvider;
      treeContextProvider = workInProgress2;
      var baseIdWithLeadingBit = treeContextId;
      workInProgress2 = treeContextOverflow;
      var baseLength = 32 - clz32(baseIdWithLeadingBit) - 1;
      baseIdWithLeadingBit &= ~(1 << baseLength);
      index2 += 1;
      var length = 32 - clz32(totalChildren) + baseLength;
      if (30 < length) {
        var numberOfOverflowBits = baseLength - baseLength % 5;
        length = (baseIdWithLeadingBit & (1 << numberOfOverflowBits) - 1).toString(32);
        baseIdWithLeadingBit >>= numberOfOverflowBits;
        baseLength -= numberOfOverflowBits;
        treeContextId = 1 << 32 - clz32(totalChildren) + baseLength | index2 << baseLength | baseIdWithLeadingBit;
        treeContextOverflow = length + workInProgress2;
      } else
        treeContextId = 1 << length | index2 << baseLength | baseIdWithLeadingBit, treeContextOverflow = workInProgress2;
    }
    function pushMaterializedTreeId(workInProgress2) {
      null !== workInProgress2.return && (pushTreeFork(workInProgress2, 1), pushTreeId(workInProgress2, 1, 0));
    }
    function popTreeContext(workInProgress2) {
      for (; workInProgress2 === treeForkProvider; )
        treeForkProvider = forkStack[--forkStackIndex], forkStack[forkStackIndex] = null, treeForkCount = forkStack[--forkStackIndex], forkStack[forkStackIndex] = null;
      for (; workInProgress2 === treeContextProvider; )
        treeContextProvider = idStack[--idStackIndex], idStack[idStackIndex] = null, treeContextOverflow = idStack[--idStackIndex], idStack[idStackIndex] = null, treeContextId = idStack[--idStackIndex], idStack[idStackIndex] = null;
    }
    function restoreSuspendedTreeContext(workInProgress2, suspendedContext) {
      idStack[idStackIndex++] = treeContextId;
      idStack[idStackIndex++] = treeContextOverflow;
      idStack[idStackIndex++] = treeContextProvider;
      treeContextId = suspendedContext.id;
      treeContextOverflow = suspendedContext.overflow;
      treeContextProvider = workInProgress2;
    }
    var hydrationParentFiber = null, nextHydratableInstance = null, isHydrating = false, hydrationErrors = null, rootOrSingletonContext = false, HydrationMismatchException = Error(formatProdErrorMessage(519));
    function throwOnHydrationMismatch(fiber) {
      var error = Error(
        formatProdErrorMessage(
          418,
          1 < arguments.length && void 0 !== arguments[1] && arguments[1] ? "text" : "HTML",
          ""
        )
      );
      queueHydrationError(createCapturedValueAtFiber(error, fiber));
      throw HydrationMismatchException;
    }
    function prepareToHydrateHostInstance(fiber) {
      var instance = fiber.stateNode, type = fiber.type, props = fiber.memoizedProps;
      instance[internalInstanceKey] = fiber;
      instance[internalPropsKey] = props;
      switch (type) {
        case "dialog":
          listenToNonDelegatedEvent("cancel", instance);
          listenToNonDelegatedEvent("close", instance);
          break;
        case "iframe":
        case "object":
        case "embed":
          listenToNonDelegatedEvent("load", instance);
          break;
        case "video":
        case "audio":
          for (type = 0; type < mediaEventTypes.length; type++)
            listenToNonDelegatedEvent(mediaEventTypes[type], instance);
          break;
        case "source":
          listenToNonDelegatedEvent("error", instance);
          break;
        case "img":
        case "image":
        case "link":
          listenToNonDelegatedEvent("error", instance);
          listenToNonDelegatedEvent("load", instance);
          break;
        case "details":
          listenToNonDelegatedEvent("toggle", instance);
          break;
        case "input":
          listenToNonDelegatedEvent("invalid", instance);
          initInput(
            instance,
            props.value,
            props.defaultValue,
            props.checked,
            props.defaultChecked,
            props.type,
            props.name,
            true
          );
          break;
        case "select":
          listenToNonDelegatedEvent("invalid", instance);
          break;
        case "textarea":
          listenToNonDelegatedEvent("invalid", instance), initTextarea(instance, props.value, props.defaultValue, props.children);
      }
      type = props.children;
      "string" !== typeof type && "number" !== typeof type && "bigint" !== typeof type || instance.textContent === "" + type || true === props.suppressHydrationWarning || checkForUnmatchedText(instance.textContent, type) ? (null != props.popover && (listenToNonDelegatedEvent("beforetoggle", instance), listenToNonDelegatedEvent("toggle", instance)), null != props.onScroll && listenToNonDelegatedEvent("scroll", instance), null != props.onScrollEnd && listenToNonDelegatedEvent("scrollend", instance), null != props.onClick && (instance.onclick = noop$1), instance = true) : instance = false;
      instance || throwOnHydrationMismatch(fiber, true);
    }
    function popToNextHostParent(fiber) {
      for (hydrationParentFiber = fiber.return; hydrationParentFiber; )
        switch (hydrationParentFiber.tag) {
          case 5:
          case 31:
          case 13:
            rootOrSingletonContext = false;
            return;
          case 27:
          case 3:
            rootOrSingletonContext = true;
            return;
          default:
            hydrationParentFiber = hydrationParentFiber.return;
        }
    }
    function popHydrationState(fiber) {
      if (fiber !== hydrationParentFiber) return false;
      if (!isHydrating) return popToNextHostParent(fiber), isHydrating = true, false;
      var tag = fiber.tag, JSCompiler_temp;
      if (JSCompiler_temp = 3 !== tag && 27 !== tag) {
        if (JSCompiler_temp = 5 === tag)
          JSCompiler_temp = fiber.type, JSCompiler_temp = !("form" !== JSCompiler_temp && "button" !== JSCompiler_temp) || shouldSetTextContent(fiber.type, fiber.memoizedProps);
        JSCompiler_temp = !JSCompiler_temp;
      }
      JSCompiler_temp && nextHydratableInstance && throwOnHydrationMismatch(fiber);
      popToNextHostParent(fiber);
      if (13 === tag) {
        fiber = fiber.memoizedState;
        fiber = null !== fiber ? fiber.dehydrated : null;
        if (!fiber) throw Error(formatProdErrorMessage(317));
        nextHydratableInstance = getNextHydratableInstanceAfterHydrationBoundary(fiber);
      } else if (31 === tag) {
        fiber = fiber.memoizedState;
        fiber = null !== fiber ? fiber.dehydrated : null;
        if (!fiber) throw Error(formatProdErrorMessage(317));
        nextHydratableInstance = getNextHydratableInstanceAfterHydrationBoundary(fiber);
      } else
        27 === tag ? (tag = nextHydratableInstance, isSingletonScope(fiber.type) ? (fiber = previousHydratableOnEnteringScopedSingleton, previousHydratableOnEnteringScopedSingleton = null, nextHydratableInstance = fiber) : nextHydratableInstance = tag) : nextHydratableInstance = hydrationParentFiber ? getNextHydratable(fiber.stateNode.nextSibling) : null;
      return true;
    }
    function resetHydrationState() {
      nextHydratableInstance = hydrationParentFiber = null;
      isHydrating = false;
    }
    function upgradeHydrationErrorsToRecoverable() {
      var queuedErrors = hydrationErrors;
      null !== queuedErrors && (null === workInProgressRootRecoverableErrors ? workInProgressRootRecoverableErrors = queuedErrors : workInProgressRootRecoverableErrors.push.apply(
        workInProgressRootRecoverableErrors,
        queuedErrors
      ), hydrationErrors = null);
      return queuedErrors;
    }
    function queueHydrationError(error) {
      null === hydrationErrors ? hydrationErrors = [error] : hydrationErrors.push(error);
    }
    var valueCursor = createCursor(null), currentlyRenderingFiber$1 = null, lastContextDependency = null;
    function pushProvider(providerFiber, context, nextValue) {
      push(valueCursor, context._currentValue);
      context._currentValue = nextValue;
    }
    function popProvider(context) {
      context._currentValue = valueCursor.current;
      pop(valueCursor);
    }
    function scheduleContextWorkOnParentPath(parent, renderLanes2, propagationRoot) {
      for (; null !== parent; ) {
        var alternate = parent.alternate;
        (parent.childLanes & renderLanes2) !== renderLanes2 ? (parent.childLanes |= renderLanes2, null !== alternate && (alternate.childLanes |= renderLanes2)) : null !== alternate && (alternate.childLanes & renderLanes2) !== renderLanes2 && (alternate.childLanes |= renderLanes2);
        if (parent === propagationRoot) break;
        parent = parent.return;
      }
    }
    function propagateContextChanges(workInProgress2, contexts, renderLanes2, forcePropagateEntireTree) {
      var fiber = workInProgress2.child;
      null !== fiber && (fiber.return = workInProgress2);
      for (; null !== fiber; ) {
        var list = fiber.dependencies;
        if (null !== list) {
          var nextFiber = fiber.child;
          list = list.firstContext;
          a: for (; null !== list; ) {
            var dependency = list;
            list = fiber;
            for (var i = 0; i < contexts.length; i++)
              if (dependency.context === contexts[i]) {
                list.lanes |= renderLanes2;
                dependency = list.alternate;
                null !== dependency && (dependency.lanes |= renderLanes2);
                scheduleContextWorkOnParentPath(
                  list.return,
                  renderLanes2,
                  workInProgress2
                );
                forcePropagateEntireTree || (nextFiber = null);
                break a;
              }
            list = dependency.next;
          }
        } else if (18 === fiber.tag) {
          nextFiber = fiber.return;
          if (null === nextFiber) throw Error(formatProdErrorMessage(341));
          nextFiber.lanes |= renderLanes2;
          list = nextFiber.alternate;
          null !== list && (list.lanes |= renderLanes2);
          scheduleContextWorkOnParentPath(nextFiber, renderLanes2, workInProgress2);
          nextFiber = null;
        } else nextFiber = fiber.child;
        if (null !== nextFiber) nextFiber.return = fiber;
        else
          for (nextFiber = fiber; null !== nextFiber; ) {
            if (nextFiber === workInProgress2) {
              nextFiber = null;
              break;
            }
            fiber = nextFiber.sibling;
            if (null !== fiber) {
              fiber.return = nextFiber.return;
              nextFiber = fiber;
              break;
            }
            nextFiber = nextFiber.return;
          }
        fiber = nextFiber;
      }
    }
    function propagateParentContextChanges(current, workInProgress2, renderLanes2, forcePropagateEntireTree) {
      current = null;
      for (var parent = workInProgress2, isInsidePropagationBailout = false; null !== parent; ) {
        if (!isInsidePropagationBailout) {
          if (0 !== (parent.flags & 524288)) isInsidePropagationBailout = true;
          else if (0 !== (parent.flags & 262144)) break;
        }
        if (10 === parent.tag) {
          var currentParent = parent.alternate;
          if (null === currentParent) throw Error(formatProdErrorMessage(387));
          currentParent = currentParent.memoizedProps;
          if (null !== currentParent) {
            var context = parent.type;
            objectIs(parent.pendingProps.value, currentParent.value) || (null !== current ? current.push(context) : current = [context]);
          }
        } else if (parent === hostTransitionProviderCursor.current) {
          currentParent = parent.alternate;
          if (null === currentParent) throw Error(formatProdErrorMessage(387));
          currentParent.memoizedState.memoizedState !== parent.memoizedState.memoizedState && (null !== current ? current.push(HostTransitionContext) : current = [HostTransitionContext]);
        }
        parent = parent.return;
      }
      null !== current && propagateContextChanges(
        workInProgress2,
        current,
        renderLanes2,
        forcePropagateEntireTree
      );
      workInProgress2.flags |= 262144;
    }
    function checkIfContextChanged(currentDependencies) {
      for (currentDependencies = currentDependencies.firstContext; null !== currentDependencies; ) {
        if (!objectIs(
          currentDependencies.context._currentValue,
          currentDependencies.memoizedValue
        ))
          return true;
        currentDependencies = currentDependencies.next;
      }
      return false;
    }
    function prepareToReadContext(workInProgress2) {
      currentlyRenderingFiber$1 = workInProgress2;
      lastContextDependency = null;
      workInProgress2 = workInProgress2.dependencies;
      null !== workInProgress2 && (workInProgress2.firstContext = null);
    }
    function readContext(context) {
      return readContextForConsumer(currentlyRenderingFiber$1, context);
    }
    function readContextDuringReconciliation(consumer, context) {
      null === currentlyRenderingFiber$1 && prepareToReadContext(consumer);
      return readContextForConsumer(consumer, context);
    }
    function readContextForConsumer(consumer, context) {
      var value = context._currentValue;
      context = { context, memoizedValue: value, next: null };
      if (null === lastContextDependency) {
        if (null === consumer) throw Error(formatProdErrorMessage(308));
        lastContextDependency = context;
        consumer.dependencies = { lanes: 0, firstContext: context };
        consumer.flags |= 524288;
      } else lastContextDependency = lastContextDependency.next = context;
      return value;
    }
    var AbortControllerLocal = "undefined" !== typeof AbortController ? AbortController : function() {
      var listeners = [], signal = this.signal = {
        aborted: false,
        addEventListener: function(type, listener) {
          listeners.push(listener);
        }
      };
      this.abort = function() {
        signal.aborted = true;
        listeners.forEach(function(listener) {
          return listener();
        });
      };
    }, scheduleCallback$2 = Scheduler.unstable_scheduleCallback, NormalPriority = Scheduler.unstable_NormalPriority, CacheContext = {
      $$typeof: REACT_CONTEXT_TYPE,
      Consumer: null,
      Provider: null,
      _currentValue: null,
      _currentValue2: null,
      _threadCount: 0
    };
    function createCache() {
      return {
        controller: new AbortControllerLocal(),
        data: /* @__PURE__ */ new Map(),
        refCount: 0
      };
    }
    function releaseCache(cache) {
      cache.refCount--;
      0 === cache.refCount && scheduleCallback$2(NormalPriority, function() {
        cache.controller.abort();
      });
    }
    var currentEntangledListeners = null, currentEntangledPendingCount = 0, currentEntangledLane = 0, currentEntangledActionThenable = null;
    function entangleAsyncAction(transition, thenable) {
      if (null === currentEntangledListeners) {
        var entangledListeners = currentEntangledListeners = [];
        currentEntangledPendingCount = 0;
        currentEntangledLane = requestTransitionLane();
        currentEntangledActionThenable = {
          status: "pending",
          value: void 0,
          then: function(resolve) {
            entangledListeners.push(resolve);
          }
        };
      }
      currentEntangledPendingCount++;
      thenable.then(pingEngtangledActionScope, pingEngtangledActionScope);
      return thenable;
    }
    function pingEngtangledActionScope() {
      if (0 === --currentEntangledPendingCount && null !== currentEntangledListeners) {
        null !== currentEntangledActionThenable && (currentEntangledActionThenable.status = "fulfilled");
        var listeners = currentEntangledListeners;
        currentEntangledListeners = null;
        currentEntangledLane = 0;
        currentEntangledActionThenable = null;
        for (var i = 0; i < listeners.length; i++) (0, listeners[i])();
      }
    }
    function chainThenableValue(thenable, result2) {
      var listeners = [], thenableWithOverride = {
        status: "pending",
        value: null,
        reason: null,
        then: function(resolve) {
          listeners.push(resolve);
        }
      };
      thenable.then(
        function() {
          thenableWithOverride.status = "fulfilled";
          thenableWithOverride.value = result2;
          for (var i = 0; i < listeners.length; i++) (0, listeners[i])(result2);
        },
        function(error) {
          thenableWithOverride.status = "rejected";
          thenableWithOverride.reason = error;
          for (error = 0; error < listeners.length; error++)
            (0, listeners[error])(void 0);
        }
      );
      return thenableWithOverride;
    }
    var prevOnStartTransitionFinish = ReactSharedInternals.S;
    ReactSharedInternals.S = function(transition, returnValue) {
      globalMostRecentTransitionTime = now();
      "object" === typeof returnValue && null !== returnValue && "function" === typeof returnValue.then && entangleAsyncAction(transition, returnValue);
      null !== prevOnStartTransitionFinish && prevOnStartTransitionFinish(transition, returnValue);
    };
    var resumedCache = createCursor(null);
    function peekCacheFromPool() {
      var cacheResumedFromPreviousRender = resumedCache.current;
      return null !== cacheResumedFromPreviousRender ? cacheResumedFromPreviousRender : workInProgressRoot.pooledCache;
    }
    function pushTransition(offscreenWorkInProgress, prevCachePool) {
      null === prevCachePool ? push(resumedCache, resumedCache.current) : push(resumedCache, prevCachePool.pool);
    }
    function getSuspendedCache() {
      var cacheFromPool = peekCacheFromPool();
      return null === cacheFromPool ? null : { parent: CacheContext._currentValue, pool: cacheFromPool };
    }
    var SuspenseException = Error(formatProdErrorMessage(460)), SuspenseyCommitException = Error(formatProdErrorMessage(474)), SuspenseActionException = Error(formatProdErrorMessage(542)), noopSuspenseyCommitThenable = { then: function() {
    } };
    function isThenableResolved(thenable) {
      thenable = thenable.status;
      return "fulfilled" === thenable || "rejected" === thenable;
    }
    function trackUsedThenable(thenableState2, thenable, index2) {
      index2 = thenableState2[index2];
      void 0 === index2 ? thenableState2.push(thenable) : index2 !== thenable && (thenable.then(noop$1, noop$1), thenable = index2);
      switch (thenable.status) {
        case "fulfilled":
          return thenable.value;
        case "rejected":
          throw thenableState2 = thenable.reason, checkIfUseWrappedInAsyncCatch(thenableState2), thenableState2;
        default:
          if ("string" === typeof thenable.status) thenable.then(noop$1, noop$1);
          else {
            thenableState2 = workInProgressRoot;
            if (null !== thenableState2 && 100 < thenableState2.shellSuspendCounter)
              throw Error(formatProdErrorMessage(482));
            thenableState2 = thenable;
            thenableState2.status = "pending";
            thenableState2.then(
              function(fulfilledValue) {
                if ("pending" === thenable.status) {
                  var fulfilledThenable = thenable;
                  fulfilledThenable.status = "fulfilled";
                  fulfilledThenable.value = fulfilledValue;
                }
              },
              function(error) {
                if ("pending" === thenable.status) {
                  var rejectedThenable = thenable;
                  rejectedThenable.status = "rejected";
                  rejectedThenable.reason = error;
                }
              }
            );
          }
          switch (thenable.status) {
            case "fulfilled":
              return thenable.value;
            case "rejected":
              throw thenableState2 = thenable.reason, checkIfUseWrappedInAsyncCatch(thenableState2), thenableState2;
          }
          suspendedThenable = thenable;
          throw SuspenseException;
      }
    }
    function resolveLazy(lazyType) {
      try {
        var init = lazyType._init;
        return init(lazyType._payload);
      } catch (x) {
        if (null !== x && "object" === typeof x && "function" === typeof x.then)
          throw suspendedThenable = x, SuspenseException;
        throw x;
      }
    }
    var suspendedThenable = null;
    function getSuspendedThenable() {
      if (null === suspendedThenable) throw Error(formatProdErrorMessage(459));
      var thenable = suspendedThenable;
      suspendedThenable = null;
      return thenable;
    }
    function checkIfUseWrappedInAsyncCatch(rejectedReason) {
      if (rejectedReason === SuspenseException || rejectedReason === SuspenseActionException)
        throw Error(formatProdErrorMessage(483));
    }
    var thenableState$1 = null, thenableIndexCounter$1 = 0;
    function unwrapThenable(thenable) {
      var index2 = thenableIndexCounter$1;
      thenableIndexCounter$1 += 1;
      null === thenableState$1 && (thenableState$1 = []);
      return trackUsedThenable(thenableState$1, thenable, index2);
    }
    function coerceRef(workInProgress2, element) {
      element = element.props.ref;
      workInProgress2.ref = void 0 !== element ? element : null;
    }
    function throwOnInvalidObjectTypeImpl(returnFiber, newChild) {
      if (newChild.$$typeof === REACT_LEGACY_ELEMENT_TYPE)
        throw Error(formatProdErrorMessage(525));
      returnFiber = Object.prototype.toString.call(newChild);
      throw Error(
        formatProdErrorMessage(
          31,
          "[object Object]" === returnFiber ? "object with keys {" + Object.keys(newChild).join(", ") + "}" : returnFiber
        )
      );
    }
    function createChildReconciler(shouldTrackSideEffects) {
      function deleteChild(returnFiber, childToDelete) {
        if (shouldTrackSideEffects) {
          var deletions = returnFiber.deletions;
          null === deletions ? (returnFiber.deletions = [childToDelete], returnFiber.flags |= 16) : deletions.push(childToDelete);
        }
      }
      function deleteRemainingChildren(returnFiber, currentFirstChild) {
        if (!shouldTrackSideEffects) return null;
        for (; null !== currentFirstChild; )
          deleteChild(returnFiber, currentFirstChild), currentFirstChild = currentFirstChild.sibling;
        return null;
      }
      function mapRemainingChildren(currentFirstChild) {
        for (var existingChildren = /* @__PURE__ */ new Map(); null !== currentFirstChild; )
          null !== currentFirstChild.key ? existingChildren.set(currentFirstChild.key, currentFirstChild) : existingChildren.set(currentFirstChild.index, currentFirstChild), currentFirstChild = currentFirstChild.sibling;
        return existingChildren;
      }
      function useFiber(fiber, pendingProps) {
        fiber = createWorkInProgress(fiber, pendingProps);
        fiber.index = 0;
        fiber.sibling = null;
        return fiber;
      }
      function placeChild(newFiber, lastPlacedIndex, newIndex) {
        newFiber.index = newIndex;
        if (!shouldTrackSideEffects)
          return newFiber.flags |= 1048576, lastPlacedIndex;
        newIndex = newFiber.alternate;
        if (null !== newIndex)
          return newIndex = newIndex.index, newIndex < lastPlacedIndex ? (newFiber.flags |= 67108866, lastPlacedIndex) : newIndex;
        newFiber.flags |= 67108866;
        return lastPlacedIndex;
      }
      function placeSingleChild(newFiber) {
        shouldTrackSideEffects && null === newFiber.alternate && (newFiber.flags |= 67108866);
        return newFiber;
      }
      function updateTextNode(returnFiber, current, textContent, lanes) {
        if (null === current || 6 !== current.tag)
          return current = createFiberFromText(textContent, returnFiber.mode, lanes), current.return = returnFiber, current;
        current = useFiber(current, textContent);
        current.return = returnFiber;
        return current;
      }
      function updateElement(returnFiber, current, element, lanes) {
        var elementType = element.type;
        if (elementType === REACT_FRAGMENT_TYPE)
          return updateFragment(
            returnFiber,
            current,
            element.props.children,
            lanes,
            element.key
          );
        if (null !== current && (current.elementType === elementType || "object" === typeof elementType && null !== elementType && elementType.$$typeof === REACT_LAZY_TYPE && resolveLazy(elementType) === current.type))
          return current = useFiber(current, element.props), coerceRef(current, element), current.return = returnFiber, current;
        current = createFiberFromTypeAndProps(
          element.type,
          element.key,
          element.props,
          null,
          returnFiber.mode,
          lanes
        );
        coerceRef(current, element);
        current.return = returnFiber;
        return current;
      }
      function updatePortal(returnFiber, current, portal, lanes) {
        if (null === current || 4 !== current.tag || current.stateNode.containerInfo !== portal.containerInfo || current.stateNode.implementation !== portal.implementation)
          return current = createFiberFromPortal(portal, returnFiber.mode, lanes), current.return = returnFiber, current;
        current = useFiber(current, portal.children || []);
        current.return = returnFiber;
        return current;
      }
      function updateFragment(returnFiber, current, fragment, lanes, key) {
        if (null === current || 7 !== current.tag)
          return current = createFiberFromFragment(
            fragment,
            returnFiber.mode,
            lanes,
            key
          ), current.return = returnFiber, current;
        current = useFiber(current, fragment);
        current.return = returnFiber;
        return current;
      }
      function createChild(returnFiber, newChild, lanes) {
        if ("string" === typeof newChild && "" !== newChild || "number" === typeof newChild || "bigint" === typeof newChild)
          return newChild = createFiberFromText(
            "" + newChild,
            returnFiber.mode,
            lanes
          ), newChild.return = returnFiber, newChild;
        if ("object" === typeof newChild && null !== newChild) {
          switch (newChild.$$typeof) {
            case REACT_ELEMENT_TYPE:
              return lanes = createFiberFromTypeAndProps(
                newChild.type,
                newChild.key,
                newChild.props,
                null,
                returnFiber.mode,
                lanes
              ), coerceRef(lanes, newChild), lanes.return = returnFiber, lanes;
            case REACT_PORTAL_TYPE:
              return newChild = createFiberFromPortal(
                newChild,
                returnFiber.mode,
                lanes
              ), newChild.return = returnFiber, newChild;
            case REACT_LAZY_TYPE:
              return newChild = resolveLazy(newChild), createChild(returnFiber, newChild, lanes);
          }
          if (isArrayImpl(newChild) || getIteratorFn(newChild))
            return newChild = createFiberFromFragment(
              newChild,
              returnFiber.mode,
              lanes,
              null
            ), newChild.return = returnFiber, newChild;
          if ("function" === typeof newChild.then)
            return createChild(returnFiber, unwrapThenable(newChild), lanes);
          if (newChild.$$typeof === REACT_CONTEXT_TYPE)
            return createChild(
              returnFiber,
              readContextDuringReconciliation(returnFiber, newChild),
              lanes
            );
          throwOnInvalidObjectTypeImpl(returnFiber, newChild);
        }
        return null;
      }
      function updateSlot(returnFiber, oldFiber, newChild, lanes) {
        var key = null !== oldFiber ? oldFiber.key : null;
        if ("string" === typeof newChild && "" !== newChild || "number" === typeof newChild || "bigint" === typeof newChild)
          return null !== key ? null : updateTextNode(returnFiber, oldFiber, "" + newChild, lanes);
        if ("object" === typeof newChild && null !== newChild) {
          switch (newChild.$$typeof) {
            case REACT_ELEMENT_TYPE:
              return newChild.key === key ? updateElement(returnFiber, oldFiber, newChild, lanes) : null;
            case REACT_PORTAL_TYPE:
              return newChild.key === key ? updatePortal(returnFiber, oldFiber, newChild, lanes) : null;
            case REACT_LAZY_TYPE:
              return newChild = resolveLazy(newChild), updateSlot(returnFiber, oldFiber, newChild, lanes);
          }
          if (isArrayImpl(newChild) || getIteratorFn(newChild))
            return null !== key ? null : updateFragment(returnFiber, oldFiber, newChild, lanes, null);
          if ("function" === typeof newChild.then)
            return updateSlot(
              returnFiber,
              oldFiber,
              unwrapThenable(newChild),
              lanes
            );
          if (newChild.$$typeof === REACT_CONTEXT_TYPE)
            return updateSlot(
              returnFiber,
              oldFiber,
              readContextDuringReconciliation(returnFiber, newChild),
              lanes
            );
          throwOnInvalidObjectTypeImpl(returnFiber, newChild);
        }
        return null;
      }
      function updateFromMap(existingChildren, returnFiber, newIdx, newChild, lanes) {
        if ("string" === typeof newChild && "" !== newChild || "number" === typeof newChild || "bigint" === typeof newChild)
          return existingChildren = existingChildren.get(newIdx) || null, updateTextNode(returnFiber, existingChildren, "" + newChild, lanes);
        if ("object" === typeof newChild && null !== newChild) {
          switch (newChild.$$typeof) {
            case REACT_ELEMENT_TYPE:
              return existingChildren = existingChildren.get(
                null === newChild.key ? newIdx : newChild.key
              ) || null, updateElement(returnFiber, existingChildren, newChild, lanes);
            case REACT_PORTAL_TYPE:
              return existingChildren = existingChildren.get(
                null === newChild.key ? newIdx : newChild.key
              ) || null, updatePortal(returnFiber, existingChildren, newChild, lanes);
            case REACT_LAZY_TYPE:
              return newChild = resolveLazy(newChild), updateFromMap(
                existingChildren,
                returnFiber,
                newIdx,
                newChild,
                lanes
              );
          }
          if (isArrayImpl(newChild) || getIteratorFn(newChild))
            return existingChildren = existingChildren.get(newIdx) || null, updateFragment(returnFiber, existingChildren, newChild, lanes, null);
          if ("function" === typeof newChild.then)
            return updateFromMap(
              existingChildren,
              returnFiber,
              newIdx,
              unwrapThenable(newChild),
              lanes
            );
          if (newChild.$$typeof === REACT_CONTEXT_TYPE)
            return updateFromMap(
              existingChildren,
              returnFiber,
              newIdx,
              readContextDuringReconciliation(returnFiber, newChild),
              lanes
            );
          throwOnInvalidObjectTypeImpl(returnFiber, newChild);
        }
        return null;
      }
      function reconcileChildrenArray(returnFiber, currentFirstChild, newChildren, lanes) {
        for (var resultingFirstChild = null, previousNewFiber = null, oldFiber = currentFirstChild, newIdx = currentFirstChild = 0, nextOldFiber = null; null !== oldFiber && newIdx < newChildren.length; newIdx++) {
          oldFiber.index > newIdx ? (nextOldFiber = oldFiber, oldFiber = null) : nextOldFiber = oldFiber.sibling;
          var newFiber = updateSlot(
            returnFiber,
            oldFiber,
            newChildren[newIdx],
            lanes
          );
          if (null === newFiber) {
            null === oldFiber && (oldFiber = nextOldFiber);
            break;
          }
          shouldTrackSideEffects && oldFiber && null === newFiber.alternate && deleteChild(returnFiber, oldFiber);
          currentFirstChild = placeChild(newFiber, currentFirstChild, newIdx);
          null === previousNewFiber ? resultingFirstChild = newFiber : previousNewFiber.sibling = newFiber;
          previousNewFiber = newFiber;
          oldFiber = nextOldFiber;
        }
        if (newIdx === newChildren.length)
          return deleteRemainingChildren(returnFiber, oldFiber), isHydrating && pushTreeFork(returnFiber, newIdx), resultingFirstChild;
        if (null === oldFiber) {
          for (; newIdx < newChildren.length; newIdx++)
            oldFiber = createChild(returnFiber, newChildren[newIdx], lanes), null !== oldFiber && (currentFirstChild = placeChild(
              oldFiber,
              currentFirstChild,
              newIdx
            ), null === previousNewFiber ? resultingFirstChild = oldFiber : previousNewFiber.sibling = oldFiber, previousNewFiber = oldFiber);
          isHydrating && pushTreeFork(returnFiber, newIdx);
          return resultingFirstChild;
        }
        for (oldFiber = mapRemainingChildren(oldFiber); newIdx < newChildren.length; newIdx++)
          nextOldFiber = updateFromMap(
            oldFiber,
            returnFiber,
            newIdx,
            newChildren[newIdx],
            lanes
          ), null !== nextOldFiber && (shouldTrackSideEffects && null !== nextOldFiber.alternate && oldFiber.delete(
            null === nextOldFiber.key ? newIdx : nextOldFiber.key
          ), currentFirstChild = placeChild(
            nextOldFiber,
            currentFirstChild,
            newIdx
          ), null === previousNewFiber ? resultingFirstChild = nextOldFiber : previousNewFiber.sibling = nextOldFiber, previousNewFiber = nextOldFiber);
        shouldTrackSideEffects && oldFiber.forEach(function(child) {
          return deleteChild(returnFiber, child);
        });
        isHydrating && pushTreeFork(returnFiber, newIdx);
        return resultingFirstChild;
      }
      function reconcileChildrenIterator(returnFiber, currentFirstChild, newChildren, lanes) {
        if (null == newChildren) throw Error(formatProdErrorMessage(151));
        for (var resultingFirstChild = null, previousNewFiber = null, oldFiber = currentFirstChild, newIdx = currentFirstChild = 0, nextOldFiber = null, step = newChildren.next(); null !== oldFiber && !step.done; newIdx++, step = newChildren.next()) {
          oldFiber.index > newIdx ? (nextOldFiber = oldFiber, oldFiber = null) : nextOldFiber = oldFiber.sibling;
          var newFiber = updateSlot(returnFiber, oldFiber, step.value, lanes);
          if (null === newFiber) {
            null === oldFiber && (oldFiber = nextOldFiber);
            break;
          }
          shouldTrackSideEffects && oldFiber && null === newFiber.alternate && deleteChild(returnFiber, oldFiber);
          currentFirstChild = placeChild(newFiber, currentFirstChild, newIdx);
          null === previousNewFiber ? resultingFirstChild = newFiber : previousNewFiber.sibling = newFiber;
          previousNewFiber = newFiber;
          oldFiber = nextOldFiber;
        }
        if (step.done)
          return deleteRemainingChildren(returnFiber, oldFiber), isHydrating && pushTreeFork(returnFiber, newIdx), resultingFirstChild;
        if (null === oldFiber) {
          for (; !step.done; newIdx++, step = newChildren.next())
            step = createChild(returnFiber, step.value, lanes), null !== step && (currentFirstChild = placeChild(step, currentFirstChild, newIdx), null === previousNewFiber ? resultingFirstChild = step : previousNewFiber.sibling = step, previousNewFiber = step);
          isHydrating && pushTreeFork(returnFiber, newIdx);
          return resultingFirstChild;
        }
        for (oldFiber = mapRemainingChildren(oldFiber); !step.done; newIdx++, step = newChildren.next())
          step = updateFromMap(oldFiber, returnFiber, newIdx, step.value, lanes), null !== step && (shouldTrackSideEffects && null !== step.alternate && oldFiber.delete(null === step.key ? newIdx : step.key), currentFirstChild = placeChild(step, currentFirstChild, newIdx), null === previousNewFiber ? resultingFirstChild = step : previousNewFiber.sibling = step, previousNewFiber = step);
        shouldTrackSideEffects && oldFiber.forEach(function(child) {
          return deleteChild(returnFiber, child);
        });
        isHydrating && pushTreeFork(returnFiber, newIdx);
        return resultingFirstChild;
      }
      function reconcileChildFibersImpl(returnFiber, currentFirstChild, newChild, lanes) {
        "object" === typeof newChild && null !== newChild && newChild.type === REACT_FRAGMENT_TYPE && null === newChild.key && (newChild = newChild.props.children);
        if ("object" === typeof newChild && null !== newChild) {
          switch (newChild.$$typeof) {
            case REACT_ELEMENT_TYPE:
              a: {
                for (var key = newChild.key; null !== currentFirstChild; ) {
                  if (currentFirstChild.key === key) {
                    key = newChild.type;
                    if (key === REACT_FRAGMENT_TYPE) {
                      if (7 === currentFirstChild.tag) {
                        deleteRemainingChildren(
                          returnFiber,
                          currentFirstChild.sibling
                        );
                        lanes = useFiber(
                          currentFirstChild,
                          newChild.props.children
                        );
                        lanes.return = returnFiber;
                        returnFiber = lanes;
                        break a;
                      }
                    } else if (currentFirstChild.elementType === key || "object" === typeof key && null !== key && key.$$typeof === REACT_LAZY_TYPE && resolveLazy(key) === currentFirstChild.type) {
                      deleteRemainingChildren(
                        returnFiber,
                        currentFirstChild.sibling
                      );
                      lanes = useFiber(currentFirstChild, newChild.props);
                      coerceRef(lanes, newChild);
                      lanes.return = returnFiber;
                      returnFiber = lanes;
                      break a;
                    }
                    deleteRemainingChildren(returnFiber, currentFirstChild);
                    break;
                  } else deleteChild(returnFiber, currentFirstChild);
                  currentFirstChild = currentFirstChild.sibling;
                }
                newChild.type === REACT_FRAGMENT_TYPE ? (lanes = createFiberFromFragment(
                  newChild.props.children,
                  returnFiber.mode,
                  lanes,
                  newChild.key
                ), lanes.return = returnFiber, returnFiber = lanes) : (lanes = createFiberFromTypeAndProps(
                  newChild.type,
                  newChild.key,
                  newChild.props,
                  null,
                  returnFiber.mode,
                  lanes
                ), coerceRef(lanes, newChild), lanes.return = returnFiber, returnFiber = lanes);
              }
              return placeSingleChild(returnFiber);
            case REACT_PORTAL_TYPE:
              a: {
                for (key = newChild.key; null !== currentFirstChild; ) {
                  if (currentFirstChild.key === key)
                    if (4 === currentFirstChild.tag && currentFirstChild.stateNode.containerInfo === newChild.containerInfo && currentFirstChild.stateNode.implementation === newChild.implementation) {
                      deleteRemainingChildren(
                        returnFiber,
                        currentFirstChild.sibling
                      );
                      lanes = useFiber(currentFirstChild, newChild.children || []);
                      lanes.return = returnFiber;
                      returnFiber = lanes;
                      break a;
                    } else {
                      deleteRemainingChildren(returnFiber, currentFirstChild);
                      break;
                    }
                  else deleteChild(returnFiber, currentFirstChild);
                  currentFirstChild = currentFirstChild.sibling;
                }
                lanes = createFiberFromPortal(newChild, returnFiber.mode, lanes);
                lanes.return = returnFiber;
                returnFiber = lanes;
              }
              return placeSingleChild(returnFiber);
            case REACT_LAZY_TYPE:
              return newChild = resolveLazy(newChild), reconcileChildFibersImpl(
                returnFiber,
                currentFirstChild,
                newChild,
                lanes
              );
          }
          if (isArrayImpl(newChild))
            return reconcileChildrenArray(
              returnFiber,
              currentFirstChild,
              newChild,
              lanes
            );
          if (getIteratorFn(newChild)) {
            key = getIteratorFn(newChild);
            if ("function" !== typeof key) throw Error(formatProdErrorMessage(150));
            newChild = key.call(newChild);
            return reconcileChildrenIterator(
              returnFiber,
              currentFirstChild,
              newChild,
              lanes
            );
          }
          if ("function" === typeof newChild.then)
            return reconcileChildFibersImpl(
              returnFiber,
              currentFirstChild,
              unwrapThenable(newChild),
              lanes
            );
          if (newChild.$$typeof === REACT_CONTEXT_TYPE)
            return reconcileChildFibersImpl(
              returnFiber,
              currentFirstChild,
              readContextDuringReconciliation(returnFiber, newChild),
              lanes
            );
          throwOnInvalidObjectTypeImpl(returnFiber, newChild);
        }
        return "string" === typeof newChild && "" !== newChild || "number" === typeof newChild || "bigint" === typeof newChild ? (newChild = "" + newChild, null !== currentFirstChild && 6 === currentFirstChild.tag ? (deleteRemainingChildren(returnFiber, currentFirstChild.sibling), lanes = useFiber(currentFirstChild, newChild), lanes.return = returnFiber, returnFiber = lanes) : (deleteRemainingChildren(returnFiber, currentFirstChild), lanes = createFiberFromText(newChild, returnFiber.mode, lanes), lanes.return = returnFiber, returnFiber = lanes), placeSingleChild(returnFiber)) : deleteRemainingChildren(returnFiber, currentFirstChild);
      }
      return function(returnFiber, currentFirstChild, newChild, lanes) {
        try {
          thenableIndexCounter$1 = 0;
          var firstChildFiber = reconcileChildFibersImpl(
            returnFiber,
            currentFirstChild,
            newChild,
            lanes
          );
          thenableState$1 = null;
          return firstChildFiber;
        } catch (x) {
          if (x === SuspenseException || x === SuspenseActionException) throw x;
          var fiber = createFiberImplClass(29, x, null, returnFiber.mode);
          fiber.lanes = lanes;
          fiber.return = returnFiber;
          return fiber;
        } finally {
        }
      };
    }
    var reconcileChildFibers = createChildReconciler(true), mountChildFibers = createChildReconciler(false), hasForceUpdate = false;
    function initializeUpdateQueue(fiber) {
      fiber.updateQueue = {
        baseState: fiber.memoizedState,
        firstBaseUpdate: null,
        lastBaseUpdate: null,
        shared: { pending: null, lanes: 0, hiddenCallbacks: null },
        callbacks: null
      };
    }
    function cloneUpdateQueue(current, workInProgress2) {
      current = current.updateQueue;
      workInProgress2.updateQueue === current && (workInProgress2.updateQueue = {
        baseState: current.baseState,
        firstBaseUpdate: current.firstBaseUpdate,
        lastBaseUpdate: current.lastBaseUpdate,
        shared: current.shared,
        callbacks: null
      });
    }
    function createUpdate(lane) {
      return { lane, tag: 0, payload: null, callback: null, next: null };
    }
    function enqueueUpdate(fiber, update, lane) {
      var updateQueue = fiber.updateQueue;
      if (null === updateQueue) return null;
      updateQueue = updateQueue.shared;
      if (0 !== (executionContext & 2)) {
        var pending = updateQueue.pending;
        null === pending ? update.next = update : (update.next = pending.next, pending.next = update);
        updateQueue.pending = update;
        update = getRootForUpdatedFiber(fiber);
        markUpdateLaneFromFiberToRoot(fiber, null, lane);
        return update;
      }
      enqueueUpdate$1(fiber, updateQueue, update, lane);
      return getRootForUpdatedFiber(fiber);
    }
    function entangleTransitions(root3, fiber, lane) {
      fiber = fiber.updateQueue;
      if (null !== fiber && (fiber = fiber.shared, 0 !== (lane & 4194048))) {
        var queueLanes = fiber.lanes;
        queueLanes &= root3.pendingLanes;
        lane |= queueLanes;
        fiber.lanes = lane;
        markRootEntangled(root3, lane);
      }
    }
    function enqueueCapturedUpdate(workInProgress2, capturedUpdate) {
      var queue = workInProgress2.updateQueue, current = workInProgress2.alternate;
      if (null !== current && (current = current.updateQueue, queue === current)) {
        var newFirst = null, newLast = null;
        queue = queue.firstBaseUpdate;
        if (null !== queue) {
          do {
            var clone = {
              lane: queue.lane,
              tag: queue.tag,
              payload: queue.payload,
              callback: null,
              next: null
            };
            null === newLast ? newFirst = newLast = clone : newLast = newLast.next = clone;
            queue = queue.next;
          } while (null !== queue);
          null === newLast ? newFirst = newLast = capturedUpdate : newLast = newLast.next = capturedUpdate;
        } else newFirst = newLast = capturedUpdate;
        queue = {
          baseState: current.baseState,
          firstBaseUpdate: newFirst,
          lastBaseUpdate: newLast,
          shared: current.shared,
          callbacks: current.callbacks
        };
        workInProgress2.updateQueue = queue;
        return;
      }
      workInProgress2 = queue.lastBaseUpdate;
      null === workInProgress2 ? queue.firstBaseUpdate = capturedUpdate : workInProgress2.next = capturedUpdate;
      queue.lastBaseUpdate = capturedUpdate;
    }
    var didReadFromEntangledAsyncAction = false;
    function suspendIfUpdateReadFromEntangledAsyncAction() {
      if (didReadFromEntangledAsyncAction) {
        var entangledActionThenable = currentEntangledActionThenable;
        if (null !== entangledActionThenable) throw entangledActionThenable;
      }
    }
    function processUpdateQueue(workInProgress$jscomp$0, props, instance$jscomp$0, renderLanes2) {
      didReadFromEntangledAsyncAction = false;
      var queue = workInProgress$jscomp$0.updateQueue;
      hasForceUpdate = false;
      var firstBaseUpdate = queue.firstBaseUpdate, lastBaseUpdate = queue.lastBaseUpdate, pendingQueue = queue.shared.pending;
      if (null !== pendingQueue) {
        queue.shared.pending = null;
        var lastPendingUpdate = pendingQueue, firstPendingUpdate = lastPendingUpdate.next;
        lastPendingUpdate.next = null;
        null === lastBaseUpdate ? firstBaseUpdate = firstPendingUpdate : lastBaseUpdate.next = firstPendingUpdate;
        lastBaseUpdate = lastPendingUpdate;
        var current = workInProgress$jscomp$0.alternate;
        null !== current && (current = current.updateQueue, pendingQueue = current.lastBaseUpdate, pendingQueue !== lastBaseUpdate && (null === pendingQueue ? current.firstBaseUpdate = firstPendingUpdate : pendingQueue.next = firstPendingUpdate, current.lastBaseUpdate = lastPendingUpdate));
      }
      if (null !== firstBaseUpdate) {
        var newState = queue.baseState;
        lastBaseUpdate = 0;
        current = firstPendingUpdate = lastPendingUpdate = null;
        pendingQueue = firstBaseUpdate;
        do {
          var updateLane = pendingQueue.lane & -536870913, isHiddenUpdate = updateLane !== pendingQueue.lane;
          if (isHiddenUpdate ? (workInProgressRootRenderLanes & updateLane) === updateLane : (renderLanes2 & updateLane) === updateLane) {
            0 !== updateLane && updateLane === currentEntangledLane && (didReadFromEntangledAsyncAction = true);
            null !== current && (current = current.next = {
              lane: 0,
              tag: pendingQueue.tag,
              payload: pendingQueue.payload,
              callback: null,
              next: null
            });
            a: {
              var workInProgress2 = workInProgress$jscomp$0, update = pendingQueue;
              updateLane = props;
              var instance = instance$jscomp$0;
              switch (update.tag) {
                case 1:
                  workInProgress2 = update.payload;
                  if ("function" === typeof workInProgress2) {
                    newState = workInProgress2.call(instance, newState, updateLane);
                    break a;
                  }
                  newState = workInProgress2;
                  break a;
                case 3:
                  workInProgress2.flags = workInProgress2.flags & -65537 | 128;
                case 0:
                  workInProgress2 = update.payload;
                  updateLane = "function" === typeof workInProgress2 ? workInProgress2.call(instance, newState, updateLane) : workInProgress2;
                  if (null === updateLane || void 0 === updateLane) break a;
                  newState = assign({}, newState, updateLane);
                  break a;
                case 2:
                  hasForceUpdate = true;
              }
            }
            updateLane = pendingQueue.callback;
            null !== updateLane && (workInProgress$jscomp$0.flags |= 64, isHiddenUpdate && (workInProgress$jscomp$0.flags |= 8192), isHiddenUpdate = queue.callbacks, null === isHiddenUpdate ? queue.callbacks = [updateLane] : isHiddenUpdate.push(updateLane));
          } else
            isHiddenUpdate = {
              lane: updateLane,
              tag: pendingQueue.tag,
              payload: pendingQueue.payload,
              callback: pendingQueue.callback,
              next: null
            }, null === current ? (firstPendingUpdate = current = isHiddenUpdate, lastPendingUpdate = newState) : current = current.next = isHiddenUpdate, lastBaseUpdate |= updateLane;
          pendingQueue = pendingQueue.next;
          if (null === pendingQueue)
            if (pendingQueue = queue.shared.pending, null === pendingQueue)
              break;
            else
              isHiddenUpdate = pendingQueue, pendingQueue = isHiddenUpdate.next, isHiddenUpdate.next = null, queue.lastBaseUpdate = isHiddenUpdate, queue.shared.pending = null;
        } while (1);
        null === current && (lastPendingUpdate = newState);
        queue.baseState = lastPendingUpdate;
        queue.firstBaseUpdate = firstPendingUpdate;
        queue.lastBaseUpdate = current;
        null === firstBaseUpdate && (queue.shared.lanes = 0);
        workInProgressRootSkippedLanes |= lastBaseUpdate;
        workInProgress$jscomp$0.lanes = lastBaseUpdate;
        workInProgress$jscomp$0.memoizedState = newState;
      }
    }
    function callCallback(callback, context) {
      if ("function" !== typeof callback)
        throw Error(formatProdErrorMessage(191, callback));
      callback.call(context);
    }
    function commitCallbacks(updateQueue, context) {
      var callbacks = updateQueue.callbacks;
      if (null !== callbacks)
        for (updateQueue.callbacks = null, updateQueue = 0; updateQueue < callbacks.length; updateQueue++)
          callCallback(callbacks[updateQueue], context);
    }
    var currentTreeHiddenStackCursor = createCursor(null), prevEntangledRenderLanesCursor = createCursor(0);
    function pushHiddenContext(fiber, context) {
      fiber = entangledRenderLanes;
      push(prevEntangledRenderLanesCursor, fiber);
      push(currentTreeHiddenStackCursor, context);
      entangledRenderLanes = fiber | context.baseLanes;
    }
    function reuseHiddenContextOnStack() {
      push(prevEntangledRenderLanesCursor, entangledRenderLanes);
      push(currentTreeHiddenStackCursor, currentTreeHiddenStackCursor.current);
    }
    function popHiddenContext() {
      entangledRenderLanes = prevEntangledRenderLanesCursor.current;
      pop(currentTreeHiddenStackCursor);
      pop(prevEntangledRenderLanesCursor);
    }
    var suspenseHandlerStackCursor = createCursor(null), shellBoundary = null;
    function pushPrimaryTreeSuspenseHandler(handler) {
      var current = handler.alternate;
      push(suspenseStackCursor, suspenseStackCursor.current & 1);
      push(suspenseHandlerStackCursor, handler);
      null === shellBoundary && (null === current || null !== currentTreeHiddenStackCursor.current ? shellBoundary = handler : null !== current.memoizedState && (shellBoundary = handler));
    }
    function pushDehydratedActivitySuspenseHandler(fiber) {
      push(suspenseStackCursor, suspenseStackCursor.current);
      push(suspenseHandlerStackCursor, fiber);
      null === shellBoundary && (shellBoundary = fiber);
    }
    function pushOffscreenSuspenseHandler(fiber) {
      22 === fiber.tag ? (push(suspenseStackCursor, suspenseStackCursor.current), push(suspenseHandlerStackCursor, fiber), null === shellBoundary && (shellBoundary = fiber)) : reuseSuspenseHandlerOnStack();
    }
    function reuseSuspenseHandlerOnStack() {
      push(suspenseStackCursor, suspenseStackCursor.current);
      push(suspenseHandlerStackCursor, suspenseHandlerStackCursor.current);
    }
    function popSuspenseHandler(fiber) {
      pop(suspenseHandlerStackCursor);
      shellBoundary === fiber && (shellBoundary = null);
      pop(suspenseStackCursor);
    }
    var suspenseStackCursor = createCursor(0);
    function findFirstSuspended(row) {
      for (var node = row; null !== node; ) {
        if (13 === node.tag) {
          var state = node.memoizedState;
          if (null !== state && (state = state.dehydrated, null === state || isSuspenseInstancePending(state) || isSuspenseInstanceFallback(state)))
            return node;
        } else if (19 === node.tag && ("forwards" === node.memoizedProps.revealOrder || "backwards" === node.memoizedProps.revealOrder || "unstable_legacy-backwards" === node.memoizedProps.revealOrder || "together" === node.memoizedProps.revealOrder)) {
          if (0 !== (node.flags & 128)) return node;
        } else if (null !== node.child) {
          node.child.return = node;
          node = node.child;
          continue;
        }
        if (node === row) break;
        for (; null === node.sibling; ) {
          if (null === node.return || node.return === row) return null;
          node = node.return;
        }
        node.sibling.return = node.return;
        node = node.sibling;
      }
      return null;
    }
    var renderLanes = 0, currentlyRenderingFiber = null, currentHook = null, workInProgressHook = null, didScheduleRenderPhaseUpdate = false, didScheduleRenderPhaseUpdateDuringThisPass = false, shouldDoubleInvokeUserFnsInHooksDEV = false, localIdCounter = 0, thenableIndexCounter = 0, thenableState = null, globalClientIdCounter = 0;
    function throwInvalidHookError() {
      throw Error(formatProdErrorMessage(321));
    }
    function areHookInputsEqual(nextDeps, prevDeps) {
      if (null === prevDeps) return false;
      for (var i = 0; i < prevDeps.length && i < nextDeps.length; i++)
        if (!objectIs(nextDeps[i], prevDeps[i])) return false;
      return true;
    }
    function renderWithHooks(current, workInProgress2, Component, props, secondArg, nextRenderLanes) {
      renderLanes = nextRenderLanes;
      currentlyRenderingFiber = workInProgress2;
      workInProgress2.memoizedState = null;
      workInProgress2.updateQueue = null;
      workInProgress2.lanes = 0;
      ReactSharedInternals.H = null === current || null === current.memoizedState ? HooksDispatcherOnMount : HooksDispatcherOnUpdate;
      shouldDoubleInvokeUserFnsInHooksDEV = false;
      nextRenderLanes = Component(props, secondArg);
      shouldDoubleInvokeUserFnsInHooksDEV = false;
      didScheduleRenderPhaseUpdateDuringThisPass && (nextRenderLanes = renderWithHooksAgain(
        workInProgress2,
        Component,
        props,
        secondArg
      ));
      finishRenderingHooks(current);
      return nextRenderLanes;
    }
    function finishRenderingHooks(current) {
      ReactSharedInternals.H = ContextOnlyDispatcher;
      var didRenderTooFewHooks = null !== currentHook && null !== currentHook.next;
      renderLanes = 0;
      workInProgressHook = currentHook = currentlyRenderingFiber = null;
      didScheduleRenderPhaseUpdate = false;
      thenableIndexCounter = 0;
      thenableState = null;
      if (didRenderTooFewHooks) throw Error(formatProdErrorMessage(300));
      null === current || didReceiveUpdate || (current = current.dependencies, null !== current && checkIfContextChanged(current) && (didReceiveUpdate = true));
    }
    function renderWithHooksAgain(workInProgress2, Component, props, secondArg) {
      currentlyRenderingFiber = workInProgress2;
      var numberOfReRenders = 0;
      do {
        didScheduleRenderPhaseUpdateDuringThisPass && (thenableState = null);
        thenableIndexCounter = 0;
        didScheduleRenderPhaseUpdateDuringThisPass = false;
        if (25 <= numberOfReRenders) throw Error(formatProdErrorMessage(301));
        numberOfReRenders += 1;
        workInProgressHook = currentHook = null;
        if (null != workInProgress2.updateQueue) {
          var children = workInProgress2.updateQueue;
          children.lastEffect = null;
          children.events = null;
          children.stores = null;
          null != children.memoCache && (children.memoCache.index = 0);
        }
        ReactSharedInternals.H = HooksDispatcherOnRerender;
        children = Component(props, secondArg);
      } while (didScheduleRenderPhaseUpdateDuringThisPass);
      return children;
    }
    function TransitionAwareHostComponent() {
      var dispatcher = ReactSharedInternals.H, maybeThenable = dispatcher.useState()[0];
      maybeThenable = "function" === typeof maybeThenable.then ? useThenable(maybeThenable) : maybeThenable;
      dispatcher = dispatcher.useState()[0];
      (null !== currentHook ? currentHook.memoizedState : null) !== dispatcher && (currentlyRenderingFiber.flags |= 1024);
      return maybeThenable;
    }
    function checkDidRenderIdHook() {
      var didRenderIdHook = 0 !== localIdCounter;
      localIdCounter = 0;
      return didRenderIdHook;
    }
    function bailoutHooks(current, workInProgress2, lanes) {
      workInProgress2.updateQueue = current.updateQueue;
      workInProgress2.flags &= -2053;
      current.lanes &= ~lanes;
    }
    function resetHooksOnUnwind(workInProgress2) {
      if (didScheduleRenderPhaseUpdate) {
        for (workInProgress2 = workInProgress2.memoizedState; null !== workInProgress2; ) {
          var queue = workInProgress2.queue;
          null !== queue && (queue.pending = null);
          workInProgress2 = workInProgress2.next;
        }
        didScheduleRenderPhaseUpdate = false;
      }
      renderLanes = 0;
      workInProgressHook = currentHook = currentlyRenderingFiber = null;
      didScheduleRenderPhaseUpdateDuringThisPass = false;
      thenableIndexCounter = localIdCounter = 0;
      thenableState = null;
    }
    function mountWorkInProgressHook() {
      var hook = {
        memoizedState: null,
        baseState: null,
        baseQueue: null,
        queue: null,
        next: null
      };
      null === workInProgressHook ? currentlyRenderingFiber.memoizedState = workInProgressHook = hook : workInProgressHook = workInProgressHook.next = hook;
      return workInProgressHook;
    }
    function updateWorkInProgressHook() {
      if (null === currentHook) {
        var nextCurrentHook = currentlyRenderingFiber.alternate;
        nextCurrentHook = null !== nextCurrentHook ? nextCurrentHook.memoizedState : null;
      } else nextCurrentHook = currentHook.next;
      var nextWorkInProgressHook = null === workInProgressHook ? currentlyRenderingFiber.memoizedState : workInProgressHook.next;
      if (null !== nextWorkInProgressHook)
        workInProgressHook = nextWorkInProgressHook, currentHook = nextCurrentHook;
      else {
        if (null === nextCurrentHook) {
          if (null === currentlyRenderingFiber.alternate)
            throw Error(formatProdErrorMessage(467));
          throw Error(formatProdErrorMessage(310));
        }
        currentHook = nextCurrentHook;
        nextCurrentHook = {
          memoizedState: currentHook.memoizedState,
          baseState: currentHook.baseState,
          baseQueue: currentHook.baseQueue,
          queue: currentHook.queue,
          next: null
        };
        null === workInProgressHook ? currentlyRenderingFiber.memoizedState = workInProgressHook = nextCurrentHook : workInProgressHook = workInProgressHook.next = nextCurrentHook;
      }
      return workInProgressHook;
    }
    function createFunctionComponentUpdateQueue() {
      return { lastEffect: null, events: null, stores: null, memoCache: null };
    }
    function useThenable(thenable) {
      var index2 = thenableIndexCounter;
      thenableIndexCounter += 1;
      null === thenableState && (thenableState = []);
      thenable = trackUsedThenable(thenableState, thenable, index2);
      index2 = currentlyRenderingFiber;
      null === (null === workInProgressHook ? index2.memoizedState : workInProgressHook.next) && (index2 = index2.alternate, ReactSharedInternals.H = null === index2 || null === index2.memoizedState ? HooksDispatcherOnMount : HooksDispatcherOnUpdate);
      return thenable;
    }
    function use(usable) {
      if (null !== usable && "object" === typeof usable) {
        if ("function" === typeof usable.then) return useThenable(usable);
        if (usable.$$typeof === REACT_CONTEXT_TYPE) return readContext(usable);
      }
      throw Error(formatProdErrorMessage(438, String(usable)));
    }
    function useMemoCache(size) {
      var memoCache = null, updateQueue = currentlyRenderingFiber.updateQueue;
      null !== updateQueue && (memoCache = updateQueue.memoCache);
      if (null == memoCache) {
        var current = currentlyRenderingFiber.alternate;
        null !== current && (current = current.updateQueue, null !== current && (current = current.memoCache, null != current && (memoCache = {
          data: current.data.map(function(array) {
            return array.slice();
          }),
          index: 0
        })));
      }
      null == memoCache && (memoCache = { data: [], index: 0 });
      null === updateQueue && (updateQueue = createFunctionComponentUpdateQueue(), currentlyRenderingFiber.updateQueue = updateQueue);
      updateQueue.memoCache = memoCache;
      updateQueue = memoCache.data[memoCache.index];
      if (void 0 === updateQueue)
        for (updateQueue = memoCache.data[memoCache.index] = Array(size), current = 0; current < size; current++)
          updateQueue[current] = REACT_MEMO_CACHE_SENTINEL;
      memoCache.index++;
      return updateQueue;
    }
    function basicStateReducer(state, action) {
      return "function" === typeof action ? action(state) : action;
    }
    function updateReducer(reducer) {
      var hook = updateWorkInProgressHook();
      return updateReducerImpl(hook, currentHook, reducer);
    }
    function updateReducerImpl(hook, current, reducer) {
      var queue = hook.queue;
      if (null === queue) throw Error(formatProdErrorMessage(311));
      queue.lastRenderedReducer = reducer;
      var baseQueue = hook.baseQueue, pendingQueue = queue.pending;
      if (null !== pendingQueue) {
        if (null !== baseQueue) {
          var baseFirst = baseQueue.next;
          baseQueue.next = pendingQueue.next;
          pendingQueue.next = baseFirst;
        }
        current.baseQueue = baseQueue = pendingQueue;
        queue.pending = null;
      }
      pendingQueue = hook.baseState;
      if (null === baseQueue) hook.memoizedState = pendingQueue;
      else {
        current = baseQueue.next;
        var newBaseQueueFirst = baseFirst = null, newBaseQueueLast = null, update = current, didReadFromEntangledAsyncAction$60 = false;
        do {
          var updateLane = update.lane & -536870913;
          if (updateLane !== update.lane ? (workInProgressRootRenderLanes & updateLane) === updateLane : (renderLanes & updateLane) === updateLane) {
            var revertLane = update.revertLane;
            if (0 === revertLane)
              null !== newBaseQueueLast && (newBaseQueueLast = newBaseQueueLast.next = {
                lane: 0,
                revertLane: 0,
                gesture: null,
                action: update.action,
                hasEagerState: update.hasEagerState,
                eagerState: update.eagerState,
                next: null
              }), updateLane === currentEntangledLane && (didReadFromEntangledAsyncAction$60 = true);
            else if ((renderLanes & revertLane) === revertLane) {
              update = update.next;
              revertLane === currentEntangledLane && (didReadFromEntangledAsyncAction$60 = true);
              continue;
            } else
              updateLane = {
                lane: 0,
                revertLane: update.revertLane,
                gesture: null,
                action: update.action,
                hasEagerState: update.hasEagerState,
                eagerState: update.eagerState,
                next: null
              }, null === newBaseQueueLast ? (newBaseQueueFirst = newBaseQueueLast = updateLane, baseFirst = pendingQueue) : newBaseQueueLast = newBaseQueueLast.next = updateLane, currentlyRenderingFiber.lanes |= revertLane, workInProgressRootSkippedLanes |= revertLane;
            updateLane = update.action;
            shouldDoubleInvokeUserFnsInHooksDEV && reducer(pendingQueue, updateLane);
            pendingQueue = update.hasEagerState ? update.eagerState : reducer(pendingQueue, updateLane);
          } else
            revertLane = {
              lane: updateLane,
              revertLane: update.revertLane,
              gesture: update.gesture,
              action: update.action,
              hasEagerState: update.hasEagerState,
              eagerState: update.eagerState,
              next: null
            }, null === newBaseQueueLast ? (newBaseQueueFirst = newBaseQueueLast = revertLane, baseFirst = pendingQueue) : newBaseQueueLast = newBaseQueueLast.next = revertLane, currentlyRenderingFiber.lanes |= updateLane, workInProgressRootSkippedLanes |= updateLane;
          update = update.next;
        } while (null !== update && update !== current);
        null === newBaseQueueLast ? baseFirst = pendingQueue : newBaseQueueLast.next = newBaseQueueFirst;
        if (!objectIs(pendingQueue, hook.memoizedState) && (didReceiveUpdate = true, didReadFromEntangledAsyncAction$60 && (reducer = currentEntangledActionThenable, null !== reducer)))
          throw reducer;
        hook.memoizedState = pendingQueue;
        hook.baseState = baseFirst;
        hook.baseQueue = newBaseQueueLast;
        queue.lastRenderedState = pendingQueue;
      }
      null === baseQueue && (queue.lanes = 0);
      return [hook.memoizedState, queue.dispatch];
    }
    function rerenderReducer(reducer) {
      var hook = updateWorkInProgressHook(), queue = hook.queue;
      if (null === queue) throw Error(formatProdErrorMessage(311));
      queue.lastRenderedReducer = reducer;
      var dispatch = queue.dispatch, lastRenderPhaseUpdate = queue.pending, newState = hook.memoizedState;
      if (null !== lastRenderPhaseUpdate) {
        queue.pending = null;
        var update = lastRenderPhaseUpdate = lastRenderPhaseUpdate.next;
        do
          newState = reducer(newState, update.action), update = update.next;
        while (update !== lastRenderPhaseUpdate);
        objectIs(newState, hook.memoizedState) || (didReceiveUpdate = true);
        hook.memoizedState = newState;
        null === hook.baseQueue && (hook.baseState = newState);
        queue.lastRenderedState = newState;
      }
      return [newState, dispatch];
    }
    function updateSyncExternalStore(subscribe, getSnapshot, getServerSnapshot) {
      var fiber = currentlyRenderingFiber, hook = updateWorkInProgressHook(), isHydrating$jscomp$0 = isHydrating;
      if (isHydrating$jscomp$0) {
        if (void 0 === getServerSnapshot) throw Error(formatProdErrorMessage(407));
        getServerSnapshot = getServerSnapshot();
      } else getServerSnapshot = getSnapshot();
      var snapshotChanged = !objectIs(
        (currentHook || hook).memoizedState,
        getServerSnapshot
      );
      snapshotChanged && (hook.memoizedState = getServerSnapshot, didReceiveUpdate = true);
      hook = hook.queue;
      updateEffect(subscribeToStore.bind(null, fiber, hook, subscribe), [
        subscribe
      ]);
      if (hook.getSnapshot !== getSnapshot || snapshotChanged || null !== workInProgressHook && workInProgressHook.memoizedState.tag & 1) {
        fiber.flags |= 2048;
        pushSimpleEffect(
          9,
          { destroy: void 0 },
          updateStoreInstance.bind(
            null,
            fiber,
            hook,
            getServerSnapshot,
            getSnapshot
          ),
          null
        );
        if (null === workInProgressRoot) throw Error(formatProdErrorMessage(349));
        isHydrating$jscomp$0 || 0 !== (renderLanes & 127) || pushStoreConsistencyCheck(fiber, getSnapshot, getServerSnapshot);
      }
      return getServerSnapshot;
    }
    function pushStoreConsistencyCheck(fiber, getSnapshot, renderedSnapshot) {
      fiber.flags |= 16384;
      fiber = { getSnapshot, value: renderedSnapshot };
      getSnapshot = currentlyRenderingFiber.updateQueue;
      null === getSnapshot ? (getSnapshot = createFunctionComponentUpdateQueue(), currentlyRenderingFiber.updateQueue = getSnapshot, getSnapshot.stores = [fiber]) : (renderedSnapshot = getSnapshot.stores, null === renderedSnapshot ? getSnapshot.stores = [fiber] : renderedSnapshot.push(fiber));
    }
    function updateStoreInstance(fiber, inst, nextSnapshot, getSnapshot) {
      inst.value = nextSnapshot;
      inst.getSnapshot = getSnapshot;
      checkIfSnapshotChanged(inst) && forceStoreRerender(fiber);
    }
    function subscribeToStore(fiber, inst, subscribe) {
      return subscribe(function() {
        checkIfSnapshotChanged(inst) && forceStoreRerender(fiber);
      });
    }
    function checkIfSnapshotChanged(inst) {
      var latestGetSnapshot = inst.getSnapshot;
      inst = inst.value;
      try {
        var nextValue = latestGetSnapshot();
        return !objectIs(inst, nextValue);
      } catch (error) {
        return true;
      }
    }
    function forceStoreRerender(fiber) {
      var root3 = enqueueConcurrentRenderForLane(fiber, 2);
      null !== root3 && scheduleUpdateOnFiber(root3, fiber, 2);
    }
    function mountStateImpl(initialState) {
      var hook = mountWorkInProgressHook();
      if ("function" === typeof initialState) {
        var initialStateInitializer = initialState;
        initialState = initialStateInitializer();
        if (shouldDoubleInvokeUserFnsInHooksDEV) {
          setIsStrictModeForDevtools(true);
          try {
            initialStateInitializer();
          } finally {
            setIsStrictModeForDevtools(false);
          }
        }
      }
      hook.memoizedState = hook.baseState = initialState;
      hook.queue = {
        pending: null,
        lanes: 0,
        dispatch: null,
        lastRenderedReducer: basicStateReducer,
        lastRenderedState: initialState
      };
      return hook;
    }
    function updateOptimisticImpl(hook, current, passthrough, reducer) {
      hook.baseState = passthrough;
      return updateReducerImpl(
        hook,
        currentHook,
        "function" === typeof reducer ? reducer : basicStateReducer
      );
    }
    function dispatchActionState(fiber, actionQueue, setPendingState, setState, payload) {
      if (isRenderPhaseUpdate(fiber)) throw Error(formatProdErrorMessage(485));
      fiber = actionQueue.action;
      if (null !== fiber) {
        var actionNode = {
          payload,
          action: fiber,
          next: null,
          isTransition: true,
          status: "pending",
          value: null,
          reason: null,
          listeners: [],
          then: function(listener) {
            actionNode.listeners.push(listener);
          }
        };
        null !== ReactSharedInternals.T ? setPendingState(true) : actionNode.isTransition = false;
        setState(actionNode);
        setPendingState = actionQueue.pending;
        null === setPendingState ? (actionNode.next = actionQueue.pending = actionNode, runActionStateAction(actionQueue, actionNode)) : (actionNode.next = setPendingState.next, actionQueue.pending = setPendingState.next = actionNode);
      }
    }
    function runActionStateAction(actionQueue, node) {
      var action = node.action, payload = node.payload, prevState = actionQueue.state;
      if (node.isTransition) {
        var prevTransition = ReactSharedInternals.T, currentTransition = {};
        ReactSharedInternals.T = currentTransition;
        try {
          var returnValue = action(prevState, payload), onStartTransitionFinish = ReactSharedInternals.S;
          null !== onStartTransitionFinish && onStartTransitionFinish(currentTransition, returnValue);
          handleActionReturnValue(actionQueue, node, returnValue);
        } catch (error) {
          onActionError(actionQueue, node, error);
        } finally {
          null !== prevTransition && null !== currentTransition.types && (prevTransition.types = currentTransition.types), ReactSharedInternals.T = prevTransition;
        }
      } else
        try {
          prevTransition = action(prevState, payload), handleActionReturnValue(actionQueue, node, prevTransition);
        } catch (error$66) {
          onActionError(actionQueue, node, error$66);
        }
    }
    function handleActionReturnValue(actionQueue, node, returnValue) {
      null !== returnValue && "object" === typeof returnValue && "function" === typeof returnValue.then ? returnValue.then(
        function(nextState) {
          onActionSuccess(actionQueue, node, nextState);
        },
        function(error) {
          return onActionError(actionQueue, node, error);
        }
      ) : onActionSuccess(actionQueue, node, returnValue);
    }
    function onActionSuccess(actionQueue, actionNode, nextState) {
      actionNode.status = "fulfilled";
      actionNode.value = nextState;
      notifyActionListeners(actionNode);
      actionQueue.state = nextState;
      actionNode = actionQueue.pending;
      null !== actionNode && (nextState = actionNode.next, nextState === actionNode ? actionQueue.pending = null : (nextState = nextState.next, actionNode.next = nextState, runActionStateAction(actionQueue, nextState)));
    }
    function onActionError(actionQueue, actionNode, error) {
      var last = actionQueue.pending;
      actionQueue.pending = null;
      if (null !== last) {
        last = last.next;
        do
          actionNode.status = "rejected", actionNode.reason = error, notifyActionListeners(actionNode), actionNode = actionNode.next;
        while (actionNode !== last);
      }
      actionQueue.action = null;
    }
    function notifyActionListeners(actionNode) {
      actionNode = actionNode.listeners;
      for (var i = 0; i < actionNode.length; i++) (0, actionNode[i])();
    }
    function actionStateReducer(oldState, newState) {
      return newState;
    }
    function mountActionState(action, initialStateProp) {
      if (isHydrating) {
        var ssrFormState = workInProgressRoot.formState;
        if (null !== ssrFormState) {
          a: {
            var JSCompiler_inline_result = currentlyRenderingFiber;
            if (isHydrating) {
              if (nextHydratableInstance) {
                b: {
                  var JSCompiler_inline_result$jscomp$0 = nextHydratableInstance;
                  for (var inRootOrSingleton = rootOrSingletonContext; 8 !== JSCompiler_inline_result$jscomp$0.nodeType; ) {
                    if (!inRootOrSingleton) {
                      JSCompiler_inline_result$jscomp$0 = null;
                      break b;
                    }
                    JSCompiler_inline_result$jscomp$0 = getNextHydratable(
                      JSCompiler_inline_result$jscomp$0.nextSibling
                    );
                    if (null === JSCompiler_inline_result$jscomp$0) {
                      JSCompiler_inline_result$jscomp$0 = null;
                      break b;
                    }
                  }
                  inRootOrSingleton = JSCompiler_inline_result$jscomp$0.data;
                  JSCompiler_inline_result$jscomp$0 = "F!" === inRootOrSingleton || "F" === inRootOrSingleton ? JSCompiler_inline_result$jscomp$0 : null;
                }
                if (JSCompiler_inline_result$jscomp$0) {
                  nextHydratableInstance = getNextHydratable(
                    JSCompiler_inline_result$jscomp$0.nextSibling
                  );
                  JSCompiler_inline_result = "F!" === JSCompiler_inline_result$jscomp$0.data;
                  break a;
                }
              }
              throwOnHydrationMismatch(JSCompiler_inline_result);
            }
            JSCompiler_inline_result = false;
          }
          JSCompiler_inline_result && (initialStateProp = ssrFormState[0]);
        }
      }
      ssrFormState = mountWorkInProgressHook();
      ssrFormState.memoizedState = ssrFormState.baseState = initialStateProp;
      JSCompiler_inline_result = {
        pending: null,
        lanes: 0,
        dispatch: null,
        lastRenderedReducer: actionStateReducer,
        lastRenderedState: initialStateProp
      };
      ssrFormState.queue = JSCompiler_inline_result;
      ssrFormState = dispatchSetState.bind(
        null,
        currentlyRenderingFiber,
        JSCompiler_inline_result
      );
      JSCompiler_inline_result.dispatch = ssrFormState;
      JSCompiler_inline_result = mountStateImpl(false);
      inRootOrSingleton = dispatchOptimisticSetState.bind(
        null,
        currentlyRenderingFiber,
        false,
        JSCompiler_inline_result.queue
      );
      JSCompiler_inline_result = mountWorkInProgressHook();
      JSCompiler_inline_result$jscomp$0 = {
        state: initialStateProp,
        dispatch: null,
        action,
        pending: null
      };
      JSCompiler_inline_result.queue = JSCompiler_inline_result$jscomp$0;
      ssrFormState = dispatchActionState.bind(
        null,
        currentlyRenderingFiber,
        JSCompiler_inline_result$jscomp$0,
        inRootOrSingleton,
        ssrFormState
      );
      JSCompiler_inline_result$jscomp$0.dispatch = ssrFormState;
      JSCompiler_inline_result.memoizedState = action;
      return [initialStateProp, ssrFormState, false];
    }
    function updateActionState(action) {
      var stateHook = updateWorkInProgressHook();
      return updateActionStateImpl(stateHook, currentHook, action);
    }
    function updateActionStateImpl(stateHook, currentStateHook, action) {
      currentStateHook = updateReducerImpl(
        stateHook,
        currentStateHook,
        actionStateReducer
      )[0];
      stateHook = updateReducer(basicStateReducer)[0];
      if ("object" === typeof currentStateHook && null !== currentStateHook && "function" === typeof currentStateHook.then)
        try {
          var state = useThenable(currentStateHook);
        } catch (x) {
          if (x === SuspenseException) throw SuspenseActionException;
          throw x;
        }
      else state = currentStateHook;
      currentStateHook = updateWorkInProgressHook();
      var actionQueue = currentStateHook.queue, dispatch = actionQueue.dispatch;
      action !== currentStateHook.memoizedState && (currentlyRenderingFiber.flags |= 2048, pushSimpleEffect(
        9,
        { destroy: void 0 },
        actionStateActionEffect.bind(null, actionQueue, action),
        null
      ));
      return [state, dispatch, stateHook];
    }
    function actionStateActionEffect(actionQueue, action) {
      actionQueue.action = action;
    }
    function rerenderActionState(action) {
      var stateHook = updateWorkInProgressHook(), currentStateHook = currentHook;
      if (null !== currentStateHook)
        return updateActionStateImpl(stateHook, currentStateHook, action);
      updateWorkInProgressHook();
      stateHook = stateHook.memoizedState;
      currentStateHook = updateWorkInProgressHook();
      var dispatch = currentStateHook.queue.dispatch;
      currentStateHook.memoizedState = action;
      return [stateHook, dispatch, false];
    }
    function pushSimpleEffect(tag, inst, create, deps) {
      tag = { tag, create, deps, inst, next: null };
      inst = currentlyRenderingFiber.updateQueue;
      null === inst && (inst = createFunctionComponentUpdateQueue(), currentlyRenderingFiber.updateQueue = inst);
      create = inst.lastEffect;
      null === create ? inst.lastEffect = tag.next = tag : (deps = create.next, create.next = tag, tag.next = deps, inst.lastEffect = tag);
      return tag;
    }
    function updateRef() {
      return updateWorkInProgressHook().memoizedState;
    }
    function mountEffectImpl(fiberFlags, hookFlags, create, deps) {
      var hook = mountWorkInProgressHook();
      currentlyRenderingFiber.flags |= fiberFlags;
      hook.memoizedState = pushSimpleEffect(
        1 | hookFlags,
        { destroy: void 0 },
        create,
        void 0 === deps ? null : deps
      );
    }
    function updateEffectImpl(fiberFlags, hookFlags, create, deps) {
      var hook = updateWorkInProgressHook();
      deps = void 0 === deps ? null : deps;
      var inst = hook.memoizedState.inst;
      null !== currentHook && null !== deps && areHookInputsEqual(deps, currentHook.memoizedState.deps) ? hook.memoizedState = pushSimpleEffect(hookFlags, inst, create, deps) : (currentlyRenderingFiber.flags |= fiberFlags, hook.memoizedState = pushSimpleEffect(
        1 | hookFlags,
        inst,
        create,
        deps
      ));
    }
    function mountEffect(create, deps) {
      mountEffectImpl(8390656, 8, create, deps);
    }
    function updateEffect(create, deps) {
      updateEffectImpl(2048, 8, create, deps);
    }
    function useEffectEventImpl(payload) {
      currentlyRenderingFiber.flags |= 4;
      var componentUpdateQueue = currentlyRenderingFiber.updateQueue;
      if (null === componentUpdateQueue)
        componentUpdateQueue = createFunctionComponentUpdateQueue(), currentlyRenderingFiber.updateQueue = componentUpdateQueue, componentUpdateQueue.events = [payload];
      else {
        var events = componentUpdateQueue.events;
        null === events ? componentUpdateQueue.events = [payload] : events.push(payload);
      }
    }
    function updateEvent(callback) {
      var ref = updateWorkInProgressHook().memoizedState;
      useEffectEventImpl({ ref, nextImpl: callback });
      return function() {
        if (0 !== (executionContext & 2)) throw Error(formatProdErrorMessage(440));
        return ref.impl.apply(void 0, arguments);
      };
    }
    function updateInsertionEffect(create, deps) {
      return updateEffectImpl(4, 2, create, deps);
    }
    function updateLayoutEffect(create, deps) {
      return updateEffectImpl(4, 4, create, deps);
    }
    function imperativeHandleEffect(create, ref) {
      if ("function" === typeof ref) {
        create = create();
        var refCleanup = ref(create);
        return function() {
          "function" === typeof refCleanup ? refCleanup() : ref(null);
        };
      }
      if (null !== ref && void 0 !== ref)
        return create = create(), ref.current = create, function() {
          ref.current = null;
        };
    }
    function updateImperativeHandle(ref, create, deps) {
      deps = null !== deps && void 0 !== deps ? deps.concat([ref]) : null;
      updateEffectImpl(4, 4, imperativeHandleEffect.bind(null, create, ref), deps);
    }
    function mountDebugValue() {
    }
    function updateCallback(callback, deps) {
      var hook = updateWorkInProgressHook();
      deps = void 0 === deps ? null : deps;
      var prevState = hook.memoizedState;
      if (null !== deps && areHookInputsEqual(deps, prevState[1]))
        return prevState[0];
      hook.memoizedState = [callback, deps];
      return callback;
    }
    function updateMemo(nextCreate, deps) {
      var hook = updateWorkInProgressHook();
      deps = void 0 === deps ? null : deps;
      var prevState = hook.memoizedState;
      if (null !== deps && areHookInputsEqual(deps, prevState[1]))
        return prevState[0];
      prevState = nextCreate();
      if (shouldDoubleInvokeUserFnsInHooksDEV) {
        setIsStrictModeForDevtools(true);
        try {
          nextCreate();
        } finally {
          setIsStrictModeForDevtools(false);
        }
      }
      hook.memoizedState = [prevState, deps];
      return prevState;
    }
    function mountDeferredValueImpl(hook, value, initialValue) {
      if (void 0 === initialValue || 0 !== (renderLanes & 1073741824) && 0 === (workInProgressRootRenderLanes & 261930))
        return hook.memoizedState = value;
      hook.memoizedState = initialValue;
      hook = requestDeferredLane();
      currentlyRenderingFiber.lanes |= hook;
      workInProgressRootSkippedLanes |= hook;
      return initialValue;
    }
    function updateDeferredValueImpl(hook, prevValue, value, initialValue) {
      if (objectIs(value, prevValue)) return value;
      if (null !== currentTreeHiddenStackCursor.current)
        return hook = mountDeferredValueImpl(hook, value, initialValue), objectIs(hook, prevValue) || (didReceiveUpdate = true), hook;
      if (0 === (renderLanes & 42) || 0 !== (renderLanes & 1073741824) && 0 === (workInProgressRootRenderLanes & 261930))
        return didReceiveUpdate = true, hook.memoizedState = value;
      hook = requestDeferredLane();
      currentlyRenderingFiber.lanes |= hook;
      workInProgressRootSkippedLanes |= hook;
      return prevValue;
    }
    function startTransition(fiber, queue, pendingState, finishedState, callback) {
      var previousPriority = ReactDOMSharedInternals.p;
      ReactDOMSharedInternals.p = 0 !== previousPriority && 8 > previousPriority ? previousPriority : 8;
      var prevTransition = ReactSharedInternals.T, currentTransition = {};
      ReactSharedInternals.T = currentTransition;
      dispatchOptimisticSetState(fiber, false, queue, pendingState);
      try {
        var returnValue = callback(), onStartTransitionFinish = ReactSharedInternals.S;
        null !== onStartTransitionFinish && onStartTransitionFinish(currentTransition, returnValue);
        if (null !== returnValue && "object" === typeof returnValue && "function" === typeof returnValue.then) {
          var thenableForFinishedState = chainThenableValue(
            returnValue,
            finishedState
          );
          dispatchSetStateInternal(
            fiber,
            queue,
            thenableForFinishedState,
            requestUpdateLane(fiber)
          );
        } else
          dispatchSetStateInternal(
            fiber,
            queue,
            finishedState,
            requestUpdateLane(fiber)
          );
      } catch (error) {
        dispatchSetStateInternal(
          fiber,
          queue,
          { then: function() {
          }, status: "rejected", reason: error },
          requestUpdateLane()
        );
      } finally {
        ReactDOMSharedInternals.p = previousPriority, null !== prevTransition && null !== currentTransition.types && (prevTransition.types = currentTransition.types), ReactSharedInternals.T = prevTransition;
      }
    }
    function noop() {
    }
    function startHostTransition(formFiber, pendingState, action, formData) {
      if (5 !== formFiber.tag) throw Error(formatProdErrorMessage(476));
      var queue = ensureFormComponentIsStateful(formFiber).queue;
      startTransition(
        formFiber,
        queue,
        pendingState,
        sharedNotPendingObject,
        null === action ? noop : function() {
          requestFormReset$1(formFiber);
          return action(formData);
        }
      );
    }
    function ensureFormComponentIsStateful(formFiber) {
      var existingStateHook = formFiber.memoizedState;
      if (null !== existingStateHook) return existingStateHook;
      existingStateHook = {
        memoizedState: sharedNotPendingObject,
        baseState: sharedNotPendingObject,
        baseQueue: null,
        queue: {
          pending: null,
          lanes: 0,
          dispatch: null,
          lastRenderedReducer: basicStateReducer,
          lastRenderedState: sharedNotPendingObject
        },
        next: null
      };
      var initialResetState = {};
      existingStateHook.next = {
        memoizedState: initialResetState,
        baseState: initialResetState,
        baseQueue: null,
        queue: {
          pending: null,
          lanes: 0,
          dispatch: null,
          lastRenderedReducer: basicStateReducer,
          lastRenderedState: initialResetState
        },
        next: null
      };
      formFiber.memoizedState = existingStateHook;
      formFiber = formFiber.alternate;
      null !== formFiber && (formFiber.memoizedState = existingStateHook);
      return existingStateHook;
    }
    function requestFormReset$1(formFiber) {
      var stateHook = ensureFormComponentIsStateful(formFiber);
      null === stateHook.next && (stateHook = formFiber.alternate.memoizedState);
      dispatchSetStateInternal(
        formFiber,
        stateHook.next.queue,
        {},
        requestUpdateLane()
      );
    }
    function useHostTransitionStatus() {
      return readContext(HostTransitionContext);
    }
    function updateId() {
      return updateWorkInProgressHook().memoizedState;
    }
    function updateRefresh() {
      return updateWorkInProgressHook().memoizedState;
    }
    function refreshCache(fiber) {
      for (var provider = fiber.return; null !== provider; ) {
        switch (provider.tag) {
          case 24:
          case 3:
            var lane = requestUpdateLane();
            fiber = createUpdate(lane);
            var root$69 = enqueueUpdate(provider, fiber, lane);
            null !== root$69 && (scheduleUpdateOnFiber(root$69, provider, lane), entangleTransitions(root$69, provider, lane));
            provider = { cache: createCache() };
            fiber.payload = provider;
            return;
        }
        provider = provider.return;
      }
    }
    function dispatchReducerAction(fiber, queue, action) {
      var lane = requestUpdateLane();
      action = {
        lane,
        revertLane: 0,
        gesture: null,
        action,
        hasEagerState: false,
        eagerState: null,
        next: null
      };
      isRenderPhaseUpdate(fiber) ? enqueueRenderPhaseUpdate(queue, action) : (action = enqueueConcurrentHookUpdate(fiber, queue, action, lane), null !== action && (scheduleUpdateOnFiber(action, fiber, lane), entangleTransitionUpdate(action, queue, lane)));
    }
    function dispatchSetState(fiber, queue, action) {
      var lane = requestUpdateLane();
      dispatchSetStateInternal(fiber, queue, action, lane);
    }
    function dispatchSetStateInternal(fiber, queue, action, lane) {
      var update = {
        lane,
        revertLane: 0,
        gesture: null,
        action,
        hasEagerState: false,
        eagerState: null,
        next: null
      };
      if (isRenderPhaseUpdate(fiber)) enqueueRenderPhaseUpdate(queue, update);
      else {
        var alternate = fiber.alternate;
        if (0 === fiber.lanes && (null === alternate || 0 === alternate.lanes) && (alternate = queue.lastRenderedReducer, null !== alternate))
          try {
            var currentState = queue.lastRenderedState, eagerState = alternate(currentState, action);
            update.hasEagerState = true;
            update.eagerState = eagerState;
            if (objectIs(eagerState, currentState))
              return enqueueUpdate$1(fiber, queue, update, 0), null === workInProgressRoot && finishQueueingConcurrentUpdates(), false;
          } catch (error) {
          } finally {
          }
        action = enqueueConcurrentHookUpdate(fiber, queue, update, lane);
        if (null !== action)
          return scheduleUpdateOnFiber(action, fiber, lane), entangleTransitionUpdate(action, queue, lane), true;
      }
      return false;
    }
    function dispatchOptimisticSetState(fiber, throwIfDuringRender, queue, action) {
      action = {
        lane: 2,
        revertLane: requestTransitionLane(),
        gesture: null,
        action,
        hasEagerState: false,
        eagerState: null,
        next: null
      };
      if (isRenderPhaseUpdate(fiber)) {
        if (throwIfDuringRender) throw Error(formatProdErrorMessage(479));
      } else
        throwIfDuringRender = enqueueConcurrentHookUpdate(
          fiber,
          queue,
          action,
          2
        ), null !== throwIfDuringRender && scheduleUpdateOnFiber(throwIfDuringRender, fiber, 2);
    }
    function isRenderPhaseUpdate(fiber) {
      var alternate = fiber.alternate;
      return fiber === currentlyRenderingFiber || null !== alternate && alternate === currentlyRenderingFiber;
    }
    function enqueueRenderPhaseUpdate(queue, update) {
      didScheduleRenderPhaseUpdateDuringThisPass = didScheduleRenderPhaseUpdate = true;
      var pending = queue.pending;
      null === pending ? update.next = update : (update.next = pending.next, pending.next = update);
      queue.pending = update;
    }
    function entangleTransitionUpdate(root3, queue, lane) {
      if (0 !== (lane & 4194048)) {
        var queueLanes = queue.lanes;
        queueLanes &= root3.pendingLanes;
        lane |= queueLanes;
        queue.lanes = lane;
        markRootEntangled(root3, lane);
      }
    }
    var ContextOnlyDispatcher = {
      readContext,
      use,
      useCallback: throwInvalidHookError,
      useContext: throwInvalidHookError,
      useEffect: throwInvalidHookError,
      useImperativeHandle: throwInvalidHookError,
      useLayoutEffect: throwInvalidHookError,
      useInsertionEffect: throwInvalidHookError,
      useMemo: throwInvalidHookError,
      useReducer: throwInvalidHookError,
      useRef: throwInvalidHookError,
      useState: throwInvalidHookError,
      useDebugValue: throwInvalidHookError,
      useDeferredValue: throwInvalidHookError,
      useTransition: throwInvalidHookError,
      useSyncExternalStore: throwInvalidHookError,
      useId: throwInvalidHookError,
      useHostTransitionStatus: throwInvalidHookError,
      useFormState: throwInvalidHookError,
      useActionState: throwInvalidHookError,
      useOptimistic: throwInvalidHookError,
      useMemoCache: throwInvalidHookError,
      useCacheRefresh: throwInvalidHookError
    };
    ContextOnlyDispatcher.useEffectEvent = throwInvalidHookError;
    var HooksDispatcherOnMount = {
      readContext,
      use,
      useCallback: function(callback, deps) {
        mountWorkInProgressHook().memoizedState = [
          callback,
          void 0 === deps ? null : deps
        ];
        return callback;
      },
      useContext: readContext,
      useEffect: mountEffect,
      useImperativeHandle: function(ref, create, deps) {
        deps = null !== deps && void 0 !== deps ? deps.concat([ref]) : null;
        mountEffectImpl(
          4194308,
          4,
          imperativeHandleEffect.bind(null, create, ref),
          deps
        );
      },
      useLayoutEffect: function(create, deps) {
        return mountEffectImpl(4194308, 4, create, deps);
      },
      useInsertionEffect: function(create, deps) {
        mountEffectImpl(4, 2, create, deps);
      },
      useMemo: function(nextCreate, deps) {
        var hook = mountWorkInProgressHook();
        deps = void 0 === deps ? null : deps;
        var nextValue = nextCreate();
        if (shouldDoubleInvokeUserFnsInHooksDEV) {
          setIsStrictModeForDevtools(true);
          try {
            nextCreate();
          } finally {
            setIsStrictModeForDevtools(false);
          }
        }
        hook.memoizedState = [nextValue, deps];
        return nextValue;
      },
      useReducer: function(reducer, initialArg, init) {
        var hook = mountWorkInProgressHook();
        if (void 0 !== init) {
          var initialState = init(initialArg);
          if (shouldDoubleInvokeUserFnsInHooksDEV) {
            setIsStrictModeForDevtools(true);
            try {
              init(initialArg);
            } finally {
              setIsStrictModeForDevtools(false);
            }
          }
        } else initialState = initialArg;
        hook.memoizedState = hook.baseState = initialState;
        reducer = {
          pending: null,
          lanes: 0,
          dispatch: null,
          lastRenderedReducer: reducer,
          lastRenderedState: initialState
        };
        hook.queue = reducer;
        reducer = reducer.dispatch = dispatchReducerAction.bind(
          null,
          currentlyRenderingFiber,
          reducer
        );
        return [hook.memoizedState, reducer];
      },
      useRef: function(initialValue) {
        var hook = mountWorkInProgressHook();
        initialValue = { current: initialValue };
        return hook.memoizedState = initialValue;
      },
      useState: function(initialState) {
        initialState = mountStateImpl(initialState);
        var queue = initialState.queue, dispatch = dispatchSetState.bind(null, currentlyRenderingFiber, queue);
        queue.dispatch = dispatch;
        return [initialState.memoizedState, dispatch];
      },
      useDebugValue: mountDebugValue,
      useDeferredValue: function(value, initialValue) {
        var hook = mountWorkInProgressHook();
        return mountDeferredValueImpl(hook, value, initialValue);
      },
      useTransition: function() {
        var stateHook = mountStateImpl(false);
        stateHook = startTransition.bind(
          null,
          currentlyRenderingFiber,
          stateHook.queue,
          true,
          false
        );
        mountWorkInProgressHook().memoizedState = stateHook;
        return [false, stateHook];
      },
      useSyncExternalStore: function(subscribe, getSnapshot, getServerSnapshot) {
        var fiber = currentlyRenderingFiber, hook = mountWorkInProgressHook();
        if (isHydrating) {
          if (void 0 === getServerSnapshot)
            throw Error(formatProdErrorMessage(407));
          getServerSnapshot = getServerSnapshot();
        } else {
          getServerSnapshot = getSnapshot();
          if (null === workInProgressRoot)
            throw Error(formatProdErrorMessage(349));
          0 !== (workInProgressRootRenderLanes & 127) || pushStoreConsistencyCheck(fiber, getSnapshot, getServerSnapshot);
        }
        hook.memoizedState = getServerSnapshot;
        var inst = { value: getServerSnapshot, getSnapshot };
        hook.queue = inst;
        mountEffect(subscribeToStore.bind(null, fiber, inst, subscribe), [
          subscribe
        ]);
        fiber.flags |= 2048;
        pushSimpleEffect(
          9,
          { destroy: void 0 },
          updateStoreInstance.bind(
            null,
            fiber,
            inst,
            getServerSnapshot,
            getSnapshot
          ),
          null
        );
        return getServerSnapshot;
      },
      useId: function() {
        var hook = mountWorkInProgressHook(), identifierPrefix = workInProgressRoot.identifierPrefix;
        if (isHydrating) {
          var JSCompiler_inline_result = treeContextOverflow;
          var idWithLeadingBit = treeContextId;
          JSCompiler_inline_result = (idWithLeadingBit & ~(1 << 32 - clz32(idWithLeadingBit) - 1)).toString(32) + JSCompiler_inline_result;
          identifierPrefix = "_" + identifierPrefix + "R_" + JSCompiler_inline_result;
          JSCompiler_inline_result = localIdCounter++;
          0 < JSCompiler_inline_result && (identifierPrefix += "H" + JSCompiler_inline_result.toString(32));
          identifierPrefix += "_";
        } else
          JSCompiler_inline_result = globalClientIdCounter++, identifierPrefix = "_" + identifierPrefix + "r_" + JSCompiler_inline_result.toString(32) + "_";
        return hook.memoizedState = identifierPrefix;
      },
      useHostTransitionStatus,
      useFormState: mountActionState,
      useActionState: mountActionState,
      useOptimistic: function(passthrough) {
        var hook = mountWorkInProgressHook();
        hook.memoizedState = hook.baseState = passthrough;
        var queue = {
          pending: null,
          lanes: 0,
          dispatch: null,
          lastRenderedReducer: null,
          lastRenderedState: null
        };
        hook.queue = queue;
        hook = dispatchOptimisticSetState.bind(
          null,
          currentlyRenderingFiber,
          true,
          queue
        );
        queue.dispatch = hook;
        return [passthrough, hook];
      },
      useMemoCache,
      useCacheRefresh: function() {
        return mountWorkInProgressHook().memoizedState = refreshCache.bind(
          null,
          currentlyRenderingFiber
        );
      },
      useEffectEvent: function(callback) {
        var hook = mountWorkInProgressHook(), ref = { impl: callback };
        hook.memoizedState = ref;
        return function() {
          if (0 !== (executionContext & 2))
            throw Error(formatProdErrorMessage(440));
          return ref.impl.apply(void 0, arguments);
        };
      }
    }, HooksDispatcherOnUpdate = {
      readContext,
      use,
      useCallback: updateCallback,
      useContext: readContext,
      useEffect: updateEffect,
      useImperativeHandle: updateImperativeHandle,
      useInsertionEffect: updateInsertionEffect,
      useLayoutEffect: updateLayoutEffect,
      useMemo: updateMemo,
      useReducer: updateReducer,
      useRef: updateRef,
      useState: function() {
        return updateReducer(basicStateReducer);
      },
      useDebugValue: mountDebugValue,
      useDeferredValue: function(value, initialValue) {
        var hook = updateWorkInProgressHook();
        return updateDeferredValueImpl(
          hook,
          currentHook.memoizedState,
          value,
          initialValue
        );
      },
      useTransition: function() {
        var booleanOrThenable = updateReducer(basicStateReducer)[0], start = updateWorkInProgressHook().memoizedState;
        return [
          "boolean" === typeof booleanOrThenable ? booleanOrThenable : useThenable(booleanOrThenable),
          start
        ];
      },
      useSyncExternalStore: updateSyncExternalStore,
      useId: updateId,
      useHostTransitionStatus,
      useFormState: updateActionState,
      useActionState: updateActionState,
      useOptimistic: function(passthrough, reducer) {
        var hook = updateWorkInProgressHook();
        return updateOptimisticImpl(hook, currentHook, passthrough, reducer);
      },
      useMemoCache,
      useCacheRefresh: updateRefresh
    };
    HooksDispatcherOnUpdate.useEffectEvent = updateEvent;
    var HooksDispatcherOnRerender = {
      readContext,
      use,
      useCallback: updateCallback,
      useContext: readContext,
      useEffect: updateEffect,
      useImperativeHandle: updateImperativeHandle,
      useInsertionEffect: updateInsertionEffect,
      useLayoutEffect: updateLayoutEffect,
      useMemo: updateMemo,
      useReducer: rerenderReducer,
      useRef: updateRef,
      useState: function() {
        return rerenderReducer(basicStateReducer);
      },
      useDebugValue: mountDebugValue,
      useDeferredValue: function(value, initialValue) {
        var hook = updateWorkInProgressHook();
        return null === currentHook ? mountDeferredValueImpl(hook, value, initialValue) : updateDeferredValueImpl(
          hook,
          currentHook.memoizedState,
          value,
          initialValue
        );
      },
      useTransition: function() {
        var booleanOrThenable = rerenderReducer(basicStateReducer)[0], start = updateWorkInProgressHook().memoizedState;
        return [
          "boolean" === typeof booleanOrThenable ? booleanOrThenable : useThenable(booleanOrThenable),
          start
        ];
      },
      useSyncExternalStore: updateSyncExternalStore,
      useId: updateId,
      useHostTransitionStatus,
      useFormState: rerenderActionState,
      useActionState: rerenderActionState,
      useOptimistic: function(passthrough, reducer) {
        var hook = updateWorkInProgressHook();
        if (null !== currentHook)
          return updateOptimisticImpl(hook, currentHook, passthrough, reducer);
        hook.baseState = passthrough;
        return [passthrough, hook.queue.dispatch];
      },
      useMemoCache,
      useCacheRefresh: updateRefresh
    };
    HooksDispatcherOnRerender.useEffectEvent = updateEvent;
    function applyDerivedStateFromProps(workInProgress2, ctor, getDerivedStateFromProps, nextProps) {
      ctor = workInProgress2.memoizedState;
      getDerivedStateFromProps = getDerivedStateFromProps(nextProps, ctor);
      getDerivedStateFromProps = null === getDerivedStateFromProps || void 0 === getDerivedStateFromProps ? ctor : assign({}, ctor, getDerivedStateFromProps);
      workInProgress2.memoizedState = getDerivedStateFromProps;
      0 === workInProgress2.lanes && (workInProgress2.updateQueue.baseState = getDerivedStateFromProps);
    }
    var classComponentUpdater = {
      enqueueSetState: function(inst, payload, callback) {
        inst = inst._reactInternals;
        var lane = requestUpdateLane(), update = createUpdate(lane);
        update.payload = payload;
        void 0 !== callback && null !== callback && (update.callback = callback);
        payload = enqueueUpdate(inst, update, lane);
        null !== payload && (scheduleUpdateOnFiber(payload, inst, lane), entangleTransitions(payload, inst, lane));
      },
      enqueueReplaceState: function(inst, payload, callback) {
        inst = inst._reactInternals;
        var lane = requestUpdateLane(), update = createUpdate(lane);
        update.tag = 1;
        update.payload = payload;
        void 0 !== callback && null !== callback && (update.callback = callback);
        payload = enqueueUpdate(inst, update, lane);
        null !== payload && (scheduleUpdateOnFiber(payload, inst, lane), entangleTransitions(payload, inst, lane));
      },
      enqueueForceUpdate: function(inst, callback) {
        inst = inst._reactInternals;
        var lane = requestUpdateLane(), update = createUpdate(lane);
        update.tag = 2;
        void 0 !== callback && null !== callback && (update.callback = callback);
        callback = enqueueUpdate(inst, update, lane);
        null !== callback && (scheduleUpdateOnFiber(callback, inst, lane), entangleTransitions(callback, inst, lane));
      }
    };
    function checkShouldComponentUpdate(workInProgress2, ctor, oldProps, newProps, oldState, newState, nextContext) {
      workInProgress2 = workInProgress2.stateNode;
      return "function" === typeof workInProgress2.shouldComponentUpdate ? workInProgress2.shouldComponentUpdate(newProps, newState, nextContext) : ctor.prototype && ctor.prototype.isPureReactComponent ? !shallowEqual(oldProps, newProps) || !shallowEqual(oldState, newState) : true;
    }
    function callComponentWillReceiveProps(workInProgress2, instance, newProps, nextContext) {
      workInProgress2 = instance.state;
      "function" === typeof instance.componentWillReceiveProps && instance.componentWillReceiveProps(newProps, nextContext);
      "function" === typeof instance.UNSAFE_componentWillReceiveProps && instance.UNSAFE_componentWillReceiveProps(newProps, nextContext);
      instance.state !== workInProgress2 && classComponentUpdater.enqueueReplaceState(instance, instance.state, null);
    }
    function resolveClassComponentProps(Component, baseProps) {
      var newProps = baseProps;
      if ("ref" in baseProps) {
        newProps = {};
        for (var propName in baseProps)
          "ref" !== propName && (newProps[propName] = baseProps[propName]);
      }
      if (Component = Component.defaultProps) {
        newProps === baseProps && (newProps = assign({}, newProps));
        for (var propName$73 in Component)
          void 0 === newProps[propName$73] && (newProps[propName$73] = Component[propName$73]);
      }
      return newProps;
    }
    function defaultOnUncaughtError(error) {
      reportGlobalError(error);
    }
    function defaultOnCaughtError(error) {
      console.error(error);
    }
    function defaultOnRecoverableError(error) {
      reportGlobalError(error);
    }
    function logUncaughtError(root3, errorInfo) {
      try {
        var onUncaughtError = root3.onUncaughtError;
        onUncaughtError(errorInfo.value, { componentStack: errorInfo.stack });
      } catch (e$74) {
        setTimeout(function() {
          throw e$74;
        });
      }
    }
    function logCaughtError(root3, boundary, errorInfo) {
      try {
        var onCaughtError = root3.onCaughtError;
        onCaughtError(errorInfo.value, {
          componentStack: errorInfo.stack,
          errorBoundary: 1 === boundary.tag ? boundary.stateNode : null
        });
      } catch (e$75) {
        setTimeout(function() {
          throw e$75;
        });
      }
    }
    function createRootErrorUpdate(root3, errorInfo, lane) {
      lane = createUpdate(lane);
      lane.tag = 3;
      lane.payload = { element: null };
      lane.callback = function() {
        logUncaughtError(root3, errorInfo);
      };
      return lane;
    }
    function createClassErrorUpdate(lane) {
      lane = createUpdate(lane);
      lane.tag = 3;
      return lane;
    }
    function initializeClassErrorUpdate(update, root3, fiber, errorInfo) {
      var getDerivedStateFromError = fiber.type.getDerivedStateFromError;
      if ("function" === typeof getDerivedStateFromError) {
        var error = errorInfo.value;
        update.payload = function() {
          return getDerivedStateFromError(error);
        };
        update.callback = function() {
          logCaughtError(root3, fiber, errorInfo);
        };
      }
      var inst = fiber.stateNode;
      null !== inst && "function" === typeof inst.componentDidCatch && (update.callback = function() {
        logCaughtError(root3, fiber, errorInfo);
        "function" !== typeof getDerivedStateFromError && (null === legacyErrorBoundariesThatAlreadyFailed ? legacyErrorBoundariesThatAlreadyFailed = /* @__PURE__ */ new Set([this]) : legacyErrorBoundariesThatAlreadyFailed.add(this));
        var stack = errorInfo.stack;
        this.componentDidCatch(errorInfo.value, {
          componentStack: null !== stack ? stack : ""
        });
      });
    }
    function throwException(root3, returnFiber, sourceFiber, value, rootRenderLanes) {
      sourceFiber.flags |= 32768;
      if (null !== value && "object" === typeof value && "function" === typeof value.then) {
        returnFiber = sourceFiber.alternate;
        null !== returnFiber && propagateParentContextChanges(
          returnFiber,
          sourceFiber,
          rootRenderLanes,
          true
        );
        sourceFiber = suspenseHandlerStackCursor.current;
        if (null !== sourceFiber) {
          switch (sourceFiber.tag) {
            case 31:
            case 13:
              return null === shellBoundary ? renderDidSuspendDelayIfPossible() : null === sourceFiber.alternate && 0 === workInProgressRootExitStatus && (workInProgressRootExitStatus = 3), sourceFiber.flags &= -257, sourceFiber.flags |= 65536, sourceFiber.lanes = rootRenderLanes, value === noopSuspenseyCommitThenable ? sourceFiber.flags |= 16384 : (returnFiber = sourceFiber.updateQueue, null === returnFiber ? sourceFiber.updateQueue = /* @__PURE__ */ new Set([value]) : returnFiber.add(value), attachPingListener(root3, value, rootRenderLanes)), false;
            case 22:
              return sourceFiber.flags |= 65536, value === noopSuspenseyCommitThenable ? sourceFiber.flags |= 16384 : (returnFiber = sourceFiber.updateQueue, null === returnFiber ? (returnFiber = {
                transitions: null,
                markerInstances: null,
                retryQueue: /* @__PURE__ */ new Set([value])
              }, sourceFiber.updateQueue = returnFiber) : (sourceFiber = returnFiber.retryQueue, null === sourceFiber ? returnFiber.retryQueue = /* @__PURE__ */ new Set([value]) : sourceFiber.add(value)), attachPingListener(root3, value, rootRenderLanes)), false;
          }
          throw Error(formatProdErrorMessage(435, sourceFiber.tag));
        }
        attachPingListener(root3, value, rootRenderLanes);
        renderDidSuspendDelayIfPossible();
        return false;
      }
      if (isHydrating)
        return returnFiber = suspenseHandlerStackCursor.current, null !== returnFiber ? (0 === (returnFiber.flags & 65536) && (returnFiber.flags |= 256), returnFiber.flags |= 65536, returnFiber.lanes = rootRenderLanes, value !== HydrationMismatchException && (root3 = Error(formatProdErrorMessage(422), { cause: value }), queueHydrationError(createCapturedValueAtFiber(root3, sourceFiber)))) : (value !== HydrationMismatchException && (returnFiber = Error(formatProdErrorMessage(423), {
          cause: value
        }), queueHydrationError(
          createCapturedValueAtFiber(returnFiber, sourceFiber)
        )), root3 = root3.current.alternate, root3.flags |= 65536, rootRenderLanes &= -rootRenderLanes, root3.lanes |= rootRenderLanes, value = createCapturedValueAtFiber(value, sourceFiber), rootRenderLanes = createRootErrorUpdate(
          root3.stateNode,
          value,
          rootRenderLanes
        ), enqueueCapturedUpdate(root3, rootRenderLanes), 4 !== workInProgressRootExitStatus && (workInProgressRootExitStatus = 2)), false;
      var wrapperError = Error(formatProdErrorMessage(520), { cause: value });
      wrapperError = createCapturedValueAtFiber(wrapperError, sourceFiber);
      null === workInProgressRootConcurrentErrors ? workInProgressRootConcurrentErrors = [wrapperError] : workInProgressRootConcurrentErrors.push(wrapperError);
      4 !== workInProgressRootExitStatus && (workInProgressRootExitStatus = 2);
      if (null === returnFiber) return true;
      value = createCapturedValueAtFiber(value, sourceFiber);
      sourceFiber = returnFiber;
      do {
        switch (sourceFiber.tag) {
          case 3:
            return sourceFiber.flags |= 65536, root3 = rootRenderLanes & -rootRenderLanes, sourceFiber.lanes |= root3, root3 = createRootErrorUpdate(sourceFiber.stateNode, value, root3), enqueueCapturedUpdate(sourceFiber, root3), false;
          case 1:
            if (returnFiber = sourceFiber.type, wrapperError = sourceFiber.stateNode, 0 === (sourceFiber.flags & 128) && ("function" === typeof returnFiber.getDerivedStateFromError || null !== wrapperError && "function" === typeof wrapperError.componentDidCatch && (null === legacyErrorBoundariesThatAlreadyFailed || !legacyErrorBoundariesThatAlreadyFailed.has(wrapperError))))
              return sourceFiber.flags |= 65536, rootRenderLanes &= -rootRenderLanes, sourceFiber.lanes |= rootRenderLanes, rootRenderLanes = createClassErrorUpdate(rootRenderLanes), initializeClassErrorUpdate(
                rootRenderLanes,
                root3,
                sourceFiber,
                value
              ), enqueueCapturedUpdate(sourceFiber, rootRenderLanes), false;
        }
        sourceFiber = sourceFiber.return;
      } while (null !== sourceFiber);
      return false;
    }
    var SelectiveHydrationException = Error(formatProdErrorMessage(461)), didReceiveUpdate = false;
    function reconcileChildren(current, workInProgress2, nextChildren, renderLanes2) {
      workInProgress2.child = null === current ? mountChildFibers(workInProgress2, null, nextChildren, renderLanes2) : reconcileChildFibers(
        workInProgress2,
        current.child,
        nextChildren,
        renderLanes2
      );
    }
    function updateForwardRef(current, workInProgress2, Component, nextProps, renderLanes2) {
      Component = Component.render;
      var ref = workInProgress2.ref;
      if ("ref" in nextProps) {
        var propsWithoutRef = {};
        for (var key in nextProps)
          "ref" !== key && (propsWithoutRef[key] = nextProps[key]);
      } else propsWithoutRef = nextProps;
      prepareToReadContext(workInProgress2);
      nextProps = renderWithHooks(
        current,
        workInProgress2,
        Component,
        propsWithoutRef,
        ref,
        renderLanes2
      );
      key = checkDidRenderIdHook();
      if (null !== current && !didReceiveUpdate)
        return bailoutHooks(current, workInProgress2, renderLanes2), bailoutOnAlreadyFinishedWork(current, workInProgress2, renderLanes2);
      isHydrating && key && pushMaterializedTreeId(workInProgress2);
      workInProgress2.flags |= 1;
      reconcileChildren(current, workInProgress2, nextProps, renderLanes2);
      return workInProgress2.child;
    }
    function updateMemoComponent(current, workInProgress2, Component, nextProps, renderLanes2) {
      if (null === current) {
        var type = Component.type;
        if ("function" === typeof type && !shouldConstruct(type) && void 0 === type.defaultProps && null === Component.compare)
          return workInProgress2.tag = 15, workInProgress2.type = type, updateSimpleMemoComponent(
            current,
            workInProgress2,
            type,
            nextProps,
            renderLanes2
          );
        current = createFiberFromTypeAndProps(
          Component.type,
          null,
          nextProps,
          workInProgress2,
          workInProgress2.mode,
          renderLanes2
        );
        current.ref = workInProgress2.ref;
        current.return = workInProgress2;
        return workInProgress2.child = current;
      }
      type = current.child;
      if (!checkScheduledUpdateOrContext(current, renderLanes2)) {
        var prevProps = type.memoizedProps;
        Component = Component.compare;
        Component = null !== Component ? Component : shallowEqual;
        if (Component(prevProps, nextProps) && current.ref === workInProgress2.ref)
          return bailoutOnAlreadyFinishedWork(current, workInProgress2, renderLanes2);
      }
      workInProgress2.flags |= 1;
      current = createWorkInProgress(type, nextProps);
      current.ref = workInProgress2.ref;
      current.return = workInProgress2;
      return workInProgress2.child = current;
    }
    function updateSimpleMemoComponent(current, workInProgress2, Component, nextProps, renderLanes2) {
      if (null !== current) {
        var prevProps = current.memoizedProps;
        if (shallowEqual(prevProps, nextProps) && current.ref === workInProgress2.ref)
          if (didReceiveUpdate = false, workInProgress2.pendingProps = nextProps = prevProps, checkScheduledUpdateOrContext(current, renderLanes2))
            0 !== (current.flags & 131072) && (didReceiveUpdate = true);
          else
            return workInProgress2.lanes = current.lanes, bailoutOnAlreadyFinishedWork(current, workInProgress2, renderLanes2);
      }
      return updateFunctionComponent(
        current,
        workInProgress2,
        Component,
        nextProps,
        renderLanes2
      );
    }
    function updateOffscreenComponent(current, workInProgress2, renderLanes2, nextProps) {
      var nextChildren = nextProps.children, prevState = null !== current ? current.memoizedState : null;
      null === current && null === workInProgress2.stateNode && (workInProgress2.stateNode = {
        _visibility: 1,
        _pendingMarkers: null,
        _retryCache: null,
        _transitions: null
      });
      if ("hidden" === nextProps.mode) {
        if (0 !== (workInProgress2.flags & 128)) {
          prevState = null !== prevState ? prevState.baseLanes | renderLanes2 : renderLanes2;
          if (null !== current) {
            nextProps = workInProgress2.child = current.child;
            for (nextChildren = 0; null !== nextProps; )
              nextChildren = nextChildren | nextProps.lanes | nextProps.childLanes, nextProps = nextProps.sibling;
            nextProps = nextChildren & ~prevState;
          } else nextProps = 0, workInProgress2.child = null;
          return deferHiddenOffscreenComponent(
            current,
            workInProgress2,
            prevState,
            renderLanes2,
            nextProps
          );
        }
        if (0 !== (renderLanes2 & 536870912))
          workInProgress2.memoizedState = { baseLanes: 0, cachePool: null }, null !== current && pushTransition(
            workInProgress2,
            null !== prevState ? prevState.cachePool : null
          ), null !== prevState ? pushHiddenContext(workInProgress2, prevState) : reuseHiddenContextOnStack(), pushOffscreenSuspenseHandler(workInProgress2);
        else
          return nextProps = workInProgress2.lanes = 536870912, deferHiddenOffscreenComponent(
            current,
            workInProgress2,
            null !== prevState ? prevState.baseLanes | renderLanes2 : renderLanes2,
            renderLanes2,
            nextProps
          );
      } else
        null !== prevState ? (pushTransition(workInProgress2, prevState.cachePool), pushHiddenContext(workInProgress2, prevState), reuseSuspenseHandlerOnStack(), workInProgress2.memoizedState = null) : (null !== current && pushTransition(workInProgress2, null), reuseHiddenContextOnStack(), reuseSuspenseHandlerOnStack());
      reconcileChildren(current, workInProgress2, nextChildren, renderLanes2);
      return workInProgress2.child;
    }
    function bailoutOffscreenComponent(current, workInProgress2) {
      null !== current && 22 === current.tag || null !== workInProgress2.stateNode || (workInProgress2.stateNode = {
        _visibility: 1,
        _pendingMarkers: null,
        _retryCache: null,
        _transitions: null
      });
      return workInProgress2.sibling;
    }
    function deferHiddenOffscreenComponent(current, workInProgress2, nextBaseLanes, renderLanes2, remainingChildLanes) {
      var JSCompiler_inline_result = peekCacheFromPool();
      JSCompiler_inline_result = null === JSCompiler_inline_result ? null : { parent: CacheContext._currentValue, pool: JSCompiler_inline_result };
      workInProgress2.memoizedState = {
        baseLanes: nextBaseLanes,
        cachePool: JSCompiler_inline_result
      };
      null !== current && pushTransition(workInProgress2, null);
      reuseHiddenContextOnStack();
      pushOffscreenSuspenseHandler(workInProgress2);
      null !== current && propagateParentContextChanges(current, workInProgress2, renderLanes2, true);
      workInProgress2.childLanes = remainingChildLanes;
      return null;
    }
    function mountActivityChildren(workInProgress2, nextProps) {
      nextProps = mountWorkInProgressOffscreenFiber(
        { mode: nextProps.mode, children: nextProps.children },
        workInProgress2.mode
      );
      nextProps.ref = workInProgress2.ref;
      workInProgress2.child = nextProps;
      nextProps.return = workInProgress2;
      return nextProps;
    }
    function retryActivityComponentWithoutHydrating(current, workInProgress2, renderLanes2) {
      reconcileChildFibers(workInProgress2, current.child, null, renderLanes2);
      current = mountActivityChildren(workInProgress2, workInProgress2.pendingProps);
      current.flags |= 2;
      popSuspenseHandler(workInProgress2);
      workInProgress2.memoizedState = null;
      return current;
    }
    function updateActivityComponent(current, workInProgress2, renderLanes2) {
      var nextProps = workInProgress2.pendingProps, didSuspend = 0 !== (workInProgress2.flags & 128);
      workInProgress2.flags &= -129;
      if (null === current) {
        if (isHydrating) {
          if ("hidden" === nextProps.mode)
            return current = mountActivityChildren(workInProgress2, nextProps), workInProgress2.lanes = 536870912, bailoutOffscreenComponent(null, current);
          pushDehydratedActivitySuspenseHandler(workInProgress2);
          (current = nextHydratableInstance) ? (current = canHydrateHydrationBoundary(
            current,
            rootOrSingletonContext
          ), current = null !== current && "&" === current.data ? current : null, null !== current && (workInProgress2.memoizedState = {
            dehydrated: current,
            treeContext: null !== treeContextProvider ? { id: treeContextId, overflow: treeContextOverflow } : null,
            retryLane: 536870912,
            hydrationErrors: null
          }, renderLanes2 = createFiberFromDehydratedFragment(current), renderLanes2.return = workInProgress2, workInProgress2.child = renderLanes2, hydrationParentFiber = workInProgress2, nextHydratableInstance = null)) : current = null;
          if (null === current) throw throwOnHydrationMismatch(workInProgress2);
          workInProgress2.lanes = 536870912;
          return null;
        }
        return mountActivityChildren(workInProgress2, nextProps);
      }
      var prevState = current.memoizedState;
      if (null !== prevState) {
        var dehydrated = prevState.dehydrated;
        pushDehydratedActivitySuspenseHandler(workInProgress2);
        if (didSuspend)
          if (workInProgress2.flags & 256)
            workInProgress2.flags &= -257, workInProgress2 = retryActivityComponentWithoutHydrating(
              current,
              workInProgress2,
              renderLanes2
            );
          else if (null !== workInProgress2.memoizedState)
            workInProgress2.child = current.child, workInProgress2.flags |= 128, workInProgress2 = null;
          else throw Error(formatProdErrorMessage(558));
        else if (didReceiveUpdate || propagateParentContextChanges(current, workInProgress2, renderLanes2, false), didSuspend = 0 !== (renderLanes2 & current.childLanes), didReceiveUpdate || didSuspend) {
          nextProps = workInProgressRoot;
          if (null !== nextProps && (dehydrated = getBumpedLaneForHydration(nextProps, renderLanes2), 0 !== dehydrated && dehydrated !== prevState.retryLane))
            throw prevState.retryLane = dehydrated, enqueueConcurrentRenderForLane(current, dehydrated), scheduleUpdateOnFiber(nextProps, current, dehydrated), SelectiveHydrationException;
          renderDidSuspendDelayIfPossible();
          workInProgress2 = retryActivityComponentWithoutHydrating(
            current,
            workInProgress2,
            renderLanes2
          );
        } else
          current = prevState.treeContext, nextHydratableInstance = getNextHydratable(dehydrated.nextSibling), hydrationParentFiber = workInProgress2, isHydrating = true, hydrationErrors = null, rootOrSingletonContext = false, null !== current && restoreSuspendedTreeContext(workInProgress2, current), workInProgress2 = mountActivityChildren(workInProgress2, nextProps), workInProgress2.flags |= 4096;
        return workInProgress2;
      }
      current = createWorkInProgress(current.child, {
        mode: nextProps.mode,
        children: nextProps.children
      });
      current.ref = workInProgress2.ref;
      workInProgress2.child = current;
      current.return = workInProgress2;
      return current;
    }
    function markRef(current, workInProgress2) {
      var ref = workInProgress2.ref;
      if (null === ref)
        null !== current && null !== current.ref && (workInProgress2.flags |= 4194816);
      else {
        if ("function" !== typeof ref && "object" !== typeof ref)
          throw Error(formatProdErrorMessage(284));
        if (null === current || current.ref !== ref)
          workInProgress2.flags |= 4194816;
      }
    }
    function updateFunctionComponent(current, workInProgress2, Component, nextProps, renderLanes2) {
      prepareToReadContext(workInProgress2);
      Component = renderWithHooks(
        current,
        workInProgress2,
        Component,
        nextProps,
        void 0,
        renderLanes2
      );
      nextProps = checkDidRenderIdHook();
      if (null !== current && !didReceiveUpdate)
        return bailoutHooks(current, workInProgress2, renderLanes2), bailoutOnAlreadyFinishedWork(current, workInProgress2, renderLanes2);
      isHydrating && nextProps && pushMaterializedTreeId(workInProgress2);
      workInProgress2.flags |= 1;
      reconcileChildren(current, workInProgress2, Component, renderLanes2);
      return workInProgress2.child;
    }
    function replayFunctionComponent(current, workInProgress2, nextProps, Component, secondArg, renderLanes2) {
      prepareToReadContext(workInProgress2);
      workInProgress2.updateQueue = null;
      nextProps = renderWithHooksAgain(
        workInProgress2,
        Component,
        nextProps,
        secondArg
      );
      finishRenderingHooks(current);
      Component = checkDidRenderIdHook();
      if (null !== current && !didReceiveUpdate)
        return bailoutHooks(current, workInProgress2, renderLanes2), bailoutOnAlreadyFinishedWork(current, workInProgress2, renderLanes2);
      isHydrating && Component && pushMaterializedTreeId(workInProgress2);
      workInProgress2.flags |= 1;
      reconcileChildren(current, workInProgress2, nextProps, renderLanes2);
      return workInProgress2.child;
    }
    function updateClassComponent(current, workInProgress2, Component, nextProps, renderLanes2) {
      prepareToReadContext(workInProgress2);
      if (null === workInProgress2.stateNode) {
        var context = emptyContextObject, contextType = Component.contextType;
        "object" === typeof contextType && null !== contextType && (context = readContext(contextType));
        context = new Component(nextProps, context);
        workInProgress2.memoizedState = null !== context.state && void 0 !== context.state ? context.state : null;
        context.updater = classComponentUpdater;
        workInProgress2.stateNode = context;
        context._reactInternals = workInProgress2;
        context = workInProgress2.stateNode;
        context.props = nextProps;
        context.state = workInProgress2.memoizedState;
        context.refs = {};
        initializeUpdateQueue(workInProgress2);
        contextType = Component.contextType;
        context.context = "object" === typeof contextType && null !== contextType ? readContext(contextType) : emptyContextObject;
        context.state = workInProgress2.memoizedState;
        contextType = Component.getDerivedStateFromProps;
        "function" === typeof contextType && (applyDerivedStateFromProps(
          workInProgress2,
          Component,
          contextType,
          nextProps
        ), context.state = workInProgress2.memoizedState);
        "function" === typeof Component.getDerivedStateFromProps || "function" === typeof context.getSnapshotBeforeUpdate || "function" !== typeof context.UNSAFE_componentWillMount && "function" !== typeof context.componentWillMount || (contextType = context.state, "function" === typeof context.componentWillMount && context.componentWillMount(), "function" === typeof context.UNSAFE_componentWillMount && context.UNSAFE_componentWillMount(), contextType !== context.state && classComponentUpdater.enqueueReplaceState(context, context.state, null), processUpdateQueue(workInProgress2, nextProps, context, renderLanes2), suspendIfUpdateReadFromEntangledAsyncAction(), context.state = workInProgress2.memoizedState);
        "function" === typeof context.componentDidMount && (workInProgress2.flags |= 4194308);
        nextProps = true;
      } else if (null === current) {
        context = workInProgress2.stateNode;
        var unresolvedOldProps = workInProgress2.memoizedProps, oldProps = resolveClassComponentProps(Component, unresolvedOldProps);
        context.props = oldProps;
        var oldContext = context.context, contextType$jscomp$0 = Component.contextType;
        contextType = emptyContextObject;
        "object" === typeof contextType$jscomp$0 && null !== contextType$jscomp$0 && (contextType = readContext(contextType$jscomp$0));
        var getDerivedStateFromProps = Component.getDerivedStateFromProps;
        contextType$jscomp$0 = "function" === typeof getDerivedStateFromProps || "function" === typeof context.getSnapshotBeforeUpdate;
        unresolvedOldProps = workInProgress2.pendingProps !== unresolvedOldProps;
        contextType$jscomp$0 || "function" !== typeof context.UNSAFE_componentWillReceiveProps && "function" !== typeof context.componentWillReceiveProps || (unresolvedOldProps || oldContext !== contextType) && callComponentWillReceiveProps(
          workInProgress2,
          context,
          nextProps,
          contextType
        );
        hasForceUpdate = false;
        var oldState = workInProgress2.memoizedState;
        context.state = oldState;
        processUpdateQueue(workInProgress2, nextProps, context, renderLanes2);
        suspendIfUpdateReadFromEntangledAsyncAction();
        oldContext = workInProgress2.memoizedState;
        unresolvedOldProps || oldState !== oldContext || hasForceUpdate ? ("function" === typeof getDerivedStateFromProps && (applyDerivedStateFromProps(
          workInProgress2,
          Component,
          getDerivedStateFromProps,
          nextProps
        ), oldContext = workInProgress2.memoizedState), (oldProps = hasForceUpdate || checkShouldComponentUpdate(
          workInProgress2,
          Component,
          oldProps,
          nextProps,
          oldState,
          oldContext,
          contextType
        )) ? (contextType$jscomp$0 || "function" !== typeof context.UNSAFE_componentWillMount && "function" !== typeof context.componentWillMount || ("function" === typeof context.componentWillMount && context.componentWillMount(), "function" === typeof context.UNSAFE_componentWillMount && context.UNSAFE_componentWillMount()), "function" === typeof context.componentDidMount && (workInProgress2.flags |= 4194308)) : ("function" === typeof context.componentDidMount && (workInProgress2.flags |= 4194308), workInProgress2.memoizedProps = nextProps, workInProgress2.memoizedState = oldContext), context.props = nextProps, context.state = oldContext, context.context = contextType, nextProps = oldProps) : ("function" === typeof context.componentDidMount && (workInProgress2.flags |= 4194308), nextProps = false);
      } else {
        context = workInProgress2.stateNode;
        cloneUpdateQueue(current, workInProgress2);
        contextType = workInProgress2.memoizedProps;
        contextType$jscomp$0 = resolveClassComponentProps(Component, contextType);
        context.props = contextType$jscomp$0;
        getDerivedStateFromProps = workInProgress2.pendingProps;
        oldState = context.context;
        oldContext = Component.contextType;
        oldProps = emptyContextObject;
        "object" === typeof oldContext && null !== oldContext && (oldProps = readContext(oldContext));
        unresolvedOldProps = Component.getDerivedStateFromProps;
        (oldContext = "function" === typeof unresolvedOldProps || "function" === typeof context.getSnapshotBeforeUpdate) || "function" !== typeof context.UNSAFE_componentWillReceiveProps && "function" !== typeof context.componentWillReceiveProps || (contextType !== getDerivedStateFromProps || oldState !== oldProps) && callComponentWillReceiveProps(
          workInProgress2,
          context,
          nextProps,
          oldProps
        );
        hasForceUpdate = false;
        oldState = workInProgress2.memoizedState;
        context.state = oldState;
        processUpdateQueue(workInProgress2, nextProps, context, renderLanes2);
        suspendIfUpdateReadFromEntangledAsyncAction();
        var newState = workInProgress2.memoizedState;
        contextType !== getDerivedStateFromProps || oldState !== newState || hasForceUpdate || null !== current && null !== current.dependencies && checkIfContextChanged(current.dependencies) ? ("function" === typeof unresolvedOldProps && (applyDerivedStateFromProps(
          workInProgress2,
          Component,
          unresolvedOldProps,
          nextProps
        ), newState = workInProgress2.memoizedState), (contextType$jscomp$0 = hasForceUpdate || checkShouldComponentUpdate(
          workInProgress2,
          Component,
          contextType$jscomp$0,
          nextProps,
          oldState,
          newState,
          oldProps
        ) || null !== current && null !== current.dependencies && checkIfContextChanged(current.dependencies)) ? (oldContext || "function" !== typeof context.UNSAFE_componentWillUpdate && "function" !== typeof context.componentWillUpdate || ("function" === typeof context.componentWillUpdate && context.componentWillUpdate(nextProps, newState, oldProps), "function" === typeof context.UNSAFE_componentWillUpdate && context.UNSAFE_componentWillUpdate(
          nextProps,
          newState,
          oldProps
        )), "function" === typeof context.componentDidUpdate && (workInProgress2.flags |= 4), "function" === typeof context.getSnapshotBeforeUpdate && (workInProgress2.flags |= 1024)) : ("function" !== typeof context.componentDidUpdate || contextType === current.memoizedProps && oldState === current.memoizedState || (workInProgress2.flags |= 4), "function" !== typeof context.getSnapshotBeforeUpdate || contextType === current.memoizedProps && oldState === current.memoizedState || (workInProgress2.flags |= 1024), workInProgress2.memoizedProps = nextProps, workInProgress2.memoizedState = newState), context.props = nextProps, context.state = newState, context.context = oldProps, nextProps = contextType$jscomp$0) : ("function" !== typeof context.componentDidUpdate || contextType === current.memoizedProps && oldState === current.memoizedState || (workInProgress2.flags |= 4), "function" !== typeof context.getSnapshotBeforeUpdate || contextType === current.memoizedProps && oldState === current.memoizedState || (workInProgress2.flags |= 1024), nextProps = false);
      }
      context = nextProps;
      markRef(current, workInProgress2);
      nextProps = 0 !== (workInProgress2.flags & 128);
      context || nextProps ? (context = workInProgress2.stateNode, Component = nextProps && "function" !== typeof Component.getDerivedStateFromError ? null : context.render(), workInProgress2.flags |= 1, null !== current && nextProps ? (workInProgress2.child = reconcileChildFibers(
        workInProgress2,
        current.child,
        null,
        renderLanes2
      ), workInProgress2.child = reconcileChildFibers(
        workInProgress2,
        null,
        Component,
        renderLanes2
      )) : reconcileChildren(current, workInProgress2, Component, renderLanes2), workInProgress2.memoizedState = context.state, current = workInProgress2.child) : current = bailoutOnAlreadyFinishedWork(
        current,
        workInProgress2,
        renderLanes2
      );
      return current;
    }
    function mountHostRootWithoutHydrating(current, workInProgress2, nextChildren, renderLanes2) {
      resetHydrationState();
      workInProgress2.flags |= 256;
      reconcileChildren(current, workInProgress2, nextChildren, renderLanes2);
      return workInProgress2.child;
    }
    var SUSPENDED_MARKER = {
      dehydrated: null,
      treeContext: null,
      retryLane: 0,
      hydrationErrors: null
    };
    function mountSuspenseOffscreenState(renderLanes2) {
      return { baseLanes: renderLanes2, cachePool: getSuspendedCache() };
    }
    function getRemainingWorkInPrimaryTree(current, primaryTreeDidDefer, renderLanes2) {
      current = null !== current ? current.childLanes & ~renderLanes2 : 0;
      primaryTreeDidDefer && (current |= workInProgressDeferredLane);
      return current;
    }
    function updateSuspenseComponent(current, workInProgress2, renderLanes2) {
      var nextProps = workInProgress2.pendingProps, showFallback = false, didSuspend = 0 !== (workInProgress2.flags & 128), JSCompiler_temp;
      (JSCompiler_temp = didSuspend) || (JSCompiler_temp = null !== current && null === current.memoizedState ? false : 0 !== (suspenseStackCursor.current & 2));
      JSCompiler_temp && (showFallback = true, workInProgress2.flags &= -129);
      JSCompiler_temp = 0 !== (workInProgress2.flags & 32);
      workInProgress2.flags &= -33;
      if (null === current) {
        if (isHydrating) {
          showFallback ? pushPrimaryTreeSuspenseHandler(workInProgress2) : reuseSuspenseHandlerOnStack();
          (current = nextHydratableInstance) ? (current = canHydrateHydrationBoundary(
            current,
            rootOrSingletonContext
          ), current = null !== current && "&" !== current.data ? current : null, null !== current && (workInProgress2.memoizedState = {
            dehydrated: current,
            treeContext: null !== treeContextProvider ? { id: treeContextId, overflow: treeContextOverflow } : null,
            retryLane: 536870912,
            hydrationErrors: null
          }, renderLanes2 = createFiberFromDehydratedFragment(current), renderLanes2.return = workInProgress2, workInProgress2.child = renderLanes2, hydrationParentFiber = workInProgress2, nextHydratableInstance = null)) : current = null;
          if (null === current) throw throwOnHydrationMismatch(workInProgress2);
          isSuspenseInstanceFallback(current) ? workInProgress2.lanes = 32 : workInProgress2.lanes = 536870912;
          return null;
        }
        var nextPrimaryChildren = nextProps.children;
        nextProps = nextProps.fallback;
        if (showFallback)
          return reuseSuspenseHandlerOnStack(), showFallback = workInProgress2.mode, nextPrimaryChildren = mountWorkInProgressOffscreenFiber(
            { mode: "hidden", children: nextPrimaryChildren },
            showFallback
          ), nextProps = createFiberFromFragment(
            nextProps,
            showFallback,
            renderLanes2,
            null
          ), nextPrimaryChildren.return = workInProgress2, nextProps.return = workInProgress2, nextPrimaryChildren.sibling = nextProps, workInProgress2.child = nextPrimaryChildren, nextProps = workInProgress2.child, nextProps.memoizedState = mountSuspenseOffscreenState(renderLanes2), nextProps.childLanes = getRemainingWorkInPrimaryTree(
            current,
            JSCompiler_temp,
            renderLanes2
          ), workInProgress2.memoizedState = SUSPENDED_MARKER, bailoutOffscreenComponent(null, nextProps);
        pushPrimaryTreeSuspenseHandler(workInProgress2);
        return mountSuspensePrimaryChildren(workInProgress2, nextPrimaryChildren);
      }
      var prevState = current.memoizedState;
      if (null !== prevState && (nextPrimaryChildren = prevState.dehydrated, null !== nextPrimaryChildren)) {
        if (didSuspend)
          workInProgress2.flags & 256 ? (pushPrimaryTreeSuspenseHandler(workInProgress2), workInProgress2.flags &= -257, workInProgress2 = retrySuspenseComponentWithoutHydrating(
            current,
            workInProgress2,
            renderLanes2
          )) : null !== workInProgress2.memoizedState ? (reuseSuspenseHandlerOnStack(), workInProgress2.child = current.child, workInProgress2.flags |= 128, workInProgress2 = null) : (reuseSuspenseHandlerOnStack(), nextPrimaryChildren = nextProps.fallback, showFallback = workInProgress2.mode, nextProps = mountWorkInProgressOffscreenFiber(
            { mode: "visible", children: nextProps.children },
            showFallback
          ), nextPrimaryChildren = createFiberFromFragment(
            nextPrimaryChildren,
            showFallback,
            renderLanes2,
            null
          ), nextPrimaryChildren.flags |= 2, nextProps.return = workInProgress2, nextPrimaryChildren.return = workInProgress2, nextProps.sibling = nextPrimaryChildren, workInProgress2.child = nextProps, reconcileChildFibers(
            workInProgress2,
            current.child,
            null,
            renderLanes2
          ), nextProps = workInProgress2.child, nextProps.memoizedState = mountSuspenseOffscreenState(renderLanes2), nextProps.childLanes = getRemainingWorkInPrimaryTree(
            current,
            JSCompiler_temp,
            renderLanes2
          ), workInProgress2.memoizedState = SUSPENDED_MARKER, workInProgress2 = bailoutOffscreenComponent(null, nextProps));
        else if (pushPrimaryTreeSuspenseHandler(workInProgress2), isSuspenseInstanceFallback(nextPrimaryChildren)) {
          JSCompiler_temp = nextPrimaryChildren.nextSibling && nextPrimaryChildren.nextSibling.dataset;
          if (JSCompiler_temp) var digest = JSCompiler_temp.dgst;
          JSCompiler_temp = digest;
          nextProps = Error(formatProdErrorMessage(419));
          nextProps.stack = "";
          nextProps.digest = JSCompiler_temp;
          queueHydrationError({ value: nextProps, source: null, stack: null });
          workInProgress2 = retrySuspenseComponentWithoutHydrating(
            current,
            workInProgress2,
            renderLanes2
          );
        } else if (didReceiveUpdate || propagateParentContextChanges(current, workInProgress2, renderLanes2, false), JSCompiler_temp = 0 !== (renderLanes2 & current.childLanes), didReceiveUpdate || JSCompiler_temp) {
          JSCompiler_temp = workInProgressRoot;
          if (null !== JSCompiler_temp && (nextProps = getBumpedLaneForHydration(JSCompiler_temp, renderLanes2), 0 !== nextProps && nextProps !== prevState.retryLane))
            throw prevState.retryLane = nextProps, enqueueConcurrentRenderForLane(current, nextProps), scheduleUpdateOnFiber(JSCompiler_temp, current, nextProps), SelectiveHydrationException;
          isSuspenseInstancePending(nextPrimaryChildren) || renderDidSuspendDelayIfPossible();
          workInProgress2 = retrySuspenseComponentWithoutHydrating(
            current,
            workInProgress2,
            renderLanes2
          );
        } else
          isSuspenseInstancePending(nextPrimaryChildren) ? (workInProgress2.flags |= 192, workInProgress2.child = current.child, workInProgress2 = null) : (current = prevState.treeContext, nextHydratableInstance = getNextHydratable(
            nextPrimaryChildren.nextSibling
          ), hydrationParentFiber = workInProgress2, isHydrating = true, hydrationErrors = null, rootOrSingletonContext = false, null !== current && restoreSuspendedTreeContext(workInProgress2, current), workInProgress2 = mountSuspensePrimaryChildren(
            workInProgress2,
            nextProps.children
          ), workInProgress2.flags |= 4096);
        return workInProgress2;
      }
      if (showFallback)
        return reuseSuspenseHandlerOnStack(), nextPrimaryChildren = nextProps.fallback, showFallback = workInProgress2.mode, prevState = current.child, digest = prevState.sibling, nextProps = createWorkInProgress(prevState, {
          mode: "hidden",
          children: nextProps.children
        }), nextProps.subtreeFlags = prevState.subtreeFlags & 65011712, null !== digest ? nextPrimaryChildren = createWorkInProgress(
          digest,
          nextPrimaryChildren
        ) : (nextPrimaryChildren = createFiberFromFragment(
          nextPrimaryChildren,
          showFallback,
          renderLanes2,
          null
        ), nextPrimaryChildren.flags |= 2), nextPrimaryChildren.return = workInProgress2, nextProps.return = workInProgress2, nextProps.sibling = nextPrimaryChildren, workInProgress2.child = nextProps, bailoutOffscreenComponent(null, nextProps), nextProps = workInProgress2.child, nextPrimaryChildren = current.child.memoizedState, null === nextPrimaryChildren ? nextPrimaryChildren = mountSuspenseOffscreenState(renderLanes2) : (showFallback = nextPrimaryChildren.cachePool, null !== showFallback ? (prevState = CacheContext._currentValue, showFallback = showFallback.parent !== prevState ? { parent: prevState, pool: prevState } : showFallback) : showFallback = getSuspendedCache(), nextPrimaryChildren = {
          baseLanes: nextPrimaryChildren.baseLanes | renderLanes2,
          cachePool: showFallback
        }), nextProps.memoizedState = nextPrimaryChildren, nextProps.childLanes = getRemainingWorkInPrimaryTree(
          current,
          JSCompiler_temp,
          renderLanes2
        ), workInProgress2.memoizedState = SUSPENDED_MARKER, bailoutOffscreenComponent(current.child, nextProps);
      pushPrimaryTreeSuspenseHandler(workInProgress2);
      renderLanes2 = current.child;
      current = renderLanes2.sibling;
      renderLanes2 = createWorkInProgress(renderLanes2, {
        mode: "visible",
        children: nextProps.children
      });
      renderLanes2.return = workInProgress2;
      renderLanes2.sibling = null;
      null !== current && (JSCompiler_temp = workInProgress2.deletions, null === JSCompiler_temp ? (workInProgress2.deletions = [current], workInProgress2.flags |= 16) : JSCompiler_temp.push(current));
      workInProgress2.child = renderLanes2;
      workInProgress2.memoizedState = null;
      return renderLanes2;
    }
    function mountSuspensePrimaryChildren(workInProgress2, primaryChildren) {
      primaryChildren = mountWorkInProgressOffscreenFiber(
        { mode: "visible", children: primaryChildren },
        workInProgress2.mode
      );
      primaryChildren.return = workInProgress2;
      return workInProgress2.child = primaryChildren;
    }
    function mountWorkInProgressOffscreenFiber(offscreenProps, mode) {
      offscreenProps = createFiberImplClass(22, offscreenProps, null, mode);
      offscreenProps.lanes = 0;
      return offscreenProps;
    }
    function retrySuspenseComponentWithoutHydrating(current, workInProgress2, renderLanes2) {
      reconcileChildFibers(workInProgress2, current.child, null, renderLanes2);
      current = mountSuspensePrimaryChildren(
        workInProgress2,
        workInProgress2.pendingProps.children
      );
      current.flags |= 2;
      workInProgress2.memoizedState = null;
      return current;
    }
    function scheduleSuspenseWorkOnFiber(fiber, renderLanes2, propagationRoot) {
      fiber.lanes |= renderLanes2;
      var alternate = fiber.alternate;
      null !== alternate && (alternate.lanes |= renderLanes2);
      scheduleContextWorkOnParentPath(fiber.return, renderLanes2, propagationRoot);
    }
    function initSuspenseListRenderState(workInProgress2, isBackwards, tail, lastContentRow, tailMode, treeForkCount2) {
      var renderState = workInProgress2.memoizedState;
      null === renderState ? workInProgress2.memoizedState = {
        isBackwards,
        rendering: null,
        renderingStartTime: 0,
        last: lastContentRow,
        tail,
        tailMode,
        treeForkCount: treeForkCount2
      } : (renderState.isBackwards = isBackwards, renderState.rendering = null, renderState.renderingStartTime = 0, renderState.last = lastContentRow, renderState.tail = tail, renderState.tailMode = tailMode, renderState.treeForkCount = treeForkCount2);
    }
    function updateSuspenseListComponent(current, workInProgress2, renderLanes2) {
      var nextProps = workInProgress2.pendingProps, revealOrder = nextProps.revealOrder, tailMode = nextProps.tail;
      nextProps = nextProps.children;
      var suspenseContext = suspenseStackCursor.current, shouldForceFallback = 0 !== (suspenseContext & 2);
      shouldForceFallback ? (suspenseContext = suspenseContext & 1 | 2, workInProgress2.flags |= 128) : suspenseContext &= 1;
      push(suspenseStackCursor, suspenseContext);
      reconcileChildren(current, workInProgress2, nextProps, renderLanes2);
      nextProps = isHydrating ? treeForkCount : 0;
      if (!shouldForceFallback && null !== current && 0 !== (current.flags & 128))
        a: for (current = workInProgress2.child; null !== current; ) {
          if (13 === current.tag)
            null !== current.memoizedState && scheduleSuspenseWorkOnFiber(current, renderLanes2, workInProgress2);
          else if (19 === current.tag)
            scheduleSuspenseWorkOnFiber(current, renderLanes2, workInProgress2);
          else if (null !== current.child) {
            current.child.return = current;
            current = current.child;
            continue;
          }
          if (current === workInProgress2) break a;
          for (; null === current.sibling; ) {
            if (null === current.return || current.return === workInProgress2)
              break a;
            current = current.return;
          }
          current.sibling.return = current.return;
          current = current.sibling;
        }
      switch (revealOrder) {
        case "forwards":
          renderLanes2 = workInProgress2.child;
          for (revealOrder = null; null !== renderLanes2; )
            current = renderLanes2.alternate, null !== current && null === findFirstSuspended(current) && (revealOrder = renderLanes2), renderLanes2 = renderLanes2.sibling;
          renderLanes2 = revealOrder;
          null === renderLanes2 ? (revealOrder = workInProgress2.child, workInProgress2.child = null) : (revealOrder = renderLanes2.sibling, renderLanes2.sibling = null);
          initSuspenseListRenderState(
            workInProgress2,
            false,
            revealOrder,
            renderLanes2,
            tailMode,
            nextProps
          );
          break;
        case "backwards":
        case "unstable_legacy-backwards":
          renderLanes2 = null;
          revealOrder = workInProgress2.child;
          for (workInProgress2.child = null; null !== revealOrder; ) {
            current = revealOrder.alternate;
            if (null !== current && null === findFirstSuspended(current)) {
              workInProgress2.child = revealOrder;
              break;
            }
            current = revealOrder.sibling;
            revealOrder.sibling = renderLanes2;
            renderLanes2 = revealOrder;
            revealOrder = current;
          }
          initSuspenseListRenderState(
            workInProgress2,
            true,
            renderLanes2,
            null,
            tailMode,
            nextProps
          );
          break;
        case "together":
          initSuspenseListRenderState(
            workInProgress2,
            false,
            null,
            null,
            void 0,
            nextProps
          );
          break;
        default:
          workInProgress2.memoizedState = null;
      }
      return workInProgress2.child;
    }
    function bailoutOnAlreadyFinishedWork(current, workInProgress2, renderLanes2) {
      null !== current && (workInProgress2.dependencies = current.dependencies);
      workInProgressRootSkippedLanes |= workInProgress2.lanes;
      if (0 === (renderLanes2 & workInProgress2.childLanes))
        if (null !== current) {
          if (propagateParentContextChanges(
            current,
            workInProgress2,
            renderLanes2,
            false
          ), 0 === (renderLanes2 & workInProgress2.childLanes))
            return null;
        } else return null;
      if (null !== current && workInProgress2.child !== current.child)
        throw Error(formatProdErrorMessage(153));
      if (null !== workInProgress2.child) {
        current = workInProgress2.child;
        renderLanes2 = createWorkInProgress(current, current.pendingProps);
        workInProgress2.child = renderLanes2;
        for (renderLanes2.return = workInProgress2; null !== current.sibling; )
          current = current.sibling, renderLanes2 = renderLanes2.sibling = createWorkInProgress(current, current.pendingProps), renderLanes2.return = workInProgress2;
        renderLanes2.sibling = null;
      }
      return workInProgress2.child;
    }
    function checkScheduledUpdateOrContext(current, renderLanes2) {
      if (0 !== (current.lanes & renderLanes2)) return true;
      current = current.dependencies;
      return null !== current && checkIfContextChanged(current) ? true : false;
    }
    function attemptEarlyBailoutIfNoScheduledUpdate(current, workInProgress2, renderLanes2) {
      switch (workInProgress2.tag) {
        case 3:
          pushHostContainer(workInProgress2, workInProgress2.stateNode.containerInfo);
          pushProvider(workInProgress2, CacheContext, current.memoizedState.cache);
          resetHydrationState();
          break;
        case 27:
        case 5:
          pushHostContext(workInProgress2);
          break;
        case 4:
          pushHostContainer(workInProgress2, workInProgress2.stateNode.containerInfo);
          break;
        case 10:
          pushProvider(
            workInProgress2,
            workInProgress2.type,
            workInProgress2.memoizedProps.value
          );
          break;
        case 31:
          if (null !== workInProgress2.memoizedState)
            return workInProgress2.flags |= 128, pushDehydratedActivitySuspenseHandler(workInProgress2), null;
          break;
        case 13:
          var state$102 = workInProgress2.memoizedState;
          if (null !== state$102) {
            if (null !== state$102.dehydrated)
              return pushPrimaryTreeSuspenseHandler(workInProgress2), workInProgress2.flags |= 128, null;
            if (0 !== (renderLanes2 & workInProgress2.child.childLanes))
              return updateSuspenseComponent(current, workInProgress2, renderLanes2);
            pushPrimaryTreeSuspenseHandler(workInProgress2);
            current = bailoutOnAlreadyFinishedWork(
              current,
              workInProgress2,
              renderLanes2
            );
            return null !== current ? current.sibling : null;
          }
          pushPrimaryTreeSuspenseHandler(workInProgress2);
          break;
        case 19:
          var didSuspendBefore = 0 !== (current.flags & 128);
          state$102 = 0 !== (renderLanes2 & workInProgress2.childLanes);
          state$102 || (propagateParentContextChanges(
            current,
            workInProgress2,
            renderLanes2,
            false
          ), state$102 = 0 !== (renderLanes2 & workInProgress2.childLanes));
          if (didSuspendBefore) {
            if (state$102)
              return updateSuspenseListComponent(
                current,
                workInProgress2,
                renderLanes2
              );
            workInProgress2.flags |= 128;
          }
          didSuspendBefore = workInProgress2.memoizedState;
          null !== didSuspendBefore && (didSuspendBefore.rendering = null, didSuspendBefore.tail = null, didSuspendBefore.lastEffect = null);
          push(suspenseStackCursor, suspenseStackCursor.current);
          if (state$102) break;
          else return null;
        case 22:
          return workInProgress2.lanes = 0, updateOffscreenComponent(
            current,
            workInProgress2,
            renderLanes2,
            workInProgress2.pendingProps
          );
        case 24:
          pushProvider(workInProgress2, CacheContext, current.memoizedState.cache);
      }
      return bailoutOnAlreadyFinishedWork(current, workInProgress2, renderLanes2);
    }
    function beginWork(current, workInProgress2, renderLanes2) {
      if (null !== current)
        if (current.memoizedProps !== workInProgress2.pendingProps)
          didReceiveUpdate = true;
        else {
          if (!checkScheduledUpdateOrContext(current, renderLanes2) && 0 === (workInProgress2.flags & 128))
            return didReceiveUpdate = false, attemptEarlyBailoutIfNoScheduledUpdate(
              current,
              workInProgress2,
              renderLanes2
            );
          didReceiveUpdate = 0 !== (current.flags & 131072) ? true : false;
        }
      else
        didReceiveUpdate = false, isHydrating && 0 !== (workInProgress2.flags & 1048576) && pushTreeId(workInProgress2, treeForkCount, workInProgress2.index);
      workInProgress2.lanes = 0;
      switch (workInProgress2.tag) {
        case 16:
          a: {
            var props = workInProgress2.pendingProps;
            current = resolveLazy(workInProgress2.elementType);
            workInProgress2.type = current;
            if ("function" === typeof current)
              shouldConstruct(current) ? (props = resolveClassComponentProps(current, props), workInProgress2.tag = 1, workInProgress2 = updateClassComponent(
                null,
                workInProgress2,
                current,
                props,
                renderLanes2
              )) : (workInProgress2.tag = 0, workInProgress2 = updateFunctionComponent(
                null,
                workInProgress2,
                current,
                props,
                renderLanes2
              ));
            else {
              if (void 0 !== current && null !== current) {
                var $$typeof = current.$$typeof;
                if ($$typeof === REACT_FORWARD_REF_TYPE) {
                  workInProgress2.tag = 11;
                  workInProgress2 = updateForwardRef(
                    null,
                    workInProgress2,
                    current,
                    props,
                    renderLanes2
                  );
                  break a;
                } else if ($$typeof === REACT_MEMO_TYPE) {
                  workInProgress2.tag = 14;
                  workInProgress2 = updateMemoComponent(
                    null,
                    workInProgress2,
                    current,
                    props,
                    renderLanes2
                  );
                  break a;
                }
              }
              workInProgress2 = getComponentNameFromType(current) || current;
              throw Error(formatProdErrorMessage(306, workInProgress2, ""));
            }
          }
          return workInProgress2;
        case 0:
          return updateFunctionComponent(
            current,
            workInProgress2,
            workInProgress2.type,
            workInProgress2.pendingProps,
            renderLanes2
          );
        case 1:
          return props = workInProgress2.type, $$typeof = resolveClassComponentProps(
            props,
            workInProgress2.pendingProps
          ), updateClassComponent(
            current,
            workInProgress2,
            props,
            $$typeof,
            renderLanes2
          );
        case 3:
          a: {
            pushHostContainer(
              workInProgress2,
              workInProgress2.stateNode.containerInfo
            );
            if (null === current) throw Error(formatProdErrorMessage(387));
            props = workInProgress2.pendingProps;
            var prevState = workInProgress2.memoizedState;
            $$typeof = prevState.element;
            cloneUpdateQueue(current, workInProgress2);
            processUpdateQueue(workInProgress2, props, null, renderLanes2);
            var nextState = workInProgress2.memoizedState;
            props = nextState.cache;
            pushProvider(workInProgress2, CacheContext, props);
            props !== prevState.cache && propagateContextChanges(
              workInProgress2,
              [CacheContext],
              renderLanes2,
              true
            );
            suspendIfUpdateReadFromEntangledAsyncAction();
            props = nextState.element;
            if (prevState.isDehydrated)
              if (prevState = {
                element: props,
                isDehydrated: false,
                cache: nextState.cache
              }, workInProgress2.updateQueue.baseState = prevState, workInProgress2.memoizedState = prevState, workInProgress2.flags & 256) {
                workInProgress2 = mountHostRootWithoutHydrating(
                  current,
                  workInProgress2,
                  props,
                  renderLanes2
                );
                break a;
              } else if (props !== $$typeof) {
                $$typeof = createCapturedValueAtFiber(
                  Error(formatProdErrorMessage(424)),
                  workInProgress2
                );
                queueHydrationError($$typeof);
                workInProgress2 = mountHostRootWithoutHydrating(
                  current,
                  workInProgress2,
                  props,
                  renderLanes2
                );
                break a;
              } else {
                current = workInProgress2.stateNode.containerInfo;
                switch (current.nodeType) {
                  case 9:
                    current = current.body;
                    break;
                  default:
                    current = "HTML" === current.nodeName ? current.ownerDocument.body : current;
                }
                nextHydratableInstance = getNextHydratable(current.firstChild);
                hydrationParentFiber = workInProgress2;
                isHydrating = true;
                hydrationErrors = null;
                rootOrSingletonContext = true;
                renderLanes2 = mountChildFibers(
                  workInProgress2,
                  null,
                  props,
                  renderLanes2
                );
                for (workInProgress2.child = renderLanes2; renderLanes2; )
                  renderLanes2.flags = renderLanes2.flags & -3 | 4096, renderLanes2 = renderLanes2.sibling;
              }
            else {
              resetHydrationState();
              if (props === $$typeof) {
                workInProgress2 = bailoutOnAlreadyFinishedWork(
                  current,
                  workInProgress2,
                  renderLanes2
                );
                break a;
              }
              reconcileChildren(current, workInProgress2, props, renderLanes2);
            }
            workInProgress2 = workInProgress2.child;
          }
          return workInProgress2;
        case 26:
          return markRef(current, workInProgress2), null === current ? (renderLanes2 = getResource(
            workInProgress2.type,
            null,
            workInProgress2.pendingProps,
            null
          )) ? workInProgress2.memoizedState = renderLanes2 : isHydrating || (renderLanes2 = workInProgress2.type, current = workInProgress2.pendingProps, props = getOwnerDocumentFromRootContainer(
            rootInstanceStackCursor.current
          ).createElement(renderLanes2), props[internalInstanceKey] = workInProgress2, props[internalPropsKey] = current, setInitialProperties(props, renderLanes2, current), markNodeAsHoistable(props), workInProgress2.stateNode = props) : workInProgress2.memoizedState = getResource(
            workInProgress2.type,
            current.memoizedProps,
            workInProgress2.pendingProps,
            current.memoizedState
          ), null;
        case 27:
          return pushHostContext(workInProgress2), null === current && isHydrating && (props = workInProgress2.stateNode = resolveSingletonInstance(
            workInProgress2.type,
            workInProgress2.pendingProps,
            rootInstanceStackCursor.current
          ), hydrationParentFiber = workInProgress2, rootOrSingletonContext = true, $$typeof = nextHydratableInstance, isSingletonScope(workInProgress2.type) ? (previousHydratableOnEnteringScopedSingleton = $$typeof, nextHydratableInstance = getNextHydratable(props.firstChild)) : nextHydratableInstance = $$typeof), reconcileChildren(
            current,
            workInProgress2,
            workInProgress2.pendingProps.children,
            renderLanes2
          ), markRef(current, workInProgress2), null === current && (workInProgress2.flags |= 4194304), workInProgress2.child;
        case 5:
          if (null === current && isHydrating) {
            if ($$typeof = props = nextHydratableInstance)
              props = canHydrateInstance(
                props,
                workInProgress2.type,
                workInProgress2.pendingProps,
                rootOrSingletonContext
              ), null !== props ? (workInProgress2.stateNode = props, hydrationParentFiber = workInProgress2, nextHydratableInstance = getNextHydratable(props.firstChild), rootOrSingletonContext = false, $$typeof = true) : $$typeof = false;
            $$typeof || throwOnHydrationMismatch(workInProgress2);
          }
          pushHostContext(workInProgress2);
          $$typeof = workInProgress2.type;
          prevState = workInProgress2.pendingProps;
          nextState = null !== current ? current.memoizedProps : null;
          props = prevState.children;
          shouldSetTextContent($$typeof, prevState) ? props = null : null !== nextState && shouldSetTextContent($$typeof, nextState) && (workInProgress2.flags |= 32);
          null !== workInProgress2.memoizedState && ($$typeof = renderWithHooks(
            current,
            workInProgress2,
            TransitionAwareHostComponent,
            null,
            null,
            renderLanes2
          ), HostTransitionContext._currentValue = $$typeof);
          markRef(current, workInProgress2);
          reconcileChildren(current, workInProgress2, props, renderLanes2);
          return workInProgress2.child;
        case 6:
          if (null === current && isHydrating) {
            if (current = renderLanes2 = nextHydratableInstance)
              renderLanes2 = canHydrateTextInstance(
                renderLanes2,
                workInProgress2.pendingProps,
                rootOrSingletonContext
              ), null !== renderLanes2 ? (workInProgress2.stateNode = renderLanes2, hydrationParentFiber = workInProgress2, nextHydratableInstance = null, current = true) : current = false;
            current || throwOnHydrationMismatch(workInProgress2);
          }
          return null;
        case 13:
          return updateSuspenseComponent(current, workInProgress2, renderLanes2);
        case 4:
          return pushHostContainer(
            workInProgress2,
            workInProgress2.stateNode.containerInfo
          ), props = workInProgress2.pendingProps, null === current ? workInProgress2.child = reconcileChildFibers(
            workInProgress2,
            null,
            props,
            renderLanes2
          ) : reconcileChildren(current, workInProgress2, props, renderLanes2), workInProgress2.child;
        case 11:
          return updateForwardRef(
            current,
            workInProgress2,
            workInProgress2.type,
            workInProgress2.pendingProps,
            renderLanes2
          );
        case 7:
          return reconcileChildren(
            current,
            workInProgress2,
            workInProgress2.pendingProps,
            renderLanes2
          ), workInProgress2.child;
        case 8:
          return reconcileChildren(
            current,
            workInProgress2,
            workInProgress2.pendingProps.children,
            renderLanes2
          ), workInProgress2.child;
        case 12:
          return reconcileChildren(
            current,
            workInProgress2,
            workInProgress2.pendingProps.children,
            renderLanes2
          ), workInProgress2.child;
        case 10:
          return props = workInProgress2.pendingProps, pushProvider(workInProgress2, workInProgress2.type, props.value), reconcileChildren(current, workInProgress2, props.children, renderLanes2), workInProgress2.child;
        case 9:
          return $$typeof = workInProgress2.type._context, props = workInProgress2.pendingProps.children, prepareToReadContext(workInProgress2), $$typeof = readContext($$typeof), props = props($$typeof), workInProgress2.flags |= 1, reconcileChildren(current, workInProgress2, props, renderLanes2), workInProgress2.child;
        case 14:
          return updateMemoComponent(
            current,
            workInProgress2,
            workInProgress2.type,
            workInProgress2.pendingProps,
            renderLanes2
          );
        case 15:
          return updateSimpleMemoComponent(
            current,
            workInProgress2,
            workInProgress2.type,
            workInProgress2.pendingProps,
            renderLanes2
          );
        case 19:
          return updateSuspenseListComponent(current, workInProgress2, renderLanes2);
        case 31:
          return updateActivityComponent(current, workInProgress2, renderLanes2);
        case 22:
          return updateOffscreenComponent(
            current,
            workInProgress2,
            renderLanes2,
            workInProgress2.pendingProps
          );
        case 24:
          return prepareToReadContext(workInProgress2), props = readContext(CacheContext), null === current ? ($$typeof = peekCacheFromPool(), null === $$typeof && ($$typeof = workInProgressRoot, prevState = createCache(), $$typeof.pooledCache = prevState, prevState.refCount++, null !== prevState && ($$typeof.pooledCacheLanes |= renderLanes2), $$typeof = prevState), workInProgress2.memoizedState = { parent: props, cache: $$typeof }, initializeUpdateQueue(workInProgress2), pushProvider(workInProgress2, CacheContext, $$typeof)) : (0 !== (current.lanes & renderLanes2) && (cloneUpdateQueue(current, workInProgress2), processUpdateQueue(workInProgress2, null, null, renderLanes2), suspendIfUpdateReadFromEntangledAsyncAction()), $$typeof = current.memoizedState, prevState = workInProgress2.memoizedState, $$typeof.parent !== props ? ($$typeof = { parent: props, cache: props }, workInProgress2.memoizedState = $$typeof, 0 === workInProgress2.lanes && (workInProgress2.memoizedState = workInProgress2.updateQueue.baseState = $$typeof), pushProvider(workInProgress2, CacheContext, props)) : (props = prevState.cache, pushProvider(workInProgress2, CacheContext, props), props !== $$typeof.cache && propagateContextChanges(
            workInProgress2,
            [CacheContext],
            renderLanes2,
            true
          ))), reconcileChildren(
            current,
            workInProgress2,
            workInProgress2.pendingProps.children,
            renderLanes2
          ), workInProgress2.child;
        case 29:
          throw workInProgress2.pendingProps;
      }
      throw Error(formatProdErrorMessage(156, workInProgress2.tag));
    }
    function markUpdate(workInProgress2) {
      workInProgress2.flags |= 4;
    }
    function preloadInstanceAndSuspendIfNeeded(workInProgress2, type, oldProps, newProps, renderLanes2) {
      if (type = 0 !== (workInProgress2.mode & 32)) type = false;
      if (type) {
        if (workInProgress2.flags |= 16777216, (renderLanes2 & 335544128) === renderLanes2)
          if (workInProgress2.stateNode.complete) workInProgress2.flags |= 8192;
          else if (shouldRemainOnPreviousScreen()) workInProgress2.flags |= 8192;
          else
            throw suspendedThenable = noopSuspenseyCommitThenable, SuspenseyCommitException;
      } else workInProgress2.flags &= -16777217;
    }
    function preloadResourceAndSuspendIfNeeded(workInProgress2, resource) {
      if ("stylesheet" !== resource.type || 0 !== (resource.state.loading & 4))
        workInProgress2.flags &= -16777217;
      else if (workInProgress2.flags |= 16777216, !preloadResource(resource))
        if (shouldRemainOnPreviousScreen()) workInProgress2.flags |= 8192;
        else
          throw suspendedThenable = noopSuspenseyCommitThenable, SuspenseyCommitException;
    }
    function scheduleRetryEffect(workInProgress2, retryQueue) {
      null !== retryQueue && (workInProgress2.flags |= 4);
      workInProgress2.flags & 16384 && (retryQueue = 22 !== workInProgress2.tag ? claimNextRetryLane() : 536870912, workInProgress2.lanes |= retryQueue, workInProgressSuspendedRetryLanes |= retryQueue);
    }
    function cutOffTailIfNeeded(renderState, hasRenderedATailFallback) {
      if (!isHydrating)
        switch (renderState.tailMode) {
          case "hidden":
            hasRenderedATailFallback = renderState.tail;
            for (var lastTailNode = null; null !== hasRenderedATailFallback; )
              null !== hasRenderedATailFallback.alternate && (lastTailNode = hasRenderedATailFallback), hasRenderedATailFallback = hasRenderedATailFallback.sibling;
            null === lastTailNode ? renderState.tail = null : lastTailNode.sibling = null;
            break;
          case "collapsed":
            lastTailNode = renderState.tail;
            for (var lastTailNode$106 = null; null !== lastTailNode; )
              null !== lastTailNode.alternate && (lastTailNode$106 = lastTailNode), lastTailNode = lastTailNode.sibling;
            null === lastTailNode$106 ? hasRenderedATailFallback || null === renderState.tail ? renderState.tail = null : renderState.tail.sibling = null : lastTailNode$106.sibling = null;
        }
    }
    function bubbleProperties(completedWork) {
      var didBailout = null !== completedWork.alternate && completedWork.alternate.child === completedWork.child, newChildLanes = 0, subtreeFlags = 0;
      if (didBailout)
        for (var child$107 = completedWork.child; null !== child$107; )
          newChildLanes |= child$107.lanes | child$107.childLanes, subtreeFlags |= child$107.subtreeFlags & 65011712, subtreeFlags |= child$107.flags & 65011712, child$107.return = completedWork, child$107 = child$107.sibling;
      else
        for (child$107 = completedWork.child; null !== child$107; )
          newChildLanes |= child$107.lanes | child$107.childLanes, subtreeFlags |= child$107.subtreeFlags, subtreeFlags |= child$107.flags, child$107.return = completedWork, child$107 = child$107.sibling;
      completedWork.subtreeFlags |= subtreeFlags;
      completedWork.childLanes = newChildLanes;
      return didBailout;
    }
    function completeWork(current, workInProgress2, renderLanes2) {
      var newProps = workInProgress2.pendingProps;
      popTreeContext(workInProgress2);
      switch (workInProgress2.tag) {
        case 16:
        case 15:
        case 0:
        case 11:
        case 7:
        case 8:
        case 12:
        case 9:
        case 14:
          return bubbleProperties(workInProgress2), null;
        case 1:
          return bubbleProperties(workInProgress2), null;
        case 3:
          renderLanes2 = workInProgress2.stateNode;
          newProps = null;
          null !== current && (newProps = current.memoizedState.cache);
          workInProgress2.memoizedState.cache !== newProps && (workInProgress2.flags |= 2048);
          popProvider(CacheContext);
          popHostContainer();
          renderLanes2.pendingContext && (renderLanes2.context = renderLanes2.pendingContext, renderLanes2.pendingContext = null);
          if (null === current || null === current.child)
            popHydrationState(workInProgress2) ? markUpdate(workInProgress2) : null === current || current.memoizedState.isDehydrated && 0 === (workInProgress2.flags & 256) || (workInProgress2.flags |= 1024, upgradeHydrationErrorsToRecoverable());
          bubbleProperties(workInProgress2);
          return null;
        case 26:
          var type = workInProgress2.type, nextResource = workInProgress2.memoizedState;
          null === current ? (markUpdate(workInProgress2), null !== nextResource ? (bubbleProperties(workInProgress2), preloadResourceAndSuspendIfNeeded(workInProgress2, nextResource)) : (bubbleProperties(workInProgress2), preloadInstanceAndSuspendIfNeeded(
            workInProgress2,
            type,
            null,
            newProps,
            renderLanes2
          ))) : nextResource ? nextResource !== current.memoizedState ? (markUpdate(workInProgress2), bubbleProperties(workInProgress2), preloadResourceAndSuspendIfNeeded(workInProgress2, nextResource)) : (bubbleProperties(workInProgress2), workInProgress2.flags &= -16777217) : (current = current.memoizedProps, current !== newProps && markUpdate(workInProgress2), bubbleProperties(workInProgress2), preloadInstanceAndSuspendIfNeeded(
            workInProgress2,
            type,
            current,
            newProps,
            renderLanes2
          ));
          return null;
        case 27:
          popHostContext(workInProgress2);
          renderLanes2 = rootInstanceStackCursor.current;
          type = workInProgress2.type;
          if (null !== current && null != workInProgress2.stateNode)
            current.memoizedProps !== newProps && markUpdate(workInProgress2);
          else {
            if (!newProps) {
              if (null === workInProgress2.stateNode)
                throw Error(formatProdErrorMessage(166));
              bubbleProperties(workInProgress2);
              return null;
            }
            current = contextStackCursor.current;
            popHydrationState(workInProgress2) ? prepareToHydrateHostInstance(workInProgress2) : (current = resolveSingletonInstance(type, newProps, renderLanes2), workInProgress2.stateNode = current, markUpdate(workInProgress2));
          }
          bubbleProperties(workInProgress2);
          return null;
        case 5:
          popHostContext(workInProgress2);
          type = workInProgress2.type;
          if (null !== current && null != workInProgress2.stateNode)
            current.memoizedProps !== newProps && markUpdate(workInProgress2);
          else {
            if (!newProps) {
              if (null === workInProgress2.stateNode)
                throw Error(formatProdErrorMessage(166));
              bubbleProperties(workInProgress2);
              return null;
            }
            nextResource = contextStackCursor.current;
            if (popHydrationState(workInProgress2))
              prepareToHydrateHostInstance(workInProgress2);
            else {
              var ownerDocument = getOwnerDocumentFromRootContainer(
                rootInstanceStackCursor.current
              );
              switch (nextResource) {
                case 1:
                  nextResource = ownerDocument.createElementNS(
                    "http://www.w3.org/2000/svg",
                    type
                  );
                  break;
                case 2:
                  nextResource = ownerDocument.createElementNS(
                    "http://www.w3.org/1998/Math/MathML",
                    type
                  );
                  break;
                default:
                  switch (type) {
                    case "svg":
                      nextResource = ownerDocument.createElementNS(
                        "http://www.w3.org/2000/svg",
                        type
                      );
                      break;
                    case "math":
                      nextResource = ownerDocument.createElementNS(
                        "http://www.w3.org/1998/Math/MathML",
                        type
                      );
                      break;
                    case "script":
                      nextResource = ownerDocument.createElement("div");
                      nextResource.innerHTML = "<script><\/script>";
                      nextResource = nextResource.removeChild(
                        nextResource.firstChild
                      );
                      break;
                    case "select":
                      nextResource = "string" === typeof newProps.is ? ownerDocument.createElement("select", {
                        is: newProps.is
                      }) : ownerDocument.createElement("select");
                      newProps.multiple ? nextResource.multiple = true : newProps.size && (nextResource.size = newProps.size);
                      break;
                    default:
                      nextResource = "string" === typeof newProps.is ? ownerDocument.createElement(type, { is: newProps.is }) : ownerDocument.createElement(type);
                  }
              }
              nextResource[internalInstanceKey] = workInProgress2;
              nextResource[internalPropsKey] = newProps;
              a: for (ownerDocument = workInProgress2.child; null !== ownerDocument; ) {
                if (5 === ownerDocument.tag || 6 === ownerDocument.tag)
                  nextResource.appendChild(ownerDocument.stateNode);
                else if (4 !== ownerDocument.tag && 27 !== ownerDocument.tag && null !== ownerDocument.child) {
                  ownerDocument.child.return = ownerDocument;
                  ownerDocument = ownerDocument.child;
                  continue;
                }
                if (ownerDocument === workInProgress2) break a;
                for (; null === ownerDocument.sibling; ) {
                  if (null === ownerDocument.return || ownerDocument.return === workInProgress2)
                    break a;
                  ownerDocument = ownerDocument.return;
                }
                ownerDocument.sibling.return = ownerDocument.return;
                ownerDocument = ownerDocument.sibling;
              }
              workInProgress2.stateNode = nextResource;
              a: switch (setInitialProperties(nextResource, type, newProps), type) {
                case "button":
                case "input":
                case "select":
                case "textarea":
                  newProps = !!newProps.autoFocus;
                  break a;
                case "img":
                  newProps = true;
                  break a;
                default:
                  newProps = false;
              }
              newProps && markUpdate(workInProgress2);
            }
          }
          bubbleProperties(workInProgress2);
          preloadInstanceAndSuspendIfNeeded(
            workInProgress2,
            workInProgress2.type,
            null === current ? null : current.memoizedProps,
            workInProgress2.pendingProps,
            renderLanes2
          );
          return null;
        case 6:
          if (current && null != workInProgress2.stateNode)
            current.memoizedProps !== newProps && markUpdate(workInProgress2);
          else {
            if ("string" !== typeof newProps && null === workInProgress2.stateNode)
              throw Error(formatProdErrorMessage(166));
            current = rootInstanceStackCursor.current;
            if (popHydrationState(workInProgress2)) {
              current = workInProgress2.stateNode;
              renderLanes2 = workInProgress2.memoizedProps;
              newProps = null;
              type = hydrationParentFiber;
              if (null !== type)
                switch (type.tag) {
                  case 27:
                  case 5:
                    newProps = type.memoizedProps;
                }
              current[internalInstanceKey] = workInProgress2;
              current = current.nodeValue === renderLanes2 || null !== newProps && true === newProps.suppressHydrationWarning || checkForUnmatchedText(current.nodeValue, renderLanes2) ? true : false;
              current || throwOnHydrationMismatch(workInProgress2, true);
            } else
              current = getOwnerDocumentFromRootContainer(current).createTextNode(
                newProps
              ), current[internalInstanceKey] = workInProgress2, workInProgress2.stateNode = current;
          }
          bubbleProperties(workInProgress2);
          return null;
        case 31:
          renderLanes2 = workInProgress2.memoizedState;
          if (null === current || null !== current.memoizedState) {
            newProps = popHydrationState(workInProgress2);
            if (null !== renderLanes2) {
              if (null === current) {
                if (!newProps) throw Error(formatProdErrorMessage(318));
                current = workInProgress2.memoizedState;
                current = null !== current ? current.dehydrated : null;
                if (!current) throw Error(formatProdErrorMessage(557));
                current[internalInstanceKey] = workInProgress2;
              } else
                resetHydrationState(), 0 === (workInProgress2.flags & 128) && (workInProgress2.memoizedState = null), workInProgress2.flags |= 4;
              bubbleProperties(workInProgress2);
              current = false;
            } else
              renderLanes2 = upgradeHydrationErrorsToRecoverable(), null !== current && null !== current.memoizedState && (current.memoizedState.hydrationErrors = renderLanes2), current = true;
            if (!current) {
              if (workInProgress2.flags & 256)
                return popSuspenseHandler(workInProgress2), workInProgress2;
              popSuspenseHandler(workInProgress2);
              return null;
            }
            if (0 !== (workInProgress2.flags & 128))
              throw Error(formatProdErrorMessage(558));
          }
          bubbleProperties(workInProgress2);
          return null;
        case 13:
          newProps = workInProgress2.memoizedState;
          if (null === current || null !== current.memoizedState && null !== current.memoizedState.dehydrated) {
            type = popHydrationState(workInProgress2);
            if (null !== newProps && null !== newProps.dehydrated) {
              if (null === current) {
                if (!type) throw Error(formatProdErrorMessage(318));
                type = workInProgress2.memoizedState;
                type = null !== type ? type.dehydrated : null;
                if (!type) throw Error(formatProdErrorMessage(317));
                type[internalInstanceKey] = workInProgress2;
              } else
                resetHydrationState(), 0 === (workInProgress2.flags & 128) && (workInProgress2.memoizedState = null), workInProgress2.flags |= 4;
              bubbleProperties(workInProgress2);
              type = false;
            } else
              type = upgradeHydrationErrorsToRecoverable(), null !== current && null !== current.memoizedState && (current.memoizedState.hydrationErrors = type), type = true;
            if (!type) {
              if (workInProgress2.flags & 256)
                return popSuspenseHandler(workInProgress2), workInProgress2;
              popSuspenseHandler(workInProgress2);
              return null;
            }
          }
          popSuspenseHandler(workInProgress2);
          if (0 !== (workInProgress2.flags & 128))
            return workInProgress2.lanes = renderLanes2, workInProgress2;
          renderLanes2 = null !== newProps;
          current = null !== current && null !== current.memoizedState;
          renderLanes2 && (newProps = workInProgress2.child, type = null, null !== newProps.alternate && null !== newProps.alternate.memoizedState && null !== newProps.alternate.memoizedState.cachePool && (type = newProps.alternate.memoizedState.cachePool.pool), nextResource = null, null !== newProps.memoizedState && null !== newProps.memoizedState.cachePool && (nextResource = newProps.memoizedState.cachePool.pool), nextResource !== type && (newProps.flags |= 2048));
          renderLanes2 !== current && renderLanes2 && (workInProgress2.child.flags |= 8192);
          scheduleRetryEffect(workInProgress2, workInProgress2.updateQueue);
          bubbleProperties(workInProgress2);
          return null;
        case 4:
          return popHostContainer(), null === current && listenToAllSupportedEvents(workInProgress2.stateNode.containerInfo), bubbleProperties(workInProgress2), null;
        case 10:
          return popProvider(workInProgress2.type), bubbleProperties(workInProgress2), null;
        case 19:
          pop(suspenseStackCursor);
          newProps = workInProgress2.memoizedState;
          if (null === newProps) return bubbleProperties(workInProgress2), null;
          type = 0 !== (workInProgress2.flags & 128);
          nextResource = newProps.rendering;
          if (null === nextResource)
            if (type) cutOffTailIfNeeded(newProps, false);
            else {
              if (0 !== workInProgressRootExitStatus || null !== current && 0 !== (current.flags & 128))
                for (current = workInProgress2.child; null !== current; ) {
                  nextResource = findFirstSuspended(current);
                  if (null !== nextResource) {
                    workInProgress2.flags |= 128;
                    cutOffTailIfNeeded(newProps, false);
                    current = nextResource.updateQueue;
                    workInProgress2.updateQueue = current;
                    scheduleRetryEffect(workInProgress2, current);
                    workInProgress2.subtreeFlags = 0;
                    current = renderLanes2;
                    for (renderLanes2 = workInProgress2.child; null !== renderLanes2; )
                      resetWorkInProgress(renderLanes2, current), renderLanes2 = renderLanes2.sibling;
                    push(
                      suspenseStackCursor,
                      suspenseStackCursor.current & 1 | 2
                    );
                    isHydrating && pushTreeFork(workInProgress2, newProps.treeForkCount);
                    return workInProgress2.child;
                  }
                  current = current.sibling;
                }
              null !== newProps.tail && now() > workInProgressRootRenderTargetTime && (workInProgress2.flags |= 128, type = true, cutOffTailIfNeeded(newProps, false), workInProgress2.lanes = 4194304);
            }
          else {
            if (!type)
              if (current = findFirstSuspended(nextResource), null !== current) {
                if (workInProgress2.flags |= 128, type = true, current = current.updateQueue, workInProgress2.updateQueue = current, scheduleRetryEffect(workInProgress2, current), cutOffTailIfNeeded(newProps, true), null === newProps.tail && "hidden" === newProps.tailMode && !nextResource.alternate && !isHydrating)
                  return bubbleProperties(workInProgress2), null;
              } else
                2 * now() - newProps.renderingStartTime > workInProgressRootRenderTargetTime && 536870912 !== renderLanes2 && (workInProgress2.flags |= 128, type = true, cutOffTailIfNeeded(newProps, false), workInProgress2.lanes = 4194304);
            newProps.isBackwards ? (nextResource.sibling = workInProgress2.child, workInProgress2.child = nextResource) : (current = newProps.last, null !== current ? current.sibling = nextResource : workInProgress2.child = nextResource, newProps.last = nextResource);
          }
          if (null !== newProps.tail)
            return current = newProps.tail, newProps.rendering = current, newProps.tail = current.sibling, newProps.renderingStartTime = now(), current.sibling = null, renderLanes2 = suspenseStackCursor.current, push(
              suspenseStackCursor,
              type ? renderLanes2 & 1 | 2 : renderLanes2 & 1
            ), isHydrating && pushTreeFork(workInProgress2, newProps.treeForkCount), current;
          bubbleProperties(workInProgress2);
          return null;
        case 22:
        case 23:
          return popSuspenseHandler(workInProgress2), popHiddenContext(), newProps = null !== workInProgress2.memoizedState, null !== current ? null !== current.memoizedState !== newProps && (workInProgress2.flags |= 8192) : newProps && (workInProgress2.flags |= 8192), newProps ? 0 !== (renderLanes2 & 536870912) && 0 === (workInProgress2.flags & 128) && (bubbleProperties(workInProgress2), workInProgress2.subtreeFlags & 6 && (workInProgress2.flags |= 8192)) : bubbleProperties(workInProgress2), renderLanes2 = workInProgress2.updateQueue, null !== renderLanes2 && scheduleRetryEffect(workInProgress2, renderLanes2.retryQueue), renderLanes2 = null, null !== current && null !== current.memoizedState && null !== current.memoizedState.cachePool && (renderLanes2 = current.memoizedState.cachePool.pool), newProps = null, null !== workInProgress2.memoizedState && null !== workInProgress2.memoizedState.cachePool && (newProps = workInProgress2.memoizedState.cachePool.pool), newProps !== renderLanes2 && (workInProgress2.flags |= 2048), null !== current && pop(resumedCache), null;
        case 24:
          return renderLanes2 = null, null !== current && (renderLanes2 = current.memoizedState.cache), workInProgress2.memoizedState.cache !== renderLanes2 && (workInProgress2.flags |= 2048), popProvider(CacheContext), bubbleProperties(workInProgress2), null;
        case 25:
          return null;
        case 30:
          return null;
      }
      throw Error(formatProdErrorMessage(156, workInProgress2.tag));
    }
    function unwindWork(current, workInProgress2) {
      popTreeContext(workInProgress2);
      switch (workInProgress2.tag) {
        case 1:
          return current = workInProgress2.flags, current & 65536 ? (workInProgress2.flags = current & -65537 | 128, workInProgress2) : null;
        case 3:
          return popProvider(CacheContext), popHostContainer(), current = workInProgress2.flags, 0 !== (current & 65536) && 0 === (current & 128) ? (workInProgress2.flags = current & -65537 | 128, workInProgress2) : null;
        case 26:
        case 27:
        case 5:
          return popHostContext(workInProgress2), null;
        case 31:
          if (null !== workInProgress2.memoizedState) {
            popSuspenseHandler(workInProgress2);
            if (null === workInProgress2.alternate)
              throw Error(formatProdErrorMessage(340));
            resetHydrationState();
          }
          current = workInProgress2.flags;
          return current & 65536 ? (workInProgress2.flags = current & -65537 | 128, workInProgress2) : null;
        case 13:
          popSuspenseHandler(workInProgress2);
          current = workInProgress2.memoizedState;
          if (null !== current && null !== current.dehydrated) {
            if (null === workInProgress2.alternate)
              throw Error(formatProdErrorMessage(340));
            resetHydrationState();
          }
          current = workInProgress2.flags;
          return current & 65536 ? (workInProgress2.flags = current & -65537 | 128, workInProgress2) : null;
        case 19:
          return pop(suspenseStackCursor), null;
        case 4:
          return popHostContainer(), null;
        case 10:
          return popProvider(workInProgress2.type), null;
        case 22:
        case 23:
          return popSuspenseHandler(workInProgress2), popHiddenContext(), null !== current && pop(resumedCache), current = workInProgress2.flags, current & 65536 ? (workInProgress2.flags = current & -65537 | 128, workInProgress2) : null;
        case 24:
          return popProvider(CacheContext), null;
        case 25:
          return null;
        default:
          return null;
      }
    }
    function unwindInterruptedWork(current, interruptedWork) {
      popTreeContext(interruptedWork);
      switch (interruptedWork.tag) {
        case 3:
          popProvider(CacheContext);
          popHostContainer();
          break;
        case 26:
        case 27:
        case 5:
          popHostContext(interruptedWork);
          break;
        case 4:
          popHostContainer();
          break;
        case 31:
          null !== interruptedWork.memoizedState && popSuspenseHandler(interruptedWork);
          break;
        case 13:
          popSuspenseHandler(interruptedWork);
          break;
        case 19:
          pop(suspenseStackCursor);
          break;
        case 10:
          popProvider(interruptedWork.type);
          break;
        case 22:
        case 23:
          popSuspenseHandler(interruptedWork);
          popHiddenContext();
          null !== current && pop(resumedCache);
          break;
        case 24:
          popProvider(CacheContext);
      }
    }
    function commitHookEffectListMount(flags, finishedWork) {
      try {
        var updateQueue = finishedWork.updateQueue, lastEffect = null !== updateQueue ? updateQueue.lastEffect : null;
        if (null !== lastEffect) {
          var firstEffect = lastEffect.next;
          updateQueue = firstEffect;
          do {
            if ((updateQueue.tag & flags) === flags) {
              lastEffect = void 0;
              var create = updateQueue.create, inst = updateQueue.inst;
              lastEffect = create();
              inst.destroy = lastEffect;
            }
            updateQueue = updateQueue.next;
          } while (updateQueue !== firstEffect);
        }
      } catch (error) {
        captureCommitPhaseError(finishedWork, finishedWork.return, error);
      }
    }
    function commitHookEffectListUnmount(flags, finishedWork, nearestMountedAncestor$jscomp$0) {
      try {
        var updateQueue = finishedWork.updateQueue, lastEffect = null !== updateQueue ? updateQueue.lastEffect : null;
        if (null !== lastEffect) {
          var firstEffect = lastEffect.next;
          updateQueue = firstEffect;
          do {
            if ((updateQueue.tag & flags) === flags) {
              var inst = updateQueue.inst, destroy = inst.destroy;
              if (void 0 !== destroy) {
                inst.destroy = void 0;
                lastEffect = finishedWork;
                var nearestMountedAncestor = nearestMountedAncestor$jscomp$0, destroy_ = destroy;
                try {
                  destroy_();
                } catch (error) {
                  captureCommitPhaseError(
                    lastEffect,
                    nearestMountedAncestor,
                    error
                  );
                }
              }
            }
            updateQueue = updateQueue.next;
          } while (updateQueue !== firstEffect);
        }
      } catch (error) {
        captureCommitPhaseError(finishedWork, finishedWork.return, error);
      }
    }
    function commitClassCallbacks(finishedWork) {
      var updateQueue = finishedWork.updateQueue;
      if (null !== updateQueue) {
        var instance = finishedWork.stateNode;
        try {
          commitCallbacks(updateQueue, instance);
        } catch (error) {
          captureCommitPhaseError(finishedWork, finishedWork.return, error);
        }
      }
    }
    function safelyCallComponentWillUnmount(current, nearestMountedAncestor, instance) {
      instance.props = resolveClassComponentProps(
        current.type,
        current.memoizedProps
      );
      instance.state = current.memoizedState;
      try {
        instance.componentWillUnmount();
      } catch (error) {
        captureCommitPhaseError(current, nearestMountedAncestor, error);
      }
    }
    function safelyAttachRef(current, nearestMountedAncestor) {
      try {
        var ref = current.ref;
        if (null !== ref) {
          switch (current.tag) {
            case 26:
            case 27:
            case 5:
              var instanceToUse = current.stateNode;
              break;
            case 30:
              instanceToUse = current.stateNode;
              break;
            default:
              instanceToUse = current.stateNode;
          }
          "function" === typeof ref ? current.refCleanup = ref(instanceToUse) : ref.current = instanceToUse;
        }
      } catch (error) {
        captureCommitPhaseError(current, nearestMountedAncestor, error);
      }
    }
    function safelyDetachRef(current, nearestMountedAncestor) {
      var ref = current.ref, refCleanup = current.refCleanup;
      if (null !== ref)
        if ("function" === typeof refCleanup)
          try {
            refCleanup();
          } catch (error) {
            captureCommitPhaseError(current, nearestMountedAncestor, error);
          } finally {
            current.refCleanup = null, current = current.alternate, null != current && (current.refCleanup = null);
          }
        else if ("function" === typeof ref)
          try {
            ref(null);
          } catch (error$140) {
            captureCommitPhaseError(current, nearestMountedAncestor, error$140);
          }
        else ref.current = null;
    }
    function commitHostMount(finishedWork) {
      var type = finishedWork.type, props = finishedWork.memoizedProps, instance = finishedWork.stateNode;
      try {
        a: switch (type) {
          case "button":
          case "input":
          case "select":
          case "textarea":
            props.autoFocus && instance.focus();
            break a;
          case "img":
            props.src ? instance.src = props.src : props.srcSet && (instance.srcset = props.srcSet);
        }
      } catch (error) {
        captureCommitPhaseError(finishedWork, finishedWork.return, error);
      }
    }
    function commitHostUpdate(finishedWork, newProps, oldProps) {
      try {
        var domElement = finishedWork.stateNode;
        updateProperties(domElement, finishedWork.type, oldProps, newProps);
        domElement[internalPropsKey] = newProps;
      } catch (error) {
        captureCommitPhaseError(finishedWork, finishedWork.return, error);
      }
    }
    function isHostParent(fiber) {
      return 5 === fiber.tag || 3 === fiber.tag || 26 === fiber.tag || 27 === fiber.tag && isSingletonScope(fiber.type) || 4 === fiber.tag;
    }
    function getHostSibling(fiber) {
      a: for (; ; ) {
        for (; null === fiber.sibling; ) {
          if (null === fiber.return || isHostParent(fiber.return)) return null;
          fiber = fiber.return;
        }
        fiber.sibling.return = fiber.return;
        for (fiber = fiber.sibling; 5 !== fiber.tag && 6 !== fiber.tag && 18 !== fiber.tag; ) {
          if (27 === fiber.tag && isSingletonScope(fiber.type)) continue a;
          if (fiber.flags & 2) continue a;
          if (null === fiber.child || 4 === fiber.tag) continue a;
          else fiber.child.return = fiber, fiber = fiber.child;
        }
        if (!(fiber.flags & 2)) return fiber.stateNode;
      }
    }
    function insertOrAppendPlacementNodeIntoContainer(node, before, parent) {
      var tag = node.tag;
      if (5 === tag || 6 === tag)
        node = node.stateNode, before ? (9 === parent.nodeType ? parent.body : "HTML" === parent.nodeName ? parent.ownerDocument.body : parent).insertBefore(node, before) : (before = 9 === parent.nodeType ? parent.body : "HTML" === parent.nodeName ? parent.ownerDocument.body : parent, before.appendChild(node), parent = parent._reactRootContainer, null !== parent && void 0 !== parent || null !== before.onclick || (before.onclick = noop$1));
      else if (4 !== tag && (27 === tag && isSingletonScope(node.type) && (parent = node.stateNode, before = null), node = node.child, null !== node))
        for (insertOrAppendPlacementNodeIntoContainer(node, before, parent), node = node.sibling; null !== node; )
          insertOrAppendPlacementNodeIntoContainer(node, before, parent), node = node.sibling;
    }
    function insertOrAppendPlacementNode(node, before, parent) {
      var tag = node.tag;
      if (5 === tag || 6 === tag)
        node = node.stateNode, before ? parent.insertBefore(node, before) : parent.appendChild(node);
      else if (4 !== tag && (27 === tag && isSingletonScope(node.type) && (parent = node.stateNode), node = node.child, null !== node))
        for (insertOrAppendPlacementNode(node, before, parent), node = node.sibling; null !== node; )
          insertOrAppendPlacementNode(node, before, parent), node = node.sibling;
    }
    function commitHostSingletonAcquisition(finishedWork) {
      var singleton = finishedWork.stateNode, props = finishedWork.memoizedProps;
      try {
        for (var type = finishedWork.type, attributes = singleton.attributes; attributes.length; )
          singleton.removeAttributeNode(attributes[0]);
        setInitialProperties(singleton, type, props);
        singleton[internalInstanceKey] = finishedWork;
        singleton[internalPropsKey] = props;
      } catch (error) {
        captureCommitPhaseError(finishedWork, finishedWork.return, error);
      }
    }
    var offscreenSubtreeIsHidden = false, offscreenSubtreeWasHidden = false, needsFormReset = false, PossiblyWeakSet = "function" === typeof WeakSet ? WeakSet : Set, nextEffect = null;
    function commitBeforeMutationEffects(root3, firstChild) {
      root3 = root3.containerInfo;
      eventsEnabled = _enabled;
      root3 = getActiveElementDeep(root3);
      if (hasSelectionCapabilities(root3)) {
        if ("selectionStart" in root3)
          var JSCompiler_temp = {
            start: root3.selectionStart,
            end: root3.selectionEnd
          };
        else
          a: {
            JSCompiler_temp = (JSCompiler_temp = root3.ownerDocument) && JSCompiler_temp.defaultView || window;
            var selection = JSCompiler_temp.getSelection && JSCompiler_temp.getSelection();
            if (selection && 0 !== selection.rangeCount) {
              JSCompiler_temp = selection.anchorNode;
              var anchorOffset = selection.anchorOffset, focusNode = selection.focusNode;
              selection = selection.focusOffset;
              try {
                JSCompiler_temp.nodeType, focusNode.nodeType;
              } catch (e$20) {
                JSCompiler_temp = null;
                break a;
              }
              var length = 0, start = -1, end = -1, indexWithinAnchor = 0, indexWithinFocus = 0, node = root3, parentNode = null;
              b: for (; ; ) {
                for (var next; ; ) {
                  node !== JSCompiler_temp || 0 !== anchorOffset && 3 !== node.nodeType || (start = length + anchorOffset);
                  node !== focusNode || 0 !== selection && 3 !== node.nodeType || (end = length + selection);
                  3 === node.nodeType && (length += node.nodeValue.length);
                  if (null === (next = node.firstChild)) break;
                  parentNode = node;
                  node = next;
                }
                for (; ; ) {
                  if (node === root3) break b;
                  parentNode === JSCompiler_temp && ++indexWithinAnchor === anchorOffset && (start = length);
                  parentNode === focusNode && ++indexWithinFocus === selection && (end = length);
                  if (null !== (next = node.nextSibling)) break;
                  node = parentNode;
                  parentNode = node.parentNode;
                }
                node = next;
              }
              JSCompiler_temp = -1 === start || -1 === end ? null : { start, end };
            } else JSCompiler_temp = null;
          }
        JSCompiler_temp = JSCompiler_temp || { start: 0, end: 0 };
      } else JSCompiler_temp = null;
      selectionInformation = { focusedElem: root3, selectionRange: JSCompiler_temp };
      _enabled = false;
      for (nextEffect = firstChild; null !== nextEffect; )
        if (firstChild = nextEffect, root3 = firstChild.child, 0 !== (firstChild.subtreeFlags & 1028) && null !== root3)
          root3.return = firstChild, nextEffect = root3;
        else
          for (; null !== nextEffect; ) {
            firstChild = nextEffect;
            focusNode = firstChild.alternate;
            root3 = firstChild.flags;
            switch (firstChild.tag) {
              case 0:
                if (0 !== (root3 & 4) && (root3 = firstChild.updateQueue, root3 = null !== root3 ? root3.events : null, null !== root3))
                  for (JSCompiler_temp = 0; JSCompiler_temp < root3.length; JSCompiler_temp++)
                    anchorOffset = root3[JSCompiler_temp], anchorOffset.ref.impl = anchorOffset.nextImpl;
                break;
              case 11:
              case 15:
                break;
              case 1:
                if (0 !== (root3 & 1024) && null !== focusNode) {
                  root3 = void 0;
                  JSCompiler_temp = firstChild;
                  anchorOffset = focusNode.memoizedProps;
                  focusNode = focusNode.memoizedState;
                  selection = JSCompiler_temp.stateNode;
                  try {
                    var resolvedPrevProps = resolveClassComponentProps(
                      JSCompiler_temp.type,
                      anchorOffset
                    );
                    root3 = selection.getSnapshotBeforeUpdate(
                      resolvedPrevProps,
                      focusNode
                    );
                    selection.__reactInternalSnapshotBeforeUpdate = root3;
                  } catch (error) {
                    captureCommitPhaseError(
                      JSCompiler_temp,
                      JSCompiler_temp.return,
                      error
                    );
                  }
                }
                break;
              case 3:
                if (0 !== (root3 & 1024)) {
                  if (root3 = firstChild.stateNode.containerInfo, JSCompiler_temp = root3.nodeType, 9 === JSCompiler_temp)
                    clearContainerSparingly(root3);
                  else if (1 === JSCompiler_temp)
                    switch (root3.nodeName) {
                      case "HEAD":
                      case "HTML":
                      case "BODY":
                        clearContainerSparingly(root3);
                        break;
                      default:
                        root3.textContent = "";
                    }
                }
                break;
              case 5:
              case 26:
              case 27:
              case 6:
              case 4:
              case 17:
                break;
              default:
                if (0 !== (root3 & 1024)) throw Error(formatProdErrorMessage(163));
            }
            root3 = firstChild.sibling;
            if (null !== root3) {
              root3.return = firstChild.return;
              nextEffect = root3;
              break;
            }
            nextEffect = firstChild.return;
          }
    }
    function commitLayoutEffectOnFiber(finishedRoot, current, finishedWork) {
      var flags = finishedWork.flags;
      switch (finishedWork.tag) {
        case 0:
        case 11:
        case 15:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
          flags & 4 && commitHookEffectListMount(5, finishedWork);
          break;
        case 1:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
          if (flags & 4)
            if (finishedRoot = finishedWork.stateNode, null === current)
              try {
                finishedRoot.componentDidMount();
              } catch (error) {
                captureCommitPhaseError(finishedWork, finishedWork.return, error);
              }
            else {
              var prevProps = resolveClassComponentProps(
                finishedWork.type,
                current.memoizedProps
              );
              current = current.memoizedState;
              try {
                finishedRoot.componentDidUpdate(
                  prevProps,
                  current,
                  finishedRoot.__reactInternalSnapshotBeforeUpdate
                );
              } catch (error$139) {
                captureCommitPhaseError(
                  finishedWork,
                  finishedWork.return,
                  error$139
                );
              }
            }
          flags & 64 && commitClassCallbacks(finishedWork);
          flags & 512 && safelyAttachRef(finishedWork, finishedWork.return);
          break;
        case 3:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
          if (flags & 64 && (finishedRoot = finishedWork.updateQueue, null !== finishedRoot)) {
            current = null;
            if (null !== finishedWork.child)
              switch (finishedWork.child.tag) {
                case 27:
                case 5:
                  current = finishedWork.child.stateNode;
                  break;
                case 1:
                  current = finishedWork.child.stateNode;
              }
            try {
              commitCallbacks(finishedRoot, current);
            } catch (error) {
              captureCommitPhaseError(finishedWork, finishedWork.return, error);
            }
          }
          break;
        case 27:
          null === current && flags & 4 && commitHostSingletonAcquisition(finishedWork);
        case 26:
        case 5:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
          null === current && flags & 4 && commitHostMount(finishedWork);
          flags & 512 && safelyAttachRef(finishedWork, finishedWork.return);
          break;
        case 12:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
          break;
        case 31:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
          flags & 4 && commitActivityHydrationCallbacks(finishedRoot, finishedWork);
          break;
        case 13:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
          flags & 4 && commitSuspenseHydrationCallbacks(finishedRoot, finishedWork);
          flags & 64 && (finishedRoot = finishedWork.memoizedState, null !== finishedRoot && (finishedRoot = finishedRoot.dehydrated, null !== finishedRoot && (finishedWork = retryDehydratedSuspenseBoundary.bind(
            null,
            finishedWork
          ), registerSuspenseInstanceRetry(finishedRoot, finishedWork))));
          break;
        case 22:
          flags = null !== finishedWork.memoizedState || offscreenSubtreeIsHidden;
          if (!flags) {
            current = null !== current && null !== current.memoizedState || offscreenSubtreeWasHidden;
            prevProps = offscreenSubtreeIsHidden;
            var prevOffscreenSubtreeWasHidden = offscreenSubtreeWasHidden;
            offscreenSubtreeIsHidden = flags;
            (offscreenSubtreeWasHidden = current) && !prevOffscreenSubtreeWasHidden ? recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              0 !== (finishedWork.subtreeFlags & 8772)
            ) : recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
            offscreenSubtreeIsHidden = prevProps;
            offscreenSubtreeWasHidden = prevOffscreenSubtreeWasHidden;
          }
          break;
        case 30:
          break;
        default:
          recursivelyTraverseLayoutEffects(finishedRoot, finishedWork);
      }
    }
    function detachFiberAfterEffects(fiber) {
      var alternate = fiber.alternate;
      null !== alternate && (fiber.alternate = null, detachFiberAfterEffects(alternate));
      fiber.child = null;
      fiber.deletions = null;
      fiber.sibling = null;
      5 === fiber.tag && (alternate = fiber.stateNode, null !== alternate && detachDeletedInstance(alternate));
      fiber.stateNode = null;
      fiber.return = null;
      fiber.dependencies = null;
      fiber.memoizedProps = null;
      fiber.memoizedState = null;
      fiber.pendingProps = null;
      fiber.stateNode = null;
      fiber.updateQueue = null;
    }
    var hostParent = null, hostParentIsContainer = false;
    function recursivelyTraverseDeletionEffects(finishedRoot, nearestMountedAncestor, parent) {
      for (parent = parent.child; null !== parent; )
        commitDeletionEffectsOnFiber(finishedRoot, nearestMountedAncestor, parent), parent = parent.sibling;
    }
    function commitDeletionEffectsOnFiber(finishedRoot, nearestMountedAncestor, deletedFiber) {
      if (injectedHook && "function" === typeof injectedHook.onCommitFiberUnmount)
        try {
          injectedHook.onCommitFiberUnmount(rendererID, deletedFiber);
        } catch (err) {
        }
      switch (deletedFiber.tag) {
        case 26:
          offscreenSubtreeWasHidden || safelyDetachRef(deletedFiber, nearestMountedAncestor);
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          deletedFiber.memoizedState ? deletedFiber.memoizedState.count-- : deletedFiber.stateNode && (deletedFiber = deletedFiber.stateNode, deletedFiber.parentNode.removeChild(deletedFiber));
          break;
        case 27:
          offscreenSubtreeWasHidden || safelyDetachRef(deletedFiber, nearestMountedAncestor);
          var prevHostParent = hostParent, prevHostParentIsContainer = hostParentIsContainer;
          isSingletonScope(deletedFiber.type) && (hostParent = deletedFiber.stateNode, hostParentIsContainer = false);
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          releaseSingletonInstance(deletedFiber.stateNode);
          hostParent = prevHostParent;
          hostParentIsContainer = prevHostParentIsContainer;
          break;
        case 5:
          offscreenSubtreeWasHidden || safelyDetachRef(deletedFiber, nearestMountedAncestor);
        case 6:
          prevHostParent = hostParent;
          prevHostParentIsContainer = hostParentIsContainer;
          hostParent = null;
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          hostParent = prevHostParent;
          hostParentIsContainer = prevHostParentIsContainer;
          if (null !== hostParent)
            if (hostParentIsContainer)
              try {
                (9 === hostParent.nodeType ? hostParent.body : "HTML" === hostParent.nodeName ? hostParent.ownerDocument.body : hostParent).removeChild(deletedFiber.stateNode);
              } catch (error) {
                captureCommitPhaseError(
                  deletedFiber,
                  nearestMountedAncestor,
                  error
                );
              }
            else
              try {
                hostParent.removeChild(deletedFiber.stateNode);
              } catch (error) {
                captureCommitPhaseError(
                  deletedFiber,
                  nearestMountedAncestor,
                  error
                );
              }
          break;
        case 18:
          null !== hostParent && (hostParentIsContainer ? (finishedRoot = hostParent, clearHydrationBoundary(
            9 === finishedRoot.nodeType ? finishedRoot.body : "HTML" === finishedRoot.nodeName ? finishedRoot.ownerDocument.body : finishedRoot,
            deletedFiber.stateNode
          ), retryIfBlockedOn(finishedRoot)) : clearHydrationBoundary(hostParent, deletedFiber.stateNode));
          break;
        case 4:
          prevHostParent = hostParent;
          prevHostParentIsContainer = hostParentIsContainer;
          hostParent = deletedFiber.stateNode.containerInfo;
          hostParentIsContainer = true;
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          hostParent = prevHostParent;
          hostParentIsContainer = prevHostParentIsContainer;
          break;
        case 0:
        case 11:
        case 14:
        case 15:
          commitHookEffectListUnmount(2, deletedFiber, nearestMountedAncestor);
          offscreenSubtreeWasHidden || commitHookEffectListUnmount(4, deletedFiber, nearestMountedAncestor);
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          break;
        case 1:
          offscreenSubtreeWasHidden || (safelyDetachRef(deletedFiber, nearestMountedAncestor), prevHostParent = deletedFiber.stateNode, "function" === typeof prevHostParent.componentWillUnmount && safelyCallComponentWillUnmount(
            deletedFiber,
            nearestMountedAncestor,
            prevHostParent
          ));
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          break;
        case 21:
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          break;
        case 22:
          offscreenSubtreeWasHidden = (prevHostParent = offscreenSubtreeWasHidden) || null !== deletedFiber.memoizedState;
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
          offscreenSubtreeWasHidden = prevHostParent;
          break;
        default:
          recursivelyTraverseDeletionEffects(
            finishedRoot,
            nearestMountedAncestor,
            deletedFiber
          );
      }
    }
    function commitActivityHydrationCallbacks(finishedRoot, finishedWork) {
      if (null === finishedWork.memoizedState && (finishedRoot = finishedWork.alternate, null !== finishedRoot && (finishedRoot = finishedRoot.memoizedState, null !== finishedRoot))) {
        finishedRoot = finishedRoot.dehydrated;
        try {
          retryIfBlockedOn(finishedRoot);
        } catch (error) {
          captureCommitPhaseError(finishedWork, finishedWork.return, error);
        }
      }
    }
    function commitSuspenseHydrationCallbacks(finishedRoot, finishedWork) {
      if (null === finishedWork.memoizedState && (finishedRoot = finishedWork.alternate, null !== finishedRoot && (finishedRoot = finishedRoot.memoizedState, null !== finishedRoot && (finishedRoot = finishedRoot.dehydrated, null !== finishedRoot))))
        try {
          retryIfBlockedOn(finishedRoot);
        } catch (error) {
          captureCommitPhaseError(finishedWork, finishedWork.return, error);
        }
    }
    function getRetryCache(finishedWork) {
      switch (finishedWork.tag) {
        case 31:
        case 13:
        case 19:
          var retryCache = finishedWork.stateNode;
          null === retryCache && (retryCache = finishedWork.stateNode = new PossiblyWeakSet());
          return retryCache;
        case 22:
          return finishedWork = finishedWork.stateNode, retryCache = finishedWork._retryCache, null === retryCache && (retryCache = finishedWork._retryCache = new PossiblyWeakSet()), retryCache;
        default:
          throw Error(formatProdErrorMessage(435, finishedWork.tag));
      }
    }
    function attachSuspenseRetryListeners(finishedWork, wakeables) {
      var retryCache = getRetryCache(finishedWork);
      wakeables.forEach(function(wakeable) {
        if (!retryCache.has(wakeable)) {
          retryCache.add(wakeable);
          var retry = resolveRetryWakeable.bind(null, finishedWork, wakeable);
          wakeable.then(retry, retry);
        }
      });
    }
    function recursivelyTraverseMutationEffects(root$jscomp$0, parentFiber) {
      var deletions = parentFiber.deletions;
      if (null !== deletions)
        for (var i = 0; i < deletions.length; i++) {
          var childToDelete = deletions[i], root3 = root$jscomp$0, returnFiber = parentFiber, parent = returnFiber;
          a: for (; null !== parent; ) {
            switch (parent.tag) {
              case 27:
                if (isSingletonScope(parent.type)) {
                  hostParent = parent.stateNode;
                  hostParentIsContainer = false;
                  break a;
                }
                break;
              case 5:
                hostParent = parent.stateNode;
                hostParentIsContainer = false;
                break a;
              case 3:
              case 4:
                hostParent = parent.stateNode.containerInfo;
                hostParentIsContainer = true;
                break a;
            }
            parent = parent.return;
          }
          if (null === hostParent) throw Error(formatProdErrorMessage(160));
          commitDeletionEffectsOnFiber(root3, returnFiber, childToDelete);
          hostParent = null;
          hostParentIsContainer = false;
          root3 = childToDelete.alternate;
          null !== root3 && (root3.return = null);
          childToDelete.return = null;
        }
      if (parentFiber.subtreeFlags & 13886)
        for (parentFiber = parentFiber.child; null !== parentFiber; )
          commitMutationEffectsOnFiber(parentFiber, root$jscomp$0), parentFiber = parentFiber.sibling;
    }
    var currentHoistableRoot = null;
    function commitMutationEffectsOnFiber(finishedWork, root3) {
      var current = finishedWork.alternate, flags = finishedWork.flags;
      switch (finishedWork.tag) {
        case 0:
        case 11:
        case 14:
        case 15:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          flags & 4 && (commitHookEffectListUnmount(3, finishedWork, finishedWork.return), commitHookEffectListMount(3, finishedWork), commitHookEffectListUnmount(5, finishedWork, finishedWork.return));
          break;
        case 1:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          flags & 512 && (offscreenSubtreeWasHidden || null === current || safelyDetachRef(current, current.return));
          flags & 64 && offscreenSubtreeIsHidden && (finishedWork = finishedWork.updateQueue, null !== finishedWork && (flags = finishedWork.callbacks, null !== flags && (current = finishedWork.shared.hiddenCallbacks, finishedWork.shared.hiddenCallbacks = null === current ? flags : current.concat(flags))));
          break;
        case 26:
          var hoistableRoot = currentHoistableRoot;
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          flags & 512 && (offscreenSubtreeWasHidden || null === current || safelyDetachRef(current, current.return));
          if (flags & 4) {
            var currentResource = null !== current ? current.memoizedState : null;
            flags = finishedWork.memoizedState;
            if (null === current)
              if (null === flags)
                if (null === finishedWork.stateNode) {
                  a: {
                    flags = finishedWork.type;
                    current = finishedWork.memoizedProps;
                    hoistableRoot = hoistableRoot.ownerDocument || hoistableRoot;
                    b: switch (flags) {
                      case "title":
                        currentResource = hoistableRoot.getElementsByTagName("title")[0];
                        if (!currentResource || currentResource[internalHoistableMarker] || currentResource[internalInstanceKey] || "http://www.w3.org/2000/svg" === currentResource.namespaceURI || currentResource.hasAttribute("itemprop"))
                          currentResource = hoistableRoot.createElement(flags), hoistableRoot.head.insertBefore(
                            currentResource,
                            hoistableRoot.querySelector("head > title")
                          );
                        setInitialProperties(currentResource, flags, current);
                        currentResource[internalInstanceKey] = finishedWork;
                        markNodeAsHoistable(currentResource);
                        flags = currentResource;
                        break a;
                      case "link":
                        var maybeNodes = getHydratableHoistableCache(
                          "link",
                          "href",
                          hoistableRoot
                        ).get(flags + (current.href || ""));
                        if (maybeNodes) {
                          for (var i = 0; i < maybeNodes.length; i++)
                            if (currentResource = maybeNodes[i], currentResource.getAttribute("href") === (null == current.href || "" === current.href ? null : current.href) && currentResource.getAttribute("rel") === (null == current.rel ? null : current.rel) && currentResource.getAttribute("title") === (null == current.title ? null : current.title) && currentResource.getAttribute("crossorigin") === (null == current.crossOrigin ? null : current.crossOrigin)) {
                              maybeNodes.splice(i, 1);
                              break b;
                            }
                        }
                        currentResource = hoistableRoot.createElement(flags);
                        setInitialProperties(currentResource, flags, current);
                        hoistableRoot.head.appendChild(currentResource);
                        break;
                      case "meta":
                        if (maybeNodes = getHydratableHoistableCache(
                          "meta",
                          "content",
                          hoistableRoot
                        ).get(flags + (current.content || ""))) {
                          for (i = 0; i < maybeNodes.length; i++)
                            if (currentResource = maybeNodes[i], currentResource.getAttribute("content") === (null == current.content ? null : "" + current.content) && currentResource.getAttribute("name") === (null == current.name ? null : current.name) && currentResource.getAttribute("property") === (null == current.property ? null : current.property) && currentResource.getAttribute("http-equiv") === (null == current.httpEquiv ? null : current.httpEquiv) && currentResource.getAttribute("charset") === (null == current.charSet ? null : current.charSet)) {
                              maybeNodes.splice(i, 1);
                              break b;
                            }
                        }
                        currentResource = hoistableRoot.createElement(flags);
                        setInitialProperties(currentResource, flags, current);
                        hoistableRoot.head.appendChild(currentResource);
                        break;
                      default:
                        throw Error(formatProdErrorMessage(468, flags));
                    }
                    currentResource[internalInstanceKey] = finishedWork;
                    markNodeAsHoistable(currentResource);
                    flags = currentResource;
                  }
                  finishedWork.stateNode = flags;
                } else
                  mountHoistable(
                    hoistableRoot,
                    finishedWork.type,
                    finishedWork.stateNode
                  );
              else
                finishedWork.stateNode = acquireResource(
                  hoistableRoot,
                  flags,
                  finishedWork.memoizedProps
                );
            else
              currentResource !== flags ? (null === currentResource ? null !== current.stateNode && (current = current.stateNode, current.parentNode.removeChild(current)) : currentResource.count--, null === flags ? mountHoistable(
                hoistableRoot,
                finishedWork.type,
                finishedWork.stateNode
              ) : acquireResource(
                hoistableRoot,
                flags,
                finishedWork.memoizedProps
              )) : null === flags && null !== finishedWork.stateNode && commitHostUpdate(
                finishedWork,
                finishedWork.memoizedProps,
                current.memoizedProps
              );
          }
          break;
        case 27:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          flags & 512 && (offscreenSubtreeWasHidden || null === current || safelyDetachRef(current, current.return));
          null !== current && flags & 4 && commitHostUpdate(
            finishedWork,
            finishedWork.memoizedProps,
            current.memoizedProps
          );
          break;
        case 5:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          flags & 512 && (offscreenSubtreeWasHidden || null === current || safelyDetachRef(current, current.return));
          if (finishedWork.flags & 32) {
            hoistableRoot = finishedWork.stateNode;
            try {
              setTextContent(hoistableRoot, "");
            } catch (error) {
              captureCommitPhaseError(finishedWork, finishedWork.return, error);
            }
          }
          flags & 4 && null != finishedWork.stateNode && (hoistableRoot = finishedWork.memoizedProps, commitHostUpdate(
            finishedWork,
            hoistableRoot,
            null !== current ? current.memoizedProps : hoistableRoot
          ));
          flags & 1024 && (needsFormReset = true);
          break;
        case 6:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          if (flags & 4) {
            if (null === finishedWork.stateNode)
              throw Error(formatProdErrorMessage(162));
            flags = finishedWork.memoizedProps;
            current = finishedWork.stateNode;
            try {
              current.nodeValue = flags;
            } catch (error) {
              captureCommitPhaseError(finishedWork, finishedWork.return, error);
            }
          }
          break;
        case 3:
          tagCaches = null;
          hoistableRoot = currentHoistableRoot;
          currentHoistableRoot = getHoistableRoot(root3.containerInfo);
          recursivelyTraverseMutationEffects(root3, finishedWork);
          currentHoistableRoot = hoistableRoot;
          commitReconciliationEffects(finishedWork);
          if (flags & 4 && null !== current && current.memoizedState.isDehydrated)
            try {
              retryIfBlockedOn(root3.containerInfo);
            } catch (error) {
              captureCommitPhaseError(finishedWork, finishedWork.return, error);
            }
          needsFormReset && (needsFormReset = false, recursivelyResetForms(finishedWork));
          break;
        case 4:
          flags = currentHoistableRoot;
          currentHoistableRoot = getHoistableRoot(
            finishedWork.stateNode.containerInfo
          );
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          currentHoistableRoot = flags;
          break;
        case 12:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          break;
        case 31:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          flags & 4 && (flags = finishedWork.updateQueue, null !== flags && (finishedWork.updateQueue = null, attachSuspenseRetryListeners(finishedWork, flags)));
          break;
        case 13:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          finishedWork.child.flags & 8192 && null !== finishedWork.memoizedState !== (null !== current && null !== current.memoizedState) && (globalMostRecentFallbackTime = now());
          flags & 4 && (flags = finishedWork.updateQueue, null !== flags && (finishedWork.updateQueue = null, attachSuspenseRetryListeners(finishedWork, flags)));
          break;
        case 22:
          hoistableRoot = null !== finishedWork.memoizedState;
          var wasHidden = null !== current && null !== current.memoizedState, prevOffscreenSubtreeIsHidden = offscreenSubtreeIsHidden, prevOffscreenSubtreeWasHidden = offscreenSubtreeWasHidden;
          offscreenSubtreeIsHidden = prevOffscreenSubtreeIsHidden || hoistableRoot;
          offscreenSubtreeWasHidden = prevOffscreenSubtreeWasHidden || wasHidden;
          recursivelyTraverseMutationEffects(root3, finishedWork);
          offscreenSubtreeWasHidden = prevOffscreenSubtreeWasHidden;
          offscreenSubtreeIsHidden = prevOffscreenSubtreeIsHidden;
          commitReconciliationEffects(finishedWork);
          if (flags & 8192)
            a: for (root3 = finishedWork.stateNode, root3._visibility = hoistableRoot ? root3._visibility & -2 : root3._visibility | 1, hoistableRoot && (null === current || wasHidden || offscreenSubtreeIsHidden || offscreenSubtreeWasHidden || recursivelyTraverseDisappearLayoutEffects(finishedWork)), current = null, root3 = finishedWork; ; ) {
              if (5 === root3.tag || 26 === root3.tag) {
                if (null === current) {
                  wasHidden = current = root3;
                  try {
                    if (currentResource = wasHidden.stateNode, hoistableRoot)
                      maybeNodes = currentResource.style, "function" === typeof maybeNodes.setProperty ? maybeNodes.setProperty("display", "none", "important") : maybeNodes.display = "none";
                    else {
                      i = wasHidden.stateNode;
                      var styleProp = wasHidden.memoizedProps.style, display = void 0 !== styleProp && null !== styleProp && styleProp.hasOwnProperty("display") ? styleProp.display : null;
                      i.style.display = null == display || "boolean" === typeof display ? "" : ("" + display).trim();
                    }
                  } catch (error) {
                    captureCommitPhaseError(wasHidden, wasHidden.return, error);
                  }
                }
              } else if (6 === root3.tag) {
                if (null === current) {
                  wasHidden = root3;
                  try {
                    wasHidden.stateNode.nodeValue = hoistableRoot ? "" : wasHidden.memoizedProps;
                  } catch (error) {
                    captureCommitPhaseError(wasHidden, wasHidden.return, error);
                  }
                }
              } else if (18 === root3.tag) {
                if (null === current) {
                  wasHidden = root3;
                  try {
                    var instance = wasHidden.stateNode;
                    hoistableRoot ? hideOrUnhideDehydratedBoundary(instance, true) : hideOrUnhideDehydratedBoundary(wasHidden.stateNode, false);
                  } catch (error) {
                    captureCommitPhaseError(wasHidden, wasHidden.return, error);
                  }
                }
              } else if ((22 !== root3.tag && 23 !== root3.tag || null === root3.memoizedState || root3 === finishedWork) && null !== root3.child) {
                root3.child.return = root3;
                root3 = root3.child;
                continue;
              }
              if (root3 === finishedWork) break a;
              for (; null === root3.sibling; ) {
                if (null === root3.return || root3.return === finishedWork) break a;
                current === root3 && (current = null);
                root3 = root3.return;
              }
              current === root3 && (current = null);
              root3.sibling.return = root3.return;
              root3 = root3.sibling;
            }
          flags & 4 && (flags = finishedWork.updateQueue, null !== flags && (current = flags.retryQueue, null !== current && (flags.retryQueue = null, attachSuspenseRetryListeners(finishedWork, current))));
          break;
        case 19:
          recursivelyTraverseMutationEffects(root3, finishedWork);
          commitReconciliationEffects(finishedWork);
          flags & 4 && (flags = finishedWork.updateQueue, null !== flags && (finishedWork.updateQueue = null, attachSuspenseRetryListeners(finishedWork, flags)));
          break;
        case 30:
          break;
        case 21:
          break;
        default:
          recursivelyTraverseMutationEffects(root3, finishedWork), commitReconciliationEffects(finishedWork);
      }
    }
    function commitReconciliationEffects(finishedWork) {
      var flags = finishedWork.flags;
      if (flags & 2) {
        try {
          for (var hostParentFiber, parentFiber = finishedWork.return; null !== parentFiber; ) {
            if (isHostParent(parentFiber)) {
              hostParentFiber = parentFiber;
              break;
            }
            parentFiber = parentFiber.return;
          }
          if (null == hostParentFiber) throw Error(formatProdErrorMessage(160));
          switch (hostParentFiber.tag) {
            case 27:
              var parent = hostParentFiber.stateNode, before = getHostSibling(finishedWork);
              insertOrAppendPlacementNode(finishedWork, before, parent);
              break;
            case 5:
              var parent$141 = hostParentFiber.stateNode;
              hostParentFiber.flags & 32 && (setTextContent(parent$141, ""), hostParentFiber.flags &= -33);
              var before$142 = getHostSibling(finishedWork);
              insertOrAppendPlacementNode(finishedWork, before$142, parent$141);
              break;
            case 3:
            case 4:
              var parent$143 = hostParentFiber.stateNode.containerInfo, before$144 = getHostSibling(finishedWork);
              insertOrAppendPlacementNodeIntoContainer(
                finishedWork,
                before$144,
                parent$143
              );
              break;
            default:
              throw Error(formatProdErrorMessage(161));
          }
        } catch (error) {
          captureCommitPhaseError(finishedWork, finishedWork.return, error);
        }
        finishedWork.flags &= -3;
      }
      flags & 4096 && (finishedWork.flags &= -4097);
    }
    function recursivelyResetForms(parentFiber) {
      if (parentFiber.subtreeFlags & 1024)
        for (parentFiber = parentFiber.child; null !== parentFiber; ) {
          var fiber = parentFiber;
          recursivelyResetForms(fiber);
          5 === fiber.tag && fiber.flags & 1024 && fiber.stateNode.reset();
          parentFiber = parentFiber.sibling;
        }
    }
    function recursivelyTraverseLayoutEffects(root3, parentFiber) {
      if (parentFiber.subtreeFlags & 8772)
        for (parentFiber = parentFiber.child; null !== parentFiber; )
          commitLayoutEffectOnFiber(root3, parentFiber.alternate, parentFiber), parentFiber = parentFiber.sibling;
    }
    function recursivelyTraverseDisappearLayoutEffects(parentFiber) {
      for (parentFiber = parentFiber.child; null !== parentFiber; ) {
        var finishedWork = parentFiber;
        switch (finishedWork.tag) {
          case 0:
          case 11:
          case 14:
          case 15:
            commitHookEffectListUnmount(4, finishedWork, finishedWork.return);
            recursivelyTraverseDisappearLayoutEffects(finishedWork);
            break;
          case 1:
            safelyDetachRef(finishedWork, finishedWork.return);
            var instance = finishedWork.stateNode;
            "function" === typeof instance.componentWillUnmount && safelyCallComponentWillUnmount(
              finishedWork,
              finishedWork.return,
              instance
            );
            recursivelyTraverseDisappearLayoutEffects(finishedWork);
            break;
          case 27:
            releaseSingletonInstance(finishedWork.stateNode);
          case 26:
          case 5:
            safelyDetachRef(finishedWork, finishedWork.return);
            recursivelyTraverseDisappearLayoutEffects(finishedWork);
            break;
          case 22:
            null === finishedWork.memoizedState && recursivelyTraverseDisappearLayoutEffects(finishedWork);
            break;
          case 30:
            recursivelyTraverseDisappearLayoutEffects(finishedWork);
            break;
          default:
            recursivelyTraverseDisappearLayoutEffects(finishedWork);
        }
        parentFiber = parentFiber.sibling;
      }
    }
    function recursivelyTraverseReappearLayoutEffects(finishedRoot$jscomp$0, parentFiber, includeWorkInProgressEffects) {
      includeWorkInProgressEffects = includeWorkInProgressEffects && 0 !== (parentFiber.subtreeFlags & 8772);
      for (parentFiber = parentFiber.child; null !== parentFiber; ) {
        var current = parentFiber.alternate, finishedRoot = finishedRoot$jscomp$0, finishedWork = parentFiber, flags = finishedWork.flags;
        switch (finishedWork.tag) {
          case 0:
          case 11:
          case 15:
            recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
            commitHookEffectListMount(4, finishedWork);
            break;
          case 1:
            recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
            current = finishedWork;
            finishedRoot = current.stateNode;
            if ("function" === typeof finishedRoot.componentDidMount)
              try {
                finishedRoot.componentDidMount();
              } catch (error) {
                captureCommitPhaseError(current, current.return, error);
              }
            current = finishedWork;
            finishedRoot = current.updateQueue;
            if (null !== finishedRoot) {
              var instance = current.stateNode;
              try {
                var hiddenCallbacks = finishedRoot.shared.hiddenCallbacks;
                if (null !== hiddenCallbacks)
                  for (finishedRoot.shared.hiddenCallbacks = null, finishedRoot = 0; finishedRoot < hiddenCallbacks.length; finishedRoot++)
                    callCallback(hiddenCallbacks[finishedRoot], instance);
              } catch (error) {
                captureCommitPhaseError(current, current.return, error);
              }
            }
            includeWorkInProgressEffects && flags & 64 && commitClassCallbacks(finishedWork);
            safelyAttachRef(finishedWork, finishedWork.return);
            break;
          case 27:
            commitHostSingletonAcquisition(finishedWork);
          case 26:
          case 5:
            recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
            includeWorkInProgressEffects && null === current && flags & 4 && commitHostMount(finishedWork);
            safelyAttachRef(finishedWork, finishedWork.return);
            break;
          case 12:
            recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
            break;
          case 31:
            recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
            includeWorkInProgressEffects && flags & 4 && commitActivityHydrationCallbacks(finishedRoot, finishedWork);
            break;
          case 13:
            recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
            includeWorkInProgressEffects && flags & 4 && commitSuspenseHydrationCallbacks(finishedRoot, finishedWork);
            break;
          case 22:
            null === finishedWork.memoizedState && recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
            safelyAttachRef(finishedWork, finishedWork.return);
            break;
          case 30:
            break;
          default:
            recursivelyTraverseReappearLayoutEffects(
              finishedRoot,
              finishedWork,
              includeWorkInProgressEffects
            );
        }
        parentFiber = parentFiber.sibling;
      }
    }
    function commitOffscreenPassiveMountEffects(current, finishedWork) {
      var previousCache = null;
      null !== current && null !== current.memoizedState && null !== current.memoizedState.cachePool && (previousCache = current.memoizedState.cachePool.pool);
      current = null;
      null !== finishedWork.memoizedState && null !== finishedWork.memoizedState.cachePool && (current = finishedWork.memoizedState.cachePool.pool);
      current !== previousCache && (null != current && current.refCount++, null != previousCache && releaseCache(previousCache));
    }
    function commitCachePassiveMountEffect(current, finishedWork) {
      current = null;
      null !== finishedWork.alternate && (current = finishedWork.alternate.memoizedState.cache);
      finishedWork = finishedWork.memoizedState.cache;
      finishedWork !== current && (finishedWork.refCount++, null != current && releaseCache(current));
    }
    function recursivelyTraversePassiveMountEffects(root3, parentFiber, committedLanes, committedTransitions) {
      if (parentFiber.subtreeFlags & 10256)
        for (parentFiber = parentFiber.child; null !== parentFiber; )
          commitPassiveMountOnFiber(
            root3,
            parentFiber,
            committedLanes,
            committedTransitions
          ), parentFiber = parentFiber.sibling;
    }
    function commitPassiveMountOnFiber(finishedRoot, finishedWork, committedLanes, committedTransitions) {
      var flags = finishedWork.flags;
      switch (finishedWork.tag) {
        case 0:
        case 11:
        case 15:
          recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          );
          flags & 2048 && commitHookEffectListMount(9, finishedWork);
          break;
        case 1:
          recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          );
          break;
        case 3:
          recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          );
          flags & 2048 && (finishedRoot = null, null !== finishedWork.alternate && (finishedRoot = finishedWork.alternate.memoizedState.cache), finishedWork = finishedWork.memoizedState.cache, finishedWork !== finishedRoot && (finishedWork.refCount++, null != finishedRoot && releaseCache(finishedRoot)));
          break;
        case 12:
          if (flags & 2048) {
            recursivelyTraversePassiveMountEffects(
              finishedRoot,
              finishedWork,
              committedLanes,
              committedTransitions
            );
            finishedRoot = finishedWork.stateNode;
            try {
              var _finishedWork$memoize2 = finishedWork.memoizedProps, id = _finishedWork$memoize2.id, onPostCommit = _finishedWork$memoize2.onPostCommit;
              "function" === typeof onPostCommit && onPostCommit(
                id,
                null === finishedWork.alternate ? "mount" : "update",
                finishedRoot.passiveEffectDuration,
                -0
              );
            } catch (error) {
              captureCommitPhaseError(finishedWork, finishedWork.return, error);
            }
          } else
            recursivelyTraversePassiveMountEffects(
              finishedRoot,
              finishedWork,
              committedLanes,
              committedTransitions
            );
          break;
        case 31:
          recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          );
          break;
        case 13:
          recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          );
          break;
        case 23:
          break;
        case 22:
          _finishedWork$memoize2 = finishedWork.stateNode;
          id = finishedWork.alternate;
          null !== finishedWork.memoizedState ? _finishedWork$memoize2._visibility & 2 ? recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          ) : recursivelyTraverseAtomicPassiveEffects(finishedRoot, finishedWork) : _finishedWork$memoize2._visibility & 2 ? recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          ) : (_finishedWork$memoize2._visibility |= 2, recursivelyTraverseReconnectPassiveEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions,
            0 !== (finishedWork.subtreeFlags & 10256) || false
          ));
          flags & 2048 && commitOffscreenPassiveMountEffects(id, finishedWork);
          break;
        case 24:
          recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          );
          flags & 2048 && commitCachePassiveMountEffect(finishedWork.alternate, finishedWork);
          break;
        default:
          recursivelyTraversePassiveMountEffects(
            finishedRoot,
            finishedWork,
            committedLanes,
            committedTransitions
          );
      }
    }
    function recursivelyTraverseReconnectPassiveEffects(finishedRoot$jscomp$0, parentFiber, committedLanes$jscomp$0, committedTransitions$jscomp$0, includeWorkInProgressEffects) {
      includeWorkInProgressEffects = includeWorkInProgressEffects && (0 !== (parentFiber.subtreeFlags & 10256) || false);
      for (parentFiber = parentFiber.child; null !== parentFiber; ) {
        var finishedRoot = finishedRoot$jscomp$0, finishedWork = parentFiber, committedLanes = committedLanes$jscomp$0, committedTransitions = committedTransitions$jscomp$0, flags = finishedWork.flags;
        switch (finishedWork.tag) {
          case 0:
          case 11:
          case 15:
            recursivelyTraverseReconnectPassiveEffects(
              finishedRoot,
              finishedWork,
              committedLanes,
              committedTransitions,
              includeWorkInProgressEffects
            );
            commitHookEffectListMount(8, finishedWork);
            break;
          case 23:
            break;
          case 22:
            var instance = finishedWork.stateNode;
            null !== finishedWork.memoizedState ? instance._visibility & 2 ? recursivelyTraverseReconnectPassiveEffects(
              finishedRoot,
              finishedWork,
              committedLanes,
              committedTransitions,
              includeWorkInProgressEffects
            ) : recursivelyTraverseAtomicPassiveEffects(
              finishedRoot,
              finishedWork
            ) : (instance._visibility |= 2, recursivelyTraverseReconnectPassiveEffects(
              finishedRoot,
              finishedWork,
              committedLanes,
              committedTransitions,
              includeWorkInProgressEffects
            ));
            includeWorkInProgressEffects && flags & 2048 && commitOffscreenPassiveMountEffects(
              finishedWork.alternate,
              finishedWork
            );
            break;
          case 24:
            recursivelyTraverseReconnectPassiveEffects(
              finishedRoot,
              finishedWork,
              committedLanes,
              committedTransitions,
              includeWorkInProgressEffects
            );
            includeWorkInProgressEffects && flags & 2048 && commitCachePassiveMountEffect(finishedWork.alternate, finishedWork);
            break;
          default:
            recursivelyTraverseReconnectPassiveEffects(
              finishedRoot,
              finishedWork,
              committedLanes,
              committedTransitions,
              includeWorkInProgressEffects
            );
        }
        parentFiber = parentFiber.sibling;
      }
    }
    function recursivelyTraverseAtomicPassiveEffects(finishedRoot$jscomp$0, parentFiber) {
      if (parentFiber.subtreeFlags & 10256)
        for (parentFiber = parentFiber.child; null !== parentFiber; ) {
          var finishedRoot = finishedRoot$jscomp$0, finishedWork = parentFiber, flags = finishedWork.flags;
          switch (finishedWork.tag) {
            case 22:
              recursivelyTraverseAtomicPassiveEffects(finishedRoot, finishedWork);
              flags & 2048 && commitOffscreenPassiveMountEffects(
                finishedWork.alternate,
                finishedWork
              );
              break;
            case 24:
              recursivelyTraverseAtomicPassiveEffects(finishedRoot, finishedWork);
              flags & 2048 && commitCachePassiveMountEffect(finishedWork.alternate, finishedWork);
              break;
            default:
              recursivelyTraverseAtomicPassiveEffects(finishedRoot, finishedWork);
          }
          parentFiber = parentFiber.sibling;
        }
    }
    var suspenseyCommitFlag = 8192;
    function recursivelyAccumulateSuspenseyCommit(parentFiber, committedLanes, suspendedState) {
      if (parentFiber.subtreeFlags & suspenseyCommitFlag)
        for (parentFiber = parentFiber.child; null !== parentFiber; )
          accumulateSuspenseyCommitOnFiber(
            parentFiber,
            committedLanes,
            suspendedState
          ), parentFiber = parentFiber.sibling;
    }
    function accumulateSuspenseyCommitOnFiber(fiber, committedLanes, suspendedState) {
      switch (fiber.tag) {
        case 26:
          recursivelyAccumulateSuspenseyCommit(
            fiber,
            committedLanes,
            suspendedState
          );
          fiber.flags & suspenseyCommitFlag && null !== fiber.memoizedState && suspendResource(
            suspendedState,
            currentHoistableRoot,
            fiber.memoizedState,
            fiber.memoizedProps
          );
          break;
        case 5:
          recursivelyAccumulateSuspenseyCommit(
            fiber,
            committedLanes,
            suspendedState
          );
          break;
        case 3:
        case 4:
          var previousHoistableRoot = currentHoistableRoot;
          currentHoistableRoot = getHoistableRoot(fiber.stateNode.containerInfo);
          recursivelyAccumulateSuspenseyCommit(
            fiber,
            committedLanes,
            suspendedState
          );
          currentHoistableRoot = previousHoistableRoot;
          break;
        case 22:
          null === fiber.memoizedState && (previousHoistableRoot = fiber.alternate, null !== previousHoistableRoot && null !== previousHoistableRoot.memoizedState ? (previousHoistableRoot = suspenseyCommitFlag, suspenseyCommitFlag = 16777216, recursivelyAccumulateSuspenseyCommit(
            fiber,
            committedLanes,
            suspendedState
          ), suspenseyCommitFlag = previousHoistableRoot) : recursivelyAccumulateSuspenseyCommit(
            fiber,
            committedLanes,
            suspendedState
          ));
          break;
        default:
          recursivelyAccumulateSuspenseyCommit(
            fiber,
            committedLanes,
            suspendedState
          );
      }
    }
    function detachAlternateSiblings(parentFiber) {
      var previousFiber = parentFiber.alternate;
      if (null !== previousFiber && (parentFiber = previousFiber.child, null !== parentFiber)) {
        previousFiber.child = null;
        do
          previousFiber = parentFiber.sibling, parentFiber.sibling = null, parentFiber = previousFiber;
        while (null !== parentFiber);
      }
    }
    function recursivelyTraversePassiveUnmountEffects(parentFiber) {
      var deletions = parentFiber.deletions;
      if (0 !== (parentFiber.flags & 16)) {
        if (null !== deletions)
          for (var i = 0; i < deletions.length; i++) {
            var childToDelete = deletions[i];
            nextEffect = childToDelete;
            commitPassiveUnmountEffectsInsideOfDeletedTree_begin(
              childToDelete,
              parentFiber
            );
          }
        detachAlternateSiblings(parentFiber);
      }
      if (parentFiber.subtreeFlags & 10256)
        for (parentFiber = parentFiber.child; null !== parentFiber; )
          commitPassiveUnmountOnFiber(parentFiber), parentFiber = parentFiber.sibling;
    }
    function commitPassiveUnmountOnFiber(finishedWork) {
      switch (finishedWork.tag) {
        case 0:
        case 11:
        case 15:
          recursivelyTraversePassiveUnmountEffects(finishedWork);
          finishedWork.flags & 2048 && commitHookEffectListUnmount(9, finishedWork, finishedWork.return);
          break;
        case 3:
          recursivelyTraversePassiveUnmountEffects(finishedWork);
          break;
        case 12:
          recursivelyTraversePassiveUnmountEffects(finishedWork);
          break;
        case 22:
          var instance = finishedWork.stateNode;
          null !== finishedWork.memoizedState && instance._visibility & 2 && (null === finishedWork.return || 13 !== finishedWork.return.tag) ? (instance._visibility &= -3, recursivelyTraverseDisconnectPassiveEffects(finishedWork)) : recursivelyTraversePassiveUnmountEffects(finishedWork);
          break;
        default:
          recursivelyTraversePassiveUnmountEffects(finishedWork);
      }
    }
    function recursivelyTraverseDisconnectPassiveEffects(parentFiber) {
      var deletions = parentFiber.deletions;
      if (0 !== (parentFiber.flags & 16)) {
        if (null !== deletions)
          for (var i = 0; i < deletions.length; i++) {
            var childToDelete = deletions[i];
            nextEffect = childToDelete;
            commitPassiveUnmountEffectsInsideOfDeletedTree_begin(
              childToDelete,
              parentFiber
            );
          }
        detachAlternateSiblings(parentFiber);
      }
      for (parentFiber = parentFiber.child; null !== parentFiber; ) {
        deletions = parentFiber;
        switch (deletions.tag) {
          case 0:
          case 11:
          case 15:
            commitHookEffectListUnmount(8, deletions, deletions.return);
            recursivelyTraverseDisconnectPassiveEffects(deletions);
            break;
          case 22:
            i = deletions.stateNode;
            i._visibility & 2 && (i._visibility &= -3, recursivelyTraverseDisconnectPassiveEffects(deletions));
            break;
          default:
            recursivelyTraverseDisconnectPassiveEffects(deletions);
        }
        parentFiber = parentFiber.sibling;
      }
    }
    function commitPassiveUnmountEffectsInsideOfDeletedTree_begin(deletedSubtreeRoot, nearestMountedAncestor) {
      for (; null !== nextEffect; ) {
        var fiber = nextEffect;
        switch (fiber.tag) {
          case 0:
          case 11:
          case 15:
            commitHookEffectListUnmount(8, fiber, nearestMountedAncestor);
            break;
          case 23:
          case 22:
            if (null !== fiber.memoizedState && null !== fiber.memoizedState.cachePool) {
              var cache = fiber.memoizedState.cachePool.pool;
              null != cache && cache.refCount++;
            }
            break;
          case 24:
            releaseCache(fiber.memoizedState.cache);
        }
        cache = fiber.child;
        if (null !== cache) cache.return = fiber, nextEffect = cache;
        else
          a: for (fiber = deletedSubtreeRoot; null !== nextEffect; ) {
            cache = nextEffect;
            var sibling = cache.sibling, returnFiber = cache.return;
            detachFiberAfterEffects(cache);
            if (cache === fiber) {
              nextEffect = null;
              break a;
            }
            if (null !== sibling) {
              sibling.return = returnFiber;
              nextEffect = sibling;
              break a;
            }
            nextEffect = returnFiber;
          }
      }
    }
    var DefaultAsyncDispatcher = {
      getCacheForType: function(resourceType) {
        var cache = readContext(CacheContext), cacheForType = cache.data.get(resourceType);
        void 0 === cacheForType && (cacheForType = resourceType(), cache.data.set(resourceType, cacheForType));
        return cacheForType;
      },
      cacheSignal: function() {
        return readContext(CacheContext).controller.signal;
      }
    }, PossiblyWeakMap = "function" === typeof WeakMap ? WeakMap : Map, executionContext = 0, workInProgressRoot = null, workInProgress = null, workInProgressRootRenderLanes = 0, workInProgressSuspendedReason = 0, workInProgressThrownValue = null, workInProgressRootDidSkipSuspendedSiblings = false, workInProgressRootIsPrerendering = false, workInProgressRootDidAttachPingListener = false, entangledRenderLanes = 0, workInProgressRootExitStatus = 0, workInProgressRootSkippedLanes = 0, workInProgressRootInterleavedUpdatedLanes = 0, workInProgressRootPingedLanes = 0, workInProgressDeferredLane = 0, workInProgressSuspendedRetryLanes = 0, workInProgressRootConcurrentErrors = null, workInProgressRootRecoverableErrors = null, workInProgressRootDidIncludeRecursiveRenderUpdate = false, globalMostRecentFallbackTime = 0, globalMostRecentTransitionTime = 0, workInProgressRootRenderTargetTime = Infinity, workInProgressTransitions = null, legacyErrorBoundariesThatAlreadyFailed = null, pendingEffectsStatus = 0, pendingEffectsRoot = null, pendingFinishedWork = null, pendingEffectsLanes = 0, pendingEffectsRemainingLanes = 0, pendingPassiveTransitions = null, pendingRecoverableErrors = null, nestedUpdateCount = 0, rootWithNestedUpdates = null;
    function requestUpdateLane() {
      return 0 !== (executionContext & 2) && 0 !== workInProgressRootRenderLanes ? workInProgressRootRenderLanes & -workInProgressRootRenderLanes : null !== ReactSharedInternals.T ? requestTransitionLane() : resolveUpdatePriority();
    }
    function requestDeferredLane() {
      if (0 === workInProgressDeferredLane)
        if (0 === (workInProgressRootRenderLanes & 536870912) || isHydrating) {
          var lane = nextTransitionDeferredLane;
          nextTransitionDeferredLane <<= 1;
          0 === (nextTransitionDeferredLane & 3932160) && (nextTransitionDeferredLane = 262144);
          workInProgressDeferredLane = lane;
        } else workInProgressDeferredLane = 536870912;
      lane = suspenseHandlerStackCursor.current;
      null !== lane && (lane.flags |= 32);
      return workInProgressDeferredLane;
    }
    function scheduleUpdateOnFiber(root3, fiber, lane) {
      if (root3 === workInProgressRoot && (2 === workInProgressSuspendedReason || 9 === workInProgressSuspendedReason) || null !== root3.cancelPendingCommit)
        prepareFreshStack(root3, 0), markRootSuspended(
          root3,
          workInProgressRootRenderLanes,
          workInProgressDeferredLane,
          false
        );
      markRootUpdated$1(root3, lane);
      if (0 === (executionContext & 2) || root3 !== workInProgressRoot)
        root3 === workInProgressRoot && (0 === (executionContext & 2) && (workInProgressRootInterleavedUpdatedLanes |= lane), 4 === workInProgressRootExitStatus && markRootSuspended(
          root3,
          workInProgressRootRenderLanes,
          workInProgressDeferredLane,
          false
        )), ensureRootIsScheduled(root3);
    }
    function performWorkOnRoot(root$jscomp$0, lanes, forceSync) {
      if (0 !== (executionContext & 6)) throw Error(formatProdErrorMessage(327));
      var shouldTimeSlice = !forceSync && 0 === (lanes & 127) && 0 === (lanes & root$jscomp$0.expiredLanes) || checkIfRootIsPrerendering(root$jscomp$0, lanes), exitStatus = shouldTimeSlice ? renderRootConcurrent(root$jscomp$0, lanes) : renderRootSync(root$jscomp$0, lanes, true), renderWasConcurrent = shouldTimeSlice;
      do {
        if (0 === exitStatus) {
          workInProgressRootIsPrerendering && !shouldTimeSlice && markRootSuspended(root$jscomp$0, lanes, 0, false);
          break;
        } else {
          forceSync = root$jscomp$0.current.alternate;
          if (renderWasConcurrent && !isRenderConsistentWithExternalStores(forceSync)) {
            exitStatus = renderRootSync(root$jscomp$0, lanes, false);
            renderWasConcurrent = false;
            continue;
          }
          if (2 === exitStatus) {
            renderWasConcurrent = lanes;
            if (root$jscomp$0.errorRecoveryDisabledLanes & renderWasConcurrent)
              var JSCompiler_inline_result = 0;
            else
              JSCompiler_inline_result = root$jscomp$0.pendingLanes & -536870913, JSCompiler_inline_result = 0 !== JSCompiler_inline_result ? JSCompiler_inline_result : JSCompiler_inline_result & 536870912 ? 536870912 : 0;
            if (0 !== JSCompiler_inline_result) {
              lanes = JSCompiler_inline_result;
              a: {
                var root3 = root$jscomp$0;
                exitStatus = workInProgressRootConcurrentErrors;
                var wasRootDehydrated = root3.current.memoizedState.isDehydrated;
                wasRootDehydrated && (prepareFreshStack(root3, JSCompiler_inline_result).flags |= 256);
                JSCompiler_inline_result = renderRootSync(
                  root3,
                  JSCompiler_inline_result,
                  false
                );
                if (2 !== JSCompiler_inline_result) {
                  if (workInProgressRootDidAttachPingListener && !wasRootDehydrated) {
                    root3.errorRecoveryDisabledLanes |= renderWasConcurrent;
                    workInProgressRootInterleavedUpdatedLanes |= renderWasConcurrent;
                    exitStatus = 4;
                    break a;
                  }
                  renderWasConcurrent = workInProgressRootRecoverableErrors;
                  workInProgressRootRecoverableErrors = exitStatus;
                  null !== renderWasConcurrent && (null === workInProgressRootRecoverableErrors ? workInProgressRootRecoverableErrors = renderWasConcurrent : workInProgressRootRecoverableErrors.push.apply(
                    workInProgressRootRecoverableErrors,
                    renderWasConcurrent
                  ));
                }
                exitStatus = JSCompiler_inline_result;
              }
              renderWasConcurrent = false;
              if (2 !== exitStatus) continue;
            }
          }
          if (1 === exitStatus) {
            prepareFreshStack(root$jscomp$0, 0);
            markRootSuspended(root$jscomp$0, lanes, 0, true);
            break;
          }
          a: {
            shouldTimeSlice = root$jscomp$0;
            renderWasConcurrent = exitStatus;
            switch (renderWasConcurrent) {
              case 0:
              case 1:
                throw Error(formatProdErrorMessage(345));
              case 4:
                if ((lanes & 4194048) !== lanes) break;
              case 6:
                markRootSuspended(
                  shouldTimeSlice,
                  lanes,
                  workInProgressDeferredLane,
                  !workInProgressRootDidSkipSuspendedSiblings
                );
                break a;
              case 2:
                workInProgressRootRecoverableErrors = null;
                break;
              case 3:
              case 5:
                break;
              default:
                throw Error(formatProdErrorMessage(329));
            }
            if ((lanes & 62914560) === lanes && (exitStatus = globalMostRecentFallbackTime + 300 - now(), 10 < exitStatus)) {
              markRootSuspended(
                shouldTimeSlice,
                lanes,
                workInProgressDeferredLane,
                !workInProgressRootDidSkipSuspendedSiblings
              );
              if (0 !== getNextLanes(shouldTimeSlice, 0, true)) break a;
              pendingEffectsLanes = lanes;
              shouldTimeSlice.timeoutHandle = scheduleTimeout(
                commitRootWhenReady.bind(
                  null,
                  shouldTimeSlice,
                  forceSync,
                  workInProgressRootRecoverableErrors,
                  workInProgressTransitions,
                  workInProgressRootDidIncludeRecursiveRenderUpdate,
                  lanes,
                  workInProgressDeferredLane,
                  workInProgressRootInterleavedUpdatedLanes,
                  workInProgressSuspendedRetryLanes,
                  workInProgressRootDidSkipSuspendedSiblings,
                  renderWasConcurrent,
                  "Throttled",
                  -0,
                  0
                ),
                exitStatus
              );
              break a;
            }
            commitRootWhenReady(
              shouldTimeSlice,
              forceSync,
              workInProgressRootRecoverableErrors,
              workInProgressTransitions,
              workInProgressRootDidIncludeRecursiveRenderUpdate,
              lanes,
              workInProgressDeferredLane,
              workInProgressRootInterleavedUpdatedLanes,
              workInProgressSuspendedRetryLanes,
              workInProgressRootDidSkipSuspendedSiblings,
              renderWasConcurrent,
              null,
              -0,
              0
            );
          }
        }
        break;
      } while (1);
      ensureRootIsScheduled(root$jscomp$0);
    }
    function commitRootWhenReady(root3, finishedWork, recoverableErrors, transitions, didIncludeRenderPhaseUpdate, lanes, spawnedLane, updatedLanes, suspendedRetryLanes, didSkipSuspendedSiblings, exitStatus, suspendedCommitReason, completedRenderStartTime, completedRenderEndTime) {
      root3.timeoutHandle = -1;
      suspendedCommitReason = finishedWork.subtreeFlags;
      if (suspendedCommitReason & 8192 || 16785408 === (suspendedCommitReason & 16785408)) {
        suspendedCommitReason = {
          stylesheets: null,
          count: 0,
          imgCount: 0,
          imgBytes: 0,
          suspenseyImages: [],
          waitingForImages: true,
          waitingForViewTransition: false,
          unsuspend: noop$1
        };
        accumulateSuspenseyCommitOnFiber(
          finishedWork,
          lanes,
          suspendedCommitReason
        );
        var timeoutOffset = (lanes & 62914560) === lanes ? globalMostRecentFallbackTime - now() : (lanes & 4194048) === lanes ? globalMostRecentTransitionTime - now() : 0;
        timeoutOffset = waitForCommitToBeReady(
          suspendedCommitReason,
          timeoutOffset
        );
        if (null !== timeoutOffset) {
          pendingEffectsLanes = lanes;
          root3.cancelPendingCommit = timeoutOffset(
            commitRoot.bind(
              null,
              root3,
              finishedWork,
              lanes,
              recoverableErrors,
              transitions,
              didIncludeRenderPhaseUpdate,
              spawnedLane,
              updatedLanes,
              suspendedRetryLanes,
              exitStatus,
              suspendedCommitReason,
              null,
              completedRenderStartTime,
              completedRenderEndTime
            )
          );
          markRootSuspended(root3, lanes, spawnedLane, !didSkipSuspendedSiblings);
          return;
        }
      }
      commitRoot(
        root3,
        finishedWork,
        lanes,
        recoverableErrors,
        transitions,
        didIncludeRenderPhaseUpdate,
        spawnedLane,
        updatedLanes,
        suspendedRetryLanes
      );
    }
    function isRenderConsistentWithExternalStores(finishedWork) {
      for (var node = finishedWork; ; ) {
        var tag = node.tag;
        if ((0 === tag || 11 === tag || 15 === tag) && node.flags & 16384 && (tag = node.updateQueue, null !== tag && (tag = tag.stores, null !== tag)))
          for (var i = 0; i < tag.length; i++) {
            var check = tag[i], getSnapshot = check.getSnapshot;
            check = check.value;
            try {
              if (!objectIs(getSnapshot(), check)) return false;
            } catch (error) {
              return false;
            }
          }
        tag = node.child;
        if (node.subtreeFlags & 16384 && null !== tag)
          tag.return = node, node = tag;
        else {
          if (node === finishedWork) break;
          for (; null === node.sibling; ) {
            if (null === node.return || node.return === finishedWork) return true;
            node = node.return;
          }
          node.sibling.return = node.return;
          node = node.sibling;
        }
      }
      return true;
    }
    function markRootSuspended(root3, suspendedLanes, spawnedLane, didAttemptEntireTree) {
      suspendedLanes &= ~workInProgressRootPingedLanes;
      suspendedLanes &= ~workInProgressRootInterleavedUpdatedLanes;
      root3.suspendedLanes |= suspendedLanes;
      root3.pingedLanes &= ~suspendedLanes;
      didAttemptEntireTree && (root3.warmLanes |= suspendedLanes);
      didAttemptEntireTree = root3.expirationTimes;
      for (var lanes = suspendedLanes; 0 < lanes; ) {
        var index$6 = 31 - clz32(lanes), lane = 1 << index$6;
        didAttemptEntireTree[index$6] = -1;
        lanes &= ~lane;
      }
      0 !== spawnedLane && markSpawnedDeferredLane(root3, spawnedLane, suspendedLanes);
    }
    function flushSyncWork$1() {
      return 0 === (executionContext & 6) ? (flushSyncWorkAcrossRoots_impl(0), false) : true;
    }
    function resetWorkInProgressStack() {
      if (null !== workInProgress) {
        if (0 === workInProgressSuspendedReason)
          var interruptedWork = workInProgress.return;
        else
          interruptedWork = workInProgress, lastContextDependency = currentlyRenderingFiber$1 = null, resetHooksOnUnwind(interruptedWork), thenableState$1 = null, thenableIndexCounter$1 = 0, interruptedWork = workInProgress;
        for (; null !== interruptedWork; )
          unwindInterruptedWork(interruptedWork.alternate, interruptedWork), interruptedWork = interruptedWork.return;
        workInProgress = null;
      }
    }
    function prepareFreshStack(root3, lanes) {
      var timeoutHandle = root3.timeoutHandle;
      -1 !== timeoutHandle && (root3.timeoutHandle = -1, cancelTimeout(timeoutHandle));
      timeoutHandle = root3.cancelPendingCommit;
      null !== timeoutHandle && (root3.cancelPendingCommit = null, timeoutHandle());
      pendingEffectsLanes = 0;
      resetWorkInProgressStack();
      workInProgressRoot = root3;
      workInProgress = timeoutHandle = createWorkInProgress(root3.current, null);
      workInProgressRootRenderLanes = lanes;
      workInProgressSuspendedReason = 0;
      workInProgressThrownValue = null;
      workInProgressRootDidSkipSuspendedSiblings = false;
      workInProgressRootIsPrerendering = checkIfRootIsPrerendering(root3, lanes);
      workInProgressRootDidAttachPingListener = false;
      workInProgressSuspendedRetryLanes = workInProgressDeferredLane = workInProgressRootPingedLanes = workInProgressRootInterleavedUpdatedLanes = workInProgressRootSkippedLanes = workInProgressRootExitStatus = 0;
      workInProgressRootRecoverableErrors = workInProgressRootConcurrentErrors = null;
      workInProgressRootDidIncludeRecursiveRenderUpdate = false;
      0 !== (lanes & 8) && (lanes |= lanes & 32);
      var allEntangledLanes = root3.entangledLanes;
      if (0 !== allEntangledLanes)
        for (root3 = root3.entanglements, allEntangledLanes &= lanes; 0 < allEntangledLanes; ) {
          var index$4 = 31 - clz32(allEntangledLanes), lane = 1 << index$4;
          lanes |= root3[index$4];
          allEntangledLanes &= ~lane;
        }
      entangledRenderLanes = lanes;
      finishQueueingConcurrentUpdates();
      return timeoutHandle;
    }
    function handleThrow(root3, thrownValue) {
      currentlyRenderingFiber = null;
      ReactSharedInternals.H = ContextOnlyDispatcher;
      thrownValue === SuspenseException || thrownValue === SuspenseActionException ? (thrownValue = getSuspendedThenable(), workInProgressSuspendedReason = 3) : thrownValue === SuspenseyCommitException ? (thrownValue = getSuspendedThenable(), workInProgressSuspendedReason = 4) : workInProgressSuspendedReason = thrownValue === SelectiveHydrationException ? 8 : null !== thrownValue && "object" === typeof thrownValue && "function" === typeof thrownValue.then ? 6 : 1;
      workInProgressThrownValue = thrownValue;
      null === workInProgress && (workInProgressRootExitStatus = 1, logUncaughtError(
        root3,
        createCapturedValueAtFiber(thrownValue, root3.current)
      ));
    }
    function shouldRemainOnPreviousScreen() {
      var handler = suspenseHandlerStackCursor.current;
      return null === handler ? true : (workInProgressRootRenderLanes & 4194048) === workInProgressRootRenderLanes ? null === shellBoundary ? true : false : (workInProgressRootRenderLanes & 62914560) === workInProgressRootRenderLanes || 0 !== (workInProgressRootRenderLanes & 536870912) ? handler === shellBoundary : false;
    }
    function pushDispatcher() {
      var prevDispatcher = ReactSharedInternals.H;
      ReactSharedInternals.H = ContextOnlyDispatcher;
      return null === prevDispatcher ? ContextOnlyDispatcher : prevDispatcher;
    }
    function pushAsyncDispatcher() {
      var prevAsyncDispatcher = ReactSharedInternals.A;
      ReactSharedInternals.A = DefaultAsyncDispatcher;
      return prevAsyncDispatcher;
    }
    function renderDidSuspendDelayIfPossible() {
      workInProgressRootExitStatus = 4;
      workInProgressRootDidSkipSuspendedSiblings || (workInProgressRootRenderLanes & 4194048) !== workInProgressRootRenderLanes && null !== suspenseHandlerStackCursor.current || (workInProgressRootIsPrerendering = true);
      0 === (workInProgressRootSkippedLanes & 134217727) && 0 === (workInProgressRootInterleavedUpdatedLanes & 134217727) || null === workInProgressRoot || markRootSuspended(
        workInProgressRoot,
        workInProgressRootRenderLanes,
        workInProgressDeferredLane,
        false
      );
    }
    function renderRootSync(root3, lanes, shouldYieldForPrerendering) {
      var prevExecutionContext = executionContext;
      executionContext |= 2;
      var prevDispatcher = pushDispatcher(), prevAsyncDispatcher = pushAsyncDispatcher();
      if (workInProgressRoot !== root3 || workInProgressRootRenderLanes !== lanes)
        workInProgressTransitions = null, prepareFreshStack(root3, lanes);
      lanes = false;
      var exitStatus = workInProgressRootExitStatus;
      a: do
        try {
          if (0 !== workInProgressSuspendedReason && null !== workInProgress) {
            var unitOfWork = workInProgress, thrownValue = workInProgressThrownValue;
            switch (workInProgressSuspendedReason) {
              case 8:
                resetWorkInProgressStack();
                exitStatus = 6;
                break a;
              case 3:
              case 2:
              case 9:
              case 6:
                null === suspenseHandlerStackCursor.current && (lanes = true);
                var reason = workInProgressSuspendedReason;
                workInProgressSuspendedReason = 0;
                workInProgressThrownValue = null;
                throwAndUnwindWorkLoop(root3, unitOfWork, thrownValue, reason);
                if (shouldYieldForPrerendering && workInProgressRootIsPrerendering) {
                  exitStatus = 0;
                  break a;
                }
                break;
              default:
                reason = workInProgressSuspendedReason, workInProgressSuspendedReason = 0, workInProgressThrownValue = null, throwAndUnwindWorkLoop(root3, unitOfWork, thrownValue, reason);
            }
          }
          workLoopSync();
          exitStatus = workInProgressRootExitStatus;
          break;
        } catch (thrownValue$165) {
          handleThrow(root3, thrownValue$165);
        }
      while (1);
      lanes && root3.shellSuspendCounter++;
      lastContextDependency = currentlyRenderingFiber$1 = null;
      executionContext = prevExecutionContext;
      ReactSharedInternals.H = prevDispatcher;
      ReactSharedInternals.A = prevAsyncDispatcher;
      null === workInProgress && (workInProgressRoot = null, workInProgressRootRenderLanes = 0, finishQueueingConcurrentUpdates());
      return exitStatus;
    }
    function workLoopSync() {
      for (; null !== workInProgress; ) performUnitOfWork(workInProgress);
    }
    function renderRootConcurrent(root3, lanes) {
      var prevExecutionContext = executionContext;
      executionContext |= 2;
      var prevDispatcher = pushDispatcher(), prevAsyncDispatcher = pushAsyncDispatcher();
      workInProgressRoot !== root3 || workInProgressRootRenderLanes !== lanes ? (workInProgressTransitions = null, workInProgressRootRenderTargetTime = now() + 500, prepareFreshStack(root3, lanes)) : workInProgressRootIsPrerendering = checkIfRootIsPrerendering(
        root3,
        lanes
      );
      a: do
        try {
          if (0 !== workInProgressSuspendedReason && null !== workInProgress) {
            lanes = workInProgress;
            var thrownValue = workInProgressThrownValue;
            b: switch (workInProgressSuspendedReason) {
              case 1:
                workInProgressSuspendedReason = 0;
                workInProgressThrownValue = null;
                throwAndUnwindWorkLoop(root3, lanes, thrownValue, 1);
                break;
              case 2:
              case 9:
                if (isThenableResolved(thrownValue)) {
                  workInProgressSuspendedReason = 0;
                  workInProgressThrownValue = null;
                  replaySuspendedUnitOfWork(lanes);
                  break;
                }
                lanes = function() {
                  2 !== workInProgressSuspendedReason && 9 !== workInProgressSuspendedReason || workInProgressRoot !== root3 || (workInProgressSuspendedReason = 7);
                  ensureRootIsScheduled(root3);
                };
                thrownValue.then(lanes, lanes);
                break a;
              case 3:
                workInProgressSuspendedReason = 7;
                break a;
              case 4:
                workInProgressSuspendedReason = 5;
                break a;
              case 7:
                isThenableResolved(thrownValue) ? (workInProgressSuspendedReason = 0, workInProgressThrownValue = null, replaySuspendedUnitOfWork(lanes)) : (workInProgressSuspendedReason = 0, workInProgressThrownValue = null, throwAndUnwindWorkLoop(root3, lanes, thrownValue, 7));
                break;
              case 5:
                var resource = null;
                switch (workInProgress.tag) {
                  case 26:
                    resource = workInProgress.memoizedState;
                  case 5:
                  case 27:
                    var hostFiber = workInProgress;
                    if (resource ? preloadResource(resource) : hostFiber.stateNode.complete) {
                      workInProgressSuspendedReason = 0;
                      workInProgressThrownValue = null;
                      var sibling = hostFiber.sibling;
                      if (null !== sibling) workInProgress = sibling;
                      else {
                        var returnFiber = hostFiber.return;
                        null !== returnFiber ? (workInProgress = returnFiber, completeUnitOfWork(returnFiber)) : workInProgress = null;
                      }
                      break b;
                    }
                }
                workInProgressSuspendedReason = 0;
                workInProgressThrownValue = null;
                throwAndUnwindWorkLoop(root3, lanes, thrownValue, 5);
                break;
              case 6:
                workInProgressSuspendedReason = 0;
                workInProgressThrownValue = null;
                throwAndUnwindWorkLoop(root3, lanes, thrownValue, 6);
                break;
              case 8:
                resetWorkInProgressStack();
                workInProgressRootExitStatus = 6;
                break a;
              default:
                throw Error(formatProdErrorMessage(462));
            }
          }
          workLoopConcurrentByScheduler();
          break;
        } catch (thrownValue$167) {
          handleThrow(root3, thrownValue$167);
        }
      while (1);
      lastContextDependency = currentlyRenderingFiber$1 = null;
      ReactSharedInternals.H = prevDispatcher;
      ReactSharedInternals.A = prevAsyncDispatcher;
      executionContext = prevExecutionContext;
      if (null !== workInProgress) return 0;
      workInProgressRoot = null;
      workInProgressRootRenderLanes = 0;
      finishQueueingConcurrentUpdates();
      return workInProgressRootExitStatus;
    }
    function workLoopConcurrentByScheduler() {
      for (; null !== workInProgress && !shouldYield(); )
        performUnitOfWork(workInProgress);
    }
    function performUnitOfWork(unitOfWork) {
      var next = beginWork(unitOfWork.alternate, unitOfWork, entangledRenderLanes);
      unitOfWork.memoizedProps = unitOfWork.pendingProps;
      null === next ? completeUnitOfWork(unitOfWork) : workInProgress = next;
    }
    function replaySuspendedUnitOfWork(unitOfWork) {
      var next = unitOfWork;
      var current = next.alternate;
      switch (next.tag) {
        case 15:
        case 0:
          next = replayFunctionComponent(
            current,
            next,
            next.pendingProps,
            next.type,
            void 0,
            workInProgressRootRenderLanes
          );
          break;
        case 11:
          next = replayFunctionComponent(
            current,
            next,
            next.pendingProps,
            next.type.render,
            next.ref,
            workInProgressRootRenderLanes
          );
          break;
        case 5:
          resetHooksOnUnwind(next);
        default:
          unwindInterruptedWork(current, next), next = workInProgress = resetWorkInProgress(next, entangledRenderLanes), next = beginWork(current, next, entangledRenderLanes);
      }
      unitOfWork.memoizedProps = unitOfWork.pendingProps;
      null === next ? completeUnitOfWork(unitOfWork) : workInProgress = next;
    }
    function throwAndUnwindWorkLoop(root3, unitOfWork, thrownValue, suspendedReason) {
      lastContextDependency = currentlyRenderingFiber$1 = null;
      resetHooksOnUnwind(unitOfWork);
      thenableState$1 = null;
      thenableIndexCounter$1 = 0;
      var returnFiber = unitOfWork.return;
      try {
        if (throwException(
          root3,
          returnFiber,
          unitOfWork,
          thrownValue,
          workInProgressRootRenderLanes
        )) {
          workInProgressRootExitStatus = 1;
          logUncaughtError(
            root3,
            createCapturedValueAtFiber(thrownValue, root3.current)
          );
          workInProgress = null;
          return;
        }
      } catch (error) {
        if (null !== returnFiber) throw workInProgress = returnFiber, error;
        workInProgressRootExitStatus = 1;
        logUncaughtError(
          root3,
          createCapturedValueAtFiber(thrownValue, root3.current)
        );
        workInProgress = null;
        return;
      }
      if (unitOfWork.flags & 32768) {
        if (isHydrating || 1 === suspendedReason) root3 = true;
        else if (workInProgressRootIsPrerendering || 0 !== (workInProgressRootRenderLanes & 536870912))
          root3 = false;
        else if (workInProgressRootDidSkipSuspendedSiblings = root3 = true, 2 === suspendedReason || 9 === suspendedReason || 3 === suspendedReason || 6 === suspendedReason)
          suspendedReason = suspenseHandlerStackCursor.current, null !== suspendedReason && 13 === suspendedReason.tag && (suspendedReason.flags |= 16384);
        unwindUnitOfWork(unitOfWork, root3);
      } else completeUnitOfWork(unitOfWork);
    }
    function completeUnitOfWork(unitOfWork) {
      var completedWork = unitOfWork;
      do {
        if (0 !== (completedWork.flags & 32768)) {
          unwindUnitOfWork(
            completedWork,
            workInProgressRootDidSkipSuspendedSiblings
          );
          return;
        }
        unitOfWork = completedWork.return;
        var next = completeWork(
          completedWork.alternate,
          completedWork,
          entangledRenderLanes
        );
        if (null !== next) {
          workInProgress = next;
          return;
        }
        completedWork = completedWork.sibling;
        if (null !== completedWork) {
          workInProgress = completedWork;
          return;
        }
        workInProgress = completedWork = unitOfWork;
      } while (null !== completedWork);
      0 === workInProgressRootExitStatus && (workInProgressRootExitStatus = 5);
    }
    function unwindUnitOfWork(unitOfWork, skipSiblings) {
      do {
        var next = unwindWork(unitOfWork.alternate, unitOfWork);
        if (null !== next) {
          next.flags &= 32767;
          workInProgress = next;
          return;
        }
        next = unitOfWork.return;
        null !== next && (next.flags |= 32768, next.subtreeFlags = 0, next.deletions = null);
        if (!skipSiblings && (unitOfWork = unitOfWork.sibling, null !== unitOfWork)) {
          workInProgress = unitOfWork;
          return;
        }
        workInProgress = unitOfWork = next;
      } while (null !== unitOfWork);
      workInProgressRootExitStatus = 6;
      workInProgress = null;
    }
    function commitRoot(root3, finishedWork, lanes, recoverableErrors, transitions, didIncludeRenderPhaseUpdate, spawnedLane, updatedLanes, suspendedRetryLanes) {
      root3.cancelPendingCommit = null;
      do
        flushPendingEffects();
      while (0 !== pendingEffectsStatus);
      if (0 !== (executionContext & 6)) throw Error(formatProdErrorMessage(327));
      if (null !== finishedWork) {
        if (finishedWork === root3.current) throw Error(formatProdErrorMessage(177));
        didIncludeRenderPhaseUpdate = finishedWork.lanes | finishedWork.childLanes;
        didIncludeRenderPhaseUpdate |= concurrentlyUpdatedLanes;
        markRootFinished(
          root3,
          lanes,
          didIncludeRenderPhaseUpdate,
          spawnedLane,
          updatedLanes,
          suspendedRetryLanes
        );
        root3 === workInProgressRoot && (workInProgress = workInProgressRoot = null, workInProgressRootRenderLanes = 0);
        pendingFinishedWork = finishedWork;
        pendingEffectsRoot = root3;
        pendingEffectsLanes = lanes;
        pendingEffectsRemainingLanes = didIncludeRenderPhaseUpdate;
        pendingPassiveTransitions = transitions;
        pendingRecoverableErrors = recoverableErrors;
        0 !== (finishedWork.subtreeFlags & 10256) || 0 !== (finishedWork.flags & 10256) ? (root3.callbackNode = null, root3.callbackPriority = 0, scheduleCallback$1(NormalPriority$1, function() {
          flushPassiveEffects();
          return null;
        })) : (root3.callbackNode = null, root3.callbackPriority = 0);
        recoverableErrors = 0 !== (finishedWork.flags & 13878);
        if (0 !== (finishedWork.subtreeFlags & 13878) || recoverableErrors) {
          recoverableErrors = ReactSharedInternals.T;
          ReactSharedInternals.T = null;
          transitions = ReactDOMSharedInternals.p;
          ReactDOMSharedInternals.p = 2;
          spawnedLane = executionContext;
          executionContext |= 4;
          try {
            commitBeforeMutationEffects(root3, finishedWork, lanes);
          } finally {
            executionContext = spawnedLane, ReactDOMSharedInternals.p = transitions, ReactSharedInternals.T = recoverableErrors;
          }
        }
        pendingEffectsStatus = 1;
        flushMutationEffects();
        flushLayoutEffects();
        flushSpawnedWork();
      }
    }
    function flushMutationEffects() {
      if (1 === pendingEffectsStatus) {
        pendingEffectsStatus = 0;
        var root3 = pendingEffectsRoot, finishedWork = pendingFinishedWork, rootMutationHasEffect = 0 !== (finishedWork.flags & 13878);
        if (0 !== (finishedWork.subtreeFlags & 13878) || rootMutationHasEffect) {
          rootMutationHasEffect = ReactSharedInternals.T;
          ReactSharedInternals.T = null;
          var previousPriority = ReactDOMSharedInternals.p;
          ReactDOMSharedInternals.p = 2;
          var prevExecutionContext = executionContext;
          executionContext |= 4;
          try {
            commitMutationEffectsOnFiber(finishedWork, root3);
            var priorSelectionInformation = selectionInformation, curFocusedElem = getActiveElementDeep(root3.containerInfo), priorFocusedElem = priorSelectionInformation.focusedElem, priorSelectionRange = priorSelectionInformation.selectionRange;
            if (curFocusedElem !== priorFocusedElem && priorFocusedElem && priorFocusedElem.ownerDocument && containsNode(
              priorFocusedElem.ownerDocument.documentElement,
              priorFocusedElem
            )) {
              if (null !== priorSelectionRange && hasSelectionCapabilities(priorFocusedElem)) {
                var start = priorSelectionRange.start, end = priorSelectionRange.end;
                void 0 === end && (end = start);
                if ("selectionStart" in priorFocusedElem)
                  priorFocusedElem.selectionStart = start, priorFocusedElem.selectionEnd = Math.min(
                    end,
                    priorFocusedElem.value.length
                  );
                else {
                  var doc = priorFocusedElem.ownerDocument || document, win = doc && doc.defaultView || window;
                  if (win.getSelection) {
                    var selection = win.getSelection(), length = priorFocusedElem.textContent.length, start$jscomp$0 = Math.min(priorSelectionRange.start, length), end$jscomp$0 = void 0 === priorSelectionRange.end ? start$jscomp$0 : Math.min(priorSelectionRange.end, length);
                    !selection.extend && start$jscomp$0 > end$jscomp$0 && (curFocusedElem = end$jscomp$0, end$jscomp$0 = start$jscomp$0, start$jscomp$0 = curFocusedElem);
                    var startMarker = getNodeForCharacterOffset(
                      priorFocusedElem,
                      start$jscomp$0
                    ), endMarker = getNodeForCharacterOffset(
                      priorFocusedElem,
                      end$jscomp$0
                    );
                    if (startMarker && endMarker && (1 !== selection.rangeCount || selection.anchorNode !== startMarker.node || selection.anchorOffset !== startMarker.offset || selection.focusNode !== endMarker.node || selection.focusOffset !== endMarker.offset)) {
                      var range = doc.createRange();
                      range.setStart(startMarker.node, startMarker.offset);
                      selection.removeAllRanges();
                      start$jscomp$0 > end$jscomp$0 ? (selection.addRange(range), selection.extend(endMarker.node, endMarker.offset)) : (range.setEnd(endMarker.node, endMarker.offset), selection.addRange(range));
                    }
                  }
                }
              }
              doc = [];
              for (selection = priorFocusedElem; selection = selection.parentNode; )
                1 === selection.nodeType && doc.push({
                  element: selection,
                  left: selection.scrollLeft,
                  top: selection.scrollTop
                });
              "function" === typeof priorFocusedElem.focus && priorFocusedElem.focus();
              for (priorFocusedElem = 0; priorFocusedElem < doc.length; priorFocusedElem++) {
                var info = doc[priorFocusedElem];
                info.element.scrollLeft = info.left;
                info.element.scrollTop = info.top;
              }
            }
            _enabled = !!eventsEnabled;
            selectionInformation = eventsEnabled = null;
          } finally {
            executionContext = prevExecutionContext, ReactDOMSharedInternals.p = previousPriority, ReactSharedInternals.T = rootMutationHasEffect;
          }
        }
        root3.current = finishedWork;
        pendingEffectsStatus = 2;
      }
    }
    function flushLayoutEffects() {
      if (2 === pendingEffectsStatus) {
        pendingEffectsStatus = 0;
        var root3 = pendingEffectsRoot, finishedWork = pendingFinishedWork, rootHasLayoutEffect = 0 !== (finishedWork.flags & 8772);
        if (0 !== (finishedWork.subtreeFlags & 8772) || rootHasLayoutEffect) {
          rootHasLayoutEffect = ReactSharedInternals.T;
          ReactSharedInternals.T = null;
          var previousPriority = ReactDOMSharedInternals.p;
          ReactDOMSharedInternals.p = 2;
          var prevExecutionContext = executionContext;
          executionContext |= 4;
          try {
            commitLayoutEffectOnFiber(root3, finishedWork.alternate, finishedWork);
          } finally {
            executionContext = prevExecutionContext, ReactDOMSharedInternals.p = previousPriority, ReactSharedInternals.T = rootHasLayoutEffect;
          }
        }
        pendingEffectsStatus = 3;
      }
    }
    function flushSpawnedWork() {
      if (4 === pendingEffectsStatus || 3 === pendingEffectsStatus) {
        pendingEffectsStatus = 0;
        requestPaint();
        var root3 = pendingEffectsRoot, finishedWork = pendingFinishedWork, lanes = pendingEffectsLanes, recoverableErrors = pendingRecoverableErrors;
        0 !== (finishedWork.subtreeFlags & 10256) || 0 !== (finishedWork.flags & 10256) ? pendingEffectsStatus = 5 : (pendingEffectsStatus = 0, pendingFinishedWork = pendingEffectsRoot = null, releaseRootPooledCache(root3, root3.pendingLanes));
        var remainingLanes = root3.pendingLanes;
        0 === remainingLanes && (legacyErrorBoundariesThatAlreadyFailed = null);
        lanesToEventPriority(lanes);
        finishedWork = finishedWork.stateNode;
        if (injectedHook && "function" === typeof injectedHook.onCommitFiberRoot)
          try {
            injectedHook.onCommitFiberRoot(
              rendererID,
              finishedWork,
              void 0,
              128 === (finishedWork.current.flags & 128)
            );
          } catch (err) {
          }
        if (null !== recoverableErrors) {
          finishedWork = ReactSharedInternals.T;
          remainingLanes = ReactDOMSharedInternals.p;
          ReactDOMSharedInternals.p = 2;
          ReactSharedInternals.T = null;
          try {
            for (var onRecoverableError = root3.onRecoverableError, i = 0; i < recoverableErrors.length; i++) {
              var recoverableError = recoverableErrors[i];
              onRecoverableError(recoverableError.value, {
                componentStack: recoverableError.stack
              });
            }
          } finally {
            ReactSharedInternals.T = finishedWork, ReactDOMSharedInternals.p = remainingLanes;
          }
        }
        0 !== (pendingEffectsLanes & 3) && flushPendingEffects();
        ensureRootIsScheduled(root3);
        remainingLanes = root3.pendingLanes;
        0 !== (lanes & 261930) && 0 !== (remainingLanes & 42) ? root3 === rootWithNestedUpdates ? nestedUpdateCount++ : (nestedUpdateCount = 0, rootWithNestedUpdates = root3) : nestedUpdateCount = 0;
        flushSyncWorkAcrossRoots_impl(0);
      }
    }
    function releaseRootPooledCache(root3, remainingLanes) {
      0 === (root3.pooledCacheLanes &= remainingLanes) && (remainingLanes = root3.pooledCache, null != remainingLanes && (root3.pooledCache = null, releaseCache(remainingLanes)));
    }
    function flushPendingEffects() {
      flushMutationEffects();
      flushLayoutEffects();
      flushSpawnedWork();
      return flushPassiveEffects();
    }
    function flushPassiveEffects() {
      if (5 !== pendingEffectsStatus) return false;
      var root3 = pendingEffectsRoot, remainingLanes = pendingEffectsRemainingLanes;
      pendingEffectsRemainingLanes = 0;
      var renderPriority = lanesToEventPriority(pendingEffectsLanes), prevTransition = ReactSharedInternals.T, previousPriority = ReactDOMSharedInternals.p;
      try {
        ReactDOMSharedInternals.p = 32 > renderPriority ? 32 : renderPriority;
        ReactSharedInternals.T = null;
        renderPriority = pendingPassiveTransitions;
        pendingPassiveTransitions = null;
        var root$jscomp$0 = pendingEffectsRoot, lanes = pendingEffectsLanes;
        pendingEffectsStatus = 0;
        pendingFinishedWork = pendingEffectsRoot = null;
        pendingEffectsLanes = 0;
        if (0 !== (executionContext & 6)) throw Error(formatProdErrorMessage(331));
        var prevExecutionContext = executionContext;
        executionContext |= 4;
        commitPassiveUnmountOnFiber(root$jscomp$0.current);
        commitPassiveMountOnFiber(
          root$jscomp$0,
          root$jscomp$0.current,
          lanes,
          renderPriority
        );
        executionContext = prevExecutionContext;
        flushSyncWorkAcrossRoots_impl(0, false);
        if (injectedHook && "function" === typeof injectedHook.onPostCommitFiberRoot)
          try {
            injectedHook.onPostCommitFiberRoot(rendererID, root$jscomp$0);
          } catch (err) {
          }
        return true;
      } finally {
        ReactDOMSharedInternals.p = previousPriority, ReactSharedInternals.T = prevTransition, releaseRootPooledCache(root3, remainingLanes);
      }
    }
    function captureCommitPhaseErrorOnRoot(rootFiber, sourceFiber, error) {
      sourceFiber = createCapturedValueAtFiber(error, sourceFiber);
      sourceFiber = createRootErrorUpdate(rootFiber.stateNode, sourceFiber, 2);
      rootFiber = enqueueUpdate(rootFiber, sourceFiber, 2);
      null !== rootFiber && (markRootUpdated$1(rootFiber, 2), ensureRootIsScheduled(rootFiber));
    }
    function captureCommitPhaseError(sourceFiber, nearestMountedAncestor, error) {
      if (3 === sourceFiber.tag)
        captureCommitPhaseErrorOnRoot(sourceFiber, sourceFiber, error);
      else
        for (; null !== nearestMountedAncestor; ) {
          if (3 === nearestMountedAncestor.tag) {
            captureCommitPhaseErrorOnRoot(
              nearestMountedAncestor,
              sourceFiber,
              error
            );
            break;
          } else if (1 === nearestMountedAncestor.tag) {
            var instance = nearestMountedAncestor.stateNode;
            if ("function" === typeof nearestMountedAncestor.type.getDerivedStateFromError || "function" === typeof instance.componentDidCatch && (null === legacyErrorBoundariesThatAlreadyFailed || !legacyErrorBoundariesThatAlreadyFailed.has(instance))) {
              sourceFiber = createCapturedValueAtFiber(error, sourceFiber);
              error = createClassErrorUpdate(2);
              instance = enqueueUpdate(nearestMountedAncestor, error, 2);
              null !== instance && (initializeClassErrorUpdate(
                error,
                instance,
                nearestMountedAncestor,
                sourceFiber
              ), markRootUpdated$1(instance, 2), ensureRootIsScheduled(instance));
              break;
            }
          }
          nearestMountedAncestor = nearestMountedAncestor.return;
        }
    }
    function attachPingListener(root3, wakeable, lanes) {
      var pingCache = root3.pingCache;
      if (null === pingCache) {
        pingCache = root3.pingCache = new PossiblyWeakMap();
        var threadIDs = /* @__PURE__ */ new Set();
        pingCache.set(wakeable, threadIDs);
      } else
        threadIDs = pingCache.get(wakeable), void 0 === threadIDs && (threadIDs = /* @__PURE__ */ new Set(), pingCache.set(wakeable, threadIDs));
      threadIDs.has(lanes) || (workInProgressRootDidAttachPingListener = true, threadIDs.add(lanes), root3 = pingSuspendedRoot.bind(null, root3, wakeable, lanes), wakeable.then(root3, root3));
    }
    function pingSuspendedRoot(root3, wakeable, pingedLanes) {
      var pingCache = root3.pingCache;
      null !== pingCache && pingCache.delete(wakeable);
      root3.pingedLanes |= root3.suspendedLanes & pingedLanes;
      root3.warmLanes &= ~pingedLanes;
      workInProgressRoot === root3 && (workInProgressRootRenderLanes & pingedLanes) === pingedLanes && (4 === workInProgressRootExitStatus || 3 === workInProgressRootExitStatus && (workInProgressRootRenderLanes & 62914560) === workInProgressRootRenderLanes && 300 > now() - globalMostRecentFallbackTime ? 0 === (executionContext & 2) && prepareFreshStack(root3, 0) : workInProgressRootPingedLanes |= pingedLanes, workInProgressSuspendedRetryLanes === workInProgressRootRenderLanes && (workInProgressSuspendedRetryLanes = 0));
      ensureRootIsScheduled(root3);
    }
    function retryTimedOutBoundary(boundaryFiber, retryLane) {
      0 === retryLane && (retryLane = claimNextRetryLane());
      boundaryFiber = enqueueConcurrentRenderForLane(boundaryFiber, retryLane);
      null !== boundaryFiber && (markRootUpdated$1(boundaryFiber, retryLane), ensureRootIsScheduled(boundaryFiber));
    }
    function retryDehydratedSuspenseBoundary(boundaryFiber) {
      var suspenseState = boundaryFiber.memoizedState, retryLane = 0;
      null !== suspenseState && (retryLane = suspenseState.retryLane);
      retryTimedOutBoundary(boundaryFiber, retryLane);
    }
    function resolveRetryWakeable(boundaryFiber, wakeable) {
      var retryLane = 0;
      switch (boundaryFiber.tag) {
        case 31:
        case 13:
          var retryCache = boundaryFiber.stateNode;
          var suspenseState = boundaryFiber.memoizedState;
          null !== suspenseState && (retryLane = suspenseState.retryLane);
          break;
        case 19:
          retryCache = boundaryFiber.stateNode;
          break;
        case 22:
          retryCache = boundaryFiber.stateNode._retryCache;
          break;
        default:
          throw Error(formatProdErrorMessage(314));
      }
      null !== retryCache && retryCache.delete(wakeable);
      retryTimedOutBoundary(boundaryFiber, retryLane);
    }
    function scheduleCallback$1(priorityLevel, callback) {
      return scheduleCallback$3(priorityLevel, callback);
    }
    var firstScheduledRoot = null, lastScheduledRoot = null, didScheduleMicrotask = false, mightHavePendingSyncWork = false, isFlushingWork = false, currentEventTransitionLane = 0;
    function ensureRootIsScheduled(root3) {
      root3 !== lastScheduledRoot && null === root3.next && (null === lastScheduledRoot ? firstScheduledRoot = lastScheduledRoot = root3 : lastScheduledRoot = lastScheduledRoot.next = root3);
      mightHavePendingSyncWork = true;
      didScheduleMicrotask || (didScheduleMicrotask = true, scheduleImmediateRootScheduleTask());
    }
    function flushSyncWorkAcrossRoots_impl(syncTransitionLanes, onlyLegacy) {
      if (!isFlushingWork && mightHavePendingSyncWork) {
        isFlushingWork = true;
        do {
          var didPerformSomeWork = false;
          for (var root$170 = firstScheduledRoot; null !== root$170; ) {
            if (0 !== syncTransitionLanes) {
              var pendingLanes = root$170.pendingLanes;
              if (0 === pendingLanes) var JSCompiler_inline_result = 0;
              else {
                var suspendedLanes = root$170.suspendedLanes, pingedLanes = root$170.pingedLanes;
                JSCompiler_inline_result = (1 << 31 - clz32(42 | syncTransitionLanes) + 1) - 1;
                JSCompiler_inline_result &= pendingLanes & ~(suspendedLanes & ~pingedLanes);
                JSCompiler_inline_result = JSCompiler_inline_result & 201326741 ? JSCompiler_inline_result & 201326741 | 1 : JSCompiler_inline_result ? JSCompiler_inline_result | 2 : 0;
              }
              0 !== JSCompiler_inline_result && (didPerformSomeWork = true, performSyncWorkOnRoot(root$170, JSCompiler_inline_result));
            } else
              JSCompiler_inline_result = workInProgressRootRenderLanes, JSCompiler_inline_result = getNextLanes(
                root$170,
                root$170 === workInProgressRoot ? JSCompiler_inline_result : 0,
                null !== root$170.cancelPendingCommit || -1 !== root$170.timeoutHandle
              ), 0 === (JSCompiler_inline_result & 3) || checkIfRootIsPrerendering(root$170, JSCompiler_inline_result) || (didPerformSomeWork = true, performSyncWorkOnRoot(root$170, JSCompiler_inline_result));
            root$170 = root$170.next;
          }
        } while (didPerformSomeWork);
        isFlushingWork = false;
      }
    }
    function processRootScheduleInImmediateTask() {
      processRootScheduleInMicrotask();
    }
    function processRootScheduleInMicrotask() {
      mightHavePendingSyncWork = didScheduleMicrotask = false;
      var syncTransitionLanes = 0;
      0 !== currentEventTransitionLane && shouldAttemptEagerTransition() && (syncTransitionLanes = currentEventTransitionLane);
      for (var currentTime = now(), prev = null, root3 = firstScheduledRoot; null !== root3; ) {
        var next = root3.next, nextLanes = scheduleTaskForRootDuringMicrotask(root3, currentTime);
        if (0 === nextLanes)
          root3.next = null, null === prev ? firstScheduledRoot = next : prev.next = next, null === next && (lastScheduledRoot = prev);
        else if (prev = root3, 0 !== syncTransitionLanes || 0 !== (nextLanes & 3))
          mightHavePendingSyncWork = true;
        root3 = next;
      }
      0 !== pendingEffectsStatus && 5 !== pendingEffectsStatus || flushSyncWorkAcrossRoots_impl(syncTransitionLanes);
      0 !== currentEventTransitionLane && (currentEventTransitionLane = 0);
    }
    function scheduleTaskForRootDuringMicrotask(root3, currentTime) {
      for (var suspendedLanes = root3.suspendedLanes, pingedLanes = root3.pingedLanes, expirationTimes = root3.expirationTimes, lanes = root3.pendingLanes & -62914561; 0 < lanes; ) {
        var index$5 = 31 - clz32(lanes), lane = 1 << index$5, expirationTime = expirationTimes[index$5];
        if (-1 === expirationTime) {
          if (0 === (lane & suspendedLanes) || 0 !== (lane & pingedLanes))
            expirationTimes[index$5] = computeExpirationTime(lane, currentTime);
        } else expirationTime <= currentTime && (root3.expiredLanes |= lane);
        lanes &= ~lane;
      }
      currentTime = workInProgressRoot;
      suspendedLanes = workInProgressRootRenderLanes;
      suspendedLanes = getNextLanes(
        root3,
        root3 === currentTime ? suspendedLanes : 0,
        null !== root3.cancelPendingCommit || -1 !== root3.timeoutHandle
      );
      pingedLanes = root3.callbackNode;
      if (0 === suspendedLanes || root3 === currentTime && (2 === workInProgressSuspendedReason || 9 === workInProgressSuspendedReason) || null !== root3.cancelPendingCommit)
        return null !== pingedLanes && null !== pingedLanes && cancelCallback$1(pingedLanes), root3.callbackNode = null, root3.callbackPriority = 0;
      if (0 === (suspendedLanes & 3) || checkIfRootIsPrerendering(root3, suspendedLanes)) {
        currentTime = suspendedLanes & -suspendedLanes;
        if (currentTime === root3.callbackPriority) return currentTime;
        null !== pingedLanes && cancelCallback$1(pingedLanes);
        switch (lanesToEventPriority(suspendedLanes)) {
          case 2:
          case 8:
            suspendedLanes = UserBlockingPriority;
            break;
          case 32:
            suspendedLanes = NormalPriority$1;
            break;
          case 268435456:
            suspendedLanes = IdlePriority;
            break;
          default:
            suspendedLanes = NormalPriority$1;
        }
        pingedLanes = performWorkOnRootViaSchedulerTask.bind(null, root3);
        suspendedLanes = scheduleCallback$3(suspendedLanes, pingedLanes);
        root3.callbackPriority = currentTime;
        root3.callbackNode = suspendedLanes;
        return currentTime;
      }
      null !== pingedLanes && null !== pingedLanes && cancelCallback$1(pingedLanes);
      root3.callbackPriority = 2;
      root3.callbackNode = null;
      return 2;
    }
    function performWorkOnRootViaSchedulerTask(root3, didTimeout) {
      if (0 !== pendingEffectsStatus && 5 !== pendingEffectsStatus)
        return root3.callbackNode = null, root3.callbackPriority = 0, null;
      var originalCallbackNode = root3.callbackNode;
      if (flushPendingEffects() && root3.callbackNode !== originalCallbackNode)
        return null;
      var workInProgressRootRenderLanes$jscomp$0 = workInProgressRootRenderLanes;
      workInProgressRootRenderLanes$jscomp$0 = getNextLanes(
        root3,
        root3 === workInProgressRoot ? workInProgressRootRenderLanes$jscomp$0 : 0,
        null !== root3.cancelPendingCommit || -1 !== root3.timeoutHandle
      );
      if (0 === workInProgressRootRenderLanes$jscomp$0) return null;
      performWorkOnRoot(root3, workInProgressRootRenderLanes$jscomp$0, didTimeout);
      scheduleTaskForRootDuringMicrotask(root3, now());
      return null != root3.callbackNode && root3.callbackNode === originalCallbackNode ? performWorkOnRootViaSchedulerTask.bind(null, root3) : null;
    }
    function performSyncWorkOnRoot(root3, lanes) {
      if (flushPendingEffects()) return null;
      performWorkOnRoot(root3, lanes, true);
    }
    function scheduleImmediateRootScheduleTask() {
      scheduleMicrotask(function() {
        0 !== (executionContext & 6) ? scheduleCallback$3(
          ImmediatePriority,
          processRootScheduleInImmediateTask
        ) : processRootScheduleInMicrotask();
      });
    }
    function requestTransitionLane() {
      if (0 === currentEventTransitionLane) {
        var actionScopeLane = currentEntangledLane;
        0 === actionScopeLane && (actionScopeLane = nextTransitionUpdateLane, nextTransitionUpdateLane <<= 1, 0 === (nextTransitionUpdateLane & 261888) && (nextTransitionUpdateLane = 256));
        currentEventTransitionLane = actionScopeLane;
      }
      return currentEventTransitionLane;
    }
    function coerceFormActionProp(actionProp) {
      return null == actionProp || "symbol" === typeof actionProp || "boolean" === typeof actionProp ? null : "function" === typeof actionProp ? actionProp : sanitizeURL("" + actionProp);
    }
    function createFormDataWithSubmitter(form, submitter) {
      var temp = submitter.ownerDocument.createElement("input");
      temp.name = submitter.name;
      temp.value = submitter.value;
      form.id && temp.setAttribute("form", form.id);
      submitter.parentNode.insertBefore(temp, submitter);
      form = new FormData(form);
      temp.parentNode.removeChild(temp);
      return form;
    }
    function extractEvents$1(dispatchQueue, domEventName, maybeTargetInst, nativeEvent, nativeEventTarget) {
      if ("submit" === domEventName && maybeTargetInst && maybeTargetInst.stateNode === nativeEventTarget) {
        var action = coerceFormActionProp(
          (nativeEventTarget[internalPropsKey] || null).action
        ), submitter = nativeEvent.submitter;
        submitter && (domEventName = (domEventName = submitter[internalPropsKey] || null) ? coerceFormActionProp(domEventName.formAction) : submitter.getAttribute("formAction"), null !== domEventName && (action = domEventName, submitter = null));
        var event = new SyntheticEvent(
          "action",
          "action",
          null,
          nativeEvent,
          nativeEventTarget
        );
        dispatchQueue.push({
          event,
          listeners: [
            {
              instance: null,
              listener: function() {
                if (nativeEvent.defaultPrevented) {
                  if (0 !== currentEventTransitionLane) {
                    var formData = submitter ? createFormDataWithSubmitter(nativeEventTarget, submitter) : new FormData(nativeEventTarget);
                    startHostTransition(
                      maybeTargetInst,
                      {
                        pending: true,
                        data: formData,
                        method: nativeEventTarget.method,
                        action
                      },
                      null,
                      formData
                    );
                  }
                } else
                  "function" === typeof action && (event.preventDefault(), formData = submitter ? createFormDataWithSubmitter(nativeEventTarget, submitter) : new FormData(nativeEventTarget), startHostTransition(
                    maybeTargetInst,
                    {
                      pending: true,
                      data: formData,
                      method: nativeEventTarget.method,
                      action
                    },
                    action,
                    formData
                  ));
              },
              currentTarget: nativeEventTarget
            }
          ]
        });
      }
    }
    for (var i$jscomp$inline_1577 = 0; i$jscomp$inline_1577 < simpleEventPluginEvents.length; i$jscomp$inline_1577++) {
      var eventName$jscomp$inline_1578 = simpleEventPluginEvents[i$jscomp$inline_1577], domEventName$jscomp$inline_1579 = eventName$jscomp$inline_1578.toLowerCase(), capitalizedEvent$jscomp$inline_1580 = eventName$jscomp$inline_1578[0].toUpperCase() + eventName$jscomp$inline_1578.slice(1);
      registerSimpleEvent(
        domEventName$jscomp$inline_1579,
        "on" + capitalizedEvent$jscomp$inline_1580
      );
    }
    registerSimpleEvent(ANIMATION_END, "onAnimationEnd");
    registerSimpleEvent(ANIMATION_ITERATION, "onAnimationIteration");
    registerSimpleEvent(ANIMATION_START, "onAnimationStart");
    registerSimpleEvent("dblclick", "onDoubleClick");
    registerSimpleEvent("focusin", "onFocus");
    registerSimpleEvent("focusout", "onBlur");
    registerSimpleEvent(TRANSITION_RUN, "onTransitionRun");
    registerSimpleEvent(TRANSITION_START, "onTransitionStart");
    registerSimpleEvent(TRANSITION_CANCEL, "onTransitionCancel");
    registerSimpleEvent(TRANSITION_END, "onTransitionEnd");
    registerDirectEvent("onMouseEnter", ["mouseout", "mouseover"]);
    registerDirectEvent("onMouseLeave", ["mouseout", "mouseover"]);
    registerDirectEvent("onPointerEnter", ["pointerout", "pointerover"]);
    registerDirectEvent("onPointerLeave", ["pointerout", "pointerover"]);
    registerTwoPhaseEvent(
      "onChange",
      "change click focusin focusout input keydown keyup selectionchange".split(" ")
    );
    registerTwoPhaseEvent(
      "onSelect",
      "focusout contextmenu dragend focusin keydown keyup mousedown mouseup selectionchange".split(
        " "
      )
    );
    registerTwoPhaseEvent("onBeforeInput", [
      "compositionend",
      "keypress",
      "textInput",
      "paste"
    ]);
    registerTwoPhaseEvent(
      "onCompositionEnd",
      "compositionend focusout keydown keypress keyup mousedown".split(" ")
    );
    registerTwoPhaseEvent(
      "onCompositionStart",
      "compositionstart focusout keydown keypress keyup mousedown".split(" ")
    );
    registerTwoPhaseEvent(
      "onCompositionUpdate",
      "compositionupdate focusout keydown keypress keyup mousedown".split(" ")
    );
    var mediaEventTypes = "abort canplay canplaythrough durationchange emptied encrypted ended error loadeddata loadedmetadata loadstart pause play playing progress ratechange resize seeked seeking stalled suspend timeupdate volumechange waiting".split(
      " "
    ), nonDelegatedEvents = new Set(
      "beforetoggle cancel close invalid load scroll scrollend toggle".split(" ").concat(mediaEventTypes)
    );
    function processDispatchQueue(dispatchQueue, eventSystemFlags) {
      eventSystemFlags = 0 !== (eventSystemFlags & 4);
      for (var i = 0; i < dispatchQueue.length; i++) {
        var _dispatchQueue$i = dispatchQueue[i], event = _dispatchQueue$i.event;
        _dispatchQueue$i = _dispatchQueue$i.listeners;
        a: {
          var previousInstance = void 0;
          if (eventSystemFlags)
            for (var i$jscomp$0 = _dispatchQueue$i.length - 1; 0 <= i$jscomp$0; i$jscomp$0--) {
              var _dispatchListeners$i = _dispatchQueue$i[i$jscomp$0], instance = _dispatchListeners$i.instance, currentTarget = _dispatchListeners$i.currentTarget;
              _dispatchListeners$i = _dispatchListeners$i.listener;
              if (instance !== previousInstance && event.isPropagationStopped())
                break a;
              previousInstance = _dispatchListeners$i;
              event.currentTarget = currentTarget;
              try {
                previousInstance(event);
              } catch (error) {
                reportGlobalError(error);
              }
              event.currentTarget = null;
              previousInstance = instance;
            }
          else
            for (i$jscomp$0 = 0; i$jscomp$0 < _dispatchQueue$i.length; i$jscomp$0++) {
              _dispatchListeners$i = _dispatchQueue$i[i$jscomp$0];
              instance = _dispatchListeners$i.instance;
              currentTarget = _dispatchListeners$i.currentTarget;
              _dispatchListeners$i = _dispatchListeners$i.listener;
              if (instance !== previousInstance && event.isPropagationStopped())
                break a;
              previousInstance = _dispatchListeners$i;
              event.currentTarget = currentTarget;
              try {
                previousInstance(event);
              } catch (error) {
                reportGlobalError(error);
              }
              event.currentTarget = null;
              previousInstance = instance;
            }
        }
      }
    }
    function listenToNonDelegatedEvent(domEventName, targetElement) {
      var JSCompiler_inline_result = targetElement[internalEventHandlersKey];
      void 0 === JSCompiler_inline_result && (JSCompiler_inline_result = targetElement[internalEventHandlersKey] = /* @__PURE__ */ new Set());
      var listenerSetKey = domEventName + "__bubble";
      JSCompiler_inline_result.has(listenerSetKey) || (addTrappedEventListener(targetElement, domEventName, 2, false), JSCompiler_inline_result.add(listenerSetKey));
    }
    function listenToNativeEvent(domEventName, isCapturePhaseListener, target) {
      var eventSystemFlags = 0;
      isCapturePhaseListener && (eventSystemFlags |= 4);
      addTrappedEventListener(
        target,
        domEventName,
        eventSystemFlags,
        isCapturePhaseListener
      );
    }
    var listeningMarker = "_reactListening" + Math.random().toString(36).slice(2);
    function listenToAllSupportedEvents(rootContainerElement) {
      if (!rootContainerElement[listeningMarker]) {
        rootContainerElement[listeningMarker] = true;
        allNativeEvents.forEach(function(domEventName) {
          "selectionchange" !== domEventName && (nonDelegatedEvents.has(domEventName) || listenToNativeEvent(domEventName, false, rootContainerElement), listenToNativeEvent(domEventName, true, rootContainerElement));
        });
        var ownerDocument = 9 === rootContainerElement.nodeType ? rootContainerElement : rootContainerElement.ownerDocument;
        null === ownerDocument || ownerDocument[listeningMarker] || (ownerDocument[listeningMarker] = true, listenToNativeEvent("selectionchange", false, ownerDocument));
      }
    }
    function addTrappedEventListener(targetContainer, domEventName, eventSystemFlags, isCapturePhaseListener) {
      switch (getEventPriority(domEventName)) {
        case 2:
          var listenerWrapper = dispatchDiscreteEvent;
          break;
        case 8:
          listenerWrapper = dispatchContinuousEvent;
          break;
        default:
          listenerWrapper = dispatchEvent;
      }
      eventSystemFlags = listenerWrapper.bind(
        null,
        domEventName,
        eventSystemFlags,
        targetContainer
      );
      listenerWrapper = void 0;
      !passiveBrowserEventsSupported || "touchstart" !== domEventName && "touchmove" !== domEventName && "wheel" !== domEventName || (listenerWrapper = true);
      isCapturePhaseListener ? void 0 !== listenerWrapper ? targetContainer.addEventListener(domEventName, eventSystemFlags, {
        capture: true,
        passive: listenerWrapper
      }) : targetContainer.addEventListener(domEventName, eventSystemFlags, true) : void 0 !== listenerWrapper ? targetContainer.addEventListener(domEven