var devvit = (function() {
  "use strict";
  function defineContentScript(definition2) {
    return definition2;
  }
  const definition = defineContentScript({
    matches: ["https://*.devvit.net/*"],
    runAt: "document_start",
    allFrames: true,
    // Run in all frames including iframes
    async main() {
      await Promise.resolve().then(() => devvit2);
    }
  });
  const browser$1 = globalThis.browser?.runtime?.id ? globalThis.browser : globalThis.chrome;
  const browser = browser$1;
  function print$1(method, ...args) {
    return;
  }
  const logger$1 = {
    debug: (...args) => print$1(console.debug, ...args),
    log: (...args) => print$1(console.log, ...args),
    warn: (...args) => print$1(console.warn, ...args),
    error: (...args) => print$1(console.error, ...args)
  };
  class WxtLocationChangeEvent extends Event {
    constructor(newUrl, oldUrl) {
      super(WxtLocationChangeEvent.EVENT_NAME, {});
      this.newUrl = newUrl;
      this.oldUrl = oldUrl;
    }
    static EVENT_NAME = getUniqueEventName("wxt:locationchange");
  }
  function getUniqueEventName(eventName) {
    return `${browser?.runtime?.id}:${"devvit"}:${eventName}`;
  }
  function createLocationWatcher(ctx) {
    let interval;
    let oldUrl;
    return {
      /**
       * Ensure the location watcher is actively looking for URL changes. If it's already watching,
       * this is a noop.
       */
      run() {
        if (interval != null) return;
        oldUrl = new URL(location.href);
        interval = ctx.setInterval(() => {
          let newUrl = new URL(location.href);
          if (newUrl.href !== oldUrl.href) {
            window.dispatchEvent(new WxtLocationChangeEvent(newUrl, oldUrl));
            oldUrl = newUrl;
          }
        }, 1e3);
      }
    };
  }
  class ContentScriptContext {
    constructor(contentScriptName, options) {
      this.contentScriptName = contentScriptName;
      this.options = options;
      this.abortController = new AbortController();
      if (this.isTopFrame) {
        this.listenForNewerScripts({ ignoreFirstEvent: true });
        this.stopOldScripts();
      } else {
        this.listenForNewerScripts();
      }
    }
    static SCRIPT_STARTED_MESSAGE_TYPE = getUniqueEventName(
      "wxt:content-script-started"
    );
    isTopFrame = window.self === window.top;
    abortController;
    locationWatcher = createLocationWatcher(this);
    receivedMessageIds = /* @__PURE__ */ new Set();
    get signal() {
      return this.abortController.signal;
    }
    abort(reason) {
      return this.abortController.abort(reason);
    }
    get isInvalid() {
      if (browser.runtime.id == null) {
        this.notifyInvalidated();
      }
      return this.signal.aborted;
    }
    get isValid() {
      return !this.isInvalid;
    }
    /**
     * Add a listener that is called when the content script's context is invalidated.
     *
     * @returns A function to remove the listener.
     *
     * @example
     * browser.runtime.onMessage.addListener(cb);
     * const removeInvalidatedListener = ctx.onInvalidated(() => {
     *   browser.runtime.onMessage.removeListener(cb);
     * })
     * // ...
     * removeInvalidatedListener();
     */
    onInvalidated(cb) {
      this.signal.addEventListener("abort", cb);
      return () => this.signal.removeEventListener("abort", cb);
    }
    /**
     * Return a promise that never resolves. Useful if you have an async function that shouldn't run
     * after the context is expired.
     *
     * @example
     * const getValueFromStorage = async () => {
     *   if (ctx.isInvalid) return ctx.block();
     *
     *   // ...
     * }
     */
    block() {
      return new Promise(() => {
      });
    }
    /**
     * Wrapper around `window.setInterval` that automatically clears the interval when invalidated.
     *
     * Intervals can be cleared by calling the normal `clearInterval` function.
     */
    setInterval(handler, timeout) {
      const id = setInterval(() => {
        if (this.isValid) handler();
      }, timeout);
      this.onInvalidated(() => clearInterval(id));
      return id;
    }
    /**
     * Wrapper around `window.setTimeout` that automatically clears the interval when invalidated.
     *
     * Timeouts can be cleared by calling the normal `setTimeout` function.
     */
    setTimeout(handler, timeout) {
      const id = setTimeout(() => {
        if (this.isValid) handler();
      }, timeout);
      this.onInvalidated(() => clearTimeout(id));
      return id;
    }
    /**
     * Wrapper around `window.requestAnimationFrame` that automatically cancels the request when
     * invalidated.
     *
     * Callbacks can be canceled by calling the normal `cancelAnimationFrame` function.
     */
    requestAnimationFrame(callback) {
      const id = requestAnimationFrame((...args) => {
        if (this.isValid) callback(...args);
      });
      this.onInvalidated(() => cancelAnimationFrame(id));
      return id;
    }
    /**
     * Wrapper around `window.requestIdleCallback` that automatically cancels the request when
     * invalidated.
     *
     * Callbacks can be canceled by calling the normal `cancelIdleCallback` function.
     */
    requestIdleCallback(callback, options) {
      const id = requestIdleCallback((...args) => {
        if (!this.signal.aborted) callback(...args);
      }, options);
      this.onInvalidated(() => cancelIdleCallback(id));
      return id;
    }
    addEventListener(target, type, handler, options) {
      if (type === "wxt:locationchange") {
        if (this.isValid) this.locationWatcher.run();
      }
      target.addEventListener?.(
        type.startsWith("wxt:") ? getUniqueEventName(type) : type,
        handler,
        {
          ...options,
          signal: this.signal
        }
      );
    }
    /**
     * @internal
     * Abort the abort controller and execute all `onInvalidated` listeners.
     */
    notifyInvalidated() {
      this.abort("Content script context invalidated");
      logger$1.debug(
        `Content script "${this.contentScriptName}" context invalidated`
      );
    }
    stopOldScripts() {
      window.postMessage(
        {
          type: ContentScriptContext.SCRIPT_STARTED_MESSAGE_TYPE,
          contentScriptName: this.contentScriptName,
          messageId: Math.random().toString(36).slice(2)
        },
        "*"
      );
    }
    verifyScriptStartedEvent(event) {
      const isScriptStartedEvent = event.data?.type === ContentScriptContext.SCRIPT_STARTED_MESSAGE_TYPE;
      const isSameContentScript = event.data?.contentScriptName === this.contentScriptName;
      const isNotDuplicate = !this.receivedMessageIds.has(event.data?.messageId);
      return isScriptStartedEvent && isSameContentScript && isNotDuplicate;
    }
    listenForNewerScripts(options) {
      let isFirst = true;
      const cb = (event) => {
        if (this.verifyScriptStartedEvent(event)) {
          this.receivedMessageIds.add(event.data.messageId);
          const wasFirst = isFirst;
          isFirst = false;
          if (wasFirst && options?.ignoreFirstEvent) return;
          this.notifyInvalidated();
        }
      };
      addEventListener("message", cb);
      this.onInvalidated(() => removeEventListener("message", cb));
    }
  }
  function initPlugins() {
  }
  function print(method, ...args) {
    return;
  }
  const logger = {
    debug: (...args) => print(console.debug, ...args),
    log: (...args) => print(console.log, ...args),
    warn: (...args) => print(console.warn, ...args),
    error: (...args) => print(console.error, ...args)
  };
  const result = (async () => {
    try {
      initPlugins();
      const { main, ...options } = definition;
      const ctx = new ContentScriptContext("devvit", options);
      return await main(ctx);
    } catch (err) {
      logger.error(
        `The content script "${"devvit"}" crashed on startup!`,
        err
      );
      throw err;
    }
  })();
  const DEFAULT_MAX_STORED_LOGS = 5e3;
  const BATCH_WRITE_INTERVAL = 1e4;
  const BATCH_SIZE_THRESHOLD = 50;
  const _Logger = class _Logger {
    constructor(context, config, parentContext) {
      this.parentContext = parentContext;
      this.config = {
        context,
        remoteLogging: config?.remoteLogging ?? true,
        remoteUrl: config?.remoteUrl ?? "http://localhost:7856/log",
        consoleLogging: config?.consoleLogging ?? true,
        storeLogs: config?.storeLogs ?? true,
        maxStoredLogs: config?.maxStoredLogs ?? DEFAULT_MAX_STORED_LOGS
      };
      if (typeof chrome !== "undefined" && chrome.storage) {
        chrome.storage.local.get(["automationConfig"], (result2) => {
          if (result2.automationConfig?.remoteLogging !== void 0) {
            this.config.remoteLogging = result2.automationConfig.remoteLogging;
          }
          if (result2.automationConfig?.storeLogs !== void 0) {
            this.config.storeLogs = result2.automationConfig.storeLogs;
          }
          if (result2.automationConfig?.maxStoredLogs !== void 0) {
            this.config.maxStoredLogs = result2.automationConfig.maxStoredLogs;
          }
        });
        chrome.storage.onChanged.addListener((changes, areaName) => {
          if (areaName === "local" && changes.automationConfig?.newValue) {
            const newConfig = changes.automationConfig.newValue;
            if (newConfig.remoteLogging !== void 0) {
              this.config.remoteLogging = newConfig.remoteLogging;
            }
            if (newConfig.storeLogs !== void 0) {
              this.config.storeLogs = newConfig.storeLogs;
            }
            if (newConfig.maxStoredLogs !== void 0) {
              this.config.maxStoredLogs = newConfig.maxStoredLogs;
            }
          }
        });
      }
    }
    /**
     * Send log to remote server
     */
    async sendToRemote(entry) {
      if (!this.config.remoteLogging) return;
      try {
        await fetch(this.config.remoteUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify(entry)
        }).catch(() => {
        });
      } catch (error) {
      }
    }
    /**
     * Flush buffered logs to chrome.storage
     * This is called periodically or when buffer reaches threshold
     */
    static async flushLogsToStorage() {
      if (typeof chrome === "undefined" || !chrome.storage) return;
      if (_Logger.logBuffer.length === 0) return;
      if (_Logger.flushTimer) {
        clearTimeout(_Logger.flushTimer);
        _Logger.flushTimer = null;
      }
      _Logger.isFlushScheduled = false;
      const logsToFlush = [..._Logger.logBuffer];
      _Logger.logBuffer = [];
      try {
        const result2 = await chrome.storage.local.get(["debugLogs", "automationConfig"]);
        const existingLogs = result2.debugLogs || [];
        const maxLogs = result2.automationConfig?.maxStoredLogs ?? DEFAULT_MAX_STORED_LOGS;
        const allLogs = [...existingLogs, ...logsToFlush];
        if (allLogs.length > maxLogs) {
          allLogs.splice(0, allLogs.length - maxLogs);
        }
        await chrome.storage.local.set({ debugLogs: allLogs });
      } catch (error) {
        _Logger.logBuffer.unshift(...logsToFlush);
        console.error("[LF] Failed to flush logs:", error);
      }
      if (_Logger.logBuffer.length > 0) {
        _Logger.scheduleFlush();
      }
    }
    /**
     * Schedule a flush to happen after the interval
     */
    static scheduleFlush() {
      if (_Logger.isFlushScheduled) return;
      _Logger.isFlushScheduled = true;
      _Logger.flushTimer = setTimeout(() => {
        _Logger.flushLogsToStorage();
      }, BATCH_WRITE_INTERVAL);
    }
    /**
     * Store log entry in buffer (will be flushed periodically)
     */
    storeLog(entry) {
      if (!this.config.storeLogs) return;
      if (typeof chrome === "undefined" || !chrome.storage) return;
      try {
        _Logger.logBuffer.push(entry);
        if (_Logger.logBuffer.length >= BATCH_SIZE_THRESHOLD) {
          _Logger.flushLogsToStorage();
        } else {
          _Logger.scheduleFlush();
        }
      } catch (error) {
        console.error("[LF] Failed to buffer log:", error);
      }
    }
    /**
     * Format message with prefix
     */
    formatMessage(message) {
      const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
      return `[LF][${fullContext}] ${message}`;
    }
    /**
     * Serialize data for logging
     */
    serializeData(data) {
      if (data === void 0) return void 0;
      try {
        return JSON.parse(JSON.stringify(data));
      } catch (error) {
        return String(data);
      }
    }
    /**
     * Core logging function
     */
    logInternal(level, ...args) {
      const message = args.map((arg) => {
        if (typeof arg === "string") return arg;
        if (typeof arg === "object") {
          try {
            return JSON.stringify(arg);
          } catch (error) {
            return "[Circular Reference]";
          }
        }
        return String(arg);
      }).join(" ");
      const entry = {
        timestamp: (/* @__PURE__ */ new Date()).toISOString(),
        context: this.config.context,
        level,
        message,
        data: args.length > 1 ? this.serializeData(args.slice(1)) : void 0
      };
      if (this.config.consoleLogging) {
        const consoleMethod = console[level] || console.log;
        const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
        consoleMethod(`[LF][${fullContext}]`, ...args);
      }
      this.storeLog(entry);
      this.sendToRemote(entry);
    }
    /**
     * Public logging methods - support unlimited parameters like console.log()
     */
    log(...args) {
      this.logInternal("log", ...args);
    }
    info(...args) {
      this.logInternal("info", ...args);
    }
    warn(...args) {
      this.logInternal("warn", ...args);
    }
    error(...args) {
      this.logInternal("error", ...args);
    }
    debug(...args) {
      this.logInternal("debug", ...args);
    }
    /**
     * Update logger configuration
     */
    setConfig(config) {
      this.config = { ...this.config, ...config };
    }
    /**
     * Enable/disable remote logging
     */
    setRemoteLogging(enabled) {
      this.config.remoteLogging = enabled;
    }
    /**
     * Enable/disable console logging
     */
    setConsoleLogging(enabled) {
      this.config.consoleLogging = enabled;
    }
    /**
     * Create a nested logger with additional context
     */
    createNestedLogger(nestedContext) {
      const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
      return new _Logger(
        nestedContext,
        {
          remoteLogging: this.config.remoteLogging,
          remoteUrl: this.config.remoteUrl,
          consoleLogging: this.config.consoleLogging
        },
        fullContext
      );
    }
    /**
     * Flush all buffered logs to storage immediately
     * Call this before extension unload to prevent log loss
     */
    static async flushLogs() {
      return _Logger.flushLogsToStorage();
    }
  };
  _Logger.logBuffer = [];
  _Logger.flushTimer = null;
  _Logger.isFlushScheduled = false;
  let Logger = _Logger;
  function createLogger(context, config, parentContext) {
    return new Logger(context, config, parentContext);
  }
  createLogger("POPUP");
  createLogger("SW");
  createLogger("REDDIT");
  const devvitLogger = createLogger("DEVVIT");
  const devvitGIAELogger = createLogger("DEVVIT-GIAE");
  class GameState {
    constructor() {
      this.livesRemaining = 3;
      this.currentEncounter = 0;
      this.totalEncounters = 0;
      this.postId = null;
      this.difficulty = null;
      this.missionMetadata = null;
      this.currentScreen = "unknown";
      this._storageLoadAttempted = false;
    }
    /**
     * Update state from DOM (called every tick)
     */
    updateFromDOM() {
      this.livesRemaining = this.readLivesFromDOM();
    }
    /**
     * Read lives from .lives-container in DOM
     */
    readLivesFromDOM() {
      const livesContainer = document.querySelector(".lives-container");
      if (!livesContainer) return 3;
      const filledHearts = livesContainer.querySelectorAll('img[src*="Heart_Full.png"]').length;
      return filledHearts;
    }
    /**
     * Set mission data from initialData message
     */
    setMissionData(metadata, postId) {
      this.postId = postId;
      this.missionMetadata = metadata;
      this.totalEncounters = metadata?.mission?.encounters?.length || 0;
      this.difficulty = metadata?.mission?.difficulty || null;
      this.currentEncounter = -1;
      devvitGIAELogger.log("[GameState] Mission data set from initialData", {
        postId,
        totalEncounters: this.totalEncounters,
        startingEncounter: this.currentEncounter,
        firstEncounterType: metadata?.mission?.encounters?.[0]?.type,
        note: "Starting at -1 (initial battle not in encounters array)",
        hasMetadata: !!metadata,
        hasMission: !!metadata?.mission,
        hasEncounters: !!metadata?.mission?.encounters,
        encountersLength: metadata?.mission?.encounters?.length || 0
      });
    }
    /**
     * Load mission metadata from storage as fallback
     * Called if initialData message doesn't have complete data
     */
    async loadMissionDataFromStorage(postId) {
      this._storageLoadAttempted = true;
      try {
        const { getMission: getMission2 } = await Promise.resolve().then(() => missions);
        const mission = await getMission2(postId);
        if (!mission?.encounters) {
          devvitGIAELogger.warn("[GameState] No encounters in storage for", postId);
          return false;
        }
        const storageMetadata = {
          mission: {
            encounters: mission.encounters,
            difficulty: mission.difficulty,
            environment: mission.environment,
            minLevel: mission.minLevel,
            maxLevel: mission.maxLevel,
            foodImage: mission.foodImage,
            foodName: mission.foodName,
            authorWeaponId: mission.authorWeaponId || "",
            chef: mission.chef || "",
            cart: mission.cart || "",
            rarity: mission.rarity
          },
          missionAuthorName: mission.missionAuthorName,
          missionTitle: mission.missionTitle,
          enemyTauntData: []
        };
        if (this.missionMetadata) {
          const initialDataEncounters = this.missionMetadata?.mission?.encounters?.length || 0;
          const storageEncounters = mission.encounters?.length || 0;
          if (initialDataEncounters !== storageEncounters) {
            devvitGIAELogger.warn("[GameState] Metadata mismatch!", {
              postId,
              initialDataEncounters,
              storageEncounters,
              initialDataDifficulty: this.missionMetadata?.mission?.difficulty,
              storageDifficulty: mission.difficulty
            });
          } else {
            devvitGIAELogger.log("[GameState] Storage metadata matches initialData");
          }
        } else {
          devvitGIAELogger.log("[GameState] Using storage metadata as fallback", {
            postId,
            encountersLength: mission.encounters?.length || 0
          });
          this.missionMetadata = storageMetadata;
          this.totalEncounters = mission.encounters?.length || 0;
          this.difficulty = mission.difficulty || null;
        }
        return true;
      } catch (error) {
        devvitGIAELogger.error("[GameState] Failed to load from storage", { postId, error: String(error) });
        return false;
      }
    }
    /**
     * Get current encounter type from mission metadata
     *
     * Note: Returns null for initial battle (currentEncounter === -1)
     * since the initial battle is not in the encounters array
     */
    getCurrentEncounterType() {
      const encounters = this.missionMetadata?.mission?.encounters;
      devvitGIAELogger.log("[GameState] getCurrentEncounterType called:", {
        currentEncounter: this.currentEncounter,
        isInitialBattle: this.currentEncounter === -1,
        hasMetadata: !!this.missionMetadata,
        hasMission: !!this.missionMetadata?.mission,
        hasEncounters: !!encounters,
        encountersLength: encounters?.length || 0,
        encounterAtCurrentIndex: encounters?.[this.currentEncounter],
        encounterAtNextIndex: encounters?.[this.currentEncounter + 1],
        storageLoadAttempted: this._storageLoadAttempted
      });
      if (this.currentEncounter === -1) {
        return null;
      }
      if (!encounters && !this._storageLoadAttempted && this.postId) {
        devvitGIAELogger.warn(
          "[GameState] No encounter metadata! Suggest calling loadMissionDataFromStorage()"
        );
      }
      if (!encounters || this.currentEncounter >= encounters.length) {
        return null;
      }
      return encounters[this.currentEncounter]?.type || null;
    }
    /**
     * Update when encounter completes
     */
    onEncounterComplete(encounterIndex) {
      devvitGIAELogger.log("[GameState] Encounter complete", {
        previousEncounter: this.currentEncounter,
        newEncounter: encounterIndex,
        totalEncounters: this.totalEncounters
      });
      this.currentEncounter = encounterIndex;
    }
    /**
     * Get progress string for display
     */
    getProgress() {
      if (this.totalEncounters === 0) return "Starting";
      if (this.currentEncounter === -1) return "Pre-Game";
      return `${this.currentEncounter + 1}/${this.totalEncounters}`;
    }
    /**
     * Should we play safe? (low on lives)
     */
    shouldPlaySafe() {
      return this.livesRemaining <= 1;
    }
    /**
     * Is player still alive?
     */
    isAlive() {
      return this.livesRemaining > 0;
    }
  }
  class DecisionMaker {
    constructor(gameState, config) {
      this.gameState = gameState;
      this.config = config;
    }
    /**
     * Crossroads: Fight or Skip mini boss
     */
    decideCrossroads() {
      return this.config.crossroadsStrategy || "fight";
    }
    /**
     * Skill Bargain: Accept or Decline
     */
    decideSkillBargain(bargainText) {
      const isPositive = this.isPositiveBargain(bargainText);
      const strategy = this.config.skillBargainStrategy || "positive-only";
      if (strategy === "always") return "accept";
      if (strategy === "never") return "decline";
      return isPositive ? "accept" : "decline";
    }
    /**
     * Pick best ability from choices
     */
    pickAbility(abilities) {
      for (const preferred of this.config.abilityTierList || []) {
        if (abilities.includes(preferred)) {
          return preferred;
        }
      }
      return abilities[0];
    }
    /**
     * Pick best blessing stat from choices
     */
    pickBlessing(blessingStats) {
      for (const preferred of this.config.blessingStatPriority || []) {
        const match = blessingStats.find(
          (stat) => stat.toLowerCase().includes(preferred.toLowerCase())
        );
        if (match) {
          return match;
        }
      }
      return blessingStats[0];
    }
    /**
     * Simple heuristic: more + than - means positive
     */
    isPositiveBargain(text) {
      const plusCount = (text.match(/\+/g) || []).length;
      const minusCount = (text.match(/-/g) || []).length;
      return plusCount > minusCount;
    }
  }
  function normalizePostId(id) {
    if (!id || typeof id !== "string") {
      return null;
    }
    if (id.startsWith("t3_")) {
      const postIdPart = id.slice(3);
      if (postIdPart && /^[a-z0-9]+$/i.test(postIdPart)) {
        return id;
      }
      return null;
    }
    if (/^[a-z0-9]+$/i.test(id)) {
      return `t3_${id}`;
    }
    return null;
  }
  function extractPostIdFromUrl(url) {
    try {
      const contextMatch = url.match(/context=([^&#]+)/);
      if (contextMatch) {
        try {
          const contextJson = decodeURIComponent(contextMatch[1]);
          const context = JSON.parse(contextJson);
          if (context.postId) {
            console.log("[extractPostIdFromUrl] Extracted postId from context parameter", {
              postId: context.postId
            });
            return context.postId;
          }
        } catch (parseError) {
          console.warn("[extractPostIdFromUrl] Failed to parse context parameter", {
            error: String(parseError)
          });
          const postIdMatch = contextMatch[1].match(/%22postId%22%3A%22(t3_[^%]+)%22/);
          if (postIdMatch) {
            console.log("[extractPostIdFromUrl] Extracted postId from context string fallback", {
              postId: postIdMatch[1]
            });
            return postIdMatch[1];
          }
        }
      }
      const tokenMatch = url.match(/webbit_token=([^&#]+)/);
      if (tokenMatch) {
        try {
          const tokenParts = tokenMatch[1].split(".");
          if (tokenParts.length === 3) {
            const payload = JSON.parse(atob(tokenParts[1]));
            if (payload["devvit-post-id"]) {
              console.log("[extractPostIdFromUrl] Extracted postId from JWT token", {
                postId: payload["devvit-post-id"]
              });
              return payload["devvit-post-id"];
            }
          }
        } catch (tokenError) {
          console.warn("[extractPostIdFromUrl] Failed to parse JWT token", {
            error: String(tokenError)
          });
        }
      }
      const redditMatch = url.match(/\/comments\/([a-zA-Z0-9]+)/);
      if (redditMatch && redditMatch[1]) {
        return normalizePostId(redditMatch[1]);
      }
    } catch (error) {
      console.warn("[extractPostIdFromUrl] Failed to extract postId from URL", {
        error: String(error)
      });
    }
    return null;
  }
  const STORAGE_KEYS = {
    MISSIONS: "missions"
  };
  function isLegacyFormat(record) {
    return record && typeof record === "object" && "metadata" in record && record.metadata !== void 0;
  }
  function migrateLegacyRecord(legacy) {
    const mission = legacy.metadata?.mission;
    const record = {
      // Core identification
      postId: legacy.postId,
      timestamp: legacy.timestamp,
      permalink: legacy.permalink,
      // Mission metadata
      missionTitle: legacy.metadata?.missionTitle || legacy.missionTitle || `Mission ${legacy.postId.slice(3)}`,
      missionAuthorName: legacy.metadata?.missionAuthorName || "Unknown",
      // Mission data (from nested mission object or top-level fields)
      environment: mission?.environment || legacy.environment || "haunted_forest",
      encounters: mission?.encounters || [],
      minLevel: mission?.minLevel || legacy.minLevel || 1,
      maxLevel: mission?.maxLevel || legacy.maxLevel || 340,
      difficulty: mission?.difficulty || legacy.difficulty || 0,
      foodImage: mission?.foodImage || "",
      foodName: mission?.foodName || legacy.foodName || "",
      authorWeaponId: mission?.authorWeaponId || "",
      chef: mission?.chef || "",
      cart: mission?.cart || "",
      rarity: mission?.rarity || "common",
      type: mission?.type
    };
    return record;
  }
  function normalizeMissionRecord(record) {
    if (isLegacyFormat(record)) {
      return migrateLegacyRecord(record);
    }
    return record;
  }
  async function saveMission(mission) {
    return new Promise((resolve, reject) => {
      if (!chrome.runtime?.id) {
        reject(new Error("Extension context invalidated"));
        return;
      }
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        const missions2 = result2[STORAGE_KEYS.MISSIONS] || {};
        missions2[mission.postId] = mission;
        chrome.storage.local.set({ [STORAGE_KEYS.MISSIONS]: missions2 }, () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            chrome.runtime.sendMessage({
              type: "MISSIONS_UPDATED"
            }).catch(() => {
            });
            resolve();
          }
        });
      });
    });
  }
  async function getAllMissions() {
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          const rawMissions = result2[STORAGE_KEYS.MISSIONS] || {};
          const migratedMissions = {};
          for (const postId in rawMissions) {
            migratedMissions[postId] = normalizeMissionRecord(rawMissions[postId]);
          }
          resolve(migratedMissions);
        }
      });
    });
  }
  async function getMission(postId) {
    const missions2 = await getAllMissions();
    return missions2[postId] || null;
  }
  const missions = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
    __proto__: null,
    getAllMissions,
    getMission,
    saveMission
  }, Symbol.toStringTag, { value: "Module" }));
  const DEFAULT_GIAE_CONFIG = {
    enabled: false,
    abilityTierList: ["IceKnifeOnTurnStart", "LightningOnCrit", "HealOnFirstTurn"],
    blessingStatPriority: ["Speed", "Attack", "Crit", "Health", "Defense", "Dodge"],
    // Speed first for faster gameplay
    autoAcceptSkillBargains: true,
    skillBargainStrategy: "positive-only",
    crossroadsStrategy: "fight",
    // Fight mini bosses by default
    clickDelay: 1e3,
    debugVisuals: true
    // Show visual indicators for debugging
  };
  class GameInstanceAutomationEngine {
    constructor(config) {
      this.intervalId = null;
      this.isProcessing = false;
      this.ACTIVE_INTERVAL_MS = 1e3;
      this.MONITORING_INTERVAL_MS = 5e3;
      this.currentPostId = null;
      this.missionMetadata = null;
      this.config = {
        ...DEFAULT_GIAE_CONFIG,
        ...config,
        // Ensure arrays are properly set, falling back to defaults if not provided
        abilityTierList: Array.isArray(config.abilityTierList) ? config.abilityTierList : DEFAULT_GIAE_CONFIG.abilityTierList,
        blessingStatPriority: Array.isArray(config.blessingStatPriority) ? config.blessingStatPriority : DEFAULT_GIAE_CONFIG.blessingStatPriority
      };
      this.gameState = new GameState();
      this.decisionMaker = new DecisionMaker(this.gameState, this.config);
      this.setupMessageListener();
      devvitGIAELogger.log("Game automation engine initialized");
      this.startInterval(this.MONITORING_INTERVAL_MS);
      devvitGIAELogger.log("Screen detection interval started (monitoring mode)");
    }
    setupMessageListener() {
      window.addEventListener("message", (event) => {
        try {
          if (event.data?.type === "devvit-message" && event.data?.data?.message?.type === "initialData") {
            const data = event.data.data.message.data;
            this.gameState.setMissionData(data.missionMetadata, data.postId);
            this.currentPostId = data.postId;
            this.missionMetadata = data.missionMetadata;
            this.gameState.loadMissionDataFromStorage(data.postId).catch((err) => {
              devvitGIAELogger.error("[GIAE] Failed to load mission data from storage", err);
            });
            devvitGIAELogger.log("Mission started", {
              postId: data.postId,
              encounters: this.gameState.totalEncounters,
              difficulty: this.gameState.difficulty
            });
            this.reportGameState();
          }
          if (event.data?.data?.message?.type === "missionComplete") {
            const postId = event.data.data.message.data?.postId;
            if (postId) {
              devvitGIAELogger.log("Mission complete", { postId });
              chrome.runtime.sendMessage({
                type: "MISSION_COMPLETED",
                postId
              });
            }
          }
        } catch (error) {
          devvitGIAELogger.error("Message error", { error: String(error) });
        }
      });
    }
    /**
     * Helper to start/restart the detection interval with a specific timing
     */
    startInterval(intervalMs) {
      if (this.intervalId) {
        clearInterval(this.intervalId);
      }
      this.intervalId = window.setInterval(() => {
        if (!this.isProcessing) {
          this.processGame();
        }
      }, intervalMs);
    }
    start() {
      this.config.enabled = true;
      devvitGIAELogger.log("Starting automation (switching to active mode)");
      this.startInterval(this.ACTIVE_INTERVAL_MS);
    }
    stop() {
      devvitGIAELogger.log("Stopping automation (switching to monitoring mode)");
      this.config.enabled = false;
      this.startInterval(this.MONITORING_INTERVAL_MS);
    }
    /**
     * Completely stops detection interval (use when leaving game)
     */
    stopDetection() {
      devvitGIAELogger.log("Stopping detection interval");
      this.config.enabled = false;
      if (this.intervalId) {
        clearInterval(this.intervalId);
        this.intervalId = null;
      }
    }
    async processGame() {
      this.isProcessing = true;
      try {
        let postId = this.currentPostId;
        if (!postId) {
          postId = extractPostIdFromUrl(window.location.href);
          if (postId) {
            devvitGIAELogger.log("Extracted postId from URL", { postId });
            this.currentPostId = postId;
            this.gameState.postId = postId;
          }
        }
        if (postId && !this.gameState.missionMetadata) {
          if (!this.gameState._storageLoadAttempted) {
            devvitGIAELogger.log("Loading mission metadata from storage (initialData not yet received)");
            await this.gameState.loadMissionDataFromStorage(postId);
          }
        }
        this.gameState.updateFromDOM();
        if (!this.gameState.isAlive()) {
          devvitGIAELogger.error("Out of lives");
          this.stop();
          chrome.runtime.sendMessage({
            type: "ERROR_OCCURRED",
            message: "Out of lives"
          });
          return;
        }
        const buttons = this.findAllButtons();
        if (buttons.length === 0) return;
        const screen = this.detectScreen(buttons);
        devvitGIAELogger.log("[processGame] Screen detected", { screen });
        const screenChanged = this.gameState.currentScreen !== screen;
        this.gameState.currentScreen = screen;
        if (screenChanged) {
          this.reportGameState();
        }
        if (screen !== "unknown" && screen !== "in_progress") {
          const encounterType = this.gameState.getCurrentEncounterType();
          const dryRun = !this.config.enabled;
          devvitGIAELogger.log("Screen", {
            screen,
            lives: this.gameState.livesRemaining,
            playSafe: this.gameState.shouldPlaySafe(),
            encounter: this.gameState.getProgress(),
            encounterType,
            // For debugging - compare with detected screen
            mode: dryRun ? "DRY-RUN" : "ACTIVE"
          });
          await this.handleScreen(screen, buttons, dryRun);
          if (!dryRun) {
            await this.delay(this.config.clickDelay || 300);
          }
        }
      } catch (error) {
        devvitGIAELogger.error("Process error", { error: String(error) });
      } finally {
        this.isProcessing = false;
      }
    }
    detectScreen(buttons) {
      const texts = buttons.map((b) => b.textContent?.trim().toLowerCase() || "");
      const classes = buttons.map((b) => b.className);
      const tooltip = document.querySelector(".navbar-tooltip");
      if (tooltip?.textContent?.includes("Find and play missions")) {
        return "inn";
      }
      if (classes.some((c) => c.includes("skip-button"))) return "skip";
      if (document.querySelector(".mission-end-footer")) return "finish";
      if (texts.some((t) => t.includes("fight")) && texts.some((t) => t.includes("nope"))) {
        return "crossroads";
      }
      if (texts.includes("refuse") || texts.includes("accept") && texts.includes("decline")) {
        return "bargain";
      }
      if (classes.filter((c) => c.includes("skill-button")).length > 1) return "choice";
      if (classes.some((c) => c.includes("advance-button"))) {
        return "battle";
      }
      if (texts.includes("continue")) return "continue";
      if (classes.some((c) => c.includes("volume-icon-button"))) {
        return "in_progress";
      }
      return "unknown";
    }
    /**
     * Helper to click a button or log dry-run action
     */
    clickButton(button, description, dryRun) {
      if (this.config.debugVisuals) {
        button.style.outline = "3px solid #ff000052";
      }
      if (dryRun) {
        devvitGIAELogger.log(`[DRY-RUN] Would click: ${description}`, {
          buttonText: button.textContent?.trim(),
          buttonClass: button.className
        });
      } else {
        button.click();
      }
    }
    async handleScreen(screen, buttons, dryRun = false) {
      let actionTaken = false;
      switch (screen) {
        case "skip": {
          const skipBtn = buttons.find((b) => b.classList.contains("skip-button"));
          if (skipBtn) {
            this.clickButton(skipBtn, "skip button", dryRun);
            actionTaken = true;
          }
          break;
        }
        case "battle": {
          const advanceBtn = buttons.find((b) => b.classList.contains("advance-button"));
          if (advanceBtn) {
            this.clickButton(advanceBtn, "advance button", dryRun);
            actionTaken = true;
          }
          break;
        }
        case "crossroads": {
          const choice = this.decisionMaker.decideCrossroads();
          devvitGIAELogger.log("Crossroads decision", { choice, dryRun });
          if (choice === "fight") {
            const fightBtn = buttons.find((b) => b.textContent?.toLowerCase().includes("fight"));
            if (fightBtn) {
              this.clickButton(fightBtn, "fight button (crossroads)", dryRun);
              actionTaken = true;
            }
          } else {
            const skipBtn = buttons.find((b) => b.textContent?.toLowerCase().includes("nope"));
            if (skipBtn) {
              this.clickButton(skipBtn, "nope button (crossroads)", dryRun);
              actionTaken = true;
            }
          }
          break;
        }
        case "bargain": {
          const bargainText = document.body.textContent || "";
          const choice = this.decisionMaker.decideSkillBargain(bargainText);
          devvitGIAELogger.log("Bargain decision", {
            choice,
            bargainText: bargainText.substring(0, 200),
            dryRun
          });
          const skillButtons = buttons.filter((b) => b.classList.contains("skill-button"));
          devvitGIAELogger.log("Bargain buttons found", {
            count: skillButtons.length,
            buttons: skillButtons.map((b) => b.textContent?.trim())
          });
          if (choice === "accept") {
            const acceptBtn = skillButtons.find((b) => {
              const text = b.textContent?.trim().toLowerCase() || "";
              return text !== "refuse" && text !== "decline";
            });
            if (acceptBtn) {
              if (!dryRun) {
                devvitGIAELogger.log("Clicking accept button", { text: acceptBtn.textContent?.trim() });
              }
              this.clickButton(acceptBtn, "accept button (bargain)", dryRun);
              actionTaken = true;
            }
          } else {
            const declineBtn = skillButtons.find((b) => {
              const text = b.textContent?.trim().toLowerCase() || "";
              return text === "refuse" || text === "decline";
            });
            if (declineBtn) {
              if (!dryRun)
                devvitGIAELogger.log("Clicking decline button", { text: declineBtn.textContent?.trim() });
              this.clickButton(declineBtn, "decline button (bargain)", dryRun);
              actionTaken = true;
            } else if (skillButtons.length > 0) {
              const fallbackBtn = skillButtons[skillButtons.length - 1];
              if (!dryRun) {
                devvitGIAELogger.log("No decline button, clicking last skill button as fallback", {
                  text: fallbackBtn.textContent?.trim()
                });
              }
              this.clickButton(fallbackBtn, "last skill button (bargain fallback)", dryRun);
              actionTaken = true;
            }
          }
          break;
        }
        case "choice": {
          const skillButtons = buttons.filter((b) => b.classList.contains("skill-button"));
          const panelHeader = document.querySelector(".ui-panel-header");
          const headerText = panelHeader?.textContent?.toLowerCase() || "";
          const blessingStats = skillButtons.map((b) => {
            const text = b.textContent?.trim() || "";
            const match = text.match(/Increase (\w+) by \d+%/);
            return match ? match[1] : null;
          }).filter((stat) => !!stat);
          let buttonClicked = false;
          if (blessingStats.length > 0 || headerText.includes("blessing") || headerText.includes("boon")) {
            devvitGIAELogger.log("Blessing detected (DOM)", { headerText, blessingStats, dryRun });
            if (blessingStats.length > 0) {
              this.recordDiscoveredBlessingStats(blessingStats);
              const chosen = this.decisionMaker.pickBlessing(blessingStats);
              devvitGIAELogger.log("Blessing choice", { chosen, available: blessingStats, dryRun });
              const btn = skillButtons.find(
                (b) => b.textContent?.toLowerCase().includes(chosen.toLowerCase())
              );
              if (btn) {
                this.clickButton(btn, `blessing: ${chosen}`, dryRun);
                buttonClicked = true;
              }
            }
          } else {
            devvitGIAELogger.log("Ability choice detected (DOM)", { headerText, dryRun });
            const abilities = skillButtons.map((b) => b.textContent?.trim() || "");
            if (abilities.length > 0) {
              this.recordDiscoveredAbilities(abilities);
              const chosen = this.decisionMaker.pickAbility(abilities);
              devvitGIAELogger.log("Ability choice", { chosen, available: abilities, dryRun });
              const btn = buttons.find((b) => b.textContent?.includes(chosen));
              if (btn) {
                this.clickButton(btn, `ability: ${chosen}`, dryRun);
                buttonClicked = true;
              }
            }
          }
          if (!buttonClicked && skillButtons.length > 0) {
            if (!dryRun) {
              devvitGIAELogger.log("No specific choice made, picking first button as fallback", {
                firstButtonText: skillButtons[0].textContent?.trim()
              });
            }
            this.clickButton(skillButtons[0], "first skill button (choice fallback)", dryRun);
            buttonClicked = true;
          }
          actionTaken = buttonClicked;
          break;
        }
        case "continue":
        case "finish": {
          const btn = buttons.find(
            (b) => b.textContent?.toLowerCase() === "continue" || b.classList.contains("end-mission-button")
          );
          if (btn) {
            this.clickButton(btn, "continue/finish button", dryRun);
            actionTaken = true;
          }
          break;
        }
        case "inn":
          devvitGIAELogger.log("Inn detected (mission complete)", {
            postId: this.gameState.postId,
            dryRun
          });
          actionTaken = true;
          break;
      }
      if (!actionTaken && buttons.length > 0) {
        if (!dryRun) {
          devvitGIAELogger.warn("No action taken for screen, clicking first available button as fallback", {
            screen,
            buttonCount: buttons.length,
            firstButton: {
              text: buttons[0].textContent?.trim(),
              classes: buttons[0].className
            }
          });
        }
        this.clickButton(buttons[0], "first available button (global fallback)", dryRun);
      }
    }
    findAllButtons() {
      const selectors = [".advance-button", ".skill-button", ".skip-button", "button"];
      const buttons = [];
      for (const selector of selectors) {
        document.querySelectorAll(selector).forEach((el) => {
          const rect = el.getBoundingClientRect();
          if (rect.width > 0 && rect.height > 0) {
            buttons.push(el);
          }
        });
      }
      return buttons;
    }
    delay(ms) {
      return new Promise((resolve) => setTimeout(resolve, ms));
    }
    // Public API
    updateConfig(config) {
      this.config = { ...this.config, ...config };
    }
    getState() {
      return this.config.enabled ? "running" : "stopped";
    }
    getGameState() {
      return {
        enabled: this.config.enabled,
        screen: this.gameState.currentScreen,
        lives: this.gameState.livesRemaining,
        progress: this.gameState.getProgress(),
        postId: this.gameState.postId,
        encounterCurrent: this.gameState.currentEncounter,
        encounterTotal: this.gameState.totalEncounters,
        playSafe: this.gameState.shouldPlaySafe(),
        difficulty: this.gameState.difficulty
      };
    }
    // Report game state to background service worker
    reportGameState() {
      try {
        chrome.runtime.sendMessage({
          type: "GAME_STATE_UPDATE",
          gameState: this.getGameState()
        });
      } catch (error) {
        devvitGIAELogger.error("Failed to report game state", { error: String(error) });
      }
    }
    // Save mission to database
    async saveMissionToDatabase(postId, username, metadata) {
      try {
        const mission = metadata.mission;
        if (!mission) return;
        const existingMission = await getMission(postId);
        const permalink = postId.startsWith("t3_") ? `https://www.reddit.com/r/SwordAndSupperGame/comments/${postId.slice(3)}/` : "";
        const record = {
          postId,
          timestamp: existingMission?.timestamp || Date.now(),
          permalink,
          missionTitle: metadata.missionTitle || mission.foodName || "Unknown",
          missionAuthorName: metadata.missionAuthorName || "Unknown",
          environment: mission.environment,
          encounters: mission.encounters || [],
          minLevel: mission.minLevel,
          maxLevel: mission.maxLevel,
          difficulty: mission.difficulty,
          foodImage: mission.foodImage,
          foodName: mission.foodName,
          authorWeaponId: mission.authorWeaponId,
          chef: mission.chef,
          cart: mission.cart,
          rarity: mission.rarity,
          type: mission.type
        };
        await saveMission(record);
        if (existingMission) {
          devvitGIAELogger.log("Mission data enriched", {
            postId,
            difficulty: record.difficulty,
            environment: record.environment
          });
        } else {
          devvitGIAELogger.log("🆕 NEW MISSION discovered", {
            postId,
            difficulty: record.difficulty,
            foodName: mission.foodName
          });
        }
      } catch (error) {
        devvitGIAELogger.error("Failed to save mission", { error: String(error) });
      }
    }
    /**
     * Record discovered abilities to storage for user reference
     */
    recordDiscoveredAbilities(abilityNames) {
      try {
        chrome.storage.local.get(["discoveredAbilities"], (result2) => {
          const existing = new Set(result2.discoveredAbilities || []);
          let added = false;
          for (const name of abilityNames) {
            if (!existing.has(name)) {
              existing.add(name);
              added = true;
            }
          }
          if (added) {
            chrome.storage.local.set({ discoveredAbilities: Array.from(existing) });
            devvitGIAELogger.log("Discovered new abilities", {
              newAbilities: abilityNames.filter((n) => !result2.discoveredAbilities?.includes(n)),
              total: existing.size
            });
          }
        });
      } catch (error) {
        devvitGIAELogger.error("Failed to record discovered abilities", { error: String(error) });
      }
    }
    /**
     * Record discovered blessing stats to storage for user reference
     */
    recordDiscoveredBlessingStats(statNames) {
      try {
        chrome.storage.local.get(["discoveredBlessingStats"], (result2) => {
          const existing = new Set(result2.discoveredBlessingStats || []);
          let added = false;
          for (const name of statNames) {
            if (!existing.has(name)) {
              existing.add(name);
              added = true;
            }
          }
          if (added) {
            chrome.storage.local.set({ discoveredBlessingStats: Array.from(existing) });
            devvitGIAELogger.log("Discovered new blessing stats", {
              newStats: statNames.filter((n) => !result2.discoveredBlessingStats?.includes(n)),
              total: existing.size
            });
          }
        });
      } catch (error) {
        devvitGIAELogger.error("Failed to record discovered blessing stats", { error: String(error) });
      }
    }
  }
  devvitLogger.log("Devvit content script loaded", {
    version: "0.16.0",
    buildTime: "2025-11-05T15:49:06.935Z",
    url: window.location.href,
    loadTime: (/* @__PURE__ */ new Date()).toISOString()
  });
  let gameAutomation = null;
  window.addEventListener("message", (event) => {
    devvitLogger.log("📨 message event", event);
    try {
      if (event.data?.type === "devvit-message") {
        const messageType = event.data?.data?.message?.type;
        devvitLogger.log("📨 devvit-message received", {
          messageType,
          origin: event.origin,
          data: event.data?.data?.message?.data || null,
          timestamp: (/* @__PURE__ */ new Date()).toISOString()
        });
        if (messageType === "initialData") {
          const postId = event.data?.data?.message?.data?.postId;
          devvitLogger.log(`✅ initialData for ${postId} captured!`, event.data?.data);
          window.__capturedInitialData = event.data.data.message.data;
        }
      }
    } catch (error) {
      devvitLogger.error("Error in early message listener", { error: String(error) });
    }
  });
  devvitLogger.log("Early message listener installed");
  function safeSendMessage(message, callback) {
    try {
      chrome.runtime.sendMessage(message, (response) => {
        if (chrome.runtime.lastError) {
          const errorMsg = String(chrome.runtime.lastError.message || chrome.runtime.lastError);
          if (errorMsg.includes("Extension context invalidated")) {
            devvitLogger.log("[ExtensionContext] Extension was updated/reloaded", {
              error: errorMsg
            });
          } else {
            devvitLogger.error("[ExtensionContext] Runtime error", { error: errorMsg });
          }
          return;
        }
        if (callback) ;
      });
    } catch (error) {
      const errorMsg = String(error);
      if (errorMsg.includes("Extension context invalidated")) {
        devvitLogger.log("[ExtensionContext] Extension was updated/reloaded", { error: errorMsg });
      } else {
        devvitLogger.error("[ExtensionContext] Runtime error", { error: errorMsg });
      }
    }
  }
  function initializeAutomation() {
    if (gameAutomation) {
      devvitLogger.warn("Automation already initialized");
      return;
    }
    devvitLogger.log("Initializing game instance automation engine");
    chrome.storage.local.get(["automationConfig"], async (result2) => {
      const config = result2.automationConfig || {};
      const giaeConfig = {
        enabled: false,
        // Will be enabled when user clicks button
        abilityTierList: config.abilityTierList || DEFAULT_GIAE_CONFIG.abilityTierList,
        blessingStatPriority: config.blessingStatPriority || DEFAULT_GIAE_CONFIG.blessingStatPriority,
        autoAcceptSkillBargains: config.autoAcceptSkillBargains !== void 0 ? config.autoAcceptSkillBargains : DEFAULT_GIAE_CONFIG.autoAcceptSkillBargains,
        skillBargainStrategy: config.skillBargainStrategy || DEFAULT_GIAE_CONFIG.skillBargainStrategy,
        crossroadsStrategy: config.crossroadsStrategy || DEFAULT_GIAE_CONFIG.crossroadsStrategy,
        clickDelay: 300,
        debugVisuals: config.debugVisuals !== void 0 ? config.debugVisuals : DEFAULT_GIAE_CONFIG.debugVisuals
      };
      gameAutomation = new GameInstanceAutomationEngine(giaeConfig);
      devvitLogger.log("Game instance automation engine initialized", { config: giaeConfig });
      const capturedData = window.__capturedInitialData;
      if (capturedData) {
        devvitLogger.log("Processing previously captured initialData", {
          postId: capturedData.postId,
          username: capturedData.username
        });
        const missionMetadata = capturedData.missionMetadata;
        const postId = capturedData.postId;
        const username = capturedData.username;
        if (missionMetadata && postId && gameAutomation) {
          await gameAutomation.saveMissionToDatabase(postId, username, missionMetadata);
          gameAutomation.currentPostId = postId;
          gameAutomation.missionMetadata = missionMetadata;
          gameAutomation.gameState.setMissionData(missionMetadata, postId);
          gameAutomation.gameState.loadMissionDataFromStorage(postId).catch((err) => {
            devvitLogger.error("Failed to load mission data from storage", err);
          });
          devvitLogger.debug("Set mission data from captured initialData", {
            postId,
            encounters: gameAutomation.gameState.totalEncounters
          });
        }
        delete window.__capturedInitialData;
      }
      safeSendMessage({
        type: "AUTOMATION_READY",
        config: giaeConfig
      });
      if (pendingStartMessage) {
        devvitLogger.log("Processing queued START_MISSION_AUTOMATION message");
        chrome.storage.local.get(["automationConfig"], (result22) => {
          if (result22.automationConfig) {
            updateAutomationConfig(result22.automationConfig);
          }
          toggleAutomation(true);
        });
        pendingStartMessage = null;
      }
    });
  }
  function toggleAutomation(enabled) {
    devvitLogger.log("toggleAutomation called", { enabled });
    if (!gameAutomation) {
      devvitLogger.error("Automation not initialized");
      return;
    }
    if (enabled) {
      gameAutomation.start();
      devvitLogger.log("Automation started");
    } else {
      gameAutomation.stop();
      devvitLogger.log("Automation stopped");
    }
    devvitLogger.log("Automation state", { state: gameAutomation.getState() });
  }
  function updateAutomationConfig(config) {
    if (!gameAutomation) {
      devvitLogger.error("Automation not initialized");
      return;
    }
    const giaeConfig = {
      abilityTierList: config.abilityTierList,
      blessingStatPriority: config.blessingStatPriority,
      autoAcceptSkillBargains: config.autoAcceptSkillBargains,
      skillBargainStrategy: config.skillBargainStrategy,
      crossroadsStrategy: config.crossroadsStrategy,
      debugVisuals: config.debugVisuals
    };
    gameAutomation.updateConfig(giaeConfig);
    chrome.storage.local.set({ automationConfig: config });
  }
  let pendingStartMessage = null;
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    devvitLogger.log(`Received ${message.type} message`, {
      message
    });
    switch (message.type) {
      case "CHECK_AUTOMATION_STATUS":
        const isReady = gameAutomation !== null;
        const automationState = gameAutomation ? gameAutomation.getState() : null;
        sendResponse({
          isReady,
          isRunning: automationState === "running",
          state: automationState
        });
        break;
      case "START_MISSION_AUTOMATION":
        if (!gameAutomation) {
          devvitLogger.log("Automation not ready, queuing START_MISSION_AUTOMATION");
          pendingStartMessage = message;
          sendResponse({ success: true, queued: true });
        } else {
          chrome.storage.local.get(["automationConfig"], (result2) => {
            if (result2.automationConfig) {
              updateAutomationConfig(result2.automationConfig);
            }
            toggleAutomation(true);
            sendResponse({ success: true });
          });
          return true;
        }
        break;
      case "STOP_MISSION_AUTOMATION":
        toggleAutomation(false);
        sendResponse({ success: true });
        break;
      case "STATE_CHANGED":
        sendResponse({ success: true });
        break;
      case "GET_GAME_STATE":
        if (gameAutomation) {
          const state = gameAutomation.getGameState();
          sendResponse({ gameState: state });
        } else {
          sendResponse({ gameState: null });
        }
        break;
      default:
        devvitLogger.warn(`Unknown message type: ${message.type}`, { message });
        sendResponse({ error: "Unknown message type: " + message.type });
    }
    return true;
  });
  setTimeout(() => {
    devvitLogger.log("Running initial game analysis");
    initializeAutomation();
  }, 2e3);
  const devvit2 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
    __proto__: null
  }, Symbol.toStringTag, { value: "Module" }));
  return result;
})();
devvit;