var background = (function() {
  "use strict";
  function defineBackground(arg) {
    if (arg == null || typeof arg === "function") return { main: arg };
    return arg;
  }
  function getGlobal() {
    if (typeof globalThis !== "undefined") {
      return globalThis;
    }
    if (typeof self !== "undefined") {
      return self;
    }
    if (typeof window !== "undefined") {
      return window;
    }
    if (typeof global !== "undefined") {
      return global;
    }
  }
  function getDevTools() {
    const w = getGlobal();
    if (w.__xstate__) {
      return w.__xstate__;
    }
    return void 0;
  }
  const devToolsAdapter = (service) => {
    if (typeof window === "undefined") {
      return;
    }
    const devTools = getDevTools();
    if (devTools) {
      devTools.register(service);
    }
  };
  class Mailbox {
    constructor(_process) {
      this._process = _process;
      this._active = false;
      this._current = null;
      this._last = null;
    }
    start() {
      this._active = true;
      this.flush();
    }
    clear() {
      if (this._current) {
        this._current.next = null;
        this._last = this._current;
      }
    }
    enqueue(event) {
      const enqueued = {
        value: event,
        next: null
      };
      if (this._current) {
        this._last.next = enqueued;
        this._last = enqueued;
        return;
      }
      this._current = enqueued;
      this._last = enqueued;
      if (this._active) {
        this.flush();
      }
    }
    flush() {
      while (this._current) {
        const consumed = this._current;
        this._process(consumed.value);
        this._current = consumed.next;
      }
      this._last = null;
    }
  }
  const STATE_DELIMITER = ".";
  const TARGETLESS_KEY = "";
  const NULL_EVENT = "";
  const STATE_IDENTIFIER$1 = "#";
  const WILDCARD = "*";
  const XSTATE_INIT = "xstate.init";
  const XSTATE_ERROR = "xstate.error";
  const XSTATE_STOP = "xstate.stop";
  function createAfterEvent(delayRef, id) {
    return {
      type: `xstate.after.${delayRef}.${id}`
    };
  }
  function createDoneStateEvent(id, output) {
    return {
      type: `xstate.done.state.${id}`,
      output
    };
  }
  function createDoneActorEvent(invokeId, output) {
    return {
      type: `xstate.done.actor.${invokeId}`,
      output,
      actorId: invokeId
    };
  }
  function createErrorActorEvent(id, error) {
    return {
      type: `xstate.error.actor.${id}`,
      error,
      actorId: id
    };
  }
  function createInitEvent(input) {
    return {
      type: XSTATE_INIT,
      input
    };
  }
  function reportUnhandledError(err) {
    setTimeout(() => {
      throw err;
    });
  }
  const symbolObservable = (() => typeof Symbol === "function" && Symbol.observable || "@@observable")();
  function matchesState(parentStateId, childStateId) {
    const parentStateValue = toStateValue(parentStateId);
    const childStateValue = toStateValue(childStateId);
    if (typeof childStateValue === "string") {
      if (typeof parentStateValue === "string") {
        return childStateValue === parentStateValue;
      }
      return false;
    }
    if (typeof parentStateValue === "string") {
      return parentStateValue in childStateValue;
    }
    return Object.keys(parentStateValue).every((key) => {
      if (!(key in childStateValue)) {
        return false;
      }
      return matchesState(parentStateValue[key], childStateValue[key]);
    });
  }
  function toStatePath(stateId) {
    if (isArray(stateId)) {
      return stateId;
    }
    const result2 = [];
    let segment = "";
    for (let i = 0; i < stateId.length; i++) {
      const char = stateId.charCodeAt(i);
      switch (char) {
        // \
        case 92:
          segment += stateId[i + 1];
          i++;
          continue;
        // .
        case 46:
          result2.push(segment);
          segment = "";
          continue;
      }
      segment += stateId[i];
    }
    result2.push(segment);
    return result2;
  }
  function toStateValue(stateValue) {
    if (isMachineSnapshot(stateValue)) {
      return stateValue.value;
    }
    if (typeof stateValue !== "string") {
      return stateValue;
    }
    const statePath = toStatePath(stateValue);
    return pathToStateValue(statePath);
  }
  function pathToStateValue(statePath) {
    if (statePath.length === 1) {
      return statePath[0];
    }
    const value = {};
    let marker = value;
    for (let i = 0; i < statePath.length - 1; i++) {
      if (i === statePath.length - 2) {
        marker[statePath[i]] = statePath[i + 1];
      } else {
        const previous = marker;
        marker = {};
        previous[statePath[i]] = marker;
      }
    }
    return value;
  }
  function mapValues(collection, iteratee) {
    const result2 = {};
    const collectionKeys = Object.keys(collection);
    for (let i = 0; i < collectionKeys.length; i++) {
      const key = collectionKeys[i];
      result2[key] = iteratee(collection[key], key, collection, i);
    }
    return result2;
  }
  function toArrayStrict(value) {
    if (isArray(value)) {
      return value;
    }
    return [value];
  }
  function toArray(value) {
    if (value === void 0) {
      return [];
    }
    return toArrayStrict(value);
  }
  function resolveOutput(mapper, context, event, self2) {
    if (typeof mapper === "function") {
      return mapper({
        context,
        event,
        self: self2
      });
    }
    return mapper;
  }
  function isArray(value) {
    return Array.isArray(value);
  }
  function isErrorActorEvent(event) {
    return event.type.startsWith("xstate.error.actor");
  }
  function toTransitionConfigArray(configLike) {
    return toArrayStrict(configLike).map((transitionLike) => {
      if (typeof transitionLike === "undefined" || typeof transitionLike === "string") {
        return {
          target: transitionLike
        };
      }
      return transitionLike;
    });
  }
  function normalizeTarget(target) {
    if (target === void 0 || target === TARGETLESS_KEY) {
      return void 0;
    }
    return toArray(target);
  }
  function toObserver(nextHandler, errorHandler, completionHandler) {
    const isObserver = typeof nextHandler === "object";
    const self2 = isObserver ? nextHandler : void 0;
    return {
      next: (isObserver ? nextHandler.next : nextHandler)?.bind(self2),
      error: (isObserver ? nextHandler.error : errorHandler)?.bind(self2),
      complete: (isObserver ? nextHandler.complete : completionHandler)?.bind(self2)
    };
  }
  function createInvokeId(stateNodeId, index) {
    return `${index}.${stateNodeId}`;
  }
  function resolveReferencedActor(machine, src) {
    const match = src.match(/^xstate\.invoke\.(\d+)\.(.*)/);
    if (!match) {
      return machine.implementations.actors[src];
    }
    const [, indexStr, nodeId] = match;
    const node = machine.getStateNodeById(nodeId);
    const invokeConfig = node.config.invoke;
    return (Array.isArray(invokeConfig) ? invokeConfig[indexStr] : invokeConfig).src;
  }
  function createScheduledEventId(actorRef, id) {
    return `${actorRef.sessionId}.${id}`;
  }
  let idCounter = 0;
  function createSystem(rootActor, options) {
    const children = /* @__PURE__ */ new Map();
    const keyedActors = /* @__PURE__ */ new Map();
    const reverseKeyedActors = /* @__PURE__ */ new WeakMap();
    const inspectionObservers = /* @__PURE__ */ new Set();
    const timerMap = {};
    const {
      clock,
      logger: logger2
    } = options;
    const scheduler = {
      schedule: (source, target, event, delay, id = Math.random().toString(36).slice(2)) => {
        const scheduledEvent = {
          source,
          target,
          event,
          delay,
          id,
          startedAt: Date.now()
        };
        const scheduledEventId = createScheduledEventId(source, id);
        system._snapshot._scheduledEvents[scheduledEventId] = scheduledEvent;
        const timeout = clock.setTimeout(() => {
          delete timerMap[scheduledEventId];
          delete system._snapshot._scheduledEvents[scheduledEventId];
          system._relay(source, target, event);
        }, delay);
        timerMap[scheduledEventId] = timeout;
      },
      cancel: (source, id) => {
        const scheduledEventId = createScheduledEventId(source, id);
        const timeout = timerMap[scheduledEventId];
        delete timerMap[scheduledEventId];
        delete system._snapshot._scheduledEvents[scheduledEventId];
        if (timeout !== void 0) {
          clock.clearTimeout(timeout);
        }
      },
      cancelAll: (actorRef) => {
        for (const scheduledEventId in system._snapshot._scheduledEvents) {
          const scheduledEvent = system._snapshot._scheduledEvents[scheduledEventId];
          if (scheduledEvent.source === actorRef) {
            scheduler.cancel(actorRef, scheduledEvent.id);
          }
        }
      }
    };
    const sendInspectionEvent = (event) => {
      if (!inspectionObservers.size) {
        return;
      }
      const resolvedInspectionEvent = {
        ...event,
        rootId: rootActor.sessionId
      };
      inspectionObservers.forEach((observer) => observer.next?.(resolvedInspectionEvent));
    };
    const system = {
      _snapshot: {
        _scheduledEvents: (options?.snapshot && options.snapshot.scheduler) ?? {}
      },
      _bookId: () => `x:${idCounter++}`,
      _register: (sessionId, actorRef) => {
        children.set(sessionId, actorRef);
        return sessionId;
      },
      _unregister: (actorRef) => {
        children.delete(actorRef.sessionId);
        const systemId = reverseKeyedActors.get(actorRef);
        if (systemId !== void 0) {
          keyedActors.delete(systemId);
          reverseKeyedActors.delete(actorRef);
        }
      },
      get: (systemId) => {
        return keyedActors.get(systemId);
      },
      getAll: () => {
        return Object.fromEntries(keyedActors.entries());
      },
      _set: (systemId, actorRef) => {
        const existing = keyedActors.get(systemId);
        if (existing && existing !== actorRef) {
          throw new Error(`Actor with system ID '${systemId}' already exists.`);
        }
        keyedActors.set(systemId, actorRef);
        reverseKeyedActors.set(actorRef, systemId);
      },
      inspect: (observerOrFn) => {
        const observer = toObserver(observerOrFn);
        inspectionObservers.add(observer);
        return {
          unsubscribe() {
            inspectionObservers.delete(observer);
          }
        };
      },
      _sendInspectionEvent: sendInspectionEvent,
      _relay: (source, target, event) => {
        system._sendInspectionEvent({
          type: "@xstate.event",
          sourceRef: source,
          actorRef: target,
          event
        });
        target._send(event);
      },
      scheduler,
      getSnapshot: () => {
        return {
          _scheduledEvents: {
            ...system._snapshot._scheduledEvents
          }
        };
      },
      start: () => {
        const scheduledEvents = system._snapshot._scheduledEvents;
        system._snapshot._scheduledEvents = {};
        for (const scheduledId in scheduledEvents) {
          const {
            source,
            target,
            event,
            delay,
            id
          } = scheduledEvents[scheduledId];
          scheduler.schedule(source, target, event, delay, id);
        }
      },
      _clock: clock,
      _logger: logger2
    };
    return system;
  }
  let executingCustomAction = false;
  const $$ACTOR_TYPE = 1;
  let ProcessingStatus = /* @__PURE__ */ (function(ProcessingStatus2) {
    ProcessingStatus2[ProcessingStatus2["NotStarted"] = 0] = "NotStarted";
    ProcessingStatus2[ProcessingStatus2["Running"] = 1] = "Running";
    ProcessingStatus2[ProcessingStatus2["Stopped"] = 2] = "Stopped";
    return ProcessingStatus2;
  })({});
  const defaultOptions = {
    clock: {
      setTimeout: (fn, ms) => {
        return setTimeout(fn, ms);
      },
      clearTimeout: (id) => {
        return clearTimeout(id);
      }
    },
    logger: console.log.bind(console),
    devTools: false
  };
  class Actor {
    /**
     * Creates a new actor instance for the given logic with the provided options,
     * if any.
     *
     * @param logic The logic to create an actor from
     * @param options Actor options
     */
    constructor(logic, options) {
      this.logic = logic;
      this._snapshot = void 0;
      this.clock = void 0;
      this.options = void 0;
      this.id = void 0;
      this.mailbox = new Mailbox(this._process.bind(this));
      this.observers = /* @__PURE__ */ new Set();
      this.eventListeners = /* @__PURE__ */ new Map();
      this.logger = void 0;
      this._processingStatus = ProcessingStatus.NotStarted;
      this._parent = void 0;
      this._syncSnapshot = void 0;
      this.ref = void 0;
      this._actorScope = void 0;
      this.systemId = void 0;
      this.sessionId = void 0;
      this.system = void 0;
      this._doneEvent = void 0;
      this.src = void 0;
      this._deferred = [];
      const resolvedOptions = {
        ...defaultOptions,
        ...options
      };
      const {
        clock,
        logger: logger2,
        parent,
        syncSnapshot,
        id,
        systemId,
        inspect
      } = resolvedOptions;
      this.system = parent ? parent.system : createSystem(this, {
        clock,
        logger: logger2
      });
      if (inspect && !parent) {
        this.system.inspect(toObserver(inspect));
      }
      this.sessionId = this.system._bookId();
      this.id = id ?? this.sessionId;
      this.logger = options?.logger ?? this.system._logger;
      this.clock = options?.clock ?? this.system._clock;
      this._parent = parent;
      this._syncSnapshot = syncSnapshot;
      this.options = resolvedOptions;
      this.src = resolvedOptions.src ?? logic;
      this.ref = this;
      this._actorScope = {
        self: this,
        id: this.id,
        sessionId: this.sessionId,
        logger: this.logger,
        defer: (fn) => {
          this._deferred.push(fn);
        },
        system: this.system,
        stopChild: (child) => {
          if (child._parent !== this) {
            throw new Error(`Cannot stop child actor ${child.id} of ${this.id} because it is not a child`);
          }
          child._stop();
        },
        emit: (emittedEvent) => {
          const listeners = this.eventListeners.get(emittedEvent.type);
          const wildcardListener = this.eventListeners.get("*");
          if (!listeners && !wildcardListener) {
            return;
          }
          const allListeners = [...listeners ? listeners.values() : [], ...wildcardListener ? wildcardListener.values() : []];
          for (const handler of allListeners) {
            try {
              handler(emittedEvent);
            } catch (err) {
              reportUnhandledError(err);
            }
          }
        },
        actionExecutor: (action) => {
          const exec = () => {
            this._actorScope.system._sendInspectionEvent({
              type: "@xstate.action",
              actorRef: this,
              action: {
                type: action.type,
                params: action.params
              }
            });
            if (!action.exec) {
              return;
            }
            const saveExecutingCustomAction = executingCustomAction;
            try {
              executingCustomAction = true;
              action.exec(action.info, action.params);
            } finally {
              executingCustomAction = saveExecutingCustomAction;
            }
          };
          if (this._processingStatus === ProcessingStatus.Running) {
            exec();
          } else {
            this._deferred.push(exec);
          }
        }
      };
      this.send = this.send.bind(this);
      this.system._sendInspectionEvent({
        type: "@xstate.actor",
        actorRef: this
      });
      if (systemId) {
        this.systemId = systemId;
        this.system._set(systemId, this);
      }
      this._initState(options?.snapshot ?? options?.state);
      if (systemId && this._snapshot.status !== "active") {
        this.system._unregister(this);
      }
    }
    _initState(persistedState) {
      try {
        this._snapshot = persistedState ? this.logic.restoreSnapshot ? this.logic.restoreSnapshot(persistedState, this._actorScope) : persistedState : this.logic.getInitialSnapshot(this._actorScope, this.options?.input);
      } catch (err) {
        this._snapshot = {
          status: "error",
          output: void 0,
          error: err
        };
      }
    }
    update(snapshot, event) {
      this._snapshot = snapshot;
      let deferredFn;
      while (deferredFn = this._deferred.shift()) {
        try {
          deferredFn();
        } catch (err) {
          this._deferred.length = 0;
          this._snapshot = {
            ...snapshot,
            status: "error",
            error: err
          };
        }
      }
      switch (this._snapshot.status) {
        case "active":
          for (const observer of this.observers) {
            try {
              observer.next?.(snapshot);
            } catch (err) {
              reportUnhandledError(err);
            }
          }
          break;
        case "done":
          for (const observer of this.observers) {
            try {
              observer.next?.(snapshot);
            } catch (err) {
              reportUnhandledError(err);
            }
          }
          this._stopProcedure();
          this._complete();
          this._doneEvent = createDoneActorEvent(this.id, this._snapshot.output);
          if (this._parent) {
            this.system._relay(this, this._parent, this._doneEvent);
          }
          break;
        case "error":
          this._error(this._snapshot.error);
          break;
      }
      this.system._sendInspectionEvent({
        type: "@xstate.snapshot",
        actorRef: this,
        event,
        snapshot
      });
    }
    /**
     * Subscribe an observer to an actor’s snapshot values.
     *
     * @remarks
     * The observer will receive the actor’s snapshot value when it is emitted.
     * The observer can be:
     *
     * - A plain function that receives the latest snapshot, or
     * - An observer object whose `.next(snapshot)` method receives the latest
     *   snapshot
     *
     * @example
     *
     * ```ts
     * // Observer as a plain function
     * const subscription = actor.subscribe((snapshot) => {
     *   console.log(snapshot);
     * });
     * ```
     *
     * @example
     *
     * ```ts
     * // Observer as an object
     * const subscription = actor.subscribe({
     *   next(snapshot) {
     *     console.log(snapshot);
     *   },
     *   error(err) {
     *     // ...
     *   },
     *   complete() {
     *     // ...
     *   }
     * });
     * ```
     *
     * The return value of `actor.subscribe(observer)` is a subscription object
     * that has an `.unsubscribe()` method. You can call
     * `subscription.unsubscribe()` to unsubscribe the observer:
     *
     * @example
     *
     * ```ts
     * const subscription = actor.subscribe((snapshot) => {
     *   // ...
     * });
     *
     * // Unsubscribe the observer
     * subscription.unsubscribe();
     * ```
     *
     * When the actor is stopped, all of its observers will automatically be
     * unsubscribed.
     *
     * @param observer - Either a plain function that receives the latest
     *   snapshot, or an observer object whose `.next(snapshot)` method receives
     *   the latest snapshot
     */
    subscribe(nextListenerOrObserver, errorListener, completeListener) {
      const observer = toObserver(nextListenerOrObserver, errorListener, completeListener);
      if (this._processingStatus !== ProcessingStatus.Stopped) {
        this.observers.add(observer);
      } else {
        switch (this._snapshot.status) {
          case "done":
            try {
              observer.complete?.();
            } catch (err) {
              reportUnhandledError(err);
            }
            break;
          case "error": {
            const err = this._snapshot.error;
            if (!observer.error) {
              reportUnhandledError(err);
            } else {
              try {
                observer.error(err);
              } catch (err2) {
                reportUnhandledError(err2);
              }
            }
            break;
          }
        }
      }
      return {
        unsubscribe: () => {
          this.observers.delete(observer);
        }
      };
    }
    on(type, handler) {
      let listeners = this.eventListeners.get(type);
      if (!listeners) {
        listeners = /* @__PURE__ */ new Set();
        this.eventListeners.set(type, listeners);
      }
      const wrappedHandler = handler.bind(void 0);
      listeners.add(wrappedHandler);
      return {
        unsubscribe: () => {
          listeners.delete(wrappedHandler);
        }
      };
    }
    /** Starts the Actor from the initial state */
    start() {
      if (this._processingStatus === ProcessingStatus.Running) {
        return this;
      }
      if (this._syncSnapshot) {
        this.subscribe({
          next: (snapshot) => {
            if (snapshot.status === "active") {
              this.system._relay(this, this._parent, {
                type: `xstate.snapshot.${this.id}`,
                snapshot
              });
            }
          },
          error: () => {
          }
        });
      }
      this.system._register(this.sessionId, this);
      if (this.systemId) {
        this.system._set(this.systemId, this);
      }
      this._processingStatus = ProcessingStatus.Running;
      const initEvent = createInitEvent(this.options.input);
      this.system._sendInspectionEvent({
        type: "@xstate.event",
        sourceRef: this._parent,
        actorRef: this,
        event: initEvent
      });
      const status = this._snapshot.status;
      switch (status) {
        case "done":
          this.update(this._snapshot, initEvent);
          return this;
        case "error":
          this._error(this._snapshot.error);
          return this;
      }
      if (!this._parent) {
        this.system.start();
      }
      if (this.logic.start) {
        try {
          this.logic.start(this._snapshot, this._actorScope);
        } catch (err) {
          this._snapshot = {
            ...this._snapshot,
            status: "error",
            error: err
          };
          this._error(err);
          return this;
        }
      }
      this.update(this._snapshot, initEvent);
      if (this.options.devTools) {
        this.attachDevTools();
      }
      this.mailbox.start();
      return this;
    }
    _process(event) {
      let nextState;
      let caughtError;
      try {
        nextState = this.logic.transition(this._snapshot, event, this._actorScope);
      } catch (err) {
        caughtError = {
          err
        };
      }
      if (caughtError) {
        const {
          err
        } = caughtError;
        this._snapshot = {
          ...this._snapshot,
          status: "error",
          error: err
        };
        this._error(err);
        return;
      }
      this.update(nextState, event);
      if (event.type === XSTATE_STOP) {
        this._stopProcedure();
        this._complete();
      }
    }
    _stop() {
      if (this._processingStatus === ProcessingStatus.Stopped) {
        return this;
      }
      this.mailbox.clear();
      if (this._processingStatus === ProcessingStatus.NotStarted) {
        this._processingStatus = ProcessingStatus.Stopped;
        return this;
      }
      this.mailbox.enqueue({
        type: XSTATE_STOP
      });
      return this;
    }
    /** Stops the Actor and unsubscribe all listeners. */
    stop() {
      if (this._parent) {
        throw new Error("A non-root actor cannot be stopped directly.");
      }
      return this._stop();
    }
    _complete() {
      for (const observer of this.observers) {
        try {
          observer.complete?.();
        } catch (err) {
          reportUnhandledError(err);
        }
      }
      this.observers.clear();
    }
    _reportError(err) {
      if (!this.observers.size) {
        if (!this._parent) {
          reportUnhandledError(err);
        }
        return;
      }
      let reportError = false;
      for (const observer of this.observers) {
        const errorListener = observer.error;
        reportError ||= !errorListener;
        try {
          errorListener?.(err);
        } catch (err2) {
          reportUnhandledError(err2);
        }
      }
      this.observers.clear();
      if (reportError) {
        reportUnhandledError(err);
      }
    }
    _error(err) {
      this._stopProcedure();
      this._reportError(err);
      if (this._parent) {
        this.system._relay(this, this._parent, createErrorActorEvent(this.id, err));
      }
    }
    // TODO: atm children don't belong entirely to the actor so
    // in a way - it's not even super aware of them
    // so we can't stop them from here but we really should!
    // right now, they are being stopped within the machine's transition
    // but that could throw and leave us with "orphaned" active actors
    _stopProcedure() {
      if (this._processingStatus !== ProcessingStatus.Running) {
        return this;
      }
      this.system.scheduler.cancelAll(this);
      this.mailbox.clear();
      this.mailbox = new Mailbox(this._process.bind(this));
      this._processingStatus = ProcessingStatus.Stopped;
      this.system._unregister(this);
      return this;
    }
    /** @internal */
    _send(event) {
      if (this._processingStatus === ProcessingStatus.Stopped) {
        return;
      }
      this.mailbox.enqueue(event);
    }
    /**
     * Sends an event to the running Actor to trigger a transition.
     *
     * @param event The event to send
     */
    send(event) {
      this.system._relay(void 0, this, event);
    }
    attachDevTools() {
      const {
        devTools
      } = this.options;
      if (devTools) {
        const resolvedDevToolsAdapter = typeof devTools === "function" ? devTools : devToolsAdapter;
        resolvedDevToolsAdapter(this);
      }
    }
    toJSON() {
      return {
        xstate$$type: $$ACTOR_TYPE,
        id: this.id
      };
    }
    /**
     * Obtain the internal state of the actor, which can be persisted.
     *
     * @remarks
     * The internal state can be persisted from any actor, not only machines.
     *
     * Note that the persisted state is not the same as the snapshot from
     * {@link Actor.getSnapshot}. Persisted state represents the internal state of
     * the actor, while snapshots represent the actor's last emitted value.
     *
     * Can be restored with {@link ActorOptions.state}
     * @see https://stately.ai/docs/persistence
     */
    getPersistedSnapshot(options) {
      return this.logic.getPersistedSnapshot(this._snapshot, options);
    }
    [symbolObservable]() {
      return this;
    }
    /**
     * Read an actor’s snapshot synchronously.
     *
     * @remarks
     * The snapshot represent an actor's last emitted value.
     *
     * When an actor receives an event, its internal state may change. An actor
     * may emit a snapshot when a state transition occurs.
     *
     * Note that some actors, such as callback actors generated with
     * `fromCallback`, will not emit snapshots.
     * @see {@link Actor.subscribe} to subscribe to an actor’s snapshot values.
     * @see {@link Actor.getPersistedSnapshot} to persist the internal state of an actor (which is more than just a snapshot).
     */
    getSnapshot() {
      return this._snapshot;
    }
  }
  function createActor(logic, ...[options]) {
    return new Actor(logic, options);
  }
  function resolveCancel(_, snapshot, actionArgs, actionParams, {
    sendId
  }) {
    const resolvedSendId = typeof sendId === "function" ? sendId(actionArgs, actionParams) : sendId;
    return [snapshot, {
      sendId: resolvedSendId
    }, void 0];
  }
  function executeCancel(actorScope, params) {
    actorScope.defer(() => {
      actorScope.system.scheduler.cancel(actorScope.self, params.sendId);
    });
  }
  function cancel(sendId) {
    function cancel2(_args, _params) {
    }
    cancel2.type = "xstate.cancel";
    cancel2.sendId = sendId;
    cancel2.resolve = resolveCancel;
    cancel2.execute = executeCancel;
    return cancel2;
  }
  function resolveSpawn(actorScope, snapshot, actionArgs, _actionParams, {
    id,
    systemId,
    src,
    input,
    syncSnapshot
  }) {
    const logic = typeof src === "string" ? resolveReferencedActor(snapshot.machine, src) : src;
    const resolvedId = typeof id === "function" ? id(actionArgs) : id;
    let actorRef;
    let resolvedInput = void 0;
    if (logic) {
      resolvedInput = typeof input === "function" ? input({
        context: snapshot.context,
        event: actionArgs.event,
        self: actorScope.self
      }) : input;
      actorRef = createActor(logic, {
        id: resolvedId,
        src,
        parent: actorScope.self,
        syncSnapshot,
        systemId,
        input: resolvedInput
      });
    }
    return [cloneMachineSnapshot(snapshot, {
      children: {
        ...snapshot.children,
        [resolvedId]: actorRef
      }
    }), {
      id,
      systemId,
      actorRef,
      src,
      input: resolvedInput
    }, void 0];
  }
  function executeSpawn(actorScope, {
    actorRef
  }) {
    if (!actorRef) {
      return;
    }
    actorScope.defer(() => {
      if (actorRef._processingStatus === ProcessingStatus.Stopped) {
        return;
      }
      actorRef.start();
    });
  }
  function spawnChild(...[src, {
    id,
    systemId,
    input,
    syncSnapshot = false
  } = {}]) {
    function spawnChild2(_args, _params) {
    }
    spawnChild2.type = "xstate.spawnChild";
    spawnChild2.id = id;
    spawnChild2.systemId = systemId;
    spawnChild2.src = src;
    spawnChild2.input = input;
    spawnChild2.syncSnapshot = syncSnapshot;
    spawnChild2.resolve = resolveSpawn;
    spawnChild2.execute = executeSpawn;
    return spawnChild2;
  }
  function resolveStop(_, snapshot, args, actionParams, {
    actorRef
  }) {
    const actorRefOrString = typeof actorRef === "function" ? actorRef(args, actionParams) : actorRef;
    const resolvedActorRef = typeof actorRefOrString === "string" ? snapshot.children[actorRefOrString] : actorRefOrString;
    let children = snapshot.children;
    if (resolvedActorRef) {
      children = {
        ...children
      };
      delete children[resolvedActorRef.id];
    }
    return [cloneMachineSnapshot(snapshot, {
      children
    }), resolvedActorRef, void 0];
  }
  function executeStop(actorScope, actorRef) {
    if (!actorRef) {
      return;
    }
    actorScope.system._unregister(actorRef);
    if (actorRef._processingStatus !== ProcessingStatus.Running) {
      actorScope.stopChild(actorRef);
      return;
    }
    actorScope.defer(() => {
      actorScope.stopChild(actorRef);
    });
  }
  function stopChild(actorRef) {
    function stop(_args, _params) {
    }
    stop.type = "xstate.stopChild";
    stop.actorRef = actorRef;
    stop.resolve = resolveStop;
    stop.execute = executeStop;
    return stop;
  }
  function evaluateGuard(guard, context, event, snapshot) {
    const {
      machine
    } = snapshot;
    const isInline = typeof guard === "function";
    const resolved = isInline ? guard : machine.implementations.guards[typeof guard === "string" ? guard : guard.type];
    if (!isInline && !resolved) {
      throw new Error(`Guard '${typeof guard === "string" ? guard : guard.type}' is not implemented.'.`);
    }
    if (typeof resolved !== "function") {
      return evaluateGuard(resolved, context, event, snapshot);
    }
    const guardArgs = {
      context,
      event
    };
    const guardParams = isInline || typeof guard === "string" ? void 0 : "params" in guard ? typeof guard.params === "function" ? guard.params({
      context,
      event
    }) : guard.params : void 0;
    if (!("check" in resolved)) {
      return resolved(guardArgs, guardParams);
    }
    const builtinGuard = resolved;
    return builtinGuard.check(
      snapshot,
      guardArgs,
      resolved
      // this holds all params
    );
  }
  const isAtomicStateNode = (stateNode) => stateNode.type === "atomic" || stateNode.type === "final";
  function getChildren(stateNode) {
    return Object.values(stateNode.states).filter((sn) => sn.type !== "history");
  }
  function getProperAncestors(stateNode, toStateNode) {
    const ancestors = [];
    if (toStateNode === stateNode) {
      return ancestors;
    }
    let m = stateNode.parent;
    while (m && m !== toStateNode) {
      ancestors.push(m);
      m = m.parent;
    }
    return ancestors;
  }
  function getAllStateNodes(stateNodes) {
    const nodeSet = new Set(stateNodes);
    const adjList = getAdjList(nodeSet);
    for (const s of nodeSet) {
      if (s.type === "compound" && (!adjList.get(s) || !adjList.get(s).length)) {
        getInitialStateNodesWithTheirAncestors(s).forEach((sn) => nodeSet.add(sn));
      } else {
        if (s.type === "parallel") {
          for (const child of getChildren(s)) {
            if (child.type === "history") {
              continue;
            }
            if (!nodeSet.has(child)) {
              const initialStates = getInitialStateNodesWithTheirAncestors(child);
              for (const initialStateNode of initialStates) {
                nodeSet.add(initialStateNode);
              }
            }
          }
        }
      }
    }
    for (const s of nodeSet) {
      let m = s.parent;
      while (m) {
        nodeSet.add(m);
        m = m.parent;
      }
    }
    return nodeSet;
  }
  function getValueFromAdj(baseNode, adjList) {
    const childStateNodes = adjList.get(baseNode);
    if (!childStateNodes) {
      return {};
    }
    if (baseNode.type === "compound") {
      const childStateNode = childStateNodes[0];
      if (childStateNode) {
        if (isAtomicStateNode(childStateNode)) {
          return childStateNode.key;
        }
      } else {
        return {};
      }
    }
    const stateValue = {};
    for (const childStateNode of childStateNodes) {
      stateValue[childStateNode.key] = getValueFromAdj(childStateNode, adjList);
    }
    return stateValue;
  }
  function getAdjList(stateNodes) {
    const adjList = /* @__PURE__ */ new Map();
    for (const s of stateNodes) {
      if (!adjList.has(s)) {
        adjList.set(s, []);
      }
      if (s.parent) {
        if (!adjList.has(s.parent)) {
          adjList.set(s.parent, []);
        }
        adjList.get(s.parent).push(s);
      }
    }
    return adjList;
  }
  function getStateValue(rootNode, stateNodes) {
    const config = getAllStateNodes(stateNodes);
    return getValueFromAdj(rootNode, getAdjList(config));
  }
  function isInFinalState(stateNodeSet, stateNode) {
    if (stateNode.type === "compound") {
      return getChildren(stateNode).some((s) => s.type === "final" && stateNodeSet.has(s));
    }
    if (stateNode.type === "parallel") {
      return getChildren(stateNode).every((sn) => isInFinalState(stateNodeSet, sn));
    }
    return stateNode.type === "final";
  }
  const isStateId = (str) => str[0] === STATE_IDENTIFIER$1;
  function getCandidates(stateNode, receivedEventType) {
    const candidates = stateNode.transitions.get(receivedEventType) || [...stateNode.transitions.keys()].filter((eventDescriptor) => {
      if (eventDescriptor === WILDCARD) {
        return true;
      }
      if (!eventDescriptor.endsWith(".*")) {
        return false;
      }
      const partialEventTokens = eventDescriptor.split(".");
      const eventTokens = receivedEventType.split(".");
      for (let tokenIndex = 0; tokenIndex < partialEventTokens.length; tokenIndex++) {
        const partialEventToken = partialEventTokens[tokenIndex];
        const eventToken = eventTokens[tokenIndex];
        if (partialEventToken === "*") {
          const isLastToken = tokenIndex === partialEventTokens.length - 1;
          return isLastToken;
        }
        if (partialEventToken !== eventToken) {
          return false;
        }
      }
      return true;
    }).sort((a, b) => b.length - a.length).flatMap((key) => stateNode.transitions.get(key));
    return candidates;
  }
  function getDelayedTransitions(stateNode) {
    const afterConfig = stateNode.config.after;
    if (!afterConfig) {
      return [];
    }
    const mutateEntryExit = (delay) => {
      const afterEvent = createAfterEvent(delay, stateNode.id);
      const eventType = afterEvent.type;
      stateNode.entry.push(raise(afterEvent, {
        id: eventType,
        delay
      }));
      stateNode.exit.push(cancel(eventType));
      return eventType;
    };
    const delayedTransitions = Object.keys(afterConfig).flatMap((delay) => {
      const configTransition = afterConfig[delay];
      const resolvedTransition = typeof configTransition === "string" ? {
        target: configTransition
      } : configTransition;
      const resolvedDelay = Number.isNaN(+delay) ? delay : +delay;
      const eventType = mutateEntryExit(resolvedDelay);
      return toArray(resolvedTransition).map((transition) => ({
        ...transition,
        event: eventType,
        delay: resolvedDelay
      }));
    });
    return delayedTransitions.map((delayedTransition) => {
      const {
        delay
      } = delayedTransition;
      return {
        ...formatTransition(stateNode, delayedTransition.event, delayedTransition),
        delay
      };
    });
  }
  function formatTransition(stateNode, descriptor, transitionConfig) {
    const normalizedTarget = normalizeTarget(transitionConfig.target);
    const reenter = transitionConfig.reenter ?? false;
    const target = resolveTarget(stateNode, normalizedTarget);
    const transition = {
      ...transitionConfig,
      actions: toArray(transitionConfig.actions),
      guard: transitionConfig.guard,
      target,
      source: stateNode,
      reenter,
      eventType: descriptor,
      toJSON: () => ({
        ...transition,
        source: `#${stateNode.id}`,
        target: target ? target.map((t) => `#${t.id}`) : void 0
      })
    };
    return transition;
  }
  function formatTransitions(stateNode) {
    const transitions = /* @__PURE__ */ new Map();
    if (stateNode.config.on) {
      for (const descriptor of Object.keys(stateNode.config.on)) {
        if (descriptor === NULL_EVENT) {
          throw new Error('Null events ("") cannot be specified as a transition key. Use `always: { ... }` instead.');
        }
        const transitionsConfig = stateNode.config.on[descriptor];
        transitions.set(descriptor, toTransitionConfigArray(transitionsConfig).map((t) => formatTransition(stateNode, descriptor, t)));
      }
    }
    if (stateNode.config.onDone) {
      const descriptor = `xstate.done.state.${stateNode.id}`;
      transitions.set(descriptor, toTransitionConfigArray(stateNode.config.onDone).map((t) => formatTransition(stateNode, descriptor, t)));
    }
    for (const invokeDef of stateNode.invoke) {
      if (invokeDef.onDone) {
        const descriptor = `xstate.done.actor.${invokeDef.id}`;
        transitions.set(descriptor, toTransitionConfigArray(invokeDef.onDone).map((t) => formatTransition(stateNode, descriptor, t)));
      }
      if (invokeDef.onError) {
        const descriptor = `xstate.error.actor.${invokeDef.id}`;
        transitions.set(descriptor, toTransitionConfigArray(invokeDef.onError).map((t) => formatTransition(stateNode, descriptor, t)));
      }
      if (invokeDef.onSnapshot) {
        const descriptor = `xstate.snapshot.${invokeDef.id}`;
        transitions.set(descriptor, toTransitionConfigArray(invokeDef.onSnapshot).map((t) => formatTransition(stateNode, descriptor, t)));
      }
    }
    for (const delayedTransition of stateNode.after) {
      let existing = transitions.get(delayedTransition.eventType);
      if (!existing) {
        existing = [];
        transitions.set(delayedTransition.eventType, existing);
      }
      existing.push(delayedTransition);
    }
    return transitions;
  }
  function formatInitialTransition(stateNode, _target) {
    const resolvedTarget = typeof _target === "string" ? stateNode.states[_target] : _target ? stateNode.states[_target.target] : void 0;
    if (!resolvedTarget && _target) {
      throw new Error(
        // eslint-disable-next-line @typescript-eslint/restrict-template-expressions, @typescript-eslint/no-base-to-string
        `Initial state node "${_target}" not found on parent state node #${stateNode.id}`
      );
    }
    const transition = {
      source: stateNode,
      actions: !_target || typeof _target === "string" ? [] : toArray(_target.actions),
      eventType: null,
      reenter: false,
      target: resolvedTarget ? [resolvedTarget] : [],
      toJSON: () => ({
        ...transition,
        source: `#${stateNode.id}`,
        target: resolvedTarget ? [`#${resolvedTarget.id}`] : []
      })
    };
    return transition;
  }
  function resolveTarget(stateNode, targets) {
    if (targets === void 0) {
      return void 0;
    }
    return targets.map((target) => {
      if (typeof target !== "string") {
        return target;
      }
      if (isStateId(target)) {
        return stateNode.machine.getStateNodeById(target);
      }
      const isInternalTarget = target[0] === STATE_DELIMITER;
      if (isInternalTarget && !stateNode.parent) {
        return getStateNodeByPath(stateNode, target.slice(1));
      }
      const resolvedTarget = isInternalTarget ? stateNode.key + target : target;
      if (stateNode.parent) {
        try {
          const targetStateNode = getStateNodeByPath(stateNode.parent, resolvedTarget);
          return targetStateNode;
        } catch (err) {
          throw new Error(`Invalid transition definition for state node '${stateNode.id}':
${err.message}`);
        }
      } else {
        throw new Error(`Invalid target: "${target}" is not a valid target from the root node. Did you mean ".${target}"?`);
      }
    });
  }
  function resolveHistoryDefaultTransition(stateNode) {
    const normalizedTarget = normalizeTarget(stateNode.config.target);
    if (!normalizedTarget) {
      return stateNode.parent.initial;
    }
    return {
      target: normalizedTarget.map((t) => typeof t === "string" ? getStateNodeByPath(stateNode.parent, t) : t)
    };
  }
  function isHistoryNode(stateNode) {
    return stateNode.type === "history";
  }
  function getInitialStateNodesWithTheirAncestors(stateNode) {
    const states = getInitialStateNodes(stateNode);
    for (const initialState of states) {
      for (const ancestor of getProperAncestors(initialState, stateNode)) {
        states.add(ancestor);
      }
    }
    return states;
  }
  function getInitialStateNodes(stateNode) {
    const set = /* @__PURE__ */ new Set();
    function iter(descStateNode) {
      if (set.has(descStateNode)) {
        return;
      }
      set.add(descStateNode);
      if (descStateNode.type === "compound") {
        iter(descStateNode.initial.target[0]);
      } else if (descStateNode.type === "parallel") {
        for (const child of getChildren(descStateNode)) {
          iter(child);
        }
      }
    }
    iter(stateNode);
    return set;
  }
  function getStateNode(stateNode, stateKey) {
    if (isStateId(stateKey)) {
      return stateNode.machine.getStateNodeById(stateKey);
    }
    if (!stateNode.states) {
      throw new Error(`Unable to retrieve child state '${stateKey}' from '${stateNode.id}'; no child states exist.`);
    }
    const result2 = stateNode.states[stateKey];
    if (!result2) {
      throw new Error(`Child state '${stateKey}' does not exist on '${stateNode.id}'`);
    }
    return result2;
  }
  function getStateNodeByPath(stateNode, statePath) {
    if (typeof statePath === "string" && isStateId(statePath)) {
      try {
        return stateNode.machine.getStateNodeById(statePath);
      } catch {
      }
    }
    const arrayStatePath = toStatePath(statePath).slice();
    let currentStateNode = stateNode;
    while (arrayStatePath.length) {
      const key = arrayStatePath.shift();
      if (!key.length) {
        break;
      }
      currentStateNode = getStateNode(currentStateNode, key);
    }
    return currentStateNode;
  }
  function getStateNodes(stateNode, stateValue) {
    if (typeof stateValue === "string") {
      const childStateNode = stateNode.states[stateValue];
      if (!childStateNode) {
        throw new Error(`State '${stateValue}' does not exist on '${stateNode.id}'`);
      }
      return [stateNode, childStateNode];
    }
    const childStateKeys = Object.keys(stateValue);
    const childStateNodes = childStateKeys.map((subStateKey) => getStateNode(stateNode, subStateKey)).filter(Boolean);
    return [stateNode.machine.root, stateNode].concat(childStateNodes, childStateKeys.reduce((allSubStateNodes, subStateKey) => {
      const subStateNode = getStateNode(stateNode, subStateKey);
      if (!subStateNode) {
        return allSubStateNodes;
      }
      const subStateNodes = getStateNodes(subStateNode, stateValue[subStateKey]);
      return allSubStateNodes.concat(subStateNodes);
    }, []));
  }
  function transitionAtomicNode(stateNode, stateValue, snapshot, event) {
    const childStateNode = getStateNode(stateNode, stateValue);
    const next = childStateNode.next(snapshot, event);
    if (!next || !next.length) {
      return stateNode.next(snapshot, event);
    }
    return next;
  }
  function transitionCompoundNode(stateNode, stateValue, snapshot, event) {
    const subStateKeys = Object.keys(stateValue);
    const childStateNode = getStateNode(stateNode, subStateKeys[0]);
    const next = transitionNode(childStateNode, stateValue[subStateKeys[0]], snapshot, event);
    if (!next || !next.length) {
      return stateNode.next(snapshot, event);
    }
    return next;
  }
  function transitionParallelNode(stateNode, stateValue, snapshot, event) {
    const allInnerTransitions = [];
    for (const subStateKey of Object.keys(stateValue)) {
      const subStateValue = stateValue[subStateKey];
      if (!subStateValue) {
        continue;
      }
      const subStateNode = getStateNode(stateNode, subStateKey);
      const innerTransitions = transitionNode(subStateNode, subStateValue, snapshot, event);
      if (innerTransitions) {
        allInnerTransitions.push(...innerTransitions);
      }
    }
    if (!allInnerTransitions.length) {
      return stateNode.next(snapshot, event);
    }
    return allInnerTransitions;
  }
  function transitionNode(stateNode, stateValue, snapshot, event) {
    if (typeof stateValue === "string") {
      return transitionAtomicNode(stateNode, stateValue, snapshot, event);
    }
    if (Object.keys(stateValue).length === 1) {
      return transitionCompoundNode(stateNode, stateValue, snapshot, event);
    }
    return transitionParallelNode(stateNode, stateValue, snapshot, event);
  }
  function getHistoryNodes(stateNode) {
    return Object.keys(stateNode.states).map((key) => stateNode.states[key]).filter((sn) => sn.type === "history");
  }
  function isDescendant(childStateNode, parentStateNode) {
    let marker = childStateNode;
    while (marker.parent && marker.parent !== parentStateNode) {
      marker = marker.parent;
    }
    return marker.parent === parentStateNode;
  }
  function hasIntersection(s1, s2) {
    const set1 = new Set(s1);
    const set2 = new Set(s2);
    for (const item of set1) {
      if (set2.has(item)) {
        return true;
      }
    }
    for (const item of set2) {
      if (set1.has(item)) {
        return true;
      }
    }
    return false;
  }
  function removeConflictingTransitions(enabledTransitions, stateNodeSet, historyValue) {
    const filteredTransitions = /* @__PURE__ */ new Set();
    for (const t1 of enabledTransitions) {
      let t1Preempted = false;
      const transitionsToRemove = /* @__PURE__ */ new Set();
      for (const t2 of filteredTransitions) {
        if (hasIntersection(computeExitSet([t1], stateNodeSet, historyValue), computeExitSet([t2], stateNodeSet, historyValue))) {
          if (isDescendant(t1.source, t2.source)) {
            transitionsToRemove.add(t2);
          } else {
            t1Preempted = true;
            break;
          }
        }
      }
      if (!t1Preempted) {
        for (const t3 of transitionsToRemove) {
          filteredTransitions.delete(t3);
        }
        filteredTransitions.add(t1);
      }
    }
    return Array.from(filteredTransitions);
  }
  function findLeastCommonAncestor(stateNodes) {
    const [head, ...tail] = stateNodes;
    for (const ancestor of getProperAncestors(head, void 0)) {
      if (tail.every((sn) => isDescendant(sn, ancestor))) {
        return ancestor;
      }
    }
  }
  function getEffectiveTargetStates(transition, historyValue) {
    if (!transition.target) {
      return [];
    }
    const targets = /* @__PURE__ */ new Set();
    for (const targetNode of transition.target) {
      if (isHistoryNode(targetNode)) {
        if (historyValue[targetNode.id]) {
          for (const node of historyValue[targetNode.id]) {
            targets.add(node);
          }
        } else {
          for (const node of getEffectiveTargetStates(resolveHistoryDefaultTransition(targetNode), historyValue)) {
            targets.add(node);
          }
        }
      } else {
        targets.add(targetNode);
      }
    }
    return [...targets];
  }
  function getTransitionDomain(transition, historyValue) {
    const targetStates = getEffectiveTargetStates(transition, historyValue);
    if (!targetStates) {
      return;
    }
    if (!transition.reenter && targetStates.every((target) => target === transition.source || isDescendant(target, transition.source))) {
      return transition.source;
    }
    const lca = findLeastCommonAncestor(targetStates.concat(transition.source));
    if (lca) {
      return lca;
    }
    if (transition.reenter) {
      return;
    }
    return transition.source.machine.root;
  }
  function computeExitSet(transitions, stateNodeSet, historyValue) {
    const statesToExit = /* @__PURE__ */ new Set();
    for (const t of transitions) {
      if (t.target?.length) {
        const domain = getTransitionDomain(t, historyValue);
        if (t.reenter && t.source === domain) {
          statesToExit.add(domain);
        }
        for (const stateNode of stateNodeSet) {
          if (isDescendant(stateNode, domain)) {
            statesToExit.add(stateNode);
          }
        }
      }
    }
    return [...statesToExit];
  }
  function areStateNodeCollectionsEqual(prevStateNodes, nextStateNodeSet) {
    if (prevStateNodes.length !== nextStateNodeSet.size) {
      return false;
    }
    for (const node of prevStateNodes) {
      if (!nextStateNodeSet.has(node)) {
        return false;
      }
    }
    return true;
  }
  function microstep(transitions, currentSnapshot, actorScope, event, isInitial, internalQueue) {
    if (!transitions.length) {
      return currentSnapshot;
    }
    const mutStateNodeSet = new Set(currentSnapshot._nodes);
    let historyValue = currentSnapshot.historyValue;
    const filteredTransitions = removeConflictingTransitions(transitions, mutStateNodeSet, historyValue);
    let nextState = currentSnapshot;
    if (!isInitial) {
      [nextState, historyValue] = exitStates(nextState, event, actorScope, filteredTransitions, mutStateNodeSet, historyValue, internalQueue, actorScope.actionExecutor);
    }
    nextState = resolveActionsAndContext(nextState, event, actorScope, filteredTransitions.flatMap((t) => t.actions), internalQueue, void 0);
    nextState = enterStates(nextState, event, actorScope, filteredTransitions, mutStateNodeSet, internalQueue, historyValue, isInitial);
    const nextStateNodes = [...mutStateNodeSet];
    if (nextState.status === "done") {
      nextState = resolveActionsAndContext(nextState, event, actorScope, nextStateNodes.sort((a, b) => b.order - a.order).flatMap((state) => state.exit), internalQueue, void 0);
    }
    try {
      if (historyValue === currentSnapshot.historyValue && areStateNodeCollectionsEqual(currentSnapshot._nodes, mutStateNodeSet)) {
        return nextState;
      }
      return cloneMachineSnapshot(nextState, {
        _nodes: nextStateNodes,
        historyValue
      });
    } catch (e) {
      throw e;
    }
  }
  function getMachineOutput(snapshot, event, actorScope, rootNode, rootCompletionNode) {
    if (rootNode.output === void 0) {
      return;
    }
    const doneStateEvent = createDoneStateEvent(rootCompletionNode.id, rootCompletionNode.output !== void 0 && rootCompletionNode.parent ? resolveOutput(rootCompletionNode.output, snapshot.context, event, actorScope.self) : void 0);
    return resolveOutput(rootNode.output, snapshot.context, doneStateEvent, actorScope.self);
  }
  function enterStates(currentSnapshot, event, actorScope, filteredTransitions, mutStateNodeSet, internalQueue, historyValue, isInitial) {
    let nextSnapshot = currentSnapshot;
    const statesToEnter = /* @__PURE__ */ new Set();
    const statesForDefaultEntry = /* @__PURE__ */ new Set();
    computeEntrySet(filteredTransitions, historyValue, statesForDefaultEntry, statesToEnter);
    if (isInitial) {
      statesForDefaultEntry.add(currentSnapshot.machine.root);
    }
    const completedNodes = /* @__PURE__ */ new Set();
    for (const stateNodeToEnter of [...statesToEnter].sort((a, b) => a.order - b.order)) {
      mutStateNodeSet.add(stateNodeToEnter);
      const actions = [];
      actions.push(...stateNodeToEnter.entry);
      for (const invokeDef of stateNodeToEnter.invoke) {
        actions.push(spawnChild(invokeDef.src, {
          ...invokeDef,
          syncSnapshot: !!invokeDef.onSnapshot
        }));
      }
      if (statesForDefaultEntry.has(stateNodeToEnter)) {
        const initialActions = stateNodeToEnter.initial.actions;
        actions.push(...initialActions);
      }
      nextSnapshot = resolveActionsAndContext(nextSnapshot, event, actorScope, actions, internalQueue, stateNodeToEnter.invoke.map((invokeDef) => invokeDef.id));
      if (stateNodeToEnter.type === "final") {
        const parent = stateNodeToEnter.parent;
        let ancestorMarker = parent?.type === "parallel" ? parent : parent?.parent;
        let rootCompletionNode = ancestorMarker || stateNodeToEnter;
        if (parent?.type === "compound") {
          internalQueue.push(createDoneStateEvent(parent.id, stateNodeToEnter.output !== void 0 ? resolveOutput(stateNodeToEnter.output, nextSnapshot.context, event, actorScope.self) : void 0));
        }
        while (ancestorMarker?.type === "parallel" && !completedNodes.has(ancestorMarker) && isInFinalState(mutStateNodeSet, ancestorMarker)) {
          completedNodes.add(ancestorMarker);
          internalQueue.push(createDoneStateEvent(ancestorMarker.id));
          rootCompletionNode = ancestorMarker;
          ancestorMarker = ancestorMarker.parent;
        }
        if (ancestorMarker) {
          continue;
        }
        nextSnapshot = cloneMachineSnapshot(nextSnapshot, {
          status: "done",
          output: getMachineOutput(nextSnapshot, event, actorScope, nextSnapshot.machine.root, rootCompletionNode)
        });
      }
    }
    return nextSnapshot;
  }
  function computeEntrySet(transitions, historyValue, statesForDefaultEntry, statesToEnter) {
    for (const t of transitions) {
      const domain = getTransitionDomain(t, historyValue);
      for (const s of t.target || []) {
        if (!isHistoryNode(s) && // if the target is different than the source then it will *definitely* be entered
        (t.source !== s || // we know that the domain can't lie within the source
        // if it's different than the source then it's outside of it and it means that the target has to be entered as well
        t.source !== domain || // reentering transitions always enter the target, even if it's the source itself
        t.reenter)) {
          statesToEnter.add(s);
          statesForDefaultEntry.add(s);
        }
        addDescendantStatesToEnter(s, historyValue, statesForDefaultEntry, statesToEnter);
      }
      const targetStates = getEffectiveTargetStates(t, historyValue);
      for (const s of targetStates) {
        const ancestors = getProperAncestors(s, domain);
        if (domain?.type === "parallel") {
          ancestors.push(domain);
        }
        addAncestorStatesToEnter(statesToEnter, historyValue, statesForDefaultEntry, ancestors, !t.source.parent && t.reenter ? void 0 : domain);
      }
    }
  }
  function addDescendantStatesToEnter(stateNode, historyValue, statesForDefaultEntry, statesToEnter) {
    if (isHistoryNode(stateNode)) {
      if (historyValue[stateNode.id]) {
        const historyStateNodes = historyValue[stateNode.id];
        for (const s of historyStateNodes) {
          statesToEnter.add(s);
          addDescendantStatesToEnter(s, historyValue, statesForDefaultEntry, statesToEnter);
        }
        for (const s of historyStateNodes) {
          addProperAncestorStatesToEnter(s, stateNode.parent, statesToEnter, historyValue, statesForDefaultEntry);
        }
      } else {
        const historyDefaultTransition = resolveHistoryDefaultTransition(stateNode);
        for (const s of historyDefaultTransition.target) {
          statesToEnter.add(s);
          if (historyDefaultTransition === stateNode.parent?.initial) {
            statesForDefaultEntry.add(stateNode.parent);
          }
          addDescendantStatesToEnter(s, historyValue, statesForDefaultEntry, statesToEnter);
        }
        for (const s of historyDefaultTransition.target) {
          addProperAncestorStatesToEnter(s, stateNode.parent, statesToEnter, historyValue, statesForDefaultEntry);
        }
      }
    } else {
      if (stateNode.type === "compound") {
        const [initialState] = stateNode.initial.target;
        if (!isHistoryNode(initialState)) {
          statesToEnter.add(initialState);
          statesForDefaultEntry.add(initialState);
        }
        addDescendantStatesToEnter(initialState, historyValue, statesForDefaultEntry, statesToEnter);
        addProperAncestorStatesToEnter(initialState, stateNode, statesToEnter, historyValue, statesForDefaultEntry);
      } else {
        if (stateNode.type === "parallel") {
          for (const child of getChildren(stateNode).filter((sn) => !isHistoryNode(sn))) {
            if (![...statesToEnter].some((s) => isDescendant(s, child))) {
              if (!isHistoryNode(child)) {
                statesToEnter.add(child);
                statesForDefaultEntry.add(child);
              }
              addDescendantStatesToEnter(child, historyValue, statesForDefaultEntry, statesToEnter);
            }
          }
        }
      }
    }
  }
  function addAncestorStatesToEnter(statesToEnter, historyValue, statesForDefaultEntry, ancestors, reentrancyDomain) {
    for (const anc of ancestors) {
      if (!reentrancyDomain || isDescendant(anc, reentrancyDomain)) {
        statesToEnter.add(anc);
      }
      if (anc.type === "parallel") {
        for (const child of getChildren(anc).filter((sn) => !isHistoryNode(sn))) {
          if (![...statesToEnter].some((s) => isDescendant(s, child))) {
            statesToEnter.add(child);
            addDescendantStatesToEnter(child, historyValue, statesForDefaultEntry, statesToEnter);
          }
        }
      }
    }
  }
  function addProperAncestorStatesToEnter(stateNode, toStateNode, statesToEnter, historyValue, statesForDefaultEntry) {
    addAncestorStatesToEnter(statesToEnter, historyValue, statesForDefaultEntry, getProperAncestors(stateNode, toStateNode));
  }
  function exitStates(currentSnapshot, event, actorScope, transitions, mutStateNodeSet, historyValue, internalQueue, _actionExecutor) {
    let nextSnapshot = currentSnapshot;
    const statesToExit = computeExitSet(transitions, mutStateNodeSet, historyValue);
    statesToExit.sort((a, b) => b.order - a.order);
    let changedHistory;
    for (const exitStateNode of statesToExit) {
      for (const historyNode of getHistoryNodes(exitStateNode)) {
        let predicate;
        if (historyNode.history === "deep") {
          predicate = (sn) => isAtomicStateNode(sn) && isDescendant(sn, exitStateNode);
        } else {
          predicate = (sn) => {
            return sn.parent === exitStateNode;
          };
        }
        changedHistory ??= {
          ...historyValue
        };
        changedHistory[historyNode.id] = Array.from(mutStateNodeSet).filter(predicate);
      }
    }
    for (const s of statesToExit) {
      nextSnapshot = resolveActionsAndContext(nextSnapshot, event, actorScope, [...s.exit, ...s.invoke.map((def) => stopChild(def.id))], internalQueue, void 0);
      mutStateNodeSet.delete(s);
    }
    return [nextSnapshot, changedHistory || historyValue];
  }
  function getAction(machine, actionType) {
    return machine.implementations.actions[actionType];
  }
  function resolveAndExecuteActionsWithContext(currentSnapshot, event, actorScope, actions, extra, retries) {
    const {
      machine
    } = currentSnapshot;
    let intermediateSnapshot = currentSnapshot;
    for (const action of actions) {
      const isInline = typeof action === "function";
      const resolvedAction = isInline ? action : (
        // the existing type of `.actions` assumes non-nullable `TExpressionAction`
        // it's fine to cast this here to get a common type and lack of errors in the rest of the code
        // our logic below makes sure that we call those 2 "variants" correctly
        getAction(machine, typeof action === "string" ? action : action.type)
      );
      const actionArgs = {
        context: intermediateSnapshot.context,
        event,
        self: actorScope.self,
        system: actorScope.system
      };
      const actionParams = isInline || typeof action === "string" ? void 0 : "params" in action ? typeof action.params === "function" ? action.params({
        context: intermediateSnapshot.context,
        event
      }) : action.params : void 0;
      if (!resolvedAction || !("resolve" in resolvedAction)) {
        actorScope.actionExecutor({
          type: typeof action === "string" ? action : typeof action === "object" ? action.type : action.name || "(anonymous)",
          info: actionArgs,
          params: actionParams,
          exec: resolvedAction
        });
        continue;
      }
      const builtinAction = resolvedAction;
      const [nextState, params, actions2] = builtinAction.resolve(
        actorScope,
        intermediateSnapshot,
        actionArgs,
        actionParams,
        resolvedAction,
        // this holds all params
        extra
      );
      intermediateSnapshot = nextState;
      if ("retryResolve" in builtinAction) {
        retries?.push([builtinAction, params]);
      }
      if ("execute" in builtinAction) {
        actorScope.actionExecutor({
          type: builtinAction.type,
          info: actionArgs,
          params,
          exec: builtinAction.execute.bind(null, actorScope, params)
        });
      }
      if (actions2) {
        intermediateSnapshot = resolveAndExecuteActionsWithContext(intermediateSnapshot, event, actorScope, actions2, extra, retries);
      }
    }
    return intermediateSnapshot;
  }
  function resolveActionsAndContext(currentSnapshot, event, actorScope, actions, internalQueue, deferredActorIds) {
    const retries = deferredActorIds ? [] : void 0;
    const nextState = resolveAndExecuteActionsWithContext(currentSnapshot, event, actorScope, actions, {
      internalQueue,
      deferredActorIds
    }, retries);
    retries?.forEach(([builtinAction, params]) => {
      builtinAction.retryResolve(actorScope, nextState, params);
    });
    return nextState;
  }
  function macrostep(snapshot, event, actorScope, internalQueue) {
    let nextSnapshot = snapshot;
    const microstates = [];
    function addMicrostate(microstate, event2, transitions) {
      actorScope.system._sendInspectionEvent({
        type: "@xstate.microstep",
        actorRef: actorScope.self,
        event: event2,
        snapshot: microstate,
        _transitions: transitions
      });
      microstates.push(microstate);
    }
    if (event.type === XSTATE_STOP) {
      nextSnapshot = cloneMachineSnapshot(stopChildren(nextSnapshot, event, actorScope), {
        status: "stopped"
      });
      addMicrostate(nextSnapshot, event, []);
      return {
        snapshot: nextSnapshot,
        microstates
      };
    }
    let nextEvent = event;
    if (nextEvent.type !== XSTATE_INIT) {
      const currentEvent = nextEvent;
      const isErr = isErrorActorEvent(currentEvent);
      const transitions = selectTransitions(currentEvent, nextSnapshot);
      if (isErr && !transitions.length) {
        nextSnapshot = cloneMachineSnapshot(snapshot, {
          status: "error",
          error: currentEvent.error
        });
        addMicrostate(nextSnapshot, currentEvent, []);
        return {
          snapshot: nextSnapshot,
          microstates
        };
      }
      nextSnapshot = microstep(
        transitions,
        snapshot,
        actorScope,
        nextEvent,
        false,
        // isInitial
        internalQueue
      );
      addMicrostate(nextSnapshot, currentEvent, transitions);
    }
    let shouldSelectEventlessTransitions = true;
    while (nextSnapshot.status === "active") {
      let enabledTransitions = shouldSelectEventlessTransitions ? selectEventlessTransitions(nextSnapshot, nextEvent) : [];
      const previousState = enabledTransitions.length ? nextSnapshot : void 0;
      if (!enabledTransitions.length) {
        if (!internalQueue.length) {
          break;
        }
        nextEvent = internalQueue.shift();
        enabledTransitions = selectTransitions(nextEvent, nextSnapshot);
      }
      nextSnapshot = microstep(enabledTransitions, nextSnapshot, actorScope, nextEvent, false, internalQueue);
      shouldSelectEventlessTransitions = nextSnapshot !== previousState;
      addMicrostate(nextSnapshot, nextEvent, enabledTransitions);
    }
    if (nextSnapshot.status !== "active") {
      stopChildren(nextSnapshot, nextEvent, actorScope);
    }
    return {
      snapshot: nextSnapshot,
      microstates
    };
  }
  function stopChildren(nextState, event, actorScope) {
    return resolveActionsAndContext(nextState, event, actorScope, Object.values(nextState.children).map((child) => stopChild(child)), [], void 0);
  }
  function selectTransitions(event, nextState) {
    return nextState.machine.getTransitionData(nextState, event);
  }
  function selectEventlessTransitions(nextState, event) {
    const enabledTransitionSet = /* @__PURE__ */ new Set();
    const atomicStates = nextState._nodes.filter(isAtomicStateNode);
    for (const stateNode of atomicStates) {
      loop: for (const s of [stateNode].concat(getProperAncestors(stateNode, void 0))) {
        if (!s.always) {
          continue;
        }
        for (const transition of s.always) {
          if (transition.guard === void 0 || evaluateGuard(transition.guard, nextState.context, event, nextState)) {
            enabledTransitionSet.add(transition);
            break loop;
          }
        }
      }
    }
    return removeConflictingTransitions(Array.from(enabledTransitionSet), new Set(nextState._nodes), nextState.historyValue);
  }
  function resolveStateValue(rootNode, stateValue) {
    const allStateNodes = getAllStateNodes(getStateNodes(rootNode, stateValue));
    return getStateValue(rootNode, [...allStateNodes]);
  }
  function isMachineSnapshot(value) {
    return !!value && typeof value === "object" && "machine" in value && "value" in value;
  }
  const machineSnapshotMatches = function matches(testValue) {
    return matchesState(testValue, this.value);
  };
  const machineSnapshotHasTag = function hasTag(tag) {
    return this.tags.has(tag);
  };
  const machineSnapshotCan = function can(event) {
    const transitionData = this.machine.getTransitionData(this, event);
    return !!transitionData?.length && // Check that at least one transition is not forbidden
    transitionData.some((t) => t.target !== void 0 || t.actions.length);
  };
  const machineSnapshotToJSON = function toJSON() {
    const {
      _nodes: nodes,
      tags,
      machine,
      getMeta,
      toJSON: toJSON2,
      can,
      hasTag,
      matches,
      ...jsonValues
    } = this;
    return {
      ...jsonValues,
      tags: Array.from(tags)
    };
  };
  const machineSnapshotGetMeta = function getMeta() {
    return this._nodes.reduce((acc, stateNode) => {
      if (stateNode.meta !== void 0) {
        acc[stateNode.id] = stateNode.meta;
      }
      return acc;
    }, {});
  };
  function createMachineSnapshot(config, machine) {
    return {
      status: config.status,
      output: config.output,
      error: config.error,
      machine,
      context: config.context,
      _nodes: config._nodes,
      value: getStateValue(machine.root, config._nodes),
      tags: new Set(config._nodes.flatMap((sn) => sn.tags)),
      children: config.children,
      historyValue: config.historyValue || {},
      matches: machineSnapshotMatches,
      hasTag: machineSnapshotHasTag,
      can: machineSnapshotCan,
      getMeta: machineSnapshotGetMeta,
      toJSON: machineSnapshotToJSON
    };
  }
  function cloneMachineSnapshot(snapshot, config = {}) {
    return createMachineSnapshot({
      ...snapshot,
      ...config
    }, snapshot.machine);
  }
  function serializeHistoryValue(historyValue) {
    if (typeof historyValue !== "object" || historyValue === null) {
      return {};
    }
    const result2 = {};
    for (const key in historyValue) {
      const value = historyValue[key];
      if (Array.isArray(value)) {
        result2[key] = value.map((item) => ({
          id: item.id
        }));
      }
    }
    return result2;
  }
  function getPersistedSnapshot(snapshot, options) {
    const {
      _nodes: nodes,
      tags,
      machine,
      children,
      context,
      can,
      hasTag,
      matches,
      getMeta,
      toJSON,
      ...jsonValues
    } = snapshot;
    const childrenJson = {};
    for (const id in children) {
      const child = children[id];
      childrenJson[id] = {
        snapshot: child.getPersistedSnapshot(options),
        src: child.src,
        systemId: child.systemId,
        syncSnapshot: child._syncSnapshot
      };
    }
    const persisted = {
      ...jsonValues,
      context: persistContext(context),
      children: childrenJson,
      historyValue: serializeHistoryValue(jsonValues.historyValue)
    };
    return persisted;
  }
  function persistContext(contextPart) {
    let copy;
    for (const key in contextPart) {
      const value = contextPart[key];
      if (value && typeof value === "object") {
        if ("sessionId" in value && "send" in value && "ref" in value) {
          copy ??= Array.isArray(contextPart) ? contextPart.slice() : {
            ...contextPart
          };
          copy[key] = {
            xstate$$type: $$ACTOR_TYPE,
            id: value.id
          };
        } else {
          const result2 = persistContext(value);
          if (result2 !== value) {
            copy ??= Array.isArray(contextPart) ? contextPart.slice() : {
              ...contextPart
            };
            copy[key] = result2;
          }
        }
      }
    }
    return copy ?? contextPart;
  }
  function resolveRaise(_, snapshot, args, actionParams, {
    event: eventOrExpr,
    id,
    delay
  }, {
    internalQueue
  }) {
    const delaysMap = snapshot.machine.implementations.delays;
    if (typeof eventOrExpr === "string") {
      throw new Error(
        // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
        `Only event objects may be used with raise; use raise({ type: "${eventOrExpr}" }) instead`
      );
    }
    const resolvedEvent = typeof eventOrExpr === "function" ? eventOrExpr(args, actionParams) : eventOrExpr;
    let resolvedDelay;
    if (typeof delay === "string") {
      const configDelay = delaysMap && delaysMap[delay];
      resolvedDelay = typeof configDelay === "function" ? configDelay(args, actionParams) : configDelay;
    } else {
      resolvedDelay = typeof delay === "function" ? delay(args, actionParams) : delay;
    }
    if (typeof resolvedDelay !== "number") {
      internalQueue.push(resolvedEvent);
    }
    return [snapshot, {
      event: resolvedEvent,
      id,
      delay: resolvedDelay
    }, void 0];
  }
  function executeRaise(actorScope, params) {
    const {
      event,
      delay,
      id
    } = params;
    if (typeof delay === "number") {
      actorScope.defer(() => {
        const self2 = actorScope.self;
        actorScope.system.scheduler.schedule(self2, self2, event, delay, id);
      });
      return;
    }
  }
  function raise(eventOrExpr, options) {
    function raise2(_args, _params) {
    }
    raise2.type = "xstate.raise";
    raise2.event = eventOrExpr;
    raise2.id = options?.id;
    raise2.delay = options?.delay;
    raise2.resolve = resolveRaise;
    raise2.execute = executeRaise;
    return raise2;
  }
  function createSpawner(actorScope, {
    machine,
    context
  }, event, spawnedChildren) {
    const spawn = (src, options) => {
      if (typeof src === "string") {
        const logic = resolveReferencedActor(machine, src);
        if (!logic) {
          throw new Error(`Actor logic '${src}' not implemented in machine '${machine.id}'`);
        }
        const actorRef = createActor(logic, {
          id: options?.id,
          parent: actorScope.self,
          syncSnapshot: options?.syncSnapshot,
          input: typeof options?.input === "function" ? options.input({
            context,
            event,
            self: actorScope.self
          }) : options?.input,
          src,
          systemId: options?.systemId
        });
        spawnedChildren[actorRef.id] = actorRef;
        return actorRef;
      } else {
        const actorRef = createActor(src, {
          id: options?.id,
          parent: actorScope.self,
          syncSnapshot: options?.syncSnapshot,
          input: options?.input,
          src,
          systemId: options?.systemId
        });
        return actorRef;
      }
    };
    return (src, options) => {
      const actorRef = spawn(src, options);
      spawnedChildren[actorRef.id] = actorRef;
      actorScope.defer(() => {
        if (actorRef._processingStatus === ProcessingStatus.Stopped) {
          return;
        }
        actorRef.start();
      });
      return actorRef;
    };
  }
  function resolveAssign(actorScope, snapshot, actionArgs, actionParams, {
    assignment
  }) {
    if (!snapshot.context) {
      throw new Error("Cannot assign to undefined `context`. Ensure that `context` is defined in the machine config.");
    }
    const spawnedChildren = {};
    const assignArgs = {
      context: snapshot.context,
      event: actionArgs.event,
      spawn: createSpawner(actorScope, snapshot, actionArgs.event, spawnedChildren),
      self: actorScope.self,
      system: actorScope.system
    };
    let partialUpdate = {};
    if (typeof assignment === "function") {
      partialUpdate = assignment(assignArgs, actionParams);
    } else {
      for (const key of Object.keys(assignment)) {
        const propAssignment = assignment[key];
        partialUpdate[key] = typeof propAssignment === "function" ? propAssignment(assignArgs, actionParams) : propAssignment;
      }
    }
    const updatedContext = Object.assign({}, snapshot.context, partialUpdate);
    return [cloneMachineSnapshot(snapshot, {
      context: updatedContext,
      children: Object.keys(spawnedChildren).length ? {
        ...snapshot.children,
        ...spawnedChildren
      } : snapshot.children
    }), void 0, void 0];
  }
  function assign(assignment) {
    function assign2(_args, _params) {
    }
    assign2.type = "xstate.assign";
    assign2.assignment = assignment;
    assign2.resolve = resolveAssign;
    return assign2;
  }
  const cache = /* @__PURE__ */ new WeakMap();
  function memo(object, key, fn) {
    let memoizedData = cache.get(object);
    if (!memoizedData) {
      memoizedData = {
        [key]: fn()
      };
      cache.set(object, memoizedData);
    } else if (!(key in memoizedData)) {
      memoizedData[key] = fn();
    }
    return memoizedData[key];
  }
  const EMPTY_OBJECT = {};
  const toSerializableAction = (action) => {
    if (typeof action === "string") {
      return {
        type: action
      };
    }
    if (typeof action === "function") {
      if ("resolve" in action) {
        return {
          type: action.type
        };
      }
      return {
        type: action.name
      };
    }
    return action;
  };
  class StateNode {
    constructor(config, options) {
      this.config = config;
      this.key = void 0;
      this.id = void 0;
      this.type = void 0;
      this.path = void 0;
      this.states = void 0;
      this.history = void 0;
      this.entry = void 0;
      this.exit = void 0;
      this.parent = void 0;
      this.machine = void 0;
      this.meta = void 0;
      this.output = void 0;
      this.order = -1;
      this.description = void 0;
      this.tags = [];
      this.transitions = void 0;
      this.always = void 0;
      this.parent = options._parent;
      this.key = options._key;
      this.machine = options._machine;
      this.path = this.parent ? this.parent.path.concat(this.key) : [];
      this.id = this.config.id || [this.machine.id, ...this.path].join(STATE_DELIMITER);
      this.type = this.config.type || (this.config.states && Object.keys(this.config.states).length ? "compound" : this.config.history ? "history" : "atomic");
      this.description = this.config.description;
      this.order = this.machine.idMap.size;
      this.machine.idMap.set(this.id, this);
      this.states = this.config.states ? mapValues(this.config.states, (stateConfig, key) => {
        const stateNode = new StateNode(stateConfig, {
          _parent: this,
          _key: key,
          _machine: this.machine
        });
        return stateNode;
      }) : EMPTY_OBJECT;
      if (this.type === "compound" && !this.config.initial) {
        throw new Error(`No initial state specified for compound state node "#${this.id}". Try adding { initial: "${Object.keys(this.states)[0]}" } to the state config.`);
      }
      this.history = this.config.history === true ? "shallow" : this.config.history || false;
      this.entry = toArray(this.config.entry).slice();
      this.exit = toArray(this.config.exit).slice();
      this.meta = this.config.meta;
      this.output = this.type === "final" || !this.parent ? this.config.output : void 0;
      this.tags = toArray(config.tags).slice();
    }
    /** @internal */
    _initialize() {
      this.transitions = formatTransitions(this);
      if (this.config.always) {
        this.always = toTransitionConfigArray(this.config.always).map((t) => formatTransition(this, NULL_EVENT, t));
      }
      Object.keys(this.states).forEach((key) => {
        this.states[key]._initialize();
      });
    }
    /** The well-structured state node definition. */
    get definition() {
      return {
        id: this.id,
        key: this.key,
        version: this.machine.version,
        type: this.type,
        initial: this.initial ? {
          target: this.initial.target,
          source: this,
          actions: this.initial.actions.map(toSerializableAction),
          eventType: null,
          reenter: false,
          toJSON: () => ({
            target: this.initial.target.map((t) => `#${t.id}`),
            source: `#${this.id}`,
            actions: this.initial.actions.map(toSerializableAction),
            eventType: null
          })
        } : void 0,
        history: this.history,
        states: mapValues(this.states, (state) => {
          return state.definition;
        }),
        on: this.on,
        transitions: [...this.transitions.values()].flat().map((t) => ({
          ...t,
          actions: t.actions.map(toSerializableAction)
        })),
        entry: this.entry.map(toSerializableAction),
        exit: this.exit.map(toSerializableAction),
        meta: this.meta,
        order: this.order || -1,
        output: this.output,
        invoke: this.invoke,
        description: this.description,
        tags: this.tags
      };
    }
    /** @internal */
    toJSON() {
      return this.definition;
    }
    /** The logic invoked as actors by this state node. */
    get invoke() {
      return memo(this, "invoke", () => toArray(this.config.invoke).map((invokeConfig, i) => {
        const {
          src,
          systemId
        } = invokeConfig;
        const resolvedId = invokeConfig.id ?? createInvokeId(this.id, i);
        const sourceName = typeof src === "string" ? src : `xstate.invoke.${createInvokeId(this.id, i)}`;
        return {
          ...invokeConfig,
          src: sourceName,
          id: resolvedId,
          systemId,
          toJSON() {
            const {
              onDone,
              onError,
              ...invokeDefValues
            } = invokeConfig;
            return {
              ...invokeDefValues,
              type: "xstate.invoke",
              src: sourceName,
              id: resolvedId
            };
          }
        };
      }));
    }
    /** The mapping of events to transitions. */
    get on() {
      return memo(this, "on", () => {
        const transitions = this.transitions;
        return [...transitions].flatMap(([descriptor, t]) => t.map((t2) => [descriptor, t2])).reduce((map, [descriptor, transition]) => {
          map[descriptor] = map[descriptor] || [];
          map[descriptor].push(transition);
          return map;
        }, {});
      });
    }
    get after() {
      return memo(this, "delayedTransitions", () => getDelayedTransitions(this));
    }
    get initial() {
      return memo(this, "initial", () => formatInitialTransition(this, this.config.initial));
    }
    /** @internal */
    next(snapshot, event) {
      const eventType = event.type;
      const actions = [];
      let selectedTransition;
      const candidates = memo(this, `candidates-${eventType}`, () => getCandidates(this, eventType));
      for (const candidate of candidates) {
        const {
          guard
        } = candidate;
        const resolvedContext = snapshot.context;
        let guardPassed = false;
        try {
          guardPassed = !guard || evaluateGuard(guard, resolvedContext, event, snapshot);
        } catch (err) {
          const guardType = typeof guard === "string" ? guard : typeof guard === "object" ? guard.type : void 0;
          throw new Error(`Unable to evaluate guard ${guardType ? `'${guardType}' ` : ""}in transition for event '${eventType}' in state node '${this.id}':
${err.message}`);
        }
        if (guardPassed) {
          actions.push(...candidate.actions);
          selectedTransition = candidate;
          break;
        }
      }
      return selectedTransition ? [selectedTransition] : void 0;
    }
    /** All the event types accepted by this state node and its descendants. */
    get events() {
      return memo(this, "events", () => {
        const {
          states
        } = this;
        const events = new Set(this.ownEvents);
        if (states) {
          for (const stateId of Object.keys(states)) {
            const state = states[stateId];
            if (state.states) {
              for (const event of state.events) {
                events.add(`${event}`);
              }
            }
          }
        }
        return Array.from(events);
      });
    }
    /**
     * All the events that have transitions directly from this state node.
     *
     * Excludes any inert events.
     */
    get ownEvents() {
      const events = new Set([...this.transitions.keys()].filter((descriptor) => {
        return this.transitions.get(descriptor).some((transition) => !(!transition.target && !transition.actions.length && !transition.reenter));
      }));
      return Array.from(events);
    }
  }
  const STATE_IDENTIFIER = "#";
  class StateMachine {
    constructor(config, implementations) {
      this.config = config;
      this.version = void 0;
      this.schemas = void 0;
      this.implementations = void 0;
      this.__xstatenode = true;
      this.idMap = /* @__PURE__ */ new Map();
      this.root = void 0;
      this.id = void 0;
      this.states = void 0;
      this.events = void 0;
      this.id = config.id || "(machine)";
      this.implementations = {
        actors: implementations?.actors ?? {},
        actions: implementations?.actions ?? {},
        delays: implementations?.delays ?? {},
        guards: implementations?.guards ?? {}
      };
      this.version = this.config.version;
      this.schemas = this.config.schemas;
      this.transition = this.transition.bind(this);
      this.getInitialSnapshot = this.getInitialSnapshot.bind(this);
      this.getPersistedSnapshot = this.getPersistedSnapshot.bind(this);
      this.restoreSnapshot = this.restoreSnapshot.bind(this);
      this.start = this.start.bind(this);
      this.root = new StateNode(config, {
        _key: this.id,
        _machine: this
      });
      this.root._initialize();
      this.states = this.root.states;
      this.events = this.root.events;
    }
    /**
     * Clones this state machine with the provided implementations.
     *
     * @param implementations Options (`actions`, `guards`, `actors`, `delays`) to
     *   recursively merge with the existing options.
     * @returns A new `StateMachine` instance with the provided implementations.
     */
    provide(implementations) {
      const {
        actions,
        guards,
        actors,
        delays
      } = this.implementations;
      return new StateMachine(this.config, {
        actions: {
          ...actions,
          ...implementations.actions
        },
        guards: {
          ...guards,
          ...implementations.guards
        },
        actors: {
          ...actors,
          ...implementations.actors
        },
        delays: {
          ...delays,
          ...implementations.delays
        }
      });
    }
    resolveState(config) {
      const resolvedStateValue = resolveStateValue(this.root, config.value);
      const nodeSet = getAllStateNodes(getStateNodes(this.root, resolvedStateValue));
      return createMachineSnapshot({
        _nodes: [...nodeSet],
        context: config.context || {},
        children: {},
        status: isInFinalState(nodeSet, this.root) ? "done" : config.status || "active",
        output: config.output,
        error: config.error,
        historyValue: config.historyValue
      }, this);
    }
    /**
     * Determines the next snapshot given the current `snapshot` and received
     * `event`. Calculates a full macrostep from all microsteps.
     *
     * @param snapshot The current snapshot
     * @param event The received event
     */
    transition(snapshot, event, actorScope) {
      return macrostep(snapshot, event, actorScope, []).snapshot;
    }
    /**
     * Determines the next state given the current `state` and `event`. Calculates
     * a microstep.
     *
     * @param state The current state
     * @param event The received event
     */
    microstep(snapshot, event, actorScope) {
      return macrostep(snapshot, event, actorScope, []).microstates;
    }
    getTransitionData(snapshot, event) {
      return transitionNode(this.root, snapshot.value, snapshot, event) || [];
    }
    /**
     * The initial state _before_ evaluating any microsteps. This "pre-initial"
     * state is provided to initial actions executed in the initial state.
     */
    getPreInitialState(actorScope, initEvent, internalQueue) {
      const {
        context
      } = this.config;
      const preInitial = createMachineSnapshot({
        context: typeof context !== "function" && context ? context : {},
        _nodes: [this.root],
        children: {},
        status: "active"
      }, this);
      if (typeof context === "function") {
        const assignment = ({
          spawn,
          event,
          self: self2
        }) => context({
          spawn,
          input: event.input,
          self: self2
        });
        return resolveActionsAndContext(preInitial, initEvent, actorScope, [assign(assignment)], internalQueue, void 0);
      }
      return preInitial;
    }
    /**
     * Returns the initial `State` instance, with reference to `self` as an
     * `ActorRef`.
     */
    getInitialSnapshot(actorScope, input) {
      const initEvent = createInitEvent(input);
      const internalQueue = [];
      const preInitialState = this.getPreInitialState(actorScope, initEvent, internalQueue);
      const nextState = microstep([{
        target: [...getInitialStateNodes(this.root)],
        source: this.root,
        reenter: true,
        actions: [],
        eventType: null,
        toJSON: null
        // TODO: fix
      }], preInitialState, actorScope, initEvent, true, internalQueue);
      const {
        snapshot: macroState
      } = macrostep(nextState, initEvent, actorScope, internalQueue);
      return macroState;
    }
    start(snapshot) {
      Object.values(snapshot.children).forEach((child) => {
        if (child.getSnapshot().status === "active") {
          child.start();
        }
      });
    }
    getStateNodeById(stateId) {
      const fullPath = toStatePath(stateId);
      const relativePath = fullPath.slice(1);
      const resolvedStateId = isStateId(fullPath[0]) ? fullPath[0].slice(STATE_IDENTIFIER.length) : fullPath[0];
      const stateNode = this.idMap.get(resolvedStateId);
      if (!stateNode) {
        throw new Error(`Child state node '#${resolvedStateId}' does not exist on machine '${this.id}'`);
      }
      return getStateNodeByPath(stateNode, relativePath);
    }
    get definition() {
      return this.root.definition;
    }
    toJSON() {
      return this.definition;
    }
    getPersistedSnapshot(snapshot, options) {
      return getPersistedSnapshot(snapshot, options);
    }
    restoreSnapshot(snapshot, _actorScope) {
      const children = {};
      const snapshotChildren = snapshot.children;
      Object.keys(snapshotChildren).forEach((actorId) => {
        const actorData = snapshotChildren[actorId];
        const childState = actorData.snapshot;
        const src = actorData.src;
        const logic = typeof src === "string" ? resolveReferencedActor(this, src) : src;
        if (!logic) {
          return;
        }
        const actorRef = createActor(logic, {
          id: actorId,
          parent: _actorScope.self,
          syncSnapshot: actorData.syncSnapshot,
          snapshot: childState,
          src,
          systemId: actorData.systemId
        });
        children[actorId] = actorRef;
      });
      function resolveHistoryReferencedState(root, referenced) {
        if (referenced instanceof StateNode) {
          return referenced;
        }
        try {
          return root.machine.getStateNodeById(referenced.id);
        } catch {
        }
      }
      function reviveHistoryValue(root, historyValue) {
        if (!historyValue || typeof historyValue !== "object") {
          return {};
        }
        const revived = {};
        for (const key in historyValue) {
          const arr = historyValue[key];
          for (const item of arr) {
            const resolved = resolveHistoryReferencedState(root, item);
            if (!resolved) {
              continue;
            }
            revived[key] ??= [];
            revived[key].push(resolved);
          }
        }
        return revived;
      }
      const revivedHistoryValue = reviveHistoryValue(this.root, snapshot.historyValue);
      const restoredSnapshot = createMachineSnapshot({
        ...snapshot,
        children,
        _nodes: Array.from(getAllStateNodes(getStateNodes(this.root, snapshot.value))),
        historyValue: revivedHistoryValue
      }, this);
      const seen = /* @__PURE__ */ new Set();
      function reviveContext(contextPart, children2) {
        if (seen.has(contextPart)) {
          return;
        }
        seen.add(contextPart);
        for (const key in contextPart) {
          const value = contextPart[key];
          if (value && typeof value === "object") {
            if ("xstate$$type" in value && value.xstate$$type === $$ACTOR_TYPE) {
              contextPart[key] = children2[value.id];
              continue;
            }
            reviveContext(value, children2);
          }
        }
      }
      reviveContext(restoredSnapshot.context, children);
      return restoredSnapshot;
    }
  }
  function resolveEmit(_, snapshot, args, actionParams, {
    event: eventOrExpr
  }) {
    const resolvedEvent = typeof eventOrExpr === "function" ? eventOrExpr(args, actionParams) : eventOrExpr;
    return [snapshot, {
      event: resolvedEvent
    }, void 0];
  }
  function executeEmit(actorScope, {
    event
  }) {
    actorScope.defer(() => actorScope.emit(event));
  }
  function emit(eventOrExpr) {
    function emit2(_args, _params) {
    }
    emit2.type = "xstate.emit";
    emit2.event = eventOrExpr;
    emit2.resolve = resolveEmit;
    emit2.execute = executeEmit;
    return emit2;
  }
  let SpecialTargets = /* @__PURE__ */ (function(SpecialTargets2) {
    SpecialTargets2["Parent"] = "#_parent";
    SpecialTargets2["Internal"] = "#_internal";
    return SpecialTargets2;
  })({});
  function resolveSendTo(actorScope, snapshot, args, actionParams, {
    to,
    event: eventOrExpr,
    id,
    delay
  }, extra) {
    const delaysMap = snapshot.machine.implementations.delays;
    if (typeof eventOrExpr === "string") {
      throw new Error(
        // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
        `Only event objects may be used with sendTo; use sendTo({ type: "${eventOrExpr}" }) instead`
      );
    }
    const resolvedEvent = typeof eventOrExpr === "function" ? eventOrExpr(args, actionParams) : eventOrExpr;
    let resolvedDelay;
    if (typeof delay === "string") {
      const configDelay = delaysMap && delaysMap[delay];
      resolvedDelay = typeof configDelay === "function" ? configDelay(args, actionParams) : configDelay;
    } else {
      resolvedDelay = typeof delay === "function" ? delay(args, actionParams) : delay;
    }
    const resolvedTarget = typeof to === "function" ? to(args, actionParams) : to;
    let targetActorRef;
    if (typeof resolvedTarget === "string") {
      if (resolvedTarget === SpecialTargets.Parent) {
        targetActorRef = actorScope.self._parent;
      } else if (resolvedTarget === SpecialTargets.Internal) {
        targetActorRef = actorScope.self;
      } else if (resolvedTarget.startsWith("#_")) {
        targetActorRef = snapshot.children[resolvedTarget.slice(2)];
      } else {
        targetActorRef = extra.deferredActorIds?.includes(resolvedTarget) ? resolvedTarget : snapshot.children[resolvedTarget];
      }
      if (!targetActorRef) {
        throw new Error(`Unable to send event to actor '${resolvedTarget}' from machine '${snapshot.machine.id}'.`);
      }
    } else {
      targetActorRef = resolvedTarget || actorScope.self;
    }
    return [snapshot, {
      to: targetActorRef,
      targetId: typeof resolvedTarget === "string" ? resolvedTarget : void 0,
      event: resolvedEvent,
      id,
      delay: resolvedDelay
    }, void 0];
  }
  function retryResolveSendTo(_, snapshot, params) {
    if (typeof params.to === "string") {
      params.to = snapshot.children[params.to];
    }
  }
  function executeSendTo(actorScope, params) {
    actorScope.defer(() => {
      const {
        to,
        event,
        delay,
        id
      } = params;
      if (typeof delay === "number") {
        actorScope.system.scheduler.schedule(actorScope.self, to, event, delay, id);
        return;
      }
      actorScope.system._relay(
        actorScope.self,
        // at this point, in a deferred task, it should already be mutated by retryResolveSendTo
        // if it initially started as a string
        to,
        event.type === XSTATE_ERROR ? createErrorActorEvent(actorScope.self.id, event.data) : event
      );
    });
  }
  function sendTo(to, eventOrExpr, options) {
    function sendTo2(_args, _params) {
    }
    sendTo2.type = "xstate.sendTo";
    sendTo2.to = to;
    sendTo2.event = eventOrExpr;
    sendTo2.id = options?.id;
    sendTo2.delay = options?.delay;
    sendTo2.resolve = resolveSendTo;
    sendTo2.retryResolve = retryResolveSendTo;
    sendTo2.execute = executeSendTo;
    return sendTo2;
  }
  function sendParent(event, options) {
    return sendTo(SpecialTargets.Parent, event, options);
  }
  function resolveEnqueueActions(actorScope, snapshot, args, actionParams, {
    collect
  }) {
    const actions = [];
    const enqueue = function enqueue2(action) {
      actions.push(action);
    };
    enqueue.assign = (...args2) => {
      actions.push(assign(...args2));
    };
    enqueue.cancel = (...args2) => {
      actions.push(cancel(...args2));
    };
    enqueue.raise = (...args2) => {
      actions.push(raise(...args2));
    };
    enqueue.sendTo = (...args2) => {
      actions.push(sendTo(...args2));
    };
    enqueue.sendParent = (...args2) => {
      actions.push(sendParent(...args2));
    };
    enqueue.spawnChild = (...args2) => {
      actions.push(spawnChild(...args2));
    };
    enqueue.stopChild = (...args2) => {
      actions.push(stopChild(...args2));
    };
    enqueue.emit = (...args2) => {
      actions.push(emit(...args2));
    };
    collect({
      context: args.context,
      event: args.event,
      enqueue,
      check: (guard) => evaluateGuard(guard, snapshot.context, args.event, snapshot),
      self: actorScope.self,
      system: actorScope.system
    }, actionParams);
    return [snapshot, void 0, actions];
  }
  function enqueueActions(collect) {
    function enqueueActions2(_args, _params) {
    }
    enqueueActions2.type = "xstate.enqueueActions";
    enqueueActions2.collect = collect;
    enqueueActions2.resolve = resolveEnqueueActions;
    return enqueueActions2;
  }
  function resolveLog(_, snapshot, actionArgs, actionParams, {
    value,
    label
  }) {
    return [snapshot, {
      value: typeof value === "function" ? value(actionArgs, actionParams) : value,
      label
    }, void 0];
  }
  function executeLog({
    logger: logger2
  }, {
    value,
    label
  }) {
    if (label) {
      logger2(label, value);
    } else {
      logger2(value);
    }
  }
  function log(value = ({
    context,
    event
  }) => ({
    context,
    event
  }), label) {
    function log2(_args, _params) {
    }
    log2.type = "xstate.log";
    log2.value = value;
    log2.label = label;
    log2.resolve = resolveLog;
    log2.execute = executeLog;
    return log2;
  }
  function createMachine(config, implementations) {
    return new StateMachine(config, implementations);
  }
  function setup({
    schemas,
    actors,
    actions,
    guards,
    delays
  }) {
    return {
      assign,
      sendTo,
      raise,
      log,
      cancel,
      stopChild,
      enqueueActions,
      emit,
      spawnChild,
      createStateConfig: (config) => config,
      createAction: (fn) => fn,
      createMachine: (config) => createMachine({
        ...config,
        schemas
      }, {
        actors,
        actions,
        guards,
        delays
      })
    };
  }
  const DEFAULT_MAX_STORED_LOGS = 5e3;
  const BATCH_WRITE_INTERVAL = 1e4;
  const BATCH_SIZE_THRESHOLD = 50;
  const _Logger = class _Logger {
    constructor(context, config, parentContext) {
      this.parentContext = parentContext;
      this.config = {
        context,
        remoteLogging: config?.remoteLogging ?? true,
        remoteUrl: config?.remoteUrl ?? "http://localhost:7856/log",
        consoleLogging: config?.consoleLogging ?? true,
        storeLogs: config?.storeLogs ?? true,
        maxStoredLogs: config?.maxStoredLogs ?? DEFAULT_MAX_STORED_LOGS
      };
      if (typeof chrome !== "undefined" && chrome.storage) {
        chrome.storage.local.get(["automationConfig"], (result2) => {
          if (result2.automationConfig?.remoteLogging !== void 0) {
            this.config.remoteLogging = result2.automationConfig.remoteLogging;
          }
          if (result2.automationConfig?.storeLogs !== void 0) {
            this.config.storeLogs = result2.automationConfig.storeLogs;
          }
          if (result2.automationConfig?.maxStoredLogs !== void 0) {
            this.config.maxStoredLogs = result2.automationConfig.maxStoredLogs;
          }
        });
        chrome.storage.onChanged.addListener((changes, areaName) => {
          if (areaName === "local" && changes.automationConfig?.newValue) {
            const newConfig = changes.automationConfig.newValue;
            if (newConfig.remoteLogging !== void 0) {
              this.config.remoteLogging = newConfig.remoteLogging;
            }
            if (newConfig.storeLogs !== void 0) {
              this.config.storeLogs = newConfig.storeLogs;
            }
            if (newConfig.maxStoredLogs !== void 0) {
              this.config.maxStoredLogs = newConfig.maxStoredLogs;
            }
          }
        });
      }
    }
    /**
     * Send log to remote server
     */
    async sendToRemote(entry) {
      if (!this.config.remoteLogging) return;
      try {
        await fetch(this.config.remoteUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify(entry)
        }).catch(() => {
        });
      } catch (error) {
      }
    }
    /**
     * Flush buffered logs to chrome.storage
     * This is called periodically or when buffer reaches threshold
     */
    static async flushLogsToStorage() {
      if (typeof chrome === "undefined" || !chrome.storage) return;
      if (_Logger.logBuffer.length === 0) return;
      if (_Logger.flushTimer) {
        clearTimeout(_Logger.flushTimer);
        _Logger.flushTimer = null;
      }
      _Logger.isFlushScheduled = false;
      const logsToFlush = [..._Logger.logBuffer];
      _Logger.logBuffer = [];
      try {
        const result2 = await chrome.storage.local.get(["debugLogs", "automationConfig"]);
        const existingLogs = result2.debugLogs || [];
        const maxLogs = result2.automationConfig?.maxStoredLogs ?? DEFAULT_MAX_STORED_LOGS;
        const allLogs = [...existingLogs, ...logsToFlush];
        if (allLogs.length > maxLogs) {
          allLogs.splice(0, allLogs.length - maxLogs);
        }
        await chrome.storage.local.set({ debugLogs: allLogs });
      } catch (error) {
        _Logger.logBuffer.unshift(...logsToFlush);
        console.error("[LF] Failed to flush logs:", error);
      }
      if (_Logger.logBuffer.length > 0) {
        _Logger.scheduleFlush();
      }
    }
    /**
     * Schedule a flush to happen after the interval
     */
    static scheduleFlush() {
      if (_Logger.isFlushScheduled) return;
      _Logger.isFlushScheduled = true;
      _Logger.flushTimer = setTimeout(() => {
        _Logger.flushLogsToStorage();
      }, BATCH_WRITE_INTERVAL);
    }
    /**
     * Store log entry in buffer (will be flushed periodically)
     */
    storeLog(entry) {
      if (!this.config.storeLogs) return;
      if (typeof chrome === "undefined" || !chrome.storage) return;
      try {
        _Logger.logBuffer.push(entry);
        if (_Logger.logBuffer.length >= BATCH_SIZE_THRESHOLD) {
          _Logger.flushLogsToStorage();
        } else {
          _Logger.scheduleFlush();
        }
      } catch (error) {
        console.error("[LF] Failed to buffer log:", error);
      }
    }
    /**
     * Format message with prefix
     */
    formatMessage(message) {
      const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
      return `[LF][${fullContext}] ${message}`;
    }
    /**
     * Serialize data for logging
     */
    serializeData(data) {
      if (data === void 0) return void 0;
      try {
        return JSON.parse(JSON.stringify(data));
      } catch (error) {
        return String(data);
      }
    }
    /**
     * Core logging function
     */
    logInternal(level, ...args) {
      const message = args.map((arg) => {
        if (typeof arg === "string") return arg;
        if (typeof arg === "object") {
          try {
            return JSON.stringify(arg);
          } catch (error) {
            return "[Circular Reference]";
          }
        }
        return String(arg);
      }).join(" ");
      const entry = {
        timestamp: (/* @__PURE__ */ new Date()).toISOString(),
        context: this.config.context,
        level,
        message,
        data: args.length > 1 ? this.serializeData(args.slice(1)) : void 0
      };
      if (this.config.consoleLogging) {
        const consoleMethod = console[level] || console.log;
        const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
        consoleMethod(`[LF][${fullContext}]`, ...args);
      }
      this.storeLog(entry);
      this.sendToRemote(entry);
    }
    /**
     * Public logging methods - support unlimited parameters like console.log()
     */
    log(...args) {
      this.logInternal("log", ...args);
    }
    info(...args) {
      this.logInternal("info", ...args);
    }
    warn(...args) {
      this.logInternal("warn", ...args);
    }
    error(...args) {
      this.logInternal("error", ...args);
    }
    debug(...args) {
      this.logInternal("debug", ...args);
    }
    /**
     * Update logger configuration
     */
    setConfig(config) {
      this.config = { ...this.config, ...config };
    }
    /**
     * Enable/disable remote logging
     */
    setRemoteLogging(enabled) {
      this.config.remoteLogging = enabled;
    }
    /**
     * Enable/disable console logging
     */
    setConsoleLogging(enabled) {
      this.config.consoleLogging = enabled;
    }
    /**
     * Create a nested logger with additional context
     */
    createNestedLogger(nestedContext) {
      const fullContext = this.parentContext ? `${this.parentContext}][${this.config.context}` : this.config.context;
      return new _Logger(
        nestedContext,
        {
          remoteLogging: this.config.remoteLogging,
          remoteUrl: this.config.remoteUrl,
          consoleLogging: this.config.consoleLogging
        },
        fullContext
      );
    }
    /**
     * Flush all buffered logs to storage immediately
     * Call this before extension unload to prevent log loss
     */
    static async flushLogs() {
      return _Logger.flushLogsToStorage();
    }
  };
  _Logger.logBuffer = [];
  _Logger.flushTimer = null;
  _Logger.isFlushScheduled = false;
  let Logger = _Logger;
  function createLogger(context, config, parentContext) {
    return new Logger(context, config, parentContext);
  }
  createLogger("POPUP");
  const extensionLogger = createLogger("SW");
  createLogger("REDDIT");
  createLogger("DEVVIT");
  createLogger("DEVVIT-GIAE");
  const STORAGE_PROPAGATION_DELAY = 500;
  const INITIAL_RETRY_DELAY = 2e3;
  const RETRY_BACKOFF_BASE = 2;
  const STATE_TIMEOUT = 1e4;
  const logger$1 = createLogger("StateMachine");
  const botMachine = setup({
    actions: {
      // Set current mission from event
      setMission: assign({
        currentMissionId: ({ event }) => {
          if (event.type === "MISSION_PAGE_LOADED" || event.type === "NEXT_MISSION_FOUND" || event.type === "NAVIGATE_TO_MISSION") {
            return event.missionId;
          }
          return null;
        },
        currentMissionPermalink: ({ event }) => {
          if (event.type === "MISSION_PAGE_LOADED" || event.type === "NEXT_MISSION_FOUND" || event.type === "NAVIGATE_TO_MISSION") {
            return event.permalink;
          }
          return null;
        }
      }),
      // Clear mission info
      clearMission: assign({
        currentMissionId: null,
        currentMissionPermalink: null
      }),
      // Set error message
      setError: assign({
        errorMessage: ({ event }) => {
          if (event.type === "ERROR_OCCURRED") {
            return event.message;
          }
          return null;
        },
        retryCount: ({ context, event }) => {
          if (event.type === "ERROR_OCCURRED") {
            return context.retryCount + 1;
          }
          return context.retryCount;
        }
      }),
      // Clear error
      clearError: assign({
        errorMessage: null,
        retryCount: 0
      }),
      // Log why we're going idle (before reset clears it)
      logIdleReason: ({ context, event }) => {
        logger$1.log("Entering idle state", {
          event: event.type,
          completionReason: context.completionReason,
          errorMessage: context.errorMessage,
          currentMissionId: context.currentMissionId
        });
      },
      // Reset context on stop
      resetContext: assign({
        currentMissionId: null,
        currentMissionPermalink: null,
        errorMessage: null,
        retryCount: 0,
        findMissionRetryCount: 0,
        completionReason: null
      }),
      // Increment find mission retry count
      incrementFindMissionRetry: assign({
        findMissionRetryCount: ({ context }) => context.findMissionRetryCount + 1
      }),
      // Reset find mission retry count
      resetFindMissionRetry: assign({
        findMissionRetryCount: 0
      }),
      // Set completion reason
      setCompletionReason: assign({
        completionReason: ({ event }) => {
          let reason = null;
          if (event.type === "STOP_BOT") {
            reason = "stopped";
          } else if (event.type === "NO_MISSIONS_FOUND") {
            reason = "no_missions";
          } else if (event.type === "ERROR_OCCURRED") {
            reason = "error";
          }
          logger$1.log("Setting completion reason", {
            event: event.type,
            reason
          });
          return reason;
        }
      }),
      // Log error state entry
      logError: ({ context, event }) => {
        logger$1.error("Entered error state", {
          event: event.type,
          errorMessage: context.errorMessage,
          retryCount: context.retryCount,
          currentMissionId: context.currentMissionId
        });
      },
      // Log state transitions
      logTransition: ({ context, event }) => {
        logger$1.log(`Transition: ${event.type}`, {
          event,
          context
        });
      }
    }
  }).createMachine({
    id: "bot",
    description: "Overall bot state machine",
    initial: "idle",
    context: {
      currentMissionId: null,
      currentMissionPermalink: null,
      errorMessage: null,
      retryCount: 0,
      findMissionRetryCount: 0,
      completionReason: null
    },
    states: {
      // ========================================================================
      // IDLE: Bot is stopped, no automation
      // Sub-states track whether game dialog is open
      // ========================================================================
      idle: {
        description: "Bot is stopped, no automation",
        initial: "stopped",
        entry: ["logIdleReason", "resetContext"],
        states: {
          stopped: {
            description: "Normal idle state - no game dialog open",
            on: {
              START_BOT: {
                // Normal flow: find mission and open game
                target: "#bot.starting"
              }
            }
          },
          dialogOpen: {
            description: "Game dialog is open and automation engine ready",
            on: {
              START_BOT: {
                // Dialog already open, send START_MISSION_AUTOMATION via gameReady
                target: "#bot.gameMission.gameReady",
                actions: ["logTransition"]
              }
            }
          }
        },
        on: {
          AUTOMATION_READY: {
            // When iframe loads while idle, go to dialogOpen sub-state
            target: ".dialogOpen",
            actions: ["logTransition"]
          }
        }
      },
      // ========================================================================
      // STARTING: User clicked Start, preparing to find/open mission
      // ========================================================================
      starting: {
        entry: ["logTransition"],
        after: {
          [STATE_TIMEOUT]: {
            target: "error",
            actions: assign({
              errorMessage: `Timeout in starting state - no mission found within ${STATE_TIMEOUT / 1e3} seconds`,
              completionReason: "error"
            })
          }
        },
        on: {
          STOP_BOT: {
            target: "idle",
            actions: ["setCompletionReason"]
          },
          MISSION_PAGE_LOADED: {
            target: "gameMission.waitingForGame",
            actions: ["setMission"]
          },
          NAVIGATE_TO_MISSION: {
            target: "navigating",
            actions: ["setMission"]
          },
          NO_MISSIONS_FOUND: {
            target: "idle",
            actions: ["setCompletionReason"]
          },
          ERROR_OCCURRED: {
            target: "error",
            actions: ["setError"]
          }
        }
      },
      // ========================================================================
      // NAVIGATING: Navigating to a mission page
      // ========================================================================
      navigating: {
        description: "Navigating to a mission page",
        entry: ["logTransition"],
        on: {
          STOP_BOT: {
            target: "idle",
            actions: ["setCompletionReason"]
          },
          MISSION_PAGE_LOADED: {
            target: "gameMission.waitingForGame"
          },
          MISSION_DELETED: { target: "starting", actions: ["logTransition"] },
          ERROR_OCCURRED: {
            description: "Error occurred while navigating to a mission page",
            target: "error",
            actions: ["setError", "setCompletionReason"]
          }
        }
      },
      // ========================================================================
      // gameMission: Nested mission subflow
      // ========================================================================
      gameMission: {
        initial: "waitingForGame",
        states: {
          waitingForGame: {
            entry: ["logTransition"],
            on: {
              GAME_LOADER_DETECTED: { target: "openingGame", actions: ["logTransition"] },
              AUTOMATION_READY: {
                // Iframe loaded while waiting for game - skip to gameReady
                target: "gameReady",
                actions: ["logTransition"]
              },
              MISSION_DELETED: { target: "completing", actions: ["logTransition"] },
              ERROR_OCCURRED: {
                target: "#bot.error",
                actions: ["setError", "setCompletionReason"]
              },
              STOP_BOT: { target: "#bot.idle", actions: ["setCompletionReason"] }
            }
          },
          openingGame: {
            entry: ["logTransition"],
            on: {
              GAME_DIALOG_OPENED: { target: "gameReady", actions: ["logTransition"] },
              AUTOMATION_READY: {
                // Iframe loaded while opening game - skip to gameReady
                target: "gameReady",
                actions: ["logTransition"]
              },
              ERROR_OCCURRED: {
                target: "#bot.error",
                actions: ["setError", "setCompletionReason"]
              },
              STOP_BOT: { target: "#bot.idle", actions: ["setCompletionReason"] }
            }
          },
          gameReady: {
            entry: ["logTransition"],
            on: {
              AUTOMATION_STARTED: { target: "running", actions: ["logTransition"] },
              ERROR_OCCURRED: {
                target: "#bot.error",
                actions: ["setError", "setCompletionReason"]
              },
              STOP_BOT: { target: "#bot.idle", actions: ["setCompletionReason"] }
            }
          },
          running: {
            entry: ["logTransition"],
            on: {
              MISSION_COMPLETED: { target: "completing", actions: ["logTransition"] },
              ERROR_OCCURRED: {
                target: "#bot.error",
                actions: ["setError", "setCompletionReason"]
              },
              STOP_BOT: { target: "#bot.idle", actions: ["setCompletionReason"] }
            }
          },
          completing: {
            entry: ["logTransition", "resetFindMissionRetry"],
            on: {
              NEXT_MISSION_FOUND: {
                target: "waitingForDialogClose",
                actions: ["setMission", "clearError", "resetFindMissionRetry"]
              },
              NO_MISSIONS_FOUND: [
                {
                  // If we've retried less than 3 times, increment retry count
                  // Internal transition (no target) - stays in completing state without re-entering
                  guard: ({ context }) => context.findMissionRetryCount < 3,
                  actions: ["incrementFindMissionRetry"]
                },
                {
                  // After 3 retries, give up and go idle
                  target: "#bot.idle",
                  actions: ["setCompletionReason"]
                }
              ],
              ERROR_OCCURRED: [
                {
                  // If we've retried less than 3 times, increment retry count
                  // Internal transition (no target) - stays in completing state without re-entering
                  guard: ({ context }) => context.findMissionRetryCount < 3,
                  actions: ["incrementFindMissionRetry", "setError"]
                },
                {
                  // After 3 retries, go to error state
                  target: "#bot.error",
                  actions: ["setError", "setCompletionReason"]
                }
              ]
            }
          },
          waitingForDialogClose: {
            description: "Waiting for game dialog to close before navigating to next mission",
            entry: ["logTransition"],
            on: {
              GAME_DIALOG_CLOSED: {
                target: "#bot.navigating"
              },
              ERROR_OCCURRED: {
                target: "#bot.error",
                actions: ["setError", "setCompletionReason"]
              },
              STOP_BOT: {
                target: "#bot.idle",
                actions: ["setCompletionReason"]
              }
            }
          }
        }
      },
      // ========================================================================
      // ERROR: Something went wrong, need user intervention or retry
      // ========================================================================
      error: {
        entry: ["logError"],
        on: {
          STOP_BOT: {
            target: "idle",
            actions: ["clearError", "setCompletionReason"]
          },
          RETRY: {
            target: "starting",
            actions: ["clearError"]
          }
        }
      }
    }
  });
  const STORAGE_KEYS = {
    MISSIONS: "missions",
    // Mission data (static, from database)
    USER_PROGRESS: "userProgress"
  };
  const CACHE_DURATION = 5 * 60 * 1e3;
  const STORAGE_KEY = "redditUserCache";
  async function getCurrentRedditUser() {
    const cachedUser = await getCachedUserIncludingExpired();
    const isCacheFresh = cachedUser ? Date.now() - cachedUser.timestamp < CACHE_DURATION : false;
    if (isCacheFresh && cachedUser) {
      return cachedUser.username;
    }
    try {
      const response = await fetch("https://www.reddit.com/api/me.json", {
        credentials: "include"
      });
      if (!response.ok) {
        if (cachedUser) {
          return cachedUser.username;
        }
        return "default";
      }
      const data = await response.json();
      const username = data?.data?.name;
      if (username && typeof username === "string") {
        await cacheUser(username);
        return username;
      } else {
        if (cachedUser) {
          return cachedUser.username;
        }
        return "default";
      }
    } catch (error) {
      if (cachedUser) {
        return cachedUser.username;
      }
      return "default";
    }
  }
  async function getCachedUserIncludingExpired() {
    return new Promise((resolve) => {
      chrome.storage.local.get([STORAGE_KEY], (result2) => {
        if (chrome.runtime.lastError) {
          resolve(null);
          return;
        }
        const cache2 = result2[STORAGE_KEY];
        resolve(cache2 || null);
      });
    });
  }
  async function cacheUser(username) {
    return new Promise((resolve, reject) => {
      const cache2 = {
        username,
        timestamp: Date.now()
      };
      chrome.storage.local.set({ [STORAGE_KEY]: cache2 }, () => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve();
        }
      });
    });
  }
  function createEmptyProgressData() {
    return {
      cleared: [],
      disabled: [],
      clearedAt: {},
      loot: {}
    };
  }
  async function getMultiUserProgress() {
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.USER_PROGRESS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve(result2[STORAGE_KEYS.USER_PROGRESS] || {});
        }
      });
    });
  }
  async function setMultiUserProgress(data) {
    return new Promise((resolve, reject) => {
      chrome.storage.local.set({ [STORAGE_KEYS.USER_PROGRESS]: data }, () => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve();
        }
      });
    });
  }
  async function getAllUserProgress() {
    const username = await getCurrentRedditUser();
    const multiUserData = await getMultiUserProgress();
    return multiUserData[username] || createEmptyProgressData();
  }
  async function markMissionCleared$1(postId) {
    const username = await getCurrentRedditUser();
    const multiUserData = await getMultiUserProgress();
    const userProgress = multiUserData[username] || createEmptyProgressData();
    if (!userProgress.cleared.includes(postId)) {
      userProgress.cleared.push(postId);
    }
    userProgress.clearedAt[postId] = {
      timestamp: Date.now(),
      duration: void 0
    };
    multiUserData[username] = userProgress;
    await setMultiUserProgress(multiUserData);
  }
  async function setMissionDisabled$1(postId, disabled) {
    const username = await getCurrentRedditUser();
    const multiUserData = await getMultiUserProgress();
    const userProgress = multiUserData[username] || createEmptyProgressData();
    {
      if (!userProgress.disabled.includes(postId)) {
        userProgress.disabled.push(postId);
      }
    }
    multiUserData[username] = userProgress;
    await setMultiUserProgress(multiUserData);
  }
  function isLegacyFormat(record) {
    return record && typeof record === "object" && "metadata" in record && record.metadata !== void 0;
  }
  function migrateLegacyRecord(legacy) {
    const mission = legacy.metadata?.mission;
    const record = {
      // Core identification
      postId: legacy.postId,
      timestamp: legacy.timestamp,
      permalink: legacy.permalink,
      // Mission metadata
      missionTitle: legacy.metadata?.missionTitle || legacy.missionTitle || `Mission ${legacy.postId.slice(3)}`,
      missionAuthorName: legacy.metadata?.missionAuthorName || "Unknown",
      // Mission data (from nested mission object or top-level fields)
      environment: mission?.environment || legacy.environment || "haunted_forest",
      encounters: mission?.encounters || [],
      minLevel: mission?.minLevel || legacy.minLevel || 1,
      maxLevel: mission?.maxLevel || legacy.maxLevel || 340,
      difficulty: mission?.difficulty || legacy.difficulty || 0,
      foodImage: mission?.foodImage || "",
      foodName: mission?.foodName || legacy.foodName || "",
      authorWeaponId: mission?.authorWeaponId || "",
      chef: mission?.chef || "",
      cart: mission?.cart || "",
      rarity: mission?.rarity || "common",
      type: mission?.type
    };
    return record;
  }
  function normalizeMissionRecord(record) {
    if (isLegacyFormat(record)) {
      return migrateLegacyRecord(record);
    }
    return record;
  }
  async function getAllMissions() {
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          const rawMissions = result2[STORAGE_KEYS.MISSIONS] || {};
          const migratedMissions = {};
          for (const postId in rawMissions) {
            migratedMissions[postId] = normalizeMissionRecord(rawMissions[postId]);
          }
          resolve(migratedMissions);
        }
      });
    });
  }
  async function markMissionCleared(postId) {
    return markMissionCleared$1(postId);
  }
  async function setMissionDisabled(postId, disabled) {
    return setMissionDisabled$1(postId);
  }
  async function getFilteredUnclearedMissions(filters) {
    const [missions, progress] = await Promise.all([getAllMissions(), getAllUserProgress()]);
    const allStarsSelected = !filters?.stars || filters.stars.length === 5 && [1, 2, 3, 4, 5].every((d) => filters.stars.includes(d));
    let unclearedMissions = Object.values(missions).filter(
      (m) => !progress.cleared.includes(m.postId) && !progress.disabled.includes(m.postId) && m.minLevel !== void 0 && m.maxLevel !== void 0 && // If all stars selected or no star filter, include missions with null difficulty
      (allStarsSelected || (m.difficulty ?? 0) > 0)
    );
    if (filters) {
      unclearedMissions = unclearedMissions.filter((m) => {
        if (filters.stars && filters.stars.length > 0) {
          if (allStarsSelected) ;
          else if (!filters.stars.includes(m.difficulty || 0)) {
            return false;
          }
        }
        if (filters.minLevel !== void 0) {
          if (m.minLevel < filters.minLevel) {
            return false;
          }
        }
        if (filters.maxLevel !== void 0) {
          if (m.maxLevel > filters.maxLevel) {
            return false;
          }
        }
        return true;
      });
    }
    unclearedMissions.sort((a, b) => (b.timestamp || 0) - (a.timestamp || 0));
    return unclearedMissions;
  }
  async function getNextUnclearedMission(filters) {
    const unclearedMissions = await getFilteredUnclearedMissions(filters);
    console.log("[getNextUnclearedMission] Found uncleared missions", {
      count: unclearedMissions.length,
      excludePostIds: filters?.excludePostIds,
      firstFew: unclearedMissions.slice(0, 3).map((m) => ({
        postId: m.postId,
        title: m.missionTitle?.substring(0, 30)
      }))
    });
    const filteredMissions = filters?.excludePostIds ? unclearedMissions.filter((m) => !filters.excludePostIds.includes(m.postId)) : unclearedMissions;
    console.log("[getNextUnclearedMission] After exclusion", {
      count: filteredMissions.length,
      nextMission: filteredMissions[0] ? {
        postId: filteredMissions[0].postId,
        title: filteredMissions[0].missionTitle?.substring(0, 30)
      } : null
    });
    return filteredMissions[0] || null;
  }
  async function getCachedUsernameForMigration() {
    return new Promise((resolve) => {
      chrome.storage.local.get(["redditUserCache"], (result2) => {
        if (chrome.runtime.lastError || !result2.redditUserCache) {
          resolve("default");
          return;
        }
        const cache2 = result2.redditUserCache;
        resolve(cache2.username || "default");
      });
    });
  }
  async function requestUsernameFromRedditTabs() {
    return new Promise((resolve) => {
      chrome.tabs.query({ url: "*://*.reddit.com/*" }, (tabs) => {
        if (chrome.runtime.lastError || !tabs.length) {
          console.log("[Migration] No Reddit tabs found to fetch username from");
          resolve();
          return;
        }
        console.log("[Migration] Found", tabs.length, "Reddit tabs, requesting username...");
        chrome.tabs.sendMessage(tabs[0].id, { type: "FETCH_REDDIT_USERNAME" }, (response) => {
          if (chrome.runtime.lastError) {
            console.log(
              "[Migration] Could not communicate with Reddit tab:",
              chrome.runtime.lastError.message
            );
          } else {
            console.log("[Migration] Username fetch requested from Reddit tab");
          }
          setTimeout(resolve, 500);
        });
      });
    });
  }
  async function migrateToSeparateProgress() {
    await requestUsernameFromRedditTabs();
    const username = await getCachedUsernameForMigration();
    console.log("[Migration] Migrating progress to username:", username);
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS, STORAGE_KEYS.USER_PROGRESS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        const oldMissions = result2[STORAGE_KEYS.MISSIONS] || {};
        const existingUserProgress = result2[STORAGE_KEYS.USER_PROGRESS] || {};
        function isUserProgressData(val) {
          return val && typeof val === "object" && (Array.isArray(val.cleared) || Array.isArray(val.disabled) || typeof val.clearedAt === "object" && val.clearedAt !== null || typeof val.loot === "object" && val.loot !== null);
        }
        const isAlreadyMultiUser = Object.values(existingUserProgress).some(isUserProgressData);
        let existingMultiUserProgress;
        if (!isAlreadyMultiUser && Object.keys(existingUserProgress).length > 0) {
          console.log("[Migration] Detected flat progress format, converting to array-based format");
          const convertedUserProgress = {
            cleared: [],
            disabled: [],
            clearedAt: {},
            loot: {}
          };
          for (const [postId, entry] of Object.entries(existingUserProgress)) {
            const progressEntry = entry;
            if (progressEntry.cleared) {
              convertedUserProgress.cleared.push(postId);
            }
            if (progressEntry.disabled) {
              convertedUserProgress.disabled.push(postId);
            }
            if (progressEntry.clearedAt !== void 0) {
              convertedUserProgress.clearedAt[postId] = progressEntry.clearedAt;
            }
            if (progressEntry.totalLoot !== void 0 && progressEntry.totalLoot.length > 0) {
              convertedUserProgress.loot[postId] = progressEntry.totalLoot;
            }
          }
          existingMultiUserProgress = {
            [username]: convertedUserProgress
          };
        } else {
          existingMultiUserProgress = existingUserProgress;
        }
        const cleanedMissions = {};
        const multiUserProgress = { ...existingMultiUserProgress };
        const userProgress = multiUserProgress[username] || {
          cleared: [],
          disabled: [],
          clearedAt: {},
          loot: {}
        };
        let migrated = 0;
        let skipped = 0;
        for (const postId in oldMissions) {
          const old = oldMissions[postId];
          const hasProgress = old.cleared !== void 0 || old.clearedAt !== void 0 || old.disabled !== void 0 || old.totalLoot !== void 0;
          if (hasProgress) {
            if (old.cleared && !userProgress.cleared.includes(postId)) {
              userProgress.cleared.push(postId);
            }
            if (old.disabled && !userProgress.disabled.includes(postId)) {
              userProgress.disabled.push(postId);
            }
            if (old.clearedAt !== void 0) {
              userProgress.clearedAt[postId] = old.clearedAt;
            }
            if (old.totalLoot !== void 0 && old.totalLoot.length > 0) {
              userProgress.loot[postId] = old.totalLoot;
            }
            migrated++;
          } else {
            skipped++;
          }
          const { cleared, clearedAt, disabled, totalLoot, ...cleanMission } = old;
          cleanedMissions[postId] = cleanMission;
        }
        multiUserProgress[username] = userProgress;
        chrome.storage.local.set(
          {
            [STORAGE_KEYS.MISSIONS]: cleanedMissions,
            [STORAGE_KEYS.USER_PROGRESS]: multiUserProgress
          },
          () => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              console.log("[Migration] Successfully migrated:", { migrated, skipped, username });
              resolve({ migrated, skipped });
            }
          }
        );
      });
    });
  }
  async function needsMigration() {
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS, STORAGE_KEYS.USER_PROGRESS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        const missions = result2[STORAGE_KEYS.MISSIONS] || {};
        const userProgress = result2[STORAGE_KEYS.USER_PROGRESS] || {};
        if (Object.keys(userProgress).length === 0 && Object.keys(missions).length > 0) {
          const hasProgressFields = Object.values(missions).some(
            (m) => m.cleared !== void 0 || m.clearedAt !== void 0 || m.disabled !== void 0 || m.totalLoot !== void 0
          );
          resolve(hasProgressFields);
        } else {
          resolve(false);
        }
      });
    });
  }
  async function migrateMissionsStorage() {
    return new Promise((resolve, reject) => {
      chrome.storage.local.get([STORAGE_KEYS.MISSIONS], (result2) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }
        const rawMissions = result2[STORAGE_KEYS.MISSIONS] || {};
        const postIds = Object.keys(rawMissions);
        const migrationResult = {
          total: postIds.length,
          migrated: 0,
          alreadyFlat: 0,
          errors: []
        };
        if (postIds.length === 0) {
          resolve(migrationResult);
          return;
        }
        const migratedMissions = {};
        for (const postId of postIds) {
          try {
            const mission = rawMissions[postId];
            if (isLegacyFormat(mission)) {
              const migrated = normalizeMissionRecord(mission);
              if (migrated.authorWeaponId === "") {
                delete migrated.authorWeaponId;
              }
              if (migrated.chef === "") {
                delete migrated.chef;
              }
              if (migrated.cart === "") {
                delete migrated.cart;
              }
              if (!migrated.timestamp || migrated.timestamp === 0) {
                delete migrated.timestamp;
              }
              migratedMissions[postId] = migrated;
              migrationResult.migrated++;
            } else {
              migratedMissions[postId] = mission;
              migrationResult.alreadyFlat++;
            }
          } catch (error) {
            migrationResult.errors.push(`${postId}: ${error instanceof Error ? error.message : String(error)}`);
            migratedMissions[postId] = rawMissions[postId];
          }
        }
        chrome.storage.local.set({ [STORAGE_KEYS.MISSIONS]: migratedMissions }, () => {
          if (chrome.runtime.lastError) {
            reject(chrome.runtime.lastError);
          } else {
            chrome.runtime.sendMessage({
              type: "MISSIONS_UPDATED"
            }).catch(() => {
            });
            resolve(migrationResult);
          }
        });
      });
    });
  }
  const definition = defineBackground(() => {
    extensionLogger.info("Starting up", {
      timestamp: (/* @__PURE__ */ new Date()).toISOString(),
      version: chrome.runtime.getManifest().version
    });
    let botActor = null;
    const gamePreviewReloadAttempts = /* @__PURE__ */ new Map();
    let gameFrameId = void 0;
    let gameTabId = void 0;
    function initializeStateMachine() {
      if (botActor) {
        extensionLogger.log("[StateMachine] Stopping existing actor before creating new one");
        try {
          botActor.stop();
        } catch (error) {
          extensionLogger.warn("[StateMachine] Error stopping existing actor", {
            error: String(error)
          });
        }
        botActor = null;
      }
      botActor = createActor(botMachine);
      botActor.start();
      extensionLogger.log("[StateMachine] Actor started in service worker", {
        initialState: botActor.getSnapshot().value
      });
      subscribeToStateChanges();
    }
    function getPresentationStateName(stateObj) {
      try {
        if (typeof stateObj?.value === "string") return stateObj.value;
        if (stateObj?.value && typeof stateObj.value === "object") {
          if (stateObj.value.gameMission && typeof stateObj.value.gameMission === "string") {
            return String(stateObj.value.gameMission);
          }
          if (stateObj.value.idle && typeof stateObj.value.idle === "string") {
            return stateObj.value.idle === "dialogOpen" ? "idleDialogOpen" : "idle";
          }
        }
        if (stateObj?.matches) {
          if (stateObj.matches("idle.stopped")) return "idle";
          if (stateObj.matches("idle.dialogOpen")) return "idleDialogOpen";
          if (stateObj.matches("gameMission.waitingForGame")) return "waitingForGame";
          if (stateObj.matches("gameMission.openingGame")) return "openingGame";
          if (stateObj.matches("gameMission.gameReady")) return "gameReady";
          if (stateObj.matches("gameMission.running")) return "running";
          if (stateObj.matches("gameMission.completing")) return "completing";
          if (stateObj.matches("gameMission.waitingForDialogClose")) return "waitingForDialogClose";
        }
      } catch {
      }
      return String(stateObj?.value ?? "unknown");
    }
    function subscribeToStateChanges() {
      botActor.subscribe((state) => {
        const context = state.context;
        const presentationState = getPresentationStateName(state);
        extensionLogger.log("[StateMachine] State changed", {
          state: presentationState,
          context
        });
        chrome.tabs.query({}, (tabs) => {
          tabs.forEach((tab) => {
            if (tab.id && tab.url?.includes("reddit.com")) {
              chrome.tabs.sendMessage(tab.id, {
                type: "STATE_CHANGED",
                state: presentationState,
                context
              });
            }
          });
        });
        handleStateTransition(state, context);
      });
    }
    initializeStateMachine();
    function sendToStateMachine(event) {
      if (!botActor) {
        extensionLogger.error("[StateMachine] Actor not initialized, cannot send event", { event });
        return false;
      }
      botActor.send(event);
      return true;
    }
    function getStateMachineSnapshot() {
      if (!botActor) {
        extensionLogger.warn("[StateMachine] Actor not initialized, returning null snapshot");
        return null;
      }
      return botActor.getSnapshot();
    }
    async function canNavigateAway() {
      return new Promise((resolve) => {
        chrome.tabs.query({ url: "https://www.reddit.com/*" }, (tabs) => {
          if (tabs.length === 0 || !tabs[0].id) {
            extensionLogger.log("[canNavigateAway] No Reddit tab found, safe to navigate");
            resolve(true);
            return;
          }
          chrome.tabs.sendMessage(
            tabs[0].id,
            { type: "CHECK_GAME_DIALOG_STATUS" },
            (response) => {
              if (chrome.runtime.lastError) {
                extensionLogger.warn("[canNavigateAway] Error checking dialog status", {
                  error: chrome.runtime.lastError.message
                });
                resolve(true);
                return;
              }
              const isDialogOpen = response?.isOpen || false;
              extensionLogger.log("[canNavigateAway] Dialog status check result", {
                isDialogOpen,
                canNavigate: !isDialogOpen
              });
              resolve(!isDialogOpen);
            }
          );
        });
      });
    }
    let lastCompletingRetryCount = -1;
    let idleStateCheckInterval = null;
    async function findAndSendNextMission() {
      try {
        const result2 = await chrome.storage.local.get(["automationFilters"]);
        const filters = result2.automationFilters;
        const snapshot = getStateMachineSnapshot();
        const currentMissionId = snapshot?.context?.currentMissionId;
        extensionLogger.log("[findAndSendNextMission] Searching for next mission", {
          filters,
          excludingCurrentMission: currentMissionId
        });
        const mission = await getNextUnclearedMission({
          stars: filters.stars,
          minLevel: filters.minLevel,
          maxLevel: filters.maxLevel,
          excludePostIds: currentMissionId ? [currentMissionId] : void 0
        });
        if (mission && mission.postId && mission.permalink) {
          extensionLogger.log("[findAndSendNextMission] Found mission", {
            missionId: mission.postId,
            permalink: mission.permalink
          });
          extensionLogger.log("[findAndSendNextMission] Getting presentation state name...");
          const currentState = getPresentationStateName(snapshot);
          extensionLogger.log("[findAndSendNextMission] Current state", { currentState });
          if (currentState === "completing") {
            extensionLogger.log(
              "[findAndSendNextMission] In completing state, sending NEXT_MISSION_FOUND"
            );
            sendToStateMachine({
              type: "NEXT_MISSION_FOUND",
              missionId: mission.postId,
              permalink: mission.permalink
            });
          } else {
            let eventSent = false;
            const timeoutId = setTimeout(() => {
              if (!eventSent) {
                extensionLogger.warn(
                  "[findAndSendNextMission] Tab query timeout, defaulting to NAVIGATE_TO_MISSION"
                );
                eventSent = true;
                sendToStateMachine({
                  type: "NAVIGATE_TO_MISSION",
                  missionId: mission.postId,
                  permalink: mission.permalink
                });
              }
            }, 1e3);
            chrome.tabs.query({ url: "https://www.reddit.com/*" }, (tabs) => {
              if (eventSent) return;
              clearTimeout(timeoutId);
              eventSent = true;
              const currentUrl = tabs[0]?.url || "";
              const isOnMissionPage = currentUrl.includes(mission.postId);
              extensionLogger.log("[findAndSendNextMission] Tab check", {
                currentUrl,
                missionPostId: mission.postId,
                isOnMissionPage,
                tabsFound: tabs.length
              });
              if (isOnMissionPage) {
                extensionLogger.log(
                  "[findAndSendNextMission] Already on mission page, sending MISSION_PAGE_LOADED"
                );
                sendToStateMachine({
                  type: "MISSION_PAGE_LOADED",
                  missionId: mission.postId,
                  permalink: mission.permalink
                });
              } else {
                extensionLogger.log(
                  "[findAndSendNextMission] Need to navigate, sending NAVIGATE_TO_MISSION"
                );
                sendToStateMachine({
                  type: "NAVIGATE_TO_MISSION",
                  missionId: mission.postId,
                  permalink: mission.permalink
                });
              }
            });
          }
        } else {
          extensionLogger.log("[findAndSendNextMission] No missions found");
          sendToStateMachine({ type: "NO_MISSIONS_FOUND" });
        }
      } catch (error) {
        extensionLogger.error("[findAndSendNextMission] Error finding mission", {
          error: String(error)
        });
        sendToStateMachine({
          type: "ERROR_OCCURRED",
          message: "Failed to find next mission: " + String(error)
        });
      }
    }
    function handleStateTransition(stateObj, context) {
      const presentationState = getPresentationStateName(stateObj);
      extensionLogger.log("[StateTransition] Entered state", { state: presentationState });
      if (presentationState !== "idle" && presentationState !== "idleDialogOpen" && idleStateCheckInterval) {
        extensionLogger.log("[StateTransition] Clearing idle state check interval");
        clearInterval(idleStateCheckInterval);
        idleStateCheckInterval = null;
      }
      if (stateObj?.matches) {
        if (stateObj.matches("gameMission.waitingForGame")) {
          broadcastToReddit({ type: "CHECK_FOR_GAME_LOADER" });
          return;
        }
        if (stateObj.matches("gameMission.openingGame")) {
          broadcastToReddit({ type: "CLICK_GAME_UI" });
          return;
        }
        if (stateObj.matches("gameMission.gameReady")) {
          broadcastToAllFrames({ type: "START_MISSION_AUTOMATION" });
          return;
        }
        if (stateObj.matches("gameMission.running")) {
          extensionLogger.log("[StateTransition] Entered running state");
          return;
        }
        if (stateObj.matches("gameMission.completing")) {
          const retryCount = context.findMissionRetryCount || 0;
          if (retryCount === lastCompletingRetryCount) {
            extensionLogger.log("[StateTransition] Skipping duplicate FIND_NEXT_MISSION", {
              retryCount,
              lastRetryCount: lastCompletingRetryCount
            });
            return;
          }
          lastCompletingRetryCount = retryCount;
          const delayMs = retryCount > 0 ? INITIAL_RETRY_DELAY * RETRY_BACKOFF_BASE ** (retryCount - 1) : STORAGE_PROPAGATION_DELAY;
          extensionLogger.log("[StateTransition] Finding next mission", {
            retryCount,
            delayMs
          });
          setTimeout(() => {
            findAndSendNextMission();
          }, delayMs);
          return;
        }
        if (stateObj.matches("gameMission.waitingForDialogClose")) {
          lastCompletingRetryCount = -1;
          extensionLogger.log("[StateTransition] Checking if game dialog is closed");
          canNavigateAway().then((canNavigate) => {
            if (canNavigate) {
              extensionLogger.log(
                "[StateTransition] Dialog is closed, sending GAME_DIALOG_CLOSED event"
              );
              sendToStateMachine({ type: "GAME_DIALOG_CLOSED" });
            } else {
              extensionLogger.log("[StateTransition] Dialog still open, will retry check in 2s");
              setTimeout(() => {
                const snapshot = getStateMachineSnapshot();
                if (snapshot && snapshot.matches && snapshot.matches("gameMission.waitingForDialogClose")) {
                  extensionLogger.log("[StateTransition] Re-checking dialog status");
                  handleStateTransition(snapshot, snapshot.context);
                }
              }, 2e3);
            }
          });
          return;
        }
      }
      switch (presentationState) {
        case "idle":
          lastCompletingRetryCount = -1;
          if (!idleStateCheckInterval) {
            extensionLogger.log("[StateTransition] Starting idle state check interval (every 5s)");
            idleStateCheckInterval = setInterval(() => {
              extensionLogger.log("[IdleCheck] Checking if automation is ready");
              checkAutomationReady().then((isReady) => {
                if (isReady) {
                  extensionLogger.log("[IdleCheck] Automation ready, sending AUTOMATION_READY");
                  sendToStateMachine({ type: "AUTOMATION_READY" });
                }
              });
            }, 5e3);
          }
          break;
        case "error":
          lastCompletingRetryCount = -1;
          break;
        case "navigating":
          extensionLogger.log("[StateTransition] Navigating state, checking permalink", {
            hasPermalink: !!context.currentMissionPermalink,
            permalink: context.currentMissionPermalink,
            fullContext: context
          });
          if (context.currentMissionPermalink) {
            chrome.tabs.query({ url: "https://www.reddit.com/*" }, (tabs) => {
              if (tabs.length > 0 && tabs[0].id) {
                extensionLogger.log("[StateTransition] Navigating tab to mission", {
                  tabId: tabs[0].id,
                  url: context.currentMissionPermalink
                });
                chrome.tabs.update(tabs[0].id, {
                  url: context.currentMissionPermalink
                });
              } else {
                extensionLogger.error("[StateTransition] No Reddit tab found!");
              }
            });
          } else {
            extensionLogger.error("[StateTransition] No permalink set! Cannot navigate!", { context });
          }
          break;
      }
    }
    function broadcastToReddit(message) {
      chrome.tabs.query({}, (tabs) => {
        tabs.forEach((tab) => {
          if (tab.id && tab.url?.includes("reddit.com")) {
            chrome.tabs.sendMessage(tab.id, message, { frameId: 0 });
          }
        });
      });
    }
    function broadcastToAllFrames(message) {
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs[0]?.id) {
          chrome.tabs.sendMessage(tabs[0].id, message, { frameId: void 0 });
        }
      });
    }
    function sendToGameFrame(message) {
      return new Promise((resolve, reject) => {
        if (gameFrameId === void 0 || gameTabId === void 0) {
          reject(new Error("Game frame not tracked yet"));
          return;
        }
        chrome.tabs.sendMessage(
          gameTabId,
          message,
          { frameId: gameFrameId },
          (response) => {
            if (chrome.runtime.lastError) {
              reject(new Error(chrome.runtime.lastError.message));
            } else {
              resolve(response);
            }
          }
        );
      });
    }
    async function checkAutomationReady() {
      try {
        const response = await sendToGameFrame({ type: "CHECK_AUTOMATION_STATUS" });
        const isReady = response?.isReady || false;
        extensionLogger.log("[checkAutomationReady] Automation status check result", {
          isReady,
          state: response?.state
        });
        return isReady;
      } catch (error) {
        extensionLogger.log("[checkAutomationReady] Error or game frame not tracked", {
          error: String(error)
        });
        return false;
      }
    }
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      extensionLogger.log("Received message", {
        type: message.type,
        source: sender.tab ? `tab ${sender.tab.id}` : "extension",
        frameId: sender.frameId
      });
      (async () => {
        try {
          await handleMessage(message, sender, sendResponse);
        } catch (error) {
          extensionLogger.error("Error handling message", {
            type: message.type,
            error: error instanceof Error ? error.message : String(error)
          });
          sendResponse({ success: false, error: String(error) });
        }
      })();
      return true;
    });
    async function handleMessage(message, sender, sendResponse) {
      switch (message.type) {
        // Messages from popup - route to state machine
        case "START_BOT":
          extensionLogger.log("START_BOT received, sending to state machine");
          if (!botActor) {
            extensionLogger.error("State machine not initialized yet!");
            sendResponse({ error: "State machine not ready" });
            break;
          }
          checkAutomationReady().then((isReady) => {
            if (isReady) {
              extensionLogger.log("[START_BOT] Automation already ready, sending AUTOMATION_READY");
              sendToStateMachine({ type: "AUTOMATION_READY" });
            }
            return getCurrentRedditUser();
          }).then((username) => {
            extensionLogger.log("[UserDetection] Current user:", username);
            chrome.storage.local.get(["automationConfig", "automationFilters"], (result2) => {
              const filters = result2.automationFilters;
              extensionLogger.log("START_BOT: Sending START_BOT event to state machine");
              sendToStateMachine({
                type: "START_BOT"
              });
              chrome.storage.local.set({
                activeBotSession: true,
                automationConfig: result2.automationConfig || {},
                automationFilters: filters
              });
              extensionLogger.log("START_BOT: Calling findAndSendNextMission");
              findAndSendNextMission();
            });
          }).catch((error) => {
            extensionLogger.error("[UserDetection] Failed to detect user on START_BOT:", error);
            chrome.storage.local.get(["automationConfig", "automationFilters"], (result2) => {
              const filters = result2.automationFilters;
              sendToStateMachine({ type: "START_BOT" });
              chrome.storage.local.set({
                activeBotSession: true,
                automationConfig: result2.automationConfig || {},
                automationFilters: filters
              });
              findAndSendNextMission();
            });
          });
          sendResponse({ success: true });
          break;
        case "STOP_BOT":
          extensionLogger.log("STOP_BOT received, sending to state machine");
          sendToStateMachine({ type: "STOP_BOT" });
          chrome.storage.local.remove(["activeBotSession"]);
          gamePreviewReloadAttempts.clear();
          broadcastToAllFrames({ type: "STOP_MISSION_AUTOMATION" });
          sendResponse({ success: true });
          break;
        // Navigate to mission - route to reddit-content
        case "NAVIGATE_TO_MISSION":
          extensionLogger.log("Forwarding NAVIGATE_TO_MISSION to reddit-content");
          chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            if (tabs[0]?.id) {
              chrome.tabs.sendMessage(
                tabs[0].id,
                {
                  type: "NAVIGATE_TO_MISSION",
                  filters: message.filters
                },
                (response) => {
                  sendResponse(response);
                }
              );
            } else {
              sendResponse({ error: "No active tab" });
            }
          });
          return true;
        // Will respond asynchronously
        // Open mission iframe - route to reddit-content
        case "OPEN_MISSION_IFRAME":
          extensionLogger.log("Forwarding OPEN_MISSION_IFRAME to reddit-content");
          chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            if (tabs[0]?.id) {
              chrome.tabs.sendMessage(
                tabs[0].id,
                {
                  type: "OPEN_MISSION_IFRAME"
                },
                (response) => {
                  sendResponse(response);
                }
              );
            } else {
              sendResponse({ error: "No active tab" });
            }
          });
          return true;
        // Will respond asynchronously
        // Start mission automation - broadcast to all frames (including game iframe)
        case "START_MISSION_AUTOMATION":
          extensionLogger.log("Broadcasting START_MISSION_AUTOMATION to all frames");
          chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            if (tabs[0]?.id) {
              chrome.tabs.sendMessage(
                tabs[0].id,
                {
                  type: "START_MISSION_AUTOMATION"
                },
                { frameId: void 0 },
                // undefined = all frames
                (response) => {
                  if (chrome.runtime.lastError) {
                    extensionLogger.warn("Message error", {
                      error: chrome.runtime.lastError.message
                    });
                  } else {
                    extensionLogger.log("Message delivered", { response });
                  }
                }
              );
            }
          });
          sendResponse({ success: true });
          break;
        // Stop mission automation - broadcast to all frames
        case "STOP_MISSION_AUTOMATION":
          extensionLogger.log("Broadcasting STOP_MISSION_AUTOMATION to all frames");
          chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            if (tabs[0]?.id) {
              chrome.tabs.sendMessage(tabs[0].id, {
                type: "STOP_MISSION_AUTOMATION"
              });
            }
          });
          sendResponse({ success: true });
          break;
        // Events from content scripts → state machine
        case "GAME_LOADER_DETECTED":
          extensionLogger.log("GAME_LOADER_DETECTED, sending to state machine");
          sendToStateMachine({ type: "GAME_LOADER_DETECTED" });
          sendResponse({ success: true });
          break;
        case "GAME_DIALOG_OPENED":
          extensionLogger.log("GAME_DIALOG_OPENED, sending to state machine");
          sendToStateMachine({ type: "GAME_DIALOG_OPENED" });
          sendResponse({ success: true });
          break;
        case "AUTOMATION_READY":
          extensionLogger.log("AUTOMATION_READY, sending AUTOMATION_STARTED to state machine");
          gameFrameId = sender.frameId;
          gameTabId = sender.tab?.id;
          extensionLogger.log("Tracked game frame", { gameFrameId, gameTabId });
          sendToStateMachine({ type: "AUTOMATION_STARTED" });
          {
            const currentState2 = botActor?.getSnapshot();
            if (currentState2?.matches("gameMission.running")) {
              extensionLogger.log(
                "Bot is running, re-broadcasting START_MISSION_AUTOMATION to newly ready iframe"
              );
              chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
                if (tabs[0]?.id) {
                  chrome.tabs.sendMessage(
                    tabs[0].id,
                    { type: "START_MISSION_AUTOMATION" },
                    { frameId: void 0 }
                  );
                }
              });
            }
          }
          sendResponse({ success: true });
          break;
        case "MISSION_DELETED":
          {
            const deletedPostId = message.missionId;
            extensionLogger.log("MISSION_DELETED received", {
              postId: deletedPostId
            });
            if (deletedPostId) {
              gamePreviewReloadAttempts.delete(deletedPostId);
            }
            if (deletedPostId) {
              try {
                await setMissionDisabled(deletedPostId, true);
                extensionLogger.log("Mission disabled in storage", {
                  postId: deletedPostId
                });
                sendToStateMachine({
                  type: "MISSION_DELETED",
                  missionId: deletedPostId
                });
                extensionLogger.log("Finding next mission after deleted post");
              } catch (error) {
                extensionLogger.error("Failed to disable mission", {
                  postId: deletedPostId,
                  error: String(error)
                });
                sendToStateMachine({
                  type: "MISSION_DELETED",
                  missionId: deletedPostId
                });
                extensionLogger.log("Finding next mission after error disabling deleted post");
              }
              findAndSendNextMission();
            } else {
              sendToStateMachine({
                type: "MISSION_DELETED",
                missionId: deletedPostId
              });
            }
            sendResponse({ success: true });
          }
          break;
        case "GAME_STATE_UPDATE":
          {
            const gameState = message.gameState;
            if (gameState) {
              const snapshot2 = getStateMachineSnapshot();
              if (snapshot2 && botActor) {
                snapshot2.context.gameState = {
                  postId: gameState.postId,
                  encounterCurrent: gameState.encounterCurrent ?? 0,
                  encounterTotal: gameState.encounterTotal ?? 0,
                  lives: gameState.lives ?? 3,
                  screen: gameState.screen ?? "unknown",
                  difficulty: gameState.difficulty
                };
                const presentationState = getPresentationStateName(snapshot2);
                broadcastToReddit({
                  type: "STATE_CHANGED",
                  state: presentationState,
                  context: snapshot2.context
                });
              }
            }
            sendResponse({ success: true });
          }
          break;
        case "GAME_PREVIEW_FAILED":
          {
            const missionId = message.missionId;
            const attempts = gamePreviewReloadAttempts.get(missionId) || 0;
            if (attempts === 0) {
              extensionLogger.warn("[GamePreview] Failed, attempting reload", { missionId });
              gamePreviewReloadAttempts.set(missionId, 1);
              sendResponse({ action: "reload" });
            } else {
              extensionLogger.error("[GamePreview] Failed after reload, skipping mission", {
                missionId
              });
              gamePreviewReloadAttempts.delete(missionId);
              try {
                await setMissionDisabled(missionId, true);
                extensionLogger.log("[GamePreview] Mission marked as disabled in storage", {
                  missionId
                });
                sendToStateMachine({
                  type: "MISSION_DELETED",
                  missionId
                });
                sendResponse({ action: "skip" });
              } catch (error) {
                extensionLogger.error(
                  "[GamePreview] CRITICAL: Failed to disable mission in storage, stopping bot",
                  {
                    missionId,
                    error: String(error)
                  }
                );
                sendToStateMachine({
                  type: "ERROR_OCCURRED",
                  message: `Failed to disable broken mission ${missionId}: ${String(error)}`
                });
                sendResponse({ action: "error", error: String(error) });
              }
            }
          }
          break;
        case "MISSION_COMPLETED":
          {
            const completedPostId = message.postId;
            extensionLogger.log("MISSION_COMPLETED received", {
              postId: completedPostId
            });
            if (completedPostId) {
              gamePreviewReloadAttempts.delete(completedPostId);
            }
            if (completedPostId) {
              markMissionCleared(completedPostId).then(() => {
                extensionLogger.log("Mission marked as cleared in storage", {
                  postId: completedPostId
                });
                sendToStateMachine({
                  type: "MISSION_COMPLETED",
                  missionId: completedPostId
                });
              }).catch((error) => {
                extensionLogger.error("Failed to mark mission as cleared", {
                  postId: completedPostId,
                  error: String(error)
                });
                sendToStateMachine({
                  type: "MISSION_COMPLETED",
                  missionId: completedPostId
                });
              });
            } else {
              sendToStateMachine({
                type: "MISSION_COMPLETED",
                missionId: completedPostId
              });
            }
            sendResponse({ success: true });
          }
          break;
        case "MISSION_FOUND":
          extensionLogger.log("MISSION_FOUND, sending event to state machine");
          const missionData = message;
          const snapshot = getStateMachineSnapshot();
          const currentState = getPresentationStateName(snapshot);
          if (currentState === "completing") {
            extensionLogger.log("In completing state, sending NEXT_MISSION_FOUND");
            sendToStateMachine({
              type: "NEXT_MISSION_FOUND",
              missionId: missionData.missionId,
              permalink: missionData.permalink
            });
          } else if (missionData.isCurrentPage) {
            extensionLogger.log("Already on page, sending MISSION_PAGE_LOADED");
            sendToStateMachine({
              type: "MISSION_PAGE_LOADED",
              missionId: missionData.missionId,
              permalink: missionData.permalink
            });
          } else {
            extensionLogger.log("Need to navigate, sending NAVIGATE_TO_MISSION");
            sendToStateMachine({
              type: "NAVIGATE_TO_MISSION",
              missionId: missionData.missionId,
              permalink: missionData.permalink
            });
          }
          sendResponse({ success: true });
          break;
        case "NO_MISSIONS_FOUND":
          extensionLogger.log("NO_MISSIONS_FOUND, sending to state machine");
          sendToStateMachine({ type: "NO_MISSIONS_FOUND" });
          sendResponse({ success: true });
          break;
        case "ERROR_OCCURRED":
          {
            const errorSnapshot = getStateMachineSnapshot();
            const errorMessage = message.message || "Unknown error";
            extensionLogger.error("ERROR_OCCURRED", {
              errorMessage,
              currentState: errorSnapshot?.value
            });
            sendToStateMachine({
              type: "ERROR_OCCURRED",
              message: errorMessage
            });
            sendResponse({ success: true });
          }
          break;
        case "MISSIONS_UPDATED":
          chrome.runtime.sendMessage({
            type: "MISSIONS_CHANGED"
          }).catch(() => {
          });
          sendResponse({ success: true });
          break;
        case "PING":
          {
            const snapshot2 = getStateMachineSnapshot();
            const state = getPresentationStateName(snapshot2);
            sendResponse({
              success: true,
              state,
              context: snapshot2?.context,
              timestamp: Date.now()
            });
          }
          break;
        case "FETCH_MISSION_DATA":
          {
            const { postId, tabId } = message;
            if (!postId || !tabId) {
              sendResponse({ success: false, error: "Missing postId or tabId" });
              break;
            }
            extensionLogger.log("[FETCH_MISSION_DATA] Forwarding request to content script", {
              postId,
              tabId
            });
            chrome.tabs.sendMessage(
              tabId,
              {
                type: "FETCH_MISSION_DATA_FROM_PAGE",
                postId
              },
              (response) => {
                if (chrome.runtime.lastError) {
                  extensionLogger.error("[FETCH_MISSION_DATA] Failed to communicate with tab", {
                    error: chrome.runtime.lastError.message,
                    tabId
                  });
                  sendResponse({
                    success: false,
                    error: `Failed to communicate with tab: ${chrome.runtime.lastError.message}`
                  });
                } else if (response?.success) {
                  extensionLogger.log("[FETCH_MISSION_DATA] Successfully fetched mission data", {
                    postId,
                    data: response.data
                  });
                  sendResponse({
                    success: true,
                    data: response.data
                  });
                } else {
                  extensionLogger.error("[FETCH_MISSION_DATA] Failed to fetch mission data", {
                    error: response?.error
                  });
                  sendResponse({
                    success: false,
                    error: response?.error || "Unknown error occurred"
                  });
                }
              }
            );
          }
          return true;
        // Will respond asynchronously
        default:
          extensionLogger.warn("Unknown message type", { type: message.type });
          sendResponse({ error: "Unknown message type: " + message.type });
      }
    }
    function cleanup() {
      extensionLogger.warn("Shutting down", {
        timestamp: (/* @__PURE__ */ new Date()).toISOString(),
        hadActiveActor: !!botActor
      });
      if (botActor) {
        try {
          botActor.stop();
          extensionLogger.log("Stopped botActor");
        } catch (error) {
          extensionLogger.warn("Error stopping botActor", { error: String(error) });
        }
        botActor = null;
      }
    }
    if (chrome.runtime.onSuspend) {
      chrome.runtime.onSuspend.addListener(() => {
        extensionLogger.warn("onSuspend event fired, running cleanup");
        cleanup();
      });
    }
    async function initializeExtension() {
      try {
        const username = await getCurrentRedditUser();
        extensionLogger.log("[UserDetection] Extension loaded, current user:", username);
      } catch (error) {
        extensionLogger.error("[UserDetection] Failed to detect user on load:", error);
      }
      try {
        extensionLogger.log("[MissionMigration] Checking for missions to migrate...");
        const result2 = await migrateMissionsStorage();
        if (result2.migrated > 0) {
          extensionLogger.log("[MissionMigration] Migrated missions from old format", {
            total: result2.total,
            migrated: result2.migrated,
            alreadyFlat: result2.alreadyFlat,
            errors: result2.errors.length
          });
        } else {
          extensionLogger.log("[MissionMigration] All missions already in flat format");
        }
      } catch (error) {
        extensionLogger.error("[MissionMigration] Failed to migrate missions", {
          error: String(error)
        });
      }
      try {
        const needsToMigrate = await needsMigration();
        if (needsToMigrate) {
          extensionLogger.log("[ProgressMigration] Storage migration needed, starting migration...");
          const result2 = await migrateToSeparateProgress();
          extensionLogger.log("[ProgressMigration] Migration completed", result2);
        } else {
          extensionLogger.log("[ProgressMigration] No migration needed");
        }
      } catch (error) {
        extensionLogger.error("[ProgressMigration] Migration check failed", { error: String(error) });
      }
    }
    initializeExtension();
    extensionLogger.log("Sword & Supper Bot background script loaded");
  });
  function initPlugins() {
  }
  globalThis.browser?.runtime?.id ? globalThis.browser : globalThis.chrome;
  function print(method, ...args) {
    return;
  }
  const logger = {
    debug: (...args) => print(console.debug, ...args),
    log: (...args) => print(console.log, ...args),
    warn: (...args) => print(console.warn, ...args),
    error: (...args) => print(console.error, ...args)
  };
  let result;
  try {
    initPlugins();
    result = definition.main();
    if (result instanceof Promise) {
      console.warn(
        "The background's main() function return a promise, but it must be synchronous"
      );
    }
  } catch (err) {
    logger.error("The background crashed on startup!");
    throw err;
  }
  const result$1 = result;
  return result$1;
})();
